//----------------------------------------------------------------------------------------;
//	Network.cc
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//	Change History:
//----------------------------------------------------------------------------------------
  
#include "Network.h"

FloorPlan* Net::fFP = NULL;					 
ByteGrid* Net::fStudyArea = NULL;

int  Net::fProcDims[2];
byte  Net::fNetInfo[8];

void Net::Startup() {

#ifdef USE_MPI
  int tag;
  MPI_Initialized(&tag);
  if( !tag ) { fprintf(stderr, "MPI Environment Not yet Initialized."); exit(11); } 
  fProcDims[0] = fProcDims[1] = 0;
  MPI_Dims_create(gNProc, 2, fProcDims);
#else
   fProcDims[0] = fProcDims[1] = 0;
#endif

   fNetInfo[kMaxScale] = 0;
   fStudyArea = NULL;
   fFP = NULL;
}

void Net::Finalize() {
  if(fFP) delete fFP;
}

int Net::SetMaxScale(int scale) { 
  fNetInfo[kMaxScale] = scale;
  return 1;
}

Region2& Net::LocalRegion(int proc) {
  if(fFP==NULL) GenerateFloorPlan();
  return (*fFP)(proc);
}

FloorPlan* Net::GetFloorPlan() {
  if(fFP==NULL) GenerateFloorPlan();
  return fFP;
}

int  Net::SetDims( int d0, int d1 ) {
  if(fFP) { 
    gPrintErr("Must Specify dims before FloorPlan is Generated, Default decomp Used."); 
    return 0;
  }
  fProcDims[0] = d0;
  fProcDims[1] = d1;
  return 1;
}


void Net::GenerateFloorPlan() {
  fFP = new FloorPlan( gNProcs() );
  for(int i=0; i<gNProcs(); i++) fFP->setowner(i,i);
  RecursiveBisectionDistribution();
}

int quantize(int index, int quantum) {
  if(quantum==0) return index;
  int quot = index/quantum; int rem = index % quantum;
  if (rem > quantum/2) return (quot + 1)*quantum;
  else return quot*quantum;
}

// rows:0, cols:1

void Net::RecursiveBisectionDistribution ( ) {
  if( fFP == NULL) { gFatal(" No FloorPlan Allocated "); } 
  if( fStudyArea == NULL) { gFatal("No StudyArea Specified for FloorPlans"); } 
  int rows = fStudyArea->extents(0);
  int cols = fStudyArea->extents(1);
  if( gNProc == 1 ) {
    (*fFP)(0).setlower(0,0); 
    (*fFP)(0).setupper(rows-1,cols-1);
    return;
  }
  int quantum = fNetInfo[kMaxScale];
  int SArea=0, SAreaR=0, SAreaC=0;
  byte mVal;
  int ir0=0, ip=0, ipg=0, irq, icq, ir, ic; 
  if(gDebug>1) {
    gPrintLog("\n---------Distribution DATA---------------\n");
    gMsgStr[cols] = '\n';
    gMsgStr[cols+1] = '\0';
    for(ir=0; ir<rows; ir++) {
      for(ic=0; ic<cols; ic++) {
				mVal = fStudyArea->BValue(ir,ic,0);
				if( mVal > 0 && (mVal <= 200) ) gMsgStr[ic] = '*';
				else gMsgStr[ic] = '_';
      } 
      gPrintLog();
    }
  }
  for(ir=0; ir<rows; ir++) {
    for(ic=0; ic<cols; ic++) {
      mVal = fStudyArea->BValue(ir,ic,0);
      if( mVal > 0 && (mVal <= 200) ) SArea++;
    } 
  }
    
  for(ir=0; ir<rows; ir++) {
    if( ip == fProcDims[0]-1 ) {
      if(gDebug>1) { sprintf(gMsgStr,"\nDistribute: Setting final row bifurcation at %d",rows-1); gPrintLog(); }
      for(ipg = ip*fProcDims[1]; ipg < (ip+1)*fProcDims[1]; ipg++) {
				(*fFP)(ipg).setlower(ir0,0); (*fFP)(ipg).setupper(rows-1,cols-1); 
				if(gDebug>1) { sprintf(gMsgStr,"Partition%d: (%d/%d)->(%d/%d): q%d ",ipg,ir0,0,rows-1,cols-1,quantum); gPrintLog(); }
      }
      break;
    }
    for(ic=0; ic<cols; ic++) {
      mVal = fStudyArea->BValue(ir,ic,0);
      if( mVal > 0 && (mVal <= 200) ) SAreaR++;
    }
    if( SAreaR >=  SArea/fProcDims[0] || ir == rows-1 ) {
			 irq = quantize(ir,quantum);
			 if(gDebug>1) { sprintf(gMsgStr,"\nDistribute: Setting row bifurcation (%d/%d) at %d(%d)/%d:", SAreaR,SArea,irq,ir,rows); gPrintLog(); }
			 for(ipg = ip*fProcDims[1]; ipg < (ip+1)*fProcDims[1]; ipg++) { 
					(*fFP)(ipg).setlower(ir0,0); (*fFP)(ipg).setupper(irq-1,cols-1); 
					if(gDebug>1) { sprintf(gMsgStr,"Partition%d: (%d/%d)->(%d/%d): q%d ",ipg,ir0,0,irq-1,cols-1,quantum); gPrintLog(); }
			 }
			 ir0 = irq; ip++; SAreaR = 0;
    }
  }
  
  for(ip=0; ip<fProcDims[0]; ip++) {
    int ic0=0, ip1=0, lrows, lcols;    
    ipg = ip*fProcDims[1]; SArea = 0;
    lrows = (*fFP)(ipg).extents(0);
    lcols = (*fFP)(ipg).extents(1);
    for(ir=0; ir<lrows; ir++) {
      for(ic=0; ic<lcols; ic++) {
				mVal = fStudyArea->BValue(ir,ic,0);
				if( mVal > 0 && (mVal <= 200) ) SArea++;
      } 
    }
    int lb = (*fFP)(ipg).lower(0);
    int ub = (*fFP)(ipg).upper(0);
    for(ic=0; ic<cols; ic++) {
      if( ip1 == fProcDims[1]-1 ) {
				if(gDebug>1) { 
					sprintf(gMsgStr,"\nDistribute: Setting final col bifurcation at %d: ",cols-1); 
					gPrintLog(); 
				}
				(*fFP)(ipg+ip1).setlower(lb,ic0); 
				(*fFP)(ipg+ip1).setupper(ub,cols-1);
				if(gDebug>1) { sprintf(gMsgStr,"Partition%d (%d/%d)->(%d/%d) ",ipg,lb,ic0,ub,cols-1); gPrintLog(); }
				break;
			}
			for(ir=lb; ir<ub; ir++) {
				mVal = fStudyArea->BValue(ir,ic,0);
				if( mVal > 0 && (mVal <= 200) ) SAreaC++;
			} 
			if( SAreaC >=  SArea/fProcDims[1]) {
				icq = quantize(ic,quantum);
				if(gDebug>1) { sprintf(gMsgStr,"\nDistribute: Setting column bifurcation (%d+%d): (%d/%d) at %d(%d)/%d: ",
							 ipg,ip1,SAreaC,SArea,icq,ic,cols); gPrintLog(); }
				(*fFP)(ipg+ip1).setlower(lb,ic0); 
				(*fFP)(ipg+ip1).setupper(ub,icq-1);
				if(gDebug>1) { 
					sprintf(gMsgStr,"Partition%d: (%d/%d)->(%d/%d): q%d ",ipg,lb,ic0,ub,icq-1,quantum); 
					gPrintLog(); 
				}
				ic0 = icq; ip1++; SAreaC = 0;
      }
    }
  }	
  for( ip=0; ip<gNProc; ip++ ) {
    sprintf(gMsgStr,"\nPartition%d(r,c): (%d,%d)->(%d,%d)", ip,
	    (*fFP)(ip).lower(0),(*fFP)(ip).lower(1),
	    (*fFP)(ip).upper(0),(*fFP)(ip).upper(1)); 
    gPrintLog(); 
  }
}





