#ifdef __GNUG__
#pragma implementation
#endif
// #include <builtin.h>
#include "PointPlex.h"
#include "Environ.h"

// IChunk support

void OrderedPointIChunk::error(const char* msg) const
{
  gPrintErr(msg);
}

void OrderedPointIChunk::index_error() const
{
  error("attempt to use invalid index");
}

void OrderedPointIChunk::empty_error() const
{
  error("invalid use of empty chunk");
}

void OrderedPointIChunk::full_error() const
{
  error("attempt to extend chunk beyond bounds");
}

OrderedPointIChunk:: ~OrderedPointIChunk() {}

OrderedPointIChunk::OrderedPointIChunk(OrderedPoint*     d,    
                     int      baseidx,
                     int      lowidx,
                     int      fenceidx,
                     int      topidx)
{
  if (d == 0 || baseidx > lowidx || lowidx > fenceidx || fenceidx > topidx)
    error("inconsistent specification");
  data = d;
  base = baseidx;
  low = lowidx;
  fence = fenceidx;
  top = topidx;
  nxt = prv = this;
}

void OrderedPointIChunk:: re_index(int lo)
{
  int delta = lo - low;
  base += delta;
  low += delta;
  fence += delta;
  top += delta;
}


void OrderedPointIChunk::clear(int lo)
{
  int s = top - base;
  low = base = fence = lo;
  top = base + s;
}

void OrderedPointIChunk::cleardown(int hi)
{
  int s = top - base;
  low = top = fence = hi;
  base = top - s;
}

int OrderedPointIChunk:: OK() const
{
  int v = data != 0;             // have some data
  v &= base <= low;              // ok, index-wise
  v &= low <= fence;
  v &= fence <= top;

  v &=  nxt->prv == this;      // and links are OK
  v &=  prv->nxt == this;
  if (!v) error("invariant failure");
  return(v);
}


// error handling


void OrderedPointPlex::error(const char* msg) const
{
  gPrintErr(msg);
}

void OrderedPointPlex::index_error() const
{
  error("attempt to access invalid index");
}

void OrderedPointPlex::empty_error() const
{
  error("attempted operation on empty plex");
}

void OrderedPointPlex::full_error() const
{
  error("attempt to increase size of plex past limit");
}

// generic plex ops

OrderedPointPlex:: ~OrderedPointPlex()
{
  invalidate();
}  


void OrderedPointPlex::append (const OrderedPointPlex& a)
{
  for (int i = a.low(); i < a.fence(); a.next(i)) add_high(a[i]);
}

void OrderedPointPlex::prepend (const OrderedPointPlex& a)
{
  for (int i = a.high(); i > a.ecnef(); a.prev(i)) add_low(a[i]);
}

void OrderedPointPlex::reverse()
{
  OrderedPoint tmp;
  int l = low();
  int h = high();
  while (l < h)
  {
    tmp = (*this)[l];
    (*this)[l] = (*this)[h];
    (*this)[h] = tmp;
    next(l);
    prev(h);
  }
}


void OrderedPointPlex::fill(const OrderedPoint& x)
{
  for (int i = lo; i < fnc; ++i) (*this)[i] = x;
}

void OrderedPointPlex::fill(const OrderedPoint& x, int lo, int hi)
{
  for (int i = lo; i <= hi; ++i) (*this)[i] = x;
}


void OrderedPointPlex::del_chunk(OrderedPointIChunk* x)
{
  if (x != 0)
  {
    x->unlink();
    OrderedPoint* data = (OrderedPoint*)(x->invalidate());
    delete [] data;
    delete x;
  }
}


void OrderedPointPlex::invalidate()
{
  OrderedPointIChunk* t = hd;
  if (t != 0)
  {
    OrderedPointIChunk* tail = tl();
    while (t != tail)
    {
      OrderedPointIChunk* nxt = t->next();
      del_chunk(t);
      t = nxt;
    } 
    del_chunk(t);
    hd = 0;
  }
}

int OrderedPointPlex::reset_low(int l)
{
  int old = lo;
  int diff = l - lo;
  if (diff != 0)
  {
    lo += diff;
    fnc += diff;
    OrderedPointIChunk* t = hd;
    do
    {
      t->re_index(t->low_index() + diff);
      t = t->next();
    } while (t != hd);
  }
  return old;
}

typedef OrderedPointIChunk* _OrderedPointIChunk_ptr;

OrderedPointRPlex:: OrderedPointRPlex()
{
  lo = fnc = 0;
  csize = DEFAULT_INITIAL_CAPACITY;
  OrderedPoint* data = new OrderedPoint[csize];
  set_cache(new OrderedPointIChunk(data,  lo, lo, fnc, lo+csize));
  hd = ch;
  maxch = MIN_NCHUNKS;
  lch = maxch / 2;
  fch = lch + 1;
  base = ch->base_index() - lch * csize;
  chunks = new _OrderedPointIChunk_ptr[maxch];
  chunks[lch] = ch;
}

OrderedPointRPlex:: OrderedPointRPlex(int chunksize)
{
  if (chunksize == 0) error("invalid constructor specification");
  lo = fnc = 0;
  if (chunksize > 0)
  {
    csize = chunksize;
    OrderedPoint* data = new OrderedPoint[csize];
    set_cache(new OrderedPointIChunk(data,  lo, lo, fnc, csize+lo));
    hd = ch;
  }
  else
  {
    csize = -chunksize;
    OrderedPoint* data = new OrderedPoint[csize];
    set_cache(new OrderedPointIChunk(data,  chunksize+lo, lo, fnc, fnc));
    hd = ch;
  }
  maxch = MIN_NCHUNKS;
  lch = maxch / 2;
  fch = lch + 1;
  base = ch->base_index() - lch * csize;
  chunks = new _OrderedPointIChunk_ptr[maxch];
  chunks[lch] = ch;
}


OrderedPointRPlex:: OrderedPointRPlex(int l, int chunksize)
{
  if (chunksize == 0) error("invalid constructor specification");
  lo = fnc = l;
  if (chunksize > 0)
  {
    csize = chunksize;
    OrderedPoint* data = new OrderedPoint[csize];
    set_cache(new OrderedPointIChunk(data,  lo, lo, fnc, lo+csize));
    hd = ch;
  }
  else
  {
    csize = -chunksize;
    OrderedPoint* data = new OrderedPoint[csize];
    set_cache(new OrderedPointIChunk(data,  chunksize+lo, lo, fnc, fnc));
    hd = ch;
  }
  maxch = MIN_NCHUNKS;
  lch = maxch / 2;
  fch = lch + 1;
  base = ch->base_index() - lch * csize;
  chunks = new _OrderedPointIChunk_ptr[maxch];
  chunks[lch] = ch;
}

void OrderedPointRPlex::make_initial_chunks(int up)
{
  int count = 0;
  int need = fnc - lo;
  hd = 0;
  if (up)
  {
    int l = lo;
    do
    {
      ++count;
      int sz;
      if (need >= csize)
        sz = csize;
      else
        sz = need;
      OrderedPoint* data = new OrderedPoint [csize];
      OrderedPointIChunk* h = new OrderedPointIChunk(data,  l, l, l+sz, l+csize);
      if (hd != 0)
        h->link_to_next(hd);
      else
        hd = h;
      l += sz;
      need -= sz;
    } while (need > 0);
  }
  else
  {
    int hi = fnc;
    do
    {
      ++count;
      int sz;
      if (need >= csize)
        sz = csize;
      else
        sz = need;
      OrderedPoint* data = new OrderedPoint [csize];
      OrderedPointIChunk* h = new OrderedPointIChunk(data,  hi-csize, hi-sz, hi, hi);
      if (hd != 0)
        h->link_to_next(hd);
      hd = h;
      hi -= sz;
      need -= sz;
    } while (need > 0);
  }
  set_cache((OrderedPointIChunk*)hd);
  
  maxch = MIN_NCHUNKS;
  if (maxch < count * 2)
    maxch = count * 2;
  chunks = new _OrderedPointIChunk_ptr[maxch];
  lch = maxch / 3;
  fch = lch + count;
  base = ch->base_index() - csize * lch;
  int k = lch;
  do
  {
    chunks[k++] = ch;
    set_cache(ch->next());
  } while (ch != hd);
}

OrderedPointRPlex:: OrderedPointRPlex(int l, int hi, const OrderedPoint& initval, int chunksize)
{
  lo = l;
  fnc = hi + 1;
  if (chunksize == 0)
  {
    csize = fnc - l;
    make_initial_chunks(1);
  }
  else if (chunksize < 0)
  {
    csize = -chunksize;
    make_initial_chunks(0);
  }
  else
  {
    csize = chunksize;
    make_initial_chunks(1);
  }
  fill(initval);
}

OrderedPointRPlex::OrderedPointRPlex(const OrderedPointRPlex& a)
{
  lo = a.lo;
  fnc = a.fnc;
  csize = a.csize;
  make_initial_chunks();
  for (int i = a.low(); i < a.fence(); a.next(i)) (*this)[i] = a[i];
}

void OrderedPointRPlex::operator= (const OrderedPointRPlex& a)
{
  if (&a != this)
  {
    invalidate();
    lo = a.lo;
    fnc = a.fnc;
    csize = a.csize;
    make_initial_chunks();
    for (int i = a.low(); i < a.fence(); a.next(i)) (*this)[i] = a[i];
  }
}


void OrderedPointRPlex::cache(const OrderedPoint* p) const
{
  const OrderedPointIChunk* old = ch;
  const OrderedPointIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->next());
    if (t == old) index_error();
  }
  set_cache(t);
}

int OrderedPointRPlex::owns(Pix px) const
{
  OrderedPoint* p = (OrderedPoint*)px;
  const OrderedPointIChunk* old = ch;
  const OrderedPointIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->next());
    if (t == old) return 0;
  }
  set_cache(t);
  return 1;
}


OrderedPoint* OrderedPointRPlex::dosucc(const OrderedPoint* p) const
{
  if (p == 0) return 0;
  const OrderedPointIChunk* old = ch;
  const OrderedPointIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->next());
    if (t == old) return 0;
  }
  int i = t->index_of(p) + 1;
  if (i >= fnc) return 0;
  if (i >= t->fence_index()) t = (t->next());
  set_cache(t);
  return t->pointer_to(i);
}

OrderedPoint* OrderedPointRPlex::dopred(const OrderedPoint* p) const
{
  if (p == 0) return 0;
  const OrderedPointIChunk* old = ch;
  const OrderedPointIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->prev());
    if (t == old) return 0;
  }
  int i = t->index_of(p) - 1;
  if (i < lo) return 0;
  if (i < t->low_index()) t = (t->prev());
  set_cache(t);
  return (t->pointer_to(i));
}

int OrderedPointRPlex::add_high(const OrderedPoint& elem)
{
  OrderedPointIChunk* t = tl();
  if (!t->can_grow_high())
  {
    if (t->OrderedPointIChunk::empty() && one_chunk())
    {
      t->clear(fnc);
      base = t->base_index() - lch * csize;
    }
    else
    {
      OrderedPoint* data = new OrderedPoint [csize];
      t = (new OrderedPointIChunk(data,  fnc, fnc, fnc,fnc+csize));
      t->link_to_prev(tl());
      if (fch == maxch)
      {
        maxch *= 2;
        OrderedPointIChunk** newch = new _OrderedPointIChunk_ptr [maxch];
        memcpy(newch, chunks, fch * sizeof(_OrderedPointIChunk_ptr));
        delete[] chunks;
        chunks = newch;
      }
      chunks[fch++] = t;
    }
  }
  *((t->OrderedPointIChunk::grow_high())) = elem;
  set_cache(t);
  return fnc++;
}

int OrderedPointRPlex::del_high ()
{
  if (empty()) empty_error();
  OrderedPointIChunk* t = tl();
  if (t->OrderedPointIChunk::empty()) // kill straggler first
  {
    OrderedPointIChunk* pred = t->prev();
    del_chunk(t);
    t = (pred);
    --fch;
  }
  t->OrderedPointIChunk::shrink_high();
  if (t->OrderedPointIChunk::empty() && !one_chunk())
  {
    OrderedPointIChunk* pred = t->prev();
    del_chunk(t);
    t = (pred);
    --fch;
  }
  set_cache(t);
  return --fnc - 1;
}

int OrderedPointRPlex::add_low (const OrderedPoint& elem)
{
  OrderedPointIChunk* t = hd;
  if (!t->can_grow_low())
  {
    if (t->OrderedPointIChunk::empty() && one_chunk())
    {
      t->cleardown(lo);
      base = t->base_index() - lch * csize;
    }
    else
    {
      OrderedPoint* data = new OrderedPoint [csize];
      hd = new OrderedPointIChunk(data,  lo-csize, lo, lo, lo);
      hd->link_to_next(t);
      t = ( hd);
      if (lch == 0)
      {
        lch = maxch;
        fch += maxch;
        maxch *= 2;
        OrderedPointIChunk** newch = new _OrderedPointIChunk_ptr [maxch];
        memcpy(&(newch[lch]), chunks, lch * sizeof(_OrderedPointIChunk_ptr));
        delete[] chunks;
        chunks = newch;
        base = t->base_index() - (lch - 1) * csize;
      }
      chunks[--lch] = t;
    }
  }
  *((t->OrderedPointIChunk::grow_low())) = elem;
  set_cache(t);
  return --lo;
}


int OrderedPointRPlex::del_low ()
{
  if (empty()) empty_error();
  OrderedPointIChunk* t = hd;
  if (t->OrderedPointIChunk::empty())
  {
    hd = t->next();
    del_chunk(t);
    t = hd;
    ++lch;
  }
  t->OrderedPointIChunk::shrink_low();
  if (t->OrderedPointIChunk::empty() && !one_chunk())
  {
    hd = t->next();
    del_chunk(t);
    t = hd;
    ++lch;
  }
  set_cache(t);
  return ++lo;
}

void OrderedPointRPlex::reverse()
{
  OrderedPoint tmp;
  int l = lo;
  int h = fnc - 1;
  OrderedPointIChunk* loch = hd;
  OrderedPointIChunk* hich = tl();
  while (l < h)
  {
    OrderedPoint* lptr = loch->pointer_to(l);
    OrderedPoint* hptr = hich->pointer_to(h);
    tmp = *lptr;
    *lptr = *hptr;
    *hptr = tmp;
    if (++l >= loch->fence_index()) loch = loch->next();
    if (--h < hich->low_index()) hich = hich->prev();
  }
}

void OrderedPointRPlex::fill(const OrderedPoint& x)
{
  for (int i = lo; i < fnc; ++i) (*this)[i] = x;
}

void OrderedPointRPlex::fill(const OrderedPoint& x, int lo, int hi)
{
  for (int i = lo; i <= hi; ++i) (*this)[i] = x;
}


void OrderedPointRPlex::clear()
{
  for (int i = lch + 1; i < fch; ++i)
    del_chunk(chunks[i]);
  fch = lch + 1;
  set_cache(chunks[lch]);
  ch->OrderedPointIChunk::clear(lo);
  fnc = lo;
}

int OrderedPointRPlex::reset_low(int l)
{
  int old = lo;
  int diff = l - lo;
  if (diff != 0)
  {
    lo += diff;
    fnc += diff;
    OrderedPointIChunk* t = hd;
    do
    {
      t->re_index(t->low_index() + diff);
      t = t->next();
    } while (t != hd);
  }
  base = hd->base_index() - lch * csize;
  return old;
}


int OrderedPointRPlex::OK () const
{
  int v = hd != 0 && ch != 0;         // at least one chunk

  v &= fnc == tl()->fence_index();  // last chunk fnc == plex fnc
  v &= lo == hd->OrderedPointIChunk::low_index();    // first lo == plex lo

  v &= base == hd->base_index() - lch * csize; // base is correct;
  v &= lch < fch;
  v &= fch <= maxch;                  // within allocation;

// loop for others:

  int k = lch;                        // to cross-check nch

  int found_ch = 0;                   // to make sure ch is in list;
  const OrderedPointIChunk* t = (hd);
  for (;;)
  {
    v &= chunks[k++] == t;             // each chunk is at proper index
    if (t == ch) ++found_ch;
    v &= t->OrderedPointIChunk::OK();              // each chunk is OK
    if (t == tl())
      break;
    else                              // and has indices contiguous to succ
    {
      v &= t->top_index() == t->next()->base_index();
      if (t != hd)                  // internal chunks full
      {
        v &= !t->empty();
        v &= !t->can_grow_low();
        v &= !t->can_grow_high();
      }
      t = t->next();
    }
  }
  v &= found_ch == 1;
  v &= fch == k;
  if (!v) error("invariant failure");
  return v;
}
