#ifndef _OrderedPointPlex_h
#ifdef __GNUG__
#pragma interface
#endif
#define _OrderedPointPlex_h 1

// #include <std.h>
#include "Globals.h"
#include "OrderedPoint.h"

// Plexes are made out of OrderedPointIChunks

#include <stddef.h>

class OrderedPointIChunk
{
//public: // kludge until C++ `protected' policies settled
protected:      

  OrderedPoint*           data;           // data, from client

  int            base;           // lowest possible index
  int            low;            // lowest valid index
  int            fence;          // highest valid index + 1
  int            top;            // highest possible index + 1

  OrderedPointIChunk*     nxt;            // circular links
  OrderedPointIChunk*     prv;

public:

// constructors

                 OrderedPointIChunk(OrderedPoint*     d,       // ptr to array of elements
                        int      base_idx, // initial indices
                        int      low_idx,  
                        int      fence_idx,
                        int      top_idx);

  virtual       ~OrderedPointIChunk();

// status reports

  int            size() const;    // number of slots

  inline virtual int    empty() const ;
  inline virtual int    full() const ; 

  int            can_grow_high () const ;  // there is space to add data
  int            can_grow_low () const;        
  
  int            base_index() const;   // lowest possible index;
  int            low_index() const;    // lowest actual index;
  inline virtual int    first_index() const;  // lowest valid index or fence if none
  inline virtual int    last_index() const;   // highest valid index or low-1 if none
  int            fence_index() const;  // highest actual index + 1
  int            top_index() const;    // highest possible index + 1

// indexing conversion

  int            possible_index(int i) const; // i between base and top
  int            actual_index(int i) const;   // i between low and fence
  inline virtual int    valid_index(int i) const;    // i not deleted (mainly for mchunks)

  int            possible_pointer(const OrderedPoint* p) const; // same for ptr
  int            actual_pointer(const OrderedPoint* p) const; 
  inline virtual int    valid_pointer(const OrderedPoint* p) const; 

  OrderedPoint*           pointer_to(int i) const ;   // pointer to data indexed by i
                                      // caution: i is not checked for validity
  int            index_of(const OrderedPoint* p) const; // index of data pointed to by p
                                      // caution: p is not checked for validity

  inline virtual int    succ(int idx) const;     // next valid index or fence if none
  inline virtual int    pred(int idx) const;     // previous index or low - 1 if none

  inline virtual OrderedPoint*   first_pointer() const;   // pointer to first valid pos or 0
  inline virtual OrderedPoint*   last_pointer() const;    // pointer to first valid pos or 0
  inline virtual OrderedPoint*   succ(OrderedPoint*  p) const;     // next pointer or 0
  inline virtual OrderedPoint*   pred(OrderedPoint* p) const;     // previous pointer or 0

// modification

  inline virtual OrderedPoint*   grow_high ();      // return spot to add an element
  inline virtual OrderedPoint*   grow_low ();  

  inline virtual void   shrink_high ();    // logically delete top index
  inline virtual void   shrink_low ();     

  virtual void   clear(int lo);     // reset to empty ch with base = lo
  virtual void   cleardown(int hi); // reset to empty ch with top = hi
  void           re_index(int lo);  // re-index so lo is new low

// chunk traversal

  OrderedPointIChunk*     next() const;
  OrderedPointIChunk*     prev() const;

  void           link_to_prev(OrderedPointIChunk* prev);
  void           link_to_next(OrderedPointIChunk* next);
  void           unlink();

// state checks

  OrderedPoint*           invalidate();     // mark self as invalid; return data
                                   // for possible deletion

  virtual int    OK() const;             // representation invariant

  void   error(const char*) const;
  void   empty_error() const;
  void   full_error() const;
  void   index_error() const;
};

// OrderedPointPlex is a partly `abstract' class: few of the virtuals
// are implemented at the Plex level, only in the subclasses

class OrderedPointPlex
{
protected:      

  OrderedPointIChunk*       hd;          // a chunk holding the data
  int              lo;          // lowest  index
  int              fnc;         // highest index + 1
  int              csize;       // size of the chunk

  void             invalidate();              // mark so OK() is false
  void             del_chunk(OrderedPointIChunk*);        // delete a chunk

  OrderedPointIChunk*       tl() const;                // last chunk;
  int              one_chunk() const;         // true if hd == tl()

public:

// constructors, etc.

                    OrderedPointPlex();                  // no-op

  virtual           ~OrderedPointPlex();

  
// Access functions 
    
  virtual OrderedPoint&      operator [] (int idx) = 0; // access by index;
  virtual OrderedPoint&      operator () (Pix p) = 0;   // access by Pix;

  virtual OrderedPoint&      high_element () = 0;      // access high element
  virtual OrderedPoint&      low_element () = 0;       // access low element

// read-only versions for const Plexes

  virtual const OrderedPoint& operator [] (int idx) const = 0; // access by index;
  virtual const OrderedPoint& operator () (Pix p) const = 0;   // access by Pix;

  virtual const OrderedPoint& high_element () const = 0;      // access high element
  virtual const OrderedPoint& low_element () const = 0;       // access low element


// Index functions

  virtual int       valid (int idx) const = 0;      // idx is an OK index

  virtual int       low() const = 0;         // lowest index or fence if none
  virtual int       high() const = 0;        // highest index or low-1 if none

  int               ecnef() const;         // low limit index (low-1)
  int               fence() const;         // high limit index (high+1)

  virtual void      prev(int& idx) const= 0; // set idx to preceding index
                                          // caution: pred may be out of bounds

  virtual void      next(int& idx) const = 0;       // set to next index
                                          // caution: succ may be out of bounds

  virtual Pix       first() const = 0;        // Pix to low element or 0
  virtual Pix       last() const = 0;         // Pix to high element or 0
  virtual void      prev(Pix& pix) const = 0;  // preceding pix or 0
  virtual void      next(Pix& pix) const = 0;  // next pix or 0
  virtual int       owns(Pix p) const = 0;     // p is an OK Pix

// index<->Pix 

  virtual int       Pix_to_index(Pix p) const = 0;   // get index via Pix
  virtual Pix       index_to_Pix(int idx) const = 0; // Pix via index

// Growth

  virtual int       add_high(const OrderedPoint& elem) =0;// add new element at high end
                                                // return new high

  virtual int       add_low(const OrderedPoint& elem) = 0;   // add new low element,
                                                // return new low

// Shrinkage

  virtual int       del_high() = 0;           // remove the element at high end
                                          // return new high
  virtual int       del_low() = 0;        // delete low element, return new lo

                                          // caution: del_low/high
                                          // does not necessarily 
                                          // immediately call OrderedPoint::~OrderedPoint


// operations on multiple elements

  virtual void      fill(const OrderedPoint& x);          // set all elements = x
  virtual void      fill(const OrderedPoint& x, int from, int to); // fill from to to
  virtual void      clear() = 0;                // reset to zero-sized Plex
  virtual int       reset_low(int newlow); // change low index,return old
  virtual void      reverse();                   // reverse in-place
  virtual void      append(const OrderedPointPlex& a);    // concatenate a copy
  virtual void      prepend(const OrderedPointPlex& a);   // prepend a copy

// status

  virtual int       can_add_high() const = 0;
  virtual int       can_add_low() const = 0;
  
  int               length () const;       // number of slots

  int               empty () const;        // is the plex empty?
  virtual int       full() const = 0;      // it it full?

  int               chunk_size() const;    // report chunk size;

  virtual int       OK() const = 0;        // representation invariant

  void		    error(const char* msg) const;
  void		    index_error() const;
  void		    empty_error() const;
  void		    full_error() const;
};


// OrderedPointIChunk ops

inline int OrderedPointIChunk:: size() const
{
  return top - base;
}


inline int OrderedPointIChunk:: base_index() const
{
  return base;
}

inline  int OrderedPointIChunk:: low_index() const
{
  return low;
}

inline  int  OrderedPointIChunk:: fence_index() const
{
  return fence;
}

inline  int  OrderedPointIChunk:: top_index() const
{
  return top;
}

inline  OrderedPoint* OrderedPointIChunk:: pointer_to(int i) const
{
  return &(data[i-base]);
}

inline  int  OrderedPointIChunk:: index_of(const OrderedPoint* p) const
{
  return (p - data) / sizeof(OrderedPoint) + base;
}

inline  int  OrderedPointIChunk:: possible_index(int i) const
{
  return i >= base && i < top;
}

inline  int  OrderedPointIChunk:: possible_pointer(const OrderedPoint* p) const
{
  return p >= data && p < &(data[top-base]);
}

inline  int  OrderedPointIChunk:: actual_index(int i) const
{
  return i >= low && i < fence;
}

inline  int  OrderedPointIChunk:: actual_pointer(const OrderedPoint* p) const
{
  return p >= data && p < &(data[fence-base]);
}

inline  int  OrderedPointIChunk:: can_grow_high () const
{
  return fence < top;
}

inline  int  OrderedPointIChunk:: can_grow_low () const
{
  return base < low;
}

inline  OrderedPoint* OrderedPointIChunk:: invalidate()
{
  OrderedPoint* p = data;
  data = 0;
  return p;
}


inline OrderedPointIChunk* OrderedPointIChunk::prev() const
{
  return prv;
}

inline OrderedPointIChunk* OrderedPointIChunk::next() const
{
  return nxt;
}

inline void OrderedPointIChunk::link_to_prev(OrderedPointIChunk* prev)
{
  nxt = prev->nxt;
  prv = prev;
  nxt->prv = this;
  prv->nxt = this;
}

inline void OrderedPointIChunk::link_to_next(OrderedPointIChunk* next)
{
  prv = next->prv;
  nxt = next;
  nxt->prv = this;
  prv->nxt = this;
}

inline void OrderedPointIChunk::unlink()
{
  OrderedPointIChunk* n = nxt;
  OrderedPointIChunk* p = prv;
  n->prv = p;
  p->nxt = n;
  prv = nxt = this;
}

inline  int OrderedPointIChunk:: empty() const
{
  return low == fence;
}

inline  int  OrderedPointIChunk:: full() const
{
  return top - base == fence - low;
}

inline int OrderedPointIChunk:: first_index() const
{
  return (low == fence)? fence : low;
}

inline int OrderedPointIChunk:: last_index() const
{
  return (low == fence)? low - 1 : fence - 1;
}

inline  int  OrderedPointIChunk:: succ(int i) const
{
  return (i < low) ? low : i + 1;
}

inline  int  OrderedPointIChunk:: pred(int i) const
{
  return (i > fence) ? (fence - 1) : i - 1;
}

inline  int  OrderedPointIChunk:: valid_index(int i) const
{
  return i >= low && i < fence;
}

inline  int  OrderedPointIChunk:: valid_pointer(const OrderedPoint* p) const
{
  return p >= &(data[low - base]) && p < &(data[fence - base]);
}

inline  OrderedPoint* OrderedPointIChunk:: grow_high ()
{
  if (!can_grow_high()) full_error();
  return &(data[fence++ - base]);
}

inline  OrderedPoint* OrderedPointIChunk:: grow_low ()
{
  if (!can_grow_low()) full_error();
  return &(data[--low - base]);
}

inline  void OrderedPointIChunk:: shrink_high ()
{
  if (empty()) empty_error();
  --fence;
}

inline  void OrderedPointIChunk:: shrink_low ()
{
  if (empty()) empty_error();
  ++low;
}

inline OrderedPoint* OrderedPointIChunk::first_pointer() const
{
  return (low == fence)? 0 : &(data[low - base]);
}

inline OrderedPoint* OrderedPointIChunk::last_pointer() const
{
  return (low == fence)? 0 : &(data[fence - base - 1]);
}

inline OrderedPoint* OrderedPointIChunk::succ(OrderedPoint* p) const
{
  return ((p+1) <  &(data[low - base]) || (p+1) >= &(data[fence - base])) ? 
    0 : (p+1);
}

inline OrderedPoint* OrderedPointIChunk::pred(OrderedPoint* p) const
{
  return ((p-1) <  &(data[low - base]) || (p-1) >= &(data[fence - base])) ? 
    0 : (p-1);
}


// generic Plex operations

inline OrderedPointPlex::OrderedPointPlex() {}

inline int OrderedPointPlex::chunk_size() const
{
  return csize;
}

inline  int OrderedPointPlex::ecnef () const
{
  return lo - 1;
}


inline  int OrderedPointPlex::fence () const
{
  return fnc;
}

inline int OrderedPointPlex::length () const
{
  return fnc - lo;
}

inline  int OrderedPointPlex::empty () const
{
  return fnc == lo;
}

inline OrderedPointIChunk* OrderedPointPlex::tl() const
{
  return hd->prev();
}

inline int OrderedPointPlex::one_chunk() const
{
  return hd == hd->prev();
}

// minimum number of chunks to index

#ifndef MIN_NCHUNKS
#define MIN_NCHUNKS  16
#endif

class OrderedPointRPlex: public OrderedPointPlex
{
  int              base;        // base index of lowest chunk
  int              lch;         // index of lowest used chunk
  int              fch;         // 1 + index of highest used chunk
  int              maxch;       // max chunks in array
  OrderedPointIChunk**      chunks;      // array of chunks
  OrderedPointIChunk*       ch;          // cached chunk

  void             make_initial_chunks(int up = 1);

  void             cache(int idx) const;
  void             cache(const OrderedPoint* p) const;
  OrderedPoint*             dopred(const OrderedPoint* p) const;
  OrderedPoint*             dosucc(const OrderedPoint* p) const;

  inline void             set_cache(const OrderedPointIChunk* t) const; // logically, 
                                               // not physically const

public:
                   OrderedPointRPlex();                 // set low = 0;
                                               // fence = 0;
                                               // csize = default

                   OrderedPointRPlex(int ch_size);      // low = 0; 
                                               // fence = 0;
                                               // csize = ch_size

                   OrderedPointRPlex(int lo,            // low = lo; 
                            int ch_size);      // fence=lo
                                               // csize = ch_size

                   OrderedPointRPlex(int lo,            // low = lo
                            int hi,            // fence = hi+1
                            const OrderedPoint& initval,// fill with initval,
                            int ch_size = 0);  // csize= ch_size
                                               // or fence-lo if 0

                   OrderedPointRPlex(const OrderedPointRPlex&);

  inline                  ~OrderedPointRPlex();

  void             operator= (const OrderedPointRPlex&);

// virtuals

  inline OrderedPoint&             high_element ();
  inline OrderedPoint&             low_element ();

  inline const OrderedPoint&       high_element () const;
  inline const OrderedPoint&       low_element () const;

  inline Pix              first() const;
  inline Pix              last() const;
  inline void             prev(Pix& ptr) const;
  inline void             next(Pix& ptr) const;
  int              owns(Pix p) const;
  inline OrderedPoint&             operator () (Pix p);
  inline const OrderedPoint&       operator () (Pix p) const;

  inline int              low() const; 
  inline int              high() const;
  inline int              valid(int idx) const;
  inline void             prev(int& idx) const;
  inline void             next(int& x) const;
  inline OrderedPoint&             operator [] (int index);
  inline const OrderedPoint&       operator [] (int index) const;
    
  inline int              Pix_to_index(Pix p) const;
  inline Pix              index_to_Pix(int idx) const;    

  inline int              can_add_high() const;
  inline int              can_add_low() const;
  inline int              full() const;

  int              add_high(const OrderedPoint& elem);
  int              del_high ();
  int              add_low (const OrderedPoint& elem);
  int              del_low ();

  void             fill(const OrderedPoint& x);
  void             fill(const OrderedPoint& x, int from, int to);
  void             clear();
  void             reverse();

  int              reset_low(int newlow);
    
  int              OK () const; 
};


inline void OrderedPointRPlex::prev(int& idx) const
{
  --idx;
}

inline void OrderedPointRPlex::next(int& idx) const 
{
  ++idx;
}

inline  int OrderedPointRPlex::full ()  const
{
  return 0;
}

inline int OrderedPointRPlex::can_add_high() const
{
  return 1;
}

inline int OrderedPointRPlex::can_add_low() const
{
  return 1;
}

inline  int OrderedPointRPlex::valid (int idx) const
{
  return idx >= lo && idx < fnc;
}

inline int OrderedPointRPlex::low() const
{
  return lo;
}

inline int OrderedPointRPlex::high() const
{
  return fnc - 1;
}

inline void OrderedPointRPlex::set_cache(const OrderedPointIChunk* t) const
{
  ((OrderedPointRPlex*)(this))->ch = (OrderedPointIChunk*)t;
}

inline void OrderedPointRPlex::cache(int idx) const
{
  if (idx < lo || idx >= fnc) index_error();
  set_cache(chunks[(idx - base) / csize]);
}  

inline  OrderedPoint& OrderedPointRPlex::low_element () 
{
  cache(lo); return *(ch->pointer_to(lo));
}

inline  OrderedPoint& OrderedPointRPlex::high_element ()
{
  cache(fnc-1); return *(ch->pointer_to(fnc - 1));
}

inline const OrderedPoint& OrderedPointRPlex::low_element () const
{
  cache(lo); return *((const OrderedPoint*)(ch->pointer_to(lo)));
}

inline const OrderedPoint& OrderedPointRPlex::high_element () const
{
  cache(fnc-1); return *((const OrderedPoint*)(ch->pointer_to(fnc - 1)));
}

inline  int OrderedPointRPlex::Pix_to_index(Pix px) const
{
  OrderedPoint* p = (OrderedPoint*)px;
  if (!ch->actual_pointer(p)) cache(p);
  return ch->index_of(p);
}

inline  Pix OrderedPointRPlex::index_to_Pix(int idx) const
{
  if (!ch->actual_index(idx)) cache(idx);
  return (Pix)(ch->pointer_to(idx));
}

inline Pix OrderedPointRPlex::first() const
{
  return Pix(hd->OrderedPointIChunk::first_pointer());
}

inline Pix OrderedPointRPlex::last() const
{
  return Pix(tl()->OrderedPointIChunk::last_pointer());
}

inline void OrderedPointRPlex::prev(Pix& p) const
{
  Pix q = Pix(ch->OrderedPointIChunk::pred((OrderedPoint*)p));
  p = (q == 0)? Pix(dopred((OrderedPoint*)p)) : q;
}

inline void OrderedPointRPlex::next(Pix& p) const
{
  Pix q = Pix(ch->OrderedPointIChunk::succ((OrderedPoint*)p));
  p = (q == 0)? Pix(dosucc((OrderedPoint*)p)) : q;
}

inline OrderedPoint& OrderedPointRPlex:: operator () (Pix p)
{
  return *((OrderedPoint*)p);
}


inline OrderedPoint& OrderedPointRPlex:: operator [] (int idx)
{
  cache(idx); return *(ch->pointer_to(idx));
}

inline const OrderedPoint& OrderedPointRPlex:: operator () (Pix p) const
{
  return *((const OrderedPoint*)p);
}

inline const OrderedPoint& OrderedPointRPlex:: operator [] (int idx) const
{
  cache(idx); return *((const OrderedPoint*)(ch->pointer_to(idx)));
}

inline OrderedPointRPlex::~OrderedPointRPlex()
{
  delete[] chunks;
}

#endif
