//----------------------------------------------------------------------------------------
//	TMap.h
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//	Change History:
//----------------------------------------------------------------------------------------

#ifndef __TMap__
#define __TMap__

#include "SDSObject.h"
#include "TMap2.h"
#include "MML_Pipe.h"
#include "MML_Frame.h"

class TMap : public TMap2,  public SDSObject {

protected:

  float* fFData;
  int* fIData;
  int fWIndex;

public:
  //  enum Definitions in MML_Pipe.h

  static EInfoIndex kMapType;
  static EInfoIndex kMapClass;
  static EInfoIndex kRegionSet;
  static EInfoIndex kDataType;

	enum EDataType { kUndefined, kBytes, kShorts, kInts, kFloats };
	
	TMap( TMap& m ) : SDSObject(), TMap2(m) { fFData = NULL; fIData = NULL; fWIndex = -1; SetObjInfo(kDataType,kBytes); }
  TMap( Region2* region, int nbytes=1 ) : SDSObject(), TMap2(region,nbytes) { fFData = NULL; fIData = NULL; fWIndex = -1; SetObjInfo(kDataType,kBytes); }
  TMap( EMapClass c =  kUnknown, EMapType t = kUnDefined ) : SDSObject(), TMap2() { 
    fFData = NULL; fIData = NULL; SetObjInfo(kMapClass,c);  SetObjInfo(kMapType,t); SetObjInfo(kDataType,kBytes);  fWIndex = -1;
  }
  TMap( const Region2* region, EDataType dtype ) : SDSObject(), TMap2() { 
		fFData = NULL; fIData = NULL; fWIndex = -1; SetDataType(dtype);  setregion(*region);  Allocate();
	}
  
  virtual int32 SDWrite( const char* info=NULL, const char* units=NULL, const char* format=NULL );
  virtual int32 SDRead( int index=0, int32 group_ref=-1, char* info=NULL, char* units=NULL, char* format=NULL );
  int32 SDSWrite( int index ); 
  inline void ClearSDS() { 
#ifdef HAS_HDF
		SDendaccess(fSDS_id); fSDS_id = 0;
#endif 
	}
  
  void BroadcastData( int srcProc, char* info = NULL, int mergeMode = 0 );
//  void  CookieCutCov( MultiCoverage& cov, int layer_index = -1 );
  
  int ReadGrassMap(const char* name, int nbytes=0 );
  int readDepthFile( const CString& pathName, LayerConfig* lc  );
  int WriteGrassMap(const char* name);
  int WriteArcMap(const char* name);
  int WriteASCIIMap(const char* name);
  int ReadArcMap(const char* name);
  int ReadArcMap2( const char* filename, const char* templatename );
  int ReadCmeMap(const char* name, int nbytes);
  int ReadNetCDFMap( const char* filename, const char* varname );
  int WriteNetCDFMap( const char* filename, const char* varname );
  void ReadTextData( FILE* file, const char* name );
  void ReadTextData2( FILE* file, const char* name, int row_min, int row_max, int col_min, int col_max, int rows, int cols );
  int WriteCmeMap( const char* name, int time_index, float time_value, Bool last_write = False );  
  int Read(const char* mapSrc, const char* fileName, int nbytes=0, LayerConfig* lc = NULL, const char* format = NULL );
  int  TMap::Read( LayerConfig& lc, int nbytes ); 
  int ReadCategories( const char* mapSrc, const char* fileName ); 
  int Write( int index = -1, Bool lastWrite = False );

  virtual int32 SDFinalize();
  
  void SetDataType( EDataType dtype ) {
		SetObjInfo(kDataType,dtype);
		switch( dtype ) {
			case kBytes:
				fNBytes = 1;
			break;
			case kShorts:
				fNBytes = 2;
			break;
			case kInts:
				fNBytes = sizeof(int);
			break;
			case kFloats:
				fNBytes = sizeof(float);
			break;
		} 
	}
	
	virtual void Set( byte val ) { 
		switch( GetObjInfo(kDataType) ) {
 			case kBytes:
				ByteGrid::Set( val );
				break;
			default:
				Set( (float) val ); 
				break;
		}
	}
	
  void Set( float val ) {
		switch( GetObjInfo(kDataType) ) {
 			case kBytes:
				ByteGrid::Set( (byte)val );
			break;
			default:
				for( int ir = lwp.elem(0); ir<=upp.elem(0); ir++ ) {
					for( int ic= lwp.elem(1);  ic<=upp.elem(1); ic++ ) {
						 SetValue( ir, ic, (float) val );
					}
				}
		}
  }

  inline byte BValue( int ir, int ic, int ibyte ) const { 
		switch( GetObjInfo(kDataType) ) {
			case kInts: {
				byte* rv = (byte*) (fIData + bindex(ir,ic));
				return rv[ ibyte ];
			}  default:
				return ByteGrid::BValue( ir, ic, ibyte );
		}
  }
  
  virtual inline float Value( int ir, int ic ) const { 
		float fval = 0.0;
		if( fS[1] > 0.0 ) { fval = fS[1]*FltValue( ir, ic ) + fS[0]; } 
		else { fval = FltValue( ir, ic ) ; }
		return  fval ;
	}
	
  virtual inline float Value( Point2& p ) const { 
		return  ( fS[1] > 0.0 ) ? fS[1]*FltValue( p ) + fS[0] : FltValue( p ) ;
	}
	virtual byte* ByteData() { return (byte*)fData; }
	virtual short* ShortData() { return (short*)fData; }
	virtual int* IntData() { return fIData; }
	virtual float* FloatData() { return fFData; }
	virtual int DataSize() { return BSize(); }  // data size in bytes; 
 
	virtual inline float FltValue( int ir, int ic ) const { 
	  if(inside(ir,ic)) {
		switch( GetObjInfo(kDataType) ) {
			case kFloats: 	return *(fFData + bindex(ir,ic));
			case kInts:			return *(fIData + bindex(ir,ic));
			default: return TMap2::IValue(ir,ic);
		}
	  }
	  else return -2;
	} 
	virtual inline float FltValue( Point2& p ) const { 
	  if(inside(p)) {
		switch( GetObjInfo(kDataType) ) {
			case kFloats: 	return *(fFData + bindex(p(0),p(1)));
			case kInts: 		return (float) *(fIData + bindex(p(0),p(1)));
			default: 				return TMap2::IValue( p );
		}
	  }
	  else return -2;
	}
	virtual inline long int IntValue( int ir, int ic ) const { 
	  if(inside(ir,ic)) {
		switch( GetObjInfo(kDataType) ) {
			case kFloats: {
			  float fval = *(fFData + bindex(ir,ic));
			  if( fval == FLT_MAX ) { return -999; }
			  else { return (int) ( fS[1]*fval + fS[0] ); }
			}
			case kInts: 		return *(fIData + bindex(ir,ic));
			default: 	return ByteGrid::IValue( ir, ic );
		}
	  }
	  else return -2;
	}
	virtual inline long int IntValue( Point2& p) const { 
	  if(inside(p)) {
		switch( GetObjInfo(kDataType) ) {
			case kFloats: 	    return (int) ( fS[1]*( *(fFData + bindex(p(0),p(1)) ) ) + fS[0] ); 
			case kInts: 		return *(fIData + bindex(p(0),p(1)));
			default: 	return ByteGrid::IValue( p );
		}
	  }
	  else return -2;
	} 
	virtual inline void SetValue( int ir, int ic, unsigned long ival ) { 
		switch( GetObjInfo(kDataType) ) {
			case kFloats: 	*(fFData + bindex(ir,ic)) = ival; break;
			case kInts: 		*(fIData + bindex(ir,ic)) = ival; break;
			default: 				ByteGrid::SetValue( ir, ic, ival );
		}
	}
	inline void SetValue( int ir, int ic, float fval ) { 
		if( !inside(ir,ic) ) { gFatal( format("Illegal grid coords in TMap::SetValue: %d %d",ir,ic) ); }
		switch( GetObjInfo(kDataType) ) {
			case kFloats: 	*(fFData + bindex(ir,ic)) = fval; break;
			case kInts: 		*(fIData + bindex(ir,ic)) = (int) floor( fval + 0.5 ); break;
			default: 				ByteGrid::SetValue( ir, ic, (unsigned long)fval );
		}
  }

  virtual inline byte* Data() const { 	
		switch( GetObjInfo(kDataType) ) {
			case kBytes: case kShorts: return fData; 
			case kInts: return (byte*)fIData;
			case kFloats: return (byte*)fFData;
		}
	}
	
  inline int Import( const char* pathName, const char* fileName, int format ) {
    CString s(pathName); (s+= '/') += fileName; 
    return ReadM2File(s);
  }

  virtual int TextDump( CPathString& path, int index, int nMaps, const char* info=NULL, const char* units=NULL, const char* format=NULL );
#ifdef HAS_X
  virtual int AnimateData( XAnim* aAnim, int index=0, int nData=1, int mag=1, char* info=NULL); 
#endif
  virtual int DumpMap( CPathString& path, int index, int nMaps, const char* info );
  virtual int DumpHDF( CPathString& path, int index, int nMaps, const char* info );
  virtual int DumpHDF5( CPathString& path, const char* info );
  unsigned short* getCompressedData( float& fmax, float& fmin);

  int32 ExportSetAttributes( int32 sds_id  );
  void Dump( FILE* oFile ){ TMap2::Dump(oFile); }
  void Dump( FILE* oFile, float background, char* label = "" ) const;

	virtual void Allocate() ;  
	virtual void ReAlloc(const Region2& r ) ;
  virtual void SetValueWithRescale( const Point2* point, float fval, Bool directScale = FALSE ) ;
  void ComposeDataPtrs( ); 
};


#endif
