#include "TMap2.h"

extern "C" {
#include <string.h>
#include <unistd.h>
}

const int kMaxElem = 35;

#ifndef SEEK_SET
#define SEEK_SET 0
#endif

#ifdef USE_MPI
MPI_Comm TMap2::fComm = 0;
unsigned int TMap2::fMapCount = 0;
#endif

//----------------------------------------------------------------------------------------
//						TMap2
//----------------------------------------------------------------------------------------

TMap2::TMap2( char* mapName, int rscale, int cscale  ) {
  fS[1] = 1.0;
  fS[0] = 0.0;
  ClearBounds();
  if(mapName) {
    fMapName = mapName;
    ReadM2File( NULL, rscale, cscale ); 
  }
#ifdef USE_MPI
  if( fComm == 0 )  { 			
		if( Env::GetComm(&fComm) ) { 
			gFatal("Error in TMap2"); 
		}   
	}
  fMapIndex = fMapCount++;
#endif
}
TMap2::TMap2( Region2* region, int nbytes  ) {
	fNBytes = nbytes;
	if(region) ReAlloc(*region);
  fMaxVal = Util::pow2(fNBytes*8)-2;    
  fS[1] = 1.0;
  fS[0] = 0.0;
	ClearBounds();
#ifdef USE_MPI
  if( fComm == 0 )  { 			 
		if( Env::GetComm(&fComm) ) { 
			gFatal("Error in TMap2"); 
		}   
	}
  fMapIndex = fMapCount++;
#endif
}
TMap2::TMap2( TMap2& m  ) {
	fNBytes = m.fNBytes;
	ReAlloc(m);
  fMaxVal = Util::pow2(fNBytes*8)-2;    
  fS[1] = 1.0;
  fS[0] = 0.0;
	ClearBounds();
	memcpy( fData, m.fData, BSize() );     
#ifdef USE_MPI
  if( fComm == 0 )  { 			 
		if( Env::GetComm(&fComm) ) { 
			gFatal("Error in TMap2"); 
		}   
	}
  fMapIndex = fMapCount++;
#endif
}

void TMap2::ReadTextData( FILE* file, const char* name ) {
	int idata, tst, nelem = NElements(), offset;
	Allocate();
	int ir, ic;
	for( int i=0; i< nelem; i++ ) {
		Util::skip_white(file);
		tst = fscanf(file,"%d",&idata);
		if( tst < 1 ) { sprintf(gMsgStr,"Format error in Map data: %s", name); gFatal(); }
		byte* mPtr = fData + i*fNBytes;
		Util::enc_Nb( &mPtr, (unsigned long)idata, fNBytes);  	
	}
}

int TMap2::ReadM2Categories( const char* pathName ) {

  if( gDebug  && (gIProc == 0) && pathName ) { 
    sprintf( gMsgStr, "Reading Categories from %s", pathName );  
    gPrintScreen(); 
    gPrintLog(); 
  }   
  if(pathName) fMapName = pathName;
	int index= -1;
  if( gIProc == 0 || gGlobalFileSystem ) { 
		FILE* file = fopen(fMapName,"r");
		if(file==NULL) { 
			sprintf(gMsgStr,"(P%d)Unable to open header file %s.\n",gIProc,fMapName.chars()); 
			gPrintLog();
			gFatal(gMsgStr,True);
		}
		if(gDebug) { 
			sprintf(gMsgStr,"(P%d)Successfully opened header file %s.\n",gIProc,fMapName.chars()); 
			gPrintLog();
		}
		int test = Util::scan_forward(file,"LABELS");
		if( test ) {
			int ch; 
			while( Util::scan_forward(file,"VALUE=") ) {
				fscanf(file,"%d",&index);
				CString* label = new CString;
				while( ( ch=fgetc(file) ) != '"' )  {;}
				while( ( ch=fgetc(file) ) != '"' )  { (*label) += (char)toupper(ch); }
				fCatMap[index] = (Pix) label;
			}
		}
	}
	return index;
}


int TMap2::ReadM2File( const char* pathName, int rscale, int cscale ) {

  if( gDebug  && (gIProc == 0) && pathName ) { 
    sprintf( gMsgStr, "Reading %s", pathName );  
    gPrintScreen(); 
    gPrintLog(); 
  }   
  if(pathName) fMapName = pathName;

  if( gIProc == 0 || gGlobalFileSystem ) { 
    setincrement(rscale,cscale);
    sprintf(gMsgStr,"Reading PPM file %s, scale(%d,%d)",fMapName.chars(),rscale,cscale); gPrintLog();
    CString bFileName;
		int remote_data = ReadHeader(bFileName);
		if(remote_data) { 
			FILE* bFile = fopen(bFileName,"rb");
			if( bFile == NULL ) { sprintf(gMsgStr,"Can't Open Mapfile: %s",bFileName.chars()); gFatal(); }
			if(fReadDataFormat) {
				ReadTextData(bFile,pathName);
			}
			else { 
				int gsize = NElements();
				if(fData) delete[] fData;
				fData = new byte[gsize*fNBytes];
				int nbytes = fread( fData, fNBytes, gsize, bFile );
				if(gDebug) { 
					sprintf(gMsgStr," %d of %d items Read for %s, size = %x bytes ",nbytes,gsize,fMapName.chars(),fNBytes); 
					gPrintLog(); 
				}
			}
			fclose(bFile);
		}
  }
  if( !gGlobalFileSystem ) BroadcastData(0);
  if( gDebug )  Dump( Env::LogFile() );
  return fNBytes;
}

int TMap2::ReadM2File2( const char* pathName, const char* templateName, int rscale, int cscale ) {

  if( gDebug  && (gIProc == 0) && pathName ) { 
    sprintf( gMsgStr, "Reading %s", pathName );  
    gPrintScreen(); 
    gPrintLog(); 
  }  
  if( templateName ) {
	fMapName = templateName; 
	Clear();
  }
  
  if( gIProc == 0 || gGlobalFileSystem ) { 
    setincrement(rscale,cscale);
    sprintf(gMsgStr,"Reading PPM file %s, scale(%d,%d)",fMapName.chars(),rscale,cscale); gPrintLog();
    CString bFileName;
    int rows, cols, bsize = 1;
	int remote_data = GetHeaderInfo( bFileName, rows, cols, bsize );
	if(remote_data == 0 ) { gFatal( format( "Must have map template data external to header: %s ", fMapName.chars()) ); }
	FILE* bFile = fopen(bFileName,"rb");
	if( bFile == NULL ) { sprintf(gMsgStr,"Can't Open Mapfile: %s",bFileName.chars()); gFatal(); }
	byte *dbuff = new byte[bsize*cols];
	int row_max=0, row_min=rows, col_max=0, col_min=cols, bcnt = 0;
	int gsize = rows*cols;
	
	for (int ir = 0; ir < rows; ir++) {
		int bytes_read = fread( dbuff, bsize, cols, bFile );
		bcnt += bytes_read;
		byte* bptr = dbuff;
		for (int ic = 0; ic < cols; ic++) {
			/* set start_point to end_point, which is buf */
			int val = (int) Util::dec_Nb( &bptr, bsize );
			if( val > 0 ) {
			  if( ir < row_min ) row_min = ir;
			  if( ir > row_max ) row_max = ir;
			  if( ic < col_min ) col_min = ic;
			  if( ic > col_max ) col_max = ic;
			}
		}
	}	
	if(gDebug) { 
		sprintf(gMsgStr,"Read %d bytes for %s, size = %x bytes ",bcnt,fMapName.chars(),bsize); 
		gPrintLog(); 
	}
	fclose(bFile);
	delete[] dbuff;
	
	int map_cols =  col_max-col_min+1;
	int map_rows = row_max-row_min+1;
	
	fMapName = pathName;
	remote_data = ReadHeader( bFileName );
	if(remote_data == 0 ) { gFatal( format( "Must have map data external to header for map template input mode: %s ", fMapName.chars()) ); }
	if( gsize != nelem() ) {
	   gFatal( format("Map size discrepancy: template %s(%d,%d) vs. Map %s(%d,%d)", 
				  templateName, rows, cols, pathName, extents(0), extents(1) ) ); 
	}
	
	SetDimensions( map_rows, map_cols );
	bFile = fopen(bFileName,"rb");
	if( bFile == NULL ) { sprintf(gMsgStr,"Can't Open Mapfile: %s",bFileName.chars()); gFatal(); }

	if(fReadDataFormat) {
		ReadTextData2( bFile, bFileName, row_min, row_max, col_min, col_max, rows, cols );
	} else { 
	  Allocate();
	  dbuff = new byte[ fNBytes*cols ];
	  for (int ir = 0; ir < rows; ir++) {
		  /* reads a row into 'dbuff' */
		  fread( dbuff, fNBytes, cols, bFile );
		  if( (ir >= row_min) && (ir <= row_max) ) {
			int offset = map_cols*(ir-row_min);
			memcpy( fData+offset*fNBytes, dbuff+col_min*fNBytes, map_cols*fNBytes );
		  }
	  }	
	}
	fclose(bFile);
	delete[] dbuff;
  }
  if( !gGlobalFileSystem ) BroadcastData(0);
  if( gDebug )  Dump( Env::LogFile() );
  return fNBytes;
}

inline int get_next_char(FILE* infile) {
		int test, incomment = 0;
		while( ( test = fgetc(infile) ) != EOF ) {
			if( test == '#' ) { incomment = 1; }
			if( incomment ) { 
				if( test == '\n' ) { incomment = 0; }
			} else {
				if( !isspace(test) ) break;
			}				
		}
		return test;
}

inline int get_next_int(FILE* infile) {
		int test, cnt=0, incomment = 0; char tmp[64];
		while( ( test = fgetc(infile) ) != EOF ) {
			if( test == '#' ) { incomment = 1; }
			if( incomment ) { 
				if( test == '\n' ) { incomment = 0; }
			} else {
				if( isdigit(test) ) break;
			}				
		}
		do { tmp[cnt++] = (char) test; }
		while( isdigit( test = fgetc(infile) ) );
		tmp[cnt++] = (char) 0; 
		test = atoi(tmp);
		return test;
}

int TMap2::ReadPPMFile( const char* pathName, int rscale, int cscale ) {

  if( gDebug  && (gIProc == 0) && pathName ) { 
    sprintf( gMsgStr, "Reading %s", pathName );  
    gPrintScreen(); 
    gPrintLog(); 
  }   
  if(pathName) fMapName = pathName;
  
  FILE* infile = fopen(fMapName,"r");
  if(infile==NULL) { 
    sprintf(gMsgStr,"(P%d)Unable to open ppm file %s.\n",gIProc,fMapName.chars()); 
    gPrintLog();
    gFatal(gMsgStr,True);
  }
  if(gDebug) { 
    sprintf(gMsgStr,"(P%d)Successfully opened ppm file %s.\n",gIProc,fMapName.chars()); 
    gPrintLog();
  } 

  if( gIProc == 0 || gGlobalFileSystem ) { 
    setincrement(rscale,cscale);
    sprintf(gMsgStr,"Reading Map file %s, scale(%d,%d)",fMapName.chars(),rscale,cscale); gPrintLog();
		int test, incomment = 0, byteDataFormat = 1;		
		char c0 = (char) get_next_char(infile);
		if( c0 != 'P' ) { sprintf(gMsgStr,"Map %s not in PPM format",fMapName.chars()); gFatal(); }
		char c1 = fgetc(infile); 
		if( c1 == '3' ) byteDataFormat = 0;
		else if( c1 == '6' ) byteDataFormat = 1;
		else  { sprintf(gMsgStr,"Map %s not in unknown PPM format: %x",fMapName.chars(),c1); gFatal(); }
		int width = get_next_int(infile);
		int height = get_next_int(infile);
		int maxval = get_next_int(infile);
		fNBytes = 3;
		fMaxVal = Util::pow2(fNBytes*8)-2;   
		Region2 r(0,0,width,height);
		setregion(r);
		int gsize = NElements();
		if(fData) delete[] fData;
		int bsize = gsize*fNBytes;
		fData = new byte[bsize];
		int nbytes = 0;
		if( byteDataFormat ) {
			nbytes = fread( fData, fNBytes, gsize, infile );
		} else {
			for( int j=0; j<bsize; j++ ) {
				fData[j] = (byte) get_next_int(infile);
			}
		}
		if(gDebug) { 
			sprintf(gMsgStr," %d of %d items Read for %s, size = %x bytes ",nbytes,gsize,fMapName.chars(),fNBytes); 
			gPrintLog(); 
		}
		fclose(infile);
	}
  if( !gGlobalFileSystem ) BroadcastData(0);
  if( gDebug )  Dump( Env::LogFile() );
  return fNBytes;
}

int TMap2::ReadERDASMap ( const char* filename, int background  ) {
	FILE	*fp;
	int	i,j;
	short	value;
	short	*ptr;
	short 	ibands;
	short   ipack;
	short   val;
	short	swapped_val;
	int	count,size;
	int	count_bck=0, count_intbck=0;
	short	min, max;
	int num_rows, num_cols;

	if ((fp = fopen (filename,"rb")) == NULL) {
		sprintf ( gMsgStr, "\nERROR reading file: %s\n",filename); gPrintErr();
		return 0;
	}

/*
 *  Read erdas image header
 */
	fread (&fImageHd,sizeof(struct image_header),1,fp);

/*
 *  For some reason, ERDAS decided to swap the order of bytes.
 *  Unswap them to get number of bands in the input image.
 */

#ifdef __GO32__
	ibands = fImageHd.nbands;
#else
//	swab ((const char*)&fImageHd.nbands,(char*)&ibands,2);
//  gcc 2.96 under Linux can't find swab.  
#endif

	if (ibands != 1) {
	   printf ("\nERROR!  This image has more than 1 band!");
	   printf ("\n  This program can only read single band images!");
	   exit(-1);
	}

/*
 *  Again, swap the bytes to get the "pack type" (either 4, 8, or 16-bit)
 */

#ifdef __GO32__
	ipack = fImageHd.pack_type;
#else
//	swab ((const char*)&fImageHd.pack_type,(char*)&ipack,2);
#endif

	if (ipack < 0 || ipack > 3) {
	   sprintf ( gMsgStr, "\nERROR! Invalid pack type: %d",ipack); gFatal();
	}

/*
 *  Again, swap the bytes to get the rows & cols
 */


#ifdef __GO32__
	num_rows = fImageHd.rrows;
	num_cols = fImageHd.rcols;
#else
//	swab ((const char*)&fImageHd.rrows,(char*)&num_rows,4);
//	swab ((const char*)&fImageHd.rcols,(char*)&num_cols,4);
#endif

	fNBytes = 2;
	fMaxVal = Util::pow2(fNBytes*8)-2;   
  Region2 r(0,0,num_rows-1,num_cols-1);
  if( !empty() ) r *= (*this);
  setregion(r);
	Allocate();
	ptr = (short*) fData;
	
	min = 9999;
	max = -9999;
	int land_border = FALSE;
	int MAX_CLASSES = -9999;
	int total_size = 0;
	size = num_rows * num_cols;
	
/*
 *  Read 8 bit data, no swapping required.  Find min, max data values.
 */
	if (ipack == 0) {
	   sprintf ( gMsgStr, "\nReading 8 bit ERDAS image ....\n"); gPrintLog();

	   for (i=0; i < size; i++) {
	      *ptr = (short) fgetc(fp);

	      if (*ptr < 0 && *ptr != background && *ptr != -background)  land_border = TRUE;

	      if (*ptr != background && *ptr != -background) {
 	         if (*ptr < min && *ptr >= 0) min = *ptr;
	         if (*ptr > max) max = *ptr;
	         if (*ptr >= 0) total_size ++;

					 if (*ptr >= 0 && *ptr > MAX_CLASSES) MAX_CLASSES = *ptr;
					 if (*ptr < 0 && -(*ptr) > MAX_CLASSES) MAX_CLASSES = -(*ptr);
	      }

	      if (*ptr == -background) {
					 count_bck ++;
					 *ptr = -999;
	      }
	      if (*ptr == background) {
					 count_intbck ++;
	         *ptr = -990;
	      }

	      ptr ++;
	   }
	}

/*
 *  Read 4-bit data packed into 1 byte. No swapping required.
 */
	if (ipack == 1) {
	   sprintf ( gMsgStr, "\nReading 4 bit ERDAS image ...\n"); gPrintLog();

	   count = 0;
	   for (i=0; i < size; i+=2) {
	      for (j=0; j < 2; j++) {
	         if (j == 0) {
	            value = (short) fgetc(fp);
                    *ptr = value & 15;
	         }
					 else {
							*ptr = value >> 4;
					 }
	         if (*ptr < 0 && *ptr != background && *ptr != -background)    land_border = TRUE;

					 if (*ptr != background && *ptr != -background) {
							if (*ptr < min && *ptr >= 0) min = *ptr;
							if (*ptr > max) max = *ptr;
							if (*ptr >= 0) total_size ++;
							if (*ptr >= 0 && *ptr > MAX_CLASSES) MAX_CLASSES = *ptr;
							if (*ptr < 0 && -(*ptr) > MAX_CLASSES) MAX_CLASSES = -(*ptr);
					 }
					 if (*ptr == -background) {
						 count_bck ++;
						 *ptr = -999;
					 }
					 if (*ptr == background) {
						 count_intbck ++;
						 *ptr = -990;
					 }

	         ptr ++;
	         count ++;
	         if (count >= size) break;
	      }
	   }
	}

/*
 *  Read 16-bit data; swapping required.
 */
	if (ipack == 2) {
	   sprintf ( gMsgStr, "\nReading 16 bit ERDAS image ...\n"); gPrintLog();

	   fread (fData,sizeof(short),size,fp);

	   for (i=0; i < size; i++) {
	      swapped_val = *ptr;
#ifdef __GO32__
	      val = swapped_val;
#else
//	      swab((const char*)&swapped_val,(char*)&val,2);
#endif
	      *ptr = val;

	      if (*ptr < 0 && *ptr != background && *ptr != -background) land_border = TRUE;
	      if (*ptr != background && *ptr != -background) {
	         if (*ptr < min && *ptr >= 0) min = *ptr;
	         if (*ptr > max) max = *ptr;
	         if (*ptr >= 0) total_size ++;
					 if (*ptr >= 0 && *ptr > MAX_CLASSES)  MAX_CLASSES = *ptr;
					 if (*ptr < 0 && -(*ptr) > MAX_CLASSES) MAX_CLASSES = -(*ptr);
	      }

	      if (*ptr == -background) {
				 count_bck ++;
				 *ptr = -999;
	      }
	      if (*ptr == background) {
				 count_intbck ++;
				 *ptr = -990;
	      }
	      ptr ++;
	   }
	}

	fclose (fp);
	MAX_CLASSES ++;

	int bcode = 0;
	if (count_bck > 0) {
		 sprintf ( gMsgStr, "\n... %d cells of background exterior to the landscape found", count_bck); gPrintLog();
	   bcode ++;
	}
	if (count_intbck > 0) {
 	   sprintf ( gMsgStr,"\n... %d cells of background interior to the landscape found", count_intbck); gPrintLog();
	   bcode ++;
	}
	if (count_bck == 0 && count_intbck == 0) {
	   sprintf ( gMsgStr,"\n... landscape does not contain background"); gPrintLog();
	}

}

int  TMap2::ReadArcInfoSVF( const char	*filename, int background ) {
	FILE	*fp;	
	int	i,j,k;
	int	count_intbck,count_bck;
	short	npairs,value;
	short	*ptr;
	char	rowflag;
	unsigned char repeat;
	short  min, max;
	int num_rows;
	int num_cols;
	
	if ((fp = fopen(filename,"rb")) == NULL) {
		sprintf ( gMsgStr, "\nERROR reading file: %s\n",filename); gPrintErr();
		return 0;
	}

/*
 *  Read the Arc/Info SVF file (compressed format):
 *  	Record 1 contains the number of rows and columns in the image.
 *	Records 2-n contain the image image in the following format.
 *		rowstart flag (-1) - indicates the start of an image
 *			row [char]
 *		npairs  - the number of pairs of (repeat,value) for 
 *		 	this row [short]
 *		repeat value - number of times the next value is to be
 *			repeated [char]
 *		image value [short]
 */
 
	fread (&num_rows,sizeof(short),1,fp);
	fread (&num_cols,sizeof(short),1,fp);

	fNBytes = 2;
	fMaxVal = Util::pow2(fNBytes*8)-2;   
  Region2 r(0,0,num_rows-1,num_cols-1);
  if( !empty() ) r *= (*this);
  setregion(r);
	Allocate();
	ptr = (short*) fData;

/*
 *  total_size is the number of cells inside the landscape (this
 *  does not include interior background or landscape border cells).
 */
	min = 9999;
	max = -9999;
	int total_size = 0;
	int land_border = FALSE;
	int MAX_CLASSES = -9999;
	count_bck = count_intbck = 0;

	for (i=0; i < num_rows; i++) {	
		fread(&rowflag,sizeof(char),1,fp);
		if (rowflag != -1 ) {
		   printf("\nROWFLAG not -1:  %d",rowflag);
		   printf ("\nFile [%s] probably not an SVF file\n",filename);
		   exit(-1);
		}
		fread (&npairs,sizeof(short),1,fp);
		for (j=0; j < npairs; j++) {
		   fread (&repeat,sizeof(char),1,fp);
		   fread (&value,sizeof(short),1,fp);

/*
 *  If there are negative cell values that are not background,
 *  then this image includes a landscape border.  Set a flag.
 */
	 	   if (value < 0 && value != background && value != -background) land_border = TRUE;

/*
 *  Keep track of the minimum and maximum class value inside the
 *  landscape.  
 */
		   if (value != background && value != -background) {
		      if (value < min && value >= 0) min = value;
		      if (value > max) max = value;

/*
 *  If the image includes a landscape border, for contagion we
 *  need to know the number of different classes in the landscape
 *  including those in the landsape border.
 */
				 if (value >= 0 && value > MAX_CLASSES) MAX_CLASSES = value;
				 if (value < 0 && -value > MAX_CLASSES) MAX_CLASSES = -value;
			}

/*
 *  The value for background cells was input by the user.  Set those 
 *  background cells interior to the landscape to -990.  Set those 
 *  background cells in the landscape border, or outside the area of 
 *  interest to -999.  This makes life easier in several other routines.
 */
		   if (value == -background) {
		      count_bck += repeat;
		      value = -999;
		   }

		   if (value == background) {
		      count_intbck += repeat;
		      value = -990;
 		   }
		   if (value >= 0) total_size += repeat;


		   for (k=0; k < repeat; k++) 
			*ptr++ = value;

		}
	}
	fclose (fp);
	MAX_CLASSES ++;

	int bcode = 0;
	if (count_bck > 0) {
		 sprintf ( gMsgStr, "\n... %d cells of background exterior to the landscape found", count_bck); gPrintLog();
	   bcode ++;
	}
	if (count_intbck > 0) {
 	   sprintf ( gMsgStr,"\n... %d cells of background interior to the landscape found", count_intbck); gPrintLog();
	   bcode ++;
	}
	if (count_bck == 0 && count_intbck == 0) {
	   sprintf ( gMsgStr,"\n... landscape does not contain background"); gPrintLog();
	}

	return 1;
}

static short get_image(FILE * imagein, int f_type, int d_type, int& count) {
    int error;
    static short i;
    union byte_int {
      short i;
      unsigned char b;
    };
    static union byte_int both;

    switch (f_type) {
    case 1: {			// ASCII grid file 
	  error = fscanf(imagein,"%hd\n",&both.i);
	  break;
      }
	
      case 2:  {		// Binary grid file 
	  error = fread(&both.i,d_type,1,imagein);
	  break;
      }
	
      case 3: {			// Packed binary grid file 
	  while (count == 0) {
	      error = fread(&count,2,1,imagein);
	      if (error != 1) break;
	      error = fread(&both.i,d_type,1,imagein);
	      if (error != 1) break;
	  }
	  
	  count--;
	  break;
      }
    }

    if (d_type == 1) i = both.b;
    else
#ifdef __GO32__
      i = both.i;
#else
 //     swab((const char*)&both.i,(char*)&i,2);
#endif

    if (error != 1) {
      count = 0;
      return(-1);
    }
    else
      return(i);
}

int TMap2::ReadIDRISIMap( const char* filename, int background ) {
	int	i,j,f_type,d_type;
	short	*ptr;
	FILE	*fp;
	static char refsystem[20],refunits[20],poserr[20],resolut[20];
	static float unitdist, min_x, max_x, min_y, max_y;
	char  record[100], *token, *open_type;
	CString wholename;
	int num_cols, num_rows;
	int count_bck = 0;
	int count_intbck = 0;
	int count = 0;

	wholename = filename;
	wholename += ".doc";
	if ((fp = fopen(wholename,"r")) == NULL) {
		sprintf ( gMsgStr, "\nERROR reading file: %s\n",wholename.chars()); gPrintErr();
		return 0;
	}
	
	// read the input parameters from the IDRISI file 
	while(fgets(record,100,fp)) {
	  token = strtok(record,":");
	  if (!strncmp(token,"data type",9)) {
	    token = strtok(NULL," \t\n\r");
	    if (!strcmp(token,"byte")) 
	      d_type = 1;
	    else if (!strcmp(token,"integer")) 
	      d_type = 2;
	    else {
	      printf("\nERROR bad input data type %s\n",token);
	      exit (-1);
	    }
	  }
		/* Number of columns */
		else if (!strncmp(token,"columns",7)) 
			num_cols = atoi(strtok(NULL,"\n"));
		/* rows ... */
		else if (!strncmp(token,"rows",4)) 
			num_rows = atoi(strtok(NULL,"\n"));
	  // file format ... 
	  else if (!strncmp(token,"file type",9)) {
	    token = strtok( NULL, " \t\n\r");
	    if (!strcmp(token,"ascii")) 
	      f_type = 1;
	    else if (!strcmp(token,"binary")) 
	      f_type = 2;
	    else if (!strcmp(token,"packed")) 
	      f_type = 3;
	    else {
	      printf("\nERROR bad input data type %s\n",token);
	      exit (-1);
	    }
	  }
	  else if (!strncmp(token,"ref. system",11)) {
	    token = strtok(NULL,"\n\r");
	    strncpy(refsystem,token,19);
	  }
	  else if (!strncmp(token,"ref. units",10)) {
	    token = strtok(NULL,"\n\r");
	    strncpy(refunits,token,19);
	  }
	  else if (!strncmp(token,"unit dist.",10))  {
	    token = strtok(NULL,"\n\r");
	    sscanf(token,"%f",&unitdist);
	  }
	  else if (!strncmp(token,"min. X",6))  {
	    token = strtok(NULL,"\n\r");
	    sscanf(token,"%f",&min_x);
	  }
	  else if (!strncmp(token,"max. X",6))  {
	    token = strtok(NULL,"\n\r");
	    sscanf(token,"%f",&max_x);
	  }
	  else if (!strncmp(token,"min. Y",6))  {
	    token = strtok(NULL,"\n\r");
	    sscanf(token,"%f",&min_y);
	  }
	  else if (!strncmp(token,"max. Y",6))  {
	    token = strtok(NULL,"\n\r");
	    sscanf(token,"%f",&max_y);
	  }
	  else if (!strncmp(token,"pos'n error",11)) {
	    token = strtok(NULL,"\n\r");
	    strncpy(poserr,token,19);
	  }
	  else if (!strncmp(token,"resolution",10)) {
	    token = strtok(NULL,"\n\r");
	    strncpy(resolut,token,19);
	  }
	}
      	    
	fclose(fp);

  fNBytes = 2;
	fMaxVal = Util::pow2(fNBytes*8)-2;   
  Region2 r(0,0,num_rows-1,num_cols-1);
  if( !empty() ) r *= (*this);
  setregion(r);
#ifdef HAS_GRASS_4_2
  SetGeoRegion ( num_rows, num_cols,  min_x, max_x, min_y, max_y ); 
#endif
  Allocate();

	int min = 9999;
	int max = -9999;
	int MAX_CLASSES = -9999;
	int land_border = FALSE;
	ptr = (short*)fData;

	wholename = filename;
	wholename += ".img";
	if (f_type == 1) open_type = "r";
	else open_type = "rb";
	if ((fp = fopen(wholename,open_type)) == NULL) {
		sprintf ( gMsgStr, "\nERROR reading file: %s\n",wholename.chars()); gPrintErr();
		return 0;
	}

//  total_size is the number of cells inside the landscape (non-background and
//  non-landscape border cells).

	int total_size = 0;
	
	for (i=0; i < num_rows; i++) {
  	   for (j=0; j < num_cols; j++) {
	        
	        *ptr = get_image(fp,f_type,d_type,count);


			//  If there are class values < 0 and not background, then this
			//  image includes a landscape border.  Set a flag.

					if (*ptr < 0 && *ptr != background && *ptr != -background)  land_border = TRUE;


			//  Find the minimum and maximum class values (only consider those
			//  classes inside the landscape). 

					if (*ptr != background && *ptr != -background) {
							 if (*ptr < min && *ptr >= 0) min = *ptr;
							 if (*ptr > max) max = *ptr;
							 if (*ptr >= 0) total_size ++;



			//   If this image has a landscape border, for contagion the total
			//  number of classes is needed including those class types in the
			//  landscape border.

								if (*ptr >= 0 && *ptr > MAX_CLASSES)  MAX_CLASSES = *ptr;
								if (*ptr < 0 && -(*ptr) > MAX_CLASSES)  MAX_CLASSES = -(*ptr);
					}


			//  The value for background cells was input by the user.  Change all
			//  background cells interior to the landscape to -990.  Change those
			//  in the landscape border and outside the area of interest to -999.
			//  This makes life easier in other routines.

					if (*ptr == -background) {
						 count_bck ++;
						 *ptr = -999;
					}
					if (*ptr == background) {
						 count_intbck ++;
						 *ptr = -990;
					}

					ptr++;
	      }
	 }

	fclose (fp);
	MAX_CLASSES ++;

	int bcode = 0;
	if (count_bck > 0) {
		 sprintf ( gMsgStr, "\n... %d cells of background exterior to the landscape found", count_bck); gPrintLog();
	   bcode ++;
	}
	if (count_intbck > 0) {
 	   sprintf ( gMsgStr,"\n... %d cells of background interior to the landscape found", count_intbck); gPrintLog();
	   bcode ++;
	}
	if (count_bck == 0 && count_intbck == 0) {
	   sprintf ( gMsgStr,"\n... landscape does not contain background"); gPrintLog();
	}
}


int  TMap2::GetHeaderInfo(CString& bFileName, int& rows, int& cols, int& size )
{
  int test, r0=0, c0=0, remote_data=0;

  // Should determine MapII Interchange format for setting origon: (r0,c0).
       	
  FILE* file = fopen(fMapName,"r");
  if(file==NULL) { 
    sprintf(gMsgStr,"(P%d)Unable to open header file %s.\n",gIProc,fMapName.chars()); 
    gPrintLog();
    gFatal(gMsgStr,True);
  }
  if(gDebug) { 
    sprintf(gMsgStr,"(P%d)Successfully opened header file %s.\n",gIProc,fMapName.chars()); 
    gPrintLog();
  } 
  Util::scan_forward(file,"ROWS=");
  fscanf(file,"%d",&rows);
  if(gDebug) { sprintf(gMsgStr,"rows = %d\n",rows); gPrintLog(); }
  rewind(file);
  Util::scan_forward(file,"COLUMNS=");
  fscanf(file,"%d",&cols);
  if(gDebug) { sprintf(gMsgStr,"cols = %d\n",cols); gPrintLog(); }
  rewind(file);
     
  test = Util::scan_forward(file,"\nSIZE=");
  if( test) fscanf(file,"%d",&size);
  rewind(file);  
  if(gDebug) { sprintf(gMsgStr,"size = %d\n",size); gPrintLog(); }
  if( size < 1 || size > sizeof(double) ) {  
    sprintf(gMsgStr,"Error, Illegal map size: %s  (%d)\n",fMapName.chars(),size); gFatal(gMsgStr); 
  }
  
  test = Util::scan_forward(file,"LOCATION");
  if(test) {
	  remote_data=1;
	  test = Util::scan_forward(file,"");
	  char ch, dirCh = '/'; bFileName = "";
	  CString pathElem[kMaxElem];
	  int nElem = split(fMapName,pathElem,kMaxElem,dirCh);
	  while( ( ch=fgetc(file) ) != '' )  {
		bFileName += ch;
		if( bFileName.length() > 100 ) { gFatal( format("Format Error reading LOCATION in MapII header file: %s", bFileName.chars()) ); }
	  }
	  if(gDebug) { sprintf(gMsgStr,"\nLOCATION = %s\n",bFileName.chars()); gPrintLog(); }
	  pathElem[nElem-1] = bFileName;
	  join(bFileName,pathElem,nElem,dirCh);
  } 
  return remote_data;
}

int  TMap2::ReadHeader(CString& bFileName)
{
  int test, rows, cols, r0=0, c0=0, i, Dsize=1, remote_data=0;
  char ch, dirCh, format_str[16], units[16];
  float cellsize;

  // Should determine MapII Interchange format for setting origon: (r0,c0).
       	
  FILE* file = fopen(fMapName,"r");
  if(file==NULL) { 
    sprintf(gMsgStr,"(P%d)Unable to open header file %s.\n",gIProc,fMapName.chars()); 
    gPrintLog();
    gFatal(gMsgStr,True);
  }
  if(gDebug) { 
    sprintf(gMsgStr,"(P%d)Successfully opened header file %s.\n",gIProc,fMapName.chars()); 
    gPrintLog();
  } 
  Util::scan_forward(file,"ROWS=");
  fscanf(file,"%d",&rows);
  if(gDebug) { sprintf(gMsgStr,"rows = %d\n",rows); gPrintLog(); }
  rewind(file);
  Util::scan_forward(file,"COLUMNS=");
  fscanf(file,"%d",&cols);
  if(gDebug) { sprintf(gMsgStr,"cols = %d\n",cols); gPrintLog(); }
  rewind(file);
  
	test = Util::scan_forward(file,"CELLSIZE=");
  if( test) fscanf( file,"%f", &cellsize );
  rewind(file);
  
	test = Util::scan_forward(file,"UNITS=");
  if( test) fscanf( file,"%s", units );
  rewind(file);

	fReadDataFormat = 0;  
	test = Util::scan_forward(file,"FORMAT=");
  if( test) fscanf( file,"%s", format_str );
  rewind(file);
  if( (strcmp(format_str,"TEXT") == 0) || ( strcmp(format_str,"DEC") == 0 ) ) fReadDataFormat = 1;  
   
	test = Util::scan_forward(file,"\nSIZE=");
  if( test) fscanf(file,"%d",&Dsize);
  rewind(file);  
  if(gDebug) { sprintf(gMsgStr,"size = %d\n",Dsize); gPrintLog(); }
  if( Dsize < 1 || Dsize > sizeof(double) ) {  
    sprintf(gMsgStr,"Error, Illegal map size: %s  (%d)\n",fMapName.chars(),Dsize); gFatal(gMsgStr); 
  }
  
  test = Util::scan_forward(file,"LOCATION");
  if(test) {
		remote_data=1;
		test = Util::scan_forward(file,""); i=1;
		dirCh = '/'; bFileName = "";
		CString pathElem[kMaxElem];
		int nElem = split(fMapName,pathElem,kMaxElem,dirCh);
		while( ( ch=fgetc(file) ) != '' )  {
		  bFileName += ch;
		  if( bFileName.length() > 100 ) { gFatal( format("Format Error reading LOCATION in MapII header file: %s", bFileName.chars()) ); }
		}
		if(gDebug) { sprintf(gMsgStr,"\nLOCATION = %s\n",bFileName.chars()); gPrintLog(); }
		pathElem[nElem-1] = bFileName;
		join(bFileName,pathElem,nElem,dirCh);
	} else rewind(file);
	
  test = Util::scan_forward(file,"LABELS");
  if( test ) {
		int index = 0; 
		while( Util::scan_forward(file,"VALUE=") ) {
			fscanf(file,"%d",&index);
			CString* label = new CString;
			while( ( ch=fgetc(file) ) != '"' )  {;}
			while( ( ch=fgetc(file) ) != '"' )  { (*label) += (char)toupper(ch); }
			fCatMap[index] = (Pix) label;
		}
	}

  fNBytes = Dsize;
	fMaxVal = Util::pow2(fNBytes*8)-2;   
  Region2 r(r0,c0,(r0+rows)*inc(0)-1,(c0+cols)*inc(1)-1,inc(0),inc(1));
  if( !empty() ) r *= (*this);
  setregion(r);
	
	if( remote_data == 0 ) {
		rewind(file);
		test = Util::scan_forward(file,"DATA=");
		if( test ) {
			ReadTextData(file,bFileName);
		} else { sprintf(gMsgStr,"(P%d) No data in map file %s.\n",gIProc,fMapName.chars());  gPrintErr(); }
	}
  fclose(file);
  return remote_data;
}


void TMap2::Dump( FILE* oFile, long background ) const
{
  int rows = extents(0);
  int cols = extents(1);
  long itmp;

  fprintf(oFile,"\n\n~~~~~~~~~~~~~~~Map Dump: %s~~~~~~~~~~~~~~~~~~\n",(const char*)fMapName);
	for(int ir = lower(0); ir<= upper(0); ir+=increment(0) ) {
		for(int ic = lower(1); ic<= upper(1); ic+=increment(1) ) {
      itmp = (long)Value(ir,ic);
      if(itmp>0) {
				if( itmp == background ) { fputc('-',oFile);  }
				else { fputc('*',oFile); }
			}
      else fputc('_',oFile); 
    }
    fputc('\n',oFile);
  }
}

void TMap2::Reflect( int dim ) {
	register byte *mPtr0, *mPtr1, *mPtr0_end, btmp;
	for(int i0=lower(dim); i0<=upper(dim); i0++ ) {
		for(int i1=upper(dim); i1<=lower(dim); i1-- ) {
			if( i1 <= i0 ) return; 
			switch(dim) {
				case 0: 
					mPtr0 = fData + bindex(i0,0)*fNBytes;
					mPtr1 = fData + bindex(i1,0)*fNBytes;
					mPtr0_end = mPtr0 + extents(1)*fNBytes;
				case 1:
					mPtr0 = fData + bindex(0,i0)*fNBytes;
					mPtr1 = fData + bindex(0,i1)*fNBytes;
					mPtr0_end = mPtr0 + extents(0)*fNBytes;
			}
			do {
				btmp = *mPtr0;
				*mPtr0 = *mPtr1;
				*mPtr1++ = btmp;
			} while ( mPtr0++ < mPtr0_end );
		}
	}
}

void TMap2::BroadcastData( int srcProc, char* info, int mergeMode ) {
#ifdef USE_MPI
  int infosize = (info) ? strlen(info)+1 : 0;
  char info_in[kMapInfoSize]; info_in[0] = 0;
  infosize = ( infosize < kMapInfoSize ) ? infosize : kMapInfoSize;
  int memLen=0, dataLen = BSize();
  static byte *tempBuff = NULL;
  TMsgBuff msgBuff(1000,fComm);
  if( gIProc == srcProc ) {
    msgBuff.PackInt(infosize);
    msgBuff.PackInt(lower(0));
    msgBuff.PackInt(lower(1));
    msgBuff.PackInt(upper(0));
    msgBuff.PackInt(upper(1));
    msgBuff.PackInt(fNBytes);
    if( infosize > 0 ) msgBuff.PackString(info);
    if( dataLen > 0 ) {  memLen = msgBuff.PackBytes(fData,dataLen); }
  }
  msgBuff.BCast(srcProc);			
  if( gIProc != srcProc ) {
    infosize = msgBuff.UnPackInt();
    int l0 = msgBuff.UnPackInt();
    int l1 = msgBuff.UnPackInt();
    int u0 = msgBuff.UnPackInt();
    int u1 = msgBuff.UnPackInt();
    int nb = msgBuff.UnPackInt();
    if(fData==NULL) {
			setlower( l0, l1 );
			setupper( u0, u1 );
			fNBytes = nb;
			fMaxVal = Util::pow2(fNBytes*8)-2;   
			Alloc();
		}
    if( infosize > 0 ) { msgBuff.UnPackString(info_in,infosize); }
    if( dataLen > 0 ) { 
			switch ( mergeMode ) {
				case 0:
					msgBuff.UnPackBytes(fData,BSize());  // overWrite
					break;
				case 1:
					if( info_in[0] == 0 ) { break; }
					if(tempBuff == NULL) tempBuff = new byte[BSize()];  // merge
					msgBuff.UnPackBytes(tempBuff,BSize());
					byte *tb_end = tempBuff + BSize(), *tb = tempBuff, *db = fData;
					if( fNBytes == 1 ) {
						while( tb < tb_end ) {
							if( *tb > 0 ) *db = *tb;
							tb++; db++;	
						}					
					} else {
						while( tb < tb_end ) {
							unsigned long ival = (unsigned long) Util::dec_Nb( &tb, fNBytes );
							if( ival > 0 ) { *db = ival; }
							db+=fNBytes;	
						}
					}
					break;
			}
		}
  }
#endif
}

int TMap2::WriteHeader( CPathString& path, CString& mapName, int index, const char* info )
{
  FILE* header;
  mapName += "M2";
  if( index >= 0 ) {
	mapName += "."; 
	mapName.appendIndex(index); 
  }
  path.Add(mapName);
  if(gDebug) { sprintf(gMsgStr,"Writing Header: filen = %s",path.chars()); gPrintLog(); }
  header = fopen(path,"w");
  if(header==NULL) { sprintf(gMsgStr,"Can't open header file: %s\n",path.chars()); gPrintErr(); return 0; }
  fprintf(header,"FILETYPE=INTERCHANGE\nROWS=%d\nCOLUMNS=%d\nCELLSIZE=1",extents(0),extents(1));
  if(info) fprintf(header,"INFO=%s\n",info);
  fprintf(header,"\nUNITS=KM\nFORMAT=BIN\nSIZE=%d\nLOCATION=%s.bin\n",fNBytes,mapName.chars());
  fclose(header);
  return 1;
}

int TMap2::WriteM2File( const CPathString& path, const CString& mapName, int index, const char* info, int iproc )
{
  if( gIProc != iproc ) return 0; 
  FILE* bFile; int rv, istat; 
  CPathString fileP(path);
  CString fileN(mapName);
  sprintf(gMsgStr,"S(%f,%f)",fS[1],fS[0]);  
  if( WriteHeader(fileP,fileN,index,info) == 0) return 0;
  if( ( bFile = fopen(fileP.Path(".bin"),"wb") ) == NULL ) { sprintf(gMsgStr,"Can't Open Mapfile: %s",(const char*)fileP.Path(".bin")); gPrintErr(); return 0; }
  fwrite(Data(),fNBytes,nelem(),bFile);
  if  ( gDebug ) {
    sprintf(gMsgStr,"Writing Map %s: scale = %f, offset = %f, index = %d, fBytes = %d",
	    (const char*)fileN,fS[1],fS[0],index,fNBytes);   	  
    gPrintLog();
  }
  sprintf(gMsgStr,"Wrote map %d to MapII file %s",index,(const char*)fileN); gPrintScreen();
  fclose(bFile);
  return 1;
}

int TMap2::WritePPMFile( const CPathString& path, const CString& mapName, int index, const char* info, int iproc )
{
  if( gIProc != iproc ) return 0; 
  FILE* outfile = NULL; 
  CPathString fileP(path);
  CString fileN(mapName);
  fileN += "ppm";
  if( index >= 0 ) {
	fileN += "."; 
	fileN.appendIndex(index); 
  }
  fileP.Add(fileN);
  if(gDebug) { sprintf(gMsgStr,"Writing ppm file: filen = %s",fileP.chars()); gPrintLog(); }
  outfile = fopen(fileP,"w");
  if(outfile==NULL) { sprintf(gMsgStr,"Can't open ppm file: %s\n",fileP.chars()); gPrintErr(); return 0; }
  int w = extents(1), h = extents(0), dsize = w*h*3;
  fprintf(outfile,"P6\n%d\n%d\n255\n",w,h);
  byte* cdata = new byte[dsize];
  float fval, maxval = -FLT_MAX, minval = FLT_MAX, nullValue = FLT_MAX;
  if( fS[0] > 0.0 ) {
	maxval = fS[0]; minval = fS[1]; 
  } else {
	for( int i=0; i<h; i++ ) {
	  for( int j=0; j<w; j++ ) {
		  fval = Value(i,j);
		  if( fval != nullValue ) {
			if( fval > maxval ) maxval = fval;
			if( fval < minval ) minval = fval;
		  }
	  }
	}
  }
  float pval, sf = (maxval-minval);
  if( sf == 0.0 ) sf = 1.0;
  int k = 0;
  for( int i=0; i<h; i++ ) {
	for( int j=0; j<w; j++ ) {
		fval = Value(i,j);
		if( fval == nullValue ) {
		  cdata[k] = cdata[k+1] = cdata[k+2] = 0;
		} else {
		  pval = (fval - minval)/sf;
		  cdata[k] = (byte) 255*pval;
		  cdata[k+1] = (byte) ( pval > 0.5 ) ? 510*(1-pval) : 510*pval;
		  cdata[k+2] = (byte) 255*(1.0-pval);
		}
		k += 3;
	}
  }
  fwrite(cdata,1,dsize,outfile);
  if  ( gDebug ) {
    sprintf(gMsgStr,"Writing ppm file %s: max = %f, min = %f, index = %d, fBytes = %d",
	    (const char*)fileN,maxval,minval,index,fNBytes);   	  
    gPrintLog();
  }
  sprintf(gMsgStr,"Wrote map %d (%f,%f) to ppm file %s",index,maxval,minval,(const char*)fileN); gPrintScreen();
  fclose(outfile);
  return 1;
}

int TMap2::WriteAscii( const char* path )
{
  if( gIProc != 0 ) return 0; 
  FILE* bFile; int rv, istat; 
//  CPathString fileP(path);
//  CString fileN(mapName);
//  fileN.appendIndex(index,4);
//  fileN += ".asc";
//  fileP.Add(fileN);
  if( ( bFile = fopen(path,"w") ) == NULL ) 
  { sprintf(gMsgStr,"Can't Open Mapfile: %s", path); gPrintErr(); return 0; }
  for (long i = 0; i < (long)extents(0); i++)
    for (long j = 0; j < (long)extents(1); j++)
      fprintf (bFile, "%ld%c", (long)Data()[(i*(long)extents(0))+j], j == extents(1) - 1 ? '\n' : ' ');
  sprintf(gMsgStr,"Wrote map to ASCII file %s", path); gPrintScreen();
  fclose(bFile);
  return 1;
}



int TMap2::WriteBinary( const CPathString& path, const CString& mapName, int index )
{
  if( gIProc != 0 ) return 0; 
  FILE* bFile; int rv, istat; 
  CPathString fileP(path);
  CString fileN(mapName);
  fileN += "bin";
  if( index >= 0 ) {
	fileN += "."; 
	fileN.appendIndex(index); 
  }
  fileP.Add(fileN);
  if( ( bFile = fopen(fileP,"wb") ) == NULL ) { sprintf(gMsgStr,"Can't Open Mapfile: %s",fileN.chars()); gPrintErr(); return 0; }
  fwrite(Data(),fNBytes,nelem(),bFile);
  if  ( gDebug ) {
    sprintf(gMsgStr,"Writing Map %s: scale = %f, offset = %f, index = %d, fBytes = %d",
	    (const char*)fileN,fS[1],fS[0],index,fNBytes);   	  
    gPrintLog();
  }
  sprintf(gMsgStr,"Wrote map %d to binary file %s",index,(const char*)fileN); gPrintScreen();
  fclose(bFile);
  return 1;
}

int TMap2::WriteHdf(const char* path, float time)
{
  return 1;
}

int TMap2::WriteArc(const char* path)
{
  return 1;
}

int TMap2::WriteGrass(const char* path)
{
  return 1;
}
