#ifndef __TMap2__
#define __TMap2__

#include "ByteGrid.h"
#include "PathString.h"
#include "MsgBuff.h"
#include "IntPixHashMap.h"

enum EMapTags { kMT_Broadcast };
const int kMapInfoSize = 100;

//----------------------------------------------------------------------------------------
//					 TMap
//----------------------------------------------------------------------------------------

//  ERDAS file image header

struct image_header {
				char         head[6];           /* version     */
				short int    pack_type;         /* 4,8,16 bits */
				short int    nbands;            /* #bands      */
				char         notused2[6];
				int          rcols;             /* #cols       */
				int          rrows;             /* #rows       */
				int          xfirst;            /* start X - UL*/
				int          yfirst;            /* start Y - UL*/
				char         notused3[56];
				short int    map_units;         /* code for units*/
				short int    num_classes;       /* #classes    */
				char         notused4[28];
				char         x_pixel_size[4];   /* X cell size */
				char         y_pixel_size[4];   /* Y cell size */
};
 
class TMap2 : public ByteGrid 
{
protected:

  CString fMapName;
  int  ReadHeader(CString& bFileName);
  int GetHeaderInfo(CString& bFileName, int& rows, int& cols, int& size );
  int WriteHeader( CPathString& path, CString& mapName,  int index, const char* info );

  float fS[2];                 // Scale factors for converting to-from float maps;
                               // S1: scale, S0: offset 
	 float fScaleBuffer[2];
	 float fMax;
	 float fMin;
	 float fUB;
	 float fLB;
	 int fReScale;
	 float fRescaleConst;
	 int fHasBounds;
	 int fReadDataFormat;
	 intPixVHMap fCatMap;
	 struct  image_header   fImageHd; 
		 
#ifdef USE_MPI
    static MPI_Comm fComm;
    static unsigned int fMapCount;
    unsigned int fMapIndex;
#endif

public:

  TMap2( char* mapName=NULL, int rscale = 1, int cscale = 1);
  TMap2( Region2* region, int nbytes=1 );
  TMap2( TMap2& m );

	virtual void ReadTextData( FILE* file, const char* name ); 
  int ReadM2File( const char* mapName = NULL, int rscale = 1, int cscale = 1 );
  int ReadM2File2( const char* mapName = NULL, const char* templateName= NULL, int rscale = 1, int cscale = 1 );
  virtual void ReadTextData2( FILE* file, const char* name, int row_min, int row_max, int col_min, int col_max, int rows, int cols ) { 
	gFatal( "Must define ReadTextData2" ); 
  }
  int ReadPPMFile( const char* pathName, int rscale = 1, int cscale = 1 );
  int ReadM2Categories( const char* pathName );
  int ReadArcInfoSVF( const char	*filename, int background=0 );
  int ReadERDASMap ( const char* filename, int background=0 );
  int ReadIDRISIMap ( const char* filename, int background=0 );
  int WriteM2File( const CPathString& path, const CString& mapName,  int index, const char* info, int iproc=0 );
  int WritePPMFile( const CPathString& path, const CString& mapName, int index, const char* info, int iproc=0 );
  int WriteBinary( const CPathString& path, const CString& mapName,  int index );
  int WriteAscii( const char* path );
  int WriteHdf(const char* path, float time);
  int WriteArc(const char* path);
  int WriteGrass(const char* path);
  inline CString& MapName() { return fMapName; }
  
  inline void SetByteConversion( float scale, float offset, int nbytes = -1 ) { 
    fS[0] = offset; fS[1] = scale; fNBytes = (nbytes>=0) ? nbytes : fNBytes; 
    fMaxVal = Util::pow2(fNBytes*8)-2;                  
    // set scale:1 and offset:0;
	}
	inline void setDataCeiling( long int val ) { fMaxVal = val; }
	
  virtual inline float Value(Point2& pt) const { 
    long itmp = IntValue( pt );
    float rv = ( ( itmp < 0 ) ? 0.0 :  fS[1]*((float)itmp)+fS[0] );
    return rv; 
  }
  virtual inline float Value( int ir, int ic) const { 
    long itmp = IntValue( ir, ic );
    float rv = ( ( itmp < 0 ) ? 0.0 :  fS[1]*((float)itmp)+fS[0] );
    return rv; 
  }
    
  inline TMap2& operator=( const TMap2& map ) { 
		map.CopyDataTo( fData, fNBytes, *this); 
		return *this;
	}
	
#ifdef USE_MPI	
  inline unsigned int getTag() {
	return 2002 + fMapIndex;
  }
#endif
  
  inline unsigned long DirectScale( float val ) {
		return (unsigned long) Util::fclamp( fS[1]*val + fS[0], 0.0, fMaxVal );
	}
		
  inline unsigned long Scale( float val ) {
		switch( fHasBounds ) {
			case 0: return (unsigned long) Util::fclamp(((val-fS[0])/fS[1]),0.0,fMaxVal);
			case 1: {
				 return (unsigned long) ((val - fMin)*fRescaleConst);
			}
			case 2: 
				if( val < fLB) val = fLB;
         if (val > fUB ) val = fUB;			   
         if( fReScale ) {
					 float cf =  fMaxVal/(fUB - fLB);
					 return (unsigned long) ((val - fLB)*cf);
         } else return (unsigned long) val;
		}
	}

  inline virtual void Allocate() { 
			Alloc();			
  }


  inline virtual void SetValueWithRescale( const Point2* point, float fval, Bool directScale = FALSE ) { 
			unsigned long dval;
			if( directScale ) { dval =  DirectScale( fval ); }
			else { dval =  Scale( fval ); }
			ByteGrid::SetValue( point, dval );
  }
	void Reflect( int dim );
	inline int NBytesClamped() { return fNBytes = Util::iclamp(fNBytes,0,sizeof(float)); }     
  void Dump( FILE* oFile, long background=0  ) const;
  virtual void BroadcastData ( int srcProc, char* info = NULL, int mergeMode = 0 );
  
	inline float ScaleFactor(int i) { return fS[1]; }
	
	inline float* ScaleData() { 
		if(fHasBounds) {
			fScaleBuffer[0] = fMax;
			fScaleBuffer[1] = fMin;
			return fScaleBuffer; 
		} else return fS; 
	}
	
	inline void SetBounds( float min, float max, float lb=0.0, float ub=0.0, int reScale = 0 ) {
		fMax = max; fMin = min;  fRescaleConst = fMaxVal/(fMax - fMin);
		if( ub==0.0 && lb==0.0 ) { fUB = max; fLB = min; fHasBounds = 1;  }
		else { fUB = ub; fLB = lb; fHasBounds = 2; }
		fReScale = reScale; 
	}
	
	inline unsigned long defaultBackground() { return fMaxVal+1; }	
  inline void ClearBounds() { fMax = 0.0;  fMin = 0.0; fUB = 0.0; fLB = 0.0; fReScale = fHasBounds = 0; fRescaleConst = 1.0; }
	inline intPixVHMap& CatMap() { return  fCatMap; }
  virtual void ComposeDataPtrs( ) {; }
};

#endif

