/**************************************************************************/
/*                         Xgraph.cc    Xgraph.cc                           */
/**************************************************************************/

/* this implementes Xgraph, a simple X based graphics drawing class */

/**************************************************************************/

#ifdef HAS_X

	// standard libraries
	
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include "XGraph.h"
#include "Environ.h"

int pDebug=0;
FILE* dFile;
#define MAX_SHORT 65536

/**************************************************************************/

TImage::TImage(Display* aDisplay) {
  height = 0;
  width = 0;
  r0=0;
  c0=0;
  display = aDisplay;
  gc = None;
  pixmap = 0;
  shape=0;
  xgraph = NULL;
  wind = DefaultRootWindow(display);
  XGCValues gcValues;
  gc = XCreateGC(display, wind, 0, &gcValues);
}

/**************************************************************************/

TImage::TImage(Xgraph* aGraph) {

  height = 0;
  width = 0;
  r0 = 0;
  c0 = 0;
  display = aGraph->getDisplay();
  aGraph->CreateColors();
  gc = None;
  xgraph = aGraph;
  pixmap = 0;
  wind = DefaultRootWindow(display);
  XGCValues gcValues;
  gc = XCreateGC(display, wind, 0, &gcValues); 
  shape = None;
  AInfo[0] = 0; AInfo[1] = 0; AInfo[2] = 0; AInfo[3] = 0;
}

/**************************************************************************/

TImage::~TImage() {
}

/**************************************************************************/
/*  Xgraph  */
/**************************************************************************/

Xgraph::Xgraph(int aWidth, int aHeight, Bool isRoot ) {

  if(pDebug) {
    CPathString path(Env::DataPath());
		path.Add("Xgraph.debug");  
		dFile = fopen(path,"w");
	}
  else dFile = NULL;
  redpixel = 0;
  midpixel = 0;
  bluepixel = 0;
  isRootNode = isRoot;
  
  if(isRoot) {
    display = XOpenDisplay(NULL);
    if (display == NULL)
      return;

    w = XCreateSimpleWindow(display, 
			    DefaultRootWindow(display), 0, 0, aWidth, aHeight, 0, 
			    WhitePixel(display, DefaultScreen(display)),
			    BlackPixel(display, DefaultScreen(display)));

    closeDisplay = TRUE;
    setup(display, w);
    resetColors = 1; 
    SetRedMap(0.0,0.75,1.0);
    SetBlueMap(1.0,0.75,0.0);
    SetGreenMap(0.0,1.0,0.0);
  }
  
  MaxImages = 100;
  ImageList = (TImage**) malloc(MaxImages*sizeof(TImage*));
  NImages = 0;
  yOffset = 0;
  xOffset[0] = 0; 
  xOffset[1] = 0;
  currentImage = NULL;
  colorBar = None;
  ngraphs = 0;
  for(int i=0; i< MAX_GRAPHS; i++) graph_cnt[i] = 0;
}

/**************************************************************************/

void Xgraph::Clear() {
  xOffset[0] = 0; 
  xOffset[1] = 0;
  refresh();
}

/**************************************************************************/

TImage* Xgraph::NewImage(EImageType type)
{
  if( NImages == MaxImages ) {
    MaxImages += 50;
    ImageList = (TImage**) realloc(ImageList,MaxImages*sizeof(TImage*));
  }
  switch(type) {
  case kXAnim: {
    XAnim* newImage = new XAnim(this);
    ImageList[NImages++] = (TImage*) newImage;
    if(dFile) fprintf(dFile,"\n Xgraph register XAnim: %d",NImages);
    return newImage;
  } break;
  case kGraph: {
    TGraph* newImage = new TGraph(this);
    ImageList[NImages++] = (TImage*) newImage;
    if(dFile) fprintf(dFile,"\n Xgraph register TGraph: %d",NImages);
    return newImage;
  } break;
  case kSheet: {
    TSheet* newImage = new TSheet(this);
    ImageList[NImages++] = (TImage*) newImage;
    if(dFile) fprintf(dFile,"\n Xgraph register TSheet: %d",NImages);
    return newImage;
  } break;
  }
}

/**************************************************************************/

void Xgraph::setup(Display* aDisplay, Window aWindow) {

  if( !isRootNode ) return;
  display = aDisplay;
  w = aWindow;
  immediateUpdate = TRUE;
  black = BlackPixel(display, DefaultScreen(display));
  white = WhitePixel(display, DefaultScreen(display));
  colormap = DefaultColormap(display, DefaultScreen(display));

  int x, y;
  Window root;
  unsigned int borderWidth;
  unsigned int depth;
  XGetGeometry(display, w, &root, 
	       &x, &y, &width, &height, &borderWidth, &depth);

  XSelectInput(display, w, ExposureMask+KeyPressMask+KeyReleaseMask+ButtonPressMask);

  XGCValues gcValues;
  gcValues.font = XLoadFont(display, "fixed");
  gcValues.foreground = black;		// black to init backup PixMap
  gcValues.background = black;
  gc = XCreateGC(display, w, GCFont+GCForeground+GCBackground, &gcValues);
  XSetGraphicsExposures(display, gc, 1);
  XMapWindow(display, w);

  backup = XCreatePixmap(display, w, width, height, depth);
  XFillRectangle(display, backup, gc, 0, 0, width, height);
  XSetForeground(display, gc, white);

  XEvent event;		// wait for window to be maped
  do 
    XNextEvent(display, &event);
  while (event.type != Expose);
}

/**************************************************************************/

Xgraph::~Xgraph() {

  if(dFile) fclose(dFile);
  free(ImageList);
  if( !isRootNode ) return;
  XDestroyWindow(display, w);
  XFreePixmap(display, backup);
  if(colorBar != None) XFreePixmap(display, colorBar);
  XFreeGC(display, gc);
  if (closeDisplay) XCloseDisplay(display);
}

/**************************************************************************/

int Xgraph::CreateLegendPixmap()
{    
  if( !isRootNode ) return 0;
  Visual *visual;
  Colormap colormap;
  XImage* legendImage;
  unsigned int depth = 8;
  unsigned int bitmap_pad = 8;

  if (display == 0) return 0;
  if (colorBar != None) {
    XFreePixmap( display, colorBar );
  }
  int w = 10;
  int h = 128;

  visual = DefaultVisual(display, DefaultScreen(display));
  colormap = DefaultColormap(display, DefaultScreen(display));

  /* then create the XImage with data = NULL and bytes_per_line = 0 */

  legendImage = XCreateImage(display, visual, depth, ZPixmap, 0,  (char*)NULL, w, h, bitmap_pad, 0);
  if (!legendImage) return 0;

  /* now that bytes_per_line must have been set properly alloc data */

  char* ldata = (char*) malloc(w*h);
  for(int i=0; i<h; i++) {
    for(int j=0; j<w; j++) {
      ldata[j+i*w] = 255-2*i;
    }
  }
  legendImage->data = ldata;
  XGCValues gcv;
  GC pgc;

  Window wind = DefaultRootWindow(display);
  colorBar = XCreatePixmap(display, wind, legendImage->width, legendImage->height, legendImage->depth);
  gcv.function = GXcopy;
  pgc = XCreateGC(display, colorBar, GCFunction, &gcv);
  XPutImage(display, colorBar, pgc, legendImage, 0, 0, 0, 0, legendImage->width, legendImage->height);

  XFreeGC(display, pgc);
  XDestroyImage(legendImage);
  return 1;
}

/**************************************************************************/

void Xgraph::DisplayLegend(Window w, int xStart, int yStart, float fmin, float fmax ) {

  if( !isRootNode ) return;
  char ltext[256]; 
  if(colorBar == None) CreateLegendPixmap();
  if (immediateUpdate)
    XCopyArea(display, colorBar, w, gc, 0, 0, 10, 128, xStart, yStart);
  sprintf(ltext,"%.3g",fmax);
  text(xStart+12,yStart,ltext,2);
  sprintf(ltext,"%.3g",fmin);
  text(xStart+12,yStart+128,ltext,2); 
  flush(); 
}

/**************************************************************************/

int Xgraph::setForeground(int cIndex) {
  if( !isRootNode ) return 0;
  if( XSetForeground(display, gc, cpixel[cIndex] )) return 1;
  else return 0;
  flush();
}
/**************************************************************************/

int Xgraph::setForeground(char* color) {

  if( !isRootNode ) return 0;
  XColor exact;
  XColor displayed;
  if( strcmp(color,"white") == 0 )  { if(XSetForeground(display, gc, white)) return 1; }
  if( strcmp(color,"red") == 0 && redpixel>0 )  { if(XSetForeground(display, gc, redpixel)) return 1; }
  if( strcmp(color,"mid") == 0 && midpixel>0 )  { if(XSetForeground(display, gc, midpixel)) return 1; }
  if( strcmp(color,"blue") == 0 && bluepixel>0 )  { if(XSetForeground(display, gc, bluepixel)) return 1; }
  if (XAllocNamedColor(display, colormap, color, &exact, &displayed) == 0) return 0;
  XSetForeground(display, gc, displayed.pixel);
  flush();
  return 1;
}
    
/**************************************************************************/

void Xgraph::rectangle(int xStart, int yStart, int xEnd, int yEnd, int isTemp) {

  if( !isRootNode ) return;
  if(isTemp==1) refresh();
  if (immediateUpdate || isTemp) 
    XDrawRectangle(display, w, gc, xStart, yStart, xEnd, yEnd);
  if(!isTemp) {
    XDrawRectangle(display, backup, gc, xStart, yStart, xEnd, yEnd);
  }
  if( isTemp < 2 ) flush();
}

/**************************************************************************/

void Xgraph::filledRectangle(int xStart, int yStart, int xEnd, int yEnd, int isTemp) {

  if( !isRootNode ) return;
  if(isTemp==1) refresh();
  if (immediateUpdate || isTemp) 
    XFillRectangle(display, w, gc, xStart, yStart, xEnd, yEnd);
  if(!isTemp) {
    XFillRectangle(display, backup, gc, xStart, yStart, xEnd, yEnd);
  }
  if(isTemp<2) flush();
}

/**************************************************************************/

void Xgraph::text(int xStart, int yStart, char* text, int isTemp) {
  if( !isRootNode ) return;
  if(isTemp==1) refresh();
  if (immediateUpdate || isTemp) 
    XDrawString(display, w, gc, xStart, yStart, text, strlen(text));
  if(!isTemp) {
    XDrawString(display, backup, gc, xStart, yStart, text, strlen(text));
  }
  if(isTemp<2) flush();
}

/**************************************************************************/

void Xgraph::line(int xStart, int yStart, int xEnd, int yEnd, int isTemp) {
  if( !isRootNode ) return;
  if(isTemp==1) refresh();
  if (immediateUpdate || isTemp)
    XDrawLine(display, w, gc, xStart, yStart, xEnd, yEnd);
  if(!isTemp) {
    XDrawLine(display, backup, gc, xStart, yStart, xEnd, yEnd);
  }
  if(isTemp<2) flush();
  if(dFile) fprintf(dFile,"\n Xgraph Draw Line: (%d,%d)-> (%d,%d)",xStart,yStart,xEnd,yEnd);
}

/**************************************************************************/

void Xgraph::circle(int xCenter, int yCenter, int radius) {
  if( !isRootNode ) return;
  if (immediateUpdate)
    XDrawArc(display, w, gc, xCenter-radius, yCenter-radius, 	
	     2*radius, 2*radius, 0, 64*360);
  XDrawArc(display, backup, gc, xCenter-radius, yCenter-radius, 	
	   2*radius, 2*radius, 0, 64*360);
  flush();
}

/**************************************************************************/

void Xgraph::filledCircle(int xCenter, int yCenter, int radius) {
  if( !isRootNode ) return;
  if (immediateUpdate)
    XFillArc(display, w, gc, xCenter-radius, yCenter-radius, 	
	     2*radius, 2*radius, 0, 64*360);
  XFillArc(display, backup, gc, xCenter-radius, yCenter-radius, 	
	   2*radius, 2*radius, 0, 64*360);
  flush();
}

/**************************************************************************/

void Xgraph::image( TImage* image,int xStart,int yStart,int width,int height,int isTemp ){
  if( !isRootNode ) return;
  if(isTemp==1) refresh();
  if (immediateUpdate|| isTemp)
    image->putDisplay(w, xStart, yStart, width, height);
  if(!isTemp) {
    image->putDisplay((Window)backup, xStart, yStart, width, height);
  }
  if(isTemp<2) flush();
}

/**************************************************************************/

void Xgraph::refresh() {
  if( !isRootNode ) return;
  XCopyArea(display, backup, w, gc, 0, 0, width, height, 0, 0);
  flush();
}

/**************************************************************************/

int Xgraph::doXEvent(XEvent& event) {
  if( !isRootNode ) return 0;
  int rctrl = 0;
  if(dFile) {
    if (event.type == ButtonPress) {
      fprintf(dFile,"XG ButtonPress: (%d,%d,%d)\n", event.xbutton.x, event.xbutton.y, event.xbutton.button );
    }
    if (event.type == KeyPress) {
      fprintf(dFile,"XG KeyPress: (%d,%d,%d)\n", event.xkey.x, event.xkey.y, event.xkey.keycode );
    }
    else fprintf(dFile,"XG Processing Event, type: %d\n", event.type );
  }
  if (event.type == Expose && event.xexpose.window == w) {
    refresh();
  }
  if (event.type == KeyPress) {
    char buffer[20];
    int bufsize=20;
    KeySym key;
    XComposeStatus compose;
    XLookupString( (XKeyEvent*)(&event),buffer,bufsize,&key,&compose);
    if( buffer[0] == breakChar ) rctrl = 1;
  }
  for(int i=0; i<NImages; i++) ImageList[i]->doXEvent(event);
  return rctrl;    
}

/**************************************************************************/

void Xgraph::flush(Bool block) {
  if( !isRootNode ) return;
  XEvent event; int ictrl;
  while(XPending(display) != 0 || block) {
    XNextEvent(display, &event);
    ictrl = doXEvent(event);
    if( block && ictrl ) break;
  }
}

/**************************************************************************/

void Xgraph::block(char eChar) {
  if( !isRootNode ) return;
  breakChar = eChar;
  flush(1);
}

/**************************************************************************/

void Xgraph::CreateColors() {
  if( !isRootNode ) return;
  if(!resetColors) return; 
  XColor color;
  Colormap colormap = DefaultColormap(display, DefaultScreen(display));
  int ncolors=0, first=1, last=1, mid=1;

  for(int i=0; i<256; i++) {
    color.red = red.GetColor(i);
    color.blue = blue.GetColor(i);
    color.green = green.GetColor(i);
    color.flags = DoRed | DoGreen | DoBlue;
    if( !XAllocColor(display,colormap,&color)) {
      cpixel[i] = 0;
    }
    else {
      ncolors++;
      cpixel[i] = color.pixel;

      if(first) {
	bluepixel = color.pixel;
	first = 0;
      }
      if(i>220 && last) {
	redpixel = color.pixel;
	last = 0;
      }	
      if( i>125 && i< 150 && mid ) {
	midpixel = color.pixel;
	mid = 0;
      }	
    }
  }
  resetColors = 0; 
  if(pDebug) printf("Allocated %d colors\n",ncolors);
}

/**************************************************************************/

float clip(float f0, float fmin, float fmax) {
  return ((f0<fmin) ? fmin : ((f0>fmax) ? fmax : f0));
}

void Xgraph::SetRedMap(float ifs, float ifm, float ife) 
{ 
  ifs = clip(ifs,0.0,1.0);
  ifm = clip(ifm,0.0,1.0);
  ife = clip(ife,0.0,1.0);
  red.Set(ifs,ifm,ife); 
}
     
void Xgraph::SetBlueMap(float ifs, float ifm, float ife) 
{ 
  ifs = clip(ifs,0.0,1.0);
  ifm = clip(ifm,0.0,1.0);
  ife = clip(ife,0.0,1.0);
  blue.Set(ifs,ifm,ife); 
}
    
void Xgraph::SetGreenMap(float ifs, float ifm, float ife) { 
  ifs = clip(ifs,0.0,1.0);
  ifm = clip(ifm,0.0,1.0);
  ife = clip(ife,0.0,1.0);
  green.Set(ifs,ifm,ife); 
}

/**************************************************************************************/
/* XAnim  */
/**************************************************************************************/

XAnim::XAnim(Xgraph* aGraph, int imag) : TImage(aGraph) {
  image = NULL;
  mag = imag;
  name = NULL;
  AInfo[2] = 0;
  fData = NULL;
}

/**************************************************************************/

void XAnim::doXEvent(XEvent& event) {

  if (event.type == ButtonPress) {
    if(dFile) {
      fprintf(dFile,"Anim Processing ButtonPress in (%d,%d),(%d,%d)\n", c0, r0, c0+width, r0+height );
    }
    ProcessButtonPress( event.xbutton.x, event.xbutton.y, event.xbutton.button );
  }
}

/**************************************************************************/

void XAnim::SetSize( int aWidth, int aHeight ) {

  width = aWidth; height = aHeight;
  int xStart, yStart, buffy=40, buffx = 50;
	
  if(xgraph->xOffset[0] == 0) {
    xStart = 10;
    xgraph->xOffset[0] = (width + 10);
  } else {
    xStart = xgraph->xOffset[0] + buffx; 
    xgraph->xOffset[0] += (width + buffx);
  }
  if( xgraph->yOffset < height+buffy ) xgraph->yOffset = height+buffy;
  yStart = buffy;
  c0 = xStart;
  r0 = yStart;
  if(dFile) fprintf(dFile,"\n XAnim Set Size: Origin: (%d,%d) // Xgraph offset: ((%d,%d),%d) // Size(w,h): (%d,%d)",c0,r0,xgraph->xOffset[0],xgraph->xOffset[1],xgraph->yOffset,width,height);
}

/**************************************************************************/

void XAnim::ProcessButtonPress( int x, int y, int button ) {
  char msg[100]; float fval; unsigned char cval;
  if( x>c0 && x < c0+width  && y>r0 && y < r0+height) {
    cval = GetValue(x-c0,y-r0,&fval);
    sprintf(msg,"Value at (%d,%d) = %d (%f)",x-c0,y-r0,cval,fval);
    xgraph->text(c0,r0+height+10,msg,1);
    xgraph->currentImage = this;
  }
}

/**************************************************************************/

unsigned char XAnim::GetValue(int x, int y, float *fval)
{
  unsigned char cval = (unsigned char)image->data[x+y*width];
  *fval = cval;
  return cval;	
}

/**************************************************************************/

void XAnim::Flush() {
  xgraph->image( this,c0,r0,width, height,2);
}
/**************************************************************************/

void XAnim::DrawRegion( Region2& r ) {
  xgraph->rectangle( c0+r.lower(1)*mag, r0+r.lower(0)*mag, r.extents(1)*mag, r.extents(0)*mag );
}
/**************************************************************************/

void XAnim::Copy(Pixmap aPixmap, int aWidth, int aHeight, int xStart, int yStart) {

  if (display == 0) 
    return;

  XCopyArea(display, aPixmap, pixmap, gc, 0, 0, aWidth, aHeight, xStart, yStart);
}

/**************************************************************************/

void XAnim::putDisplay(Window w, int xStart, int yStart, int aWidth, int aHeight) {

  static int first_time = 1;
  if (display == 0) return;

  if (aWidth == 0 && aHeight == 0) {
    aWidth = width;
    aHeight = height;
  }

  if (xStart == -1 || yStart == -1) {
    xStart = c0;
    yStart = r0;
  } else {
    c0 = xStart;
    r0 = yStart;
  }

  if( shape != None ) {
    XSetClipMask(display, gc, shape);
    XSetClipOrigin(display, gc, xStart, yStart);
  }

  if( AInfo[0] == 'i') {
    XPutImage(display, w, gc, image, 0, 0, xStart, yStart, aWidth, aHeight);
  }
  else if( AInfo[0] == 'p') {
    XCopyArea(display, pixmap, w, gc, 0, 0, aWidth, aHeight, xStart, yStart);
  }
  DisplayLegend(w,xStart+aWidth+10,yStart);
  if(name) { xgraph->text(xStart+10,yStart-10,name); }  

  if(dFile) fprintf(dFile,"\nXAnim: display image at (%d,%d), size: (%d,%d)",c0,r0,width,height); 
  first_time = 0;
}

/**************************************************************************/

void XAnim::DisplayLegend(Window w, int xStart, int yStart) {
  xgraph->DisplayLegend( w, xStart, yStart, 0.0, 255.0 );
}

/**************************************************************************/

XAnim::~XAnim() {
  if (image != NULL) {
    if(AInfo[1] == 's') image->data = NULL;
    XDestroyImage(image);
    AInfo[3] = 0;
  }
  if(name) delete[] name;
}

/**************************************************************************/

void XAnim::SetMapData( char* aName )
{    
  int nLen = strlen(aName);
  name  = new char[nLen+1];
  strcpy(name,aName);
}

/**************************************************************************/

int XAnim::CreateNewImage( int aWidth, int aHeight, char* data, int imag, int deepCopy )
{    
  Visual *visual;
  Colormap colormap;
  unsigned int depth = 8;
  unsigned int bitmap_pad = 8;

  if (image != NULL) {
    if(AInfo[1] == 's') image->data = NULL;
    XDestroyImage(image);
    AInfo[1] = 0;
  }
  if(imag<1) {;}
  else if(imag>10) mag = 10;
  else mag = imag;
  SetSize( aWidth*mag, aHeight*mag ); 
  if (display == 0) return 0;

  visual = DefaultVisual(display, DefaultScreen(display));
  colormap = DefaultColormap(display, DefaultScreen(display));

  /* then create the XImage with data = NULL and bytes_per_line = 0 */

  image = XCreateImage(display, visual, depth, ZPixmap, 0,  (char*)NULL, width, height, bitmap_pad, 0);
  if (!image) return 0;
  AInfo[0] = 'i';
  if(deepCopy || mag>1) AInfo[1] = 'd';
  else AInfo[1] = 's';

  /* now that bytes_per_line must have been set properly alloc data */

  if(data) { SetImageData( data ); fData = data; }
  AInfo[2] = 1;
  return 1;
}/**************************************************************************/

void XAnim::SetImageData( char* data )
{
  if(!image) return;
  if(data==NULL) data = fData;
  if( AInfo[1] == 'd' ) {
    int mLen;
    if(AInfo[3] == 0) {
      mLen = image->bytes_per_line * height;
      if(dFile) fprintf(dFile,"Image bytes per line = %d at mag %d\n",image->bytes_per_line,mag);
      image->data = (char*)malloc(mLen);
      if (!image->data) {
	XDestroyImage(image);
	image = NULL;
	return;
      }    
      AInfo[3] = 1;
    }
    char* idata = image->data;
    if(mag==1) memcpy(idata,data,mLen);
    else if(mag>1) {
      int h = image->height;
      int w = image->width;
      for(int i=0; i<h; i++) {
	for(int j=0; j<w; j++) {
	  idata[j+i*w] = data[(j/mag)+(i/mag)*(w/mag)];
	}
      }
    }
  }
  else if( AInfo[1] == 's' ){
    image->data = data;
  }
}

/**************************************************************************/

void XAnim::GetMinMax( int* minp, int* maxp )
{
  int h = image->height;
  int w = image->width;
  char* data = image->data, ctmp;
  unsigned char min=255, max = 0;
  for(int i=0; i<h; i++) {
    for(int j=0; j<w; j++) {
      ctmp = data[j+i*w];
      if(ctmp<min) min = ctmp;
      if(ctmp>max) max = ctmp;
    }
  }
  *minp = (int)min;
  *maxp = (int)max;
}

/**************************************************************************/

int XAnim::CreatePixmap(int aWidth, int aHeight, char* data )
{    
  unsigned int depth=8;
  if (display == 0) return 0;
  width = aWidth;
  height = aHeight;
  int rv = CreateNewImage( aWidth, aHeight, data, 0 );
  if (rv == 0) return 0;

  XGCValues gcv;
  GC pgc;

  pixmap = XCreatePixmap(display, wind, image->width, image->height, image->depth);
  gcv.function = GXcopy;
  pgc = XCreateGC(display, pixmap, GCFunction, &gcv);
  XPutImage(display, pixmap, pgc, image, 0, 0, 0, 0, image->width, image->height);

  XFreeGC(display, pgc);
  AInfo[0] = 'p';
  return 1;
}

 
unsigned short ColorParm::GetColor( int i ) {
  unsigned short rv;
  float f1, f2; unsigned int itmp;
  if( i<128 && i>0 ) {
    f1 = (128-i)*(fs/128);
    f2 = i*(fm/128);
    itmp = (unsigned int)((f1+f2)*MAX_SHORT);
    rv = (unsigned short)itmp;
  } else if(i<256) {
    f1 = (255-i)*(fm/128);
    f2 = (i-128)*(fe/128);
    itmp = (unsigned int)((f1+f2)*MAX_SHORT);
    rv = (unsigned short)itmp;
  } else rv = 0;
  return rv;
}


/**************************************************************************************/
/* TGraph  */
/**************************************************************************************/

TGraph::TGraph( Xgraph* aGraph, int buffSize) : TImage(aGraph) {
  name = NULL;
  nPoints = 0;
  mSize = buffSize;
  nPoints = buffSize;
  iPoint = 0;
  data = new float[mSize];
  xgraph = aGraph;    
  width = 0;
  height = 0;
}

/**************************************************************************************/

void TGraph::SetDomain(float xmin, float xmax, float dx) {
  Xmin = xmin; Xmax = xmax; dX = dx; nPoints = (int)((Xmax-Xmin)/dX);
  nPoints = ( nPoints < 3 ) ? 3 : nPoints;
  if(width>0) width = (width/nPoints)*nPoints;
}

/**************************************************************************/

void TGraph::SetIndex( int index ) {
  if(index<0) {
    if( xgraph->ngraphs == MAX_GRAPHS ) fIndex0 = 0;
    else fIndex0 = xgraph->ngraphs++;
  }
  else if(index<MAX_GRAPHS) fIndex0 = index;
  else fIndex0 = 0;
  fIndex1 = xgraph->graph_cnt[fIndex0]++;
}

/**************************************************************************/

void TGraph::SetSize( int aWidth, int aHeight ) {

  width = aWidth; height = aHeight;
  int xStart, yStart, buff=30;
	
  if( xgraph->xOffset[1] == 0 ) xStart = buff;
  else xStart = xgraph->xOffset[1] + 2*buff;  
  xgraph->xOffset[1] += (width + buff);
  yStart = xgraph->yOffset + buff;
  //    c0 = xStart;
  r0 = yStart + xgraph->graph_cnt[fIndex0]*12;
  c0 = 30 + fIndex0*(width+buff);
  if(dFile) fprintf(dFile,"\n Xgraph Set Size: Origin: (%d,%d) // Xgraph offset: ((%d,%d),%d) // Size(w,h): (%d,%d)",
		    c0,r0,xgraph->xOffset[0],xgraph->xOffset[1],xgraph->yOffset,width,height);
}


/**************************************************************************/

void TGraph::SetRange(float ymin, float ymax) {
  Ymin = ymin; Ymax = ymax;
  if(dFile) fprintf(dFile,"\n TGraph Set Range : %f %f",ymin,ymax);
}

/**************************************************************************/

void TGraph::doXEvent(XEvent& event) {

  if (event.type == ButtonPress) {
    if(dFile) {
      fprintf(dFile,"TGraph Processing ButtonPress in (%d,%d),(%d,%d)\n", c0, r0, c0+width, r0+height );
    }
    ProcessButtonPress( event.xbutton.x, event.xbutton.y, event.xbutton.button );
  }
  if (event.type == KeyPress && xgraph->currentImage == this ) {
    char buffer[20];
    int bufsize=20;
    KeySym key;
    XComposeStatus compose;
    XLookupString( (XKeyEvent*)(&event),buffer,bufsize,&key,&compose);
    if( buffer[0] == 'b' ) { Resize(1.50); }
    if( buffer[0] == 's' ) { Resize(0.66); }
    if( buffer[0] == 'u' ) { Rescale(1.50); }
    if( buffer[0] == 'd' ) { Rescale(0.66); }
  }
}


/**************************************************************************/

void TGraph::ProcessButtonPress( int x, int y, int button ) {
  char msg[100]; float fval; 
  if( x>c0 && x < c0+width  && y>r0 && y < r0+height) {
    GetValue(x-c0,&fval);
    sprintf(msg,"Value at (%d,%d) = %f",x-c0,y-r0,fval);
    xgraph->text(c0,r0+height+10,msg,1);
    xgraph->currentImage = this;
  }
}

/**************************************************************************/

int TGraph::GetValue( int x, float* value )
{
  int npoints = (nPoints>width) ?  width : nPoints;
  int npp = width/npoints;
  int ix = x/npp;
  if( ix < 0 || ix >= npoints) { 
    fprintf(dFile,"TGraph: Data index out of range in GetValue: %d\n",ix); 
    *value = 0.0;
    return 0; 
  }
  *value = data[ix];
  return 1;	
}

/**************************************************************************/

void TGraph::DisplayGrid(int isTemp) {
  char label[50];
  if(fIndex1 == 0) {
    xgraph->rectangle(c0, r0, width, height,isTemp);
    sprintf(label,"%.2f",Xmin);
    xgraph->text(c0, r0+height+10, label, isTemp);
    sprintf(label,"%.2f",Xmax);
    xgraph->text(c0+width, r0+height+10, label, isTemp);
  }
  SetColor();
  sprintf(label,"%.2f",Ymax);
  xgraph->text(c0-30, r0+fIndex1*12, label, isTemp);
  sprintf(label,"%.2f",Ymin);
  xgraph->text(c0-30, r0+height+fIndex1*12, label, isTemp);
  sprintf(label,"%s",name);
  int len = strlen(name);
  int r0s = r0 -  xgraph->graph_cnt[fIndex0]*12;
  xgraph->text(c0+10, r0s + fIndex1*12, label, isTemp);

  //	if(xgraph->setForeground("blue")) xgraph->filledRectangle(c0+1, r0+1, width-2, height-2,isTemp);
  //	xgraph->setForeground("white");
}

/**************************************************************************/

void TGraph::Flush(int isTemp) {

  DisplayGrid(isTemp);
  if(iPoint>1) {	
    int y0, y1, x0, x1;
    int npoints = (iPoint>width) ?  width : iPoint;
    int npp = width/(npoints);
    y0 = (r0+height) - ScalePoint(data[0]);
    x0 = c0+1;
    for( int i=1; i<npoints; i++ )  {
      y1 = (r0+height) - ScalePoint(data[i]);
      x1 = x0 + npp; 
      xgraph->line(x0, y0, x1, y1, 2);
      x0 = x1; y0 = y1;
    }
  }
  xgraph->flush();
}

/**************************************************************************/

TGraph::~TGraph() {
  if(name) delete[] name;
  if(nPoints) delete[] data;
}

/**************************************************************************/

void TGraph::SetName( const char* aName )
{    
  int nLen = strlen(aName);
  name  = new char[nLen+1];
  strcpy(name,aName);
}
/**************************************************************************/

float clamp ( float y, float ymax, float ymin)  {
  float tmp = ( y > ymax ) ? ymax : y;
  tmp = ( tmp < ymin ) ? ymin : tmp;
  return tmp;
}
int iclamp ( int y, int ymax, int ymin)  {
  int tmp = ( y > ymax ) ? ymax : y;
  tmp = ( tmp < ymin ) ? ymin : tmp;
  return tmp;
}

/**************************************************************************/

int TGraph::ScalePoint(float y) {
  float sy = ( (y-Ymin) / (Ymax-Ymin) );
  sy = clamp(sy,1.0,0.0);
  int rv = (int)(sy*height); 
  //	if(dFile) fprintf(dFile,"\n TGraph Scale Point: (%f->%f): %d",y,sy,rv);
  return rv; 
}

/**************************************************************************/

void TGraph::FlushPoint(int isTemp) {

  static int y0, y1, x0, x1;
  int npoints = (nPoints>width) ? width : nPoints;
  int npp = width/(npoints);
  npp = (npp < 1) ? 1 : npp;
  int i = iPoint;
  if(i<2) return;
  y0 = (r0+height) - ScalePoint(data[i-2]);
  x0 = c0+(i-2)*npp;
  y1 = (r0+height) - ScalePoint(data[i-1]);
  x1 = c0+(i-1)*npp; 
  SetColor();
  xgraph->line(x0, y0, x1, y1, isTemp);
  xgraph->setForeground("white");
}

/**************************************************************************/

void TGraph::SetColor( ) {
  if(fIndex1==0) xgraph->setForeground("white");
  else if(fIndex1==1) xgraph->setForeground("red");
  else xgraph->setForeground( ((fIndex1-1)*100) % 256 );
}

/**************************************************************************/

void TGraph::AddPoint( float y ) {
  if( iPoint >= mSize ) {
    float* ndata = new float[ iPoint+100 ];
    for( int i=0; i<iPoint; i++ ) ndata[i] = data[i];
    delete[] data;
    data = ndata;
    mSize = iPoint+100; 		
  }
  data[ iPoint++ ] = y;
}

/**************************************************************************/
void TGraph::AddPoint( float y, float t ) {
  if( iPoint >= mSize ) {
    float* ndata = new float[ iPoint+100 ];
    for( int i=0; i<iPoint; i++ ) ndata[i] = data[i];
    delete[] data;
    data = ndata;
    mSize = iPoint+100; 		
  }
  data[ iPoint++ ] = y;
}

/**************************************************************************/

void TGraph::putDisplay(Window w, int xStart, int yStart, int aWidth, int aHeight) {
  Flush();
}

/**************************************************************************/

void TGraph::SetData(  float* idata, int npoints )
{
  if( mSize <= npoints ) {
    delete[] data;
    data = new float[npoints+1];
    mSize = npoints+1;
  }
  for(int i=0; i<npoints; i++) {
    data[i] = idata[i];
  }
  nPoints = iPoint = npoints;
  dX = (Xmax-Xmin)/npoints;
}

/**************************************************************************/

void TGraph::Resize(float factor) {

  SetSize((int)(width*factor),(int)(height*factor));
  Flush();
}

/**************************************************************************/

void TGraph::Rescale(float factor) {

  SetRange(Ymin*factor,Ymax*factor);
  Flush();
}

/**************************************************************************/


/**************************************************************************************/
/* TSheet  */
/**************************************************************************************/

TSheet::TSheet( Xgraph* aGraph) : TImage(aGraph) {
  for(int i=0; i<MAX_S_COLS; i++) name[i] = NULL;
  mSize = 0;
  data = NULL;
  xgraph = aGraph;    
  width = 0;
  height = 0;
  rows = 0;
  cols = 0;
  ysp=10;
  xsp=70;
  SetDomain(0.0,1.0);
}

/**************************************************************************************/

void TSheet::SetDomain(float xmin, float dx) {
  Xmin = xmin; dX = dx;
}
/**************************************************************************************/

#define DINDEX(ci,ri) ((ci)+(ri)*(cols))

int TSheet::AddColumn(char* aName) {

  if(data != NULL) return -1;
  int nLen = strlen(aName);
  name[cols]  = new char[nLen+1];
  strcpy(name[cols++],aName);
  width = (cols+1)*xsp;
  cIndex = cols-1;
  return cIndex;
}

void TSheet::SetRange(int nPoints, float dt) {
  if(data != NULL) return;
  Xmin = 0; dX = dt; rows = nPoints;
  height = (ysp)*(nPoints+1);
}

void TSheet::FlushPoint(float value, int colIndex, int rowIndex, Bool doFlush ) {
  if(data == NULL) {
    mSize = rows*cols;
    data = new float[mSize+1];
    for ( int ri=0; ri<rows; ri++) {
      for(int ci=0; ci<cols; ci++) {
	data[ DINDEX(ci,ri) ] = 0.0;
      }
    }
  }
  data[ DINDEX(colIndex,rowIndex) ] = value;
  if(doFlush) Flush(rowIndex);
}

/**************************************************************************/

void TSheet::SetSize( int aWidth, int aHeight ) {

  if(aWidth>0) width = aWidth; 
  if(aHeight>0) height = aHeight;
  int xStart, yStart, buff=40;
	
  xStart = xgraph->xOffset[1] + buff; 
  yStart = xgraph->yOffset + buff;
  c0 = xStart;
  r0 = yStart;

  if(dFile) fprintf(dFile,"TSheet Set Size: Origin: (%d,%d) // Xgraph offset: (%d,%d) // Size(w,h): (%d,%d)",c0,r0,xgraph->xOffset[0],xgraph->xOffset[1],width,height);
}


/**************************************************************************/

void TSheet::doXEvent(XEvent& event) {

  if (event.type == ButtonPress) {
    if(dFile) {
      fprintf(dFile,"TSheet Processing ButtonPress in (%d,%d),(%d,%d)\n", c0, r0, c0+width, r0+height );
    }
    ProcessButtonPress( event.xbutton.x, event.xbutton.y, event.xbutton.button );
  }
  if (event.type == KeyPress && xgraph->currentImage == this ) {
    char buffer[20];
    int bufsize=20;
    KeySym key;
    XComposeStatus compose;
    XLookupString( (XKeyEvent*)(&event),buffer,bufsize,&key,&compose);
  }
}


/**************************************************************************/

void TSheet::ProcessButtonPress( int x, int y, int button ) {
  char msg[100]; float fval; 
  if( x>c0 && x < c0+width  && y>r0 && y < r0+height) {
    xgraph->currentImage = this;
  }
}

/**************************************************************************/

int TSheet::GetValue( int x, int y, float* value )
{
  int ix = y*cols + x;
  if( ix < 0 || ix >= rows*cols) { 
    fprintf(dFile,"TSheet: Data index out of range in GetValue: %d\n",ix); 
    *value = 0.0;
    return 0; 
  }
  *value = data[ix];
  return 1;	
}

/**************************************************************************/

void TSheet::Flush( int frows, int isTemp ) {

  int y0, x0;
  char label[50];
  if(frows == -1) frows = rows;
  xgraph->rectangle(c0-1, r0, width, height+1,2);

  for( int j=1; j<=cols; j++ )  {
    sprintf(label,"%s",name[j-1]);
    x0 = c0+xsp*j;
    xgraph->text(x0, r0-10-10*((j%2)-1), label, isTemp);
  }

  for( int i=0; i<frows; i++ )  {
    y0 = r0+ysp*(i+1);
    x0 = c0;
    sprintf(label,"%.2g",Xmin+i*dX);
    xgraph->text(x0, y0, label, 2);
    for( int j=1; j<=cols; j++ )  {
      x0 = c0+xsp*j;
      sprintf(label,"%.4g", data[ DINDEX(j-1,i) ] );
      xgraph->text(x0, y0, label, 2);
    }
  }
  sprintf(label,"%s",name[0]);
  xgraph->text(c0+20, r0-1, label, 2);
  xgraph->flush();
}

/**************************************************************************/

TSheet::~TSheet() {
  for(int i=0; i<MAX_S_COLS; i++) if(name[i]) delete[] name[i];
  delete[] data;
}

/**************************************************************************/

void TSheet::SetName( const char* aName )
{    
  int nLen = strlen(aName);
  name[0]  = new char[nLen+1];
  strcpy(name[0],aName);
}
/**************************************************************************/

void TSheet::putDisplay(Window w, int xStart, int yStart, int aWidth, int aHeight) {
  Flush();
}

/**************************************************************************/

void TSheet::SetData(  float* idata, int irows, int icols )
{
  rows = irows;
  cols = icols;
  int npoints = rows*cols;
  if( mSize <= npoints ) {
    if(data != NULL) delete[] data;
    data = new float[npoints+1];
    mSize = npoints+1;
  }
  for(int i=0; i<npoints; i++) {
    data[i] = idata[i];
  }
  SetSize( (cols+1)*xsp, rows*ysp );
}

/**************************************************************************/

#endif
