#include "CVariable.h"
#include "ActivationLayer.h"
#include "SSModule.h"
#include "TPipe.h"
#include "MemoryMgr.h"
#include "SSModel.h"
#include "HDepParm.h"
#include "HDFMgr.h"
#include "Externals.h"
#include "GOFCalc.h"
#include "SimLib.h"

FlagType	FFluxOriginMap ;		//  Flux Origin has different Grid
FlagType	FFluxDestinationMap ;		//  Flux Destination has different Grid
FlagType	FMapRemoteData ;		//  Input Var and Connection have different Grids
FlagType	FICFlux  ;		//  calls ICFlux method
FlagType	FAllocated ;		//  Allocate() Method has been called.
FlagType	FEdgesLinked;		//  LinkEdges() Method has been called since last update.

float maxCombiner(float f1, float f2 ) { return (f1 > f2) ? f1 : f2; }
float minCombiner(float f1, float f2 ) { return (f1 < f2) ? f1 : f2; }

#ifdef POSTGRESS

TDBaseSocket::TDBaseSocket(const char* name) : TNamedObject(name) { 
	sme_pg95* db = TModel::I().DataBase();
	if(db) {
		const CString& port = TModel::I().DataBase()->port();
		const CString& host = TModel::I().DataBase()->host();
		fDBase = new sme_pg95(name,0,port,host); 
	} else {
		fDBase = new sme_pg95(name); 	
	}
}

#endif

//========================================================================================
// 				CLASS TVariable
//========================================================================================

byte TVariable::kNoDbgCnfg = 255;
TNamedObjectList  TVariable::fDBaseList("Variable DBase List",8);
float TVariable::fErrorVal = floatVec::ErrorVal();

TVariable::TVariable(const char* name) : Variable ( name )
{ 
	SetCInfo(kType,kNULL); 
	SetCInfo(kInitMode,kDefault); 
	SetCInfo( kDebug, kNoDbgCnfg );	
  SetF( FisOptimizable, True, "TVariable constructor" );
}

void TVariable::SetupInternals() {
  if( GetF(FisStateVar) && !GetF(FisImport) ) {
	int rank =  TModel::I().addStateVariable( *this );
  } 	
  AddUserDeps(); 
}

void TVariable::AddFlags() {
	Variable::AddFlags();
  AddFlag("FEdgesLinked",FEdgesLinked);
  AddFlag("FFluxDestinationMap",FFluxDestinationMap);
  AddFlag("FFluxOriginMap",FFluxOriginMap);
  AddFlag("FMapRemoteData",FMapRemoteData);
  AddFlag("FAllocated",FAllocated);
	AddFlag("FICFlux",FICFlux);
}

int  TVariable::Config( TConfigData& cd) {
  const CString& cmd = cd.Cmd();
  if(gDebug) { 
    sprintf(gMsgStr,"Reading Variable Config for %s: Nargs = %d, CMD = %s",
	    fName.chars(), cd.NArgs(), cmd.chars() );	
    gPrintLog();  
  }
  int iarg;
  TPipeCommand* pc = NULL;
  Pipe* aPipe = TPipe::GetPipeFromConfig( cd, this, pc ); 
  if(  aPipe != NULL ) {
	if(gDebug) {  sprintf(gMsgStr,"Processing pipe cmd" ); gPrintLog(); }
	if(pc==NULL) {
	  pc = TPipeCommand::CreatePipeCommand( cd, aPipe, this );
	  AddCommand( pc );
	}
	aPipe->SetPipeCommand(pc);    
  } else {
	if(gDebug) {  sprintf(gMsgStr,"Processing var cmd" ); gPrintLog(); }
    if( cmd == "d" ) {
      if( cd.IntArg(0,iarg) > 0 ) SetCInfo( kDebug, iarg );
    } else if ( cmd == "vi" ) {
      if( cd.IntArg(0,iarg) > 0 ) TModel::I().AddIVar(iarg,this);
    } else if( cmd == "EI" ) {
	  int rank = -1;
      if( !GetF(FisImport) ) { 
		rank =  TModel::I().addStateVariable( *this );
	  } else {
		if( fConnection == NULL ) {
		  sprintf(gMsgStr,"Attempt to add up unconnected input to State Vector: %s", Name() ); gFatal();
		}
		rank =  TModel::I().addStateVariable( *fConnection );
	  }
	} else return Variable::Config(cd);
  }
  return 1;
}


#ifdef POSTGRESS
          
sme_pg95* TVariable::GetDBase(const char* name) {
	TDBaseSocket* db = (TDBaseSocket*) fDBaseList.GetObjectByName(name);
	if( db == NULL ) db = new TDBaseSocket(name);
	fDBaseList.append(*db);
	return db->DBase();
}

#endif

//========================================================================================
// 				CLASS CVariable
//========================================================================================

CVariable::CVariable(const char* name) 
  : TVariable ( name )    
{
  SetCInfo(kType,kNULL); 
  SetCInfo(kInitMode,kDefault);
  fMaxDelay = 0.0; 
  fCatMapPtr = NULL;
  fValueMode = kValueNonSpatial;
  fUpdateMode = kTemporal;
  for(int i=0; i<4; i++) fMaps[i] = NULL;
  for(int j=0; j<8; j++) { _Array_Offests[j] = _Array_Sizes[j] = -1; }
	fClass = kNULL;
	fActivationLayer = -1;
}  

void CVariable::Update( float value ) {
	fTHistory.Update(gTime(),value);
	if( !isValidValue( value ) ) { ProcessBoundsError( value, NULL , 'p' ); }
#ifdef DEBUG
  if( GetCInfo(kDebug) ) {
    DoBoundsCheck(value);
    PrintUpdateMsg(value,value,"UPDATE");
  }
#endif
}

float CVariable::ArraySum() { 
  float* data = fSHistory.LastData();
  int dsize = getArraySize();
  float rv = 0;
  for( int i=0; i<dsize; i++ ) {
	rv += data[i];
  }
  return rv;
}

void CVariable::Dump( char* msg, int pl0, int pl1, FILE* file, int s0, int s1 ) { 
	if( GetF(FisSpatial) ) {
		CString result = "\n\n  " + SName() + " Data Dump ( " + msg + " ):";	
		Point2 pt(pl0,pl1);
		int iproc = Grid().GetOwner( pt );
		int layer_index = -1;
		if( gIProc == iproc ) {
			for ( int p0 = pl0; p0 < pl0+s0; p0++ ) {
				result += "\n  ";
				for ( int p1 = pl1; p1 < pl1+s1; p1++ ) {
					float value = Value(p0,p1,layer_index);
					if( value == floatVec::ErrorVal() ) {
						sprintf(gMsgStr,"    *     ");
					} else {
						sprintf(gMsgStr,"%9.2e ",value);
					}
					result += gMsgStr;
				}
			}
		}
		result += "\n\n";
		result.ProcSend(iproc,0);
		fputs( result.chars(), file );
		fflush( file );
	} else { gPrintErr(" NonSpatial Variable." ); }
}
			
float CVariable::ArraySum(int index0, int index1, int index2, int index3) { 
  float* data = fSHistory.LastData();
  int dsize = getArraySize();
  float rv = 0.0;
  int i[4], wc[4];
  if( index0 == gWildCardIndex ) { wc[0]=1; i[0]=0; } else { wc[0]=0; i[0]=index0; }
  if( index1 == gWildCardIndex ) { wc[1]=1; i[1]=0; } else { wc[1]=0; i[1]=index1; }
  if( index2 == gWildCardIndex ) { wc[2]=1; i[2]=0; } else { wc[2]=0; i[2]=index2; }
  if( index3 == gWildCardIndex ) { wc[3]=1; i[3]=0; } else { wc[3]=0; i[3]=index3; }
  
  while(1) {
  
	rv += ArrayValue(i[0],i[1],i[2],i[3]);
	
	int count = 1;
	for( int j=0; j<4; j++ ) {
	  if( wc[j] && count ) {			  
		if( ++i[j] == _Array_Sizes[j] ) {
		  i[j] = 0;
		} else {
		  count = 0;
		}			  
	  } 
	}
	if( count ) { break; }
  }	
  	  
  return rv;
}

float CVariable::ArrayMean() { 
  float* data = fSHistory.LastData();
  int dsize = getArraySize();
  float rv = 0;
  for( int i=0; i<dsize; i++ ) {
	rv += data[i];
  }
  return rv/dsize;
}

float CVariable::ArrayMean( int index0, int index1, int index2, int index3 ) { 
  float* data = fSHistory.LastData();
  int dsize = getArraySize();
  float rv = 0.0;
  int i[4], wc[4], nelem = 0; 
  if( index0 == gWildCardIndex ) { wc[0]=1; i[0]=0; } else { wc[0]=0; i[0]=index0; }
  if( index1 == gWildCardIndex ) { wc[1]=1; i[1]=0; } else { wc[1]=0; i[1]=index1; }
  if( index2 == gWildCardIndex ) { wc[2]=1; i[2]=0; } else { wc[2]=0; i[2]=index2; }
  if( index3 == gWildCardIndex ) { wc[3]=1; i[3]=0; } else { wc[3]=0; i[3]=index3; }
  
  while(1) {

	rv += ArrayValue(i[0],i[1],i[2],i[3]);
	nelem++;
	
	int count = 1;
	for( int j=0; j<4; j++ ) {
	  if( wc[j] && count ) {			  
		if( ++i[j] == _Array_Sizes[j] ) {
		  i[j] = 0;
		} else {
		  count = 0;
		}			  
	  } 
	}
	if( count ) { break; }
  }	
  return rv/nelem;
}

CVariable* CVariable::GetSimilarVariable(char* name) {
	CVariable* rv = NULL;
	switch ( fClass ) {
		case kState:
			rv = new CStateVariable(name);
			break;
		case kFlux:
			rv = new CFluxVariable(name);
			break;
		case kAux:
			rv = new CAuxVariable(name);
			break;
	}
	if( GetF(FisSpatial) ) { 
	  rv->SetF(FisSpatial,True,"GetSimilarVariable"); 
	}
	rv->SSInit(fModule);
	rv->SetupInternals();
	rv->Allocate();
	return rv; 
}

float CVariable::Delay( int array_index[4], float dt, float ival ) {
  float dTime = gTime()-dt;
  const float* fdata;
  if( dTime <= 0.0 ) { fdata = fSHistory.LastData(); }
  else if( dTime < fTHistory.FirstTimeStamp() ) {
	  if( ival < FLT_MAX ) { return ival; }
	  else { fdata = fSHistory.FirstData(); }
  } else {
	fdata = fSHistory.Data(dTime);
  }     
  return fdata[ getArrayIndex(array_index) ];
}

float CVariable::Delay( float dt, float ival ) {
  float dTime = gTime()-dt;
  if( dTime <= 0.0 ) { return fTHistory.LastValue(); }
  else if( dTime < fTHistory.FirstTimeStamp() ) {
		if( ival < FLT_MAX ) { return ival; }
		else { return fTHistory.FirstValue(); }
	} else {
	return fTHistory.Value(dTime);
  }     
}


void CVariable::PrintUpdateMsg(float value, float dv, char* label ) {

  if( GetCInfo(kDebug)>1 ) {
    sprintf(gMsgStr,"\n%s %s (%f) by %f -> %E",(const char*)fName,label,gTime(),dv,value); 
    gPrintLog();
    PrintFlags();
  }
}

void CVariable::SetupInternals() {
  TVariable::SetupInternals();
  if( GetF( FisSpatial ) ) {
	int op = TModel::I().GetCInfo(Model::kOptLevel);
	if( ( op > 0 ) && GetF( FisOptimizable ) ) {
	  fUpdateMode = kFastSpatial;
	} else {
	  fUpdateMode = kSpatial;
	}
	fValueMode = kValueSpatial;
  } 
  if( GetF(FisIgnored) ) {
		int first_cmd = 0;
		for( Pix p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
			TCommand &c0 = (TCommand&) fCommandList(p0);
			TCommand::EType type =  (TCommand::EType) c0.GetObjInfo(TCommand::kType);
			TCommand::EExType extype =  (TCommand::EExType) c0.GetObjInfo(TCommand::kExType);
			if( (extype == TCommand::kCode) && (type != TCommand::kIntegrate) && (first_cmd==0) ) {
				TExecuteCommand ec = (TExecuteCommand&) c0;
				ec.SetExecutable(&IGNORE_exe_g); 
				ec.Code() = fDefValue.elem(0); 
				ec.SetObjInfo( TCommand::kStatus, TCommand::kActive); 
				first_cmd = 1;
			} else {
				c0.SetObjInfo( TCommand::kStatus, TCommand::kInActive); 
			}
		}
		if( first_cmd == 0 ) { 
			sprintf(gMsgStr,"Can't find executable command for ignored variable %s",Name()); gPrintErr();
		}
	}
}

int CVariable::addArrayDeclaration( ArrayRec* array, int array_index, int is_dimension_name, int global_index ) {
	fValueMode = kValueNonSpatialArray;
	if( !GetF(FisImport) ) { 
	  for( Pix p = fConnectionList.first(); p; fConnectionList.next(p))  {
		CVariable& aVar = (CVariable&) fConnectionList(p);
		aVar.addArrayDeclaration( array, array_index, is_dimension_name, global_index ); 
	  }
	}
	return Variable::addArrayDeclaration( array, array_index, is_dimension_name, global_index );
}

void CVariable::MakeStatusDataString( CString& str, Bool append ) {
  byte initmode = GetCInfo(kInitMode);
  byte isspatial = GetF(FisSpatial);
  byte isimport = GetF(FisImport);
  if( append ) {
	str += (isspatial) ? '1' : '0';
  } else {
	str = (isspatial) ? '1' : '0';
  }
  str += (GetF(FisConstant)) ? '1' : '0';
  str += (GetF(FisStateVar)) ? '1' : '0';
  str += (GetF(FisFlux)) ? '1' : '0';
  str += (isimport) ? '1' : '0';
  str += (GetF(FisExport)) ? '1' : '0';
  str += (GetF(FisClamped)) ? '1' : '0';
  str += (GetF(FDualConnection)) ? '1' : '0';
//  str += (GetF(FHasNegFlux)) ? '1' : '0';
  str += (char) ('0' + GetCInfo(kMode)     );
  str += (char) ('0' + initmode );
  str += (char) ('0' + GetCInfo(kType)     );
  float value = ( (initmode == kParameter) &&  !isspatial && !isimport ) ? Value() : 0.0;
  str +=  value;
}

int CVariable::readGeometryFile( const CString& pathName, LayerConfig* lc ) {
	if( lc == NULL ) { 
	  sprintf(gMsgStr,"Undefined LayerConfig in CVariable::readGeometryFile: %s", pathName.chars()); gFatal(); 
	} else {
	  sprintf(gMsgStr,"Reading GeometryFile(rows=%d,cols=%d,%d): %s", lc->Dim(0), lc->Dim(1), lc->Dim(2), pathName.chars() ); gPrintScreen(); 
	}
	int Nx = lc->Dim(1); // cols
	int Ny = lc->Dim(0); // rows
	int Nz = lc->Dim(2);
	int tt, ix, iy, jx, jy, jz, index, debug = 1, partitioned_layer = -1;
	FILE *tfile;
	int ncols = Nx*Ny;
	byte *in = new byte[2*Nx];
	byte *inptr, t1, t2;
	TPartition2* p = fCov.GetPartition(partitioned_layer);

	if (gIProc == 0) {
		tfile = fopen (pathName.chars(), "r");
	}
	if ((tfile == 0) && (gIProc == 0)) {
		gFatal ("fopen on geometry failed.\n");
	}

	int go = 1;	
	for (jz = 0; jz < Nz; jz++ ) if( go ) {
		for (jy = 0; jy < Ny; jy++) if( go ) {
			if ( gIProc == 0 ) {
			  int test = fgetc(tfile); ungetc(test,tfile);
			  if( test != EOF ) {
				fread (in, sizeof (char), 2 * Nx, tfile);
			  } else { 
				go = 0; 
#ifdef USE_MPI	
				MPI_Bcast (&go, 1, MPI_INT, 0, MPI_COMM_WORLD);
#endif          
				break;
			  }
			}
#ifdef USE_MPI	
			MPI_Bcast (in, 2 * Nx, MPI_CHAR, 0, MPI_COMM_WORLD);
#endif
			inptr = in;
			for (jx = 0; jx < Nx; jx++) {
				t1 = *inptr++;
				t2 = *inptr++;
				tt = t1 * 256 + t2;
				if( (p==NULL) || (p->getowner( jy, jx ) == gIProc) ) {
				  SetValue( jy, jx, (float) tt, jz ); 

				  if( (debug) && (jz == 0) ) {
					  fprintf(Env::LogFile(),"\n(%3i, %3i, %3i): %f", jx,jy,jz, (float) tt  );
				  }
				}
			}
		}
	}
	if (gIProc == 0) {
	  fclose (tfile);
	}		
	delete[] in;
	return 0;
}

void CVariable::InitValue(float value) {
     fTHistory.Add( gTime(), value );
    if(GetCInfo(kDebug)) {
#ifdef DEBUG
      DoBoundsCheck(value);
#endif 
      PrintUpdateMsg(value,value,"INIT");
    }
}

inline int iabs( int ival ) { return (ival<0) ? -ival : ival; }

void CVariable::UpdateValueExternal( const char* external_mod_name,  const char* external_var_name, EVarLink::EModality modality ) {
  ActivationLayer* aL = fCov.activationLayer();
  int32 offset, i, ncells = aL->nCells();
  int init = ((modality == EVarLink::kInitIn) || (modality == EVarLink::kInitOut)); 
#ifdef OM3
  if( gDebug ) {
	sprintf( gMsgStr, "UpdateValueExternal: %s:%s -> %s, mode: %d, time: %f", external_mod_name, external_var_name, Name(), modality, gTime() );  
	gPrintLog();
	if( gIProc == 0 ) { gPrintScreen(); }
  }
  if( Util::scmp(external_mod_name,"om3") == 0 ) {
	if( om3_set_current_data_array( external_var_name, init ) == 0 ) {
	  sprintf( gMsgStr, "Can't identify external var %s:%s in %s external update", external_mod_name, external_var_name, Name() );  
	  gPrintErr(); 
	  return;
	}
	switch( modality ) {
	  case EVarLink::kInitIn: 
	  case EVarLink::kIn: 
		for( i=0; i<ncells; i++ ) {
		  TCell* c = aL->cell( i );
		  int iz = (int)c->GetObjInfo( TCell::kLayerIndex );
		  int col = c->col();
		  int row = c->row();
		  float value = om3_get_data_value( col, row, iz  ) ;
		  if( ( value < 0.0 ) && GetF(FisClamped) ) { value = 0.0; }
		  SetValue( c, value ); 
		  if( gDebug > 1 ) {
			if( ( iabs(row-49) < 3 ) && ( iabs(col-31) < 3 ) && (iz == 0) ) {
			  sprintf( gMsgStr, "cell(%d,%d,%d) -> %f", c->col(), c->row(), iz, value );  gPrintLog();
			}
		  }
		}
		break;
	  case EVarLink::kInitOut:  
	  case EVarLink::kOut: 
		for( i=0; i<ncells; i++ ) {
		  TCell* c = aL->cell( i );
		  int iz = (int)c->GetObjInfo( TCell::kLayerIndex );
		  int col = c->col();
		  int row = c->row();
		  float value = Value( c );	
		  om3_set_data_value( col, row, iz, value ); 
		  if( gDebug > 1 ) {
			if( ( iabs(row-49) < 3 ) && ( iabs(col-31) < 3 ) && (iz == 0) ) {
			  sprintf( gMsgStr, "om3 cell(%d,%d,%d) <- %f", c->col(), c->row(), iz, value );  gPrintLog();
			}
		  }
		}
		om3_wrap_boundary();
		break;
	  case EVarLink::kInOut:  
		gPrintErr("Ambiguous mode for UpdateValueExternal: InOut");
		break;
	 }
  }
#endif
}

void CVariable::Map ( CVariable* aMVar ) {
  for(int i=0; i<4; i++) if( fMaps[i] == NULL ) { fMaps[i] = aMVar; break; }
  if( !GetF(FisImport) ) { 
    for( Pix p = fConnectionList.first(); p; fConnectionList.next(p))  {
      CVariable& aVar = (CVariable&) fConnectionList(p);
      aVar.Map(aMVar);
    }
  }
}

int  CVariable::AllocateSpatialVariable( int activationLevel, int linkLayer ) {
	setActivationLevel( activationLevel, linkLayer );
	SetF(FisSpatial,True,"TMultiFrame::Setup");
	SetCInfo(kInitMode,kDefault);
	SetupInternals();
	Allocate();
	return 0;
}


TMultiFrame* CVariable::GetMultiFrame() {
  TMultiFrame* f = (TMultiFrame*) &(TModel::I().Frame());
  if( GetF(FisSpatial) && ( (f==NULL) || !f->hasLayersConfigured() ) ) { 
		sprintf(gMsgStr,"Attempt to allocate spatial variable %s in non-spatial frame.",Name()); gPrintScreen(); 
		sprintf(gMsgStr,"Hint 1: If you are trying to run spatially then you probably forgot to configure a spatial frame in the Config file."); gPrintScreen();
		sprintf(gMsgStr,"Hint 2: If you are trying to run non-spatially then check the Config file for spatial input commands."); gPrintScreen(); 
		sprintf(gMsgStr,"Hint 3: If you are trying to run non-spatially and there are arrays in your Stella model"); gPrintScreen(); 
		sprintf(gMsgStr,"        then you must run with the following switch: SME -ignore_STELLA_arrays run <modelName>.");  gPrintScreen(); 
		gFatal(); 
  }
  return f;
}

int  CVariable::Allocate( ) {
  if( Allocated() || !GetF(FisSpatial) ) return 0;
  TMultiFrame* f = GetMultiFrame();
  if( fActivationLayer < 0 ) { fActivationLayer = fModule->activationLayerIndex(); }
  if( fLinkLayer < 0 ) { fLinkLayer = fModule->linkLayerIndex(); }
  int ghostSize = f->GetParm(TMultiFrame::kMaxGhost);
  int memoryIndex; 
  if( TModel::I().MemoryOp() == 0 ) { memoryIndex = MemObject::kPermanentMemory; }
  else { memoryIndex = ( GetCInfo(kInitMode) ==  kDefault ) ? fModule->Index() : MemObject::kPermanentMemory; }
  
  if( GetF(FisImport) ) {
		if( fConnection == NULL ) {
			sprintf(gMsgStr,"Undefined Connection for input Variable %s:%s", fModule->Name(), Name() );
			gFatal();
		}
    CVariable* svar = (CVariable*)fConnection;
		svar->Allocate();
    if( fActivationLayer == svar->fActivationLayer ) {
      if( gDebug) { 
				sprintf(gMsgStr,"\nImport MAlloc: %s::%s (%d), SameGrid",fModule->Name(),Name(),memoryIndex); 
				gPrintLog(); 
      }
#ifdef DEBUG
			if(gDebug>2){
				sprintf(gMsgStr,"\n &&& Allocating memory, %s:%s -> %s:%s \n",Name(),GetModule()->Name(),svar->Name(),svar->GetModule()->Name()); 
				gPrintLog(); 
			}
#endif      
      fCov.Setup( svar->Cov(), -1, memoryIndex );
    }  else {
			if( GetF( FisConstant ) ) { 
				memoryIndex = MemObject::kPermanentMemory; 
			}
      if( gDebug) { 
				sprintf(gMsgStr,"\nImport MAlloc: %s::%s (%d), RemoteGrid",fModule->Name(),Name(),memoryIndex); 
				gPrintLog(); 
      }
      fCov.Setup( f->Grid(), fActivationLayer, ghostSize, memoryIndex );
      SetF(FMapRemoteData,True,"Allocate");
      if( GetF(FDualConnection) ) svar->SetF(FMapRemoteData,True,"Allocate");
    }
  } else {
    if( GetF( FisExport ) || GetF( FisStateVar)  ) { 
      memoryIndex = MemObject::kPermanentMemory; 
    }
    if( gDebug) { sprintf(gMsgStr,"\nMAlloc: %s::%s (%d)",fModule->Name(),Name(),memoryIndex); gPrintLog(); }
    fCov.Setup( f->Grid(), fActivationLayer, ghostSize, memoryIndex);
  }
  SetF(FAllocated,True,"Allocate");
  ActivationLayer* aL = fCov.activationLayer();     // initializes MultiCoverage::fNCells
  return 1;
}

void CVariable::PrintUpdateMsg( TCell* p0, float value, float dv, char* label, TCell* pR ) {
/*
  const OrderedPoint& pt = Grid()->GetPoint(p0);
  byte dMode = pt.GetObjInfo(DistributedGrid::kDebug);
  if( dMode ) {
    switch(dMode) {
    case 1: if( GetCInfo(kDebug) > 1 ) {
      OrderedPoint& pr = (OrderedPoint&)((pR) ? Grid()->GetPoint(pR) : pt);
      sprintf(gMsgStr,"\n(%d,%d)%s %s by %f -> %f at (%d,%d): ",
	      pt(0),pt(1),label,(const char*)fName,dv,value,pr(0),pr(1)); 
      gPrintLog();
      PrintFlags();
    }
    break;
    case 2: 
      OrderedPoint& pr = (OrderedPoint&)((pR) ? Grid()->GetPoint(pR) : pt);
      sprintf(gMsgStr,"\n(%d,%d)%s %s by %f -> %f at (%d,%d):  ",
	      pt(0),pt(1),label,(const char*)fName,dv,value,pr(0),pr(1)); 
      gPrintLog();
      PrintFlags();
      break;
    }
  }
*/
}

void CVariable::CopyData(CVariable& v) {
	fCov = v.fCov;
}

void CVariable::AddData(CVariable& v) {
	fCov += v.fCov;
}	

int CVariable::GraphData( const char* timeseries_name, const char* graph_name, float* fd, int ds, Bool overwrite ) {
  if( graph_name == NULL ) { graph_name = Name(); }
  TGraphPipe::GraphData( this, graph_name, timeseries_name, fd, ds, overwrite );
  return 0;
}

int CVariable::DisplayData( const char* display_name, float* fd, float fmax, float fmin, int s0, int s1, int s2 ) {
  if( display_name == NULL ) { display_name = Name(); }
  return TDataDisplayPipe::DisplayData( display_name, fd, fmax, fmin, s0, s1, s2 );
} 


void CVariable::SSInit( Module* mod ) {
	TVariable::SSInit( mod );
	fSHistory.SetStack(((TModule*)mod)->SpatialArrays());
}

void CVariable::UpdateInternals() {
	if( GetF( FisSpatial ) ) {
		GrabMemory();
		TVariable::UpdateInternals();

		if( fMaxDelay > 0 ) {
			fSHistory.Add( gStep(), fCov );
			fSHistory.DeleteBelow( gTime()-fMaxDelay );
		}

		if( GetF(FICFlux) ) fCov.Set(0.0); 

		if( GetF(FLinkEdges) ) { LinkEdges(); }
		else { SetF(FEdgesLinked,False,"UpdateInternals"); } 
	} else {
		TVariable::UpdateInternals();
	}     
}


void CVariable::BroadcastValue() {
  
  for( Pix p = fConnectionList.first(); p; fConnectionList.next(p))  {
    CVariable& aSVar = (CVariable&) fConnectionList(p);
    aSVar.RefreshData();
  }
}

void CVariable::UpdateRel( TCell* p0, float value, int dr, int dc, int dv  ) {
  
  TCell* pR = (TCell*) Grid().Translate( p0, dr, dc, dv );
  
  if( pR ) SetValue(pR,value);
  else return;

	if(GetCInfo(kDebug)) {
		DoBoundsCheck(value,pR);
#ifdef DEBUG  	  
		PrintUpdateMsg(p0,Value(pR),value,"UpdateRel",pR);
#endif
	}
}

void CVariable::collapseVolumeToSurface( CVariable& surfVar, CVariable* cellWeights, int layer_index ) {
	ActivationLayer *aL_vol; 
	if( GetF(FisSpatial) && surfVar.GetF(FisSpatial) ) {
	  aL_vol = fCov.activationLayer();
	  int vol_aL_index = activationLayerIndex();
	  int surf_aL_index = surfVar.activationLayerIndex();
	  int weights_aL_index = -1, use_vol_weights = 0;
	  if( cellWeights != NULL ) {
		weights_aL_index = cellWeights->activationLayerIndex();
	  }
	  if( weights_aL_index == vol_aL_index ) { use_vol_weights = 1; }
	  else if ( weights_aL_index == surf_aL_index ) { use_vol_weights = 0; }
	  else { sprintf(gMsgStr,"Illegal Weights Activation Level in collapseVolumeToSurface: %d", weights_aL_index); gPrintErr(); return; }
	  int32 ncells = aL_vol->nActiveCells();
	  TLayer* surf_layer = surfVar.Cov().multiGrid()->getCellLayer( layer_index );
	  surfVar.Set(0.0);
	  TCell *c, *c1; float value;
	  for( int cindex=0; cindex < ncells; cindex++ ) {
		  c = aL_vol->cell( cindex );  
		  c1 = surf_layer->getCell( c->row(), c->col() );
		  value = Value();
		  if( cellWeights != NULL ) { value = (use_vol_weights) ? value*(*cellWeights)(c) : value*(*cellWeights)(c1); }
		  surfVar.AddValue( c1, Value( c ) );
	  }
	}
}

TMap* CVariable::GetDataArray( EMapClass dataClass, int destProc ) {
  TMap* rv = new TMap( dataClass );
  if( dataClass == kSurfaceMap ) {
	rv->SetDataType(TMap::kFloats);
  } else if( dataClass == kClassMap ) {
	rv->SetDataType(TMap::kInts);	
  }  
  CopyToMap( *rv, destProc );
  return rv;
}

void CVariable::PrettyPrint( CString& result, int start, int end ) {
    switch( fValueMode ) {
    case kValueSpatial: 
	case kValueSpatialArray: 
	case kDBParm: {
	  result = "(  ";
	  float* data = Data(); 
	  int dataLen = ( (end < 0) || (end > DataSize()) ) ? DataSize() : end;
	  for (int i=0; i<dataLen; i++ ) {
		  if( i >= start ) {
			  if( i % 5 == 0 ) result += "\n   ";
			  else result += " ";
			  float fv = fabs(data[i]);
			  if( (fv < 0.001) || (fv>10000)  ) {
				result += format("%e",data[i]); 
			  } else {
				result += data[i]; 
			  }
		  }
	  }
	  result += "\n)\n";
	  } break;
    case kValueNonSpatialArray: {
      fSHistory.PrettyPrint( result, start, end ); 
    } break;
    case kValueNonSpatial: {
      fTHistory.PrettyPrint( result, start, end ); 
    } break;
  }  
}

int CVariable::CopyToMap(  TMap& map, int layerIndex, int destProc ) { 
//	 fC[0].elem(k0) = Model::I0().UTM(0,0);
//	 fC[0].elem(k1) = Model::I0().UTM(0,1);
//	 l->getUTM( 0, fC[0] );  l->getUTM( 1, fC[1] ); 
	EMapType mt = (EMapType) GetObjInfo(TMap::kMapType);
	TMap::EDataType dt = (TMap::EDataType) map.GetObjInfo(TMap::kDataType);
	if( (dt == TMap::kUndefined) || (dt == TMap::kBytes)  ) {
	  if( ( map.NBytes() > 0) && ( mt != kGRASS ) && ( mt != kASCII ) && ( mt != kArc ) && ( mt != kHDF5 )) {
		 map.SetDataType(TMap::kBytes);
	  } else if( map.GetObjInfo(TMap::kMapClass) == kClassMap ) {
		  map.SetDataType(TMap::kInts);	
	  } else {
		  map.SetDataType(TMap::kFloats);
		  map.SetObjInfo(TMap::kMapClass,kSurfaceMap);
	  }
	} 
    EScaleType st = kDefaultScaling;
	if( mt == kPPM ) st = kRainbowScaling;
	return fCov.CopyToMap( map, FALSE, st, layerIndex, destProc ); 
}          

// Copies a Variable's data to the DataSet ( SpaceSim/src/DataSet.h ), 
// filling in all the relevant fields of the DatSet object in the process.  
// The CVariable's data is converted to Raster format ( if DataSet.format = 0 ) in the process, 
// and stored in the DataSet as an array of shorts ranging 0 to MAX_SHORT ( 256*256-1 ).  
// It also fills in the _max and _min fields which represent the actual (float) max 
// and min values of the CVariable data.  The flush_time_buffers is used for 1D (timeseries)
// data.  In this case the method, which is called at each relevant timestep,
//  buffers the data ( for flush_time_buffers == 0) so it can rescale it as a new 
// point is added at each timestep.  If the method is called with (flush_time_buffers == 1)
// then the buffers are flushed and data scaling operation is reset.   Ant 1D dataSet will
// contain timeseries data starting from  the last time the buffers were flushed ( or
// the beginning of the simulation if no flush has occurred. )   The return value gives
// a suggestion as to when to write the resulting DataSet, depending on the CVariable type.
// For constants it will be non-zero only on the first call of CopyToDataSet.
//  For timeseries (1D) data it will be non-zero on the first call of CopyToDataSet and
// subsequently each time the buffers are flushed.  For 2D and 3D data it always is non-zero.
// Formats:
// DataSet.format = 0: raster format     ( required (& default) if NProcs > 1 )
// DataSet.format = 1: compressed format ( SME internal format, like raster format, but skip any 
//                     data not in study area, and place ghost cells at end. )

int CVariable::CopyToDataSet( DataSet& d, int flush_time_buffers, int time_buffer_size, int data_index ) {
		int datasize = 0, local_debug = 0, rv = 1;
		int maxval = 256*256-2;
		ActivationLayer* aL; 
		sprintf(gMsgStr,"CVariable::CopyToDataSet %s:%d", Name(), data_index ); gPrintLog(); 
		switch( fValueMode ) {
		  case kValueSpatial: {
			d.fill(0);
			int format = d.format();
			aL = fCov.activationLayer();
			d.setTimestamp( gTime() - gDT(), gDT() );
			float max = d.max() = getMax();
			float min = d.min() = getMin();
			float nullValue = FLT_MAX;
			int32 index, dims, pindex = 0, memindex = 0, aindex = fCov.activationIndex();
			TLayer* l = fCov.multiGrid()->getCellLayer( data_index );
			TPartition2* p = l->partition();				
			d.setMap( (byte*) p->cellDistribution()->Data() );
			int32 ncells = aL->nActiveCells();
			int32 nLayers = fCov.multiGrid()->nCellLayers();
			int nCellsGlobal = fCov.multiGrid()->getNCells();
			Region2 r; int32 n_grid_pts;
			if( (gIProc == 0) || (format == 1) ) {  
				l->globalRegion(r);
				n_grid_pts = (format == 1) ? nCellsGlobal : r.size()*nLayers; 
			} else {  
				r = *(p->region( gIProc )); 
				n_grid_pts =  r.size()*nLayers;
			}
			d.setDims( dims = ( aL->multiLayers()) ? 3 : 2 );
			d.setSize( r.extents(1), r.extents(0), nLayers, n_grid_pts ); 
			int partition_size =  r.extents(1) * r.extents(0);
			float value, s = maxval/(max - min);
			unsigned short sval;
			for( int cindex=0; cindex < ncells; cindex++ ) {
				TCell* c = aL->cell( cindex ); 
				if( dims > 2 ) { 
					data_index = c->GetObjInfo( TCell::kLayerIndex );
					l = fCov.multiGrid()->getCellLayer(data_index);  
				}
				if(  c->GetObjInfo( TCell::kGhostIndex ) == 0 ) {
					value = Value( c );
					if( (value > -FLT_MAX) && (value < FLT_MAX) ) {
						sval = (unsigned short) ( ( max > min ) ? (( value - min)*s + 1 ) : 1 );
						if( format == 0  ) {
							index = l->getCellPartitionIndex(c) + ((dims<3) ? 0 : partition_size*data_index);
						} else {
							index =  c->getGlobalIndex();
						}
						if( (index < 0) || ( index >= nCellsGlobal ) ) {
						  sprintf(gMsgStr,"CVariable::CopyToDataSet index error: %d (max %d)",index,nCellsGlobal); gPrintErr();
						  int32 cellindex = c->Index();
						  memindex = c->memoryIndex( aindex ); 
						  Point2& pt = l->getCellPartitionPoint( c );
						  sprintf(gMsgStr,"cell(%d,%d,%d) (%d:%d:%d), setting value -> %d (%f)", pt(0), pt(1), data_index, 
																		  cellindex, memindex, index, sval, value ); gPrintScreen(); 
						} else {
						  d.setDataValue( index, sval );
						  if( local_debug > 1 ) {
							Point2& pt = l->getCellLocPoint( c );
							int32 cellindex = c->Index();
							memindex = c->memoryIndex( aindex ); 
							sprintf(gMsgStr,"cell(%d,%d,%d) (%d:%d:%d:%d), setting value -> %d (%f)", pt(0), pt(1), data_index, 
																cellindex, memindex, index, c->getGlobalIndex(), sval, value ); gPrintLog(); 
						  }
						}
					}
				}
			}
			if( local_debug ) { d.check_data( Name() ); }
			if( format == 1 ) {
			  LayerConfig* lc = GetModule()->getLayerConfig();
			  if( (lc->Type() != kVolume) && ( nLayers > 1 ) ) {
				d.setFormat( 2 );
			  }
			}
			
			if( gNProc > 1 ) {
#ifdef USE_MPI
			static unsigned short *dTemp = NULL;
			static int dTempSize = 0;
			int condenseTag  = 3352;
			if( gIProc == 0 ) {
				MPI_Status status; int flag, source, count, msgCnt=1;
				while(1) {
					MPI_Iprobe(MPI_ANY_SOURCE,condenseTag,MPI_COMM_WORLD,&flag,&status);
					if( flag == True ) {
						MPI_Get_count(&status,MPI_UNSIGNED_SHORT,&count);
						source = status.MPI_SOURCE;
						if( count != nCellsGlobal ) {
						  sprintf(gMsgStr,"Data size mismatch for proc %d: expected %d, got %d",source,nCellsGlobal,count); gFatal();
						}
						if( count > dTempSize ) {
							if( dTemp ) { delete[] dTemp; }
							dTemp = new unsigned short[ dTempSize = count ];
						}
						if( local_debug ) { sprintf(gMsgStr,"CondenseToMap: proc %d: Receiving data from proc %d: %d shorts",gIProc,source,count); gPrintLog(); }
						MPI_Recv(dTemp,count,MPI_UNSIGNED_SHORT,source,condenseTag,MPI_COMM_WORLD,&status);
						unsigned short stmp; 
						if( format == 0  ) {
							r = *(p->region( source ));
							unsigned short *sdata = dTemp; 
							int gcols = p->cols(), psize = p->cols()*p->rows();
							for( int iL=0; iL<nLayers; iL++ ) {
								for(int ir=r.lower(0); ir<=r.upper(0); ir++ ) {
									for(int ic=r.lower(1); ic<=r.upper(1); ic++) { 
										pindex =  ic + ir * gcols + iL*psize;
										d.setDataValue( pindex, *sdata++ );
									}
								}
							}
						} else {
							for( int ic=0; ic<nCellsGlobal; ic++ ) {
							  if( (stmp = dTemp[ic]) > 0 ) {
								  d.setDataValue( ic, stmp );
								  if( local_debug > 1 ) { 
									  sprintf(gMsgStr,"cell(%d:%d) setting value -> %d ", source, ic, stmp ); gPrintLog(); 
								  }
							  }
							}
						}
						if(++msgCnt == gNProc) break;
					}
				}
			} else {
					if( local_debug ) { sprintf(gMsgStr,"CondenseToMap: Sending data from proc %d to proc 0: %d shorts", gIProc, d.dataSize() ); gPrintLog(); }
					MPI_Send( d.data(), d.dataSize(), MPI_UNSIGNED_SHORT, 0, condenseTag, MPI_COMM_WORLD );
			}
#endif  
		  }
		} break;
		case kValueNonSpatial: {
		  d.setFormat( 0 );
		  if( gIProc == 0 ) {
			if( GetF(FisConstant)  ) {
				float max = -FLT_MAX, min = FLT_MAX;
				d.setDims( 0 ); 
				if( d.getSize(0) == 0 ) { flush_time_buffers = 1; }    // if const: send first one.
				if(  flush_time_buffers ) {
					d.setTimestamp( gTime(), gDT() ); 
					float tmp, *data = DataStart();
					int i;
					d.setSize( datasize = DataSize() );
					for( i=0; i<datasize; i++ ) {
						tmp = data[i];
						if( tmp > max ) { max = tmp; }
						if( tmp < min ) { min = tmp; }
					}
					float s = maxval/(max - min);
					d.max() = max; d.min() = min;
					for( i=0; i<datasize; i++ ) {
						unsigned short dval = (unsigned short) ( ( max > min ) ? ((data[i] - min)*s + 1) : 1 );
						d.setDataValue( i,  dval );
					}
				}
				rv = flush_time_buffers;
			} else {
				if( fDataBuffer.Size() == 0 ) { 
					d.reset(); 
					d.setDims( 1 );
					if( time_buffer_size == 0 ) {
						d.setSize( fTHistory.Size() );				
						for( Pix p = fTHistory.first(); p;  fTHistory.next(p) ) {
						  TTemporalTimeSlice& tts = fTHistory.elem (p);
						  if( p == fTHistory.first() ) { d.setTimestamp( tts.Time(), gDT() ); }
						  fDataBuffer.add( tts() );					
						}
					} else {
						d.setSize( time_buffer_size );				
						fDataBuffer.add( Value() );
					}
				}
				if( flush_time_buffers ) {
					float tmp, max= -FLT_MAX, min = FLT_MAX; int i;
					datasize = fDataBuffer.Size();
					for( i=0; i<datasize; i++ ) {
						tmp = fDataBuffer[i];
						if( tmp > max ) { max = tmp; }
						if( tmp < min ) { min = tmp; }
					}
					float s = (max == min) ? 1 : maxval/(max - min);
					d.max() = max; d.min() = min;
					fprintf(stderr,"\n\n  debug: graph data %s: bounds(%f,%f), npoints=%d, curval=%f, flush=%d",Name(),max,min,datasize,Value(),flush_time_buffers);
					for( i=0; i<datasize; i++ ) {
						unsigned short dval = (unsigned short) ( ( max > min ) ? ((fDataBuffer[i] - min)*s + 1) : 1 );
						d.setDataValue( i, dval );
					}
					fDataBuffer.clear();				
				} else { rv = 0; }
			}
		}
	  } break;
	  case kValueNonSpatialArray: {
		d.setFormat( 3 );
		d.setTimestamp( gTime(), gDT() ); 
		if( _Array_dimension > 3 ) { _Array_dimension = 3; }
		d.setDims( _Array_dimension );
		int s0 = ( _Array_Sizes[0] < 1 ) ? 1 : _Array_Sizes[0];
		int s1 = ( _Array_Sizes[1] < 1 ) ? 1 : _Array_Sizes[1];
		int s2 = ( _Array_Sizes[2] < 1 ) ? 1 : _Array_Sizes[2];
		int s01 = s0*s1;
		int s012 = s01*s2;;
		d.setSize( s0, s1, s2 ); 
		if( gIProc == 0 ) {
		  float max= -FLT_MAX, min = FLT_MAX; 
		  for( int i2=0; i2<s2; i2++ ) {
			  for(int i1=0; i1<s1; i1++ ) {
				  for(int i0=0; i0<s0; i0++) { 
					  float value = ArrayValue( i0, i1, i2  );
					  if( value > max ) { max = value; }
					  if( value < min ) { min = value; }
					  fDataBuffer.add(value);
				  }
			  }
		  }
		  float s = (max == min) ? 1 : maxval/(max - min);
		  for( int i=0; i<s012; i++ ) {
			  unsigned short dval = (unsigned short) ( ( max > min ) ? ((fDataBuffer[i] - min)*s + 1) : 1 );
			  d.setDataValue( i, dval );
		  }
		  d.max() = max; d.min() = min;
		  fDataBuffer.clear();				
		}
	  } break;
	}
	return rv;
}

float CVariable::RelValue( TCell* p0, int dr, int dc, int dv, EBoundaryCondition bc  ) {
	static int printWarning = 1, print_debug = 0; 	  
  if( GetF(FisSpatial) ) {
		TCell* pR = (TCell*) Grid().Translate( p0, dr, dc, dv );
		if( bc == kTestBC ) return (pR != 0) ? 1.0 : -1.0;
		if( print_debug ) {
			sprintf(gMsgStr,"RelValue: (%d,%d)->(%d,%d):%p(G%d) -> %f ",p0->row(),p0->col(),dr,dc,pR, 
				(pR) ? pR->GetObjInfo( TCell::kGhostIndex ) : 0,  (pR) ? Value(pR) : 0.0 );
			gPrintScreen();
		}
		if(pR != 0)  return Value(pR);
		switch(bc) {
			case kDefaultBC: 
			case kContinuousBC:
				return (p0) ? Value(p0) : 0.0; 
			case kZeroBC: 
			case kPeriodicBC:
				return 0.0;
		}
  } else { 
		if( printWarning ) {
			sprintf(gMsgStr,"Attempt to access neighboring cell in non-spatial variable %s ignored.",Name()); 
			gPrintErr();
			printWarning = 0;
		}
	}
	return 0.0; 
}


float CVariable::InitVal( TCell* p ) {
 
		if( fSHistory.Size() == 0 ) { fSHistory.Add( gStep(), fCov ); }
		int32 index = p->memoryIndex( fActivationLayer );
    return fSHistory.FirstValue(index);
}

float* CVariable::InitValData() {
	if( fSHistory.Size() == 0 ) { fSHistory.Add( gStep(), fCov ); } 
	return fSHistory.FirstData();
}


float CVariable::Delay( TCell* p, float dTime, float ival ) {
	if( GetF(FisSpatial) ) {     
		if( dTime <= 0.0 ) { return Value( p ); }
		else if( fMaxDelay == 0.0 ) {
			fMaxDelay = dTime;
			if( ival < FLT_MAX ) {
				MultiCoverage cov( fCov );
				cov.Set( ival);
				fSHistory.Add( gTime()-dTime, cov, kTS_Deep );
				return ival;
			} else {
				if( fSHistory.Size() == 0 ) { fSHistory.Add( gTime(), fCov, kTS_Deep ); }
				return Value( p );
			}
		}
		else if ( dTime > fMaxDelay ) {
			fMaxDelay = dTime;
			int32 index = p->memoryIndex( fActivationLayer );
			return fSHistory.FirstValue( index );
		}
		else {
			int32 index = p->memoryIndex( fActivationLayer );
			return fSHistory.Value( gTime()-dTime, index );
		}
	} else {
		return Delay(dTime,ival);
	}
}

int  CVariable::Config( TConfigData& cd ) {
  TVariable::Config( cd );
  const CString& cmd = cd.Cmd();
  int iarg; float farg;
    
  if( cmd == "cc" ) {
	if( cd.IntArg(0,iarg) ) {
	  TMultiFrame& f = (TMultiFrame&) TModel::I().Frame(); 
	  f.SetCoordVariable( (byte)iarg, this );
	}
  }

  if( cmd == "b" ) {
    if( cd.IntArg(0,iarg) ) fBuffsize = iarg;
  }
  else return 0;
  return 1;
}      

int  CVariable::SetDataValues( float* data, int layOut, int size ) {
		Model::I0().SetContext(NULL,NULL,this,fModule);
		fCov.SetDataValues( data, layOut, size);
		UpdateInternals();
		UpdateReserve(); 
		if( GetF(FVisible) || (GetF(FRegistered) &&  Model::I0().MemoryOp()) ) {
				Model::I0().QueryInterface();
		} 
		return 0;
}

int  CVariable::BuildConfigXML( ArgArray& msg, int mode ) {
	int rv = 0;
    msg.Add(format("\n\n\t<atom name=\"%s\" id=\"%s\" >\n",Name(),Name()));
	if( GetF(FisDisplayVar ) ) {
	  msg.Add(format("\n\t\t<config name=\"DD\">"));
	  msg.Add(format("\n\t\t</config>"));
	  msg.Add(format("\n\t\t<doc>"));
	  msg.Add(format("\n\t\t%s",fDoc.chars()));
	  msg.Add(format("\n\t\t</doc>"));
	  rv = 1;
	}
	if( GetCInfo(kInitMode) == kParameter ) { 
	  if( fParm.isWebParm() ) {
		msg.Add(format("\n\t\t<config name=\"pm\">"));
		msg.Add(format("\n\t\t\t<arg name=\"value\"> %f </arg>",fValue));
		msg.Add(format("\n\t\t\t<arg name=\"pmax\"> %f </arg>",fParm.Max_Value()));
		msg.Add(format("\n\t\t\t<arg name=\"pmin\"> %f </arg>",fParm.Min_Value()));
		msg.Add(format("\n\t\t\t<arg name=\"label\"> %s </arg>",fComment));
		msg.Add(format("\n\t\t</config>"));
		msg.Add(format("\n\t\t<doc>"));
		msg.Add(format("\n\t\t%s",fDoc.chars()));
		msg.Add(format("\n\t\t</doc>"));
		rv = 1;
	  }
	}
	msg.Add(format("\n\t</atom>\n"));
	return rv;
}

void  CVariable::WriteConfigXML( FILE* oFile ) {
	if( GetCInfo(kMode) == kIgnored ) return;
	if( GetF(FisImport ) ) {
		fprintf(oFile,"\n\n\t<port type=\"input\" name=\"%s\" >",Name() );
		fprintf(oFile,"\n\t\t<default> %s </default>", DefaultValue(0).chars() );
		fprintf(oFile,"\n\t</port>\n");
	} else {
		CString mod;
		GetXMLAttributes(mod);
		CString status;
		MakeStatusDataString( status, False );
		fprintf(oFile,"\n\n\t<atom name=\"%s\" id=\"%s\" status_data=\"%s\" %s>\n",Name(),Name(),status.chars(),mod.chars());
		if( GetCInfo(kInitMode) == kParameter ) { 
		  fprintf(oFile,"\n\t\t<config name=\"pm\">");
		  fprintf(oFile,"\n\t\t\t<arg name=\"value\"> %f </arg>",fValue);
		  fprintf(oFile,"\n\t\t</config>");
		} else {
		  for( Pix p = fCommandList.first(); p; fCommandList.next(p) ) {
			Pipe* pipe = ((TCommand&)fCommandList(p)).GetPipe();
			if( pipe ) { pipe->WriteMML(oFile); }
		  }		
		}
		if( fArrayRec != NULL ) {
		  fArrayRec->WriteConfigXML( oFile );
		}
		if( _Arrays[0] != NULL ) {
		  fprintf(oFile,"\n\t\t<config name=\"ArDep\">");
		  for ( int array_index = 0; array_index < kMaxNVarArrays; array_index++ ) {
			ArrayRec* array = _Arrays[array_index];
			if( array == NULL ) break;
			fprintf(oFile,"\n\t\t\t<arg name=\"dep%d\"> %s </arg>", array_index, array->Name() );
		  }
		  fprintf(oFile,"\n\t\t</config>");
		}
		if( GetF(FisDisplayVar ) ) {
		  fprintf(oFile,"\n\t\t<config name=\"DD\">");
		  fprintf(oFile,"\n\t\t</config>");
		}
		fprintf(oFile,"\n\t</atom>\n");
	}
}

void CVariable::RefreshData() {
	  switch( fValueMode ) {
		case kValueSpatial: 
		  if( GetF(FMapRemoteData) ) { 
			  if( fConnection == NULL ) {
				  sprintf(gMsgStr,"Undefined Connection for remote Variable %s:%s", fModule->Name(), Name() );
				  gFatal();
			  }
			  CVariable* c = (CVariable*)fConnection;
			  fCov = c->fCov;
		  }
		break;
		case kValueNonSpatial: 
		  if( GetF(FisImport) ) { 
			  if( fConnection == NULL ) {
				  sprintf(gMsgStr,"Undefined Connection for input Variable %s:%s", fModule->Name(), Name() );
				  gFatal();
			  }
			  CVariable* aVar = (CVariable*) fConnection;
			  if( aVar->GetF(FisIgnored) ) {
				  float dval = atof(fDefValue[0]);
				  fTHistory.Add( gTime(), dval );
			  } else {     
				  fTHistory.Add( aVar->fTHistory.LastTimeStamp(), aVar->fTHistory.LastValue() );
			  }
		  }
		break;
		case kValueNonSpatialArray:
		  if( GetF(FisImport) ) { 
			  if( fConnection == NULL ) {
				  sprintf(gMsgStr,"Undefined Connection for input Variable %s:%s", fModule->Name(), Name() );
				  gFatal();
			  }
			  CVariable* aVar = (CVariable*) fConnection;
			  int dsize = aVar->getArraySize();
			  if( aVar->GetF(FisIgnored) ) {
				  float fval = atof(fDefValue[0]);
				  float* dval = new float[ dsize ];
				  for( int i=0; i<dsize; i++ ) { dval[i] = fval; }
				  fSHistory.Add( aVar->fSHistory.LastTimeStamp(), dval,  dsize, kTS_Shallow );
			  } else {     
				  fSHistory.Add( aVar->fSHistory.LastTimeStamp(), aVar->fSHistory.LastData(),  dsize, kTS_Shallow );
			  }
		  }
		break;
	}
}

int CVariable::ProcessBoundsError( float value, TCell* p , byte error ) {
		switch( error ) {
			case 'd': {
				if( p ) {
					TCell& pt = *p;
					sprintf(gMsgStr," BOUNDS ERROR(%d,%d,%d) %s -> %E: ",pt(0),pt(1),p->GetObjInfo( TCell::kLayerIndex ),(const char*)fName,value);
				} else {
					sprintf(gMsgStr," BOUNDS ERROR var %s -> %E: ",(const char*)fName,value); gPrintLog();				
				} 
				gPrintErr(); 
				Env::debugger_stop_here();
				SS_SetCurrentVariable(this);
				PrintFlags();
				} break;
			case 'p': {
				if( p ) {
					TCell& pt = *p;
					sprintf(gMsgStr,"\nBOUNDS ERROR(%d,%d,%d) %s -> %E: ",pt(0),pt(1),p->GetObjInfo( TCell::kLayerIndex ),(const char*)fName,value); 
				} else {
					sprintf(gMsgStr,"\nBOUNDS ERROR var %s -> %E: ",(const char*)fName,value); gPrintLog();
				} 
				gPrintErr(); 
				if( !Env::UseJavaSocket() ) { TModel::I().Interrupt(); }
				SS_SetCurrentVariable(this);
				} break;
			case 'f': {
				if( p ) {
					TCell& pt = *p;
					sprintf(gMsgStr,"\nBOUNDS ERROR(%d,%d,%d) %s -> %E: ",pt(0),pt(1),p->GetObjInfo( TCell::kLayerIndex ),(const char*)fName,value); 
				} else {
					sprintf(gMsgStr,"\nBOUNDS ERROR var %s -> %E: ",(const char*)fName,value);  gPrintLog();
				} 
				PrintFlags();
				gFatal();
				} break;
			default: return 0;	       
		}
		return 1;
}



//========================================================================================
// 				CLASS CStateVariable
//========================================================================================

EConfigInfo CStateVariable::kUpdateMode = kCInfo4;

CStateVariable::CStateVariable(const char* name ) 
  : CVariable( name )
{
  SetCInfo(kType,kState); 
  SetCInfo(kInitMode,kDefault);
  SetF(FisStateVar,1,"CStateVariable");
  SetF(FisConstant,False,"CStateVariable");
  fCurrentOrder = 1;
  fOrder = 1;
  fClass = kState;
  fAReserve = NULL;
}

float CStateVariable::ArrayUpdate( float value, int array_index[4], float dt ) {
  float ndt = dt;
  int mIndex = getArrayIndex(array_index);
  float tval, ival1, ival2, ival3, newvalue;
  
  switch( fCurrentOrder ) {
    case 1:
      newvalue = fSHistory.LastValueOffset(1,mIndex)  + value*dt;
	  if( fOrder > 1 ) { fSIntegratorMemory.LastValue(mIndex) = value; }
      break;
    case 2: 
	  fSIntegratorMemory.LastValue(mIndex) = value;
	  tval = fSHistory.LastValueOffset(1,mIndex);
	  ival1 = fSIntegratorMemory.LastValueOffset(1,mIndex);
      newvalue = tval + ( 1.5*value - 0.5*ival1 )*dt;
      break;
    case  3:
	  fSIntegratorMemory.LastValue(mIndex) = value;
	  tval = fSHistory.LastValueOffset(1,mIndex);
	  ival1 = fSIntegratorMemory.LastValueOffset(1,mIndex);
	  ival2 = fSIntegratorMemory.LastValueOffset(2,mIndex);
      newvalue = tval + ( 1.92*value - 1.33*ival1 + .42*ival2 )*ndt;
      break;
    case  4:
	  fSIntegratorMemory.LastValue(mIndex) = value;
	  tval = fSHistory.LastValueOffset(1,mIndex);
	  ival1 = fSIntegratorMemory.LastValueOffset(1,mIndex);
	  ival2 = fSIntegratorMemory.LastValueOffset(2,mIndex);
	  ival3 = fSIntegratorMemory.LastValueOffset(3,mIndex);
      newvalue = tval + ( 2.29*value - 2.46*ival1 + 1.54*ival2 - .375*ival3)*ndt;
      break;
    }
	if( GetF(FisClamped) ) { 
		fAReserve[mIndex] = newvalue =  (newvalue < 0) ? 0 : newvalue; 
		if( fOrder > 1 ) { fAReserve[mIndex] -= GetReserveAdjustment(mIndex); }
	}
	fSHistory.LastValue(mIndex) = newvalue;

	if( !isValidValue( value ) ) { ProcessBoundsError( value, NULL, 'p' ); }
	
#ifdef DEBUG
    if( GetCInfo(kDebug) ) {
      DoBoundsCheck(newvalue);
      PrintUpdateMsg(newvalue,value,"UPDATESV");
    }
#endif
	return ndt;
}

float CStateVariable::Update( TCell* c, float value, float dt ) {
    float ndt = dt;
	int32 mIndex = c->memoryIndex( fActivationLayer );
	float& ftmp = fCov(mIndex);
    switch( fCurrentOrder ) {
    case 1:
      ftmp += value*dt;
	  if( fOrder > 1 ) { fSIntegratorMemory.LastValue(mIndex) = value; }
      break;
    case 2:
	  fSIntegratorMemory.LastValue(mIndex) = value;
      ftmp = ftmp + ( 1.5*value - 0.5*fSIntegratorMemory.LastValueOffset(1,mIndex) )*dt;
      break;
    case  3:
	  fSIntegratorMemory.LastValue(mIndex) = value;
      ftmp = ftmp+ ( 1.92*value - 1.33*fSIntegratorMemory.LastValueOffset(1,mIndex) + .42*fSIntegratorMemory.LastValueOffset(2,mIndex))*ndt;
      break;
    case  4:
			fSIntegratorMemory.LastValue(mIndex) = value;
      ftmp = ftmp + ( 2.29*value - 2.46*fSIntegratorMemory.LastValueOffset(1,mIndex) + 1.54*fSIntegratorMemory.LastValueOffset(2,mIndex) - .375*fSIntegratorMemory.LastValueOffset(3,mIndex))*ndt;
      break;
    }
    if( GetF(FisClamped) ) { 
		if( ftmp < 0 ) {   ftmp = 0;  }
		fSReserve.elem(mIndex) = (ftmp - GetReserveAdjustment(mIndex)); 
	}
	if( !isValidValue( value ) ) { ProcessBoundsError( value, c , 'p' ); }
	
#ifdef DEBUG
    if( GetCInfo(kDebug) ) {
      DoBoundsCheck(ftmp,c);
      PrintUpdateMsg(c,ftmp,value,"UPDATESV");
    }
#endif
    return ndt;
}

float CStateVariable::GetIntegratorValue(float value ) {
  float dt = fModule->dT();
  float ival = 0.0;
  switch( fCurrentOrder ) {
    case 1:
      ival = value*dt;
      break;
    case  2:
      ival =  ( 1.5*value )*dt;
      break;
    case  3:
      ival = ( 1.92*value )*dt;
      break;
    case  4:
      ival = ( 2.29*value )*dt;
      break;
  };
  return ival;
}

float CStateVariable::GetFluxValueFromIntegrator( float value ) {
  float dt = fModule->dT();
  float ival = 0.0;
  switch( fCurrentOrder ) {
    case 1: 
      ival =  value/dt;
      break;
    case  2:
      ival =  value/( 1.5*dt);
      break;
    case  3:
      ival =  value/( 1.92*dt );
      break;
    case  4:
      ival =  value/( 2.29*dt );
      break;
  };
  return ival;
}

float CStateVariable::GetReserveAdjustment( int mIndex ) {
	if( fOrder < 2 ) return 0.0;
	float dt = fModule->dT();
	float ival = 0.0;
    switch( fCurrentOrder ) {
    case 1: case 0:
      break;
    case 2:
      ival =  ( - 0.5*fSIntegratorMemory.LastValueOffset(1,mIndex) )*dt;
      break;
    case  3:
      ival =  ( - 1.33*fSIntegratorMemory.LastValueOffset(1,mIndex) + .42*fSIntegratorMemory.LastValueOffset(2,mIndex) )*dt;
      break;
    case  4:
      ival = ( - 2.46*fSIntegratorMemory.LastValueOffset(1,mIndex) + 1.54*fSIntegratorMemory.LastValueOffset(2,mIndex)
																   - .375*fSIntegratorMemory.LastValueOffset(3,mIndex) )*dt;
      break;
    }
    return ival;
}

float CStateVariable::GetReserveAdjustment() {
  if( fOrder < 2 ) return 0.0;
  float dt = fModule->dT();
  float ival = 0.0;
  switch( fCurrentOrder ) {
    case 1:  case 0:      
	  break;
    case  2:
      ival = ( - 0.5*fTIntegratorMemory.LastValue() )*dt;
      break;
    case  3:
      ival = ( - 1.33*fTIntegratorMemory.LastValue() + .42*fTIntegratorMemory.LastValueOffset(1) )*dt;
      break;
    case  4:
      ival = ( - 2.46*fTIntegratorMemory.LastValue() + 1.54*fTIntegratorMemory.LastValueOffset(1) - .375*fTIntegratorMemory.LastValueOffset(2) )*dt;
      break;
  };
  return ival;
}
  
float CStateVariable::Update(float value, float dt) {

  float ndt = dt;
  float newvalue;
  switch( fCurrentOrder ) {
    case 1:
      newvalue = fTHistory.LastValue() + value*ndt;
	  if( fOrder > 1 ) { fTIntegratorMemory.Add( gTime(), value ); }
      break;
    case  2:
      newvalue = fTHistory.LastValue() + ( 1.5*value - 0.5*fTIntegratorMemory.LastValue() )*ndt;
			fTIntegratorMemory.Add( gTime(), value );
      break;
    case  3:
      newvalue = fTHistory.LastValue() + ( 1.92*value - 1.33*fTIntegratorMemory.LastValue() + .42*fTIntegratorMemory.LastValueOffset(1))*ndt;
			fTIntegratorMemory.Add( gTime(), value );
      break;
    case  4:
      newvalue = fTHistory.LastValue() + ( 2.29*value - 2.46*fTIntegratorMemory.LastValue() + 1.54*fTIntegratorMemory.LastValueOffset(1) - .375*fTIntegratorMemory.LastValueOffset(2))*ndt;
			fTIntegratorMemory.Add( gTime(), value );
      break;
  };
	if( GetF(FisClamped) ) { 
		fTReserve = newvalue =  ((newvalue < 0) ? 0 : newvalue); 
		if( fOrder > 1 ) { fTReserve -= GetReserveAdjustment(); }
	}
	fTHistory.Add( gTime(), newvalue );
	if( !isValidValue( value ) ) { ProcessBoundsError( value, NULL, 'p' ); }
  
#ifdef DEBUG
  if(GetCInfo(kDebug)) {
    DoBoundsCheck(newvalue);
    PrintUpdateMsg(newvalue,value,"UPDATE_SV");
  }
#endif
  return ndt;
}

int  CStateVariable::Config( TConfigData& cd ) {
  int rv = CVariable::Config( cd );
	if( fIntegrationMethod.length() > 0 && isdigit(fIntegrationMethod[0]) ) {
		fOrder = fIntegrationMethod[0] - '0';
	}
	return rv;
}      

void CStateVariable::AddFlux( CFluxVariable* f ) { fFluxList.Insert(*f); }

void CStateVariable::InitValue(float value) {
		fTHistory.Add( gTime(), value );
		if( GetF(FisClamped) ) {  
		  fTReserve = value - GetReserveAdjustment();  
		}

    if(GetCInfo(kDebug)) {
      DoBoundsCheck(value);
#ifdef DEBUG
      PrintUpdateMsg(value,value,"INIT");
#endif 
    }
}

void CStateVariable::ArrayInitValue(  Pix p, float value, int array_index[4] ) {
  int mIndex = getArrayIndex(array_index);

	if( GetF(FisClamped) ) { 
		fAReserve[mIndex] =  (value = (value < 0) ? 0 : value) - GetReserveAdjustment(mIndex); 
	}
	fSHistory.LastValue(mIndex) = value;
	
    if(GetCInfo(kDebug)) {
      DoBoundsCheck(value);
#ifdef DEBUG
      PrintUpdateMsg(value,value,"INIT");
#endif 
    }
}

void CStateVariable::ArrayInitValue( float value, int array_index[4] ) {
  int mIndex = getArrayIndex(array_index);

	if( GetF(FisClamped) ) { 
		fAReserve[mIndex] =  (value = (value < 0) ? 0 : value) - GetReserveAdjustment(mIndex); 
	}
	fSHistory.LastValue(mIndex) = value;
	
    if(GetCInfo(kDebug)) {
      DoBoundsCheck(value);
#ifdef DEBUG
      PrintUpdateMsg(value,value,"INIT");
#endif 
    }
}

void CStateVariable::UpdateInternals() {
	CVariable::UpdateInternals();
}

void CStateVariable::UpdateIntegrator( int isInit ) {
	if(  getArraySize() > 0  ) { 
	  fSHistory.Allocate( gTime(), getArraySize() ); 
	  if( !isInit ) {			
		fSIntegratorMemory.Allocate( gTime(), getArraySize() );
		fSIntegratorMemory.DeleteExtraSlicesBelow( fOrder ); 
		int ilen = fSIntegratorMemory.length();
		fCurrentOrder = Util::imin( fOrder, ilen );
		fCurrentOrder = (fCurrentOrder < 1) ? 1 : fCurrentOrder;
	  }
	}			
	else if( !isInit && (fOrder > 1) ) {
	  if( GetF(FisSpatial) ) { 
		  fSIntegratorMemory.Allocate( gTime(), fCov.Size() );
		  fSIntegratorMemory.DeleteExtraSlicesBelow( fOrder ); 
		  int ilen = fSIntegratorMemory.length();
		  fCurrentOrder = Util::imin( fOrder, ilen );
		  fCurrentOrder = (fCurrentOrder < 1) ? 1 : fCurrentOrder;
	  } else {		
		  int ilen = fTIntegratorMemory.length();
		  fCurrentOrder = Util::imin( fOrder, ilen );
		  fCurrentOrder = (fCurrentOrder < 1) ? 1 : fCurrentOrder;
	  }
	}
	if( gDebug) {
		sprintf(gMsgStr,"\nUpdating Integrator %s at %f: %d ",Name(),gTime(),fCurrentOrder);
		gPrintLog();
	}
}

int  CStateVariable::Allocate() {
	if( fAReserve == NULL ) { fAReserve = new float[getArraySize()]; }
	
	if( !CVariable::Allocate() ) return 0;
	
	int memoryIndex = MemObject::kPermanentMemory;
	//  int memoryIndex = fModule->Index();
	if( GetF(FisClamped) ) { 
	  if( fSReserve.Setup( fCov.Grid( -1, fLinkLayer ), 0, memoryIndex ) && gDebug ) {
	    sprintf(gMsgStr,"\nSVReserve MAlloc: %s::%s (%d)",fModule->Name(),Name(),memoryIndex); 
	    gPrintLog(); 
	  }
	}
	return 1;
}

void CStateVariable::SSInit( Module* mod ) {
	CVariable::SSInit( mod );
	fSIntegratorMemory.SetStack(((TModule*)mod)->SpatialArrays());
}

//========================================================================================
// 				CLASS CAuxVariable
//========================================================================================

EConfigInfo CAuxVariable::kDBOrder = kCInfo4;

CAuxVariable::CAuxVariable(const char* name ) 
  : CVariable( name ) {
    SetCInfo(kType,kAux); 
    SetCInfo(kInitMode,kDefault);
		fClass = kAux;
}


void  CAuxVariable::MakeDBaseVar( ) {
  Variable::MakeDBaseVar( );
	if( !GetF(FisImport) ) { 
		for( Pix p = fConnectionList.first(); p; fConnectionList.next(p))  {
			CAuxVariable& aVar = (CAuxVariable&) fConnectionList(p);
			aVar.MakeDBaseVar( );
		}
	}
} 

float* CAuxVariable::GetDBaseData(const char *filename, int sIndex, int pIndex, int index, int order, int& size ) {
  gPrintLog("Doing ReadDBase"); 
  if( gGlobalFileSystem ) return HParm::Read( filename, sIndex, pIndex, index, order, size );
  else {
    int root = 0;
    float* data=NULL;
    if(  gIProc == 0  ) {
      data = HParm::Read( filename, sIndex, pIndex, index, order, size );
      if(data == NULL) return 0;
    }
    if(gNProc > 1) {
#ifdef USE_MPI
      MPI_Bcast( &size, 1, MPI_INT, root, MPI_COMM_WORLD );
      if(size>0) { 
				if(data == NULL) { data = new float[size]; }
				MPI_Bcast( data, size, MPI_FLOAT, root, MPI_COMM_WORLD );
      }
#endif
    }
    return data;
  }
}

int CAuxVariable::CheckCategoryLabels( ) {
	int all_labels_marked = 1;
	if( fCatMapPtr ) { 
		for( Pix p = fCatMapPtr->first(); p; fCatMapPtr->next(p) ) {
			if( fCatMapPtr->is_marked(p) < 1 ) {
				int key =  fCatMapPtr->key(p);
				if( key > 0 ) {
					all_labels_marked = 0;
					CString* name =  (CString*)((*fCatMapPtr)[key]);
					sprintf(gMsgStr,"%s: No category info for %s (%d) in map variable %s\n",Name(),name->chars(),key,fMapVariable[0]->Name()); 
					gPrintErr(); 
				}
			}
		}
	}
	return all_labels_marked;
}


float CAuxVariable::Value( TCell* c ) {
  static int first_error = 1;
  switch( fValueMode ) {
  case kValueSpatial: {
//			if( c->activated( fCov.activationIndex() ) ) { return fCov( c ); }
//			else { 
			  return fCov.activationLayer()->mapValue( c, fCov.activationIndex(), fCov.Data(), fCov.isIntensive(), fErrorVal );
//			}
  } case kDBParm: {
		   float fval = ((CVariable*)fMapVariable[0])->Value(c);
		   if( fval == fErrorVal ) return fErrorVal;
		  int index = (int)fval;
		  float& rv = fV.elem( index );
#ifdef DEBUG
		  if( rv == kfloatVecError ) { 
			  sprintf(gMsgStr,"DBParm %s:%s initialized improperly (missing category info?): requested index: %d, capacity: %d, map: %s:%s",GetModule()->Name(),Name(),index,fV.capacity(),fMapVariable[0]->GetModule()->Name(),fMapVariable[0]->Name());
			  gPrintErr();
		  }
#endif
	return rv;
  } case kTSeries:
		  if( GetF(FisImport) )  { 			
			  if( fConnection == NULL ) {
				  sprintf(gMsgStr,"Undefined Connection for input Variable %s:%s", fModule->Name(), Name() );
				  gFatal();
			  }
			   fValue = ((CAuxVariable*)fConnection)->fSeries.Value( gTime() ); 
			   if( (fValue == kSeriesNullValue) && first_error ) {
				  gPrintErr( format("Empty timeseries in %s", Name() ) );
				  first_error = 0;
				  fValue = 0.0;
			   }
			   return fValue;
		  } else {
			   fValue = fSeries.Value( gTime() ); 
			   if( (fValue == kSeriesNullValue) && first_error ) {
				  gPrintErr( format("Empty timeseries in %s", Name() ) );
				  first_error = 0;
				  fValue = 0.0;
			   }
			   return fValue;
		  }
  case kParm:
	return fValue;
  case kValueNonSpatial: 
		 fValue = fTHistory.LastValue(); 
		 if( fValue == kSeriesNullValue ) {
			gPrintErr( format("Empty timeseries in %s", Name() ) );
			fValue = 0.0;
		 }
		 return fValue;  
  }
}


int CAuxVariable::GetCategoryIndex( const char* label ) {
	if( fArrayRec ) {
	   int rv = -1;
	   ArgArray* a = fArrayRec->getArgs(); 	
	   if( a != NULL ) { rv = a->Find(label); }
	   if( rv < 0 ) { gPrintErr( format("Unknown Array Element in %s:%s[%d]: %s", fModule->Name(), Name(), a->NArgs(), label) ); }
	   return rv;
	} else {
	  if(fCatMapPtr == NULL) { 
		  CAuxVariable* cv = (CAuxVariable*)fMapVariable[0];
		  if( cv == NULL ) { cv = this; }
		  if( cv->GetF(FisImport) ) { 
			  if( cv->fConnection == NULL ) {
				  sprintf(gMsgStr,"Undefined Connection for input Variable %s:%s", cv->fModule->Name(), cv->Name() );
				  gFatal();
			  }
			  cv = (CAuxVariable*)cv->fConnection; 
		  }
		  fCatMapPtr = cv->ReadCategories(); 
	  } 
	  if(fCatMapPtr == NULL) { return -2; }
	  for( Pix p = fCatMapPtr->first(); p; fCatMapPtr->next(p) ) {
		  int key =  fCatMapPtr->key(p);
		  CString* name =  (CString*)((*fCatMapPtr)[key]);
		  if( name->equals_nocase(label) ) { fCatMapPtr->mark(p); return key; }	
	  }
	}
	return -1;
}

intPixVHMap* CAuxVariable::ReadCategories( )  {
	if( ( GetCInfo(Variable::kInitMode) != Variable::kMap ) || ( fCatMapPtr==NULL ) ) { 
		return (intPixVHMap*)NULL; 
	}
	if(  fCatMapPtr->empty() ) {
		TMapInputPipe *mp = NULL;
		for( Pix p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
			TCommand& c0 = (TCommand&) fCommandList(p0);
			if( TCommand::kPipe == c0.GetObjInfo(TCommand::kExType) ) { 
				TPipeCommand& pc = (TPipeCommand&)c0;
				Pipe* p = pc.GetPipe();
				if( p != NULL ) {
					if( p->GetObjInfo( Pipe::kType ) == Pipe::kMapInput ) {
						mp = (TMapInputPipe*)p;
						break;
					}
				}
			}
		}
		if( mp == NULL ) return (intPixVHMap*)NULL;
		mp->ReadCategories();
		if( fCatMapPtr->empty() ) { fCatMapPtr = NULL; }
	}
	return fCatMapPtr;
}  

int CAuxVariable::PrintCategories( CString& cats ) {
	cats.clear();
	if(fCatMapPtr == NULL) return 0;
	for( Pix p = fCatMapPtr->first(); p; fCatMapPtr->next(p) ) {
		int key =  fCatMapPtr->key(p);
		CString* name =  (CString*)((*fCatMapPtr)[key]);
		cats += (*name); cats += " ";	
	}
	return 1;
}

void CAuxVariable::PrettyPrint( CString& result, int start, int end ) {
    switch( fValueMode ) {
	  case kValueSpatial: 
	  case kValueNonSpatial: 
	  case kValueNonSpatialArray: 
	  case kDBParm: 
		CVariable::PrettyPrint( result, start, end );
	  break;
	  case kTSeries: {
		if( GetF(FisImport) )  { 			
			 if( fConnection == NULL ) {
				 sprintf(gMsgStr,"Undefined Connection for input Variable %s:%s", fModule->Name(), Name() );
				 gFatal();
			 }
			 ((CAuxVariable*)fConnection)->fSeries.PrettyPrint( result, start, end ); 
		} else {
			 fSeries.PrettyPrint( result, start, end ); 
		}
	 } break;
    case kParm: {
      result += "( "; result +=  fValue;  result += " ) \n";
      } break;
  }  
}

void CAuxVariable::ReadArrayParameter(const char* fileName) {
	CPathString cpath(Env::DataPath()); cpath.Add(fileName);
  if(gDebug) { sprintf(gMsgStr,"Reading array parameter file %s",cpath.chars()); gPrintLog(); } 
  FILE* file = fopen(cpath,"r");
  if(file==NULL) { sprintf(gMsgStr,"\nERROR: Unable to open array parameter file %s\n", cpath.chars() ); gFatal(); }
  CString mapName, className, value;
  Variable* map = NULL;
	if( (fMapVariable[0] == NULL) || (fCatMapPtr == NULL) ) { 
			sprintf(gMsgStr,"DBase Variable %s:%s not set up properly\n",fModule->Name(),Name()); gPrintErr(); return; 
	}
	fV.alloc(256);
  while( Util::findNextRecord( file, Name() ) ) {
		Util::readNextField( file, mapName );
		Util::readNextField( file, className );
		Util::readNextField( file, value );
		SetArrayParameterValue(mapName, className, value);
	}
	fclose(file);
}

void CAuxVariable::SetArrayParameterValue(CString& mapName, CString& className, CString& value) { 
	int index; float fval;
	CString arrayName = mapName;
	int isMap = 0, mapPos = mapName.length()-3;
	if( arrayName.matches("MAP", mapPos) ) { arrayName.del( mapPos, 3 ); isMap = 1; }
	
	if( fMapVariable[0] == NULL ) {
		sprintf(gMsgStr,"%s: No map data\n",Name()); 
		gPrintErr(); 
		return;
	}
	if( fV.Size() == 0 ) {
		fV.resize( MAX_NHAB, kfloatVecError  ); 
	} 
	if( fMapVariable[0]->SName() == mapName ) {
		if(  className == arrayName ) {
			fval = atof(value);
			fV.fill( fval );  
		} else {
			index = GetCategoryIndex(className);
			if( index < 0 ) {
				if( index == -2 ) {
					if(isMap) sprintf(gMsgStr,"%s: No category info for map variable %s ( arrayName %s )\n",Name(),fMapVariable[0]->Name(),arrayName.chars());
					else sprintf(gMsgStr,"%s: No category info for map variable %s\n",Name(),fMapVariable[0]->Name()); 
 					gPrintErr(); 
				} else { 
					if( gDebug > 2 ) {
						sprintf(gMsgStr,"%s: Can't find category label \"%s\" for map variable %s\n",Name(),className.chars(),fMapVariable[0]->Name()); 
						gPrintErr(); 
						static CString cats;
						if( PrintCategories( cats ) ) {
							sprintf(gMsgStr,"cats: ( %s )", cats.chars() ); 
							gPrintErr(); 
						}
					}
				}
			} else {
				if( index >= fV.capacity() ) {
					int newsize = ( index < 100 ) ? 100 : index * 2; 
					fV.resize( newsize ); 
				}
				fval = atof(value); 
				fV.elem(index) = fval;
				if( gDebug > 1 ) { 
					sprintf(gMsgStr,"MDP %s: Set value for category \"%s\" (%d) in map %s to %f\n",
														Name(),className.chars(),index,fMapVariable[0]->Name(),fval); 
					gPrintLog(); 
				}
			}
		}
	} 
}

int CAuxVariable::SetupDBParam( const CString& map  ) {
	static CString mapvar; mapvar = map; mapvar += "MAP";
	if( GetCInfo(kInitMode) != kDBase ) { 
		sprintf(gMsgStr,"Setting up DBParameter %s", Name() ); gPrintLog();
		MakeDBaseVar(); 
		DBaseInputPipe* p = TPipe::GetDBaseInputPipe( "m10", this );
		p->AddMap( mapvar );
		p->SetObjInfo(DBaseInputPipe::kOrder,10);
		RegisterPipe( *p );
		
		TPipeCommand* pc = TPipeCommand::CreatePipeCommand( "m10", p, this );
		if( pc) {
			AddCommand( pc );
			if( p ) p->SetPipeCommand(pc); 
			pc->AddPipeDependencies(p);	
		}                       
		return 1;
	}
	else if( fMap[0] != mapvar ) { 
		sprintf(gMsgStr,"Map discrepancy in SetupDBParmdata for %s: %s vs %s", Name(), fMap[0].chars(), map.chars() );  gPrintErr(); 
	}
	return 0;
}

int CAuxVariable::SetDBParamValues() {
	int cnt = 0;
	for(  Pix p = fDBParamValueList.first(); p; fDBParamValueList.next(p) ) {
		TDBParamValue& dbp = (TDBParamValue&)fDBParamValueList(p);
		SetArrayParameterValue( fMap[0], (CString&)dbp.SName(), (CString&)dbp.Value() );
		cnt++;
	}
	CheckCategoryLabels();
	if( !GetF(FisImport) && ( fV.capacity() > 0 ) ) { 
		for( Pix p = fConnectionList.first(); p; fConnectionList.next(p))  {
			CAuxVariable& aVar = (CAuxVariable&) fConnectionList(p);
			aVar.fV.shallow_copy(fV); 
		}
	}
	return cnt;
}
  
void CAuxVariable::ReadDBase(const char* fileName, int sIndex, int cellIndex, int index ) {
	CPathString cpath(Env::DataPath()); cpath.Add(fileName);
  int len; float* data;
  data = GetDBaseData(cpath, sIndex, cellIndex, index, GetCInfo(kDBOrder), len);
  fV.setData(data,len);
  for( Pix p = fConnectionList.first(); p; fConnectionList.next(p))  {
    CAuxVariable& aVar = (CAuxVariable&) fConnectionList(p);
    aVar.fV.shallow_copy(fV);
  }	
  if(GetCInfo(kDebug)) {
    char ctmp[50]; CString s;
    sprintf(gMsgStr,"\nINIT DBase Parm%s -> ", (const char*)fName); 
    for(int i=0; i<fV.Size(); i++) { sprintf(ctmp,"%f ",fV.elem(i)); s += ctmp; }
    gPrintLog(s.chars());
  }

}

const int kMaxCols = 10;

#ifdef POSTGRESS

void CAuxVariable::ReadDBase( const char* dataBase, const pgSQLRec* sqlRec, const ArgArray& args ) {
	sme_pg95* db = GetDBase(dataBase);
	db->exec_sql( *sqlRec, args );
	int nt = db->GetNTuples();
	CString value, colVal[kMaxCols];
	int ival = sqlRec->ValueFieldIndex();
	int mval = sqlRec->MapFieldIndex(0);
	CString val;
	if( GetCInfo(kInitMode) == kParameter ) {
		for( int i=0; i<nt; i++ ) {
			val = db->GetTuples(i);
			val.splitColumns(colVal,kMaxCols);
			fValue = atof(colVal[ival]);
		}
		for( Pix p = fConnectionList.first(); p; fConnectionList.next(p))  {
			CAuxVariable& aVar = (CAuxVariable&) fConnectionList(p);
			aVar.fValue = fValue;
		}	
		if(GetCInfo(kDebug)) {
			char ctmp[50]; CString s;
			sprintf(gMsgStr,"\nINIT Parm %s -> ", (const char*)fName); 
			sprintf(ctmp,"%f ",fValue); s += ctmp;
			gPrintLog(s.chars());
		}
	} else if( GetCInfo(kInitMode) == kDBase ) {
		for( int i=0; i<nt; i++ ) {
			val = db->GetTuples(i);
			val.splitColumns(colVal,kMaxCols);
			int iv = atoi(colVal[mval]);
			float fv = atof(colVal[ival]);
			if( (iv<0) || (iv>=MAX_NHAB) ) { sprintf(gMsgStr,"Illegal Index in %s :: ReadDBase",Name()); gPrintErr(); }
			else { fV.add(iv) = fv; }
		}
		for( Pix p = fConnectionList.first(); p; fConnectionList.next(p))  {
			CAuxVariable& aVar = (CAuxVariable&) fConnectionList(p);
			aVar.fV.shallow_copy(fV);
		}	
		if(GetCInfo(kDebug)) {
			char ctmp[50]; CString s;
			sprintf(gMsgStr,"\nINIT DBase Parm%s -> ", (const char*)fName); 
			for(int i=0; i<fV.Size(); i++) { sprintf(ctmp,"%f ",fV.elem(i)); s += ctmp; }
			gPrintLog(s.chars());
		}
	}
}
#endif

/*
float CAuxVariable::ArraySum() { 
  if( GetCInfo(kInitMode) != kParameter ) {
	sprintf(gMsgStr,"May only invode ArraySum on DBParmVariables: %s", fName.chars() );
	gFatal();
  }
  float rv = 0.0;
  for( int i=0; i<fV.Size(); i++ ) {
	float val = fV.elem(i);
	if( val != kfloatVecError ) {
	  rv += val;
	}
  }
  return rv; 
}
*/

float CAuxVariable::DBParmValue( int vIndex ) { 
	float rv = fV.elem(vIndex);
	if( rv == kfloatVecError ) {
		sprintf(gMsgStr,"Data structures initialized improperly in DBParmVariable %s", fName.chars() );
		gFatal();
	}
	return rv; 
}

int  CAuxVariable::MakeParameterVariable( ) {
  int rv = Variable::MakeParameterVariable( );
	if( gDebug > 1 ) { sprintf(gMsgStr,"MakeParameterVariable: %s", Name() ); gPrintLog(); }
	if( !GetF(FisImport) ) { 
		for( Pix p = fConnectionList.first(); p; fConnectionList.next(p))  {
			CAuxVariable& aVar = (CAuxVariable&) fConnectionList(p);
			aVar.MakeParameterVariable( );
		}
	}
	return rv;
}  
    

int CAuxVariable::SetupParameter(float init_val, float min_value, float max_val, int  ustep ) { 

	int dim = Variable::SetupParameter( init_val, min_value, max_val, ustep ); 
  
//   This functionality handled by RefreshData method:
//
//  if( GetF(FisExport ) ) {
//    for( Pix p = fConnectionList.first(); p; fConnectionList.next(p))  {
//      CAuxVariable& aVar = (CAuxVariable&) fConnectionList(p);
//      aVar.fParm = fParm;
//      aVar.fValue = aVar.fParm.Value();
//    }
//  } else 

	if( GetF(FisImport ) ) { 
		if( fConnection == NULL ) {
			sprintf(gMsgStr,"Undefined Connection for input Variable %s:%s", fModule->Name(), Name() );
			gFatal();
		}
		((CAuxVariable*)fConnection)->SetupParameter( init_val, min_value, max_val, ustep ); 
	}
	return 0;
}

void CAuxVariable::ReadFloatData(const char* mapPath, int column ) {
	 if( gIProc == 0 || gGlobalFileSystem ) { 
		FILE* file = fopen(mapPath,"r");
		Module* m = GetModule();
		LayerConfig* lc = m->getLayerConfig();
		if(file==NULL) { 
			sprintf(gMsgStr,"(P%d)Unable to open float data file %s.\n",gIProc,mapPath); 
			gPrintLog();
			gFatal(gMsgStr,True);
		}
		if(gDebug) { 
			sprintf(gMsgStr,"(P%d)Successfully opened float data file %s.\n",gIProc,mapPath); 
			gPrintLog();
		}
		int ch, col = 0, row_index, data_index = 0;
		char cbuf[128];
		float* dbuf = DataStart(); 
		float* dstart = dbuf; 
		const float* dend = GridEnd();
		while ( (ch = fgetc(file)) !=  EOF ) {
			if( col == 0 ) {
				while ( ( ch == ' ' ) || ( ch == '\t' ) ) {	if( (ch=fgetc(file)) == EOF ) break; }     // skip white space at beginning of col.
				if( isdigit(ch) ) {
					char* bptr =  cbuf;
					*bptr++ = ch;
					while ( isdigit(ch = fgetc(file)) ) {
						*bptr++ = ch;
					}   
					*bptr++ = 0;
					row_index = atoi(cbuf);
					col = 1; 
				} else {
					while ( (ch = fgetc(file)) !=  '\n' ) {	if( ch == EOF ) break; }     // skip lines not starting with a digit.
				}
			} else {
				while ( ( ch == ' ' ) || ( ch == '\t' ) ) {	if( (ch=fgetc(file)) == EOF ) break; }     // ship white space at beginning of col.
				if( ch == '\n' ) col = 0;
				else if( Util::isEnum( ch ) ) {
					char* bptr =  cbuf;
					*bptr++ = ch;
					while ( Util::isEnum(ch = fgetc(file)) ) { 
						if( ch == EOF ) break; 
						*bptr++ = ch; 
					}
					*bptr++ = 0;
					
					if( col++ == column ) {
						if( dbuf >=  dend ) { gFatal("Float Data read error: too much data." ); }
						*dbuf++ = atof(cbuf);
//						sprintf(gMsgStr, "FData Read row %d: %f ", row_index, *(dbuf-1) ); gPrintLog();
					}
					if( ch == '\n' ) col = 0;
				} else {
					sprintf(gMsgStr, "Unrecognized character in Float Data: %c", ch );
					gFatal( gMsgStr );
				}
			}
		}
		int iL0 = lc->CellLayer(), iL1 = iL0 + lc->Dim( 2 ) - 1;
		for( int iL = iL0; iL < iL1; iL++ ) {
		  int msize = dbuf-dstart;
		  memcpy( dbuf, dstart, msize*sizeof(float) ); 
		  dbuf +=  msize;
		}
		if( dbuf !=  dend ) { 
			sprintf(gMsgStr,"Float Data read error: too little data: read %d cells, %d cells short", dbuf-dstart, dend-dbuf); gFatal();
		}
		fclose(file);	
	}	
}

int  CAuxVariable::Config( TConfigData& cd ) {
  return CVariable::Config( cd );
}

void CAuxVariable::MakeTimeSeriesVariable() {
  Variable::MakeTimeSeriesVariable();
	if( !GetF(FisImport) ) { 
		for( Pix p = fConnectionList.first(); p; fConnectionList.next(p))  {
			CAuxVariable& aVar = (CAuxVariable&) fConnectionList(p);
			aVar.MakeTimeSeriesVariable( );
		}
	}
}  


void CAuxVariable::Read( const char* fileName, int format ) {
	if(  GetCInfo(kInitMode) == kTimeSeries ) {
		if( format == 2 ) {
			HDFM::Open(HDFM::kFInput);
			fSeries.DSetName(fileName);
			fSeries.SDRead();
		} else {
			CPathString name(Env::DataPath()); name.Add(fileName);
			fSeries.Import( name, format );
		}
	} else if( GetCInfo(kInitMode) == kPtTimeSeries ) {
		if( format == 2 ) { HDFM::Open(HDFM::kFInput); }
		fSeriesList.SeriesName(fileName);
		fSeriesList.Read(format);
	}
}


void CAuxVariable::MakePtTimeSeriesVariable() {
  Variable::MakePtTimeSeriesVariable();
	if( !GetF(FisImport) ) { 
		for( Pix p = fConnectionList.first(); p; fConnectionList.next(p))  {
			CAuxVariable& aVar = (CAuxVariable&) fConnectionList(p);
			aVar.MakePtTimeSeriesVariable( );
		}
	}
} 
 
void CAuxVariable::MakeMapInputVariable( int isInt ) {
  Variable::MakeMapInputVariable( isInt );
	if( !GetF(FisImport) ) { 
		for( Pix p = fConnectionList.first(); p; fConnectionList.next(p))  {
			CAuxVariable& aVar = (CAuxVariable&) fConnectionList(p);
			aVar.MakeMapInputVariable( isInt );
		}
	}
}  

void CAuxVariable::MakeExternalIOVariable( EVarLink::EModality mod ) {
  Variable::MakeExternalIOVariable(mod);
	if( !GetF(FisImport) ) { 
		for( Pix p = fConnectionList.first(); p; fConnectionList.next(p))  {
			CAuxVariable& aVar = (CAuxVariable&) fConnectionList(p);
			aVar.MakeExternalIOVariable( mod );
		}
	}
}  

void CAuxVariable::RefreshData() {
	switch(  GetCInfo(kInitMode)  ) {
		case kPtTimeSeries:
			if( GetF(FisImport) )  CVariable::RefreshData();
			else {
			  if( gDebug > 1 ) { sprintf( gMsgStr," -- Create Point Map for %s -- ",Name()); gPrintLog(); }
 			  fSeriesList.CreatePointMap( Cov(), gTime() );
			}
			break;
		case kTimeSeries:
			break;
		case kParameter:
			if( fConnection ) {
				fParm = ((CAuxVariable*)fConnection)->fParm;
				fValue = ((CAuxVariable*)fConnection)->fValue;
			}
			break;
		default:  CVariable::RefreshData();
	}
}

void CAuxVariable::SetupInternals() {
  CVariable::SetupInternals();
  switch(  GetCInfo(kInitMode) ) {
    case kDefault: break;
    case kMap: fValueMode = kValueSpatial; break;
    case kTimeSeries: fValueMode = kTSeries; break;
    case kPtTimeSeries: fValueMode = kValueSpatial; break; 
    case kDBase: fValueMode = kDBParm; break;
    case kParameter: 
			if( GetF(FisSpatial) ) { fValueMode = kValueSpatial; }
			else { fValueMode = kParm; }
			break;
  } 
  if( gDebug ) {
    sprintf( gMsgStr,"\nSetupInternals for %s: upd= %d, val= %d",Name(),fUpdateMode,fValueMode); gPrintLog();
  }
}


//========================================================================================
// 				CLASS CFluxVariable
//========================================================================================

CFluxVariable::CFluxVariable(const char* name ) 
  : CAuxVariable( name )
{
  memset(fFluxInfo,0,4);
  memset(fClampData,0,4);
  SetCInfo(kType,kFlux); 
  SetCInfo(kInitMode,kDefault); 
  SetF(FisFlux,1,"CStateVariable");
  fClass = kFlux;
}

void CFluxVariable::Update( float value ) {
	if( GetCInfo(kInitMode) == kParameter ) {
		value = fParm.Value();
	}
	value = Clamp(value);
	CVariable::Update( value );
}

void CFluxVariable::ArrayUpdate( float value, int array_index[4] ) {
	value = ArrayClamp(value,array_index); 
	CVariable::ArrayUpdate( value, array_index );
}

void CFluxVariable::Update( TCell* c, float value ) {
	if( GetCInfo(kInitMode) == kParameter ) {
		value = fParm.Value();
	}
	value = Clamp(c,value);
	CVariable::Update( c, value );
} 


float CFluxVariable::Clamp( float v ) {
  if( v < 0 ) {
	if( fFluxInfo[kF_FluxType] == kF_UniFlow ) { return 0.0; }
	if( (gDebug > 1) && (fFluxInfo[kF_FluxType] == kF_Unknown) ) {
	  gPrintErr( format("Variable %s has negative flux- assuming biflow (if uniflow- configure with ft(1) command)",Name()));
	  fFluxInfo[kF_FluxType] = kF_BiFlow;
	  Env::ArchiveBiFlow( fModule->Name(), Name() );
	}
  }
  if( !GetF(FisClamped) ) { return v; }
  CStateVariable *o = (CStateVariable*) Origin(); 
  CStateVariable *sv, *d = (CStateVariable*) Destination(); 
  int ao = 0;
  float value = 0.0;
  if( o ) {
	  sv = o;
	  value =  o->GetIntegratorValue( v );
	  if( (value > 0.0) && (o->Reserve() < value) ) { 
#ifdef DEBUG
		  if(GetCInfo(kDebug)) PrintUpdateMsg(o->Reserve(),value,"CLAMP");
#endif
		  value = o->Reserve(); 
		  o->Reserve() = 0;
	  } else {
		 ao = 1; 
	  }
  }
  
  if( d ) { 
	  if( o == NULL ) { sv = d; value =  d->GetIntegratorValue( v ); }
	  if( (value < 0.0) && ((d->Reserve()+value) < 0.0 ) ) { 
#ifdef DEBUG
		  if(GetCInfo(kDebug)) PrintUpdateMsg(d->Reserve(),value,"CLAMP destination");
#endif
		  value = -d->Reserve(); 
		  d->Reserve() = 0;
	  } else {
		 d->Reserve() += value; 
	  }
  }
  if( ao ) { o->Reserve() -= value; }
  return sv->GetFluxValueFromIntegrator(value);
}

float CFluxVariable::ArrayClamp( float v, int array_index[4] ) {
  if( v < 0 ) {
	if( fFluxInfo[kF_FluxType] == kF_UniFlow ) { return 0.0; }
	if( (gDebug > 1) && (fFluxInfo[kF_FluxType] == kF_Unknown) ) {
	  gPrintErr( format("Variable %s has negative flux- assuming biflow (if uniflow- configure with ft(1) command)",Name()));
	  fFluxInfo[kF_FluxType] = kF_BiFlow;
	  Env::ArchiveBiFlow( fModule->Name(), Name() );
	}
  }
  if( !GetF(FisClamped) ) { return v; }
  CStateVariable *o = (CStateVariable*) Origin(); 
  CStateVariable *sv, *d = (CStateVariable*) Destination(); 
  int ao = 0;
  float value = 0.0;
  if( o ) {
	sv = o;
	float& rval = o->ArrayReserve(array_index);
	value =  o->GetIntegratorValue( v );
	if( (value>0.0) && (rval < value) ) { 
#ifdef DEBUG
		if(GetCInfo(kDebug)) PrintUpdateMsg(rval,value,"CLAMP");
#endif
		value = rval; 
		rval = 0;
	} else {
		ao = 1;
	}
  }

  if( d ) { 
	float& rval = d->ArrayReserve(array_index);
	if( o == NULL ) { sv = d; value =  d->GetIntegratorValue( v ); }
	if( ( value < 0.0 ) && ( (rval+value) < 0.0 ) ) { 
#ifdef DEBUG
		if(GetCInfo(kDebug)) PrintUpdateMsg(rval,value,"CLAMP destination");
#endif
		value = -rval; 
		rval = 0;
	} else {
		rval += value;  
	}
  }
  if( ao ) { o->ArrayReserve(array_index) -= value;  }
  return sv->GetFluxValueFromIntegrator(value);
}

int  CFluxVariable::Allocate() {
  if( !CVariable::Allocate() ) return 0;
  if(Origin()) {
    CStateVariable* o = (CStateVariable*) Origin(); 
    o->Allocate();
    if( !(o->activationLayerIndex() == fActivationLayer) ) {
			SetF( FFluxOriginMap, True, "Allocate" ); 
			sprintf(gMsgStr,"Flux and Flux origon belong to different activation layers: %s & %s",Name(),o->Name()); 
			gFatal();
		}
  }
  if( Destination() ) {
    CStateVariable* d = (CStateVariable*) Destination();  
    d->Allocate();
    if( !(d->activationLayerIndex() == fActivationLayer) ) SetF( FFluxDestinationMap, True, "Allocate" ); 
  } 
  return 1;
}

float CFluxVariable::Clamp( TCell* p, float v ) {
  if( v < 0 ) {
	if( fFluxInfo[kF_FluxType] == kF_UniFlow ) { return 0.0; }
	if( (gDebug > 1) && (fFluxInfo[kF_FluxType] == kF_Unknown) ) {
	  gPrintErr( format("Variable %s has negative flux- assuming biflow (if uniflow- configure with ft(1) command)",Name()));
	  fFluxInfo[kF_FluxType] = kF_BiFlow;
	  Env::ArchiveBiFlow( fModule->Name(), Name() );
	}
  }
  if( !GetF(FisClamped) ) { return v; }
  
  int printLog = (( gDebug > 1 ) && ( fClampData[0] == 0 ));
  fClampData[0] = 1;
  
  float tmp = 0.0;
  TCell* p0=p; 
  CStateVariable *o = (CStateVariable*) Origin(); 
  CStateVariable *sv, *d = (CStateVariable*) Destination(); 
  int ao = 0; float value = 0.0, value0 = 0.0;
  
  if( fFluxInfo[kF_OriginClamped] ) {
	sv = o;
	if( p0 ) {
		value0 = value =  o->GetIntegratorValue( v );
		if( (value > 0) && ((tmp = o->Reserve(p0)) < value ) ) { 
			value = tmp; 
			o->SetReserve(p0,0);
		} else {
			ao = 1;
		}
		if( printLog ) {
		  gPrintLog( format("CLAMP-orgin: Flux %s from %s; clamp-val: %f, val0: %f ",Name(),o->Name(),value,value0));
		}
	}
  }
  if( fFluxInfo[kF_DestinationClamped] ) {
	if( p0 ) {
		if( fFluxInfo[kF_OriginClamped] == 0 ) { sv = d; value0 = value =  d->GetIntegratorValue( v ); }
		if( (value < 0) && (((tmp = d->Reserve(p0))+value) < 0.0 ) ) { 
			value = -tmp; 
			d->SetReserve(p0,0);
		} else {
		   int index = d->AddReserve(p0, value);
		   if( index < 0 ) {
			  gFatal( format("In flux %s: Error adding %f to reserve of flux destination %s:%s",
				Name(),value,d->GetModule()->Name(),d->Name()));
		   }
		}
		if( printLog ) {
		  gPrintLog( format("CLAMP-destination: Flux %s to %s; clamp-val: %f, val0: %f ",Name(),d->Name(),value,value0));
		}
	}
  }
  if( ao ) { 
	 int index = o->AddReserve(p0,-value); 
	 if( index < 0 ) {
		gFatal( format("In flux %s: Error taking %f from reserve of flux origin %s:%s",
		  Name(),value,o->GetModule()->Name(),o->Name()));
	 }
  }
  return sv->GetFluxValueFromIntegrator( value );
}

void CFluxVariable::ICFlux( TCell* p0, float value, int dr, int dc, int dv ) {

  SetF(FICFlux,True,"ICFlux");
  TCell* pR = (TCell*)Grid().Translate( p0, dr, dc, dv );

  if( p0 && pR ) {
    fCov(p0) += value;
    fCov(pR) -= value;
  }

	if(GetCInfo(kDebug)) {
		DoBoundsCheck(Value(p0),p0);
		DoBoundsCheck(Value(pR),pR);
#ifdef DEBUG  	  
		PrintUpdateMsg(pR,Value(pR),value,"ICFlux:R");
		PrintUpdateMsg(p0,Value(p0),value,"ICFlux:O");
#endif
	}

}
