//----------------------------------------------------------------------------------------
//	CVariable.h
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//	Change History:
//----------------------------------------------------------------------------------------

#ifndef __CVariable__
#define __CVariable__

#include "Globals.h"
#include "Bounds.h"
#include "MultiCoverage.h"
#include "SList.h"
#include "DataSet.h"
#include "Series.h"
#include "TMap.h"
#include "MML_Variable.h"
#include "MML_Module.h"
#include "MML_Time.h"
#include "CnfgProcessor.h"
#include "Utilities.h"
#include "Externals.h"
#include "sme_postgres.h"

typedef OrderedPoint GridPoint;
typedef class TModule TModule;
typedef class TMultiFrame TMultiFrame;

float maxCombiner(float f1, float f2 );
float minCombiner(float f1, float f2 );

#ifdef POSTGRESS

class TDBaseSocket : public TNamedObject {
	sme_pg95* fDBase;	  
public:
	TDBaseSocket(const char* name);
	sme_pg95* DBase() { return fDBase; }
};

#endif

//----------------------------------------------------------------------------------------
//						TVariable
//----------------------------------------------------------------------------------------


class  TVariable : public Variable {	
 
protected:

	static TNamedObjectList  fDBaseList;
	static float fErrorVal;
  	
public:

  static byte kNoDbgCnfg;

  enum EInfo { kFrameType, kOutMode, kRemDataMap };
  
  TVariable(const char* name);
      
    virtual int Config( TConfigData& config );
    virtual int Allocate() { return 1; }
    virtual void Dump( FILE* file ) {;}
		virtual void SetupInternals();
		
    virtual inline float Value() const { return 0.0;}
    virtual inline float Value( TCell* c ) const { return 0.0; }
		inline virtual void UpdateInternals() { Variable::UpdateInternals(); }
		inline void SetLinkEdges() { SetF(FLinkEdges,True,"SetLinkEdges"); }
#ifdef POSTGRESS
		sme_pg95* GetDBase(const char* name);
#endif

    virtual void AddFlux( TVariable* f ) {;}
    inline virtual float& Reserve() { return fGarbage; }

    inline float GenericValue( TCell* c=0 ) const {
      if( GetF(FisSpatial) ) return Value( c );
      else return Value();
    }
    virtual void AddFlags(); 
      
    virtual void Reset() {
			for( Pix p = fCommandList.first(); p; fCommandList.next(p) ) {
				TCommand& c = (TCommand&) fCommandList(p);
				if( c.GetObjInfo(TCommand::kExType) == TCommand::kPipe ) {
					c.GetPipe()->Reset();
				}
			}
    }
    
    virtual void SSInit( Module* mod ) {
      Variable::Init( mod );
    }
     
};

//  TVariable Flags

 extern FlagType	FFluxOriginMap ;		//  Flux Origin has different Grid
 extern FlagType	FFluxDestinationMap ;		//  Flux Destination has different Grid
 extern FlagType	FMapRemoteData ;		//  Input Var and Connection have different Grids
 extern FlagType	FICFlux  ;		//  calls ICFlux method
 extern FlagType	FAllocated ;		//  Allocate() Method has been called.
 extern FlagType	FEdgesLinked;		//  LinkEdges() Method has been called since last update.

class  CVariable : public TVariable  {	
public:
  enum EUpdateMode { kFastSpatial, kSpatial, kTemporal };     
  enum EValueMode { kValueSpatial, kValueSpatialArray, kValueNonSpatial, kValueNonSpatialArray, kDBParm, kTSeries, kParm };
  enum EBoundaryCondition { kDefaultBC, kPeriodicBC, kContinuousBC, kZeroBC, kTestBC };
    
  protected:

  EValueMode fValueMode;
  EUpdateMode fUpdateMode;
  EType fClass;
  MultiCoverage fCov;   
  float fMaxDelay;
  intPixVHMap* fCatMapPtr;

    int fBuffsize;
    CVariable* fMaps[4];
    TObjectList fPSetList; 
    int _Array_Offests[8];
    int _Array_Sizes[8];
    int _Array_dimension;
    
    TSpatialSeries fSHistory;   
	TTemporalSeries fTHistory; 
	floatVec fDataBuffer;  

    void PrintUpdateMsg( TCell* c, float value, float dv, char* label, TCell* cR=0 );    
    void PrintUpdateMsg(float value, float dv, char* label ); 

  public:
 
    CVariable(const char* name);
    ~CVariable() {;}
    void Connect( CVariable* aVar ) { MakeConnection(aVar); }
    void Map( CVariable* aVar );
		inline ByteGrid* RegionMap() { int cell_layer; return fCov.RegionMap(cell_layer); }
		inline void SetCategories( intPixVHMap& catmap ) { fCatMapPtr = &catmap; }
		inline intPixVHMap* GetCategories() { return fCatMapPtr; }
	
    virtual int  Allocate();
    TMultiFrame* GetMultiFrame();
    
    virtual int  AllocateSpatialVariable( int activationLevel, int linkLayer );

    inline int  Allocate( float ival ) { 
      int rv = Allocate(); 
      if(rv) { Set(ival); } 
      return rv; 
    }

    inline Bool Allocated() { return GetF(FAllocated); }
    
    inline void setActivationLevel( int activationLayer, int linkLayer=-1 ) { 
	  if( GetF(FAllocated) && ( activationLayer >= 0 ) && (activationLayer != fActivationLayer) ) { 
		sprintf(gMsgStr,"Can't change activation Level After Variable is Allocated: %s",Name()); gPrintErr(); return; 
	  }
	  fActivationLayer = activationLayer; 
	  if( linkLayer >= 0 ) { fLinkLayer = linkLayer; }
	}


    void BroadcastValue();

    void UpdateRel( TCell* c0, float value, int dr, int dc, int dv=0  );

    float RelValue( TCell* c0, int dr, int dc, int dv=0, EBoundaryCondition bc = kDefaultBC );
    
    void collapseVolumeToSurface( CVariable& surfVar, CVariable* cellWeights=NULL, int layer_index=-1 );

    float Delay( TCell* c, float dTime, float ival=FLT_MAX );
    float Delay( int array_index[4], TCell* c, float dTime, float ival=FLT_MAX ) { return CVariable::Delay( array_index, dTime, ival ); }
    
		inline float Delay( TCell* c, CVariable& v, float ival=FLT_MAX ) { return Delay( c, v.Value(c), ival ); }
      //  Warning: Increasing delayTime during simulation can result in initval being returned.
    float InitVal(TCell* p);              
    float* InitValData();              

    virtual int Config(TConfigData& config);
    
    int  SetDataValues( float* data, int layOut, int size );

    void RefreshData();
    void WriteConfigXML( FILE* oFile );
    int BuildConfigXML( ArgArray& msg, int mode );
		
	virtual inline float Value( TCell* c ) = 0;
	virtual inline float Value( unsigned int p0, unsigned int p1, int level = -1 )  = 0;
	virtual inline float Value() = 0;

	virtual inline float ArrayValue( int index0, int index1 = -2, int index2 = -2, int index3 = -2  )  { 
	  return fSHistory.CurValue( getArrayIndex(index0,index1,index2,index3) );  
	} 

	virtual inline float ArrayValue()  { return Value(); }
	virtual inline float ArrayValue( TCell* c, int index0, int index1 = -2, int index2 = -2, int index3 = -2  )  { 
	  return fSHistory.CurValue( getArrayIndex(index0,index1,index2,index3) );  
	} 
		   
    inline MultiCoverage& Cov() { return  fCov; }
    inline void DumpToFile(FILE* oFile) { 
	  if( !fSHistory.empty() ) { fSHistory.DumpToFile(oFile); }
	  if( !fTHistory.empty() ) { fTHistory.DumpToFile(oFile); }
	}
    
    inline DistributedGrid& Grid( int gridLayer = -1 ) { 
			return *(fCov.Grid(gridLayer,fLinkLayer));
		}
    
    inline int32 SetValue( TCell* c, float value ) { return fCov.SetValue( c, value ); }
    
	virtual void UpdateValueExternal( const char* external_mod_name, const char* external_var_name, EVarLink::EModality modality );
 
    inline void SetValue( unsigned int row, unsigned int col, float value, int layer_index = -1 ) { 
			if( GetF(FisSpatial) ) {
				TLayer* l = fCov.multiGrid()->getCellLayer( layer_index );
				TCell* c = l->getCell(row, col);
				if (c != NULL) { Update( c, value ); }
			} else {
				Update(value);
			}
		}
		
    inline int32 AddValue( TCell* c, float value ) { return fCov.AddValue( c, value ); }
    inline Region2& Region(int iproc=gIProc ) const { return fCov.Region(iproc); } 
    inline  void GlobalRegion( Region2& r ) const { fCov.GlobalRegion(r); } 
    int CopyToMap(  TMap& map, int layerIndex = -1, int destProc = 0  );
    TMap* GetDataArray( EMapClass dataClass, int destProc ); 
    void CopyFromMap( TMap& map, int layer_index=-1, int srcProc=0 ) { fCov.CopyFromMap(map,layer_index,srcProc,Name()); }
    int  CopyToDataSet( DataSet& d, int flush_time_buffer = 1, int time_buffer_size = 1, int layer_index = -1 );
	void collapseVolumeToSurface( CVariable& surfVar, int doAve );
	int readGeometryFile( const CString& pathName, LayerConfig* lc );
    
	virtual float getMax(){
	  switch( fValueMode ) {
		case kValueSpatial: 
			return fCov.Reduce ( kR_Max, -1, NULL, 1 );  
		case kValueNonSpatial: 
			return  fTHistory.max();  
		case kValueNonSpatialArray: {
		  float rv = -FLT_MAX;
		  int asize = getArraySize();
		  float* data = fSHistory.LastData();
		  for( int i=0; i<asize; i++ ) {
			float val = data[ i ];  
			if(  val > rv ) rv = val;
		  }
		  return rv;
		}
	  } 
	}
	void MakeStatusDataString( CString& str, Bool append );
	virtual int  addArrayDeclaration( ArrayRec* array, int array_index, int is_dimension_name, int index ) ;
	
	virtual float getMax( CVariable& mask  ){
	  switch( fValueMode ) {
		case kValueSpatial: 
			return fCov.Reduce ( kR_Max, -1, mask.Cov(), 1 );  
		case kValueNonSpatial: 
			return  fTHistory.max();  
		case kValueNonSpatialArray: 
		  return getMax();
	  }
	}

	virtual float getSum(){
		switch( fValueMode ) {
		case kValueSpatial: 
			return fCov.Reduce ( kR_Sum, -1, NULL, 1 );  
		case kValueNonSpatial: 
			return  fTHistory.max();   
		case kValueNonSpatialArray: {
		  float rv = 0;
		  int asize = getArraySize();
		  float* data = fSHistory.LastData();
		  for( int i=0; i<asize; i++ ) {
			rv += data[ i ];  
		  }
		  return rv;
		}
	  }
	}

	virtual float getSum( CVariable& mask ){
	  switch( fValueMode ) {
		case kValueSpatial: 
			return fCov.Reduce ( kR_Sum, -1, mask.Cov(), 1 );  
		case kValueNonSpatial: 
			return  fTHistory.max();   
		case kValueNonSpatialArray: 
		  return getSum();
	  }
	}

	virtual float getMin(){
		switch( fValueMode ) {
		case kValueSpatial: 
			return fCov.Reduce ( kR_Min, -1, NULL, 1 );  
		case kValueNonSpatial: 
			return  fTHistory.min();       
		case kValueNonSpatialArray: {
		  float rv = FLT_MAX;
		  int asize = getArraySize();
		  float* data = fSHistory.LastData();
		  for( int i=0; i<asize; i++ ) {
			float val = data[ i ];  
			if(  val < rv ) rv = val;
		  }
		  return rv;
		}
	  }
	}  
	
	virtual float getMin( CVariable& mask ){
		switch( fValueMode ) {
		case kValueSpatial: 
			return fCov.Reduce ( kR_Min, -1, mask.Cov(), 1 );  
		case kValueNonSpatial: 
			return  fTHistory.min();     
		case kValueNonSpatialArray: 
		  return getMin();
		}
	}    
  
    inline void Set(float value) { 
		if( GetF(FisSpatial) ) {fCov.Set(value); }
		else { 
		  switch( fValueMode ) {
			case kValueNonSpatial: 
			  fTHistory.Update(gTime(),value);
			case kValueNonSpatialArray: {
			  int dsize = getArraySize();
			  if( fSHistory.LastTimeStamp() < gTime() ) {
				fSHistory.Allocate( gTime(), dsize ); 
			  }
			  float* data = fSHistory.LastData();
			  for( int i=0; i<dsize; i++ ) { data[i] = value; }
			}
		  }		
		}
	}

    inline const float* DataEnd() { return  fCov.DataEnd(); }
		inline const float* GridEnd() { return  fCov.GridEnd(); }

		virtual float* Data() { 
			if( GetF(FisSpatial) ) { return  fCov.DataStart(); }
			else { 
			  switch( fValueMode ) {
				case kValueNonSpatial: 
				 return (float*) fTHistory.TimeSeriesData(); 			 
				case kValueSpatialArray: 
				case kValueNonSpatialArray: 
				  return fSHistory.LastData();
			  }
			}
		}
		
		virtual void PrettyPrint( CString& result, int start=0, int end=-1 );
		
		virtual float* DataStart() { 
			if( GetF(FisSpatial) ) { return  fCov.DataStart(); }
			else { 
			  switch( fValueMode ) {
				case kValueNonSpatial: 
				  return (float*) &fTHistory.LastValue(); 
				case kValueSpatialArray: 
				case kValueNonSpatialArray: 
				  return fSHistory.LastData();
			  }
			}
		}
		
		inline const float* TimeSeriesData( int array_index, int& length ) {
		  length = fSHistory.length();
		  return fSHistory.TimeSeriesData(array_index);
		}
		
		virtual int DataSize() { 
			if( GetF(FisSpatial) ) { return  fCov.Size(); }
			else {  
			  switch( fValueMode ) {
				case kValueNonSpatial: 
				  return  fTHistory.Size(); 
				case kValueNonSpatialArray: 
				  return getArraySize();  
			  }
			}
		}


		inline void GrabMemory() { fCov.GrabMemory(); }
		inline int CheckMemory() { return fCov.CheckMemory(); }

    virtual inline CVariable* Connection() { if(fConnection) return (CVariable*) fConnection; else return this; }
    inline int activationLayerIndex() { return fActivationLayer; }

    inline int32 InitValue(TCell* c, float value) {
      int32 rv = SetValue(c,value);  
      if( GetCInfo(kDebug) ) {
				DoBoundsCheck(value,c);
#ifdef DEBUG
				PrintUpdateMsg(c,Value(c),value,"INIT");
#endif
			}
			return rv;
    }
    
		inline int LinkEdges() {
				lprint2( "\nLinkEdges(%f): %s", gTime(), Name() ); 
//			if( GetF(FEdgesLinked) ) return 0;
//			else { 
				fCov.LinkEdges( Name() ); 
//				SetF(FEdgesLinked,True,"LinkEdges"); 
				return 1;
//			} 
		}
		
		void CVariable::Dump( char* msg, int pl0=0, int pl1=0, FILE* file=stdout, int s0 = 15, int s1 = 15 );
		
		inline virtual void Reset() {
			TVariable::Reset();    
			fSHistory.Reset();  
			fTHistory.Reset();
		}

		virtual float ArraySum();		
		virtual float ArrayMean();		
		virtual float ArraySum(int index0, int index1 = -1, int index2 = -1, int index3 = -1);	
		virtual float ArrayMean(int index0, int index1 = -1, int index2 = -1, int index3 = -1);
		
		virtual inline float ArraySum(Pix p, int index0, int index1 = -1, int index2 = -1, int index3 = -1) { 
		  return ArraySum(index0,index1,index2,index3); 
		}
		virtual inline float ArrayMean(Pix p, int index0, int index1 = -1, int index2 = -1, int index3 = -1) { 
		  return ArrayMean(index0,index1,index2,index3); 
		}
		
		void CopyData(CVariable& v);			
		void AddData(CVariable& v);
		
	virtual int GraphData( const char* timeseries_name, const char* graph_name=NULL, float* fd=NULL, int ds=0, Bool overwrite = False  );
	virtual int DisplayData( const char* display_name, float* fd, float fmax, float fmin, int s0, int s1, int s2=1 ); 

    virtual void UpdateInternals();
    
    inline void UpdateDelayData() {
			if( fMaxDelay > 0 ) {
				fSHistory.Add( gStep(), fCov );
				fSHistory.DeleteBelow( gTime()-fMaxDelay );
			}
		}
    virtual void SSInit( Module* mod );
    
	CVariable* GetSimilarVariable(char* name);
		
    virtual inline void Update( TCell* c, float value ) {
	  switch( fValueMode ) {
		case kValueSpatial: {
		  fCov.SetValue( c, value ); 
		  if( !isValidValue( value ) ) { ProcessBoundsError( value, c , 'p' ); }
#ifdef DEBUG
		  if( GetCInfo(kDebug) ) {
			DoBoundsCheck(value,c);
			PrintUpdateMsg(c,value,Value(c),"UPDATE");
		  }
#endif
		} break;
		case kValueNonSpatial: { 
			Update(value);
		} break;
	  };     
/*
#ifdef HAS_MPE
	  if( GetF(FcalcGOF) ) {
		GOFCalc::updateData(FullName(),c,value); 
	  }
#endif
*/
    }

    virtual inline void ArrayUpdate( TCell* c, float value, int array_index[4] ) {
	  switch( fValueMode ) {
		case kValueSpatialArray: {
		  if( fSHistory.LastTimeStamp() < gTime() ) {
			fSHistory.Allocate( gTime(), getArraySize() ); 
		  }
		  float* fdata = fSHistory.LastData();
		  fdata[ getArrayIndex(array_index) ] = value;
		  if( !isValidValue( value ) ) { ProcessBoundsError( value, c , 'p' ); }
		} break;
		default:
		  gPrintErr( format("Illegal call to %s.ArrayUpdate()", Name()) );
	  };     
    }

    virtual inline void ArrayUpdate(  float value, int array_index[4] ) {
	  switch( fValueMode ) {
		case kValueNonSpatialArray: {
		  if( fSHistory.LastTimeStamp() < gTime() ) {
			fSHistory.Allocate( gTime(), getArraySize() ); 
		  }
		  float* fdata = fSHistory.LastData();
		  fdata[ getArrayIndex(array_index) ] = value;
		  if( !isValidValue( value ) ) { ProcessBoundsError( value, NULL , 'p' ); }
		} break;
		default:
		  gPrintErr( format("Illegal call to %s.ArrayUpdate()", Name()) );
	  };     
    }
    
    inline int GetCategorySize( const char* array_element ) {
	  ArgArray* a = NULL;
	  CString name(array_element);
	  if( fArrayRec ) {
		 if( !fArrayRec->SName().equals_nocase( name ) ) {
			gPrintErr( format("Array Name mismatch in %s: %s vs %s", Name(), fArrayRec->Name(), array_element) );
		 } else {
		   a = fArrayRec->getArgs(); 	
		 }   
	  }
	  if( a == NULL ) {
		 int index, is_dimension_name;
		 ArrayRec* ar = Model::I0().getArrayDeclaration( name, index, is_dimension_name ); 
		 if( ar == NULL ) {
			gPrintErr( format("Unknown Array: %s", array_element) );
			return 0;
		 }
		 a = ar->getArgs(); 
	   } 
	   return a->items();
    }
    
    inline int getArraySize() {
	  if( _Array_Offests[0] < 0 ) {
		if( GetF(FisImport) ) {
		  if( fConnection == NULL ) {
			  sprintf(gMsgStr,"Undefined Connection for input Variable %s:%s", fModule->Name(), Name() );
			  gFatal();
		  }
		  CVariable* svar = (CVariable*)fConnection;
		  svar->getArraySize();
		  for( int i=0; i<8; i++ ) { 
			_Array_Offests[i] = svar->_Array_Offests[i]; 
			_Array_Sizes[i] = svar->_Array_Sizes[i]; 
		  }
		} else {
		  int asize = 1, index = 0;
		  for( int iA = 0; iA < kMaxNVarArrays; iA++ ) {
			ArrayRec* ar =  _Arrays[iA];
			if( ar != NULL ) {
			  ArgArray* a = ar->getArgs(); 
			  asize = asize*a->items();
			  _Array_Sizes[index] = a->items();
			  _Array_Offests[++index] = asize;
			} else { break; }
		  } 
		  _Array_Offests[0] = (index>0) ? _Array_Offests[index] : 0;
		  _Array_dimension = index;
		}
	  } 
	  return  _Array_Offests[0]; 
    }
    
    inline int getArrayDimension() {
	  if( fValueMode == kValueNonSpatialArray ) {
		int as = getArraySize();
		return _Array_dimension;
	  }
	  return 0;
	}

   
    inline int getArrayIndex( int array_index[4] ) {
/*
		int i1= -1, i2 = -1, i3 = -1;
		int i0 =  array_index[ _Array_Index_Map[0] ];
		if( _Array_Index_Map[1] >= 0 ) {
		  i1 =  array_index[ _Array_Index_Map[1] ];
		  if( _Array_Index_Map[2] >= 0 ) {
			i2 =  array_index[ _Array_Index_Map[2] ];
			if( _Array_Index_Map[3] >= 0 ) {
			  i3 =  array_index[ _Array_Index_Map[3] ];
			}
		  }		
		}
*/
		return getArrayIndex( array_index[0], array_index[1], array_index[2], array_index[3] );
    }

    inline int getArrayIndex( int index0, int index1=-1, int index2=-1, int index3=-1 ) {
		int rv = index0;
		if( index1 > 0 ) {
		  rv += index1*_Array_Offests[1]; 
		  if( index2 > 0 ) {
			rv += index2*_Array_Offests[2]; 
			if( index3 > 0 ) {
			  rv += index3*_Array_Offests[3]; 
			}
		  }
		}
		return rv; 
    }
    
    inline float& operator() ( const OrderedPoint& pt ) {
      return fCov( (TCell*) &pt );
    }    
    inline int ClassValue( TCell* p) {
      return (int) fCov.Value( (TCell*)p );
    }
    
//   inline float Time() { return  fTHistory.LastTimeStamp(); }

   inline float& operator() () {
     return fTHistory.LastValue();
   }

   inline float LastValue() { return fTHistory.LastValue(); }

   inline float InitVal( ) {
     return fTHistory.FirstValue();
   }
   
   float Delay( float dTime, float ival=FLT_MAX );
   float Delay( int array_index[4], float dt, float ival=FLT_MAX );
   
   inline float Delay( CVariable& v, float ival=FLT_MAX ) { return Delay( v.Value(), ival ); }
 
   virtual void Update( float value );

   void InitValue(float value);
   
	 inline EUpdateMode UpdateMode() { return fUpdateMode; }
	 
	virtual void SetupInternals();
	int SetDBParamValues();

	virtual int ExecuteIgnoreFunction()  { 
		float value = atof(fDefValue.elem(0));
		Update( value ); 
		return 0;
	}

	inline int DoBoundsCheck( float value, TCell* c = 0 ) { 	
		byte error = fBounds.Check(value);
		return (error) ? ProcessBoundsError( value, c, error ) : 0;
	}
	
	inline int isValidValue( float val ) { return ( val < FLT_MAX ) && ( val > -FLT_MAX ); }
	
	int ProcessBoundsError( float value, TCell* p , byte error );	
};
	
//----------------------------------------------------------------------------------------
//						CStateVariable
//----------------------------------------------------------------------------------------
typedef class CFluxVariable CFluxVariable;

class  CStateVariable : public CVariable {
    
protected:
	
	MultiCoverage fSReserve;
	float* fAReserve;
	TSpatialSeries fSIntegratorMemory; 
	int fCurrentOrder;  

	float fTReserve;
	TTemporalSeries fTIntegratorMemory;    

	
public:

  static EConfigInfo kUpdateMode;

  CStateVariable(const char* name);
  virtual int Allocate();
  inline void SetUpdateMode(EUpdateMode upDateMode) {  SetCInfo(kUpdateMode,upDateMode); }
  inline EUpdateMode GetUpdateMode() {   return (EUpdateMode) GetCInfo(kUpdateMode); }
  inline void Connect( CStateVariable* aVar ) { MakeConnection(aVar); }
  inline float Reserve(TCell* c) { return fSReserve.Value( c ); }
  inline void SetReserve(TCell* c, float value) { fSReserve.SetValue( c, value ); }
  inline int AddReserve(TCell* c, float value) { 
	return fSReserve.AddValue( c, value ); 
  }
  inline float* RDataStart() { return  fSReserve.DataStart(); }
  inline void SetMap ( CStateVariable* aMVar ) { Map( aMVar ); }
  float Update( float value, float dt );
   
   inline float& Reserve() { return fTReserve; }
  inline float& ArrayReserve( int array_index[4] ) { return fAReserve[ getArrayIndex(array_index) ]; }
  float GetIntegratorValue(float value);
  float GetReserveAdjustment();
  float GetReserveAdjustment( int mIndex ); 
  float GetFluxValueFromIntegrator( float value );
  
  void AddFlux( CFluxVariable* f );
  inline void SetOrder(int o) { fOrder = o; }
	virtual void SSInit( Module* mod );
 
	virtual int Config(TConfigData& config);

    virtual inline int  AllocateSpatialVariable( int activationLevel, int linkLayer ) {
	  CVariable::AllocateSpatialVariable( activationLevel, linkLayer );
	  SetCInfo(kType,kState);
	  return 0;
	}
	
	void InitValue(float value);
	void ArrayInitValue( float value, int array_index[4] );
	void ArrayInitValue( Pix p, float value, int array_index[4] );

	 inline float* GetDataPtr( int index ) {
	   switch( fValueMode ) {
		 case kValueSpatial: 
		   return DataStart() + index; 
		 case kValueNonSpatial: 
		 case kValueNonSpatialArray: 
		   return Data(); 
	   }
	 }
	
	inline void UpdateDataPtr( float*& dRtr ) {
		switch( fValueMode ) {
			case kValueSpatial: dRtr++; break;
			case kValueNonSpatial: break;
			case kValueNonSpatialArray: break;
		}
	}

	inline virtual void Reset() {
		CVariable::Reset();
		fSIntegratorMemory.Reset();  
		fTIntegratorMemory.Reset();    
	}
	
	inline float Value( TCell* c ) {
		switch( fValueMode ) {
			case kValueSpatial: 
			  return fCov( c ); 
			case kValueNonSpatial: 
			  return Value();
		}      
	}
	
	inline float Value( unsigned int p0, unsigned int p1, int level = -1 ) {
		switch( fValueMode ) {
			case kValueSpatial: return fCov.Value( p0, p1, level ); 
			case kValueNonSpatial: return Value();
		}      
	}
	
	inline float Value(void) {
#ifdef DEBUG
		if( fTHistory.length() == 0 ) {
		  gPrintErr( SName() + ": Error, accessing uninitialized timeseries variable.");
		  return 0.0;
		}
#endif
		return  fTHistory.LastValue();
	}
  
	inline virtual void UpdateReserve() {
	  if( GetF(FisClamped) ) {
		if( fOrder > 1 ) {
		  DistributedGrid& g = Grid();
		  for( TCell* c = (TCell*)g.first(); g.onGrid(c); g.next(c) ) {
			int32 mIndex = c->memoryIndex( fActivationLayer );			
			fSReserve.elem(mIndex) = fCov.elem( mIndex ) - GetReserveAdjustment(mIndex);
		  }
		} else {
		  float* data_Rptr = fSReserve.DataStart();
		  float* data_ptr = DataStart();
		  const float* data_Eptr = DataEnd();
		  while( data_ptr < data_Eptr ) {
			  *data_Rptr++ = *data_ptr++;
		  }	
		}		
	  }	
	}
	
  virtual  inline float ArrayUpdate( TCell* c, float value, int array_index[4], float dt ) {
	return ArrayUpdate( value, array_index, dt );
  }
  
  virtual float ArrayUpdate( float value, int array_index[4], float dt );

  virtual float Update( TCell* c, float value, float dt );

  inline void InitValue(TCell* c, float value) {
    int32 index = CVariable::InitValue( c, value );
    if( GetF(FisClamped) ) { fSReserve.elem(index) = value; }	
  }

	virtual void UpdateInternals();
    	
	void UpdateIntegrator( int isInit = 0 );
	
	inline int CurrentOrder() { return fCurrentOrder; }
	inline int Order() { return fOrder; }

	inline TSpatialSeries& SIntegratorMemory() { return fSIntegratorMemory; }

};


//----------------------------------------------------------------------------------------
//						CAuxVariable
//----------------------------------------------------------------------------------------
// fValueMode: kValueSpatial, kValueNonSpatial, kDBParm, kTSeries, kParm

class  CAuxVariable : public CVariable {
protected:

// DBase fields;  
   floatVec fV;
//   EInitMode fInitMode; //  kDefault, kMap, kTimeSeries, kPtTimeSeries, kDBase, kParameter;
   float* fCurrentMapPtr;
	
  float* GetDBaseData(const char *filename, int sIndex, int pIndex, int index, int order, int& size ); 
  
  
// TSeries Fields
 TDistributedSeries fSeries;

// TPtSeries fields
  TPtSeriesList fSeriesList;

public:
  static EConfigInfo kDBOrder;
  CAuxVariable(const char* name);
  ~CAuxVariable() {;}
  inline void Connect( CAuxVariable* aVar ) { MakeConnection(aVar); SetCInfo(kInitMode,aVar->GetCInfo(kInitMode)); } 
  inline void SetMap ( CAuxVariable* aMVar ) { Map( aMVar ); }
  virtual int Config( TConfigData& config );
	virtual void RefreshData();

  virtual inline float* Data() { 
    switch( GetCInfo(kInitMode) ) {
    case kParameter: 
			if( GetF(FisSpatial) ) { return CVariable::DataStart(); }
      fData[0] = fValue; 
      return fData;
    case kDBase:
      return fV.DataStart();
    case kTimeSeries: 
      return (float*) fSeries.TimeSeriesData();
    default: 
      return CVariable::Data();
    }
  }

  virtual inline int  AllocateSpatialVariable( int activationLevel, int linkLayer ) {
	CVariable::AllocateSpatialVariable( activationLevel, linkLayer );
	SetCInfo(kType,kAux);
	return 0;
  }
  
  void ReadFloatData(const char* mapPath, int column );
  	
  virtual inline float* DataStart() { 
    switch( GetCInfo(kInitMode) ) {
    case kParameter:  
			if( GetF(FisSpatial) ) { return CVariable::DataStart(); }
      fData[0] = fValue; 
      return fData;
    case kDBase:
      fCurrentMapPtr = ((CVariable*)fMapVariable[0])->DataStart();
      return fV.DataStart() + (int)*fCurrentMapPtr;
    case kTimeSeries: 
      return (float*) &fSeries.LastValue();
    default: 
      return CVariable::DataStart();
    }
  }
  virtual inline int DataSize() { 
		switch( GetCInfo(kInitMode) ) {
		  case kParameter: 
				if( GetF(FisSpatial) ) { return CVariable::DataSize(); } 
				return 1;
			case kDBase:  return  fV.capacity();;
			case kTimeSeries: return fSeries.Size();
			default: return CVariable::DataSize();
		}
	}
	
  inline float Value( unsigned int row, unsigned int col, int layer_index = -1 ) {
    switch( fValueMode ) {
			case kValueSpatial: {
				TLayer* l = fCov.multiGrid()->getCellLayer(layer_index);
				TCell* c = l->getCell(row,col);
				if( c == NULL ) return fErrorVal; 
				return Value(c);
			} 
			case kDBParm: {
				TLayer* l = ((CVariable*)fMapVariable[0])->Cov().multiGrid()->getCellLayer(layer_index);
				TCell* c = l->getCell(row,col); 
				if( c == NULL ) return fErrorVal; 
				return Value(c);
			} 
			case kTSeries:
			case kParm:
			case kValueNonSpatial: 
				return Value(0);
		}
	 }


	virtual float getMax(){
		switch( fValueMode ) {
		case kValueSpatial: 
			return fCov.Reduce ( kR_Max, -1, NULL, 1 );  
		case kDBParm: {
			return fV.reduce( maxCombiner, -FLT_MAX );
		} case kTSeries:
			return fSeries.max(); 
		case kParm:
			return fValue;
		case kValueNonSpatial: 
			return  fTHistory.max();  
		case kValueNonSpatialArray: {
		  float rv = -FLT_MAX;
		  int asize = getArraySize();
		  float* data = fSHistory.LastData();
		  for( int i=0; i<asize; i++ ) {
			float val = data[i]; 
			if(  val > rv ) rv = val;
		  }
		  return rv;
		}
	  }
	}

	virtual float getMin(){
		switch( fValueMode ) {
		case kValueSpatial: 
			return fCov.Reduce ( kR_Min, -1, NULL, 1 );  
		case kDBParm: {
			return fV.reduce( minCombiner, FLT_MAX );
		} case kTSeries:
			return fSeries.min(); 
		case kParm:
			return fValue;
		case kValueNonSpatial: 
			return  fTHistory.min();       
		case kValueNonSpatialArray: {
		  float rv = FLT_MAX;
		  int asize = getArraySize();
		  float* data = fSHistory.LastData();
		  for( int i=0; i<asize; i++ ) {
			float val = data[i]; 
			if(  val < rv ) rv = val;
		  }
		  return rv;
		}
	  }
	}
	
	float Value( TCell* c );

	virtual inline float ArrayValue( int index0, int index1 = -2, int index2 = -2, int index3 = -2 ) { 
	  return CVariable::ArrayValue( index0, index1, index2, index3  ); 
	}
	virtual inline float ArrayValue( TCell* c, int index0, int index1 = -2, int index2 = -2, int index3 = -2  ) { 
	  return CVariable::ArrayValue( index0, index1, index2, index3  ); 
	}
	virtual inline float ArrayValue()  { 
	  return CVariable::ArrayValue(); 
	}
  
   virtual inline void Update( float value ) {
			switch( GetCInfo(kInitMode) ) {
				case kParameter:
//				  fValue = fParm.Value();
				  fValue = value;
				break;
				default:
					CVariable::Update( value );
			}	
   }
   
   virtual inline void Update( TCell* c, float value ) {
					CVariable::Update( c, value );
	 }

	 
   inline float ParameterValue( float default_value = 0, TCell* cell = NULL ) {
		 if( fParm.Configured() ) {
			 if( cell == NULL ) return fParm.Value();
			 return fParm.Value( fCov.CellIndex( cell ) );
		 } else {
			 return default_value;
		 }
   }
      
  inline float Value( ) {
    switch( fValueMode ) {
		case kValueSpatial: 
		case kDBParm: 
		  sprintf(gMsgStr,"\n\nNonspatial value call on spatial variable: %s\n\n", Name() ); gFatal();
		case kTSeries:
				if( GetF(FisImport) )  { 			
					 if( fConnection == NULL ) {
						 sprintf(gMsgStr,"Undefined Connection for input Variable %s:%s", fModule->Name(), Name() );
						 gFatal();
					 }
					 fValue = ((CAuxVariable*)fConnection)->fSeries.Value( gTime() ); 
					 if( fValue == kSeriesNullValue ) {
						gPrintErr( format("Empty timeseries in %s", Name() ) );
						fValue = 0.0;
					 }
					 return fValue;
				} else {
					 fValue = fSeries.Value( gTime() ); 
					 if( fValue == kSeriesNullValue ) {
						gPrintErr( format("Empty timeseries in %s", Name() ) );
						fValue = 0.0;
					 }
					 return fValue;
				}
		case kParm:
		  return fValue;
		case kValueNonSpatial: 
		 fValue = fTHistory.LastValue(); 
		 if( fValue == kSeriesNullValue ) {
			gPrintErr( format("Empty timeseries in %s", Name() ) );
			fValue =  0.0;
		 }
		 return fValue;
	}
  }
  

   inline float operator() () { return Value(); }

   inline float InitVal( ) {
    switch( fValueMode ) {
	  case kTSeries: {
		 float val = fSeries.FirstValue();
		 if( val == kSeriesNullValue ) {
			gPrintErr( format("Empty timeseries in %s", Name() ) );
			val =  0.0;
		 }
		 return val;
	  } default:
		return CVariable::InitVal();
	  }
   }

   inline void InitValue() {
     switch( fValueMode ) {
     case kParm:
       fValue = fParm.Value();
       if(GetCInfo(kDebug)) {
				 sprintf(gMsgStr,"\nINIT PARM %s -> %f",(const char*)fName,fValue); gPrintLog();
       }
       break;
     default:
       sprintf(gMsgStr,"\nIllegal usage of InitValue by %s",(const char*)fName); gPrintErr();
     }
   }
   
	 inline float* GetDataPtr( int index ) {
     switch( fValueMode ) {
     case kValueSpatial: return DataStart() + index; 
     case kValueNonSpatial: return Data(); 
     case kValueNonSpatialArray: return Data(); 
     case kDBParm: return fV.DataStart() + (int)*(fCurrentMapPtr+index);
     case kTSeries: return Data(); 
     }
	 }
	 
   inline void UpdateDataPtr( float*& dRtr ) {
     switch( fValueMode ) {
     case kValueSpatial: dRtr++; break;
     case kValueNonSpatial: break;
     case kValueNonSpatialArray: break;
     case kDBParm: { 
       fCurrentMapPtr++;
       dRtr = fV.DataStart() + (int)*fCurrentMapPtr;
       } break; 
     case kTSeries: break;
     }
   }
  
// DBaseParm methods
    
	void  MakeDBaseVar( );
	void ReadDBase(const char* fileName, int sIndex, int cellIndex, int index);
#ifdef POSTGRESS
  void ReadDBase(const char* dataBase, const pgSQLRec* sqlRec, const ArgArray& args );
#endif
  float DBParmValue( int vIndex );

  virtual int SetupParameter(float init_val, float min_value, float max_val, int  ustep );

	void ReadArrayParameter(const char* fileName);
	void SetArrayParameterValue(CString& mapName, CString& className, CString& value);
	int GetCategoryIndex( const char* label  );
	int CheckCategoryLabels( ); 
	intPixVHMap* ReadCategories( ); 
	int PrintCategories( CString& cats ); 
	virtual void PrettyPrint( CString& result, int start, int end );

// Parameter methods

  int MakeParameterVariable( );
  
// TSeries methods.

   void MakeTimeSeriesVariable();	  

//   inline float Time() { return fSeries.LastTimeStamp(); }
   

    virtual void Dump( FILE* file ) {;}

// TPtSeries methods

	void MakePtTimeSeriesVariable();
   void Read( const char* fileName, int format );

// generic timeseries methods

		inline float DT()  { 
			switch( GetCInfo(kInitMode) ) {
				case kTimeSeries: return fSeries.DT();
				case kPtTimeSeries: return fSeriesList.DT(); 
				default: return gDT();
			}
		}
		inline int Length()  { 
			switch( GetCInfo(kInitMode) ) {
				case kTimeSeries: return fSeries.length();
				case kPtTimeSeries: return fSeriesList.Size();
				default: return 1;
			}
		}
		inline void Reset()  { 
			CVariable::Reset();
			switch( GetCInfo(kInitMode) ) {
				case kTimeSeries: 
					fSeries.Reset(); break;
				case kPtTimeSeries: 
					fSeriesList.Reset(); break;
				default: break;
			}
		}
		
		virtual void SetupInternals();
		virtual int SetupDBParam( const CString& map );
		int SetDBParamValues();

// Map Input methods:

		  virtual void MakeMapInputVariable( int isInt );
		  virtual void MakeExternalIOVariable( EVarLink::EModality mod );
};

//----------------------------------------------------------------------------------------
//						CFluxVariable
//----------------------------------------------------------------------------------------


class  CFluxVariable : public CAuxVariable {
  
	byte fClampData[4];  
public:

    CFluxVariable(const char* name);
    ~CFluxVariable() {;}
    virtual int Allocate();
    float Clamp( TCell* p, float value );
    float Clamp( float value );
    float ArrayClamp( float value, int array_index[4] );
    void ICFlux( TCell* p0, float value, int dr, int dc, int dv=0 );
    		
    virtual inline int  AllocateSpatialVariable( int activationLevel, int linkLayer ) {
	  CVariable::AllocateSpatialVariable( activationLevel, linkLayer );
	  SetCInfo(kType,kFlux);
	  return 0;
	}

    inline void SetDestination(CStateVariable* aVar) { 
      Variable::SetDestination(aVar); aVar->AddFlux(this); 
    }
    inline void SetOrigin(CStateVariable* aVar) { 
      Variable::SetOrigin(aVar); aVar->AddFlux(this); 
    }
    
    virtual void Update( float value );
    virtual void ArrayUpdate( float value, int array_index[4]  );

    virtual void Update( TCell* c, float value );
    virtual inline void ArrayUpdate( TCell* c, float value, int array_index[4]  ) { 
	   CFluxVariable::ArrayUpdate( value, array_index ); 
	}

    virtual inline void SSInit( Module* mod ) {
      CVariable::SSInit( mod );
      SetF(FisFlux,True,"Init");
    }

};

#endif
