//----------------------------------------------------------------------------------------
//	DataSet.h
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//	Change History:
//----------------------------------------------------------------------------------------

#ifndef __DataSet__
#define __DataSet__

#include "Globals.h"
#include "Region2.h"

//----------------------------------------------------------------------------------------
//						DataSet
//----------------------------------------------------------------------------------------


class  DataSet : public TOrderedObject {	
	
		int _format;   // 0 : raster format,   1: reduced format
		int _dims;
		int _L0;
		int _L1;
		int _L2;
		int _datasize;
		int _buffersize;
		unsigned short* _data;
		byte* _data_map;
		float _time;
		float _dt;
		float _max;
		float _min;
		int _index;
		CString _name;

	public:

						
		DataSet( int format, byte* data_map = NULL, const char* name = NULL ) { 
			_index = -1;
			setMap( data_map );
			setFormat( format );
			_data = NULL; _buffersize = 0; 
			if( name != NULL ) { _name = name; }
			reset();
		}

		inline void setFormat( int format ) { 
		  _format = format; 
		}
		
		void setMap( byte* data_map );
		
		inline void setTimestamp( float time, float dt ) { _time = time; _dt = dt; }


		inline void setSize( Region2& r, int datasize = 0 ) {  
			setSize( r.extents(1), r.extents(0), 1, datasize ); 
		}
		 			
		void setSize( int l0, int l1=1, int l2=1, int datasize = 0 );
		
		inline void reset() {
			_datasize = _L0 = _L1 = _L2 = 0; 
			_max = -FLT_MAX; _min = FLT_MAX;
		}
		
		inline void setDataValue( int index, unsigned short val ) {
			_data[index] = val;
		}

		inline void fill( unsigned short val ) {
			for( int index=0; index<_datasize; index++ ) { _data[index] = val; }
		}
		
		inline void setDims( int dim ) {
			_dims = dim;
		}
		
		inline int getDims() {
			return _dims;
		}

		inline int getSize( int dim ) {
			switch( dim ) {
				case 0: return _L0;
				case 1: return _L1;
				case 2: return _L2;
				default: return 0;
			}
		}		

		inline void	check_data( const char* name ) {
			int val, max=-1, min=-1;
			for( int i=0; i<_datasize; i++ ) {
				val = _data[i];
				if( (max<0) || (val>max) ) { max = val; }
				if( (min<0) || (val<min) ) { min = val; }
			}
			sprintf(gMsgStr,"\nChecking data for %s(%d): max: %d (%f), min: %d (%f)\n", name, _datasize, max, _max, min, _min ); 
			gPrintLog();
		}
		
//		inline CString setName( const char* name ) { _name = name; }
		inline void setIndex( int index ) { _index = index; }

		inline CString name() { return _name; }
		inline int index() { return _index; }
		inline int format() { return _format; }
		inline float& max() { return  _max; }
		inline float& min() { return  _min; }
		inline byte* dataMap() { return _data_map; }
		inline int dataSize() { return _datasize; }
		
		inline void Dump( FILE* oFile ) {
			fprintf(oFile,"%f%f",_max,_min);
			byte* data = (byte*)_data;
			byte* data_end = (byte*)(_data+_datasize);
			while( data < data_end ) {
				fputc(*data++,oFile);
			}
		}
		
		inline void dbg_dump(char* msg) {
			fprintf(stderr,"\n    DataSet dump: %s; heXdata=",msg);
			for( int i=0; i<_datasize; i++ ) {
				fprintf(stderr," %x ",_data[i]);
			}
			fprintf(stderr,"\n"); fflush(stderr);	
		}


		inline void Read( FILE* oFile ) {
			fscanf(oFile,"%f%f",&_max,&_min);
			byte* data = (byte*)_data;
			byte* data_end = (byte*)(_data+_datasize);
			while( data < data_end ) {
				*data++ = fgetc(oFile);
			}
		}

		inline unsigned short* data() { return _data; }
		inline float time() { return _time; }
		inline float DT() { return _dt; }

};

#endif
