#include "Fluxes.h"
#include "SSModel.h"

#define iabs(x) (((x)>0) ? (x) : -(x))

/*********************************************************************************************/

TSpatialFluxVariable Flux::fSWfluxWE("SWfluxWE");
TSpatialFluxVariable Flux::fSWfluxNS("SWfluxNS");
TSpatialFluxVariable Flux::fGWfluxWE("GWfluxWE"); 
TSpatialFluxVariable Flux::fGWfluxNS("GWfluxNS");
TVariable* Flux::fCellSize = NULL;

float Flux::fAlpha[5] = { 0.5, 0.5, 0.5, 0.5, 0.5 };
float Flux::fSFluxIn, Flux::fSFluxOut, Flux::fGFluxIn, Flux::fGFluxOut;  
float Flux::fBaseFluxRate = 1.0;
Pix Flux::fCurrentSPoint = 0, Flux::fCurrentGPoint = 0;
TModule* Flux::fCurrentModule = NULL;
DGrid*  Flux::fCurrentGrid = NULL;
int FSetup = 0;
			
void Flux::Setup() {
	if(FSetup) return;
  if( fCurrentModule == NULL ) { gPrintErr( "No Frame Module Specified for Flux Module."); return; }
  fCurrentGrid = fCurrentModule->Frame().Grid();
  if( fCurrentGrid->GetInfo(DGrid::kFrameType) != DGrid::kGrid) gFatal( "Flux must be declared in Grid Frame. " );
  ((Variable&)fSWfluxWE).Init(fCurrentModule);
  ((Variable&)fSWfluxNS).Init(fCurrentModule);
  ((Variable&)fGWfluxWE).Init(fCurrentModule);
  ((Variable&)fGWfluxNS).Init(fCurrentModule);
  ((TSpatialVariable&)fSWfluxWE).Allocate(0.0);
  ((TSpatialVariable&)fSWfluxNS).Allocate(0.0);
  ((TSpatialVariable&)fGWfluxWE).Allocate(0.0);
  ((TSpatialVariable&)fGWfluxNS).Allocate(0.0);
  FSetup = 1;
}

int Flux::Config(TConfigData& cd) {
  const CString& cmd = cd.Cmd();
  if(gDebug) { 
    sprintf(gMsgStr,"Reading Flux Config: Nargs = %d, CMD = %s", cd.NArgs(), cmd.chars() );	
    gPrintLog();  
  }
  if( cmd == "cs" ) {
    CString* modName = cd.Arg(0,CD::kRequired );
    CString* varName = cd.Arg(1,CD::kRequired );
    TModule* mod = (TModule*) ((modName) ? Model::I0().GetModule( *modName, False ) : NULL);
    if( mod == NULL ) { gPrintErr( *modName + ": Can't find Module."); return 0; } 
    TVariable* var =  (TVariable*) mod->GetVariable(*varName,False);
    if( var == NULL ) { gPrintErr( *varName + ": Can't find Variable."); return 0; }   
    fCellSize = var;
  } else if( cmd == "m" ) {
    CString* modName = cd.Arg(0,CD::kRequired );
    fCurrentModule = (TModule*) ((modName) ? Model::I0().GetModule( *modName, False ) : NULL);
    if( fCurrentModule == NULL ) { gPrintErr( *modName + ": Can't find Module."); return 0; } 
  } else if ( cmd == "f" ) {
    cd.FloatArg(0,fBaseFluxRate,CD::kRequired );
  } else if ( cmd == "a" ) {
    cd.FloatArg(0,fAlpha[0],CD::kRequired );
    cd.FloatArg(1,fAlpha[1]);
    cd.FloatArg(2,fAlpha[2]);
  } else return 0;
  return 1;
}

float Flux::SWInP( Pix p, TSpatialVariable& SWater, TSpatialVariable& Elevation, TSpatialVariable& MC ) {
  if(fCurrentSPoint != p) {
    SWP( p, SWater, Elevation, MC );
    fCurrentSPoint == p;
  }
  return fSFluxIn;
}

float Flux::SWInHeadP( Pix p, TSpatialVariable& SWater, TSpatialVariable& Elevation, TSpatialVariable& MC ) {
  if(fCurrentSPoint != p) {
    SWP( p, SWater, Elevation, MC, kHead );
    fCurrentSPoint == p;
  }
  return fSFluxIn;
}

float Flux::SWOutP( Pix p, TSpatialVariable& SWater, TSpatialVariable& Elevation, TSpatialVariable& MC ) {
  if(fCurrentSPoint != p) {
    SWP( p, SWater, Elevation, MC );
    fCurrentSPoint == p;
  }
  return fSFluxOut;
}

float Flux::SWOutHeadP( Pix p, TSpatialVariable& SWater, TSpatialVariable& Elevation, TSpatialVariable& MC ) {
  if(fCurrentSPoint != p) {
    SWP( p, SWater, Elevation, MC, kHead );
    fCurrentSPoint == p;
  }
  return fSFluxOut;
}

/*********************************************************************************************/

float Flux::SWP( Pix p0, TSpatialVariable& SWater, TSpatialVariable& Elevation, TSpatialVariable& MC, EFluxUnits uf )
{	
  float ff;
  float SWfluxS = 0., SWfluxW = 0., SWfluxN = 0., SWfluxE = 0.;
  Setup();

  float flux = 0.0; 
  fSFluxOut = fSFluxIn = 0.0;

  if ( AllowFlux(kSW,p0 ) )
    {   /* the pairs of cells are "from" "to" */
      // Point2 pSS(1,0); pSS = Point2(x,y);
      Pix pS = fCurrentGrid->TranslateByLinks(p0,pSS);
      SWfluxS = SurfaceWater ( p0, pS, SWater, Elevation, MC );  
      if ( SWfluxS > 0 ) fSFluxOut += SWfluxS;
      else fSFluxIn -= SWfluxS;
	  	
      Pix pE = fCurrentGrid->TranslateByLinks(p0,pEE); 
      SWfluxE = SurfaceWater ( p0, pE, SWater, Elevation, MC );
      if ( SWfluxE > 0 ) fSFluxOut += SWfluxE;
      else fSFluxIn -= SWfluxE;
	  
      Pix pN = fCurrentGrid->TranslateByLinks(p0,pNN);  
      if ( pN == 0 )
	SWfluxN = SurfaceWater ( pN, p0, SWater, Elevation, MC );
      else SWfluxN = fSWfluxNS.Value(pN);
      if ( SWfluxN < 0 ) fSFluxOut -= SWfluxN;
      else fSFluxIn += SWfluxN;
	  
      Pix pW = fCurrentGrid->TranslateByLinks(p0,pWW); 
      if ( pW == 0 )
	SWfluxW = SurfaceWater ( pW, p0, SWater, Elevation, MC );
      else SWfluxW = fSWfluxWE.Value(pW);
      if ( SWfluxW < 0 ) fSFluxOut -= SWfluxW;
      else fSFluxIn += SWfluxW;
	  	  
      flux = fSFluxIn - fSFluxOut;
	  
      float cs = fCellSize->GenericValue(p0);

      if ( (ff = SWater.Value(p0)*cs + flux) < 0  && fSFluxOut > 0) { 
	ff = (fSFluxOut + ff)/fSFluxOut; 
	if ( SWfluxS > 0 ) SWfluxS *= ff;
	if ( SWfluxE > 0 ) SWfluxE *= ff;
	if ( SWfluxN < 0 ) SWfluxN *= ff;
	if ( SWfluxW < 0 ) SWfluxW *= ff;
	flux = - SWater.Value(p0)*cs;
	fSFluxIn *= ff;
	fSFluxOut *= ff;
      }
      if( uf == kVolume ) {
	fSWfluxNS.Update(p0,SWfluxS);
	fSWfluxWE.Update(p0,SWfluxE);
      } else {
	fSWfluxNS.Update(p0,SWfluxS/cs);
	fSWfluxWE.Update(p0,SWfluxE/cs);
	fSFluxIn /= cs;
	fSFluxOut /= cs;
      }
		  
    }

  if( fCurrentGrid->GetPointInfo ( p0, DGrid::kDebug )  && gDebug ) {
    sprintf( gMsgStr, 
	     "(%x):SW Fluxes: Sout= %.3f, Eout= %.3f, Nin=%.3f, Win= %.3f, Total= %.3f, SFOut= %.3f\n",
	     p0,SWfluxS,SWfluxE,SWfluxN,SWfluxW,flux,fSFluxOut); 
    gPrintLog(); 
  }
		
  return(flux);
}
/*********************************************************************************************/

float Flux::SurfaceWater( Pix p0, Pix p1, TSpatialVariable& SWater, TSpatialVariable& Elevation, TSpatialVariable& MC )
{	
  float w, fr, H, cs,  F = 0.;
  double dh, adh;
  /* the pairs of cells are "from" p0  "to" p1 */
 	
  if(fCellSize==NULL) gFatal("Error, CELLSIZE undefined in Flux Module ( use cs() command )" ); 
  if( AllowFlux(kSW,p0, p1) ) {
    dh = ( Elevation.Value(p0) + SWater.Value(p0)) - (Elevation.Value(p1) + SWater.Value(p1) );
    /* dh is "from -- to" */
    adh = fabs (dh);
			
    if (dh > 0) 
      {
	cs = fCellSize->GenericValue(p0);
	w = Util::ramp(SWater.Value(p0));
	fr = pow(cs,0.25) * pow(w,fAlpha[1]) * fBaseFluxRate / MC.Value(p0);
	
	F = fr * pow(adh,fAlpha[2]);
	H = F/cs;
	if ( (Elevation.Value(p0) + SWater.Value(p0) - H) < (Elevation.Value(p1) + SWater.Value(p1) + H) )
	  F = dh/2 * cs;				
      }
    else
      {	
	cs = fCellSize->GenericValue(p1);
	w = Util::ramp(SWater.Value(p1));
	fr = pow(cs,0.25) * pow(w,fAlpha[1]) * fBaseFluxRate / MC.Value(p1);
	
	F = - fr * pow(adh,fAlpha[2]);
	H = F/cs;
	if ( (Elevation.Value(p0) + SWater.Value(p0) - H) > (Elevation.Value(p1) + SWater.Value(p1) + H) )
	  F = dh/2 * cs;				
      }
  }
  return F;
}

/*********************************************************************************************/

float Flux::GWP( Pix p0, TSpatialVariable& GW_head, TSpatialVariable& tot_head, TSpatialVariable& rate, EFluxUnits uf ) {
  float ff;
  float GWfluxS = 0., GWfluxW = 0., GWfluxN = 0., GWfluxE = 0.;
  Setup();

  float flux = 0.0; 
  fGFluxOut = fGFluxIn = 0.0;

  if ( AllowFlux(kGW,p0 ) ) {   /* the pairs of cells are "from" "to" */
    Pix pS = fCurrentGrid->TranslateByLinks(p0,pSS);
    GWfluxS = GroundWater ( p0, pS, GW_head, tot_head, rate );  
    if ( GWfluxS > 0 ) fGFluxOut += GWfluxS;
    else fGFluxIn -= GWfluxS;
	  	
    Pix pE = fCurrentGrid->TranslateByLinks(p0,pEE);  
    GWfluxE = GroundWater ( p0, pE, GW_head, tot_head, rate );
    if ( GWfluxE > 0 ) fGFluxOut += GWfluxE;
    else fGFluxIn -= GWfluxE;
	  
    Pix pN = fCurrentGrid->TranslateByLinks(p0,pNN);  
    if ( pN == 0 ) GWfluxN = GroundWater ( pN, p0, GW_head, tot_head, rate );
    else GWfluxN = fGWfluxNS.Value (pN);
    if ( GWfluxN < 0 ) fGFluxOut -= GWfluxN;
    else fGFluxIn += GWfluxN;
	  
    Pix pW = fCurrentGrid->TranslateByLinks(p0,pWW);
    if ( pW == 0 ) GWfluxW = GroundWater ( pW, p0, GW_head, tot_head, rate );
    else GWfluxW = fGWfluxWE.Value(pW);
    if ( GWfluxW < 0 ) fGFluxOut -= GWfluxW;
    else fGFluxIn += GWfluxW;
	  	  
    flux = fGFluxIn - fGFluxOut;
	  
    float cs = fCellSize->GenericValue(p0);
    if ( (ff = GW_head.Value(p0)*cs + flux) < 0 && fGFluxOut > 0 ) { 
      ff = (fGFluxOut + ff)/fGFluxOut; 
      if ( GWfluxS > 0 ) GWfluxS *= ff;
      if ( GWfluxE > 0 ) GWfluxE *= ff;
      if ( GWfluxN < 0 ) GWfluxN *= ff;
      if ( GWfluxW < 0 ) GWfluxW *= ff;
      flux = - GW_head.Value(p0)*cs;
      fGFluxIn *= ff;
      fGFluxOut *= ff;
    }
    if( uf == kVolume ) {	
      fGWfluxNS.Update(p0,GWfluxS);
      fGWfluxWE.Update(p0,GWfluxE);
    } else {
      fGWfluxNS.Update(p0,GWfluxS/cs);
      fGWfluxWE.Update(p0,GWfluxE/cs);
      fGFluxIn /= cs;
      fGFluxOut /= cs;
    }
  }

  if( fCurrentGrid->GetPointInfo ( p0, DGrid::kDebug )  && gDebug ) {
    sprintf( gMsgStr, 
	     "(%x):GW Fluxes: Sout= %.3f, Eout= %.3f, Nin=%.3f, Win= %.3f, Total= %.3f, SFOut= %.3f\n",
	     p0,GWfluxS,GWfluxE,GWfluxN,GWfluxW,flux,fGFluxOut ); 
    gPrintLog(); 
  }
		
  return(flux);
}

float Flux::GWInP( Pix p0, TSpatialVariable& GW_head, TSpatialVariable& tot_head, TSpatialVariable& rate ) {
  if(fCurrentGPoint != p0) {
    GWP( p0, GW_head, tot_head, rate );
    fCurrentGPoint == p0;
  }
  return fGFluxIn;
}

float Flux::GWOutP( Pix p0, TSpatialVariable& GW_head, TSpatialVariable& tot_head, TSpatialVariable& rate ) {
  if(fCurrentGPoint != p0) {
    GWP( p0, GW_head, tot_head, rate );
    fCurrentGPoint == p0;
  }
  return fGFluxOut;
}

float Flux::GWInHeadP( Pix p0, TSpatialVariable& GW_head, TSpatialVariable& tot_head, TSpatialVariable& rate ) {
  if(fCurrentGPoint != p0) {
    GWP( p0, GW_head, tot_head, rate, kHead );
    fCurrentGPoint == p0;
  }
  return fGFluxIn;
}

float Flux::GWOutHeadP( Pix p0, TSpatialVariable& GW_head, TSpatialVariable& tot_head, TSpatialVariable& rate ) {
  if(fCurrentGPoint != p0) {
    GWP( p0, GW_head, tot_head, rate, kHead );
    fCurrentGPoint == p0;
  }
  return fGFluxOut;
}


/*********************************************************************************************/

float Flux::GroundWater( Pix p0, Pix p1, TSpatialVariable& GW_head, TSpatialVariable& tot_head, TSpatialVariable& rate )
{	
  float fr, dh, H, F = 0.;
	
  if(fCellSize==NULL) gFatal("Error, CELLSIZE undefined in Flux Module ( use cs() command )" ); 
  if ( AllowFlux(kGW,p0,p1) ) {   

    dh = tot_head.Value(p0) - tot_head.Value(p1);
    /* dh is "from p0 -- to p1" */
    fr = fAlpha[3] * dh;
			
    if (dh > 0) 
      {
	float cs = fCellSize->GenericValue(p0);
	F = fr *  rate.Value(p0) * Util::ramp(GW_head.Value(p0));
	H = F/ cs;
	if ( (tot_head.Value(p0) - H) < (tot_head.Value(p1) + H) )
	  F = dh/2 * cs;				
      }
    else
      {	
	float cs = fCellSize->GenericValue(p1);
	F = fr * rate.Value(p1) * Util::ramp(GW_head.Value(p1));
	H = F/ cs;
	if ( (tot_head.Value(p0) - H) > (tot_head.Value(p1) + H) )
	  F = dh/2 * cs;				
      }
  }
  return F;
}

/*********************************************************************************************/

float Flux::SWTransportInP (Pix p0, TSpatialVariable& SWater_height, TSpatialVariable& Constituents)
{	
  float flux=0.0; 
  Setup();
  Pix pN = fCurrentGrid->TranslateByLinks(p0,pNN);  
  Pix pS = fCurrentGrid->TranslateByLinks(p0,pSS);  
  Pix pE = fCurrentGrid->TranslateByLinks(p0,pEE);  
  Pix pW = fCurrentGrid->TranslateByLinks(p0,pWW);  

  if(SWater_height.Value(p0)<=0.01) return(0.0);
  if(pS>0) if(fSWfluxNS.Value(p0) < 0) flux -= Constituents.Value(pS) * fSWfluxNS.Value(p0);
  if(pE>0) if(fSWfluxWE.Value(p0) < 0) flux -= Constituents.Value(pE) * fSWfluxWE.Value(p0);
  if(pN>0) if(fSWfluxNS.Value(pN) > 0) flux += Constituents.Value(pN) * fSWfluxNS.Value(pN);
  if(pW>0) if(fSWfluxWE.Value(pW) > 0) flux += Constituents.Value(pW) * fSWfluxWE.Value(pW);
  return(flux);
}
	  
/*********************************************************************************************/

float Flux::SWTransportOutP (Pix p0, TSpatialVariable& SWater_height, TSpatialVariable& Constituents)
{	
  float flux=0.0; 
  Setup();
  Pix pN = fCurrentGrid->TranslateByLinks(p0,pNN);  
  Pix pS = fCurrentGrid->TranslateByLinks(p0,pSS);  
  Pix pE = fCurrentGrid->TranslateByLinks(p0,pEE);  
  Pix pW = fCurrentGrid->TranslateByLinks(p0,pWW);  

  if(SWater_height.Value(p0)<=0.01) return(0.0);
  if(fSWfluxNS.Value(p0) > 0) flux += Constituents.Value(p0) * fSWfluxNS.Value(p0);
  if(fSWfluxWE.Value(p0) > 0) flux += Constituents.Value(p0) * fSWfluxWE.Value(p0);
  if(pN>0) if(fSWfluxNS.Value(pN) < 0) flux -= Constituents.Value(p0) * fSWfluxNS.Value(pN);
  if(pW>0) if(fSWfluxWE.Value(pW) < 0) flux -= Constituents.Value(p0) * fSWfluxWE.Value(pW);
  return(flux);
}	  
/***************************************************************************************/

float Flux::GWTransportInP (Pix p0, TSpatialVariable& GWater_head, TSpatialVariable& Constituents)
{	
  float flux=0.0;
  Setup();
  Pix pN = fCurrentGrid->TranslateByLinks(p0,pNN);  
  Pix pS = fCurrentGrid->TranslateByLinks(p0,pSS);  
  Pix pE = fCurrentGrid->TranslateByLinks(p0,pEE);  
  Pix pW = fCurrentGrid->TranslateByLinks(p0,pWW);  
 
  if(GWater_head.Value(p0)<=0.01) return(0.0);
  if(pS>0) if(fGWfluxNS.Value(p0) < 0) flux -= Constituents.Value(pS) * fGWfluxNS.Value(p0);
  if(pE>0) if(fGWfluxWE.Value(p0) < 0) flux -= Constituents.Value(pE) * fGWfluxWE.Value(p0);
  if(pN>0) if(fGWfluxNS.Value(pN) > 0) flux += Constituents.Value(pN) * fGWfluxNS.Value(pN);
  if(pW>0) if(fGWfluxWE.Value(pW) > 0) flux += Constituents.Value(pW) * fGWfluxWE.Value(pW);
  return(flux);
}
	  
/*********************************************************************************************/

float Flux::GWTransportOutP (Pix p0, TSpatialVariable& GWater_head, TSpatialVariable& Constituents)
{	
  float flux=0.0;
  Setup();
  Pix pN = fCurrentGrid->TranslateByLinks(p0,pNN);  
  Pix pS = fCurrentGrid->TranslateByLinks(p0,pSS);  
  Pix pE = fCurrentGrid->TranslateByLinks(p0,pEE);  
  Pix pW = fCurrentGrid->TranslateByLinks(p0,pWW);  
 
  if(GWater_head.Value(p0)<=0.01) return(0.0);
  if(fGWfluxNS.Value(p0) > 0) flux += Constituents.Value(p0) * fGWfluxNS.Value(p0);
  if(fGWfluxWE.Value(p0) > 0) flux += Constituents.Value(p0) * fGWfluxWE.Value(p0);
  if(pN>0) if(fGWfluxNS.Value(pN) < 0) flux -= Constituents.Value(p0) * fGWfluxNS.Value(pN);
  if(pW>0) if(fGWfluxWE.Value(pW) < 0) flux -= Constituents.Value(p0) * fGWfluxWE.Value(pW);
  return(flux);
}	  
/***************************************************************************************/



