#include "FrameLink.h"
#include "DistributedGrid.h"
#include "CVariable.h"
#include "MML_Frame.h"
#include "MML_Module.h"
#include "SSModel.h"
#include "TMap.h"
#ifdef USE_MPI
#include "mpi.h"
#endif

/*********************************************************************************************/

int FLmapProcessIndex = 0;

/*********************************************************************************************/
			
void FrameLink::Setup() {
	if(FSetup) return;
	for( int i=0; i<kMaxMaps; i++ ) {
		if( ! fMapSource[i].empty() ) {
			fSetupMap[i] = new TMap(kClassMap);
			fSetupMap[i]->VariableName("Input");
			fSetupMap[i]->DSetName(fMapName[i]);
			const char* format = (( fFormat[i] == "" ) || ( fFormat[i] == "default" ))  ? ((const char*)NULL) : fFormat[i].chars();
			fSetupMap[i]->Read(fMapSource[i],fMapName[i],1,NULL,format);
			if( (fLinkMap == NULL) && (gDebug > 0) ) { 
				fLinkMap = new TMap(fSetupMap[i],1); 
				fLinkMap->DSetName("LinkMap"); 
				fLinkMap->Set((byte)0); 
			}
// #ifdef USE_MPI
//			if( fNewLinkMap == NULL) { 
//				fNewLinkMap = new TMap(fSetupMap[i],1); 
//				fNewLinkMap->DSetName("NewLinkMap"); 
//				fNewLinkMap->Set((byte)0); 
//			}
// #endif
		} else fSetupMap[i] = NULL;
	}
  FSetup = 1;
}

/*********************************************************************************************/

void FrameLink::DynamicRelink() {
	if( fDynamicLinkStep < 0.0 ) return;
	static float relink_time = fDynamicLinkStep;
	static CVariable* var = NULL; 
	if( gTime() >= relink_time ) {
		relink_time += fDynamicLinkStep;
		if( var == NULL ) {
			var = (CVariable*) fCurrentModule->GetVariable(fDynamicLinkVar,False);
			if( var == NULL ) { gPrintErr( fDynamicLinkVar + ": Can't Find Dynamic Link Variable for FrameLink."); return; }
		}
		DynamicRelink(var); 
	}
}
/*********************************************************************************************/

int FrameLink::ConfigObject(TConfigData& cd) {
  const CString& cmd = cd.Cmd();
	sprintf(gMsgStr,"Reading FrameLink Config: Nargs = %d, CMD = %s", cd.NArgs(), cmd.chars() );	
	gPrintLog();  
  int iarg; float farg; CString* s;
	if( cmd == "i" ) { 
		int index = 0;
		do {
			if( cd.IntArg(	index, fIndex[index] ) == 0 ) break;
		} while( index++ < kMaxIndices );
		fNIndices = index;
	}
	else if( cmd[0] == 'p' ) {
		cd.IntArg(	0, FLmapProcessIndex );
	} else if( cmd == "dl" ) {
		if( (s = cd.Arg(0)) != NULL  ) { fDynamicLinkVar = *s; }
		else { gPrintErr("Must specify dynamic link Variable in FrameLink dl() command."); }
		fDynamicLinkStep = gDT();
		cd.FloatArg(	1, fDynamicLinkStep );
	}	else if( cmd == "lm" ) {
		cd.IntArg(	0, fLinkMode );
	} else if( cmd[0] == 'm' ) {
		int mIndex = (cmd[1]-'0');
		if( (mIndex >= 0) && (mIndex < kMaxMaps) ) {
	    if( (s = cd.Arg(0)) != NULL  ) {
				fMapSource[mIndex] = *s;
				if( (s = cd.Arg(1)) != NULL ) {
				  fMapName[mIndex] = *s;
				  if( (s = cd.Arg(2)) != NULL ) {
					  fFormat[mIndex] = *s;
				  } else return 0;
				} else return 0;
			} else return 0;
		} else return 0;
	} else return 0;
  return 1;
}


/*********************************************************************************************/

int FrameLink::Config( TConfigData& cd, Module* m ) {
	int fl_index;
	FrameLink* fl = GetByModule(m,fl_index);
	if( fl == NULL ) {
		CString FLName("C");
		if(fl_index >= 0) FLName.appendIndex(fl_index);
		fl = New(FLName);
		fl->SetModule(m);
	}
	if(fl) { 
		fl->ConfigObject(cd);
		if(m) { 
			MFrame& f = TModel::I().Frame();
			LayerConfig* lc = f.getLayerConfig( m  );
			if( lc == NULL ) {
				sprintf( gMsgStr, "FrameLink: Null LayerConfig for Module %s",m->Name());
				gFatal();
			}
			lc->SetFrameLink(fl); 
			fl->SetLinkLayer(lc->LinkLayer());
			sprintf(gMsgStr,"\nSetting FrameLink for %s(%s): %x", m->Name(), f.Name(), fl );	
			gPrintLog();  
		}
		return 1; 
	}
}

void FrameLink::Run( const char* methodName, TMap2* linkMap, int activationLevel, int gridLayer, int linkLayer ) {
 CString name(methodName);
 if( fCurrentGrid == NULL ) { fCurrentGrid = new DistributedGrid( TModel::I().Grid(), linkMap, activationLevel, gridLayer, linkLayer); }
 sprintf(gMsgStr,"Linking cells for activation layer %d, link layer %d, grid layer %d,  using Framelink %s ",
									activationLevel,linkLayer, gridLayer, methodName ); gPrintScreen(); 
 Setup();
 if( name == "default" ) RiverLink();
 if( name == "RiverLink" ) RiverLink();
 if( name == "RiverLink1" ) RiverLink1();
}

/*
int FrameLink::RiverLink0( )
{
	if( fSetupMap[0] == NULL ) { 
		gPrintErr( SName() + ": Error, No River map specified in FluxLink!" ); return 0; 
	}
	if( fCurrentGrid == NULL ) { 
		gPrintErr( SName() + ": Error, No grid specified in FluxLink!" ); return 0; 
	}
	if( ((Region2&)(*(fSetupMap[0]))) !=  ((Region2&)(*(fCurrentGrid->RegionMap())))) { 
		gPrintErr( SName() + ": Error, Wrong region for River map specified in FluxLink!" ); return 0; 
	}
	if( fSetupMap[1] ) {
		if( ((Region2&)(*(fSetupMap[1]))) !=  ((Region2&)(*(fCurrentGrid->RegionMap())))) { 
			gPrintErr( SName() + ": Error, Wrong region for Elevation map specified in FluxLink!" ); return 0; 
		}
	}
	
	PixArray pa[2];
	int rp_index = 0;

	TLayer* l = TModel::I().Grid()->getCellLayer();	
	fLinkedPoints.ReAlloc(*fSetupMap[0]); fLinkedPoints.Set( (byte)0 );
	TCell* c;
	for( c = fCurrentGrid->first(); c; fCurrentGrid->next(c) ) if( c->GetObjInfo(TCell::kGhostIndex) == 0 ) {
		 int ir = l->getCellLocCoord(TLayer::kRows,c);
		 int ic = l->getCellLocCoord(TLayer::kCols,c);
		 byte rVal0 = fSetupMap[0]->BValue( ir, ic, 0 );
		 for( int i=0; i<fNIndices; i++) {
			 if( rVal0 == fIndex[i] ) {
				 fLinkedPoints.SetValue( ir, ic, rVal0 );
				 if(i==1) { fCurrentGrid->RootPointVec().elem(rp_index++) = c; }
			 }
		 }
	}
	for( int irp = 0; irp < rp_index; irp++ ) {
		LinkNeighbors( (TCell*) fCurrentGrid->RootPointVec().elem(irp), l, pa[0], fLinkMode );
	}
	
	int cnt=0, na=0, ir, ic;
	while(1) {
		PixArray& pa0 = pa[ cnt % 2 ];
		PixArray& pa1 = pa[ ++cnt % 2 ];
//		if(fNewLinkMap) fNewLinkMap->Set((byte)0);
		pa1.clear();
		int nas = (na = pa0.nElem());
#ifdef USE_MPI
		MPI_Allreduce(&na,&nas,1,MPI_INT,MPI_SUM,MPI_COMM_WORLD);
#endif
		if( nas == 0 ) break;
		for( int i=0; i<na; i++ ) {
			TCell* c = (TCell*) pa0[i];
			LinkNeighbors(c, l, pa1,fLinkMode);
		}
		AddBoundaryPoints(pa1,l);
	}
	if( fSetupMap[1] ) {
		for( c = fCurrentGrid->first(); c; fCurrentGrid->next(c) ) if( c->GetObjInfo(TCell::kGhostIndex) == 0 ) {
			LinkNeighborsDownslope(c,l);
		}
	}
	 
	if( fLinkMap ) {
		fLinkMap->SetObjInfo(TMap::kMapType,kMap2);
		fLinkMap->Write(0);
		if(FLmapProcessIndex==1) {
#ifdef USE_MPI
			gPrintErr( "This Map process (Modifify Elevation Map) must be done in serial mode.");
#else
			CreateModifiedElevationMap();
#endif
		}
	}
}
*/

int FrameLink::RiverLink( ) {

	TLayer* l = TModel::I().Grid()->getCellLayer();	
	if( fSetupMap[2] ) {
	
		sprintf(gMsgStr,"Linking Tree Network Frame using map %s ",fMapName[2].chars()); gPrintScreen();
		LinkNeighborsUsingMap(fSetupMap[2]);
		
	} else {

		if( fSetupMap[0] == NULL ) { 
			gPrintErr( SName() + ": Error, No River map specified in FluxLink!" ); return 0; 
		}
		if( fCurrentGrid == NULL ) { 
			gPrintErr( SName() + ": Error, No grid specified in FluxLink!" ); return 0; 
		}
		if( ((Region2&)(*(fSetupMap[0]))) !=  ((Region2&)(*(fCurrentGrid->RegionMap())))) { 
			gPrintErr( SName() + ": Error, Wrong region for River map specified in FluxLink!" ); return 0; 
		}
		if( fSetupMap[1] ) {
			if( ((Region2&)(*(fSetupMap[1]))) !=  ((Region2&)(*(fCurrentGrid->RegionMap())))) { 
				gPrintErr( SName() + ": Error, Wrong region for Elevation map specified in FluxLink!" ); return 0; 
			}
		}

		PixArray pa[2];
		int rp_index = 0;
		gPrintScreen("Linking Tree Network Frame using RiverLink ");
		fLinkedPoints.ReAlloc(*fSetupMap[0]); fLinkedPoints.Set( (byte)0 );
		TCell* c;
		for( c = fCurrentGrid->first(); c; fCurrentGrid->next(c) )  {
			 int ir = l->getCellLocCoord(TLayer::kRows,c);
			 int ic = l->getCellLocCoord(TLayer::kCols,c);
			 byte rVal0 = fSetupMap[0]->BValue( ir, ic, 0 );
			 for( int i=0; i<fNIndices; i++) {
				 if( rVal0 == fIndex[i] ) {
					 fLinkedPoints.SetValue( ir, ic, rVal0 );
					 if( (i==1) && (c->GetObjInfo(TCell::kGhostIndex) == 0) ) { 
						 fCurrentGrid->RootPointVec().elem(rp_index++) = c; 
					 }
				 }
			 }
		}
		for( int irp = 0; irp < rp_index; irp++ ) {
			LinkNeighbors( (TCell*) fCurrentGrid->RootPointVec().elem(irp), l, pa[0], fLinkMode );
		}

		int cnt=0, na=0, ir, ic;
		while(1) {
			PixArray& pa0 = pa[ cnt % 2 ];
			PixArray& pa1 = pa[ ++cnt % 2 ];
	//		if(fNewLinkMap) fNewLinkMap->Set((byte)0);
			pa1.clear();
			int nas = (na = pa0.nElem());
#ifdef USE_MPI
			fCellArray.clear();
			MPI_Allreduce(&na,&nas,1,MPI_INT,MPI_SUM,MPI_COMM_WORLD);
#endif
			if( nas == 0 ) break;
			for( int i=0; i<na; i++ ) {
				TCell* c = (TCell*) pa0[i];
				LinkNeighbors(c, l, pa1,fLinkMode);
			}
			AddBoundaryPoints(l,&pa1);
		}
		
#ifdef USE_MPI	
		fCellArray.clear();
#endif

		if( fSetupMap[1] ) {
			for( c = fCurrentGrid->first(); c; fCurrentGrid->next(c) ) {
				LinkNeighborsDownslope(c,l);
			}
		}
	}
	AddBoundaryPoints(l);
	 
	if( fLinkMap ) {
		fLinkMap->SetObjInfo(TMap::kMapType,kMap2);
		fLinkMap->Write(0);
		if(FLmapProcessIndex==1) {
#ifdef USE_MPI
			gPrintErr( "This Map process (Modifify Elevation Map) must be done in serial mode.");
#else
			CreateModifiedElevationMap();
#endif
		}
	}
	return 0;
}

int FrameLink::LinkNeighborsUsingMap( TMap* m ) {
	TLayer* l = TModel::I().Grid()->getCellLayer();	
	int rr, rc;
	unsigned int C0, C1;
	for( TCell* c = fCurrentGrid->first(); c; fCurrentGrid->next(c) )  {
		 int ir = l->getCellLocCoord(TLayer::kRows,c);
		 int ic = l->getCellLocCoord(TLayer::kCols,c);
		 byte dir =  m->BValue( ir, ic, 0 );
		 Grid_Direction il = (Grid_Direction) dir;
		 if( !GMR(rr,rc,il) ) { sprintf(gMsgStr,"Illegal Index in LinkNeighbors: (%d,%d): %d(%x)",ir,ic,il,dir);  gPrintErr(); }
		 else {
			 C0 = ir+rr; C1 = ic+rc;
			 TCell* cR = l->getCell(C0,C1);
			 if( cR ) { 
					MakeLink(c,cR,l);
					if( fLinkMap ) {
						fLinkMap->SetValue(ir,ic,(unsigned long)il);
					}
					RegisterLink( ir, ic, C0, C1, l );
					if(gDebug > 1 ) {
						sprintf(gMsgStr,"add(%d,%d), ",ir,ic);
						gPrintLog();
					}
			}
		}
	}
	return 0;
}

int FrameLink::RiverLink1( )
{
	TLayer* l = TModel::I().Grid()->getCellLayer();	
	if( fSetupMap[2] ) {
	
		LinkNeighborsUsingMap(fSetupMap[2]);

	} else {

		if( fSetupMap[0] == NULL ) { 
			gPrintErr( SName() + ": Error, No River map specified in FluxLink!" ); return 0; 
		}
		if( fCurrentGrid == NULL ) { 
			gPrintErr( SName() + ": Error, No grid specified in FluxLink!" ); return 0; 
		}
		if( ((Region2&)(*(fSetupMap[0]))) !=  ((Region2&)(*(fCurrentGrid->RegionMap())))) { 
			gPrintErr( SName() + ": Error, Wrong region for River map specified in FluxLink!" ); return 0; 
		}
		if( fSetupMap[1] ) {
			if( ((Region2&)(*(fSetupMap[1]))) !=  ((Region2&)(*(fCurrentGrid->RegionMap())))) { 
				gPrintErr( SName() + ": Error, Wrong region for Elevation map specified in FluxLink!" ); return 0; 
			}
		}

		PixArray pa[3];
		int rp_index = 0;
		gPrintScreen("Linking Tree Network Frame using RiverLink1 ");
		fLinkedPoints.ReAlloc(*fSetupMap[0]); fLinkedPoints.Set( (byte)0 );
		TCell* c;
		for( c = fCurrentGrid->first(); c; fCurrentGrid->next(c) )  {
			 int ir = l->getCellLocCoord(TLayer::kRows,c);
			 int ic = l->getCellLocCoord(TLayer::kCols,c);
			 byte rVal0 = fSetupMap[0]->BValue( ir, ic, 0 );
			 for( int i=0; i<fNIndices; i++) {
				 if( rVal0 == fIndex[i] ) {
					 fLinkedPoints.SetValue( ir, ic, rVal0 );
					 if( (i==1) && (c->GetObjInfo(TCell::kGhostIndex) == 0) ) { 
						 fCurrentGrid->RootPointVec().elem(rp_index++) = c; 
					 }
				 }
			 }
		}
		for( int irp = 0; irp < rp_index; irp++ ) {
			LinkNeighbors( (TCell*) fCurrentGrid->RootPointVec().elem(irp), l, pa[0], fLinkMode );
		}

		int cnt=0, na=0, ir, ic;
		pa[3] = pa[0];
		while(1) {
			PixArray& pa0 = pa[ cnt % 2 ];
			PixArray& pa1 = pa[ ++cnt % 2 ];
	//		if(fNewLinkMap) fNewLinkMap->Set((byte)0);
			pa1.clear();
			int nas = (na = pa0.nElem());
#ifdef USE_MPI
			fCellArray.clear();
			MPI_Allreduce(&na,&nas,1,MPI_INT,MPI_SUM,MPI_COMM_WORLD);
#endif
			if( nas == 0 ) break;
			for( int i=0; i<na; i++ ) {
				TCell* c = (TCell*) pa0[i];
				LinkNeighbors(c, l, pa1,fLinkMode);
			}
			AddBoundaryPoints(l,&pa1);
			pa[3] += pa1;
		}

#ifdef USE_MPI	
		fCellArray.clear();
#endif

		if( fSetupMap[1] ) {
			int cnt=0, na=0, ir, ic;
			pa[0] = pa[3];
			while(1) {
				PixArray& pa0 = pa[ cnt % 2 ];
				PixArray& pa1 = pa[ ++cnt % 2 ];
	//		if(fNewLinkMap) fNewLinkMap->Set((byte)0);
				pa1.clear();
				int nas = (na = pa0.nElem());
#ifdef USE_MPI
				fCellArray.clear();
				MPI_Allreduce(&na,&nas,1,MPI_INT,MPI_SUM,MPI_COMM_WORLD);
#endif
				if( nas == 0 ) break;
				for( int i=0; i<na; i++ ) {
					TCell* c = (TCell*) pa0[i];
					LinkNeighborsBestPath(c, l, pa1,fLinkMode);
				}
				AddBoundaryPoints(l,&pa1);
			}
		}
	}

	AddBoundaryPoints(l);
	 
	if( fLinkMap ) {
		fLinkMap->SetObjInfo(TMap::kMapType,kMap2);
		fLinkMap->Write(0);
	}
	return 0;
}

int FrameLink::DynamicRelink(CVariable* v) {
	sprintf(gMsgStr," Executing Dynamic ReLink ");
	gPrintLog(); gPrintScreen();
	int layer_index = -1;
	TLayer* l = TModel::I().Grid()->getCellLayer();	
	if( fCurrentGrid == NULL )  { fCurrentGrid = v->Cov().Grid( layer_index, fLinkLayer ); }
	for( TCell* c = fCurrentGrid->first(); fCurrentGrid->onGrid(c); fCurrentGrid->next(c) ) {
		LinkNeighborsDownslope(c,l,v);
	}
	return 0;
}

int FrameLink::CreateModifiedElevationMap(int mapIndex) {
	TMap* oldElevMap = fSetupMap[mapIndex];
	if( oldElevMap == NULL ) return 0;
	TMap* newMap = new TMap( *oldElevMap ); 
	newMap->DSetName("newElevMap"); 
	TLayer* l = TModel::I().Grid()->getCellLayer();
	TCell* c;	
	PixArray pa[2]; int na, ir, ic, cnt = 0; 
	if( fSetupMap[mapIndex] ) {
		for( c = fCurrentGrid->first(); fCurrentGrid->onGrid(c); fCurrentGrid->next(c) ) {
			AdjustMapDownslope(c,l,pa[0],mapIndex,newMap,True);
		}
		while(1) { 
			PixArray& pa0 = pa[ cnt % 2 ];
			PixArray& pa1 = pa[ ++cnt % 2 ];
			pa1.clear();
			if( (na = pa0.nElem()) == 0 ) break;
			for( int i=0; i<na; i++ ) {
				c  = (TCell*) pa0[i];
				AdjustMapDownslope(c,l,pa1,mapIndex,newMap,False);
			}
		}	
	}	else { gPrintErr("No Elevation Map Specified."); return 0; }
	newMap->SetObjInfo(TMap::kMapType,kMap2);
	newMap->Write(0);
	return 1;
}

//			Point2& pR = getCellLocPoint( TCell*  c )
// 			if( cR ) { c->addNeighbor( link_layer, cR, index ); }

int FrameLink::MakeLink( TCell* c0, TCell* c1, TLayer* l ) {  // create link from c0 to c1
	c0->addNeighbor( fLinkLayer, c1, 0 );                // create forward link using link # 0.
	int index = c1->addNeighborFAS( fLinkLayer, c0, 1 );     // create backward link using link # index ( first avail > 0 ).
	if(gDebug > 1 ) {
		int ir0 = l->getCellLocCoord(TLayer::kRows,c0);
		int ic0 = l->getCellLocCoord(TLayer::kCols,c0);
		int ir1 = l->getCellLocCoord(TLayer::kRows,c1);
		int ic1 = l->getCellLocCoord(TLayer::kCols,c1);
		sprintf(gMsgStr,"\nMakeLink(%d): (%d,%d):%p -> (%d,%d):%p  ",index,ir0,ic0,c0,ir1,ic1,c1);
		gPrintLog();
		LinkLayer* ll0 = c0->neighborLinkLayer( fLinkLayer );
		LinkLayer* ll1 = c1->neighborLinkLayer( fLinkLayer );
		sprintf(gMsgStr,"\nLinks: ->(%p %p) <-(%p %p) ", ll0->Link(0), ll0->Link(1), ll1->Link(0), ll1->Link(1) );
		gPrintLog();
	}
	return index;
}

int FrameLink::LinkNeighbors( TCell* c, TLayer* l, PixArray& pa, int allowDiag ) {
	 int rr, ir = l->getCellLocCoord(TLayer::kRows,c);
	 int rc, ic = l->getCellLocCoord(TLayer::kCols,c);
	 if( fLinkedPoints.IntValue(ir,ic) == 0 ) { return 0; }
	 if( c->GetObjInfo( TCell::kGhostIndex ) > 1 ) return 0;
	 unsigned int cnt = 0, C0, C1;
	 unsigned long dir, lmv = 1;
	 TCell* clink = c->getNeighbor( fLinkLayer, 0 );
	 for( Grid_Direction il = firstGD(); moreGD(il); (allowDiag) ? incrGD(il) : incrGD_NEWS(il) ) {
			if( !GMR(rr,rc,il) ) { gFatal("Illegal Index in LinkNeighbors");  }
			C0 = ir+rr; C1 = ic+rc;
			TCell* cR = l->getCell(C0,C1);
			if( cR && ( clink != cR ) && (fLinkedPoints.IntValue(C0,C1) > 0) ) {
				int root_point_index = fCurrentGrid->RootPointVec().index(cR);
				TCell* clink0 = cR->getNeighbor( fLinkLayer, 0 );
				if(  ( clink0 == 0 )  && (root_point_index < 0) ) {
					MakeLink(cR,c,l);
					if( fLinkMap ) {
						dir = (il<6) ? il+4 : il-4;
						fLinkMap->SetValue(C0,C1,dir);
					}
					RegisterLink( C0, C1, ir, ic, l );
					if(gDebug > 1 ) {
						if( cR->GetObjInfo(TCell::kGhostIndex) > 0 ) { sprintf(gMsgStr,"addGhost(%d,%d), ",C0,C1); }
						else { sprintf(gMsgStr,"add(%d,%d), ",C0,C1); }
						gPrintLog();
					}
					pa.add(cR); 
					cnt++;	
				}
			}			
	}
	return cnt; 
}

// River cell if:  (fLinkedPoints.IntValue(C0,C1) > 0)

int FrameLink::LinkNeighborsBestPath( TCell* c, TLayer* l, PixArray& pa, int allowDiag ) {
	 int rr, ir = l->getCellLocCoord(TLayer::kRows,c);
	 int rc, ic = l->getCellLocCoord(TLayer::kCols,c);
	 if( c->GetObjInfo( TCell::kGhostIndex ) > 1 ) return 0;
	 unsigned int cnt = 0, C0, C1;
	 unsigned long dir, lmv = 1;
	 for( Grid_Direction il = firstGD(); moreGD(il); (allowDiag) ? incrGD(il) : incrGD_NEWS(il) ) {
			if( !GMR(rr,rc,il) ) { gFatal("Illegal Index in LinkNeighbors");  }
			C0 = ir+rr; C1 = ic+rc;
			TCell* cR = l->getCell(C0,C1);
			if( cR  ) {
				int root_point_index = fCurrentGrid->RootPointVec().index(cR);
				TCell* clink0 = cR->getNeighbor( fLinkLayer, 0 );
				if(  ( clink0 == 0 )  && (root_point_index < 0) ) {
					MakeLink(cR,c,l);
					if( fLinkMap ) {
						dir = (il<6) ? il+4 : il-4;
						fLinkMap->SetValue(C0,C1,dir);
					}
					RegisterLink( C0, C1, ir, ic, l );
					if(gDebug > 1 ) {
						if( cR->GetObjInfo(TCell::kGhostIndex) > 0 ) { sprintf(gMsgStr,"addGhost(%d,%d), ",C0,C1); }
						else { sprintf(gMsgStr,"add(%d,%d), ",C0,C1); }
						gPrintLog();
					}
					pa.add(cR); 
					cnt++;	
				}
			}			
	}
	return cnt; 
}

int FrameLink::LinkNeighborsBestPath1( TCell* c, int mapIndex, TLayer* l, PixArray& pa, int allowDiag ) {
	 int rr, ir = l->getCellLocCoord(TLayer::kRows,c);
	 int rc, ic = l->getCellLocCoord(TLayer::kCols,c);
	 if( c->GetObjInfo( TCell::kGhostIndex ) > 1 ) return 0;
	 unsigned int cnt = 0, C0, C1;
	 unsigned long dir, lmv = 1;
	 for( Grid_Direction il = firstGD(); moreGD(il); (allowDiag) ? incrGD(il) : incrGD_NEWS(il) ) {
			if( !GMR(rr,rc,il) ) { gFatal("Illegal Index in LinkNeighbors");  }
			C0 = ir+rr; C1 = ic+rc;
			TCell* cR = l->getCell(C0,C1);
			if( cR  ) {
				int root_point_index = fCurrentGrid->RootPointVec().index(cR);
				float vr = fSetupMap[mapIndex]->FltValue(C0,C1);
				TCell* clink0 = cR->getNeighbor( fLinkLayer, 0 );
				if(  ( clink0 == 0 )  && (root_point_index < 0) ) {
					MakeLink(cR,c,l);
					if( fLinkMap ) {
						dir = (il<6) ? il+4 : il-4;
						fLinkMap->SetValue(C0,C1,dir);
					}
					RegisterLink( C0, C1, ir, ic, l );
					if(gDebug > 1 ) {
						if( cR->GetObjInfo(TCell::kGhostIndex) > 0 ) { sprintf(gMsgStr,"addGhost(%d,%d), ",C0,C1); }
						else { sprintf(gMsgStr,"add(%d,%d), ",C0,C1); }
						gPrintLog();
					}
					pa.add(cR); 
					cnt++;	
				}
			}			
	}
	return cnt; 
}

int FrameLink::RegisterLink( int r0, int c0, int r1, int c1, TLayer* l ) {
	int rv = 0;
#ifdef USE_MPI
	 if( gNProcs() <= 1 ) return rv;
	 int ig, idata[NCellRecordInts];
		TPartition2* p = l->partition();
		if( p ) {
			for( int ip=0; ip<gNProcs(); ip++ ) if( ip != gIProc ) {
				if( (ig = p->ghostIndex( ip, r0, c0 ) ) > 0 ) {
					idata[0] = r0; idata[1] = c0; idata[2] = r1;  idata[3] = c1; 
					fCellArray.addCellRecord( ip, idata ); rv++;
				}
			}
			if ( ( ig = p->ghostIndex( gIProc, r0, c0 ) ) > 0 ) {
				idata[0] = r0; idata[1] = c0; idata[2] = r1;  idata[3] = c1; 
				int ip1 = p->getowner( r0, c0 ); 
				fCellArray.addCellRecord( ip1, idata ); rv++;
			}
		} else  { gFatal( "Undefined partition in FrameLink::LinkNeighbors" ); }
#endif
	return rv;
}

int FrameLink::AddBoundaryPoints( TLayer* l, PixArray* pa ) { 
	if( gNProcs() <= 1 ) return 0;
	unsigned int r0, c0, r1, c1, cnt=0;
#ifdef USE_MPI
	fCellArray.generateInArray();
	CellRecordArray& cra = fCellArray.CellRecordInArray();
	for( int i=0; i<cra.nCellRecords(); i++ ) {
		CellRecord& cr = cra[i];
		r0 = cr.idata(0); c0 = cr.idata(1); r1 = cr.idata(2); c1 = cr.idata(3); 
		TCell* C0 = l->getCell(r0,c0);
		TCell* C1 = l->getCell(r1,c1);
		if(gDebug > 1 ) {
			sprintf(gMsgStr,"\nAddBoundary Link (%d,%d) -> (%d,%d) (%p,%p) ",r0,c0,r1,c1,C0,C1);
			gPrintLog();
		}
		if( C0 && C1 ) { 
			MakeLink(C0,C1,l);
			if( pa ) {
				if( (C0->GetObjInfo(TCell::kGhostIndex)) == 0 ) { pa->add(C0); cnt++; gPrintLog(" ** Add BP0 "); }
				if( (C1->GetObjInfo(TCell::kGhostIndex)) == 0 ) { pa->add(C1); cnt++; gPrintLog(" ** Add BP1 "); }					
			}
		}
	}
#endif
	return cnt;
}

int FrameLink::AdjustMapDownslope( TCell* c, TLayer* l, PixArray& pa, int mapIndex, TMap* newMap, int firstRun ) { 
	 Point2& point0 = l->getCellLocPoint(c);
	 int ival, cnt = 0;
	 if( (ival=fLinkedPoints.IntValue( point0 )) == 0 ) { return 0; }
	 TCell* cR = c->getNeighbor( fLinkLayer, 0 );
	 if( cR == 0 ) { 
		 if( ival == fIndex[1] ) {
			 sprintf(gMsgStr,"Missing Link in AdjustMapDownslope() at (%d,%d).",point0(0),point0(1));
			 gPrintErr(); 
		 }
		 return 0; 
	 }
	 Point2& point1 = l->getCellLocPoint(cR);
	 float  v0 = ( firstRun || newMap == NULL ) ? fSetupMap[mapIndex]->FltValue(point0) : newMap->FltValue(point0);
	 float  v1 = ( firstRun || newMap == NULL ) ? fSetupMap[mapIndex]->FltValue(point1) : newMap->FltValue(point1);
	 if( v1 > v0 ) {
			if(newMap) { newMap->SetValueWithRescale( &point1, v0 ); }
			else  { fSetupMap[mapIndex]->SetValueWithRescale( &point1, v0 ); }
			pa.add(cR); cnt++; 
	 } 
	 return cnt;
}

TCell* FrameLink::LinkNeighborsDownslope( TCell* c, TLayer* l, int mapIndex, Bool checkNeighbors ) { 
	 if( c->GetObjInfo(TCell::kGhostIndex) > 0 ) return 0;
	 int rr, iR, ir = l->getCellLocCoord(TLayer::kRows,c);
	 int rc, iC, ic = l->getCellLocCoord(TLayer::kCols,c);
	 if( fLinkedPoints.IntValue(ir,ic) == fIndex[1] ) { return 0; }
	 TCell* cR = c->getNeighbor( fLinkLayer, 0 );
	 if( cR ) return cR;
	 if( c->GetObjInfo( TCell::kGhostIndex ) > 1 ) return 0;
	 unsigned int C0, C1;
	 unsigned long iL;
	 float  v0 = fSetupMap[mapIndex]->FltValue(ir,ic);
	 float rvm, vm = v0; 
	 TCell* cm = 0;
	 for( Grid_Direction il = firstGD(); moreGD(il); incrGD(il) ) {
			if( !GMR(rr,rc,il) ) { gFatal("Illegal Index in LinkNeighbors");  }
			float vr = fSetupMap[mapIndex]->FltValue(C0=ir+rr,C1=ic+rc);
			if( (vr < vm) || (cm==0) ) { 
				TCell* cR = l->getCell(C0,C1);
				TCell* cR_link = (cR) ? cR->getNeighbor( fLinkLayer, 0 ) : 0;
				TCell* cR_link1 =  (cR_link) ? cR_link->getNeighbor( fLinkLayer, 0 ) : 0;
				if(cR && ( cR_link != c ) && ( cR_link1 != c ) ) { 
					cm = cR; vm = vr;
					if(checkNeighbors) { rvm = ProcessMapNeighbors( cR, l, mapIndex, kAve ); }
					iL = il; iR = C0; iC = C1;
				}
			} else if ( (vr == vm) && checkNeighbors ) {
				TCell* cR = l->getCell(C0,C1);
				TCell* cR_link = (cR) ? cR->getNeighbor( fLinkLayer, 0 ) : 0;
				TCell* cR_link1 =  (cR_link) ? cR_link->getNeighbor( fLinkLayer, 0 ) : 0;
				if( cR && ( cR_link != c ) && ( cR_link1 != c ) ) { 
					float rvm1 = ProcessMapNeighbors( cR, l, mapIndex, kAve ); 
					if( rvm1 < rvm ) {
						cm = cR; vm = vr;
						rvm = rvm1;
						iL = il; iR = C0; iC = C1;
					}
				}
			}
	 }
	 if( cm ) {
			static int cnt = 0;
			MakeLink(c,cm,l);
			RegisterLink( ir, ic, iR, iC, l );
			if( fLinkMap ) fLinkMap->SetValue(ir,ic,iL);
			if( gDebug && ( vm > v0 ) ) {
				sprintf(gMsgStr,"Uphill link configured: (%d,%d:%f) -> (%d,%d:%f)", ir, ic, v0, iR, iC, vm );
				gPrintErr();
			}
	 }
	 return cm; 
}

TCell* FrameLink::LinkNeighborsDownslope( TCell* c, TLayer* l, CVariable* v ) { 
	 if( c->GetObjInfo(TCell::kGhostIndex) > 0 ) return 0;
	 int rr, iR, ir = l->getCellLocCoord(TLayer::kRows,c);
	 int rc, iC, ic = l->getCellLocCoord(TLayer::kCols,c);
	 Grid_Direction iL;
	 if( fLinkedPoints.IntValue(ir,ic) ==  fIndex[1] ) { return 0; }
	 float vm, vr, v0 = v->Value(c);
	 unsigned int C0, C1;
	 TCell *cm = 0, *cR = c->getNeighbor( fLinkLayer, 0 );
	 if( cR && ((vr = v->Value(cR)) <= v0) ) return 0;
	 if( (gDebug > 1) && cR ) {
			 Point2& p1 = l->getCellLocPoint(cR);
			 sprintf(gMsgStr,"ReLink:: Found link backflow: (%d,%d:%f) -> (%d,%d:%f)",ir,ic,v0,p1(0),p1(1),vr);
			 gPrintLog();
	 } 
	 vm = vr;
	 for( Grid_Direction il = firstGD(); moreGD(il); incrGD(il) ) {
			if( !GMR(rr,rc,il) ) { gFatal("Illegal Index in LinkNeighbors");  }
			cR = l->getCell(C0=ir+rr,C1=ic+rc);
			if( cR && ((vr = v->Value(cR)) < vm) ) { 
				cm = cR;
				vm = vr;
				iL = il; iR = C0; iC = C1;
			}
	 }
	 if( cm ) {
			MakeLink(c,cm,l);
			RegisterLink( ir, ic, iR, iC, l );
			if( gDebug ) {
				 Point2& p1 = l->getCellLocPoint(cm);
				 sprintf(gMsgStr,"ReLinkNeighborsDownslope:  Create new link: (%d,%d:%f) -> (%d,%d:%f)",ir,ic,v0,p1(0),p1(1),vm);
				 gPrintLog();
			} 
	 }
	 return cm; 
}


float FrameLink::ProcessMapNeighbors( TCell* c, TLayer* l, int mapIndex, EMapOperation mop ) {
	 int cnt=0, rr, ir = l->getCellLocCoord(TLayer::kRows,c);
	 int rc, ic  = l->getCellLocCoord(TLayer::kCols,c);
	 unsigned int C0, C1;
	 float  v0 = fSetupMap[mapIndex]->FltValue(ir,ic);
	 float vm = v0;
	 for( Grid_Direction il = firstGD(); moreGD(il); incrGD(il) ) {
			if( !GMR(rr,rc,il) ) { gFatal("Illegal Index in LinkNeighbors"); }
			C0 = ir+rr; C1 = ic+rc;
			if( l->getCell(C0,C1) ) {
				float vr = fSetupMap[mapIndex]->FltValue(C0,C1);
				switch(mop) {
					case kMax:
						if( vr > vm ) { vm = vr; }
						break;
					case kMin:
						if( vr < vm ) { vm = vr; }
						break;
					case kAve:
						vm += vr;
						cnt++;
						break;
					case kSum:
						vm += vr;
						break;
				}
			}
	 }
	 if(mop==kAve) { vm /= cnt; } 
	 return vm; 
}


