#include "RunOff.h"
#include "SSModel.h"

#define iabs(x) (((x)>0) ? (x) : -(x))

/*********************************************************************************************/

TSpatialAuxVariable RunOff::fFlux("Flux");

float RunOff::fAlpha[10] = { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 };
int RunOff::fNAlpha = 0;
float RunOff::fBeta[10] = { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 };
int RunOff::fNBeta = 0;
TModule* RunOff::fCurrentModule = NULL;
DistributedGrid*  RunOff::fCurrentGrid = NULL;
TVariable* RunOff::fCellSize = NULL;
TSpatialVariable* RunOff::fStudyArea = NULL;
TSpatialVariable* RunOff::fForcingFunction = NULL;
TSpatialVariable* RunOff::fOutput = NULL;
PixVec RunOff::fPointVec(100,(Pix)0);
ByteBuffer RunOff::fCommBuff;
byte RunOff::fInfo[8] = { 0,0,0,0,0,0,0,0 };

int RunOff::FOutFlowIndex0 = 10;
int RunOff::FOutFlowIndex1 = 10;
int RunOff::FForcingIndex = -1;
int RunOff::FOutputIndex = -1;

/*********************************************************************************************/
			
void RunOff::Setup( TSpatialVariable& v ) {
	if( fInfo[kSetup] == 0 ) {
		fInfo[kSetup]  = 1;
		if( fOutput ) {
			if( !fOutput->GetF(FisStateVar) ) { fOutput->Set(0.0); }
			fOutput->GrabMemory();
		}
		if( fCurrentModule == NULL ) { fCurrentModule = (TModule*) v.GetModule(); }
		if( fCurrentModule == NULL ) { gPrintErr( "No Frame Module Specified for Flux Module."); return; }
		TFrame& currentFrame = fCurrentModule->Frame();
		fCurrentGrid = currentFrame.Grid();
		if( fCurrentGrid->GetInfo(DGrid::kFrameType) != DGrid::kTree ) gFatal( "RunOff must be declared in Tree Network Frame. " );
		((Variable&)fFlux).Init(fCurrentModule);
		((TSpatialVariable&)fFlux).Allocate(0.0);
		if( fStudyArea ) { 
			fStudyArea->LinkEdges();
			TModule* m = (TModule*)fStudyArea->GetModule();
			if( !m->Frame().SamePointsAs( currentFrame )  )
				gFatal( "StudyArea map in RunOff module has wrong frame type. " ); 
		}
		if( fForcingFunction ) { 
			TModule* m = (TModule*)fForcingFunction->GetModule();
			if( !m->Frame().SamePointsAs( currentFrame ) ) 
				gFatal( "ForcingFunction varible in RunOff module has wrong frame type. " ); 
		}
		if( fOutput ) { 
			TModule* m = (TModule*)fOutput->GetModule();
			if( !m->Frame().SamePointsAs( currentFrame ) ) 
				gFatal( "Output variable in RunOff module has wrong frame type. " ); 
		}
	}
}

/*********************************************************************************************/

void RunOff::Wrapup() {
	if( fOutput  ) {
		if( (fOutput->GetF(FRegistered) && Model::I0().MemoryOp()) || fOutput->GetF(FVisible) ) {
			Model::I0().QueryInterface(fOutput->GetModule()->Name(),fOutput->Name());
		}
	}
}

/*********************************************************************************************/

int RunOff::Config(TConfigData& cd) {
  const CString& cmd = cd.Cmd();
  if(gDebug) { 
    sprintf(gMsgStr,"Reading Flux Config: Nargs = %d, CMD = %s", cd.NArgs(), cmd.chars() );	
    gPrintLog();  
  }
  if( cmd == "m" ) {
    CString* modName = cd.Arg(0,CD::kRequired );
    fCurrentModule = (TModule*) ((modName) ? Model::I0().GetModule( *modName, False ) : NULL);
    if( fCurrentModule == NULL ) { gPrintErr( *modName + ": Can't find Module."); return 0; } 
  } else if ( cmd == "a" ) {
    if( cd.FloatArg(0,fAlpha[0],CD::kRequired ) > 0 ) { fNAlpha = 1;
			if( cd.FloatArg(1,fAlpha[1]) > 0 ) { fNAlpha = 2;
				if( cd.FloatArg(2,fAlpha[2]) > 0 ) { fNAlpha = 3;
					if( cd.FloatArg(3,fAlpha[3]) > 0 ) { fNAlpha = 4;
						if( cd.FloatArg(4,fAlpha[4]) > 0 ) { fNAlpha = 5; }
					}
				}
			}
		}
  } else if ( cmd == "b" ) {
    if( cd.FloatArg(0,fBeta[0],CD::kRequired ) > 0 ) { fNBeta = 1;
			if( cd.FloatArg(1,fBeta[1]) > 0 ) { fNBeta = 2;
				if( cd.FloatArg(2,fBeta[2]) > 0 ) { fNBeta = 3;
					if( cd.FloatArg(3,fBeta[3]) > 0 ) { fNBeta = 4;
						if( cd.FloatArg(4,fBeta[4]) > 0 ) { fNBeta = 5; }
					}
				}
			}
		}
  } else   if( cmd == "cs" ) {
    CString* modName = cd.Arg(0,CD::kRequired );
    CString* varName = cd.Arg(1,CD::kRequired );
    TModule* mod = (TModule*) ((modName) ? Model::I0().GetModule( *modName, False ) : NULL);
    if( mod == NULL ) { gPrintErr( *modName + ": Can't find Module."); return 0; } 
    TVariable* var =  (TVariable*) mod->GetVariable(*varName,False);
    if( var == NULL ) { gPrintErr( *varName + ": Can't find Variable."); return 0; }   
    fCellSize = var;
  }  else   if( cmd == "sa" ) {
    CString* modName = cd.Arg(0,CD::kRequired );
    CString* varName = cd.Arg(1,CD::kRequired );
    cd.IntArg( 2, FOutFlowIndex0 );
    if( cd.IntArg( 3, FOutFlowIndex1 ) == 0 ) FOutFlowIndex1 = FOutFlowIndex0;
    TModule* mod = (TModule*) ((modName) ? Model::I0().GetModule( *modName, False ) : NULL);
    if( mod == NULL ) { gPrintErr( *modName + ": Can't find Module."); return 0; } 
    TVariable* var =  (TVariable*) mod->GetVariable(*varName,False);
    if( var == NULL ) { gPrintErr( *varName + ": Can't find Variable."); return 0; } 
    if( TModel::TranslationMode() == 0 ) {
      if( !var->GetF(FisSpatial ) ) { 
	gPrintErr( "NonSpatial Variable specified for studyArea in RunOff Module!");
      }
      fStudyArea = (TSpatialVariable*)var; 
    } else {
      var->SetF(FisSpatial,True,"RunOff");
      fStudyArea = (CAuxVariable*)var; 
    }
  }  else   if( cmd == "ff" ) {
    CString* modName = cd.Arg(0,CD::kRequired );
    CString* varName = cd.Arg(1,CD::kRequired );
		cd.IntArg( 2, FForcingIndex );
    TModule* mod = (TModule*) ((modName) ? Model::I0().GetModule( *modName, False ) : NULL);
    if( mod == NULL ) { gPrintErr( *modName + ": Can't find Module."); return 0; } 
    TVariable* var =  (TVariable*) mod->GetVariable(*varName,False);
    if( var == NULL ) { gPrintErr( *varName + ": Can't find Variable."); return 0; } 
    if( !var->GetF(FisSpatial ) ) { gPrintErr( "NonSpatial Variable specified for forcingFunction in RunOff Module!"); }  
    else { fForcingFunction = (TSpatialVariable*)var; }
	}  else   if( cmd == "ov" ) {
    CString* modName = cd.Arg(0,CD::kRequired );
    CString* varName = cd.Arg(1,CD::kRequired );
    cd.IntArg( 2, FOutputIndex );
    TModule* mod = (TModule*) ((modName) ? Model::I0().GetModule( *modName, False ) : NULL);
    if( mod == NULL ) { gPrintErr( *modName + ": Can't find Module."); return 0; } 
    TVariable* var =  (TVariable*) mod->GetVariable(*varName,False);
    if( var == NULL ) { gPrintErr( *varName + ": Can't find Variable."); return 0; } 
    if( !var->GetF(FisSpatial ) ) { gPrintErr( "NonSpatial Variable specified for output in RunOff Module!"); }  
    else { fOutput = (TSpatialVariable*)var; }
	} else return 0;
  return 1;
}


/*********************************************************************************************/

void RunOff::Full( TSpatialVariable& AvailWater, TTemporalVariable& fluxDistance )
{	
	Setup(AvailWater);
	AvailWater.LinkEdges();
	float flux_in = 0.0;
	int niter = (int) fluxDistance.Value();
	PointSetCoverage& water = AvailWater.Cov();
	PointSetCoverage& flux = fFlux.Cov();
	const float* fFlux_Eptr = flux.GridEnd();
	
	for(int i=0; i<niter; i++) {	
		flux.Set(0.0);
		for( Pix p = fCurrentGrid->first(); p; fCurrentGrid->next(p) ) {
			 Pix rp = fCurrentGrid->TranslateByLinkIndex( p );
			 if(rp) {
				 flux.AddValue( rp, water(p) );
				 water(p) = 0.0;
			 }
		}
		if( fStudyArea ) {
			float df=0.0;
			float* flux_ptr = flux.DataStart();
			float* studyArea_ptr = fStudyArea->Cov().DataStart();
			float* output_ptr = (fOutput) ? fOutput->Cov().DataStart() : (float*)NULL;
			while( flux_ptr < fFlux_Eptr ) {
				if( (*studyArea_ptr >= FOutFlowIndex0) && (*studyArea_ptr <= FOutFlowIndex1) ) {
					df = *flux_ptr;
					*flux_ptr = 0.0;
					if( output_ptr && ((FOutputIndex<0) || (*studyArea_ptr == FOutputIndex)) ) { 
						*output_ptr += df; 
					}
				}
				flux_ptr++;
				studyArea_ptr++;
				if( output_ptr ) { output_ptr++; } 
			}	
		}

		flux.LinkEdges();	
		float* fFlux_ptr = flux.DataStart();
		float* AvailWater_ptr = water.DataStart();
		float* water_Rptr = ( AvailWater.GetF(FisClamped) ) ? ((TSpatialStateVariable&)AvailWater).RDataStart() : (float*)NULL;
		while( fFlux_ptr < fFlux_Eptr ) {
			*AvailWater_ptr += *fFlux_ptr;
			if( water_Rptr ) { *water_Rptr = *AvailWater_ptr; water_Rptr++; }
			fFlux_ptr++;
			AvailWater_ptr++;
		}	
	}
	Wrapup();
}

/*********************************************************************************************/

void RunOff::FullEquilHead( TSpatialVariable& AvailWater, TSpatialVariable& SurfaceElev, TTemporalVariable& fluxDistance )
{	
	Setup(AvailWater);
	SurfaceElev.LinkEdges();
	AvailWater.LinkEdges();
	float flux_in = 0.0;
	int niter = (int) fluxDistance.Value();
	PointSetCoverage& water = AvailWater.Cov();
	PointSetCoverage& elev = SurfaceElev.Cov();
	PointSetCoverage& flux = fFlux.Cov();
	const float* fFlux_Eptr = flux.GridEnd();
	register float we0, wer, wr, e0, er;
	
	for(int i=0; i<niter; i++) {	
		flux.Set(0.0);
		for( Pix p = fCurrentGrid->first(); p; fCurrentGrid->next(p) ) {
			 float& w0 = water(p);
			 e0 = elev(p);
			 we0 =  w0 + e0;
			 register Pix rp = fCurrentGrid->TranslateByLinkIndex( p );
			 if( (gDebug > 1) && rp ) {
				 OrderedPoint& point0 = (OrderedPoint&)(*fCurrentGrid)(p);
				 OrderedPoint& point1 = (OrderedPoint&)(*fCurrentGrid)(rp);
				 int pc00 = point0(0);  // rows
				 int pc01 = point0(1);  // cols
				 int pc10, pc11;
				 if( (pc00 == 18) && (pc01==10) ) {
					 pc10 = point1(0); 
					 pc11 = point1(1);
					 int tst = 0;
				 }
			 }
			 if(rp) {
				 wr = water(rp);
				 er = elev(rp);
				 wer = wr + er;
				 if( (wer+w0) <= e0 ) {
					 flux.AddValue( rp, w0 );
					 w0 = 0.0;
				 } else if( wer < we0 ) {
					 float dw = (we0-wer);
					 float wmax = w0;
					 dw = (dw>wmax) ? wmax : dw;
					 flux.AddValue( rp, dw );
					 w0 -= dw;
//				 } else if( wer > we0 ){
//					 float dw = (wer-we0)/8;
//					 dw = (dw>wr/8) ? wr/8 : dw;
//					 flux.AddValue( rp, -dw );
//					 w0 += dw;
				 }
			 }
		}
		
		if( fStudyArea ) {
			float df=0.0;
			float* flux_ptr = flux.DataStart();
			float* studyArea_ptr = fStudyArea->Cov().DataStart();
			float* forcing_ptr = (fForcingFunction) ? fForcingFunction->Cov().DataStart() : (float*)NULL;
			float* output_ptr = (fOutput) ? fOutput->Cov().DataStart() : (float*)NULL;
			while( flux_ptr < fFlux_Eptr ) {
				if( (*studyArea_ptr >= FOutFlowIndex0) && (*studyArea_ptr <= FOutFlowIndex1) ) {
					if( forcing_ptr && ((FForcingIndex<0) || (*studyArea_ptr == FForcingIndex) ) ) {
						df = (*flux_ptr - *forcing_ptr);
						*flux_ptr -= df;

					} else {
						df = *flux_ptr;
						*flux_ptr = 0.0;
					}
					if( output_ptr && ((FOutputIndex<0) || (*studyArea_ptr == FOutputIndex)) ) { 
						*output_ptr = df; 
					}
				}
				flux_ptr++;
				studyArea_ptr++;
				if( output_ptr ) output_ptr++; 
				if( forcing_ptr ) forcing_ptr++;
			}	
		}
		
		flux.LinkEdges();	
		float* flux_ptr = flux.DataStart();
		float* AvailWater_ptr = water.DataStart();
		float* water_Rptr = ( AvailWater.GetF(FisClamped) ) ? ((TSpatialStateVariable&)AvailWater).RDataStart() : (float*)NULL;
		while( flux_ptr < fFlux_Eptr ) {
			*AvailWater_ptr += *flux_ptr;
			if( water_Rptr ) { *water_Rptr = *AvailWater_ptr; water_Rptr++; }
			flux_ptr++;
			AvailWater_ptr++;
		}
	}
	Wrapup();
}

void RunOff::FullEquilHead1( TSpatialVariable& AvailWater, TSpatialVariable& SurfaceElev )
{	
	if( fNAlpha < 2 ) { gPrintErr("Must define at least two parameters for FullEquilHead in config."); return; }
	int niter = (int) fAlpha[0];
	float flux_parm = fAlpha[1];
	float outflux_parm = fAlpha[2];
	Pix p, pl;
	Setup(AvailWater);
	SurfaceElev.LinkEdges();
	PointSetCoverage& water = AvailWater.Cov();
	PointSetCoverage& elev = SurfaceElev.Cov();
	const float* water_Eptr = water.GridEnd();
	CreatePointVec( );
	
	register int active_link[9];
	register float water_val[9];
	register float water_head[9];
	register float base_val[9];
	register Pix pix_val[9];
	register float water_level;
	register int nlinks;
	register int n_active_links = 0;
	
	for(int iter=0; iter<niter; iter++) {
		AvailWater.LinkEdges();	
		int pv_start = 0;
		while( p = fPointVec.elem(pv_start++) ) {		
			nlinks=0;
			TOrderedObjectDLListNode* pnode = (TOrderedObjectDLListNode*)p;
			float& w0 = water(p);
			water_val[0] = water(p)*flux_parm;
			base_val[0] = elev(p) + water_val[0]*(flux_parm-1);
			water_head[0] = ( water_val[0] + base_val[0] );
			pix_val[0] = p;
			active_link[0] = 1;
			water_level = water_head[0];
			while(  pl =  pnode->plink( ++nlinks ) ) {
				pix_val[nlinks] = pl;				
				water_val[nlinks] = water(pl)*flux_parm;
				base_val[nlinks] = elev(pl) + water_val[nlinks]*(flux_parm-1);
				water_head[nlinks] = ( water_val[nlinks] + base_val[nlinks] );
				water_level += water_head[nlinks];
				active_link[nlinks] = 1;
			}
			water_level /= nlinks;
			n_active_links = nlinks;
			

/*			
			if(gDebug > 1) {
				sprintf(gMsgStr,"\n\nRIVER FLUX (%d): wl(%f), nl(%d)\n",niter,water_level,nlinks);
				gPrintLog();
				for(int il=0; il<nlinks; il++ ) {
					sprintf(gMsgStr," ( %.1f, %.4f, %.2f, %d ) ",base_val[il],water_val[il],water_head[il],active_link[il]);
					gPrintLog();
				}
			}
*/
			start_equilibration:
			
			float w_tmp = 0;			
			int do_loop=0;
			for( int i=1; i<nlinks; i++ ) if( active_link[i] ) {
				if ( base_val[i] >= water_level ) {
					do_loop = 1;
					float& w1 = water(pix_val[i]);
					w0 += water_val[i];
					water_val[0] += water_val[i];
					w1 -= water_val[i];
					active_link[i] = 0;
					n_active_links--;
				} else w_tmp += water_head[i];
			}
			water_head[0] = ( water_val[0] + base_val[0] );
			w_tmp += 	water_head[0];
			water_level = w_tmp / n_active_links;
/*			
			if(gDebug > 1) {
				sprintf(gMsgStr,"\nRIVEREQUIL: wl(%f), nl(%d), nal(%d) \n",water_level,nlinks,n_active_links);
				gPrintLog();
				for(int il=0; il<nlinks; il++ ) {
					sprintf(gMsgStr," ( %.1f, %.4f, %.2f, %d ) ",base_val[il],water_val[il],water_head[il],active_link[il]);
					gPrintLog();
				}
			}   */
			if( base_val[0] > water_level ) {
				w0 -= water_val[0];
				for( int i=1; i<nlinks; i++ ) {
					if( active_link[i] ) {
						float& w1 = water(pix_val[i]);
						w1 += water_val[0]/(n_active_links-1);
					}
				}
			}	
			else if ( n_active_links == 1 ) {;}
			else if (do_loop) {
				goto start_equilibration;
			} else {
				 w0 += (water_level - water_head[0]);
				 for( int i=1; i<nlinks; i++ ) if( active_link[i] ) {
					 float& wl = water(pix_val[i]);
					 wl += (water_level - water_head[i]);
				 }			
			}
		}
		
		if( fStudyArea ) {
			float* water_ptr = water.DataStart();
			float* studyArea_ptr = fStudyArea->Cov().DataStart();
			float* forcing_ptr = (fForcingFunction) ? fForcingFunction->Cov().DataStart() : (float*)NULL;
			float* output_ptr = (fOutput) ? fOutput->Cov().DataStart() : (float*)NULL;
			if( forcing_ptr || output_ptr ) {
				while( water_ptr < water_Eptr ) {
					if( (*studyArea_ptr >= FOutFlowIndex0) && (*studyArea_ptr <= FOutFlowIndex1) ) {
						if( forcing_ptr && ((FForcingIndex<0) || (*studyArea_ptr == FForcingIndex) ) ) {
							*water_ptr = *forcing_ptr;
						} 
						if( output_ptr && ((FOutputIndex<0) || (*studyArea_ptr == FOutputIndex)) ) { 
							*output_ptr = (*water_ptr)*outflux_parm;
							*water_ptr *= (1-outflux_parm);
						}
					}
					water_ptr++;
					studyArea_ptr++;
					if( output_ptr ) output_ptr++; 
					if( forcing_ptr ) forcing_ptr++;
				}	
			}
		}
/*		
		if(gDebug > 1) {
			sprintf(gMsgStr,"\nDONE: wl(%f), nl(%d), nal(%d) ",water_level,nlinks,n_active_links);
			gPrintLog();
			for(int il=0; il<nlinks; il++ ) {
				sprintf(gMsgStr," (%.4f,%d) ",water(pix_val[il]),active_link[il]);
				gPrintLog();
			}
		}  */
	}
	float* water_Rptr = ( AvailWater.GetF(FisClamped) ) ? ((TSpatialStateVariable&)AvailWater).RDataStart() : (float*)NULL;
	if( water_Rptr ) {
		float* water_ptr = water.DataStart();
		while( water_ptr < water_Eptr ) {
			*water_Rptr = *water_ptr;
			water_Rptr++;
			water_ptr++;
		}	
	}
	Wrapup();
}

void RunOff::PartialHead( TSpatialVariable& AvailWater )
{	
	if( fNAlpha < 2 ) { gPrintErr("Must define at least two parameters for PartialHead in config ( RunOff a(x,y) )."); return; }
	int niter = (int) fAlpha[0];
	float flux_parm = fAlpha[1];
	Pix p, pl;
	float inv_flux_parm = (1-flux_parm);
	float outflux_parm = fAlpha[2];
	Setup(AvailWater);
	PointSetCoverage& water = AvailWater.Cov();
	const float* water_Eptr = water.GridEnd();
	CreatePointVec( );
	
	for(int iter=0; iter<niter; iter++) {	
		AvailWater.LinkEdges();	
		int pv_start = 0;
		
		while( p = fPointVec.elem(pv_start++) ) {		
			TOrderedObjectDLListNode* pnode = (TOrderedObjectDLListNode*)p;
			float& w0 = water(p);

			int link_index = 0;					
			while( pl =  pnode->plink( ++link_index ) ) {	
				float& w1 = water(pl);
				if( w1 ) {
					w0 += w1*flux_parm;
					w1 *= inv_flux_parm;
				}
			}
		}
		if( fStudyArea ) {
			float* water_ptr = water.DataStart();
			float* studyArea_ptr = fStudyArea->Cov().DataStart();
			float* forcing_ptr = (fForcingFunction) ? fForcingFunction->Cov().DataStart() : (float*)NULL;
			float* output_ptr = (fOutput) ? fOutput->Cov().DataStart() : (float*)NULL;
			if( forcing_ptr || output_ptr ) {
				while( water_ptr < water_Eptr ) {
					if( (*studyArea_ptr >= FOutFlowIndex0) && (*studyArea_ptr <= FOutFlowIndex1) ) {
						if( forcing_ptr && ((FForcingIndex<0) || (*studyArea_ptr == FForcingIndex) ) ) {
							*water_ptr = *forcing_ptr;
						} 
						if( output_ptr && ((FOutputIndex<0) || (*studyArea_ptr == FOutputIndex)) ) { 
							*output_ptr = (*water_ptr)*outflux_parm;
							*water_ptr *= (1-outflux_parm);
						}
					}
					water_ptr++;
					studyArea_ptr++;
					if( output_ptr ) output_ptr++; 
					if( forcing_ptr ) forcing_ptr++;
				}	
			}
		}
	}
	float* water_Rptr = ( AvailWater.GetF(FisClamped) ) ? ((TSpatialStateVariable&)AvailWater).RDataStart() : (float*)NULL;
	if( water_Rptr ) {
		float* water_ptr = water.DataStart();
		while( water_ptr < water_Eptr ) {
			*water_Rptr = *water_ptr;
			water_Rptr++;
			water_ptr++;
		}	
	}		
	if( gDebug > 1 ) { AvailWater.LinkEdges(); }	
	Wrapup();
}
void RunOff::FullEquilVol( TSpatialVariable& AvailWater, TSpatialVariable& SurfaceElev, TTemporalVariable& fluxDistance )
{	
	Setup(AvailWater);
	SurfaceElev.LinkEdges();
	AvailWater.LinkEdges();
	if(fCellSize==NULL) gFatal("Error, CELLSIZE undefined in Flux Module ( use cs() command )" );
	float flux_in = 0.0;
	int niter = (int) fluxDistance.Value();
	PointSetCoverage& water = AvailWater.Cov();
	PointSetCoverage& elev = SurfaceElev.Cov();
	PointSetCoverage& flux = fFlux.Cov();
	const float* fFlux_Eptr = flux.GridEnd();
	register float we0, wer, e0, er, cs0, csr;
	
	for(int i=0; i<niter; i++) {	
		flux.Set(0.0);
		for( Pix p = fCurrentGrid->first(); p; fCurrentGrid->next(p) ) {
			 cs0 = fCellSize->GenericValue(p);
			 float& w0 = water(p);
			 e0 = elev(p);
			 we0 =  w0/cs0 + e0;
			 register Pix rp = fCurrentGrid->TranslateByLinkIndex( p );
			 if(rp) {
				 float& wr = water(rp);
				 er = elev(rp);
				 wer = wr/csr + er;
				 if( wer <= e0 ) {
					 flux.AddValue( rp, w0 );
					 w0 = 0.0;
				 } else if( wer < we0 ) {
					 float dw = (we0-wer)*cs0/4;
					 dw = (dw>w0) ? w0 : dw;
					 flux.AddValue( rp, dw );
					 w0 -= dw;
				 } else if( wer > we0 ){
					 float dw = (wer-we0)*csr/4;
					 dw = (dw>wr/8) ? wr/8 : dw;
					 flux.AddValue( rp, -dw );
					 w0 += dw;
				 }
			 }
		}
		
		if( fStudyArea ) {
			float df=0.0;
			float* flux_ptr = flux.DataStart();
			float* studyArea_ptr = fStudyArea->Cov().DataStart();
			float* forcing_ptr = (fForcingFunction) ? fForcingFunction->Cov().DataStart() : (float*)NULL;
			float* output_ptr = (fOutput) ? fOutput->Cov().DataStart() : (float*)NULL;
			while( flux_ptr < fFlux_Eptr ) {
				if( (*studyArea_ptr >= FOutFlowIndex0) && (*studyArea_ptr <= FOutFlowIndex1) ) {
					if( forcing_ptr && ((FForcingIndex<0) || (*studyArea_ptr == FForcingIndex) ) ) {
						df = (*flux_ptr - *forcing_ptr);
						*flux_ptr -= df;
					} else {
						df = *flux_ptr;
						*flux_ptr = 0.0;
					}
					if( output_ptr && ((FOutputIndex<0) || (*studyArea_ptr == FOutputIndex)) ) { 
						*output_ptr = df; 
					}
				}
				flux_ptr++;
				studyArea_ptr++;
				if( output_ptr ) output_ptr++; 
				if( forcing_ptr ) forcing_ptr++;
			}	
		}
		
		flux.LinkEdges();	
		float* flux_ptr = flux.DataStart();
		float* AvailWater_ptr = water.DataStart();
		float* water_Rptr = ( AvailWater.GetF(FisClamped) ) ? ((TSpatialStateVariable&)AvailWater).RDataStart() : (float*)NULL;
		while( flux_ptr < fFlux_Eptr ) {
			*AvailWater_ptr += *flux_ptr;
			if( water_Rptr ) { *water_Rptr = *AvailWater_ptr; water_Rptr++; }
			flux_ptr++;
			AvailWater_ptr++;
		}	
	}
	Wrapup();
}

/*********************************************************************************************/

void RunOff::Partial( TSpatialVariable& AvailWater, TTemporalVariable& fluxDistance )
{
	Setup(AvailWater);	
	AvailWater.LinkEdges();
	float flux_out = 0.0, rate = fAlpha[0];
	int niter = (int) fluxDistance.Value();
	PointSetCoverage& water = AvailWater.Cov();
	PointSetCoverage& flux = fFlux.Cov();
	const float* fFlux_Eptr = flux.GridEnd();
	
	for(int i=0; i<niter; i++) {	
		flux.Set(0.0);
		for( Pix p = fCurrentGrid->first(); p; fCurrentGrid->next(p) ) {
			 Pix rp = fCurrentGrid->TranslateByLinkIndex( p );
			 if(rp)  {
				 float& water_val = water(p);
				 flux_out = rate*water_val;
				 flux.AddValue( rp, flux_out );
				 water_val -= flux_out;
			 }
		}

		flux.LinkEdges();	
		float* fFlux_ptr = flux.DataStart();
		float* AvailWater_ptr = water.DataStart();
		float* water_Rptr = ( AvailWater.GetF(FisClamped) ) ? ((TSpatialStateVariable&)AvailWater).RDataStart() : (float*)NULL;
		while( fFlux_ptr < fFlux_Eptr ) {
			*AvailWater_ptr += *fFlux_ptr;
			if( water_Rptr ) { *water_Rptr = *AvailWater_ptr; water_Rptr++; }
			fFlux_ptr++;
			AvailWater_ptr++;
		}	
	}
	Wrapup();
}

/*********************************************************************************************/

void RunOff::GWHead1( TSpatialVariable& GWater, TSpatialVariable& GWHead, TSpatialVariable& Porosity, TSpatialVariable& FluxRate )
{
	if( fNBeta < 2 ) { gPrintErr("Must define at least two parameters for GWHead1 in config ( RunOff b(x,y) )."); return; }
	Setup(GWHead);	
	GWHead.LinkEdges();
	DistributedGrid::EFrameType ft = (DistributedGrid::EFrameType) fCurrentGrid->GetInfo(DistributedGrid::kFrameType);
	FluxRate.LinkEdges();
	int niter = (int) fBeta[0];
	float base_rate = fBeta[1], ftmp, fmax;
	PointSetCoverage& head = GWHead.Cov();
	PointSetCoverage& rate = FluxRate.Cov();
	PointSetCoverage& flux = fFlux.Cov();
	const float* flux_Eptr = flux.GridEnd();
	
	for(int i=0; i<niter; i++) {	
		flux.Set(0.0);
		for( Pix p = fCurrentGrid->first(); p; fCurrentGrid->next(p) ) {
			int rr, rc;
			for( Grid_Direction il = firstGD(); moreGD(il); incrGD(il) ) { 
				 if( !GMR(rr,rc,il) ) { gFatal("Illegal Index in GWHead1");  }
				 Pix rp = 0;
				 if( ft == DistributedGrid::kGrid ) { 
					 rp = fCurrentGrid->TranslateByLinks( p, rr, rc ); 
				 } else { 
					 rp = fCurrentGrid->TranslateByGridCoord( p, rr, rc ); 
				 }
				 if(rp)  {
					 float dh = ( head(rp) - head(p) );
					 if( dh > 0 ) {
						 ftmp = base_rate * dh * rate(rp) * Util::ramp( head(rp) );
						 fmax = dh/2;
						 ftmp = ( ftmp > fmax ) ? fmax : ftmp;
					 } else {
						 ftmp = base_rate * dh * rate(p) * Util::ramp( head(p) );
						 fmax = dh/2;
						 ftmp = ( ftmp < fmax ) ? fmax : ftmp;
					 }
					 flux(p) += ftmp;
					 flux(rp) -= ftmp;
				 }
			}
		}
		flux.LinkEdges();	
		float* flux_ptr = flux.DataStart();
		float* poros_ptr = Porosity.DataStart();
		float* AvailWater_ptr = GWater.DataStart();
		float* water_Rptr = ( GWater.GetF(FisClamped) ) ? ((TSpatialStateVariable&)GWater).RDataStart() : (float*)NULL;
		while( flux_ptr < flux_Eptr ) {
			*AvailWater_ptr += (*flux_ptr) * (*poros_ptr);
			if( water_Rptr ) { *water_Rptr = *AvailWater_ptr; water_Rptr++; }
			flux_ptr++;
			poros_ptr++;
			AvailWater_ptr++;
		}	
	}
	Wrapup();
}

/*********************************************************************************************/

void RunOff::GWHead2( TSpatialVariable& GWater, TSpatialVariable& Porosity )
{
	if( fNBeta < 2 ) { gPrintErr("Must define at least two parameters for GWHead1 in config ( RunOff b(x,y) )."); return; }
	Setup(GWater);	
	int niter = (int) fBeta[0];
	float base_rate = fBeta[1];                        // base_rate is in range ( 0.0 -> 1.0 )
	PointSetCoverage& porosity = Porosity.Cov();
	PointSetCoverage& water = GWater.Cov();
	register Pix rpix[10];
	static int call_count = 0;
	Grid_Direction il;
	Pix p;
	int first_tcall=0;
	
	for(int i=0; i<niter; i++) {	
		GWater.LinkEdges();
		Porosity.LinkEdges();
		int mc0 = call_count++ % 2;
		for( p = fCurrentGrid->first(); p; fCurrentGrid->next(p) ) {
			const OrderedPoint& pt = fCurrentGrid->GetPoint(p);
			int ir = pt(0), ic = pt(1);
			int mc1 = (mc0+ir/2) % 2;
			if( (pt.GetObjInfo(DistributedGrid::kGhostType) == 2) && ((ir % 2) == mc0) && ((ic % 2) == mc1) ) {
				int rr, rc; 
				float w_sum = water(p);
				float p_sum = porosity(p);
				for( il = firstGD(); moreGD(il); incrGD(il) ) {  
					 if( !GMR(rr,rc,il) ) { gFatal("Illegal Index in GWHead1");  }
					 Pix rp =  fCurrentGrid->Translate( p, rr, rc ); 
					 rpix[il] = rp;
					 if( rp )  {
						 w_sum += water(rp);
						 p_sum += porosity(rp);
					 } 
				}
				float h0 = w_sum / p_sum;
				
				if( (gDebug>1) && (first_tcall == 150) ) {
					sprintf(gMsgStr,"\nGWHead2: (%d,%d:%f) : (%f/%f)->%f : ",ir,ic,water(p)/porosity(p),w_sum,p_sum,h0);
					gPrintLog();
					for( il = firstGD(); moreGD(il); incrGD(il) ) {  
						Pix rp = rpix[il];
						if( rp ) {
							sprintf(gMsgStr," %f ",water(rp)/porosity(rp));
							gPrintLog();
						}
					}	
				}
																			
				water(p) += (  h0 * porosity(p) - water(p) ) * base_rate;
				for( il = firstGD(); moreGD(il); incrGD(il) ) {  
					Pix rp = rpix[il];
					if( rp ) {
						water(rp) += (  h0 * porosity(rp) - water(rp) ) * base_rate;
					}
				}	
				
				if( (gDebug>1) && (first_tcall++ == 150) ) {
					sprintf(gMsgStr," -> (%f) : ",water(p)/porosity(p));
					gPrintLog();
					for( il = firstGD(); moreGD(il); incrGD(il) ) {
						Pix rp = rpix[il];
						if( rp ) {
							sprintf(gMsgStr," %f ",water(rp)/porosity(rp));
							gPrintLog();
						}
					}	
				}				 
			}
		}
	}
	Wrapup();
}

void RunOff::GWHead3( TSpatialVariable& GWater, TSpatialVariable& Porosity )
{
	if( fNBeta < 2 ) { gPrintErr("Must define at least two parameters for GWHead1 in config ( RunOff b(x,y) )."); return; }
	Setup(GWater);	
	int niter = (int) fBeta[0];
	float base_rate = fBeta[1];                        // base_rate is in range ( 0.0 -> 1.0 )
	PointSetCoverage& porosity = Porosity.Cov();
	PointSetCoverage& water = GWater.Cov();
	register Pix rpix[10];
	static int call_count = 0;
	int first_tcall=0, il;
	
	for(int i=0; i<niter; i++) {	
		GWater.LinkEdges();
		Porosity.LinkEdges();
		int mc0 = call_count++ % 2;
		for( Pix p = fCurrentGrid->first(); p; fCurrentGrid->next(p) ) {
			const OrderedPoint& pt = fCurrentGrid->GetPoint(p);
			int ir = pt(0), ic = pt(1);
			int mc1 = (mc0+ir/2) % 2;
			if( (pt.GetObjInfo(DistributedGrid::kGhostType) == 2) && ((ir % 2) == mc0) && ((ic % 2) == mc1) ) {
				int rr, rc; 
				float w_sum = water(p);
				float p_sum = porosity(p);
				Grid_Direction il;
				for( il = firstGD(); moreGD(il); incrGD(il) ) {
					 if( !GMR(rr,rc,il) ) { gFatal("Illegal Index in GWHead1");  }
					 Pix rp =  fCurrentGrid->Translate( p, rr, rc ); 
					 rpix[il] = rp;
					 if( rp )  {
						 w_sum += water(rp);
						 p_sum += porosity(rp);
					 } 
				}
				float h0 = w_sum / p_sum;
				
				if( (gDebug>1) && (first_tcall == 150) ) {
					sprintf(gMsgStr,"\nGWHead2: (%d,%d:%f) : (%f/%f)->%f : ",ir,ic,water(p)/porosity(p),w_sum,p_sum,h0);
					gPrintLog();
					for( il = firstGD(); moreGD(il); incrGD(il) ) {
						Pix rp = rpix[il];
						if( rp ) {
							sprintf(gMsgStr," %f ",water(rp)/porosity(rp));
							gPrintLog();
						}
					}	
				}
																			
				water(p) += (  h0 * porosity(p) - water(p) ) * base_rate;
				for( il = firstGD(); moreGD(il); incrGD(il) ) {
					Pix rp = rpix[il];
					if( rp ) {
						water(rp) += (  h0 * porosity(rp) - water(rp) ) * base_rate;
					}
				}	
				
				if( (gDebug>1) && (first_tcall++ == 150) ) {
					sprintf(gMsgStr," -> (%f) : ",water(p)/porosity(p));
					gPrintLog();
					for( il = firstGD(); moreGD(il); incrGD(il) ) {
						Pix rp = rpix[il];
						if( rp ) {
							sprintf(gMsgStr," %f ",water(rp)/porosity(rp));
							gPrintLog();
						}
					}	
				}				 
			}
		}
	}
	Wrapup();
}

/*********************************************************************************************/


void RunOff::CreatePointVec( ) {
	if( fInfo[kPointVecGenerated] == 0 ) {
		fInfo[kPointVecGenerated] = 1;
		int pv_start=0;
		Pix p, pl;
		Region2& local_region = fCurrentGrid->Region();
		TMap point_map( &(fCurrentGrid->GlobalRegion()) );
		point_map.Set(0);
		int pv_end = fPointVec.copy(fCurrentGrid->RootPointVec());
		while(1) {
			int new_points = 0, new_bpoints=0, go=0;						
			while( p = fPointVec.elem(pv_start) ) {	
				pv_start++;	
				int nlinks=0;
				TOrderedObjectDLListNode* pnode = (TOrderedObjectDLListNode*)p;
				OrderedPoint& pt = (OrderedPoint&) pnode->Object();
				int gi = pt.GetObjIndex( DGrid::kGhostLayer );

				if( gDebug > 1 ) {
					sprintf(gMsgStr,"\n\nRIVER FLUX (%d): (%d,%d:%d)",pv_start,pt(0),pt(1),gi);
					gPrintLog();
				}
				if( gi == 0 ) {
					while( pl =  pnode->plink( ++nlinks ) ) {
						TOrderedObjectDLListNode* pnode1 = (TOrderedObjectDLListNode*)pl;
						OrderedPoint& pt1 = (OrderedPoint&) pnode1->Object();
						int gi1 =	pt1.GetObjIndex( DGrid::kGhostLayer );
						int ir = pt1(0), ic = pt1(1);
						if( point_map.Value(ir,ic) == 0 ) {
							point_map.SetValue(ir,ic,1);
							fPointVec.elem(pv_end++) = pl; 
							new_points++;
							if(gDebug > 1) {
								sprintf(gMsgStr,"\nAdd Point(%d:%d:%d): (%d,%d)\n",pv_end,nlinks,gi1,ir,ic);
								gPrintLog();
							}
#ifdef USE_MPI
							if( gi1 > 0 ) {
								fCommBuff.resize((++new_bpoints)*sizeof(int)*2);
								int* ibuff = (int*)fCommBuff.buff();
								ibuff[new_bpoints*2-2] = ir;
								ibuff[new_bpoints*2-1] = ic;
							}
#endif
						}
					}
				}
			}
#ifdef USE_MPI
			for( int ip=0; ip<gNProc; ip++ ) {
				int size = new_bpoints;
				MPI_Bcast( &size, 1, MPI_INT, ip, MPI_COMM_WORLD );
				if(size>0) {
					go = 1; 
					fCommBuff.resize(size*sizeof(int)*2);
					MPI_Bcast( fCommBuff.buff(), size*2, MPI_INT, ip, MPI_COMM_WORLD );
					for( int ipt=0; ipt<size; ipt++ ) {
						int* ibuff = (int*)fCommBuff.buff();
						int ir = ibuff[ipt*2], ic = ibuff[ipt*2+1];
						if( local_region.inside(ir,ic) && ( point_map.Value(ir,ic) == 0 ) ) {
							point_map.SetValue(ir,ic,1);
							PointRef pr(ir,ic,local_region);
							Pix pb = fCurrentGrid->GetPixFast( pr );
							if(pb) { 
								fPointVec.elem(pv_end++) = pb; 
								if(gDebug > 1) {
									sprintf(gMsgStr,"\nAdd Boundary Point(%d:%d): (%d,%d)\n",pv_end,ipt,ir,ic);
									gPrintLog();
								}
							}
						}
					}
				}
			}
#endif
			if( !go ) break;
		} 
		if( gDebug>1 ) { 
			point_map.SetObjInfo(TMap::kMapType,kMap2);
			CString mapName =  "RunOff_PointMap";
			CPathString pathName(Env::ArchivePath());
			point_map.WriteM2File( pathName, mapName, gIProc, NULL, gIProc);
		}
	}
}





