#include "SimLib.h"
#include "Globals.h"
#include "SSModel.h"
#include "SSModule.h"
#include "CVariable.h"
#include "PixVec.h"
#include "TPipe.h"
#include "controlsocket.h"

#include "SSComm.h"

#ifdef HAS_SOCKETS
extern "C" {
#include <unistd.h>
#include <sys/mman.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <netdb.h>
#include <ctype.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <fcntl.h>
// pid_t wait3( int*, int, struct rusage*);
}
#endif

#define VAR_UNSET  0
#define VAR_SET    1
#define VAR_LOCKED 2
#define VAR_ERROR  3

#define MAX_SHARED_SCALARS 40
#define MAX_SHARED_SPATIAL 20

// spatial variables must fit into this space
#define SPATIAL_VAR_PAGESIZE 124000
// scalar variables timeseries must fit into this space
#define SCALAR_VAR_PAGESIZE 8000
FILE* dfp = NULL;

#ifdef HAS_SOCKETS

// -- util class ------------------------------------------------------


void sleep_u(long m)
{
  timeval v;
  v.tv_sec  = 0l;
  v.tv_usec = m;
  select(0,NULL,NULL,NULL,&v);
}

#endif


PathName::PathName(const char* v) : CString(v)
{
  dot = length() - 1;
  // might have newlines at the end
  while (isspace((*this)[dot]) && dot)
    (*this)(dot--) = '\0';
  while(dot >= 0) 
    if ((*this)[dot] == '.') break; 
    else dot--;
  if (dot >= 0) (*this)(dot) = '\0';
}

// -- run-length encoder -------------------------------------------------

class RLEncoder
{
  byte*    _d;      // pointer to data to be encoded
  long     _ndata;  // number of floats to encode
  unsigned _bsize;  // byte size of encoded result
  unsigned _maxc;   // max bytes for single chunk of encoded data
  long     _ofs;    // current offset in data

  Bool     _bw;     // read data backwards

  int    _encode(byte,  byte*);
  int    _encode(short, byte*);
  int    _encode(long,  byte*);

public:

  // buf is passed to hold data (no smaller than _maxc)
  // returns n. of bytes in buf; called until 0 is returned
  unsigned int encode_b(byte* buf);
  unsigned int encode_i(byte* buf);
  unsigned int encode_l(byte* buf);

  void encode (byte* buf);

  RLEncoder(byte* f, long ndata, int bsize, int maxc, Bool scale = FALSE) :
    _d(f), _ndata(ndata), _bsize(bsize), _maxc(maxc), _ofs(0l), _bw(0)
  { if (_bw) _ofs = _ndata - 1; }
  ~RLEncoder() {;}
};

inline int RLEncoder::_encode(byte v, byte* b)
{
  *b = v;
  return 1;
}

inline int RLEncoder::_encode(short i, byte* b)
{
  *b     = (byte)(i >> 8);
  *(b+1) = (byte)i;
  return 2;
}

inline int RLEncoder::_encode(long l, byte* b)
{
  *b     = (byte)(l >> 24);
  *(b+1) = (byte)(l >> 16);
  *(b+2) = (byte)(l >> 8);
  *(b+3) = (byte)l;
  return 4;
}

unsigned RLEncoder::encode_b(byte* b)
{
  if ((_bw && _ofs <= 0) || (!_bw && _ofs >= (_ndata - 1))) 
    return 0;
  
  byte rd = _d[_ofs];
  long i = _ofs; 
  unsigned di = 0, dc = sizeof(unsigned);
  const unsigned chm = _maxc - 1 - _bsize;

  for (; _bw ? (i >= 0) : (i < _ndata); _bw ? i-- : i++)
    {
      if (_d[i] == rd) di++;

      if (_d[i] != rd || (_bw ? (i == 0) : (i == (_ndata - 1))) || di == 255)
	{
	  if (dc < chm)
	    {
	      // write run length
	      dc += _encode((byte)di, b+dc);
	      dc += _encode(rd, b+dc);
	      // next pass
	      _ofs = i;
	      rd = _d[i];
	      di = 1;
	    }
	  if (dc >= chm || (_bw ? (i == 0) : (i == (_ndata - 1))))
	   {
	     // encode remaining
	     dc += _encode((byte)di, b+dc);
	     dc += _encode(rd, b+dc);
	     // write (unsigned) chunk size at start of chunk and return
	     memcpy(b, &dc, sizeof(unsigned));
	     return dc;
	   }
	}
    }
}

unsigned RLEncoder::encode_i(byte* b)
{
  if ((_bw && _ofs <= 0) || (!_bw && _ofs >= (_ndata - 1))) 
    return 0;
  
  short rd = _d[_ofs];
  long i = _ofs; 
  unsigned di = 0, dc = sizeof(unsigned);
  const unsigned chm = _maxc - 1 - _bsize;

  for (; _bw ? (i >= 0) : (i < _ndata); _bw ? i-- : i++)
    {
      if (_d[i] == rd) di++;

      if (_d[i] != rd || (_bw ? (i == 0) : (i == (_ndata - 1))) || di == 255)
	{
	  if (dc < chm)
	    {
	      // write run length
	      dc += _encode((byte)di, b+dc);
	      dc += _encode(rd, b+dc);
	      // next pass
	      _ofs = i;
	      rd = _d[i];
	      di = 1;
	    }
	  if (dc >= chm || (_bw ? (i == 0) : (i == (_ndata - 1))))
	   {
	     // encode remaining
	     dc += _encode((byte)di, b+dc);
	     dc += _encode(rd, b+dc);
	     // write (unsigned) chunk size at start of chunk and return
	     memcpy(b, &dc, sizeof(unsigned));
	     return dc;
	   }
	}
    }
}

unsigned RLEncoder::encode_l(byte* b)
{
  if ((_bw && _ofs <= 0) || (!_bw && _ofs >= (_ndata - 1))) 
    return 0;
  
  byte rd = _d[_ofs];
  long i = _ofs; 
  unsigned di = 0, dc = sizeof(unsigned);
  const unsigned chm = _maxc - 1 - _bsize;

  for (; _bw ? (i >= 0) : (i < _ndata); _bw ? i-- : i++)
    {
      if (_d[i] == rd) di++;

      if (_d[i] != rd || (_bw ? (i == 0) : (i == (_ndata - 1))) || di == 255)
	{
	  if (dc < chm)
	    {
	      // write run length
	      dc += _encode((byte)di, b+dc);
	      dc += _encode(rd, b+dc);
	      // next pass
	      _ofs = i;
	      rd = _d[i];
	      di = 1;
	    }
	  if (dc >= chm || (_bw ? (i == 0) : (i == (_ndata - 1))))
	   {
	     // encode remaining
	     dc += _encode((byte)di, b+dc);
	     dc += _encode(rd, b+dc);
	     // write (unsigned) chunk size at start of chunk and return
	     memcpy(b, &dc, sizeof(unsigned));
	     return dc;
	   }
	}
    }
}


void RLEncoder::encode(byte* buf)
{
  unsigned n, end = 0; 
  long ofs = 0l;

  switch (_bsize)
  {
  case 1:
    while ((n = encode_b(buf+ofs)) > 0)
      ofs += (long)n;
    // mark end of RLE data
    memcpy((void*)(buf+ofs), &end, sizeof(unsigned));
    break;
  case 2:
    while ((n = encode_i(buf+ofs)) > 0)
      ofs += (long)n;
    // mark end of RLE data
    memcpy((void*)(buf+ofs), &end, sizeof(unsigned));
    break;
  case 4:
    while ((n = encode_l(buf+ofs)) > 0)
      ofs += (long)n;
    // mark end of RLE data
    memcpy((void*)(buf+ofs), &end, sizeof(unsigned));
    break;
  default:
    /* unimplemented */
    break;
  }   
}

// -- shared object manager ----------------------------------------------


struct obj_desc {
  char  _path[256];
  byte  _state; 

  unsigned _idx;
  unsigned _bsz;
  TMap*   _map;
  float    _val;
  float    _umin;
  float    _umax;
  unsigned _x1;
  unsigned _y1;
  unsigned _x2;
  unsigned _y2;
  Bool     _resc;
  char     _format;
  float    _downsample;
};

struct var_desc
{
  unsigned long dSize;
  unsigned int  xSize;
  unsigned int  ySize;
  float    _min;
  float    _max;
};

#ifdef HAS_SOCKETS

class SharedObjectManager 
{
  // do NOT add virtual functions to this class!!!!

  int   _n_scalar;
  int   _n_spatial;
  float _time;
  Bool  _lock;
  int   _parent;
  int   _syncflag;

  struct obj_desc _scalars[MAX_SHARED_SCALARS];
  struct obj_desc _spatial[MAX_SHARED_SPATIAL]; 

  // no action is needed since it's not allocated but
  // mapped on shared zero-filled space
  void init() {;}
  // needs to munmap() all segments
  void destroy();

  struct obj_desc*  find_slot(char*, Bool, Bool spatial); 
  TVariable* get_variable(char*);
  
public:
  
  // size of shared memory chunk to be allocated by server
  static size_t shmem_size()
  {
    return sizeof(SharedObjectManager) +
      (MAX_SHARED_SPATIAL * SPATIAL_VAR_PAGESIZE) +
      (MAX_SHARED_SCALARS * SCALAR_VAR_PAGESIZE);
  }

  // address of page for spatial obj n
  static void* s_page(int n)
  {
    return (void*)((byte*)(Env::DataRep()) + 
		   ((long)sizeof(SharedObjectManager) +
		    ((long)n * (long)SPATIAL_VAR_PAGESIZE)));
  }

  // address of page for scalar timeseries obj n
  static void* t_page(int n)
  {
    return (void*)((byte*)(Env::DataRep()) + 
		   ((long)sizeof(SharedObjectManager) +
		    ((long)MAX_SHARED_SPATIAL * (long)SPATIAL_VAR_PAGESIZE) +
		    ((long)n * (long)SCALAR_VAR_PAGESIZE)));
  }

  Bool lock() const { return _lock; } 
  Bool lock(Bool v) {  return _lock = v;   } 

  // called on driver side to update shared memory
  void update(const char* moduleName = NULL, const char* variableName = NULL);

  // inquire functions called by client
  float time() const { return _time; }
  int&  ppid() { return _parent; }
  int&  syncflag() { return _syncflag; }
  
  void list_vars(FILE*, Bool);

  // this creates a slot for a new var to be filled by
  // server (will alloc the memory the first time)
  int register_variable(char* p, Bool s);
  // this munmaps() the space and makes var available
  int unregister_variable(char* p);
  // this modifies configuration options for var - takes
  // same format of register-variable but does not create
  // slot - does nothing if var is wrong
  void configure_variable(char* p);

  // find available slot for pathname (use new one if create == true)
  struct obj_desc* find_spatial(char* p, Bool create) 
  { return find_slot(p, create, 1); }
  struct obj_desc* find_scalar(char* p, Bool create) 
  { return find_slot(p, create, 0); }

  // allocates shared memory for spatial variables
  int alloc_pages();

  SharedObjectManager()  { init();    }
  ~SharedObjectManager() { destroy(); }
};

TVariable* SharedObjectManager::get_variable(char* path) 
{
  PathName p(path);
  return TModel::I().GetVariable((char*)p.mod(), (char*)p.var());
}

struct obj_desc* SharedObjectManager::find_slot(char* p, Bool create,
						Bool spatial)
{
  struct obj_desc* a = spatial ? _spatial : _scalars;
  int max = spatial ? MAX_SHARED_SPATIAL : MAX_SHARED_SCALARS;

  int pp = strlen(p)-1;
  while (isspace(p[pp]) && pp)
    p[pp--] = '\0';

  int i, ffree = -1;
  for (i = 0; i < max; i++)
  {
    struct obj_desc* o = a + i;
    if (create && ffree == -1 && o->_path[0] == '\0')
      ffree = i;
    if (strcmp(o->_path, p) == 0)
      break;
  }

  if (i < max)
    {
      if( dfp != NULL) {
	struct obj_desc* o = a + i;
	fprintf(dfp,"\nGot Slot %x (%d:%d): %s (matched: %s)",o,i,max,o->_path,p); 
      } 
      return &a[i];
    }
  else if (create && ffree > -1)
    {
      if( dfp != NULL) { fprintf(dfp,"\nCreate Slot %d: %s",ffree,p); } 
      strcpy(a[ffree]._path, p);
      a[ffree]._state = VAR_UNSET;
      return &a[ffree];
    }
  else return NULL;
}

void SharedObjectManager::update(const char* moduleName, 
				 const char* variableName)
{

  static FILE* fp = NULL;
  if( fp == NULL ) { 
		CPathString pathName(Env::ArchivePath()); 
    pathName.Add("CME.log");
 		fp = fopen(pathName,"w"); 
	}
  fprintf(fp,"\nCalling update for %s::%s",moduleName,variableName);  fflush(fp);
  
  // client CANNOT call this
  int c_factor = 1, i;
  
  syncflag() = 0; 
  while (lock()) sleep_u(10000);
  lock(TRUE);
  
  _time = TTime::Current();
  
  // update all variables: scalars
  for (i = 0; i < MAX_SHARED_SCALARS; i++)
    {
      struct obj_desc* o = _scalars + i; 
      
      if (o->_path[0] != '\0')
	{
	  // check pathname
	  PathName p(o->_path);
	  if (moduleName != NULL && strcmp(p.mod(), moduleName) != 0) 
	    continue;
	  if (variableName != NULL && strcmp(p.var(), variableName) != 0) 
	    continue;
	  
	  // get/set value;
	  TVariable* v = get_variable(o->_path);
	  if (v && o->_state != VAR_ERROR)
	    {
	      o->_idx = (unsigned)i;
	      
	      // TBI assumes 'f' format for scalar vars
	      var_desc* sd = (var_desc*)(t_page(i));
	      sd->dSize = v->DataSize();

	      // start of data in shared memory
	      float* d = (float*)((byte*)t_page(i) + sizeof(struct var_desc));
	      // determine min/max 
	      sd->_min = 100000.0; sd->_max = -100000.0;
	      sd->xSize = 2;             // number of columns
	      sd->ySize = v->DataSize(); // number of rows
	      int k = 0;
	      for (int l = 0; l < v->DataSize(); l++)
		{
		  if (v->Data()[l] > sd->_max) sd->_max = v->Data()[l];
		  if (v->Data()[l] < sd->_min) sd->_min = v->Data()[l];
		  d[k++] = (float)l
		    /* CHANGE TO THIS WHEN IMPLEMENTED  v->TimeData()[l] */
		    ;
		  d[k++] = v->Data()[l];
		}
	      
	      if (fp != NULL)
		{
		  float ftmp = v->Value();
		  o->_val = ftmp;
		  o->_state = VAR_SET;
		  fprintf(fp," (%f) -> %f ",_time,ftmp);  fflush(fp);
		}
	    }
	}
    }
  // spatials
  for ( i = 0; i < MAX_SHARED_SPATIAL; i++)
    {
      struct obj_desc* o = _spatial + i; 
      if (o->_path[0] != '\0')
	{
	  // check pathname
	  PathName p(o->_path);
	  if (moduleName != NULL && strcmp(p.mod(), moduleName) != 0)
	    continue;
	  if (variableName != NULL && strcmp(p.var(), variableName) != 0)
	    continue;
	  
	  // to avoid sending the wrong variables when that 'continue' is executed
	  var_desc* sd = (var_desc*)(s_page(i));
	  sd->xSize = 0;
	  sd->ySize = 0;
	  sd->dSize = 0l;
	  
	  TVariable* v = (TVariable*)get_variable(o->_path);

	  fprintf(fp,"\nspatial update: ");   
	  if (v && !(v->GetF(FisSpatial))) continue;
	  Region2* vreg = NULL;
	  MultiCoverage* vcov = NULL;
		CVariable* sv = (CVariable*)v;
		if (sv && !sv->CheckMemory() ) continue;
		vreg = &sv->Region();
		vcov = &sv->Cov();
	  fprintf(fp,"eligible");
	  
	  if (v && o->_state != VAR_ERROR)
	    {	      
	      o->_idx    = (unsigned)i;
	      
	      // TBI assumes 'B' format for spatial vars
	      var_desc* sd = (var_desc*)(s_page(i));
	      sd->xSize = vreg->extents(1);
	      sd->ySize = vreg->extents(0);
	      unsigned long ds = sd->xSize * sd->ySize * o->_bsz;
	      // determine min/max 
	      sd->_min = 100000.0; sd->_max = -100000.0;
	      for (int l = 0; l < v->DataSize(); l++) {
		if (v->Data()[l] > sd->_max) sd->_max = v->Data()[l];
		if (v->Data()[l] < sd->_min) sd->_min = v->Data()[l];
	      }
	      fprintf(fp,"\ndatasize = %d, Max = %f, Min = %f",v->DataSize(),sd->_max,sd->_min);  fflush(fp);
	      // copy to temporary map: pointer is only valid in 
	      // parent process
	      Region2 r(*vreg);
	      
	      // redefine region as needed
	      if (o->_x1 != 0 || o->_y1 != 0 || o->_x2 != 0 || o->_y2 != 0) 	
		r.setregion(o->_x1, o->_y1, o->_x2, o->_y2, 1, 1);
	      
	      // downsampling according to o->_downsample
	      int dsf = (int)(1.0/o->_downsample);
	      if (dsf != 1) r.setincrement(dsf, dsf);
	      
	      TMap* m = o->_map = new TMap(&r, (int)o->_bsz);
	      
	      // scale and offset are dealt with directly by interface
	      m->SetByteConversion(1.0, 0.0, (int)o->_bsz);
	      
	      // clip and rescale if needed
	      m->ClearBounds();
	      if (o->_umin != 0.0 || o->_umax != 0.0 || o->_resc) 	
		m->SetBounds(sd->_min, sd->_max, o->_umin, o->_umax, o->_resc);
	      
	      // use coordinate swapping
	      vcov->CopyToMap(*m, TRUE);
	      
	      sd->dSize = (unsigned long)(m->BSize());	
	      
	      if (o->_format == 'B')  	{
		// encode bytestream
		RLEncoder rle(m->Data(), (unsigned long)sd->dSize, o->_bsz, Env::SockBufsize());
		
		// start of data in shared memory
		byte* b = (byte*)s_page(i) + sizeof(struct var_desc);
		// encode into buffer
		rle.encode(b);
	      } 
	      else if (o->_format == 'b')
		{      
		  memcpy((byte*)s_page(i) + sizeof(struct var_desc),
			 m->Data(), (size_t)(sd->dSize*sizeof(byte)));
		}
	      else if (o->_format == 's')
		{      
		  memcpy((short*)s_page(i) + sizeof(struct var_desc),
			 m->Data(), (size_t)(sd->dSize*sizeof(short)));
		}
	      else if (o->_format == 'l')
		{      
		  memcpy((long*)s_page(i) + sizeof(struct var_desc),
			 m->Data(), (size_t)(sd->dSize*sizeof(long)));
		}
	      else if (o->_format == 'f')
		{      
		  memcpy((float*)s_page(i) + sizeof(struct var_desc),
			 m->Data(), (size_t)(sd->dSize*sizeof(float)));
		}
	      
	      delete o->_map;
	      o->_map = NULL;
	      
	      // return
	      o->_state = VAR_SET;
	    }
	}
    }
  lock(FALSE);
  // **DBG** fclose(fp);
}

int SharedObjectManager::register_variable(char* p, Bool spatial)
{
  // registers variable for updating; defines bytesize (def 1)
  // parse line and seek slot
  while (lock()) sleep(1); lock(TRUE);
  
  char c, f, path[256]; int b, resc, x1, x2, y1, y2; float umin, umax, dwn;
  int it = sscanf(p,"%c %s %c %d %f %f %d %d %d %d %d %g",
		  &c, path, &f, &b, &umin, &umax, &resc, 
		  &x1, &y1, &x2, &y2, &dwn);

  if (it < 2) return 0;

  int pp = strlen(path)-1;
  while (isspace(path[pp]) && pp)
    path[pp--] = '\0';
  struct obj_desc* slot = NULL;
  if (spatial) slot = find_spatial(path, TRUE);
  else slot = find_scalar(path, TRUE);
  if (slot != NULL) 
  {
    if (it > 2)  slot->_format     = f;
    if (it > 3)  slot->_bsz        = (unsigned)b;
    if (it > 4)  slot->_umin       = umin;
    if (it > 5)  slot->_umax       = umax;
    if (it > 6)  slot->_resc       = (Bool)resc;
    if (it > 7)  slot->_x1         = (unsigned)x1;
    if (it > 8)  slot->_y1         = (unsigned)y1;
    if (it > 9)  slot->_x2         = (unsigned)x2;
    if (it > 10) slot->_y2         = (unsigned)y2;
    if (it > 11) slot->_downsample = dwn;
    if( dfp != NULL) {
      fprintf(dfp,"\nRegister Variable %s in slot %x",path,slot); 
    }   
  }
  lock(FALSE);

  // if sync enabled, try updating the value right now
  if (slot != NULL && Env::Sync())
    {
      syncflag() = 1;
      kill(ppid(), SIGCONT);
    }
  
  return slot != NULL;
}

void SharedObjectManager::configure_variable(char* p)
{
  // modifies var options without raising errors on wrong var
  while (lock()) sleep(1); lock(TRUE);
  
  char c, f, path[256]; int b, resc, x1, x2, y1, y2; float umin, umax, dwn;

  int it = sscanf(p,"%c %s %c %d %f %f %d %d %d %d %d %f",
		  &c, path, &f, &b, &umin, &umax, &resc, 
		  &x1, &y1, &x2, &y2, &dwn);

  if (it < 2) return;
  int pp = strlen(path)-1;
  while (isspace(path[pp]) && pp)
    path[pp--] = '\0';

  struct obj_desc* slot = find_spatial(path, FALSE);
  if (slot == NULL)
      slot = find_scalar(path, FALSE);

  if (slot) 
  {
    if (it > 2) slot->_format  = f;
    if (it > 3) slot->_bsz     = (unsigned)b;
    if (it > 4) slot->_umin    = umin;
    if (it > 5) slot->_umax    = umax;
    if (it > 6) slot->_resc    = (Bool)resc;
    if (it > 7) slot->_x1      = (unsigned)x1;
    if (it > 8) slot->_y1      = (unsigned)y1;
    if (it > 9) slot->_x2      = (unsigned)x2;
    if (it > 10) slot->_y2     = (unsigned)y2;
    if (it > 11) slot->_downsample = dwn;
  }
  lock(FALSE);
  // if sync enabled, try updating the value right now
  if (slot != NULL && Env::Sync())
  {
    syncflag() = 1;
    kill(ppid(), SIGCONT);  
  }    
}

int SharedObjectManager::unregister_variable(char* p)
{
  while (lock()) sleep(1); lock(TRUE);
  struct obj_desc* slot = find_spatial(p, FALSE);
  if (slot == NULL) slot = find_scalar(p, FALSE);
  if (slot) 
  {
    slot->_path[0] = '\0';
    slot->_state   = VAR_UNSET;
    if (slot->_map != NULL)
    {
      delete slot->_map;
      slot->_map = NULL;
    }
    if( dfp != NULL) {      
      fprintf(dfp,"\nUnRegister Variable %s in slot %x",p,slot); 
    }   
  }
  lock(FALSE);
  return slot != NULL;
}

void SharedObjectManager::list_vars(FILE* o, Bool s)
{
  struct obj_desc* a = s ? _spatial : _scalars;
  int max = s ? MAX_SHARED_SPATIAL : MAX_SHARED_SCALARS;

  while (lock()) sleep(1); lock(TRUE);

  for (int i = 0; i < max; i++)
    {
      struct obj_desc* op = a + i;
      if (op->_path[0] != '\0')
	{
	  fprintf(o, "slot %d: %s: state %d\n", i, op->_path, op->_state);
	  if (s)
	    {
	      struct var_desc* sp = 
		(struct var_desc*)s_page(op->_idx);
	      fprintf(o, "pars: %lu %u %u %g %g\n", sp->dSize,  sp->xSize,
		      sp->ySize, sp->_min, sp->_max);
	    }
	}
    }
  fflush(o);
  lock(FALSE);
}

// a single OM is mapped into shared memory; every byte is zero at
// startup
static SharedObjectManager& OM()
{
  return *(SharedObjectManager*)(Env::DataRep());
}

#endif

// -- server code -------------------------------------------------------

void DataServer(const char* moduleName, const char* variableName)
{
#ifdef HAS_SOCKETS
  SharedObjectManager* om = (SharedObjectManager*)(Env::DataRep());
  if(om) om->update(moduleName,variableName);
#endif
}

// sig handler for child termination
void sigchld_handler(int dummy)
{ 
#ifdef HAS_SOCKETS
  OM().lock(FALSE); 
  while (wait3(NULL, WNOHANG, NULL) > 0)
    ; 
  signal(SIGCHLD, sigchld_handler);
#endif
}

// sig handler for updating
void sigusr1_handler(int dummy)
{
#ifdef HAS_SOCKETS 
  if (OM().time() > 0.0) 
    OM().update(); 
#endif
}

int SS_InitializeNetwork(const char* host, const char* timeout, const char* bufsize, const char* datasize, const char* sync)
{
#ifdef HAS_SOCKETS 
  Env::Host() = host;
  // timeout (not used actually)
  int to = atoi(timeout);
  // socket buffer
  int bs = atoi(bufsize);
  // size of shared memory data swap area
  size_t dsize = (size_t)atol(datasize);
  if (dsize > 0)
    Env::DataSize() = dsize;
  // if TRUE driver receives SIGCONT after each registration or
  // configuration of variables, to update shared memory before
  // next time unit
  Env::Sync() = atoi(sync);

  if (to > 0) Env::Timeout() = to;
  if (bs > 0) Env::SockBufsize() = bs;
  // create and bind a TCP/INET socket
  // peer (interface) registers localhost, localport for connecting
  Env::Socket();

  // create shared memory space for communicating data
  // between child (server) and parent (driver)
  int f;
  if ((f = open("/dev/zero", O_RDWR)) == -1)
    gFatal("can't access /dev/zero");
  
  // create shared object manager/ data by mapping /dev/zero in shared memory
  void* v = mmap(NULL, SharedObjectManager::shmem_size(), 
		 PROT_READ|PROT_WRITE, MAP_SHARED, f, 0);

  if (v == NULL)
    gFatal("can't map %lu bytes of shared memory", 
	   (unsigned long)SharedObjectManager::shmem_size());
  else /* if (SS_GoToDebug) */
    printf("mapped %lu bytes of shared memory onto /dev/zero\n"
           "Update mode is %s\n",
	   (unsigned long)SharedObjectManager::shmem_size(),
      Env::Sync() ? "syncronous" : "asyncronous");    
  close(f);
  Env::DataRep(v);

  // put driver process ID in shared memory
  OM().ppid() = getpid();

  // setup child cleanup from signal
  signal(SIGCHLD, sigchld_handler);
  // SIGUSR1 to parent causes updating of shared memory
  // signal(SIGCONT, sigusr1_handler);

  // fork server
  if (Env::Port())
  {
    if (!fork())
    {
      DispatchData();
      exit(100);
    } else {
      // parent: 
    }
  }
#endif
	return 0;
}

int SS_InitializeNetworkJava( const char* host, const char* port, const char* debug )
{
#ifdef JAVA
  int iport = atoi(port);
  int idebug = atoi(debug);
  JavaSocketReader::initializeSocket( host, iport, idebug );
#endif
	return 0;
}

void SS_WaitForJavaConnection() {
#ifdef JAVA
  if( JavaSocketReader::connect1() ) {
	  Env::UseJavaSocket() = TRUE;
	  Env::SetOutputStream( JavaSocketReader::OStream() );
	  Env::SetErrorStream(  JavaSocketReader::OStream() );
  }
#endif
}

// -- child code: manage connections ------------------------------------- 


// DispatchData(): called upon client connect() - serves data
// to client (on socket)
void DispatchData()
{
#ifdef HAS_SOCKETS 
  struct sockaddr_in sin;
  int s;
	int len;
  
  for (;;)
  {
    if ((s = accept(Env::Socket(), (sockaddr*)&sin, (socklen_t*) &len)) >= 0)  
    {
      // spawn child to handle request
      if (!fork())
      {
        DataClient(s);
        exit(101);  
      }
      close(s);
    }
  }
#endif
}


// ---------------------------------------------------------------------
// dataclient returns data packets for any variable in this format:
//
// 1) header (ASCII) with the following info in a single row:
//    data format (a single letter, either b (byte) s (short) l (long) or
//                 f (float); letter is uppercase if RLE compression is used)
//    number of bytes sent (has no meaning if RLE is not used) (long integer)
//    number of rows (integer)
//    number of columns (integer)
//    min value (float)
//    max value (float)
//    newline
//
// 2) if RLE is not used, numbers are sent one by one in row-first
// order, one for each row, in ASCII. If RLE is used, the output is <n>
// chunks of binary data, each prefixed by the chunk size in ASCII on a
// single line. The end of the data is marked by a 0 (in ASCII) as the
// chunk size. RLE data are encoded using RLEncoder and are supposed
// to be processed by RLEdecoder (in table.C in <CME>/src/generic right
// now).
// 
// Right now only single byte data are sent RLE encoded. All of the
// others are sent as ASCII values.Scalar variables are sent as ASCII
// 2 by N tables of floats, each row reporting <time> <value> of each
// point in the current timeseries.
// ---------------------------------------------------------------------

void DataClient(int s)
{
#ifdef HAS_SOCKETS 
  // called by fork()'ed proc to ask parent for data
  // use buffered I/O on socket for convenience
  FILE* in  = fdopen(s,"r");
  FILE* out = fdopen(s,"w");
  
  if( dfp == NULL ) { 
    // dfp = fopen("dataclient.log","a"); 
  }
  
  char buf[256];
  
  byte* b = (byte*)Env::DataRep();
  int go = 1;
  
  while (go && fgets(buf, 256, in))
    {
      go = !feof(in);
      
      if (*buf ==  '.')
	{
	  switch(buf[1])
	    {
	    case 'T':  // get current time (.T)
	      fprintf(out,"%g\n", (double)OM().time());
	      break;
	      // put other admin commands here
	    case 's': // list registered spatial variables (.s)
	      OM().list_vars(out, TRUE);
	      break;
	    case 'r': // list registered scalars (.r)
	      OM().list_vars(out, FALSE);
	      break;
	    case 'q': // quit (.q)
	      go = 0;
	      break;
	    }
	}
      else {
	if (buf[0] == 's' && buf[1] == ' ')
	  // register spatial variable command
	  fprintf(out, "%d\n", OM().register_variable(buf,TRUE));
	else if (buf[0] == 'r' && buf[1] == ' ')
	  // register scalar variable command
	  fprintf(out, "%d\n", OM().register_variable(buf,FALSE));
	else if (buf[0] == 'c' && buf[1] == ' ')
	  // configure variable command
	  OM().configure_variable(buf);
	else if (buf[0] == 'u' && buf[1] == ' ') 
	  {
	  // unregister variable command
	  fprintf(out, "%d\n", OM().unregister_variable(&buf[2]));
	  } 
	else 
	  {
	    // name of variable: return data
	    Bool spatial = TRUE;
	    struct obj_desc* o = OM().find_spatial(buf, FALSE);
	    if (o == NULL) {
	      o = OM().find_scalar(buf, FALSE); 
	      spatial = FALSE;
	    }
	    
	    if (o == NULL)
	      fprintf(out, "X\n");
	    else
	      {
		// data descriptor
		struct var_desc* vd = spatial ? 
		  (struct var_desc*)OM().s_page(o->_idx):
		  (struct var_desc*)OM().t_page(o->_idx);

		// send header with format, size and extent info
		fprintf(out, "%c %lu %u %u %g %g\n", o->_format, vd->dSize,
			vd->xSize, vd->ySize, vd->_min, vd->_max);

		// send data: only B and f formats are currently
		// supported
		while (OM().lock()) sleep(1);
		OM().lock(TRUE);

		// start of data buffer
		byte* base = (byte*)vd + sizeof(struct var_desc);
		  
		switch(o->_format)
		  {

		    long i, n;
		    float* fdata;
		    byte* bdata;
		    short* sdata;
		    long* ldata;

		  case 'B':
		    // send RL encoded byte stream
		    for (;;)
		      {
			// read chunk size at start of chunk
			unsigned cs;
			memcpy(&cs, base, sizeof(unsigned));
			fprintf(out, "%u\n", cs); 
			fflush(out);
			if (cs == 0)
			  break;
			// write chunk
			fwrite(base, 1, (int)cs, out);
			fflush(out);
			base += (int)cs;
		      }
		    break;
		  case 'f':
		    // send ASCII data one by one
		    fdata = (float*)base;
		    n = vd->xSize * vd->ySize;
		    for (i = 0; i < n ; i ++)
		      fprintf(out, "%g\n", fdata[i]);
		    fflush(out);
		    break;
		  case 'b':
		    // send ASCII data one by one
		    bdata = (byte*)base;
		    n = vd->xSize * vd->ySize;
		    for (i = 0; i < n ; i ++)
		      fprintf(out, "%d\n", (int)bdata[i]);
		    fflush(out);
		    break;
		  case 's':
		    // send ASCII data one by one
		    sdata = (short*)base;
		    n = vd->xSize * vd->ySize;
		    for (i = 0; i < n ; i ++)
		      fprintf(out, "%d\n", sdata[i]);
		    fflush(out);
		    break;
		  case 'l':
		    // send ASCII data one by one
		    ldata = (long*)base;
		    n = vd->xSize * vd->ySize;
		    for (i = 0; i < n ; i ++)
		      fprintf(out, "%ld\n", ldata[i]);
		    fflush(out);
		    break;
		  }
		o->_state = VAR_SET;
		OM().lock(FALSE);
	      }
	  }
      }
      fflush(out);
    } // while
#endif
}


