//----------------------------------------------------------------------------------------
// Model.cc
// Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//----------------------------------------------------------------------------------------

#include "viewserver.h"
#include "SSModel.h"
#include "Environ.h"
#include "HDFMgr.h"
#include "SSModule.h"
#include "CVariable.h"
#include "UserCode.h"
#include "MML_Pipe.h"
#include "FrameLink.h"
#include "Externals.h"
#include "GOFCalc.h"
#include <unistd.h>

extern void gInitializeModel();
extern void gSetupExternals();

//========================================================================================
// 				CLASS TModel
//========================================================================================


int TModel::WriteSetupFile( char* cnfgExtension ) {
	int rv = 1, i;
  CPathString path(Env::ConfigPath());
  path.Add(Env::ModelName()); 
  FILE* sf = fopen(path.Path(cnfgExtension),"w");
  if( sf ) {
		fprintf(sf,"config input File names = ( ");
		for( i=0; i<fConfigFileList.NArgs(); i++ ) {
			if( i > 0 ) { fprintf(sf,","); }
			fprintf(sf," %s",fConfigFileList[i].chars());
		}
		fprintf(sf," ); ");
		fprintf(sf,"\nmml input File names = ( ");
		for( i=0; i<fMMLFileList.NArgs(); i++ ) {
			if( i > 0 ) { fprintf(sf,","); }
			fprintf(sf," %s",fMMLFileList[i].chars());
		}
		fprintf(sf," ); ");
		fclose(sf);
	} else {
		sprintf(gMsgStr,"Can't generate setup file %s",path.chars());
		gPrintErr();
		rv = 0; 
	} 
  return rv;
}

void TModel::Startup()  {	

	ExprBuilder::SetMode(ExprBuilder::kCCode);
	int readDBParamValues = 0;
	int useDefaultConfig = 1;
	int Argc = Env::Argc();
	for( int i=0; i< Argc; i++ ) {
		if( Env::Argv(i) == "-mop" ) { 
			fMemoryOp = 1;
		} else 	if( Env::Argv(i) == "-ignore_STELLA_arrays" ) { 
			readDBParamValues = 0;
		} else if( Env::Argv(i) == "-mi" ) { 
			if ((i+1) < Argc) fMMLFileList.Add( Env::Argv(++i) ); 
		} else if ( Env::Argv(i) == "-ci" ) {
			if ((i+1) < Argc) fConfigFileList.Add( Env::Argv(++i) ); 
			useDefaultConfig = 0;
    } else if (Env::Argv(i) == "-help") {
      gPrintScreen("Driver command line Arguments:");
      gPrintScreen(" -mi FILE				Use FILE as MML input file");
      gPrintScreen(" -ci FILE				Use FILE as configuration input file  (default: <ProjDir>/Config/<ModelName>.config.new)");
      gPrintScreen(" -mop				 		Use memory optimization");
      exit(-10);
    }  
	}
	
	gInitializeModel();
	DBaseObjectFileRW('r');

  CPathString cpath(Env::ModelPath());
	cpath.Add("UserCode");  
	cpath.Add("SQLFunctions.MML");  
	ExprBuilder::FreeNodes();
	if( ParseMMLFile( cpath ) ) {
		sprintf(gMsgStr,"Processing SQLFunctions file: %s",cpath.chars());
		ProcessNode(ExprBuilder::GetRootNode());
	} else {
		sprintf(gMsgStr,"Can't open SQLFunctions file: %s: Using defaults.",cpath.chars());
	}	
	if( gDebug > 1 ) {
		gPrintScreen();	
	}
		  
  if( gInterface || !Env::ReadConf()) { ; }
  else {
	if(useDefaultConfig) { 
	  CString config(Env::ModelName()); ( ((config += ".") += Env::ScenarioName() ) += ".config" );
	  fConfigFileList.Add(config); 	  
	}				
	gPrintScreen("Reading Config Files"); 
    ReadConfigFiles( );
  }
  
	if( readDBParamValues ) { 
		ReadDBParamValues("DBaseParameters.out"); 
	}
	
	InitArrays();
}

int TModel::addStateVariable( const Variable& var ) { 
   for( int i=0; i<fNIVars; i++ ) {
	 if( fIVars[i] == &var ) return i;
   }
   fIVars[fNIVars++] = (TVariable*) &var;
   return fNIVars-1;
}

void TModel::InitArrays() { 
	for( Pix p = fModuleList.first(); p; fModuleList.next(p) ) {
	  Module& aModule = (Module&) fModuleList(p);
	  aModule.InitArrays();
	}
}

//----------------------------------------------------------------------------------------
//				TModel::PostExtVarUpdateCommands:
//----------------------------------------------------------------------------------------

void TModel::PostExtVarUpdateCommands() {

  Module* gm  = GlobalMod();                             // Post at beginning of init event.
  if( gm == NULL ) { gFatal( "No Root Module Specified." ); }
  CString cName; TExternalsUpdateCommand* c;
  
  if( Externals::has_Externals( EVarLink::kInitIn ) ) {
	cName = "ExternalsUpdate-InitIn";
	TEvent* e = gm->GetEvent("StateVarInit",TEvent::kInit);
	c = new TExternalsUpdateCommand( cName, EVarLink::kInitIn );
	e->AddCommand( *c,TEvent::kBeginning);
  }
  
  if( Externals::has_Externals( EVarLink::kInitOut ) ) {
	cName = "ExternalsUpdate-InitOut";
	TEvent* e = gm->GetEvent("StateVarInit__S__",TEvent::kInit);
	c = new TExternalsUpdateCommand( cName, EVarLink::kInitOut );
	e->AddCommand( *c,TEvent::kEnd);
  }

  if( Model::I0().GetCInfo(Model::kOptLevel) >= 5) { 
	  TEvent* e = gm->GetEvent("GlobalUpdate",TEvent::kGlobalUpdate);
	  	  
	  if( Externals::has_Externals( EVarLink::kIn ) ) {
		cName = "ExternalsUpdate-In";
		c = new TExternalsUpdateCommand( cName, EVarLink::kIn );
		e->AddCommand( *c,TEvent::kBeginning);
	  }
	  
	  if( Externals::has_Externals( EVarLink::kOut ) ) {
		cName = "ExternalsUpdate-Out";
		c = new TExternalsUpdateCommand( cName, EVarLink::kOut );
		e->AddCommand( *c,TEvent::kEnd);
	  }
	  
  } else {
//	int hasInitIn = 0, hasInitOut=0; 
	for( Pix p = fModuleList.first(); p; fModuleList.next(p) ) {
	  Module& aModule = (Module&) fModuleList(p);

	  if( Externals::has_Externals( EVarLink::kIn, &aModule ) ) {
		cName = "ExternalsUpdate-In";
		c = new TExternalsUpdateCommand( cName, EVarLink::kIn );
		TEvent* pe = aModule.GetEvent("InputPipes",TEvent::kFirstUpdate);	  
		pe->AddCommand( *c );
		pe->Schedule(TEvent::kBeginning);
/*
		if( hasInitIn == 0 ) {
			c = new TExternalsUpdateCommand( cName, EVarLink::kInitIn );
			TEvent* e = gm->GetEvent("StateVarInit",TEvent::kInit);
			e->AddCommand( *c, TEvent::kBeginning );
			hasInitIn = 1;
		}
*/
		sprintf(gMsgStr,"\nPosting ExternalsUpdate-In: %s:%s to event %s",Name(),c->Name(),pe->Name()); gPrintLog();
	  }
	}

	if( Externals::has_Externals( EVarLink::kOut ) ) {
	  cName = "ExternalsUpdate-Out";

//	  if( hasInitOut == 0 ) {
//		  TEvent* e = gm->GetEvent("StateVarInit__S__",TEvent::kInit);
//		  c = new TExternalsUpdateCommand( cName, EVarLink::kInitOut );
//		  e->AddCommand( *c, TEvent::kEnd );
//		  hasInitOut = 1;
//	  }

	  int hasFinalUpdate = 0;
	  c = new TExternalsUpdateCommand( cName, EVarLink::kOut );
	  for( Pix p = fEventList.last(); p; fEventList.prev(p) ) {
		  TEvent& fe = (TEvent&) fEventList(p);
		  if( (hasFinalUpdate == 0) && (fe.GetObjInfo(TEvent::kType) == TEvent::kFinalUpdate) ) {
			  fe.AddCommand( *c );
			  sprintf(gMsgStr,"\nPosting ExternalsUpdate-Out: %s:%s to event %s",Name(),c->Name(),fe.Name()); gPrintLog();
			  hasFinalUpdate = 1;
		  }
	  }
	}

  }
}


void TModel::InitializeEvents() {  
  gPrintScreen("Setup Events");    
  for(Pix p = fModuleList.first(); p; fModuleList.next(p))  {
    TModule& aModule = (TModule&) fModuleList(p);
    if( &aModule != I().GlobalMod() ) aModule.SetupEvents();
  }
  ((TModule*)(I().GlobalMod()))->SetupEvents();

  CreateEventLists();
  Env::SetInfo(Env::kEventsDefined,1); 
    
  //  UpdateDBaseInCache(); 
  DumpEventList();				
}

void TModel::SetupFrames() {
  static int frameSetup = 0;
  
  if( frameSetup == 0 ) {
    gPrintScreen("Setting Up Frames & Schedules");    
    HDFM::Open(HDFM::kFInput);
//    ((TModule*)(I().GlobalMod()))->SetupFrame();
		TMultiFrame& f = (TMultiFrame&)Frame();
		f.Setup();
    ((TModule*)(I().GlobalMod()))->ConfigureSchedules();
    Pix p;
    for(p = fModuleList.first(); p; fModuleList.next(p))  {
      TModule& aModule = (TModule&)fModuleList(p);
      aModule.SetupDefaults(); 
    }
    gPrintScreen("Allocating Memory");    
    for(p = fModuleList.first(); p; fModuleList.next(p))  {
      TModule& aModule = (TModule&)fModuleList(p);
      aModule.AllocMemory();
    }
   
    //		gSetupExternals();
    frameSetup = 1;
  }
}

void TModel::SetupVariables() {   
	if( gDebug > 0 ) { 
	  gPrintScreen("Setup Variables"); 
	}
	if( GetCInfo(kDebug) ==  kNoConfig ) {  SetCInfo( kDebug, 0 ); }
	for(Pix p = fModuleList.first(); p; fModuleList.next(p))  {
		TModule& aModule = (TModule&)fModuleList(p);
		if( aModule.GetCInfo(Module::kDebug) == kNoConfig ) { 
			aModule.SetCInfo( Module::kDebug, GetCInfo(kDebug) );
		}
		aModule.SetupVariables();
	}
}

void TModel::QueryInterface() { 
	DataServer(fCurrentModule->Name(),fCurrentVariable->Name()); 
}

void TModel::PostEvents() {
  if( gDebug > 0 ) { 
	gPrintScreen("Posting Events"); 
  }
  Pix p;   
  for(p = fModuleList.first(); p; fModuleList.next(p))  {
    TModule& aModule = (TModule&) fModuleList(p);
    if( &aModule != I().GlobalMod() ) aModule.PostEvents();
  }
  ((TModule*)(I().GlobalMod()))->PostEvents();

	TNamedObjectList* pipeList = NULL;
  for(p = fModuleList.first(); p; fModuleList.next(p))  {
    TModule& aModule = (TModule&) fModuleList(p);
    pipeList = aModule.PostCommandsFromConfig();
  }
  if( pipeList ) {
		for( Pix pv = pipeList->last(); pv; pipeList->prev(pv) ) {
			TPipeCommand& pc = (TPipeCommand&) (*pipeList)(pv);
			PostPipe(pc);                                                  // do global level (init) post.
		}
	}
  PostExtVarUpdateCommands();
}

void TModel::Reset() {
  for(Pix p = fModuleList.first(); p; fModuleList.next(p))  {
    TModule& aModule = (TModule&) fModuleList(p);
		aModule.Reset();
  }
}
 
int TModel::InitSimulation() {
#ifdef JAVA
	ViewServer::init_simulation( Env::ProjName(), Env::ModelName(), Env::ArchivePath() ) ;
#endif
	TTime::Update( fSchedule.StartTime() );
	if( Env::GetInfo(Env::kEventsDefined) != 1 ) { 
	  InitializeEvents(); 
	} 
	SetupVariables();
	SetupFrames();
	PostEvents(); 
    CPathString configDataPath(Env::ModelPath()); 
	configDataPath.Add("Models");
    configDataPath.Add(Env::ModelName());
    configDataPath.Add(Env::ScenarioName());
	WriteConfigXML( configDataPath.Path() );  
	return fNSim;
}

void TModel::Shutdown() {
  Finalize();
  HDFM::Close();
  Env::Finalize();
  exit(98);
}


int  TModel::Config( TConfigData& cd) {
  const CString& cmd = cd.Cmd();
  int rv = Model::Config(cd);
  return rv;
}



int  TModel::ConfigExternal( TConfigData& cd, Module* m ) {
  const CString& cmd = cd.Cmd();
	sprintf(gMsgStr,"\nConfigExternal: %s:%s, %x",cd.Name().chars(),cmd.chars(),m); 
	gPrintLog(); 
  if( cd.Name() == "UserCode" ) return UserCode::Config(cd);
  else if( cd.Name() == "FrameLink" ) {
		if( m ) { return m->ConfigExternal(cd,m); } 
		else { return Model::ConfigExternal(cd,m); }
	}
  else return Model::ConfigExternal(cd,m);
}

//----------------------------------------------------------------------------------------
//				TModel::CreateNewModule:
//----------------------------------------------------------------------------------------

Module* TModel::CreateNewModule( const CString& name ) {
  TModule *mod = new TModule(name);
  return mod; 
}

  //----------------------------------------------------------------------------------------
  //				TModel::GetModule:
  //----------------------------------------------------------------------------------------
  
  
Module* TModel::GetModule( const CString& name, Bool create )
{
  Module* thisModule = (Module*) fModuleList.GetObjectByName( name );

  if( thisModule == NULL && create ) {
    thisModule = CreateNewModule(name);						
  }
  return fCurrentModule = thisModule;
  
} // TModule::CreateNewModule

//----------------------------------------------------------------------------------------
// TModel::Run
//----------------------------------------------------------------------------------------

		
void TModel::Run() 
{

  int& iSim = Env::BatchIndex();
  do {
    
    InitSimulation();
    
    RunModel( FLT_MAX );
    
    CloseSimulation(iSim);
    
	} while ( ++iSim < fNSim );

}
void TModel::CloseSimulation(int iSim) {
		ClearEventList();
		SaveDBaseOutCache();
		Reset();
}

int TModel::Open() {
  if( !Env::GetInfo(Env::kSimOpen) ) { 
    InitSimulation(); 
    Env::SetInfo(Env::kSimOpen,1); 
    return 1;
  }
  return 0;
}

void TModel::Close () {
  static int sim_cnt = 0;
  if( Env::GetInfo(Env::kSimOpen)  ) {
	CloseSimulation(sim_cnt++);
  }
  if( !Env::UseMPE() ) { 
	gPrintScreen("Simulation Reset To Initial Time Condition (Spatial Frames Retained).");
  }
  Env::SetInfo(Env::kSimOpen,0);
}

float TModel::GetValue(char* moduleName, char* varName, Pix p) {
  TModule* mod = (TModule*) GetModule(moduleName);
  if(mod==NULL) { gProgramBreak( CString(moduleName) += ":  Can't find module"); return 0.0; }
  TVariable* var =  (TVariable*) mod->GetVariable(varName);
  if(var==NULL) gProgramBreak( CString(varName) += ": Can't find variable");
	return ((CVariable*)var)->Value((TCell*)p);
}

TVariable* TModel::GetVariable(char* moduleName, char* varName) {
  TModule* mod = (TModule*) GetModule(moduleName);
  if(mod==NULL) { gProgramBreak( CString(moduleName) += ":  Can't find module"); return NULL; }
  TVariable* var = (TVariable*) mod->GetVariable(varName);
  if(var==NULL) gProgramBreak( CString(varName) += ": Can't find variable");
  return var;
}

TVariable* TModel::GetVariable( CString& name ) {
  TVariable* var = NULL;
  int mIndex = name.index('.');
  if(  mIndex > 0 ) { 
 	CString moduleName(name.before('.'));
	TModule* mod = (TModule*) GetModule(moduleName);
	if(mod==NULL) { gProgramBreak( CString(moduleName) += ":  Can't find module");  }
	else {
	  CString varName(name.after('.'));
	  var = (TVariable*) mod->GetVariable(varName);
	  if(var==NULL) gProgramBreak( CString(varName) += ": Can't find variable");
	}
  } else { 
	for( Pix p = fModuleList.first(); p; fModuleList.next(p))  {
	  TModule& aModule = (TModule&) fModuleList(p);
	  TVariable* aVar = (TVariable*) aModule.GetVariable(name);
	  if(  aVar != NULL ) {
		 if( aVar->GetF(FisImport) ) {
		   if( var == NULL ) { var = aVar; }
		 } else {
		   var = aVar;
		 }
	  }
	}
  }
  return var;
}

long int TModel::ReadMapFromHDF( TMap* m, const char* dsetName,  const char* variableName, int index ) {
	HDFM::Open(HDFM::kFReadArchive);
  m->DSetName(dsetName);
  m->VariableName(variableName);
  long int rv = m->SDRead(index);
  m->ClearSDS();
  return rv;
}
  
float  TModel::GetValue(char* moduleName, char* varName, unsigned int row, unsigned int col, int layer_index ) {

  TModule* mod = (TModule*) GetModule( moduleName );
  if(mod==NULL) { gProgramBreak( " Can't find module in TModel::GetValue" ); return 0.0; }
  TVariable* var = (TVariable*) mod->GetVariable(varName);
  if(var==NULL) { gProgramBreak( "Undefined variable index."); return 0.0; }
  TLayer* l = TModel::I().Grid()->getCellLayer(layer_index);
	TCell* c = l->getCell(row,col); 
	return ((CVariable*)var)->Value(c);
}

float  TModel::GetValue(byte ivar, unsigned int row, unsigned int col, int layer_index) {
  
  TVariable* var = fIVars[ivar];
  if(var==NULL) { gProgramBreak( "Undefined variable index."); return 0.0; }
	TModule* mod = (TModule*) var->GetModule();
	if(mod==NULL) { gProgramBreak( " Can't find module in TModel::GetValue" ); return 0.0; }
  TLayer* l = TModel::I().Grid()->getCellLayer(layer_index);
	TCell* c = l->getCell(row,col); 
	return ((CVariable*)var)->Value(c);
}

void TModel::SetValue(byte ivar, Pix p, float value) {
  
  TVariable* var = IVar(ivar);
  if(var==NULL) { gProgramBreak( "Undefined variable index."); return; }
	((CVariable*)var)->Update( (TCell*) p, value );
}

void TModel::SetValue(byte ivar, unsigned int row, unsigned int col, float value,  int layer_index) {
    
  TVariable* var = IVar(ivar);
  if(var==NULL) { gProgramBreak( "Undefined variable index."); return; }
	TModule* mod = (TModule*) var->GetModule();
	if(mod==NULL) { gProgramBreak( " Can't find module TModel::SetValue" ); return; }
  TLayer* l = TModel::I().Grid()->getCellLayer(layer_index);
	TCell* c = l->getCell(row,col); 
	((CVariable*)var)->Update( c, value );
}

float TModel::GetValue(byte ivar, Pix p ) {
  
  TVariable* var = IVar(ivar);
  if(var==NULL) { gProgramBreak( "Undefined variable index."); return 0.0; }
	return ((CVariable*)var)->Value( (TCell*) p );
}










