// ------------------------------------------------------------------------
// SSTcl.cc
// fv Jul 30 1997
// TCL support for SME driver
// ------------------------------------------------------------------------

#ifdef USE_TCL

#include "HDFMgr.h"
#include "SSTcl.h"

#include "SimLib.h"
#include "Globals.h"
#include "Encode.h"
#include "SSModel.h"
#include "SSModule.h"
#include "CVariable.h"
#include "PixVec.h"
#include "TPipe.h"
#include "SSComm.h"
#include "Utility.h"

static Tcl_Interp* _interp;
static Module* current_module;
static Variable* current_variable;

// output function for RLE encoder
void add_res (void* p, const char* b) 
{
  Tcl_AppendResult((Tcl_Interp*)p, b, NULL);
}


// -------------------------------------------------------------------------
// Configuration commands are issued directly to objects after 'configure';
// the first argument can be the SME command name or a more descriptive TCL
// word, which is mapped to the command through the table here. If not found
// in the table it's given to SME as is. In parsing the configure command line,
// arguments introduced by a dash after the 'configure' keyword are ignored,
// to allow command lines like the following at different levels of clarity:
//
//  MOD.VAR configure g DB elevation debugmap
//  MOD.VAR configure gridFrame DB elevation debugmap
//  MOD.VAR configure gridFrame -type DB -areaMap elevation -debugMap debmap
// 
// Whoever introduces a new configuration command should add a descriptive 
// entry here.

struct {
  char  cmd_class;
  char* TCL_command;
  char* SME_command;
} conf_translation_table[] = {
  {'a', "nSimulations", "n"},
  {'a', "randomSeed", "s"},
  {'a', "optimization", "op"},
  {'a', "outputStep", "OS"},
  {'a', "schedule", "OT"},
  {'a', "debugLevel", "d"},
  {'m', "gridFrame", "g"},
  {'m', "treeFrame", "t"},
  {'m', "inputDatabase", "db"},
  {'m', "ignore", "i"},
  {'m', "schedule", "OT"},
  {'m', "executeScript", "es"},
  {'m', "debugLevel", "d"},
  {'v', "ignore", "i"},
  {'v', "surfaceMap", "d"},
  {'v', "classMap", "c"},
  {'v', "callFunc", "f"},
  {'v', "timeSeries", "t"},
  {'v', "stateVariable", "s"},
  {'v', "outputStep", "OS"},
  {'v', "schedule", "OT"},
  {'v', "scale", "S"},
  {'v', "pointTimeSeries", "p"},
  {'v', "animation", "A"},
  {'v', "mapOutput", "M"},
  {'v', "ptsOutput", "P"},
  {'v', "window", "W"},
  {'v', "combine", "C"},
  {'v', "graph", "G"},
  {'v', "sheetData", "GS"},
  {'v', "debugLevel", "db"},
  {'v', "parameter", "pm"},
  {'v', "reInitalize", "oi"},
  {'v', "random", "rs"},
  {'v', "bounds", "B"},
  {'v', "reInput", "ip"},
  {'v', "parameterMap", "m1"},
  {'\0', NULL, NULL}             // leave this at the end
};

struct module_info {
  void*  module;
  external_command_table* cmd_table;
  external_config_table*  cnf_table;
};

// support structures and functions

inline const char* descvar(Variable* v, int property, const char* y, 
			   const char* n)
{
  return (v->GetF(property)) ? y : n;
}

Tcl_Interp* SSTcl_Interp() 
{ return _interp; }

extern TConfigData SS_CD;
extern ECnfgLevel SS_Level;

/*
char* format(const char* fmt, ...)
{
  static char __tmp_string[2048];
  
  va_list pars;
  
  va_start(pars, fmt);
  const int tot = vsprintf(__tmp_string, fmt, pars);
  va_end(pars);
  return(__tmp_string);
}
*/

int make_tcl_list_from_pix(Tcl_Interp* i, TNamedObjectList& l, int full_names=0 )
{
  Tcl_ResetResult(i);
  for( Pix p = l.first(); p; l.next(p) ) 
    {
      TNamedObject& obj = (TNamedObject&) l(p);
      if( full_names ) {
		CString name; ((Variable&)obj).FullName(name,'.');
		Tcl_AppendElement(i, (char*)name.chars());
	  } else {
		Tcl_AppendElement(i, (char*)obj.Name());
	  }
    }
  return TCL_OK;
}

// TCL/SME code

int SMETcl_error(Tcl_Interp* i, const char* format, ...)
{
  va_list pars;

  va_start(pars, format);
  const int tot = vsprintf(i->result, format, pars);
  va_end(pars);
  return TCL_ERROR;
}

int SME_SetConfigArgs(Tcl_Interp* i, ECnfgLevel level, int argc, char* argv[])
{

  if (argc < 3) 
    return SMETcl_error (i, "configure command requires at least an argument");

  char cclass = 'v';
  if (level == kSS_Module)
    cclass = 'm';
  else if (level == kSS_Model)
    cclass = 'a';

  SS_Level = level;

  char* cmd = argv[2]; int j;
  
  // look for command in translation list
  for (j = 0;  conf_translation_table[j].SME_command != NULL; j++)
    {
      if (cclass == conf_translation_table[j].cmd_class && 
	  strcmp(cmd, conf_translation_table[j].TCL_command) == 0)
	{
	  cmd = conf_translation_table[j].SME_command;
	  break;
	}
    }

  // printf ("Cmd is %s\n", cmd);

  SS_CD.SetCmd(cmd);
  int ac = 0;
  for (j = 3; j < argc; j++)
    {
      if ((*argv[j] == '-' && argv[j][1] == '\0') || 
	  (*argv[j] == '-' && argv[j][1] >= 'A'))
	continue;

      SS_CD.SetArg(ac++, argv[j]);
      // printf("\targ %d is %s\n", ac, argv[j]);
    }
  return TCL_OK;
}

int SME_VarInfo(Tcl_Interp* interp, Variable* v, int argc, char* argv[])
{
  if (argc < 3)
    return SMETcl_error(interp,
			"usage: info {dependencies|actions|command|min|max|type"
			"|spatial|scalar|constant|dynamic|import|export|flux|"
			"ignored|clamped|state|actionInfo|actions}");
  
  int ret = TCL_OK;
  if (strcmp(argv[2], "dependencies") == 0)
    {
      TCommand* c = NULL;
      if( argc > 3 ) {
	const CString name(argv[3]);
	c = v->GetCommand( name, TCommand::kUndefined, TCommand::kNoEx, False );
      } else {
	c = v->GetDefaultCommand();
      }
      if( c != NULL ) {
	TNamedObjectList& ld = c->DependencyList();
	ret = make_tcl_list_from_pix(interp, ld);
      }
    }
  else if (strcmp(argv[2], "actioninfo") == 0) {
      TCommand* c = NULL;
      if( argc > 3 ) {
				const CString name(argv[3]);
				c = v->GetCommand( name, TCommand::kUndefined, TCommand::kNoEx, False );
      } else {
				c = v->GetDefaultCommand();
      }
      if( c != NULL ) {
				char cmd_info[300];
				c->getCommandInfoTextString( cmd_info, 300 );
				Tcl_SetResult(interp, cmd_info, TCL_STATIC);
      }
    }
  else if (strcmp(argv[2], "actions") == 0)
    {
      TNamedObjectList& ld = v->CommandList();
      ret = make_tcl_list_from_pix(interp, ld);
    }
  else if (strcmp(argv[2], "comment") == 0)
    {   
	  CString& cs = v->GetComment(); 
      Tcl_SetResult(interp, (char*)format("%s", cs.chars() ), TCL_STATIC);
    }
  else if (strcmp(argv[2], "doc") == 0)
    {   
	  CString& cs = v->GetDoc(); 
      Tcl_SetResult(interp, (char*)format("%s", cs.chars() ), TCL_STATIC);
    }
  else if (strcmp(argv[2], "pmax") == 0)
    {   
	  TParmRec& prec = v->getParmRec(); 
	  float pmax = prec.Max_Value();
      Tcl_SetResult(interp, (char*)format("%g", pmax), TCL_STATIC);
    }
  else if (strcmp(argv[2], "pmin") == 0)
    {   
	  TParmRec& prec = v->getParmRec(); 
	  float pmin = prec.Min_Value();
      Tcl_SetResult(interp, (char*)format("%g", pmin), TCL_STATIC);
    }
  else if (strcmp(argv[2], "pval") == 0)
    {   
	  float pval = v->Value();
      Tcl_SetResult(interp, (char*)format("%g", pval), TCL_STATIC);
    }
  else if (strcmp(argv[2], "max") == 0)
    {
      int size;
      const float* data = SS_GetCurrentData(&size);
      float max = data[0];
      for (int l = 0; l < size; l++)
	if (data[l] > max) max = data[l];
      Tcl_SetResult(interp, (char*)format("%g", max), TCL_STATIC);
    }
  else if (strcmp(argv[2], "min") == 0)
    {
      int size;
      const float* data = SS_GetCurrentData(&size);
      float min = data[0];
      for (int l = 0; l < size; l++)
	if (data[l] < min) min = data[l];
      Tcl_SetResult(interp, (char*)format("%g", min), TCL_STATIC);
    }
  else if (strcmp(argv[2], "size") == 0) {
	if (!(v->GetF(FisSpatial))) {
	  Tcl_SetResult(interp, format("%d", v->DataSize()), TCL_STATIC);
	} else {
	  Region2* vreg = NULL;
	  int cell_layer_index = -1;
	  CVariable* sv = (CVariable*)v;
	  MultiGrid* mg = sv->GetMultiFrame()->Grid();
	  if( mg != NULL ) {
		TPartition2* p2 = mg->GetPartition(cell_layer_index);
		if( p2 != NULL ) {
		  ByteGrid* bg = p2->cellDistribution();
		  if (bg != NULL) {
			Tcl_SetResult(interp,  format("%d %d %d", bg->extents(0), bg->extents(1), mg->nCellLayers() ),  TCL_STATIC);
		  }
		}
	  }
	}
  }
  else if (strcmp(argv[2], "type") == 0)
    Tcl_SetResult(interp, (char*)descvar(v, FisSpatial, "spatial", "scalar"), 
		  TCL_STATIC);
  else if (strcmp(argv[2], "clamped") == 0)
    Tcl_SetResult(interp, (char*)descvar(v, FisClamped, "1", "0"), TCL_STATIC);
  else if (strcmp(argv[2], "ignored") == 0)
    Tcl_SetResult(interp, (char*)descvar(v, FisIgnored, "1", "0"), TCL_STATIC);
  else if (strcmp(argv[2], "spatial") == 0)
    Tcl_SetResult(interp, (char*)descvar(v, FisSpatial, "1", "0"), TCL_STATIC);
  else if (strcmp(argv[2], "scalar") == 0)
    Tcl_SetResult(interp, (char*)descvar(v, FisSpatial, "0", "1"), TCL_STATIC);
  else if (strcmp(argv[2], "import") == 0)
    Tcl_SetResult(interp, (char*)descvar(v, FisImport, "1", "0"), TCL_STATIC);
  else if (strcmp(argv[2], "export") == 0)
    Tcl_SetResult(interp, (char*)descvar(v, FisExport, "1", "0"), TCL_STATIC);
  else if (strcmp(argv[2], "flux") == 0)
    Tcl_SetResult(interp, (char*)descvar(v, FisFlux, "1", "0"), TCL_STATIC);
  else if (strcmp(argv[2], "constant") == 0)
    Tcl_SetResult(interp, (char*)descvar(v, FisConstant, "1", "0"), 
		  TCL_STATIC);
  else if (strcmp(argv[2], "dynamic") == 0)
    Tcl_SetResult(interp, (char*)descvar(v, FisConstant, "0", "1"), 
		  TCL_STATIC);
  else if (strcmp(argv[2], "state") == 0)
    Tcl_SetResult(interp, (char*)descvar(v, FisStateVar, "1", "0"), 
		  TCL_STATIC);
  else if (strcmp(argv[2], "all") == 0)
    {
      /* returns list: spatial constant import export flux ignored 
	 clamped state */
      Tcl_ResetResult(interp);
      Tcl_AppendElement(interp, (char*)descvar(v, FisSpatial, "1", "0"));
      Tcl_AppendElement(interp, (char*)descvar(v, FisConstant, "1", "0"));
      Tcl_AppendElement(interp, (char*)descvar(v, FisImport, "1", "0"));
      Tcl_AppendElement(interp, (char*)descvar(v, FisExport, "1", "0"));
      Tcl_AppendElement(interp, (char*)descvar(v, FisFlux, "1", "0"));
      Tcl_AppendElement(interp, (char*)descvar(v, FisIgnored, "1", "0"));
      Tcl_AppendElement(interp, (char*)descvar(v, FisClamped, "1", "0"));
      Tcl_AppendElement(interp, (char*)descvar(v, FisStateVar, "1", "0"));
    }
  else 
    ret = SMETcl_error(interp,
		       "usage: info {dependencies|min|max|type|spatial"
		       "|scalar|constant|dynamic|import|export|flux|"
		       "ignored|clamped|state}");
  return ret;
}

TMap* SME_Var2Map(Variable* v, int argc, char* argv[], float& min, float& max) 
{
  // returns a TMap2 made from v according to switches supplied,
  // turns processed switches into empty strings

  if( v == NULL ) 
    { gPrintErr("Undefined varialbe in SME_Var2Map"); return NULL; }
  CVariable* sv = (CVariable*)v;
  if( !sv->GetF(FisSpatial) ) 
    { gPrintErr("NonSpatial Variable in SME_Var2Map"); return NULL; }
  if( !sv->CheckMemory() ) 
    { /* buggy - gPrintErr("Variable does not own memory in SME_Var2Map"); return NULL;*/ }

  int x1 = 0, y1 = 0, x2 = 0, y2 = 0;
  float downsample = 1.0;
  float umin = 0.0, umax = 0.0;
  int rescale = 0, clip = 0, background = 0;
  int bsize = 1;
  float scale = 1.0, offset = 0.0;

  // process relevant flags and take them off the list
  for (int i = 0; i < argc; i++) 
    {
      if (strcmp(argv[i], "-x1") == 0)
	{
	  *argv[i] = '\0';
	  x1 = atoi(argv[++i]);
	  *argv[i] = '\0';
	}
      else if (strcmp(argv[i], "-y1") == 0)
	{
	  *argv[i] = '\0';
	  y1 = atoi(argv[++i]);
	  *argv[i] = '\0';
	}
      else if (strcmp(argv[i], "-x2") == 0)
	{
	  *argv[i] = '\0';
	  x2 = atoi(argv[++i]);
	  *argv[i] = '\0';
	}
      else if (strcmp(argv[i], "-y2") == 0)
	{
	  *argv[i] = '\0';
	  y2 = atoi(argv[++i]);
	  *argv[i] = '\0';
	}
      else if (strcmp(argv[i], "-min") == 0 || strcmp(argv[i], "-f") == 0)
	{
	  *argv[i] = '\0';
	  umin = atof(argv[++i]);
	  *argv[i] = '\0';
	}
      else if (strcmp(argv[i], "-max") == 0 || strcmp(argv[i], "-x") == 0)
	{
	  *argv[i] = '\0';
	  umax = atof(argv[++i]);
	  *argv[i] = '\0';
	}
      else if (strcmp(argv[i], "-downsample") == 0 || 
	       strcmp(argv[i], "-d") == 0)
	{
	  *argv[i] = '\0';
	  downsample = atof(argv[++i]);
	  *argv[i] = '\0';
	}
      else if (strcmp(argv[i], "-rescale") == 0 || strcmp(argv[i], "-s") == 0)
	{
	  *argv[i] = '\0';
	  rescale = 1;
	}
      else if (strcmp(argv[i], "-clip") == 0 || strcmp(argv[i], "-p") == 0)
	{
	  *argv[i] = '\0';
	  clip = 1;
	}
      else if (strcmp(argv[i], "-resolution") == 0 ||
	       strcmp(argv[i], "-b") == 0)
	{
	  *argv[i] = '\0';
	  bsize = atoi(argv[++i]);
	  *argv[i] = '\0';
	}
      else if (strcmp(argv[i], "-scale") == 0)
	{
	  *argv[i] = '\0';
	  scale = atof(argv[++i]);
	  *argv[i] = '\0';
	}
      else if (strcmp(argv[i], "-offset") == 0)
	{
	  *argv[i] = '\0';
	  offset = atof(argv[++i]);
	  *argv[i] = '\0';
	}
      else if (strcmp(argv[i], "-bg") == 0 || strcmp(argv[i], "-g") == 0)
	{
	  *argv[i] = '\0';
	  background = atoi(argv[++i]);
	  *argv[i] = '\0';
	}
    }

  Region2* r = &sv->Region();

  if( r == NULL ) 
    { gPrintErr("Undefined region in SME_Var2Map"); return NULL; }

  MultiCoverage* vcov = &sv->Cov(); 
  if( vcov == NULL ) 
    { gPrintErr("Undefined coverage in SME_Var2Map"); return NULL; }

  if (x1 != 0 || y1 != 0 || x2 != 0 || y2 != 0) 
    r->setregion(x1, y1, x2, y2, 1, 1);
  
  // downsampling according to o->_downsample
  int dsf = (int)(1.0/downsample);
  if (dsf != 1) r->setincrement(dsf, dsf);

  min = max = 0.0;

  if (rescale)
    {
      int size; 
      const float* data = SS_GetCurrentData(&size);
      min = max = data[0];
      for (int l = 0; l < size; l++)
	{
	  if (data[l] > max) max = data[l];
	  if (data[l] < min) min = data[l];
	}	  
    }
  
  TMap* m = new TMap(r, bsize);
  m->SetByteConversion(scale, offset, bsize);
  m->ClearBounds();
  if (min != 0.0 || max != 0.0 || rescale) 
    m->SetBounds(min, max, umin, umax, rescale);
  
  // make map
  m->Set((byte)background);
  vcov->CopyToMap(*m, TRUE);
  
  return m;
}

int SME_SetVariable(Tcl_Interp* in, Variable* v, int argc, char* argv[])
{
  if (argc < 3)
    return SMETcl_error(in, "usage: <var> set [-r <row> -c <col>] [-f] "
			"[-l <layer>] <value>");
  
  int row = -1, col = -1, layer = -1;
  float val; Bool valset = FALSE, force = FALSE;

  for (int i = 2; i < argc; i++)
    {
      if (strcmp(argv[i], "-r") == 0)
	row = atoi(argv[++i]);
      else if (strcmp(argv[i], "-c") == 0)
	col = atoi(argv[++i]);
      else if (strcmp(argv[i], "-l") == 0)
	layer = atoi(argv[++i]);
      else if (strcmp(argv[i], "-f") == 0)
	force = TRUE;    // force commands to use simple update on non-state 
                         // variable, just in case
      else val = atof(argv[i]);
    }
  
  if ((row+col) != -2 && !v->GetF(FisSpatial))
    return SMETcl_error(in, "can't set value in non-spatial variable");
  if ( ((row+col) == -2 || (row*col) < 0) && v->GetF(FisSpatial))
    return SMETcl_error(in, "please specify row and column");

  const char* command = NULL;

  
  if ((row+col) != -2)
    {
      if (layer == -1)
	command = format("VS %d %d %g", row, col, val);
      else
	command = format("VS %d %d %d %g", row, col, layer, val);
    }
  else
    command = format("VS %g", val);
  
  // if state variable, it's simple. I guess this also applies to 
  // constants, which can't have a breakpoint set on update cmds
  // the force parameter (-f) is there just in case
  if (v->GetF(FisStateVar) || v->GetF(FisConstant) || force)
      SS_ExecuteCommand ((char*)command);  
  else
    {
      // kasino - hope it works
      SS_ExecuteCommand("b u");            // break on update cmd
      SS_ExecuteCommand("r");              // run to that point
      SS_ExecuteCommand("s c");            // execute command
      SS_ExecuteCommand((char*)command);   // set value
      SS_ExecuteCommand("b D");            // delete breakpoint
    }

  return TCL_OK;
}

int SME_RetValue(Tcl_Interp* interp, Variable* v, int argc, char* argv[])
{
  int use_map = 0;
  int use_rle = 0;
  int force_float = 0;

  if (argc < 3) 
    {
      int size; 
      const float* data = SS_GetCurrentData(&size);
      Tcl_ResetResult(interp);
      for (int i = 0; i < size; i++) 
	Tcl_AppendElement(interp, format("%g", data[i]));
    }
  else 
    {
      int r = -1, c = -1, at = -1; 
      for (int i = 2; i < argc; i++) 
	{
	  if (strcmp(argv[i], "-r") == 0)
	    {
	      use_map = 1;
	      *argv[i] = '\0';
	      r  = atoi(argv[++i]);
	      *argv[i] = '\0';
	    }
	  else if (strcmp(argv[i], "-c") == 0)
	    {
	      use_map = 1;
	      *argv[i] = '\0';
	      c  = atoi(argv[++i]);
	      *argv[i] = '\0';
	    }
	  else if (strcmp(argv[i], "-map") == 0 || strcmp(argv[i], "-m") == 0)
	    {
	      *argv[i] = '\0';
	      use_map = 1;
	    }
	  else if (strcmp(argv[i], "-force-float") == 0 ||  
		   strcmp(argv[i], "-ff") == 0)
	    {
	      *argv[i] = '\0';
	      force_float = 1;
	    }
	  else if (strcmp(argv[i], "-rle") == 0 || strcmp(argv[i], "-l") == 0)
	    {
	      *argv[i] = '\0';
	      use_rle = 1;
	    }
	  else if (strcmp(argv[i], "-at") == 0)
	    at = atoi(argv[++i]);
	  else 	
	    // TBC options go to Var2Map and are checked there
	    if (v->GetF(FisSpatial))
		  use_map = 1; 
	}
      
      // check feasible
      if (at != -1 && v->GetF(FisSpatial))
	return SMETcl_error(interp, 
			    "value: -at option applies to scalar variables");
      if (use_map && !(v->GetF(FisSpatial)))
	return SMETcl_error(interp, 
			    "value: options apply to spatial variables");

      if (use_map)
	{
	  // return partial value
	  float min, max;
	  TMap* m = SME_Var2Map(v, argc, argv, min, max);
	  if (r != -1 || c != -1)
	    {
	      /* return either entire column, entire row, or pixel */
	      if (r != -1 && c != -1)
		{
		  if (force_float)
		    {
		      CVariable* cv = (CVariable*)v;
		      Tcl_SetResult(interp, 
				    format("%g", cv->Value(r, c)), TCL_STATIC);
		    }
		  else
		    {
		      Tcl_SetResult(interp, 
				    format("%ld", (long)m->Value(r,c)), 
				    TCL_STATIC);
		    }
		}
	      else if (r != -1 && c == -1)
		;
	      else
		;
	    }
	  else
	    {
	      // return map values as a list of lists or as RLE-encoded
	      // string
	      if (use_rle)
		{
		  SRLE_Encoder encoder(add_res, interp);

		  Tcl_ResetResult(interp);
		  
		  int rows  = m->extents(0);
		  int cols  = m->extents(1);
		  int cnt = 0;
		  
		  if (force_float)
		    {
		      // send column first as it's supposed to be
		      // x, y for generality
		      Tcl_AppendResult(interp, 
				       format("%d,%d,8,%f,%f,", 
					      cols, rows, min, max), 
				       NULL);
		      CVariable* cv = (CVariable*)v;

		      // encode in column-first order as in map
		      for (int r = 0; r < rows; r++)
			for (int c = 0; c < cols; c++)
			  {
			    float fv = cv->Value(r,c);
			    if (fv == floatVec::ErrorVal()) 
			      fv = 0.0;
			    encoder << fv;
			  }
		      encoder.flush();
		    }
		  else 
		    {
		      Tcl_AppendResult(interp, 
				       format("%d,%d,%d,%f,%f,", 
					      cols, rows, m->NBytes(), 
					      min, max), 
				       NULL);
		      
		      long size = rows*cols;     

		      for (long i = 0; i < size; i++)
			{ 
			  long dat;
			  switch (m->NBytes())
			    {
			    case 1: 
			      dat = (long)((byte*)(m->Data()))[i]; 
			      break;
			    case 2: 
			      dat = (long)((short*)(m->Data()))[i]; 
			      break;
			    case 4: 
			      dat = ((long*)(m->Data()))[i]; 
			      break;
			    }
			  encoder << dat;
			} 
		      encoder.flush();
		    }
		}
	      else 
		{
		}
	    }
	  delete m;
	}
      else if (at != -1)
	{
	}
      else 
	{
	  /* not map; can  be timeseries or spatial not returned as
	     map. The option of using RLE encoding is still honored
	     (only syntactically at the moment)
	  */

	  // calculate min/max (TBC: something should be there
	  // already in SME - also beware of parallelism)
	  const float* data = v->Data(); float min, max;
	  min = max = data[0];
	  for (int l = 0; l < v->DataSize(); l++)
	    {
	      if (data[l] < min) min = data[l];
	      if (data[l] > max) max = data[l];
	    }

	  if (use_rle)
	    {
	      if (v->GetF(FisSpatial))
		{
		  Tcl_AppendResult(interp, 
				   format("1,%d,8,%f,%f", v->DataSize(), min, max), 
				   NULL);
		  for(int l = 0; l < v->DataSize(); l++)
		    Tcl_AppendResult(interp, format(",%g", v->Data()[l]),
				     NULL);
		} 
	      else 
		{
		  Tcl_AppendResult(interp, 
				   format("2,%d,8,%f,%f", v->DataSize(), min, max), 
				   NULL);
		  for(int l = 0; l < v->DataSize(); l++)
		    Tcl_AppendResult(interp, format(",%d,%g", l, v->Data()[l]),
				     NULL);
		}
	    }
	}
    }
  return TCL_OK;
}

int SME_DumpVariable(Tcl_Interp* interp, Variable* d, int argc, char* argv[])
{
  int i, frm = 0;  // text
  char* outfile = NULL;
  int r_from = 0, r_to = 0;
  int smap = 1, remap = 0;

  for ( i = 2; i < argc; i++)
    {
      if (strcmp(argv[i], "-hdf") == 0) 
	{
	  *argv[i] = '\0';
	  frm = 1;
	}
      else if (strcmp(argv[i], "-txt") == 0)
	{
	  *argv[i] = '\0';
	  frm = 0;
	}
      else if (strcmp(argv[i], "-ppm") == 0)
	{
	  *argv[i] = '\0';
	  frm = 2;
	}
      else if (strcmp(argv[i], "-gif") == 0)
	{
	  *argv[i] = '\0';
	  frm = 7;
	}
      else if (strcmp(argv[i], "-grass") == 0)
	{
	  *argv[i] = '\0';
	  frm = 4;
	}
      else if (strcmp(argv[i], "-mapII") == 0)
	{
	  *argv[i] = '\0';
	  frm = 5;
	}
      else if (strcmp(argv[i], "-arcInfo") == 0)
	{
	  *argv[i] = '\0';
	  frm = 6;
	}
      else if (strcmp(argv[i], "-bcmap") == 0) // "binary" colormap
	{
	  *argv[i] = '\0';
	  smap = 0;
	}
      else if (strcmp(argv[i], "-remap") == 0) // "binary" colormap
	{
	  *argv[i] = '\0';
	  remap = 1;
	  r_from  = atoi(argv[++i]);
	  *argv[i] = '\0';
	  r_to  = atoi(argv[++i]);
	  *argv[i] = '\0';
	}
    }
  
  int size; const float* data = SS_GetCurrentData(&size);
  
  TMap2* m = NULL; float min, max;
  if (d->GetF(FisSpatial))
    m = SME_Var2Map(d, argc, argv, min, max);

  // reprocess arguments to find outfile and errors
  for ( i = 2; i < argc; i++)
    {
      if (*argv[i] == '\0') 
	{
	  continue;
	}
      else if (*argv[i] == '-')
	{
	  delete m;
	  return SMETcl_error(interp, "dump: unrecognized switch %s", argv[i]);
	}
      else
	{
	  outfile = argv[i];
	}
    }
  
  if (frm > 0 && outfile == NULL)
    return SMETcl_error(interp, "non-text format requires an output file");

  if (frm > 0 &&  !(d->GetF(FisSpatial)))
    return SMETcl_error(interp, "the variable is not spatial");

  int ret = 1;
  
  switch(frm)
    {
    case 0: // text
      if (outfile == NULL)
	{
	}
      else 
	{
	  if (m) 
	    ret = m->WriteAscii(outfile);
	  else 
	    {
	      FILE* fp = fopen(outfile, "w");
	      for (int i = 0; i < size; i++)
		fprintf(fp, "%g\n", data[i]);
	      fclose(fp);
	    }
	}
      break;
    case 1: // hdf
      ret = m->WriteHdf(outfile, 0);
      break;
    case 2: // ppm
    case 7: // gif

      if (m != NULL)
      {
	// 1) create colormap with size corresponding to map resolution
	  const float curv = 1.4;
	  const float bias = 1.0;
	  const float rfct = 0.5 * bias;

	  int size = 256;

	  float* red, *green, *blue;
	  blue = new float[size];
	  green = new float[size];
	  red = new float[size];

	  float rm, rx, gm, gx, bm, bx;
	  rm = rx = gm =gx = bm = bx = 0.0; 

	  for (i = 0; i < size; i++)
	    {
	      float s = (float)i/(float)size;
	      float tt;
	      
	      tt       = curv * (s - rfct); 
	      red[i]   = 128.0 + 127.0 * atan(7.0*tt) / 1.57;
	      green[i] = 128.0 + 127.0 * (2 * exp(-7*tt*tt) - 1);
	      blue[i]  = 128.0 + 127.0 * atan(-7.0*tt) / 1.57;
	      
	      // store min/max
	      if (red[i] < rm)   rm = red[i]; 
	      if (red[i] > rx)   rx = red[i]; 
	      if (green[i] < gm) gm = green[i]; 
	      if (green[i] > gx) gx = green[i]; 
	      if (blue[i] < bm)  bm = blue[i]; 
	      if (blue[i] > bx)  bx = blue[i]; 
	    }
	  for (i = 0; i < size; i++)
	    {
	      red[i]   = 255.0 * ((red[i]   - rm) / (rx - rm));
	      green[i] = 255.0 * ((green[i] - gm) / (gx - gm));
	      blue[i]  = 255.0 * ((blue[i] - bm) / (bx - bm));
	    }  

	  if (!smap)
	    {
	      red[0] = green[0] = blue[0] = 0.0;
	      red[1] = green[1] = blue[1] = 1.0;
	    }
	  
	  
	  FILE* fp = fopen((const char*)outfile, "w");;
	  byte* b = new byte[m->extents(1)*3*sizeof(byte)];
	  
	  // write header (binary format - necessarily 1-byte)
	  fprintf(fp, "P6 %d %d 255\n", m->extents(1), m->extents(0));
	  fflush(fp);
	  
	  int dx = m->extents(1);
	  int dy = m->extents(0);

	  for (int j = 0; j < dy; j++)
	    {
	      for (int i = 0; i < dx; i++)
		{
		  
		  int dat = m->Data()[j*dx+i];

		  if (remap && (dat == r_from))
		    dat = r_to;
		  
		  b[i*3]   = (byte)red[dat];
		  b[i*3+1] = (byte)green[dat];
		  b[i*3+2] = (byte)blue[dat];
		}
	      fwrite(b, 1, dx*sizeof(byte)*3, fp);
	    }
	  
	  fclose(fp);
	  delete b;
	  
	  delete red;
	  delete green;
	  delete blue;
      } else
	return SMETcl_error(interp, "Null MAP pointer for spatial variable");

      break;
    case 3: // tbc
      break;
    case 4: // grass
      ret = m->WriteGrass(outfile);
      break;
    case 5: // mapII
      ret = m->WriteM2File(CPathString(outfile), CString(outfile), 0, "");
      break;    
    case 6: // arcInfo
      ret = m->WriteArc(outfile);
      break;
    }

  if (m != NULL)
    delete m;

  return ret == 0 ? TCL_ERROR : TCL_OK;
}

int SME_ModelCallback(ClientData d, Tcl_Interp* i, int argc, char* argv[])
{
  int ret = TCL_OK;

  if (strcmp(argv[1], "elapsed") == 0)
    ret =  SMETcl_elapsed(d, i, argc - 1, &(argv[1]));
  else if (strcmp(argv[1], "dt") == 0)
    ret =  SMETcl_dt(d, i, argc - 1, &(argv[1]));
  else if (strcmp(argv[1], "t0") == 0)
    ret =  SMETcl_t0(d, i, argc - 1, &(argv[1]));
  else if (strcmp(argv[1], "step") == 0)
    ret = SMETcl_step(d, i, argc - 1, &(argv[1]));
  else if (strcmp(argv[1], "run") == 0)
    ret = SMETcl_run(d, i, argc - 1, &(argv[1]));
  else if (strcmp(argv[1], "exec") == 0 || strcmp(argv[1], "e") == 0) 
    ret = SMETcl_execute(d, i, argc - 1, &(argv[1]));	
  else if (strcmp(argv[1], "open") == 0)
    SS_Open();
  else if (strcmp(argv[1], "module_list") == 0) 
      return make_tcl_list_from_pix(i, TModel::I().ModuleList());
  else if (strcmp(argv[1], "parameter_list") == 0) 
      return make_tcl_list_from_pix(i, TModel::I().ParameterList(), True );
  else if (strcmp(argv[1], "restart") == 0)
    SS_Close();
  else if (strncmp(argv[1], "configure", 6) == 0 || strcmp(argv[1], "c") == 0)
    {
      if ((ret = SME_SetConfigArgs(i, kSS_Model, argc, argv)) == TCL_OK)
	SS_RegisterConfigData(kSS_Model);
    }
  else if (strcmp(argv[1], "info") == 0)
    {
      if (strcmp(argv[2], "grid") == 0)
	{
	  if (argc > 3) 
	    {
	      // returns dx, dy of chosen grid
	      int nl = TModel::I().Grid()->nCellLayers();
	      int n  = atoi(argv[3]);
	      if (n >= nl || n < 0)
		ret = SMETcl_error(i, "error: non-existent grid requested");
	      else
		{
		  int dx =0, dy = 0;
		  TLayer* l =  TModel::I().Grid()->getCellLayer(n);
		  if (l != NULL)
		    {
		      dx = l->Dim(0);
		      dy = l->Dim(1);
		    }
		  Tcl_SetResult(i, format("%d %d", dx, dy), TCL_STATIC);
		}
	    }
	  else 
	    {
	      ret = SMETcl_error(i, "usage: SME info grid <n>");
	    }
	}
      else if (strcmp(argv[2], "ngrids") == 0)
	{
	  Tcl_SetResult(i, 
			format("%d", TModel::I().Grid()->nCellLayers()),
			TCL_STATIC);
	}
    }
  else if (strcmp(argv[1], "cget") == 0)
    {
    }
  return ret;
}

int SME_ModuleCallback(ClientData d, Tcl_Interp* i, int argc, char* argv[])
{
  int ret = TCL_OK;

  // necessary to configure extension modules in module scope  
  if (SS_SetCurrentModule(argv[0]) == 0)
    ret = SMETcl_error(i, "Internal error in SetCurrentModule");
  
  if (strcmp(argv[1], "info") == 0)
    {
    }
  if (strcmp(argv[1], "makeCurrent") == 0)
    {
      // done already: just an empty command
    }
	else if (strcmp(argv[1], "exec") == 0 || strcmp(argv[1], "e") == 0) {
	  ret = SMETcl_execute(d, i, argc - 1, &(argv[1]));	
	}
  else if (strcmp(argv[1], "variable_list") == 0)
    {
      return make_tcl_list_from_pix(i, ((Module*)d)->VarList());
    }
  else if (strncmp(argv[1], "configure", 6) == 0 || strcmp(argv[1], "c") == 0)
    {
      if ((ret = SME_SetConfigArgs(i, kSS_Module, argc, argv)) == TCL_OK)
	SS_RegisterConfigData(kSS_Module);
    }
  else if (strcmp(argv[1], "cget") == 0)
    {
    }
  
  return ret;
}

int SME_VariableCallback(ClientData d, Tcl_Interp* i, int argc, char* argv[])
{
  int ret = TCL_OK;

  PathName p(argv[0]);


  if (SS_SetCurrentModule(p.mod()) == 0 ||
      SS_SetCurrentVariable(p.var()) == 0)
    ret = SMETcl_error(i, "Internal error in SetCurrentVariable");

  sprintf(gMsgStr,"SME_VariableCallback(%d): %s.%s: %s",argc,p.mod(),p.var(),argv[1]);
  gPrintLog();
  
  if (argc < 2 || strcmp(argv[1], "value") == 0 || strcmp(argv[1], "v") == 0) 
    {
      // return value as scalar or list
      ret = SME_RetValue(i, (Variable*)d, argc, argv);
    }  
  else if (strcmp(argv[1], "info") == 0 || strcmp(argv[1], "i") == 0)
    {
      ret = SME_VarInfo(i, (Variable*)d, argc, argv);
    }
  else if (strcmp(argv[1], "exec") == 0 || strcmp(argv[1], "e") == 0) {
    ret = SMETcl_execute(d, i, argc - 1, &(argv[1]));	
  }
  else if (strcmp(argv[1], "makeCurrent") == 0)
    {
      // done already: just an empty command
    }
  else if (strncmp(argv[1], "configure", 6) == 0 || strcmp(argv[1], "c") == 0)
    {
      if ((ret = SME_SetConfigArgs(i, kSS_Variable, argc, argv)) == TCL_OK)
	SS_RegisterConfigData(kSS_Variable);
    }
  else if (strcmp(argv[1], "commands") == 0)
    {
      ret = make_tcl_list_from_pix(i, ((Variable*)d)->CommandList());
    }
  else if (strcmp(argv[1], "cget") == 0)
    {
    }
  else if (strcmp(argv[1], "dump") == 0)
    {
      ret = SME_DumpVariable(i, (Variable*)d, argc, argv);
    }
  else if (strcmp(argv[1], "get") == 0)
    {
    }
  else if (strcmp(argv[1], "set") == 0)
    {
      ret = SME_SetVariable(i, (Variable*)d, argc, argv);
    }
  else if (strcmp(argv[1], "update") == 0)
    {
    }
  return ret;
}

int SMETcl_step(ClientData d, Tcl_Interp* i, int argc, char* argv[])
{
  if (argc < 2)
    SS_Step();
  else 
    {
      if (strncmp(argv[1], "-event", 2) == 0)
	SS_Step(kSS_EventS);
      else if (strncmp(argv[1], "-command", 2) ==0)
	SS_Step(kSS_CommandS);
      else if (strncmp(argv[1], "-update", 2) == 0)
	SS_Step(kSS_UpdateD);
      else 
	return SMETcl_error(i, "step: unknown flag %s", argv[1]);
    }
  return TCL_OK;
}

int SMETcl_run(ClientData d, Tcl_Interp* i, int argc, char* argv[])
{
  float res = 0.0;
  
  if (argc < 2)
      res = SS_Run(0);
  else 
      res = SS_Update(atof(argv[1]),0);
  
  Tcl_SetResult(i, format("%f", res), TCL_STATIC);

  return TCL_OK;
}

int SMETcl_execute(ClientData d, Tcl_Interp* i, int argc, char* argv[]) 
{
  if (argc < 2) 
    return TCL_OK;
  
  CString cmd;
  for( int j=1; j<argc; j++ ) {
    cmd += argv[j]; cmd += " "; 
  }		
  
  const char* rv = SS_ExecuteCommand ((char*)cmd.chars());  
  Tcl_SetResult(i, (char*)rv, TCL_STATIC);  
  
  return TCL_OK ;
}


int SMETcl_restart(ClientData d, Tcl_Interp* i, int argc, char* argv[])
{
  SS_Close();
  int& iSim = Env::BatchIndex();
  if( iSim < 0 ) iSim = 1;
  else iSim++;
  
  return TCL_OK;
}

int SMETcl_elapsed(ClientData d, Tcl_Interp* i, int argc, char* argv[])
{
  Tcl_SetResult(i, format("%g", TTime::Current()), TCL_STATIC);

  return TCL_OK;
}

int SMETcl_dt(ClientData d, Tcl_Interp* i, int argc, char* argv[])
{
  Tcl_SetResult(i, format("%g", TTime::DT()), TCL_STATIC);

  return TCL_OK;
}

int SMETcl_t0(ClientData d, Tcl_Interp* i, int argc, char* argv[])
{
  Tcl_SetResult(i, format("%g", TTime::Start()), TCL_STATIC);

  return TCL_OK;
}

int SME_Init(Tcl_Interp* interp)
{

  // "publish" pointer to interpreter for external apps 
  _interp = interp;

  Tcl_CreateExitHandler( SMETcl_Exit,  NULL );

  // create shortened version of major commands as well
  Tcl_CreateCommand(interp, "s", SMETcl_step, 0, NULL);
  Tcl_CreateCommand(interp, "r", SMETcl_run, 0, NULL);
  Tcl_CreateCommand(interp, "S", SMETcl_restart, 0, NULL);

  // install SME module 
  Tcl_CreateCommand(interp, "SME", SME_ModelCallback, 0, NULL);

  // create list of modules, variables, array of variables indexed by module,
  // commands for each variable and module
  
  TNamedObjectList& ml = TModel::I().ModuleList();

  for( Pix p = ml.first(); p; ml.next(p) ) 
    {
      TNamedObject& obj = (TNamedObject&)ml(p);
      const char* mn = obj.Name();
      Module* mdl = TModel::I().GetModule(mn, False);
      Tcl_CreateCommand(interp, (char*)mn, SME_ModuleCallback, mdl, NULL);

      Tcl_SetVar(interp, "SME_modules", format("%s", mn),  TCL_GLOBAL_ONLY | TCL_APPEND_VALUE | TCL_LIST_ELEMENT );

      TNamedObjectList& vl = mdl->VarList();

      for( Pix v = vl.first(); v; vl.next(v) ) 
	{
	  TNamedObject& obj = (TNamedObject&)vl(v);
	  const char* vn = obj.Name();
	  Variable* var = mdl->GetVariable(vn, False);

	  Tcl_SetVar(interp, "SME_variables", 
		     format("%s.%s", mn, vn), 
		      TCL_GLOBAL_ONLY | TCL_APPEND_VALUE | TCL_LIST_ELEMENT);

	  Tcl_CreateCommand(interp, format("%s.%s", mn, vn), 
			    SME_VariableCallback, var, NULL);
	}
    }

  // install built_in external module FrameLink
  static external_config_table framelink_config_table[] = {
    {"i",  "indices"},
    {"p",  "mapProcessIndex"},
    {"m0", "mapInput0"},
    {"m1", "mapInput1"},
    {"m2", "mapInput2"},
    {"m3", "mapInput3"},
    {"m4", "mapInput4"},
    {NULL, NULL}
  };
  
  SMETcl_install_module ("FrameLink", NULL, NULL, framelink_config_table);

  // if a TCL file has been passed, execute it and exit
  const char* tcl_file = SS_TclFile();
  if ( tcl_file ) {
	  sprintf(gMsgStr, "Executing Tcl_EvalFile: %s ", tcl_file );  gPrintScreen();
      int ret = Tcl_EvalFile(_interp, (char*) tcl_file );
      if (ret == TCL_ERROR) {
		sprintf(gMsgStr, "Tcl Error: %d ", ret );  gPrintErr();
		exit (ret == TCL_ERROR ? 0xff : 105 );
	  }
	  sprintf(gMsgStr, "Tcl_EvalFile completed, exiting." );  gPrintScreen();
      SS_Exit();
    }

  return TCL_OK;
}

int Tcl_AppInit(Tcl_Interp* interp) 
{
  Tcl_SetVar(interp, "tcl_rcFileName", "~/.smerc", TCL_GLOBAL_ONLY);

  if (Tcl_Init(interp) == TCL_ERROR) {
    return TCL_ERROR;
  }
  if (SME_Init(interp) == TCL_ERROR) {
    return TCL_ERROR;
  }

	if( SS_UseJavaSocket() ) {
    char inBuf[256];
    while( 1 ) {
			SS_ExecuteJavaCommand ( inBuf, 1, interp );
		} 
  }
  
  return TCL_OK;
}

void SMETcl_Exit(ClientData clientData) {
 SS_Exit( False );
}

int SMETcl_install_module (char* name, void* module, 
			   external_command_table* commands,
			   external_config_table* config)
{ 
  module_info* mi = new module_info;

  mi->module = module;
  mi->cmd_table = commands;
  mi->cnf_table = config;

  Tcl_CreateCommand(SSTcl_Interp(), name, SME_extmodule_callback, mi, NULL);
  
  return TCL_OK;
}

int SME_extmodule_callback(ClientData d, Tcl_Interp* interp, 
			   int argc, char* argv[])
{
  
  int ret = TCL_OK;
  
  void* module = ((module_info*)d)->module;
  external_command_table* cmdtab = ((module_info*)d)->cmd_table;
  external_config_table*  cnftab = ((module_info*)d)->cnf_table;

  if (strcmp(argv[1], "info") == 0)
    {
    }
  else if (strcmp(argv[1], "configure") == 0 || strcmp(argv[1], "c") == 0)
    {
      SS_CD.SName() = argv[0];

      char* cmd = argv[2];

      if (cnftab != NULL)
	{
	  // look for command in translation list
	  for (int i = 0;  cnftab[i].command_name != NULL; i++)
	    {
	      if (strcmp(cmd, cnftab[i].command_name) == 0)
		{
		  cmd = cnftab[i].command;
		  break;
		}
	    }
	}

      SS_CD.SetCmd(cmd);
      int ac = 0;
      for (int j = 3; j < argc; j++)
	if (*argv[j] != '-')
	  SS_CD.SetArg(ac++, argv[j]);
      SS_RegisterConfigData(kSS_External);
    }
  else if (cmdtab != NULL)
    {
      // look in command list and invoke command
      for (int i = 0; cmdtab[i].command != NULL; i++)
	{
	  if (strcmp(argv[1], cmdtab[i].command) == 0)
	    {
	      ret = (*(cmdtab[i].func))(interp, module);
	      break;
	    }
	}
    } 
  return ret;
}

#endif
