#include "SSVariable.h"
#include "SSModule.h"
#include "TPipe.h"
#include "MemoryMgr.h"
#include "SSModel.h"
#include "HDepParm.h"
#include "HDFMgr.h"

 unsigned int	FFluxOriginMap ;		//  Flux Origin has different Grid
 unsigned int	FFluxDestinationMap ;		//  Flux Destination has different Grid
 unsigned int	FMapRemoteData ;		//  Input Var and Connection have different Grids
 unsigned int	FICFlux  ;		//  calls ICFlux method
 unsigned int	FAllocated ;		//  Allocate() Method has been called.
 unsigned int	FEdgesLinked ;		//  LinkEdges() Method has been called.
  
//========================================================================================
// 				CLASS TVariable
//========================================================================================

byte TVariable::kNoDbgCnfg = 255;
TNamedObjectList  TVariable::fDBaseList("Variable DBase List",8);

TVariable::TVariable(const char* name) : Variable ( name )
{ 
	if(fFlagList==NULL) {
    fFlagList = new TObjectList();
    AddFlags();
  } 
	SetCInfo(kType,kNULL); 
	SetCInfo(kInitMode,kDefault); 
	SetCInfo( kDebug, kNoDbgCnfg );	
  SetF( FisOptimizable, True, "TVariable constructor" );
}

void TVariable::SetupInternals() { ; }

void TVariable::AddFlags() {
	Variable::AddFlags();
  AddFlag("FEdgesLinked",FEdgesLinked);
  AddFlag("FFluxDestinationMap",FFluxDestinationMap);
  AddFlag("FFluxOriginMap",FFluxOriginMap);
  AddFlag("FMapRemoteData",FMapRemoteData);
  AddFlag("FAllocated",FAllocated);
	AddFlag("FICFlux",FICFlux);
}


int  TVariable::Config( TConfigData& cd) {
  const CString& cmd = cd.Cmd();
  if(gDebug) { 
    sprintf(gMsgStr,"Reading Variable Config for %s: Nargs = %d, CMD = %s",
	    fName.chars(), cd.NArgs(), cmd.chars() );	
    gPrintLog();  
  }
  int iarg;
  TPipeCommand* pc = NULL;
  Pipe* aPipe = TPipe::GetPipeFromConfig( cd, this, pc ); 
  if( aPipe != NULL ) {
	if(pc==NULL) {
	  pc = TPipeCommand::CreatePipeCommand( cd, aPipe, this );
	  AddCommand( pc );
	}
	aPipe->SetPipeCommand(pc);    
  } else {
    if( cmd == "d" ) {
      if( cd.IntArg(0,iarg) > 0 ) SetCInfo( kDebug, iarg );
    } else if ( cmd == "vi" ) {
      if( cd.IntArg(0,iarg) > 0 ) TModel::I().AddIVar(iarg,this);
      else { addStateVariable( *this ); }
    } else return Variable::Config(cd);
  }
  return 1;
}

#ifdef POSTGRESS

sme_pg95* TVariable::GetDBase(const char* name) {
	TDBaseSocket* db = (TDBaseSocket*) fDBaseList.GetObjectByName(name);
	if( db == NULL ) db = new TDBaseSocket(name);
	fDBaseList.append(*db);
	return db->DBase();
}
 
#endif

/*
//========================================================================================
// 				CLASS TSpatialVariable
//========================================================================================

TSpatialVariable::TSpatialVariable(char* name) 
  : TVariable ( name )    
{
  SetCInfo(kType,kNULL); 
  SetCInfo(kInitMode,kDefault);
  fMaxDelay = 0.0; 
  fCatMapPtr = NULL;
  for(int i=0; i<4; i++) fMaps[i] = NULL;
}  

void TSpatialVariable::Map ( TSpatialVariable* aMVar ) {
  for(int i=0; i<4; i++) if( fMaps[i] == NULL ) { fMaps[i] = aMVar; break; }
  if( !GetF(FisImport) ) { 
    for( Pix p = fConnectionList.first(); p; fConnectionList.next(p))  {
      TSpatialVariable& aVar = (TSpatialVariable&) fConnectionList(p);
      aVar.Map(aMVar);
    }
  }
}

int  TSpatialVariable::Allocate( ) {
  if( Allocated() ) return 0;
  TFrame* f = (TFrame*) &((TModule*)fModule)->Frame();
  int ghostSize = f->GetParm(TFrame::kMaxGhost);
  int memoryIndex; 
  if( TModel::I().MemoryOp() == 0 ) { memoryIndex = MemObject::kPermanentMemory; }
  else { memoryIndex = ( GetCInfo(kInitMode) ==  kDefault ) ? fModule->Index() : MemObject::kPermanentMemory; }
  
  if( GetF(FisImport) ) {
    TSpatialVariable* svar = (TSpatialVariable*)fConnection;
		TFrame& rf = ((TModule*)svar->GetModule())->Frame();
		svar->Allocate();
    if( *(f->Grid()) == svar->Grid() ) {
      if( gDebug) { 
				sprintf(gMsgStr,"\nImport MAlloc: %s::%s (%d), SameGrid",fModule->Name(),Name(),memoryIndex); 
				gPrintLog(); 
      }
#ifdef DEBUG
			if(gDebug>2){
				sprintf(gMsgStr,"\n &&& Allocating memory, %s:%s -> %s:%s \n",Name(),GetModule()->Name(),svar->Name(),svar->GetModule()->Name()); 
				gPrintLog(); 
			}
#endif      
      fCov.Setup( svar->Cov(), memoryIndex );
    } else if( f->PointsSubsetOf(rf) ) {
      if( gDebug) { 
				sprintf(gMsgStr,"\nImport MAlloc: %s::%s (%d), SamePoints",fModule->Name(),Name(),memoryIndex); 
				gPrintLog(); 
      }
      fCov.Setup( f->Grid(), svar->Cov(), ghostSize, memoryIndex );
    } else {
			if( GetF( FisConstant ) ) { 
				memoryIndex = MemObject::kPermanentMemory; 
			}
      if( gDebug) { 
				sprintf(gMsgStr,"\nImport MAlloc: %s::%s (%d), RemoteGrid",fModule->Name(),Name(),memoryIndex); 
				gPrintLog(); 
      }
      fCov.Setup( f->Grid(), ghostSize, memoryIndex );
      SetF(FMapRemoteData,True,"Allocate");
      if( GetF(FDualConnection) ) svar->SetF(FMapRemoteData,True,"Allocate");
    }
  } else {
    if( GetF( FisExport ) || GetF( FisStateVar)  ) { 
      memoryIndex = MemObject::kPermanentMemory; 
    }
    if( gDebug) { sprintf(gMsgStr,"\nMAlloc: %s::%s (%d)",fModule->Name(),Name(),memoryIndex); gPrintLog(); }
    fCov.Setup( f->Grid(), ghostSize, memoryIndex);
  }
  SetF(FAllocated,True,"Allocate");
  return 1;
}

void TSpatialVariable::PrintUpdateMsg( Pix p0, float value, float dv, char* label, Pix pR ) {
  const OrderedPoint& pt = Grid().GetPoint(p0);
  byte dMode = pt.GetObjInfo(DGrid::kDebug);
  if( dMode ) {
    switch(dMode) {
    case 1: if( GetCInfo(kDebug) > 1 ) {
      OrderedPoint& pr = (OrderedPoint&) ((pR) ? Grid().GetPoint(pR) : pt);
      sprintf(gMsgStr,"\n(%d,%d)%s %s by %f -> %f at (%d,%d): ",
	      pt(0),pt(1),label,(const char*)fName,dv,value,pr(0),pr(1)); 
      gPrintLog();
      PrintFlags();
    }
    break;
    case 2: 
      OrderedPoint& pr = (OrderedPoint&) ((pR) ? Grid().GetPoint(pR) : pt);
      sprintf(gMsgStr,"\n(%d,%d)%s %s by %f -> %f at (%d,%d):  ",
	      pt(0),pt(1),label,(const char*)fName,dv,value,pr(0),pr(1)); 
      gPrintLog();
      PrintFlags();
      break;
    }
  }
}

void TSpatialVariable::CopyData(TSpatialVariable& v) {
#ifdef DEBUG
	if( gDebug) {
		TFrame& f = (TFrame&) v.GetModule()->Frame();
		if( ! f.SamePointsAs( (const TFrame&) fModule->Frame() ) ) {
			gPrintErr("Can't Copy Data between Frames with different points!");
			return;
		}
	}
#endif
	float* data_vptr = v.DataStart();
	float* data_ptr = DataStart();
	const float* data_Eptr = DataEnd();
	while( data_ptr < data_Eptr ) {
		*data_ptr++ = *data_vptr++;
	}			
}

void TSpatialVariable::AddData(TSpatialVariable& v) {
#ifdef DEBUG
	if( gDebug) {
		TFrame& f = (TFrame&) v.GetModule()->Frame();
		if( ! f.SamePointsAs( (const TFrame&) fModule->Frame() ) ) {
			gPrintErr("Can't Copy Data between Frames with different points!");
			return;
		}
	}
#endif
	float* data_vptr = v.DataStart();
	float* data_ptr = DataStart();
	const float* data_Eptr = DataEnd();
	while( data_ptr < data_Eptr ) {
		*data_ptr++ += *data_vptr++;
	}			
}	

Pix TSpatialVariable::SSInit( Module* mod ) {
	Pix p0 = TVariable::SSInit( mod );
	fHistory.SetStack(((TModule*)mod)->SpatialArrays());
	SetF(FisSpatial,True,"Init");
	return p0;
}

void TSpatialVariable::UpdateInternals() {
	GrabMemory();
	TVariable::UpdateInternals();

	if( fMaxDelay > 0 ) {
		fHistory.Add( gStep(), fCov );
		fHistory.DeleteBelow( gTime()-fMaxDelay );
	}

	if( GetF(FICFlux) ) fCov.Set(0.0); 

	if( GetF(FLinkEdges) ) { LinkEdges(); }
	else { SetF(FEdgesLinked,False,"UpdateInternals"); }      
}

int TSpatialVariable::DoBoundsCheck(Pix p, float value) {
		switch( fBounds.Check(value) ) {
      case 'd': {
				const OrderedPoint& pt = fCov.Grid().GetPoint(p);
				sprintf(gMsgStr,"\nBOUNDS ERROR(%d,%d) %s -> %f: ",pt(0),pt(1),(const char*)fName,value); 
				gPrintLog(); 
				Env::debugger_stop_here();
				PrintFlags();
				} break;
      case 'p': {
				const OrderedPoint& pt = fCov.Grid().GetPoint(p);
				sprintf(gMsgStr,"\nBOUNDS ERROR(%d,%d) %s -> %f: ",pt(0),pt(1),(const char*)fName,value); 
				gPrintErr(); 
				TModel::I().Interrupt();
				} break;
			case 'f': {
				const OrderedPoint& pt = fCov.Grid().GetPoint(p);
				sprintf(gMsgStr,"\nBOUNDS ERROR(%d,%d) %s -> %f: ",pt(0),pt(1),(const char*)fName,value); 
				PrintFlags();
				gFatal();
				} break;
			default: return 0;	       
		}
		return 1;
}

void TSpatialVariable::BroadcastValue() {
  
  for( Pix p = fConnectionList.first(); p; fConnectionList.next(p))  {
    TSpatialVariable& aSVar = (TSpatialVariable&) fConnectionList(p);
    aSVar.RefreshData();
  }
}

void TSpatialVariable::UpdateRel( Pix p0, Point2 rp, float value ) {
  
  Pix pR = fCov.Grid().TranslateByLinks( p0, rp );
  
  if( pR ) SetValue(pR,value);
  else return;

#ifdef DEBUG  	  
    if(GetCInfo(kDebug)) {
      DoBoundsCheck(pR,value);
      PrintUpdateMsg(p0,Value(pR),value,"UpdateRel",pR);
    }
#endif
}

float TSpatialVariable::RelValue( Pix p0, Point2 rp ) {
  
  Pix pR = fCov.Grid().TranslateByLinks( p0, rp );

  if(pR == 0) return 0.0;
  else return Value(pR);
}


float TSpatialVariable::InitVal( Pix p ) {
 
		if( fHistory.Size() == 0 ) { fHistory.Add( gStep(), fCov ); }
    const int index = fCov.Grid().Index( p );
    return fHistory.FirstValue(index);
}

float* TSpatialVariable::InitValData() {
	if( fHistory.Size() == 0 ) { fHistory.Add( gStep(), fCov ); } 
	return fHistory.FirstData();
}


float TSpatialVariable::Delay( Pix p, float dTime, float ival ) {
       
  if( dTime <= 0.0 ) { return Value( p ); }
  else if( fMaxDelay == 0.0 ) {
    PointSetCoverage cov( fCov.GridP() );
    cov.Set( ival);
    fHistory.Add( gTime()-dTime, cov, kTS_Deep );
    fMaxDelay = dTime;
    return ival;
  }
  else if ( dTime > fMaxDelay ) {
    fMaxDelay = dTime;
    const int index = fCov.Grid().Index( p );
    return fHistory.FirstValue( index );
  }
  else {
    const int index = fCov.Grid().Index( p );
    return fHistory.Value( gTime()-dTime, index );
  }
}

int  TSpatialVariable::Config( TConfigData& cd ) {
  TVariable::Config( cd );
  const CString& cmd = cd.Cmd();
  int iarg; float farg;

  if( cmd == "b" ) {
    if( cd.IntArg(0,iarg) ) fBuffsize = iarg;
  } else return 0;
  return 1;
}      

int  TSpatialVariable::SetDataValues( float* data, int layOut, int size ) {
		Model::I0().SetContext(NULL,NULL,this,fModule);
		fCov.SetDataValues( data, layOut, size);
		UpdateInternals();
		UpdateReserve(); 
		if( GetF(FVisible) || (GetF(FRegistered) &&  Model::I0().MemoryOp()) ) {
				Model::I0().QueryInterface();
		} 
		return 0;
}

//========================================================================================
// 				CLASS TSpatialFluxVariable
//========================================================================================

TSpatialFluxVariable::TSpatialFluxVariable(char* name, int isConstant ) 
  : TSpatialVariable( name )
{
  memset(fFluxInfo,0,4);
  SetCInfo(kType,kFlux); 
  SetCInfo(kInitMode,kDefault); 
  SetF(FisConstant,isConstant,"TSpatialFluxVariable");
}

int  TSpatialFluxVariable::Allocate() {
  if( !TSpatialVariable::Allocate() ) return 0;
  if(Origin()) {
    TSpatialStateVariable* o = (TSpatialStateVariable*) Origin(); 
    o->Allocate();
    if( !(o->Grid() == Grid()) ) SetF( FFluxOriginMap, True, "Allocate" ); 
  }
  if( Destination() ) {
    TSpatialStateVariable* d = (TSpatialStateVariable*) Destination();  
    d->Allocate();
    if( !(d->Grid() == Grid()) ) SetF( FFluxDestinationMap, True, "Allocate" ); 
  } 
  return 1;
}

void TSpatialFluxVariable::Clamp( Pix p ) {
    
  float tmp, value = Value(p);
  Pix pO=p, pD=0; 
  if( Origin() ) { 
    TSpatialStateVariable* o = (TSpatialStateVariable*) Origin(); 
    if( GetF(FFluxOriginMap) ) { pO = o->Cov().MapPix(p,Grid()); }
    if( pO ) {
      if( tmp = o->Reserve(pO) < value ) { 
				fCov(p) = tmp; 
				o->Reserve(pO) = 0;
#ifdef DEBUG
				if( GetCInfo(kDebug) ) PrintUpdateMsg(p,value,tmp,"CLAMP");
#endif
			} else {
				o->Reserve(pO) -= value;  
      }
    }
  }
  if( Destination() ) { 
    TSpatialStateVariable* d = (TSpatialStateVariable*) Destination();  
    if( GetF(FFluxDestinationMap) ) { pD = d->Cov().MapPix(p,Grid()); }
    if(pD) d->Reserve(pD) += value; 
  }
}

void TSpatialFluxVariable::ICFlux( Pix p0, Point2 rp, float value ) {

  SetF(FICFlux,True,"ICFlux");
  Pix pR = fCov.Grid().TranslateByLinks( p0, rp );

  if( p0 && pR ) {
    fCov(p0) += value;
    fCov(pR) -= value;
  }

#ifdef DEBUG  	  
    if(GetCInfo(kDebug)) {
      DoBoundsCheck(p0,Value(p0));
      DoBoundsCheck(pR,Value(pR));
      PrintUpdateMsg(pR,Value(pR),value,"ICFlux:R");
      PrintUpdateMsg(p0,Value(p0),value,"ICFlux:O");
    }
#endif

}

//========================================================================================
// 				CLASS TSpatialStateVariable
//========================================================================================

EConfigInfo TSpatialStateVariable::kUpdateMode = kCInfo4;

TSpatialStateVariable::TSpatialStateVariable(char* name ) 
  : TSpatialVariable( name )
{
  SetCInfo(kType,kState); 
  SetCInfo(kInitMode,kDefault); 
	SetF(FisConstant,False,"TSpatialStateVariable");
	fCurrentOrder = fOrder = 1;
}

void TSpatialStateVariable::UpdateInternals() {
	TSpatialVariable::UpdateInternals();

	if( fOrder > 0 ) {
		fIntegratorMemory.DeleteExtraSlicesBelow( fOrder );
	}
}

void TSpatialStateVariable::AddFlux( TVariable* f ) { fFluxList.Insert(*f); }

int  TSpatialStateVariable::Allocate() {
	if( !TSpatialVariable::Allocate() ) return 0;	
  int memoryIndex = MemObject::kPermanentMemory;
  //  int memoryIndex = fModule->Index();
  if( gDebug) { sprintf(gMsgStr,"\nSVReserve MAlloc: %s::%s (%d)",fModule->Name(),Name(),memoryIndex); gPrintLog(); }
  fReserve.Setup( fCov.GridP(), 0, memoryIndex );
  return 1;
}

Pix TSpatialStateVariable::SSInit( Module* mod ) {
	Pix p0 = TSpatialVariable::SSInit( mod );
	fIntegratorMemory.SetStack(((TModule*)mod)->SpatialArrays());
	SetF(FisStateVar,True,"Init");
	return p0;
}

int  TSpatialStateVariable::Config( TConfigData& cd ) {
  return TSpatialVariable::Config( cd );
}      

//========================================================================================
// 				CLASS TSpatialAuxVariable
//========================================================================================

TSpatialAuxVariable::TSpatialAuxVariable(char* name, int isConstant ) 
  : TSpatialVariable( name ) {
    SetCInfo(kType,kAux); 
    SetCInfo(kInitMode,kDefault); 
    SetF(FisConstant,isConstant,"TSpatialAuxVariable");
}

int  TSpatialAuxVariable::Allocate( ) {
  return TSpatialVariable::Allocate( );
}

//========================================================================================
// 				CLASS TTemporalVariable
//========================================================================================

TTemporalVariable::TTemporalVariable(char* name) 
  : TVariable ( name ) 
{
  SetCInfo(kType,kNULL); 
  SetCInfo(kInitMode,kDefault); 
  for(int i=0; i<4; i++) fMaps[i] = NULL;
  
}  

void TTemporalVariable::Map ( TTemporalVariable* aMVar ) {
  for(int i=0; i<4; i++) if( fMaps[i] == NULL ) { fMaps[i] = aMVar; break; }
  if( !GetF(FisImport) ) { 
    for(Pix p = fConnectionList.first(); p; fConnectionList.next(p) ) {
      TTemporalVariable& aVar = (TTemporalVariable&) fConnectionList(p); 
      aVar.Map(aMVar);
    }
  }
}

float* TTemporalVariable::Data() { return (float*) fHistory.TimeSeriesData(); }

int TTemporalVariable::DataSize() { return  fHistory.Size(); }

float TTemporalVariable::Value(void) const {	
  return  fHistory.LastValue();
}

float& TTemporalVariable::Value(void) {
#ifdef DEBUG
  if( fHistory.length() == 0 ) gFatal( SName() + ": Error, accessing uninitialized variable.");
#endif
  return  fHistory.LastValue();
}

void TTemporalVariable::Update( float value ) {
  fHistory.Add( gTime(), value );
#ifdef DEBUG
  if( GetCInfo(kDebug) ) {
    DoBoundsCheck(value);
    PrintUpdateMsg(value,value,"UPDATE");
  }
#endif
}

float TTemporalVariable::Delay( float dt, float ival ) {

  float dTime = gTime()-dt;
  if( dTime <= 0.0 ) { return fHistory.LastValue(); }
  else if( dTime < fHistory.FirstTimeStamp() ) return ival;
  else {
    return fHistory.Value(dTime);
  }
}

void TTemporalVariable::RefreshData() {
  if( GetF(FisImport) ) { 
    TTemporalVariable* aVar = (TTemporalVariable*) fConnection;     
      fHistory.Add( aVar->fHistory.LastTimeStamp(), aVar->fHistory.LastValue() );
  }
}

void TTemporalVariable::PrintUpdateMsg(float value, float dv, char* label ) {

  if( GetCInfo(kDebug)>1 ) {
    sprintf(gMsgStr,"\n%s %s (%f) by %f -> %E",(const char*)fName,label,gTime(),dv,value); 
    gPrintLog();
    PrintFlags();
  }
}

int TTemporalVariable::DoBoundsCheck(float value) {
		switch(fBounds.Check(value)) {
      case 'd': 
				sprintf(gMsgStr,"\nBOUNDS ERROR var %s -> %E: ",(const char*)fName,value); gPrintLog();
				PrintFlags();
				gPrintLog();
				Env::debugger_stop_here();
				break;
      case 'p': 
				sprintf(gMsgStr,"\nBOUNDS ERROR var %s -> %E: ",(const char*)fName,value); gPrintLog();
				gPrintErr();
				TModel::I().Interrupt();
				break;
      case 'f':
				sprintf(gMsgStr,"\nBOUNDS ERROR var %s -> %E: ",(const char*)fName,value);  gPrintLog();
				gPrintLog(); 
				gFatal();
      default: 
				return 0;	       
		}
		return 1;
}

void TTemporalVariable::InitValue(float value) {
     fHistory.Add( gTime(), value );

#ifdef DEBUG
    if(GetCInfo(kDebug)) {
      DoBoundsCheck(value);
      PrintUpdateMsg(value,value,"INIT");
    }
#endif 
}

int  TTemporalVariable::Config( TConfigData& cd ) {
  const CString& cmd = cd.Cmd();
  int iarg; float farg;
  return TVariable::Config( cd );

}   

//========================================================================================
// 				CLASS TTemporalFluxVariable
//========================================================================================

TTemporalFluxVariable::TTemporalFluxVariable(char* name, int isConstant ) 
  : TTemporalVariable( name )
{
  memset(fFluxInfo,0,4);
  SetCInfo(kType,kFlux); 
  SetCInfo(kInitMode,kDefault); 
	SetF(FisConstant,isConstant,"TTemporalFluxVariable");
}

void TTemporalFluxVariable::Clamp() {
  float value = fHistory.LastValue();
    
  if( Origin() ) {
    TTemporalStateVariable* o = (TTemporalStateVariable*) Origin(); 
    if( o->Reserve() < value ) { 
      float dVal = ( fHistory.LastValue() = o->Reserve()); 
      o->Reserve() = 0;

#ifdef DEBUG
	if(GetCInfo(kDebug)) PrintUpdateMsg(value,dVal,"CLAMP");
#endif
    } else {
      o->Reserve() -= value;  
    }
  }
  if( Destination() ) { 
    TTemporalStateVariable* d = (TTemporalStateVariable*) Destination(); 
    d->Reserve() += value; 
  }
}


//========================================================================================
// 				CLASS TTemporalStateVariable
//========================================================================================

EConfigInfo TTemporalStateVariable::kUpdateMode = kCInfo4;

TTemporalStateVariable::TTemporalStateVariable(char* name ) 
  : TTemporalVariable( name )
{
  SetCInfo(kType,kState); 
  SetCInfo(kInitMode,kDefault); 
	SetF(FisConstant,False,"TTemporalStateVariable");
	fOrder = fCurrentOrder = 1;
}

float TTemporalStateVariable::Update(float value, float dt) {

  float ndt = dt;
  fCurrentOrder = Util::imin( fOrder, fIntegratorMemory.length()+1 );
  float newvalue;
  switch( fCurrentOrder ) {
    case 1:
      newvalue = fHistory.LastValue() + value*ndt;
      break;
    case  2:
      newvalue = fHistory.LastValue() + ( 1.5*value - 0.5*fIntegratorMemory.LastValue() )*ndt;
			fIntegratorMemory.Add( gTime(), value );
      break;
    case  3:
      newvalue = fHistory.LastValue() + ( 1.92*value - 1.33*fIntegratorMemory.LastValue() + .42*fIntegratorMemory.LastValueOffset(1))*ndt;
			fIntegratorMemory.Add( gTime(), value );
      break;
    case  4:
      newvalue = fHistory.LastValue() + ( 2.29*value - 2.46*fIntegratorMemory.LastValue() + 1.54*fIntegratorMemory.LastValueOffset(1) - .375*fIntegratorMemory.LastValueOffset(2))*ndt;
			fIntegratorMemory.Add( gTime(), value );
      break;
  };
	if( GetF(FisClamped) ) { newvalue =  (newvalue < 0) ? 0 : newvalue; }
  fHistory.Add( gTime(), newvalue );
#ifdef DEBUG
  if(GetCInfo(kDebug)) {
    DoBoundsCheck(value);
    PrintUpdateMsg(value,value,"UPDATE_SV");
  }
#endif
  return ndt;
}

int  TTemporalStateVariable::Config( TConfigData& cd ) {
  int rv = TTemporalVariable::Config( cd );
	if( fIntegrationMethod.length() > 1 && isdigit(fIntegrationMethod[1]) ) {
		fOrder = fIntegrationMethod[1] - '0';
	}
	return rv;
}      

void TTemporalStateVariable::AddFlux( TVariable* f ) { fFluxList.Insert(*f); }

void TTemporalStateVariable::InitValue(float value) {
	fHistory.Add( gTime(), value );

#ifdef DEBUG
    if(GetCInfo(kDebug)) {
      DoBoundsCheck(value);
      PrintUpdateMsg(value,value,"INIT");
    }
#endif 
}

//========================================================================================
// 				CLASS TTemporalAuxVariable
//========================================================================================

TTemporalAuxVariable::TTemporalAuxVariable(char* name, int isConstant ) 
  : TTemporalVariable( name )
{
  SetCInfo(kType,kAux); 
  SetCInfo(kInitMode,kDefault); 
	SetF(FisConstant,isConstant,"TTemporalAuxVariable");
}

//========================================================================================
// 				CLASS TDBParmVariable
//========================================================================================

TDBaseSocket::TDBaseSocket(const char* name) : TNamedObject(name) { 
	sme_pg95* db = TModel::I().DataBase();
	if(db) {
		const CString& port = TModel::I().DataBase()->port();
		const CString& host = TModel::I().DataBase()->host();
		fDBase = new sme_pg95(name,0,port,host); 
	} else {
		fDBase = new sme_pg95(name); 	
	}
}

EConfigInfo TDBParmVariable::kDBOrder = kCInfo4;

TDBParmVariable::TDBParmVariable( char* name, int order ) 
  : TVariable ( name )
{
  SetCInfo(kType,kData);
  SetCInfo(kInitMode,kDBase); 
  SetCInfo(kDBOrder,order);
} 


float* TDBParmVariable::GetDBaseData(const char *filename, int sIndex, int pIndex, int index, int order, int& size ) {
  gPrintLog("Doing ReadDBase"); 
  if( gGlobalFileSystem ) return HParm::Read( filename, sIndex, pIndex, index, order, size );
  else {
    int root = 0;
    float* data=NULL;
    if(  gIProc == 0  ) {
      data = HParm::Read( filename, sIndex, pIndex, index, order, size );
      if(data == NULL) return 0;
    }
    if(gNProc > 1) {
#ifdef USE_MPI
      MPI_Bcast( &size, 1, MPI_INT, root, MPI_COMM_WORLD );
      if(size>0) { 
				if(data == NULL) { data = new float[size]; }
				MPI_Bcast( data, size, MPI_FLOAT, root, MPI_COMM_WORLD );
      }
#endif
    }
    return data;
  }
}
    
void TDBParmVariable::ReadDBase(const char* fileName, int sIndex, int cellIndex, int index ) {
	CPathString cpath(Env::DataPath()); cpath.Add(fileName);
  int len; float* data;
  data = GetDBaseData(cpath, sIndex, cellIndex, index, GetCInfo(kDBOrder), len);
  fV.setData(data,len);
  for( Pix p = fConnectionList.first(); p; fConnectionList.next(p))  {
    TDBParmVariable& aVar = (TDBParmVariable&) fConnectionList(p);
    aVar.fV.shallow_copy(fV);
  }	
  if(GetCInfo(kDebug)) {
    char ctmp[50]; CString s;
    sprintf(gMsgStr,"\nINIT DBase Parm%s -> ", (const char*)fName); 
    for(int i=0; i<fV.Size(); i++) { sprintf(ctmp,"%f ",fV.elem(i)); s += ctmp; }
    gPrintLog(s.chars());
  }

}

const int kMaxCols = 10;

#ifdef POSTGRESS

void TDBParmVariable::ReadDBase( const char* dataBase, const pgSQLRec* sqlRec, const ArgArray& args ) {
	sme_pg95* db = GetDBase(dataBase);
	db->exec_sql( *sqlRec, args );
	int nt = db->GetNTuples();
	CString colVal[kMaxCols];
	int ival = sqlRec->ValueFieldIndex();
	int mval = sqlRec->MapFieldIndex(0);
	CString val;
	for( int i=0; i<nt; i++ ) {
		val = db->GetTuples(i);
		val.splitColumns(colVal,kMaxCols);
		int iv = atoi(colVal[mval]);
		float fv = atof(colVal[ival]);
		if( (iv<0) || (iv>=MAX_NHAB) ) { sprintf(gMsgStr,"Illegal Index in %s :: ReadDBase",Name()); gPrintErr(); }
		else { fV.add(iv) = fv; }
	}
  for( Pix p = fConnectionList.first(); p; fConnectionList.next(p))  {
    TDBParmVariable& aVar = (TDBParmVariable&) fConnectionList(p);
    aVar.fV.shallow_copy(fV);
  }	
  if(GetCInfo(kDebug)) {
    char ctmp[50]; CString s;
    sprintf(gMsgStr,"\nINIT DBase Parm%s -> ", (const char*)fName); 
    for(int i=0; i<fV.Size(); i++) { sprintf(ctmp,"%f ",fV.elem(i)); s += ctmp; }
    gPrintLog(s.chars());
  }
}
#endif

float TDBParmVariable::Value( float classVal1, float classVal2 ) {

  int offset = 0;
  int vIndex1 = (int)classVal1;
  int vIndex2 = (int)classVal2;
  int vIndex = vIndex1*offset + vIndex2;
  if( vIndex >= fV.Size()|| vIndex2 >= fV.Size() ) { 
    sprintf(gMsgStr,"DBParm::Value Memory fault for %s: indices = %d,%d, size = %d\n",(const char*)fName,vIndex1,vIndex2,fV.Size()); gFatal();
  }	
  return fV.elem(vIndex);
}

float TDBParmVariable::DBParmValue( int vIndex ) { 
	float rv = fV.elem(vIndex);
	if( rv == kfloatVecError ) {
		sprintf(gMsgStr,"Data structures initialized improperly in DBParmVariable %s", fName.chars() );
		gFatal();
	}
	return rv; 
}

//----------------------------------------------------------------------------------------
//						TParmRec
//----------------------------------------------------------------------------------------       
       
TParmRec::TParmRec() {
  fUStep = 0;
  fModStep = -1;
}
     
void TParmRec::Setup(float value, float increment, int step, int modStep) {
  fValue = value;
  fIncrement = increment;
  fStepLength = step;
  fModStep = modStep;  
  if( fModStep < 0 || fStepLength < 0 ) {
    sprintf(gMsgStr,"\nTParmRec Error: Illegal ModStep/StepLength Value: %f\n",fValue); gPrintErr();
  }
}

TParmRec& TParmRec::operator = ( TParmRec& p ) {
  fValue = p.fValue;
  fIncrement = p.fIncrement;
  fStepLength = p.fStepLength;
  fModStep = p.fModStep;  
  fUStep = 0;
  return *this;
}
    
float TParmRec::Value() const {
  float rv;
  if( fModStep < 0 ) {
    sprintf(gMsgStr,"\nError, Accessing UInitialized Parameter: %f",fValue); gPrintErr();
  }
  else if( fModStep > 0 ) {  
    if( fStepLength>0 ) 
      rv = ( fValue + fIncrement*( (fUStep/fStepLength) % fModStep ) );
    else  rv =  fValue;
  }
  else if( fStepLength>0 )  
    rv = ( fValue + fIncrement*( fUStep/fStepLength ) );
  else  rv =  fValue;
  return rv;
}

//========================================================================================
// 				CLASS TParmVariable
//========================================================================================

TParmVariable::TParmVariable( char* name ) 
  : TVariable ( name )
{
  SetCInfo(kType,kData);
  SetCInfo(kInitMode,kParameter); 
  SetF(FisConstant,True,"TParmVariable");
	SetF(FisSpatial,False,"TParmVariable");
}  
    
void TParmVariable::InitValue() {
  fValue = fParm.Value();
  if(GetCInfo(kDebug)) {
    sprintf(gMsgStr,"\nINIT PARM %s -> %f",(const char*)fName,fValue); gPrintLog();
  }
}

void TParmVariable::Setup(float value, float increment, int step, int modStep) { 
  fParm.Setup(value,increment,step,modStep);
  
//   This functionality handled by RefreshData method:
//
//  if( GetF(FisExport ) ) {
//    for( Pix p = fConnectionList.first(); p; fConnectionList.next(p))  {
//      TParmVariable& aVar = (TParmVariable&) fConnectionList(p);
//      aVar.fParm = fParm;
//      aVar.fValue = aVar.fParm.Value();
//    }
//  } else 

  if( GetF(FisImport ) ) { ((TParmVariable*)fConnection)->Setup(value,increment, step, modStep); }
  fValue = fParm.Value();
}

#ifdef POSTGRESS

void TParmVariable::ReadDBase( const char* dataBase, const pgSQLRec* sqlRec, const ArgArray& args ) {
	sme_pg95* db = GetDBase(dataBase);
	db->exec_sql( *sqlRec, args );
	int nt = db->GetNTuples();
	CString value, colVal[kMaxCols];
	int ival = sqlRec->ValueFieldIndex();
	CString val;
	for( int i=0; i<nt; i++ ) {
		val = db->GetTuples(i);
		val.splitColumns(colVal,kMaxCols);
		fValue = atof(colVal[ival]);
	}
  for( Pix p = fConnectionList.first(); p; fConnectionList.next(p))  {
    TParmVariable& aVar = (TParmVariable&) fConnectionList(p);
    aVar.fValue = fValue;
  }	
  if(GetCInfo(kDebug)) {
    char ctmp[50]; CString s;
    sprintf(gMsgStr,"\nINIT Parm %s -> ", (const char*)fName); 
		sprintf(ctmp,"%f ",fValue); s += ctmp;
    gPrintLog(s.chars());
  }
}

#endif

int  TParmVariable::Config( TConfigData& cd ) {
  TVariable::Config( cd );
  const CString& cmd = cd.Cmd();
  int argc=0; float pval[5];
	if( cmd == "pm" ) {
    cd.FloatArg(0,pval[0],CD::kRequired );
    for(int i=1; i<4; i++) if( cd.FloatArg(i,pval[i]) <= 0 ) pval[i] = 0;
    Setup(pval[0],pval[1],(int)pval[2],(int)pval[3]);
		if(gDebug) { sprintf(gMsgStr,"\n%s Parm Config: pm(%f,%f,%f,%f)",(const char*)fName,pval[0],pval[1],pval[2],pval[3]); gPrintLog(); } 	
  } else return 0;
  return 1;
}

//========================================================================================
// 				CLASS TSeriesVariable
//========================================================================================

TSeriesVariable::TSeriesVariable( char* name ) 
  : TTemporalVariable ( name )
{
  SetCInfo(kType,kData);
  SetCInfo(kInitMode,kTimeSeries); 
}  


void TSeriesVariable::Read( const char* fileName, int format ) {
  if( format == 2 ) {
    HDFM::Open(HDFM::kFInput);
    fHistory.DSetName(fileName);
    fHistory.SDRead();
  } else {
    CPathString name(Env::DataPath()); name.Add(fileName);
    fHistory.Import( name, format );
  }
}

float TSeriesVariable::Delay( float dt, float ival ) {

  float dTime = gTime()-dt;
  if( dTime <= 0.0 ) { return fHistory.LastValue(); }
  else if( dTime < fHistory.FirstTimeStamp() ) return ival;
  else {
    return fHistory.Value(dTime);
  }
}

int  TSeriesVariable::Config( TConfigData& cd ) { return TTemporalVariable::Config( cd ); }

//========================================================================================
// 				CLASS TPtSeriesVariable
//========================================================================================

TPtSeriesVariable::TPtSeriesVariable( char* name ) 
  : TSpatialVariable ( name )
{
  SetCInfo(kType,kData);
  SetCInfo(kInitMode,kPtTimeSeries); 
}  

void TPtSeriesVariable::Read( const char* fileName, int format ) {
  if( format == 2 ) HDFM::Open(HDFM::kFInput);
  fHistoryList.SeriesName(fileName);
  fHistoryList.Read(Grid(),format);
}

void TPtSeriesVariable::RefreshData() {
  if( GetF(FisImport) )  TSpatialVariable::RefreshData();
  else fHistoryList.CreatePointMap( Cov(), gTime() );
}

int  TPtSeriesVariable::Config( TConfigData& cd ) { return TSpatialVariable::Config( cd ); }

*/



