//----------------------------------------------------------------------------------------
//	SSVariable.h
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//	Change History:
//----------------------------------------------------------------------------------------

#ifndef __SSVariable__
#define __SSVariable__

#include "Globals.h"
#include "Bounds.h"
#include "Coverage.h"
#include "SList.h"
#include "Series.h"
#include "TMap.h"
#include "MML_Variable.h"
#include "MML_Time.h"
#include "CnfgProcessor.h"
#include "Utilities.h"
#include "sme_postgres.h"

typedef OrderedPoint GridPoint;
typedef class TModule TModule; 
typedef class TSpatialVariable TSpatialVariable;

#ifdef POSTGRESS

class TDBaseSocket : public TNamedObject {
	sme_pg95* fDBase;	  
public:
	TDBaseSocket(const char* name);
	sme_pg95* DBase() { return fDBase; }
};

#endif

//----------------------------------------------------------------------------------------
//						TVariable
//----------------------------------------------------------------------------------------


class  TVariable : public Variable {	
 
protected:

	static TNamedObjectList  fDBaseList;
  	
public:

  static byte kNoDbgCnfg;

  enum EInfo { kFrameType, kOutMode, kRemDataMap };
  
  TVariable(const char* name);
      
    virtual int Config( TConfigData& config );
    virtual int Allocate() { return 1; }
    virtual void Dump( FILE* file ) {;}
		virtual void SetupInternals();
		
    virtual inline float Value() const { return 0.0;}
    virtual inline float Value( Pix p ) const { return 0.0; }
		inline virtual void UpdateInternals() { Variable::UpdateInternals(); }
		inline void SetLinkEdges() { SetF(FLinkEdges,True,"SetLinkEdges"); }
#ifdef POSTGRESS
		sme_pg95* GetDBase(const char* name);
#endif

    virtual void AddFlux( TVariable* f ) {;}
    inline virtual float& Reserve() { return fGarbage; }

    inline float GenericValue( Pix p=0 ) const {
      if( GetF(FisSpatial) ) return Value( p );
      else return Value();
    }
    virtual void AddFlags();   
    virtual void Reset() {;} 
    
    virtual Pix SSInit( Module* mod ) {
      Pix p0 = Variable::Init( mod );
      return p0;
    }
     
};

//  TVariable Flags

 extern unsigned int	FFluxOriginMap ;		//  Flux Origin has different Grid
 extern unsigned int	FFluxDestinationMap ;		//  Flux Destination has different Grid
 extern unsigned int	FMapRemoteData ;		//  Input Var and Connection have different Grids
 extern unsigned int	FICFlux  ;		//  calls ICFlux method
 extern unsigned int	FAllocated ;		//  Allocate() Method has been called.
 extern unsigned int	FEdgesLinked;		//  LinkEdges() Method has been called since last update.

/* 
//----------------------------------------------------------------------------------------
//						TSpatialVariable
//----------------------------------------------------------------------------------------


  class  TSpatialVariable : public TVariable  {	
    
  protected:

    PointSetCoverage fCov;    
    float fMaxDelay;
    intPixVHMap* fCatMapPtr;

    int fBuffsize;
    TVariable* fMaps[4];
    TObjectList fPSetList; 
    TSpatialSeries fHistory;   

    int DoBoundsCheck( Pix p, float value );
    void PrintUpdateMsg( Pix p, float value, float dv, char* label, Pix pR=0 );
           
  public:
       
    TSpatialVariable(char* name);
    ~TSpatialVariable() {;}
    void Connect( TSpatialVariable* aVar ) { MakeConnection(aVar); }
    void Map( TSpatialVariable* aVar );
		inline TMap2* RegionMap() { return fCov.RegionMap(); }
		inline void SetCategories( intPixVHMap& catmap ) { fCatMapPtr = &catmap; }
		inline intPixVHMap* GetCategories() { return fCatMapPtr; }
	
    virtual int Allocate();
    inline int Allocate( float ival ) { int rv = Allocate(); if(rv) { Set(ival); } return rv; }

    inline Bool Allocated() { return GetF(FAllocated); }

    void BroadcastValue();

    void UpdateRel( Pix p0, Point2 rp, float value );

    float RelValue( Pix p0, Point2 rp );

    float RelValue( Pix p0, int i0, int i1 ) { 
      return RelValue( p0, Point2(i0,i1) ); }

    float Delay( Pix p, float dTime, float ival=0.0 );
       //  Warning: Increasing delayTime during simulation can result in initval being returned.
    float InitVal(Pix p);              
    float* InitValData();              

    virtual int Config(TConfigData& config);
    
    int  SetDataValues( float* data, int layOut, int size );

    inline void RefreshData() {
      if( GetF(FMapRemoteData) ) { 
				TSpatialVariable* c = (TSpatialVariable*)fConnection;
				fCov = c->fCov;
      }
    }

    inline PointSetCoverage& Cov() { return  fCov; }
    inline DistributedGrid& Grid() { return fCov.Grid(); }
    inline int SetValue( Pix p, float value ) { return fCov.SetValue( p, value ); }
    inline int AddValue( Pix p, float value ) { return fCov.AddValue( p, value ); }
    inline Region2& Region(int iproc=gIProc ) const { return fCov.Region(iproc); } 
    inline Region2& GlobalRegion( ) const { return fCov.GlobalRegion(); } 
    inline void CopyToMap(  TMap2& map ) { fCov.CopyToMap( map ); }           // Convert to map;
    inline float RemoteValue( Pix p, int recvProc=0 ){ return fCov.RemoteValue( p,recvProc); } // p should be non-zero on only one proc!;
    inline Pix GetPix ( Point2& point ) const { return fCov.GetPix ( point ); }
    inline void Set(float value) { fCov.Set(value); }
    inline float* DataStart() { return  fCov.DataStart(); }
    inline const float* DataEnd() { return  fCov.DataEnd(); }
		inline const float* GridEnd() { return  fCov.GridEnd(); }
		virtual float* Data() { return  fCov.DataStart(); }
    virtual int DataSize() { return  fCov.Size(); }
		inline void GrabMemory() { fCov.GrabMemory(); }
		inline int CheckMemory() { return fCov.CheckMemory(); }

    virtual inline TSpatialVariable* Connection() { if(fConnection) return (TSpatialVariable*) fConnection; else return this; }

    inline int InitValue(Pix p, float value) {
      int rv = SetValue(p,value);  
#ifdef DEBUG
      if( GetCInfo(kDebug) ) {
				DoBoundsCheck(p,value);
				PrintUpdateMsg(p,Value(p),value,"INIT");
			}
#endif
			return rv;
    }
    
		inline int LinkEdges() { 
			if( GetF(FEdgesLinked) ) return 0;
			else { 
				fCov.LinkEdges( Name() ); 
				SetF(FEdgesLinked,True,"LinkEdges"); 
				return 1;
			} 
		}
		
		
		void CopyData(TSpatialVariable& v);			
		void AddData(TSpatialVariable& v);
				
    virtual void UpdateInternals();

    virtual Pix SSInit( Module* mod );
    
		inline TSpatialVariable* GetSimilarVariable(char* name) {
			TSpatialVariable* rv = new TSpatialVariable(name);
			rv->SSInit(fModule);
			rv->Allocate();
			return rv; 
		}
		
    inline void Update( Pix p, float value ) {
      fCov.SetValue( p, value ); 
#ifdef DEBUG
      if( GetCInfo(kDebug) ) {
				DoBoundsCheck(p,value);
				PrintUpdateMsg(p,value,Value(p),"UPDATE");
      }
#endif
    }
    inline float Value( Pix p) const {
      return fCov.Value( p );
    }
    
    inline float& operator() ( const OrderedPoint& pt ) {
      return fCov.operator()( pt );
    }    
    inline int ClassValue( Pix p) const {
      return (int) fCov.Value( p );
    }
    inline float& Value( Pix p ) {
      return fCov.operator()( p );
    }
    inline int ClassValue( Pix p) {
      return (int) fCov.operator()( p );
    }

};



//----------------------------------------------------------------------------------------
//						TSpatialStateVariable
//----------------------------------------------------------------------------------------
typedef class TSpatialFluxVariable TSpatialFluxVariable;


class  TSpatialStateVariable : public TSpatialVariable {
    
protected:
  PointSetCoverage fReserve;
	TSpatialSeries fIntegratorMemory; 
	int fCurrentOrder;  

public:
  static EConfigInfo kUpdateMode;

  TSpatialStateVariable(char* name);
	~TSpatialStateVariable() {;}
  virtual int Allocate();
  inline void SetUpdateMode(PointSetCoverage::EUpdateMode upDateMode) {  SetCInfo(kUpdateMode,upDateMode); }
  inline PointSetCoverage::EUpdateMode GetUpdateMode() {   return (PointSetCoverage::EUpdateMode) GetCInfo(kUpdateMode); }
  inline void Connect( TSpatialStateVariable* aVar ) { MakeConnection(aVar); }
  inline void SetMap ( TSpatialStateVariable* aMVar ) { Map( aMVar ); }
  inline float& Reserve(Pix p) { return fReserve(p); }
  inline float* RDataStart() { return  fReserve.DataStart(); }
 
	inline virtual void UpdateReserve() {
		float* data_Rptr = ( GetF(FisClamped) ) ? fReserve.DataStart() : (float*)NULL;
		if(data_Rptr) {
			float* data_ptr = DataStart();
			const float* data_Eptr = DataEnd();
			while( data_ptr < data_Eptr ) {
				*data_Rptr++ = *data_ptr++;
			}	
		} 		
	}
 
  void AddFlux( TVariable* f );

  inline float Update( Pix p, float value, float dt ) {
    float ndt = dt;
    int index = Grid().Index(p);
		float& ftmp = fCov.elem(index);
    switch( fCurrentOrder ) {
    case 1:
      ftmp += value*dt;
      break;
    case 2:
			fIntegratorMemory.LastValue(index) = value;
      ftmp = ftmp + ( 1.5*value - 0.5*fIntegratorMemory.LastValueOffset(1,index) )*dt;
      break;
    case  3:
			fIntegratorMemory.LastValue(index) = value;
      ftmp = ftmp+ ( 1.92*value - 1.33*fIntegratorMemory.LastValueOffset(1,index) + .42*fIntegratorMemory.LastValueOffset(2,index))*ndt;
      break;
    case  4:
			fIntegratorMemory.LastValue(index) = value;
      ftmp = ftmp + ( 2.29*value - 2.46*fIntegratorMemory.LastValueOffset(1,index) + 1.54*fIntegratorMemory.LastValueOffset(2,index) - .375*fIntegratorMemory.LastValueOffset(3,index))*ndt;
      break;
    }
    if( GetF(FisClamped) ) { 
			if( ftmp < 0 ) { ftmp = 0; }
			fReserve.elem(index) = ftmp; 
		}
#ifdef DEBUG
    if( GetCInfo(kDebug) ) {
      DoBoundsCheck(p,Value(p));
      PrintUpdateMsg(p,value,Value(p),"UPDATESV");
    }
#endif
    return ndt;
  }

  inline void InitValue(Pix p, float value) {
    int index = TSpatialVariable::InitValue( p, value );
    if( GetF(FisClamped) ) { fReserve.elem(index) = value; }	
  }

  virtual Pix SSInit( Module* mod );
	virtual int Config(TConfigData& config);
	
	virtual void UpdateInternals();
    	
	inline int InitializeIntegrator() {  int rv=0; 
		if( fOrder > 1 ) { 
			fIntegratorMemory.grow_high(rv);
		  fCurrentOrder = Util::imin( fOrder, fIntegratorMemory.length() + 1 );
		}
		return rv; 
	}
};

//----------------------------------------------------------------------------------------
//						TSpatialFluxVariable
//----------------------------------------------------------------------------------------

enum EFluxInfo { kF_ClampMsg };

class  TSpatialFluxVariable : public TSpatialVariable {
    
  protected:
    byte fFluxInfo[4];

  public:
    TSpatialFluxVariable(char* name, int isConstant = False);
    ~TSpatialFluxVariable() {;}
    virtual int Allocate();
    void Clamp( Pix p );
    void ICFlux( Pix p0, Point2 rp, float value );

    inline void SetDestination(TSpatialStateVariable* aVar) { 
      Variable::SetDestination(aVar); aVar->AddFlux(this); 
    }
    inline void SetOrigin(TSpatialStateVariable* aVar) { 
      Variable::SetOrigin(aVar); aVar->AddFlux(this); 
    }

    inline void Connect( TSpatialFluxVariable* aVar ) { MakeConnection(aVar); }
    inline void SetMap ( TSpatialFluxVariable* aMVar ) { Map( aMVar ); }

    inline void Update( Pix p, float value ) {
      TSpatialVariable::Update( p, value );
#ifdef DEBUG
      if( GetF(FisClamped) && value<0 && fFluxInfo[kF_ClampMsg]==0 ) {
				sprintf(gMsgStr,"Negative Fluxes not allowed for clamped SVars: %s",(const char*)fName); gPrintErr();
				fFluxInfo[kF_ClampMsg]=1;
      }
#endif
    }

    virtual inline Pix SSInit( Module* mod ) {
      Pix p0 = TSpatialVariable::SSInit( mod );
      SetF(FisFlux,True,"Init");
      return p0;
    }

  };

//----------------------------------------------------------------------------------------
//						TSpatialAuxVariable
//----------------------------------------------------------------------------------------

class  TSpatialAuxVariable : public TSpatialVariable {
protected:
public:
  TSpatialAuxVariable(char* name, int isConstant = False);
  ~TSpatialAuxVariable() {;}
  virtual int Allocate( );
  inline void Connect( TSpatialAuxVariable* aVar ) { MakeConnection(aVar); } 
  inline void SetMap ( TSpatialAuxVariable* aMVar ) { Map( aMVar ); }
};

//----------------------------------------------------------------------------------------
  //						TTemporalVariable
  //----------------------------------------------------------------------------------------
  
class  TTemporalVariable : public TVariable  {
    
  protected:

    TTemporalVariable* fMaps[4];
		TTemporalSeries fHistory;   
    
    void PrintUpdateMsg(float value, float dv, char* label ); 
    int DoBoundsCheck(float value);

  public:

   TTemporalVariable( char* name );
   ~TTemporalVariable() {;}
 
   void Map ( TTemporalVariable* aMVar );

   float Value(void) const;	
   float& Value(void);	

   inline float Time() { return  fHistory.LastTimeStamp(); }

   inline void Connect( TTemporalVariable* aVar ) { MakeConnection( aVar ); }
   
   inline float& operator() () {
     return fHistory.LastValue();
   }

   inline float LastValue() { return fHistory.LastValue(); }

   inline float InitVal( ) {
     return fHistory.FirstValue();
   }
   void RefreshData();

   float Delay( float dTime, float ival=0.0 );
 
   void Update( float value );

    virtual int ExecuteIgnoreFunction()  { 
			float value = atof(fDefValue.elem(0));
			Update( value ); 
			return 0;
		}

   virtual int Config( TConfigData& config );

   void InitValue(float value);
   
   virtual float* Data();
   virtual int DataSize();

  };



//----------------------------------------------------------------------------------------
//						TTemporalStateVariable
//----------------------------------------------------------------------------------------

typedef class TTemporalFluxVariable TTemporalFluxVariable;

class  TTemporalStateVariable : public TTemporalVariable {
    
protected:
  float fReserve;
	TTemporalSeries fIntegratorMemory; 
	int fCurrentOrder;
	
public:

  static EConfigInfo kUpdateMode;

  TTemporalStateVariable(char* name);
  inline void SetUpdateMode(Cov::EUpdateMode upDateMode) {  SetCInfo(kUpdateMode,upDateMode); }
  inline Cov::EUpdateMode GetUpdateMode() {  return (Cov::EUpdateMode) GetCInfo(kUpdateMode); }
  inline void Connect( TTemporalStateVariable* aVar ) { MakeConnection( aVar ); } 
  inline void SetMap ( TTemporalStateVariable* aMVar ) { Map( aMVar ); }
  float Update( float value, float dt );
  inline float& Reserve() { return fReserve; }
  void AddFlux( TVariable* f );
  inline void SetOrder(int o) { fOrder = o; }

  virtual inline Pix SSInit( Module* mod ) {
    Pix p0 = TVariable::SSInit( mod );
    SetF(FisStateVar,True,"Init");
    return p0;
  }
	virtual int Config(TConfigData& config);
	
	void InitValue(float value);
};

//----------------------------------------------------------------------------------------
//						TTemporalFluxVariable
//----------------------------------------------------------------------------------------


class  TTemporalFluxVariable : public TTemporalVariable {
    
  protected:
    byte fFluxInfo[4];

  public:
    TTemporalFluxVariable(char* name, int isConstant = False );
    void Clamp();

    inline void SetDestination(TVariable* aVar) { 
      Variable::SetDestination(aVar); aVar->AddFlux(this); 
    }
    inline void SetOrigin(TVariable* aVar) { 
      Variable::SetOrigin(aVar); aVar->AddFlux(this); 
    }

    inline void Connect( TTemporalFluxVariable* aVar ) { MakeConnection( aVar ); }
    inline void SetMap ( TTemporalFluxVariable* aMVar ) { Map( aMVar ); }

    inline void Update( float value ) {
      TTemporalVariable::Update( value );
#ifdef DEBUG
      if( GetF(FisClamped) && value<0 && fFluxInfo[kF_ClampMsg]==0 ) {
				sprintf(gMsgStr,"Negative Fluxes not allowed for clamped SVars: %s\n",(const char*)fName); gPrintErr();
				fFluxInfo[kF_ClampMsg]=1;
      }
#endif
    }
    virtual inline Pix SSInit( Module* mod ) {
      Pix p0 = TVariable::SSInit( mod );
      SetF(FisFlux,True,"Init");
      return p0;
    }
  };

//----------------------------------------------------------------------------------------
//						TTemporalAuxVariable
//----------------------------------------------------------------------------------------

class  TTemporalAuxVariable : public TTemporalVariable {
protected:
public:
  TTemporalAuxVariable(char* name, int isConstant = False );
  inline void Connect( TTemporalAuxVariable* aVar ) { MakeConnection( aVar ); } 
  inline void SetMap ( TTemporalAuxVariable* aMVar ) { Map( aMVar ); }
};

//----------------------------------------------------------------------------------------
//						TDBParmVariable
//----------------------------------------------------------------------------------------

class TDBParmVariable : public TVariable {

protected:

	floatVec fV;
  float* GetDBaseData(const char *filename, int sIndex, int pIndex, int index, int order, int& size ); 

public:

  static EConfigInfo kDBOrder;

  TDBParmVariable( char* name, int order=1 );

  void ReadDBase(const char* fileName, int sIndex, int cellIndex, int index);
#ifdef POSTGRESS
  void ReadDBase(const char* dataBase, const pgSQLRec* sqlRec, const ArgArray& args );
#endif
  void Connect( TDBParmVariable* aVar ) { MakeConnection( aVar ); } 
  inline float Value( int vIndex ) const { return fV[vIndex]; }
 // inline float DBParmValue( int vIndex ) const { return fV.elem(vIndex); }
  float DBParmValue( int vIndex );
  float Value( float classVal1, float classVal2 );
  virtual int Config( TConfigData& config ) { return TVariable::Config(config); }  
  inline float* DataStart() { return fV.DataStart(); }  
  virtual float* Data() { return  fV.DataStart(); }
	virtual int DataSize() { return  fV.capacity(); }

};

//----------------------------------------------------------------------------------------
  //						TParmRec
  //----------------------------------------------------------------------------------------
  
  class TParmRec {
    
    float fValue;
    float fIncrement;
    int fStepLength;
    int fModStep;
    int fUStep;
    
  public:
    
    TParmRec();
    void Setup(float value, float increment=0, int step=0, int modStep=0);
    float Value() const;
    TParmRec& operator = ( TParmRec& p );
    inline int Initialized() { return( fModStep >= 0 ); }
  };

//----------------------------------------------------------------------------------------
//						TParmVariable
//----------------------------------------------------------------------------------------

const int kPVMaxArgs = 10;

class TParmVariable : public TVariable {

protected:

  TParmRec fParm;
  float fValue;
  float fData[5];

public:

  TParmVariable(char* name );

  void InitValue();
  void Setup(float value, float increment=0, int step=0, int modStep=0);
  void Connect( TParmVariable* aVar ) { MakeConnection( aVar ); } 

	virtual inline void RefreshData() {
		fParm = ((TParmVariable*)fConnection)->fParm;
		fValue = ((TParmVariable*)fConnection)->fValue;
	}
  inline float Value() const { return fValue; }
  inline float& Value() { return fValue; }
  virtual int Config( TConfigData& config );
  virtual float* Data() { fData[0] = fValue; return fData; }
  virtual int DataSize() { return  1; }
#ifdef POSTGRESS
	void ReadDBase( const char* dataBase, const pgSQLRec* sqlRec, const ArgArray& args ) ;
#endif
};

//----------------------------------------------------------------------------------------
//						TSeriesVariable
//----------------------------------------------------------------------------------------

class  TSeriesVariable : public TTemporalVariable  {	
    
  protected:

 TDistributedSeries fHistory;

  public:

   TSeriesVariable( char* name );
   void Read( const char* fileName, int format );
	virtual void Reset() { fHistory.Reset(); }

   void Connect( TSeriesVariable* aVar ) { MakeConnection( aVar ); } 
   inline void SetMap ( TSeriesVariable* aMVar ) { Map( aMVar ); }

   inline float Value() const {	
     return fHistory.LastValue();
   }

   inline float Time() { return fHistory.LastTimeStamp(); }
   
   inline float& operator() () {
     return fHistory.LastValue();
   }

   inline float InitVal( ) {
     return fHistory.FirstValue();
   }

   float Delay( float dTime, float ival );
   virtual int Config( TConfigData& config ); 

    virtual int Allocate() { return 1; }
    virtual void Dump( FILE* file ) {;}
    virtual float* Data() { return (float*)fHistory.TimeSeriesData(); }
    virtual int DataSize() { return  fHistory.length(); }
		inline float DT() { return fHistory.DT(); }
		inline int Length() { return fHistory.length(); }

  };


//----------------------------------------------------------------------------------------
//						TPtSeriesVariable
//----------------------------------------------------------------------------------------

class  TPtSeriesVariable : public TSpatialVariable  {	
    
  protected:

  TPtSeriesList fHistoryList;

   public:

   TPtSeriesVariable( char* name );
   void Read( const char* fileName, int format );
   virtual int Config( TConfigData& config ); 
   void RefreshData();
	virtual void Reset() { fHistoryList.Reset(); }

   virtual int Allocate( ) { return TSpatialVariable::Allocate(); }
   inline void Connect( TPtSeriesVariable* aVar ) { MakeConnection(aVar); } 
   inline void SetMap ( TPtSeriesVariable* aMVar ) { Map( aMVar ); }
   inline float DT() { return fHistoryList.DT(); }
	 inline int Length() { return fHistoryList.Size(); }
};

*/

typedef TVariable Var;

#endif
