//----------------------------------------------------------------------------------------
//	Series.cc
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//	Change History:
//----------------------------------------------------------------------------------------

#include "Series.h"
#include "SSModel.h"
#include "HDFMgr.h"

float sum( int istart, int nitems, TTemporalSeries& s );
float ave( int istart, int nitems, TTemporalSeries& s );


float sum( int istart, int nitems, TTemporalSeries& s ) {
  float tmp=0;
  for(int i=0; i<nitems; i++ ) tmp += s(i+istart);
  return tmp;
}

float ave( int istart, int nitems, TTemporalSeries& s ) {
  float tmp=0;
  for(int i=0; i<nitems; i++ ) tmp += s(i+istart);
  return tmp/nitems;
}

//========================================================================================
// 				CLASS TDistributedSeries
//========================================================================================


//----------------------------------------------------------------------------------------
//  TDistributedSeries:::Read
//----------------------------------------------------------------------------------------

void TDistributedSeries::Read( int nameIndex, int dataIndex, int clearOldData, int root ) {  
   // reads vdata sX from series vGroup fNameY
   //  X : dataIndex,  Y: nameIndex
   // dataIndex < 0 reads most recent vdata.
   // dataIndex > 0 reads consequtive vdatas (with each call) starting from S0.
  // nameIndex < 0 reads from series vGroup fName
	SetupComm();
  int timeIndex; float t0;							       
  if(dataIndex>0) timeIndex = ++fReadIndex;						
  else timeIndex = -1;
  int npt;
  if( root < 0 ) SDRead( dataIndex, -1, NULL, NULL, NULL   );
  else {
    float* data = NULL; int npt=0;
    if(  gIProc == root  ) {
      SDRead( dataIndex, -1, NULL, NULL, NULL   );
      npt = GetData( data );
    }
    if( gNProc > 1 ) {
#ifdef USE_MPI
      MPI_Bcast( &npt, 1, MPI_INT, root, fComm );
      if(npt>0) { 
	if(data == NULL) { data = new float[npt+1]; }
	MPI_Bcast( data, npt+1, MPI_FLOAT, root, fComm );
      }
#endif
    }
    fTS = data[npt];
    t0 = LastTimeStamp() + fTS;
    AddData( data, npt, t0, fTS );
    delete [] data;
  }
  if(clearOldData) DeleteBelow( t0 );
}

void TDistributedSeries::Write( char* info ) {
  if(  gIProc == 0  ) SDWrite( info );
}

void  TDistributedSeries::AddLocal( float time, float value, int root ) {
	SetupComm();
  if( gNProc > 1 ) {
    float data[2];
    data[0] = time; data[1] = value;
#ifdef USE_MPI
    MPI_Bcast( data, 2, MPI_FLOAT, root, fComm );
#endif
    Add(data[0],data[1]);
  } else Add(time,value);
}
  
void TDistributedSeries::Condense( int ostep, char mode, int ostart )
{ 
  if( fSeriesInfo[kTS_ConstantStep] == False ) return;
  int cnt = 0, i;
  if( ostep < 1 || ostart < 0 ) return;
  if( ostep == 1 && ostart == 0 ) return;
  TDistributedSeries newS;
  float t0 = FirstTimeStamp();
  switch( mode ) {
  case 'D' :
    for(i=ostart; i<Size(); i+=ostep ) { 
      float tdata = (*this)[low()+i]();
      float ctime = t0+i*fTS;
      newS.Add(ctime,tdata); 
    }
    break;
  case 'C' :
    for(i=ostart; i<=(Size()-ostep); i+=ostep ) { 
      float ctime = t0+i*fTS; 
      newS.Add(ctime,sum(low()+i,ostep,*this));
    }
    break;
  case 'A' :
    for(i=ostart; i<=(Size()-ostep); i+=ostep ) { 
      float ctime = t0+i*fTS; 
      newS.Add(ctime,ave(low()+i,ostep,*this));
    }
    break;
  }
}


//========================================================================================
// 				CLASS TLocalSeries
//========================================================================================

//----------------------------------------------------------------------------------------
//  TLocalSeries::AddData
//----------------------------------------------------------------------------------------

int TLocalSeries::AddData( float* data, int np, float t0, float ts, int addProc, int clearOldData ) 
{  
  fSeriesInfo[kTS_ConstantStep] = True;
  fTS = ts;
  if( fOwner == -1) fOwner = addProc;

  if( gIProc == addProc ) {
    if(clearOldData) clear(); 
    for(int i=0; i<np; i++) Add(t0+i*ts, data[i]);
  }
  return 1;
}
//----------------------------------------------------------------------------------------
//  TLocalSeries::MakeGhost
//----------------------------------------------------------------------------------------

void TLocalSeries::MakeGhost( int ghostProc ) 
{  
  Copy(fOwner,ghostProc);
}

//----------------------------------------------------------------------------------------
//  TLocalSeries:::Read
//----------------------------------------------------------------------------------------

int TLocalSeries::Read( int nameIndex, int dataIndex, char* info, int clearOldData, int rootProc ) {  
   // reads vdata sX from series vGroup fNameY
   //  X : dataIndex,  Y: nameIndex
   // dataIndex < 0 reads most recent vdata.
   // dataIndex > 0 reads consequtive vdatas (with each call) starting from S0.
  // nameIndex < 0 reads from series vGroup fName

  int timeIndex;
  if(dataIndex>0) timeIndex = ++fReadIndex;						
  else timeIndex = -1;
  int npt; float* data;
  if( rootProc < 0 ) rootProc = fOwner; 
  if(  gIProc == rootProc  ) {
    SDRead( dataIndex, -1, NULL, NULL, NULL   );
  }
  Move(rootProc,fOwner,info);
  if( clearOldData && gIProc == rootProc  ) { 
    float t0 = LastTimeStamp() + fTS;
    DeleteBelow( t0 );
  }
  return 1;
}

//----------------------------------------------------------------------------------------
//  TLocalSeries::Move
//----------------------------------------------------------------------------------------

int TLocalSeries::Move( int srcProc, int destProc, char* info, int deleteOld ) {
	SetupComm();
  int infosize = (info) ? strlen(info) : 0;
  infosize = ( infosize < kSeriesInfoSize ) ? infosize : kMapInfoSize;
  if(srcProc == destProc) return infosize;
  int dataLen = length();
  int tag =  fSeriesIndex*10 + kST_Move;
#ifdef USE_MPI
  TMsgBuff msgBuff(1000,fComm);
  if( gIProc == srcProc ) {
    msgBuff.PackInt(infosize);
    msgBuff.PackInt(dataLen);
    msgBuff.PackFloat(fTS);
    if( infosize > 0 ) msgBuff.PackString(info);
    if( dataLen > 0 ) { 
      for( int i = low(); i<=high(); i++ ) {
				TimeSlice& t = (TimeSlice&) (*this)[i];
				msgBuff.PackFloat(t.Time());
				msgBuff.PackFloat(t.Value());
      }
    }
    msgBuff.Send(tag,destProc);
    if( deleteOld ) clear();			
  }
  if( gIProc == destProc ) {
    msgBuff.Recv(tag,srcProc);
    infosize = msgBuff.UnPackInt();
    dataLen = msgBuff.UnPackInt();
    fTS = msgBuff.UnPackFloat();
    if( infosize > 0 ) { msgBuff.UnPackString(info,infosize); }
    if( dataLen > 0 ) {
      clear();
      for( int i=0; i<dataLen; i++ ) {
	float time = msgBuff.UnPackFloat();
	float value = msgBuff.UnPackFloat();
	Add(time,value);
      }
    }
  }
#endif
  return infosize;
}

void TLocalSeries::Write( char* info, int rootProc ) {
  if(rootProc<0) rootProc = fOwner;  
  Copy(fOwner,rootProc,info);
  if(  gIProc == rootProc  ) SDWrite( info );  
}

void TLocalSeries::Condense( int ostep, char mode, int ostart )
{ 
  if( fSeriesInfo[kTS_ConstantStep] == False ) return;
  if( gIProc != fOwner ) return;
  int cnt = 0, i;
  if( ostep < 1 || ostart < 0 ) return;
  if( ostep == 1 && ostart == 0 ) return;
  TLocalSeries newS;
  float t0 = FirstTimeStamp();
  switch( mode ) {
  case 'D' :
    for(i=ostart; i<Size(); i+=ostep ) { 
      float tdata = (*this)(low()+i);
      float ctime = t0+i*fTS;
      newS.Add(ctime,tdata); 
    }
    break;
  case 'C' :
    for(i=ostart; i<=(Size()-ostep); i+=ostep ) { 
      float ctime = t0+i*fTS; 
      newS.Add(ctime,sum(low()+i,ostep,*this));
    }
    break;
  case 'A' :
    for(i=ostart; i<=(Size()-ostep); i+=ostep ) { 
      float ctime = t0+i*fTS; 
      newS.Add(ctime,ave(low()+i,ostep,*this));
    }
    break;
  }
}


//----------------------------------------------------------------------------------------
// TPtSeries
//----------------------------------------------------------------------------------------;

TPtSeries::TPtSeries (CString name, int owner ) : TLocalSeries(name,owner), fPoint(-1,-1) {  fLayerIndex = -1; } 
TPtSeries::TPtSeries (int owner) : TLocalSeries(owner), fPoint(-1,-1) { fLayerIndex = -1;  } 
TPtSeries::TPtSeries (char* name, int owner ) : TLocalSeries(name,owner), fPoint(-1,-1) { fLayerIndex = -1;  } 

TPtSeries::~TPtSeries () {
  clear();
} 


int TPtSeries::Read (int nameIndex, int tIndex, int clearOldData, int rootProc )
{
  char info[ kAttrStrLen ];
  int dims[3];

  int infoLen = TLocalSeries::Read(nameIndex,tIndex,info,clearOldData,rootProc);
  if( infoLen == 0) return 0;
  Point2 p;
  GetPointInformation( nameIndex, p);
  SetPoint( p );
  return 1;
}

//----------------------------------------------------------------------------------------
// TPtSeriesList
//----------------------------------------------------------------------------------------

TPtSeriesList::TPtSeriesList(char* name) 
{
  fNptTS = -1;
  fSeriesName = name;
  fTS=fT0=fT1=0.0;
  fReadIndex = -1;
  fFormat = 0; fLayerIndex = -1;
  fUseReadIndex = 0;
	fDoneFirstRead = 0;
 fSiteMap = NULL;
}
TPtSeriesList::TPtSeriesList() 
{
  fNptTS = -1;
  fTS=fT0=fT1=0.0;
  fReadIndex = -1;
  fFormat = 0; fLayerIndex = -1;
  fUseReadIndex = 0;
	fDoneFirstRead = 0;
  fSiteMap = NULL;
}


TPtSeriesList::~TPtSeriesList() {;}

int TPtSeriesList::Read( int format, int deleteOld) 
{
  int index = 0, tst;
  if(format==0) format = fFormat;
  
  if( fUseReadIndex ) { fReadIndex++; }
  else { fReadIndex = ( ((int)TTime::Year()) % 100 ); }
  
  if(  fDoneFirstRead == 0 ) {
    if( format == 3 ) ReadHDF(deleteOld);
    else { 
      tst = Import( Env::DataPath(), fSeriesName.data(), format );
      if( tst <= 0 ) { gPrintErr( fSeriesName + ": Error reading PTS File\n"); return 0; } 
    }
    fDoneFirstRead = 1;
  } else {
    for( Pix p = fList.first(); p; fList.next(p) ){
      TPtSeries& ps = (TPtSeries&)fList(p);
      if( format == 3 ) { 
				if( !ps.Read(index++,fReadIndex,deleteOld) ) { return 0; }
      } else {
				CString path( fPTSDataPath.Path("/") );  
				path += ps.SeriesName(); 
				path.appendIndex(fReadIndex); path += ".ts";
				int rv =  ps.Import(path,format);
				if( rv == 0 ) return 0;
      }
    }
  }
  return 1;
}

int TPtSeriesList::ReadHDF( int deleteOld ) {
  int owner = 0, index = 0, cnt=0;
  while (1) {
    TPtSeries* ps = new TPtSeries(fSeriesName,owner);
    if( !ps->Read(index++,1,deleteOld) ) {
      delete ps;
      if(cnt) return 1;
      else return 0; 
    }
    if( Size() < 0) { fNptTS = ps->Size(); fTS = ps->TS(); }
    else if ( fNptTS != ps->Size() ) { 
      sprintf(gMsgStr,"Uneven timeseries length in %s, new %d vs %d",ps->SeriesName(),ps->Size(),fNptTS); gPrintScreen(); 
	} else if (  fTS != ps->TS() ) { 
      sprintf(gMsgStr,"Uneven timestep length in %s: new %f vs %f",ps->SeriesName(),ps->TS(),fTS); gPrintScreen(); 
	}
    fList.append(*ps); cnt++;
  }
}

int  TPtSeriesList::ReadHeaderInfo(FILE* cfile, int format, const char* root_path ) {
	char tmp[250]; 

  int itest = Util::scan_forward(cfile,"DefaultData"); 
  if( itest > 0 ) {  
    itest = Util::scan_forward(cfile,"=",1); if(itest==-1)  { return 0; }
		int i=0;
		fscanf(cfile,"%s",tmp);
		while ( tmp[i] == ' ' ) { i++; }
		if( tmp[i] == '/' ) {
			fDefaultDataPath.Add(tmp);
		} else {
			fDefaultDataPath.Add( (root_path==NULL) ? Env::DataPath() : root_path );
			fDefaultDataPath.Add(tmp);
		}
		FILE* dfile = fopen(fDefaultDataPath,"r");
		if(dfile==NULL) { 
			sprintf(gMsgStr,"Unable to open default PTS data file %s\n",fDefaultDataPath.chars()); gPrintErr(); 
		}
		else { sprintf(gMsgStr,"Reading default PTS data file %s",fDefaultDataPath.chars()); gPrintScreen(); }
		ReadHeaderInfo(dfile,format,root_path);
  }

	rewind(cfile);
  itest = Util::scan_forward(cfile,"Data Directory"); 
  if( itest <= 0 )  { fPTSDataPath.Add( (root_path==NULL) ? Env::DataPath() : root_path  );  } 
  else {  
		itest = Util::scan_forward(cfile,"=",1); 
		if( itest==-1 )  { 
			fPTSDataPath.Add( (root_path==NULL) ? Env::DataPath() : root_path  );
		} else {
			int i=0;
			fscanf(cfile,"%s",tmp);
			while ( tmp[i] == ' ' ) { i++; }
			if( tmp[i] == '/' ) {
				fPTSDataPath.Add(tmp);
			} else {
				fPTSDataPath.Add( (root_path==NULL) ? Env::DataPath() : root_path  );
				fPTSDataPath.Add(tmp);
			}
		}
	}

  rewind(cfile);
  int start_index = -1;
  itest = Util::scan_forward(cfile,"Start Index"); 
  if( itest > 0  ) {  
    itest = Util::scan_forward(cfile,"=",1); if(itest==-1)  { return 0; }
    fscanf(cfile,"%d",&start_index);
    fUseReadIndex = 1;
  }
  fReadIndex = start_index;

  rewind(cfile);
  fInterpolate = 1;
  itest = Util::scan_forward(cfile,"Interpolate"); 
  if( itest > 0  ) {  
    itest = Util::scan_forward(cfile,"=",1); if(itest==-1)  { return 0; }
    fscanf(cfile,"%d",&fInterpolate);
  }

  rewind(cfile);
  itest = Util::scan_forward(cfile,"PointMap"); 
  if( itest > 0 ) { 
  	CPathString pointMapPath;
    itest = Util::scan_forward(cfile,"=",1); if(itest==-1)  { return 0; }
		int i=0;
		fscanf(cfile,"%s",tmp);
		while ( tmp[i] == ' ' ) { i++; }
		if( tmp[i] == '/' ) {
			pointMapPath.Add(tmp);
		} else {
			pointMapPath.Add( (root_path==NULL) ? Env::DataPath() : root_path  );
			pointMapPath.Add(tmp);
		}
		rewind(cfile);
		itest = Util::scan_forward(cfile,"PointMapSource"); 
		if( itest > 0 ) { 
			CString mapSource; 
			itest = Util::scan_forward(cfile,"=",1); if(itest==-1)  { return 0; }
			int i=0;
			fscanf(cfile,"%s",tmp);
			mapSource = tmp;
			fSiteMap = new TMap();
			fSiteMap->Read( mapSource, pointMapPath );
		} else { sprintf( gMsgStr, "Must specify map source for PointMap %s", pointMapPath.chars() ); gPrintErr(); }
	}

	if( format == 0 ) {  
		rewind(cfile);
		int iformat = 0;
		itest = Util::scan_forward(cfile,"Format"); 
		if( itest > 0 ) {  
			itest = Util::scan_forward(cfile,"=",1); if(itest==-1)  { return 0; }
			fscanf(cfile,"%d",&iformat);
		} 
		fFormat = iformat;
	}  else fFormat = format;

	rewind(cfile);
	int iyear = -1;
	itest = Util::scan_forward(cfile,"Year"); 
	if( itest > 0 ) {  
		itest = Util::scan_forward(cfile,"=",1); if(itest==-1)  { return 0; }
		fscanf(cfile,"%d",&iyear);
	} 
	fYear = iyear;
	rewind(cfile);
}

int  TPtSeriesList::Import( const CPathString& dataPath, char* fileName, int format ) {
  CPathString s0(dataPath); s0.Add(fileName);
	char tmp[250]; 
  FILE* cfile = fopen(s0,"r");
  if(cfile==NULL) { 
    sprintf(gMsgStr,"\nERROR: Unable to open pts file %s\n",fileName); gFatal(); 
  }
  else { sprintf(gMsgStr,"Reading pts file %s",fileName); gPrintScreen(); }
	ReadHeaderInfo( cfile, format, s0.Path(1).chars() );
	
  int itest = Util::scan_forward(cfile,"Data ="); 
  if(itest<=0)  { return 0; } 

  int owner = 0, error = 0;
  int isFloat=0, sCnt=0, sIndex;
  float fx, fy;
  while(1) {
  
		if( fSiteMap == NULL ) { 
			int tst = Util::read_number ( cfile, fx);
			if( tst < 0 ) { error = 1; }
			if(tst) { isFloat = 1; }    
			tst = Util::read_number ( cfile, fy);
			if( tst < 0 ) { error = 1; }
			if(tst) { isFloat = 1; }
		} else {
			int tst = Util::read_number ( cfile, fx);
			if( tst != 0 ) { error = 1; }
			sIndex = (int) fx;  
		}
    
    if( Util::skip_white(cfile) == 0 ) break;
	fscanf(cfile,"%s",tmp);
	if( error ) { sprintf(gMsgStr,"Index format error in PTS file %s",tmp); gPrintErr(); break; }

    TPtSeries* ptseries = new TPtSeries(tmp,owner);
    if( fYear >= 0 ) { 
	  ptseries->SetStartYear(fYear); 
	}

	if( fSiteMap == NULL ) { 
		if( isFloat ) {
			ptseries->SetPoint(fx, fy);
		} else {
			Point2 pt( (int)fx, (int)fy );
			ptseries->SetPoint(pt);
		}
	} else ptseries->SetIndex(sIndex);

	if( (fReadIndex < 0) && (fYear < 0) ) {
		fReadIndex = ( ((int)TTime::Year()) % 100 ); 
	}		
    CString path( fPTSDataPath.Path("/") );  
    path += (ptseries->SeriesName());
    if( fFormat != 6 ) {
	  if( fReadIndex >= 0 ) { path.appendIndex(fReadIndex); }
	  path += ".ts";
	}
    int rv =  ptseries->Import(path,fFormat);
    if(rv == 0) { return 0; }
    ptseries->SeriesIndex(sCnt++);
    
    if( Size() < 0) { fNptTS = ptseries->Size(); fTS = ptseries->TS(); }
    else if ( fNptTS != ptseries->Size() ) { 
      sprintf(gMsgStr,"Uneven timeseries length in %s, new %d vs %d",ptseries->SeriesName(),ptseries->Size(),fNptTS); gPrintScreen(); 
	} else if (  fTS != ptseries->TS() ) { 
      sprintf(gMsgStr,"Uneven timestep length in %s: new %f vs %f",ptseries->SeriesName(),ptseries->TS(),fTS); gPrintScreen(); 
	}
    fList.append(*ptseries);
  }
  
  if( fSiteMap ) {
		for (int ir = fSiteMap->lower(0); ir <= fSiteMap->upper(0); ir++) {
			for (int ic = fSiteMap->lower(1); ic <= fSiteMap->upper(1); ic++) {
				long ival = fSiteMap->IntValue( ir, ic );
				if( ival > 0 ) {
					Pix p = fList.find(ival);
					if(p) {
						TPtSeries* ptseries = (TPtSeries*)&fList(p);
						Point2 pt( ir, ic );
						ptseries->SetPoint(pt);
					} else { sprintf( gMsgStr, "No series associated with index %d in PointMap %s", ival, fSiteMap->MapName().chars() ); gPrintErr(); }
				}
			}
		}
	}  
  fclose(cfile);  
  return sCnt;
}

void TPtSeriesList::Reset() { 
	fReadIndex = -1; fTS=fT0=fT1=0.0; fNptTS = -1; fDoneFirstRead = 0;  fFormat=0;
  for( Pix p = fList.first(); p; fList.next(p) ){
    TPtSeries* ps = (TPtSeries*)&fList(p);
    delete ps;
	}	
	fList.clear(); 
}

float TPtSeriesList::GetInterpolatedValue( TCell* c, TLayer* l,  float time, int print_log )  
{
  float InterpValue = 0, weight = 0, distance = 0, r, value; 
  int series_cnt = 0; 

	if( fList.one_node() ) { return  ((TPtSeries*)fList.firstObject())->InterpValue(time); }
	if( print_log ) { sprintf(gMsgStr,"\n PTS:GetInterpolatedValue values at %f : ", gTime() ); gPrintLog(); } 
	for( Pix p = fList.first(); p; fList.next(p) ){
	  TPtSeries& ps = (TPtSeries&)fList(p);
	  float last_time = ps.LastTime(); 
	  float first_time = ps.FirstTime(); 
	  if( (time >= first_time) && (time <= last_time) ) {
		value = ps.InterpValue(time);
		if( value != kSeriesNullValue ) {
			series_cnt++;
			if( ps.GetSeriesInfo(kTS_SpatialReferenced) ) {
				float dr0 = l->getUTMResolution(TLayer::kRows)/2.0; 
				float dc0 = l->getUTMResolution(TLayer::kCols)/2.0; 
				float c0 = l->getCellGblCoord( TLayer::kRows, c);
				float c1 = l->getCellGblCoord( TLayer::kCols, c);
				float dr = ps.Coord(0) - c0;
				float dc = ps.Coord(1) - c1;
				if( (fabs(dr) < dr0) && (fabs(dc) < dc0) ) {
					return ps.CurValue();
				}
				r = (double)(dr*dr + dc*dc);
				if( print_log ) { sprintf(gMsgStr," [cell %d, series %d] P:(%.2f,%.2f) C:(%.2f,%.2f) dR:(%.2f,%.2f) Res:(%.2f,%.2f) R:%.2f-> %.2f ", 
										  c->Index(), series_cnt, ps.Coord(0), ps.Coord(1), c0, c1, dr, dc, dr0, dc0, r, value ); gPrintLog(); } 
			} else {
				Point2& point = l->getCellLocPoint( c );
				long int dr = ps.Point()(0) - point(0);
				long int dc = ps.Point()(1) - point(1);
				if( (dr == 0) && (dc == 0) ) {
					return ps.CurValue();
				}
				r = (double)(dr*dr + dc*dc);
				if( print_log ) { sprintf(gMsgStr," ( %d: %d,%d,%f ) ", series_cnt, ps.Point()(0), ps.Point()(1), value ); gPrintLog(); } 
			}
			if( fInterpolate  ) {
				weight = (1.0)/(r*r);
				InterpValue += value*weight;
				distance += weight;
			}
		}
	 }
  }
  if( series_cnt == 0 ) { gFatal( "No Data in Point Time Series."); }
  if( distance == 0 ) { return 0.0; } 
  InterpValue /= distance;
  if( print_log ) { sprintf(gMsgStr," Result::  D:%.2f-> %.2f ",  distance, InterpValue ); gPrintLog(); } 
  return (float) InterpValue;
}

void TPtSeriesList::CreatePointMap(MultiCoverage& cov, float time )
{
	if( gDebug ) {
		TPtSeries* ps = (TPtSeries*)fList.firstObject();
		if( (ps==NULL) || ps->empty() ) {
			sprintf(gMsgStr,"\nNo Data in TPtSeriesList read from: %s\n", fSeriesName.chars() ); gFatal();
		}
	}
	MultiGrid* g = TModel::I().Grid();
	TLayer* l = g->getCellLayer(fLayerIndex);
	ActivationLayer* al = cov.activationLayer();
	int start = al->layerOffset(fLayerIndex);
	int stop = al->layerOffset(fLayerIndex+1);
	int print_log = ( gDebug > 1 ), index = 0;
	for( int32 i = start; i < stop; i++ ) {
		TCell* c = al->cell(i);
		cov.SetValue( c, GetInterpolatedValue( c, l, time, print_log ) );
		if( index++ > 3 ) { print_log = 0; }
  }
  if( gDebug > 1 ) {
    gPrintLog(format("\nCreatePointMap (time %.2f):\n",time)); int cnt=0;
	for( int32 i = start; i < stop; i++ ) {
		TCell* c = al->cell(i);
		Point2& pt = l->getCellLocPoint( c );
		sprintf(gMsgStr," %.2f ",cov.Value( c ));
		gPrintLog();
		if( cnt++ == 100 ) break;
    } 
  } 
}




