#include "SimLib.h"
#include "Globals.h"
#include "SSModel.h"
#include "SSModule.h"
#include "CVariable.h"
#include "PixVec.h"
#include "TPipe.h"
#include "SSComm.h"
#include "HDFMgr.h"
#include "viewserver.h"
#include "controlsocket.h"
#include "Externals.h"
#include "GOFCalc.h"

#ifdef USE_TCL
#include <tcl.h>
#endif

int SS_GetConfigData(ECnfgLevel level, CString& config_str );

Module* SS_CurrentModule = NULL;
Variable* SS_CurrentVariable = NULL;
Variable* SS_ListVariable = NULL;
Variable* SS_CurrentSocketVariable = NULL;
TCommand* SS_CurrentCommand = NULL;
TCommand* SS_ListCommand = NULL;
TConfigData SS_CD;
ArgArray SS_Args;
ArgArray SS_Path;
CString SS_CmdResult;
PixVec SS_Pix;
ECnfgLevel SS_Level;
int SS_GoToDebug = 0;
int SS_iSim = 0;
int SS_repeat_index = 0;
int SS_error = 0;

int SS_error_index() { return SS_error; }
int SS_DumpToDebug() { return SS_GoToDebug; SS_GoToDebug = 0; }
void MakeVarDataList( Variable* v );
void MakeVarDataString( CString& s, Bool append );
void MakeModuleList(int& index);
void MakeCompactVarDataList( Variable* v );
void MakeList( TNamedObjectList&  mList, const char* hint, int  addVarData = 0 );
int SS_UseTcl() { return (int)Env::UseTcl(); }
const char* SS_TclFile() { return (Env::TclFile().empty()) ? NULL : (const char*) Env::TclFile(); }

//----------------------------------------------------------------------------------------
// SS_ExecuteCommand 
//----------------------------------------------------------------------------------------

void clearComments( CString& commands ) {
  int sz = commands.size();
  char* data =  (char*) commands.chars();
  for( int i=0; i<sz; i++ ) {
	if( data[i] == '|' ) { 
	  data[i] = '\0'; 
	  break;
	}
  }
  commands = data;
}

const char* SS_ExecuteJavaCommand (char* buf, int useSocket, void* interp ) {
#ifdef JAVA
	const char* rv = JavaSocketReader::processMessages(buf, useSocket, interp, SS_ExecuteTelnetCommand );
#else
	const char* rv = NULL;
#endif
	return rv;
}

int SS_UseJavaSocket () { return Env::UseJavaSocket (); }

const char* SS_ExecuteTelnetCommand (char* cmdList, int* cmd_code, int * repeat, void* interp ) {
  ArgArray msg;
  if(cmd_code) { *cmd_code = 200; }
  if(repeat) { *repeat = 0; }
  int nargs = SS_Args.AddList( cmdList );
  
  CString cmd(SS_Args[0]); cmd.trim('\n','\n'); cmd.trim('\r','\r');
	sprintf(gMsgStr, "Executing Telnet cmd: %s (%s)" , cmdList, cmd.chars() );  gPrintLog();
//	if( gDebug ) { gPrintScreen();  }
	SS_Args.ShiftLow();
	if( cmd == "COMM" ) {
//			CString comm(SS_Args[0]); cmd.trim('\n','\n'); cmd.trim('\r','\r');
			if( interp != NULL ) {
#ifdef USE_TCL
				const char* script = SS_Args.string();				
				if( gDebug ) { sprintf(gMsgStr, "Evaluating tcl script: %s" , script );  gPrintScreen();  }
				int ret = Tcl_Eval( (Tcl_Interp*) interp, (char*)script );
				if (ret == TCL_ERROR) {
					gFatal( format("Executing tcl script: %s",script) );
				}
				SS_CmdResult = (const char *) Tcl_GetStringResult( (Tcl_Interp*) interp );
#endif
			} else {
				SS_CmdResult = "This command requires the SME to be built with tcl."; 
				if(cmd_code) { *cmd_code = 500; }
			}
	} else	if( cmd == "USER" ) {
		if(cmd_code) { *cmd_code = 300; }
	
	} else	if( cmd == "PASS" ) {
	
	} else	if( cmd == "ANSW" ) {

	} else	if( cmd == "NINF" ) {
		char tmp[32];
		sprintf(tmp,"%d",Env::getNInfo());
		SS_CmdResult = tmp;
	} else	if( cmd == "NERR" ) {
		char tmp[32];
		sprintf(tmp,"%d",Env::getNError());
		SS_CmdResult = tmp;
	} else	if( cmd == "NWAR" ) {
		char tmp[32];
		sprintf(tmp,"%d",Env::getNWarn());
		SS_CmdResult = tmp;
	} else	if( cmd == "WARN" ) {
		CString comm(SS_Args[0]); cmd.trim('\n','\n'); cmd.trim('\r','\r');
		int index = atoi(comm.chars());
		SS_CmdResult = Env::getWarn(index);	
	} else	if( cmd == "INFO" ) {
		CString comm(SS_Args[0]); cmd.trim('\n','\n'); cmd.trim('\r','\r');
		int index = atoi(comm.chars());
		SS_CmdResult = Env::getInfo(index);	
	} else	if( cmd == "CNFG" ) {
		CString comm(SS_Args[0]); cmd.trim('\n','\n'); cmd.trim('\r','\r');
		int index = atoi(comm.chars());	
		TModel::I().BuildConfigXML( msg, index ); 	
		SS_CmdResult = msg.dump();
	} else	if( cmd == "ERRO" ) {
		CString comm(SS_Args[0]); cmd.trim('\n','\n'); cmd.trim('\r','\r');
		int index = atoi(comm.chars());
		SS_CmdResult = Env::getError(index);	
	} else	if( cmd == "exit" ) {
		SS_CmdResult = "exit";
	} else { 
		SS_CmdResult = "Processed Command: ";  SS_CmdResult += cmd;
	}
	return SS_CmdResult.chars(); 
}

const char* SS_ExecuteCommand (char* cmdList, int* cmd_code, int * repeat, void* interp, int onProc ) {
	if( (onProc>=0) && (gIProc != onProc) ) return NULL;
	int nargs = 0;
	if( cmdList != NULL ) {
		sprintf(gMsgStr, "Executing cmd: %s" , cmdList );  gPrintLog();
		if( gDebug > 2 ) { gPrintScreen();  }
#ifdef USE_MPI 
		if( onProc < 0 ) {
		  int size = strlen(cmdList)+1; 
		  MPI_Bcast( &size, 1, MPI_INT, 0, MPI_COMM_WORLD );
		  MPI_Bcast( cmdList, size, MPI_CHAR, 0, MPI_COMM_WORLD );
		}
#endif
		if(cmd_code) { *cmd_code = 200; }
		nargs = SS_Args.AddList( cmdList );
	} else {
		nargs = SS_Args.high()+1;
	}
	SS_error = 0;	
	SS_repeat_index = 0;
 	float ftmp;
  if( nargs == 0 ) {
		return NULL;
	}
  const CString& cmd = SS_Args[0];

  switch( cmd[0] ) {
  case 'h': {
    if( nargs == 1 ) {
      gPrintErr( "Available Commands:\n\ts step\n\tr run\n\tm set current module\n\tv set current variable\n");	
      gPrintErr( "\tl list\n\tb break\n\td dump\n\tc choose from list\n\tC input config info\n\tS stop\n\tN initialize network parameters\n\tX exit");	
    } else {
      switch(SS_Args[1][0]) {
      case 's': 
	gPrintScreen( "Args{ (char type) } > Execute next Event(type=e), Command(type=c), SharedData Update(type=u)" ); 
	break;
      case 'r': 
	gPrintScreen( "Args{ (float t=inf) } > Run sim until time = t" ); 
	break;
      case 'm': 
	gPrintScreen( "Args{ (string mod_name) } > Set current module to mod_name" ); 
	break;
      case 'v': 
	gPrintScreen( "Args{ (string var_path) (char action=0) } > Set current variable to var_path and update sharedData(action=u)" ); 
	break;
      case 'N': 
	gPrintScreen( "Args{ (host port [timeout] [bufsize] [datasize] [sync] } > Set parameters for communication with interface" ); 
	break;					
      case 'l': 
	gPrintScreen( "Args{ (char level = {current_level}) } > list Modules(level=m), Variables(level=v), Commands(level=c), or Dependencies(level=d)" ); 
	break;
      case 'q': 
	gPrintScreen( "Args{ (char type) } > Query current variable for information, type=( spatial(s) temporal(t) mode(m) initmode(i) class(c) )" ); 
	break;
      case 'b': 
	gPrintScreen( "Args{ (char type) } > break on Commands for type: all(a), update(u), current(c), init(i), pipe(p), clear(C), delete(D)" ); 
	break;
      case 'k': 
	gPrintScreen( "Args{ (string pipeConfigCmd = all pipes) } > delete pipes configured with Cmd" ); 
	break;
      case 'd': 
	gPrintScreen( "Args{ (int start=0) (int end=100) } > dump data in array from start to end." ); 
	break;
      case 'c': 
	gPrintScreen( "Args{ (int index) } > Choose (from current list) item with index." ); 
	break;
      case 'D': 
	gPrintScreen( "Args{ } > Dump to Debugger." ); 
	break;
      case 'S': 
	gPrintScreen( "Args{ } > Stop: Reset Simulation To Initial Time Condition (Spatial Frames Retained)." ); 
	break;
      case 'X': 
	gPrintScreen( "Args{ } > Exit Simulation." ); 
	break;
      case 'C': 
	gPrintScreen( "Args{ (char level) (string cmd) } > Input Configuration Command cmd to level { Model(a), Module(m), Variable(v) }" ); 
	break;
      case 'R':
	gPrintScreen( "Args{ (int index) } > set replication index for data input from CME data" );
	break;
      case 'I':
	gPrintScreen( "Args { ( string HDFfilename ) } set HDF input file name");
	break;
      default:
	gPrintScreen( "Unknown or Undocumented Command." ); 
	break;
      }
    }
  } break;
  case 's':
    if(nargs == 1) SS_Step ();
    else {
      switch(SS_Args[1][0]) {
				case 'e': 	SS_Step ( kSS_EventS ); 		SS_CmdResult = "Step Event"; break;	
				case 'c': 	SS_Step ( kSS_CommandS ); 		SS_CmdResult = "Step Command"; break;	
				case 'u': 	SS_Step ( kSS_UpdateD ); 		SS_CmdResult = "Update Shared Data"; break;					
				case 'T': 	{
					float end_time = gTime() + gDT();
					SS_Update ( end_time, 0 );  
					SS_CmdResult = end_time;
					break;					
				}
			}
		}
    break;
  case 'r': {
    float rv;
	if( Env::UseMPE() ) {
	  if(nargs > 1) {
		int niter = atoi(SS_Args[1]); 
		rv = SS_RunMPE (niter);
		SS_CmdResult = "mpe run completed";
	  } else {
		rv = SS_RunMPE (100);
		SS_CmdResult = "mpe run completed";
	  }
	} else { 
	  int useTimer = ( cmd[1] == 't' );
	  if(nargs > 1) {
		ftmp = atof(SS_Args[1]); 
		rv = SS_Update (ftmp,useTimer);
		SS_CmdResult = SS_Args[1];
	  } else {
		  rv = SS_Run(useTimer);
		  SS_CmdResult = gTime();
	  }	
	}
	} break;
  case 'o':
    if( SS_Open() ) {
      SS_CmdResult = "Simulation opened";
    } else {
      SS_CmdResult = "Simulation already open";
    }
    break;
  case 'm':
	if( cmd == "mpg" ) {
	  Bool debug = False;
	  if( nargs < 3 ) { gPrintErr(" Must specify parameter indices (use \"l p\" cmd to list)." ); break; } 
	  int p0 = atoi(SS_Args[1]); 
	  int p1 = atoi(SS_Args[2]); 
#ifdef HAS_MPE
	  GOFCalc::verbose(debug);
	  GOFCalc::readReferenceData();	
	  GOFCalc::plotGOF(p0,p1);
#endif
	} else if(nargs > 1) {
      if( !SS_SetCurrentModule( SS_Args[1]) ) 
	gPrintErr( SS_Args[1] + ": Can't Find Module.");
      else gPrintErr( SS_Args[1] + ": Current Module.\n");
    } 
    else  { SS_CurrentModule = NULL; }
    break;
  case 'v':
    if(nargs > 1) {
			if( cmd[1] == 'c' ) {
				if( !SS_SetCurrentCommand( SS_Args[1] ) )  { gPrintErr( SS_Args[1] + ": Can't Find Command."); }
				else { gPrintErr( SS_Args[1] + ": Current Command.\n"); }
			} else {
				int nargs = SS_Path.AddList( SS_Args[1], '.' );
				if( nargs > 1 ) {
					if( !SS_SetCurrentModule( SS_Path[0] ) )  gPrintErr( SS_Path[0] + ": Can't Find Module.");
					else gPrintErr( SS_Path[0] + ": Current Module.\n");
				}
				if( !SS_SetCurrentVariable( SS_Path.Top() ) ) 
					gPrintErr( SS_Path.Top() + ": Can't Find Variable.");
				else {
					gPrintErr( SS_Path.Top() + ": Current Variable.\n");
					SS_CurrentCommand = SS_CurrentVariable->GetDefaultCommand();
				}
			}
    } 
    else   { SS_CurrentModule = NULL; SS_CurrentCommand = NULL; SS_CurrentVariable = NULL; }
    if(nargs > 2) {
      if( SS_Args[2][0] == 'u' ) {
				if( SS_CurrentModule && SS_CurrentVariable ) {
					TModel::I().QueryInterface( SS_CurrentModule->Name(), SS_CurrentVariable->Name() ); 
					SS_CurrentVariable->SetF(FRegistered,True,"SS_ExecuteCommand");
					SS_CurrentVariable->SetF(FVisible,True,"SS_ExecuteCommand");
					if( gDebug > 1 ) {
						sprintf(gMsgStr,"\nRegistering %s::%s",SS_CurrentModule->Name(), SS_CurrentVariable->Name() ); 
						gPrintLog();
					}
				}
      }
    }
    break;
  case 'l': {
	CString hint( ( nargs > 2) ? SS_Args[2].chars() : "" );
    if( nargs == 1 ) SS_MakeList(SS_Level,hint);
    else {
      switch ( SS_Args[1][0] ) {
      case 'a':
				if(repeat) { SS_repeat_index = *repeat; }
				SS_MakeList(kSS_Model,hint); 
				if( cmd_code ) { *cmd_code = 201; } 
				break;
      case 'm': SS_MakeList(kSS_Module,hint); break;
      case 'v': SS_MakeList(kSS_Variable,hint); break;
      case 'c': SS_MakeList(kSS_Command,hint); break;
      case 'p': SS_MakeList(kSS_Parameter,hint); break;
      case 'd': SS_MakeList(kSS_Dependency,hint); break;
      case 'H': 
				if( nargs > 2) SS_MakeHDFList(hint[0]);
				else SS_MakeHDFList(0);
				break;
      case 'h': 
				if( SS_CurrentVariable == NULL ) { gPrintErr("Error, Variable Undefined."); break; }
				else SS_MakeHDFList('d');
				break;
      case 'M': 
				SS_MakeHDFList('M');
				break;
      }
    }
    } break;
  case 'k':
    if( SS_CurrentVariable == NULL ) { gPrintErr("Error, Variable Undefined."); break; }
    if( nargs == 1 ) SS_CurrentVariable->ProcessPipes(Pipe::kUndefined, Variable::kDelete);
    else {
      if( TAnimPipe::Cmd(SS_Args[1]) )      		SS_CurrentVariable->ProcessPipes(Pipe::kAnim, Variable::kDelete);
      else if( TGraphPipe::Cmd(SS_Args[1]) ) 		SS_CurrentVariable->ProcessPipes(Pipe::kGraph, Variable::kDelete); 
      else if( TSheetPipe::Cmd(SS_Args[1]) ) 		SS_CurrentVariable->ProcessPipes(Pipe::kSheet, Variable::kDelete); 
      else if( TSocketPipe::Cmd(SS_Args[1]) ) 	SS_CurrentVariable->ProcessPipes(Pipe::kSocket, Variable::kDelete); 
      else if( TMapArchivePipe::Cmd(SS_Args[1]) ) SS_CurrentVariable->ProcessPipes(Pipe::kMapArchive, Variable::kDelete); 
//      else if( TWindowPipe::Cmd(SS_Args[1]) ) 	SS_CurrentVariable->ProcessPipes(Pipe::kWindow, Variable::kDelete); 
      else if( TPtSeriesPipe::Cmd(SS_Args[1]) ) 	SS_CurrentVariable->ProcessPipes(Pipe::kPtSeries, Variable::kDelete); 
      else if( TCondensePipe::Cmd(SS_Args[1]) ) SS_CurrentVariable->ProcessPipes(Pipe::kCondense, Variable::kDelete); 
      else if( TTimeSeriesInputPipe::Cmd(SS_Args[1]) ) 		SS_CurrentVariable->ProcessPipes(Pipe::kTimeSeriesInput, Variable::kDelete); 
      else if( TPtTimeSeriesInputPipe::Cmd(SS_Args[1]) ) 	SS_CurrentVariable->ProcessPipes(Pipe::kPtTimeSeriesInput, Variable::kDelete); 
      else if( TDBaseInputPipe::Cmd(SS_Args[1]) ) 				SS_CurrentVariable->ProcessPipes(Pipe::kDBaseInput, Variable::kDelete); 
      else if( TMapInputPipe::Cmd(SS_Args[1]) ) 					SS_CurrentVariable->ProcessPipes(Pipe::kMapInput, Variable::kDelete); 
    }
    break;
  case 'q': {
      if( SS_CurrentVariable == NULL ) { gPrintErr("Error, Variable Undefined."); break; }
      int rv=0;
      if( cmd[1] == 'A' ) {
		MakeVarDataString( SS_CmdResult, False );
      } else if( nargs > 1 ) {
				SS_CmdResult = "( info "; SS_CmdResult += SS_CurrentVariable->SName(); 
				switch ( SS_Args[1][0] ) {
					case 's':  SS_CmdResult += " space ";
					rv = SS_CurrentVariable->GetF(FisSpatial);
					SS_CmdResult.appendIndex(rv);
					break;
					case 't':   SS_CmdResult += " time ";
					rv = !SS_CurrentVariable->GetF(FisConstant);
					SS_CmdResult.appendIndex(rv);
					break;
					case 'm':   SS_CmdResult += " mode ";
					rv = SS_CurrentVariable->GetCInfo(Variable::kMode);
					SS_CmdResult.appendIndex( rv );
					break;
					case 'i':  SS_CmdResult += " initmode "; 
					rv = SS_CurrentVariable->GetCInfo(Variable::kInitMode);
					SS_CmdResult.appendIndex( rv );
					break;
					case 'c':   SS_CmdResult += " class "; 
					rv = SS_CurrentVariable->GetCInfo(Variable::kType);
					SS_CmdResult.appendIndex( rv );
					break;					
				}
				SS_CmdResult += " ) ";
			}
  } break;
  case 'b': {
    if( nargs > 1 ) {
      int breakOn  = 1;
      if( nargs == 3 ) breakOn = atoi(SS_Args[2]);
      if( SS_CurrentVariable == NULL ) { gPrintErr("Error, Variable Undefined."); break; }
      switch ( SS_Args[1][0] ) {
      case 'c': 
				if( SS_CurrentCommand == NULL ) { gPrintErr("Error, Command Undefined."); break; }
				SS_CurrentCommand->SetBreak( breakOn );
				break;
			case 'a': 
				SS_CurrentVariable->SetBreak( TCommand::kUndefined, breakOn ); 
				break;
			case 'i': 
				SS_CurrentVariable->SetBreak( TCommand::kInit, breakOn ); 
				break;
			case 'u': 
				SS_CurrentVariable->SetBreak( TCommand::kImport, breakOn ); 
				SS_CurrentVariable->SetBreak( TCommand::kUpdate, breakOn ); 
				SS_CurrentVariable->SetBreak( TCommand::kIntegrate, breakOn ); 
				break;
			case 'p': 
				SS_CurrentVariable->SetBreak( TCommand::kPipe, breakOn ); 
				break;
			case 'C': 
				SS_CurrentVariable->SetBreak( TCommand::kUndefined, 0 ); 
				break;
			case 'D': 
				SS_CurrentCommand->SetBreak( 0 ); 
				break;
     }
    }
    SS_MakeBreakList();
    } break;
   case 'd': {
		   if( cmd[1] == 'm' ) {
			 CString& mod_name = (CString&) SS_Args[1];
			 if( Externals::deactivate(mod_name) ) {
				gPrintErr(" Unrecognized module: " + mod_name);
			 }
			 break;
		   }
			if( SS_CurrentVariable == NULL ) { gPrintErr(" Current Variable Undefined. " ); SS_CmdResult = "";  }	
			else if( cmd[1] == 'c' ) {
				if( SS_CurrentCommand ) {
					SS_CmdResult = SS_CurrentCommand->Code();
				} else 	{ gPrintErr(" Current Command Undefined. " ); SS_CmdResult = ""; }			
			} else if( cmd[1] == 'C' ) {
				if( SS_CurrentVariable->GetF(FisSpatial) ) {
					intPixVHMap* cats; 
					CVariable* sv = (CVariable*)SS_CurrentVariable;
					cats = sv->GetCategories();
					if( cats ) {
						int index; CString* cat_name;
						SS_CmdResult = "(  \n";
						for( Pix p = cats->first(); p; cats->next(p) ) {
							index = cats->key(p);
							cat_name = (CString*) cats->contents(p);
							SS_CmdResult += "(  ";
							SS_CmdResult.appendIndex(index);
							SS_CmdResult += " ";
							SS_CmdResult += (*cat_name);
							SS_CmdResult += "  ) \n";
						}
						SS_CmdResult += " )\n";
					} else 	{ gPrintErr(" Current Variable is not a Class map. " ); SS_CmdResult = ""; }			
				} else 	{ gPrintErr(" Current Variable is not spatial. " ); SS_CmdResult = ""; }			
			} else if( cmd[1] == 'p' ) {
				if( nargs < 3 ) { gPrintErr(" Must specify point coodrs." ); break; } 
				int p0 = atoi(SS_Args[1]); 
				int p1 = atoi(SS_Args[2]); 
				if( SS_CurrentVariable->GetF(FisSpatial) ) {
				CVariable* sv =  (CVariable*)SS_CurrentVariable;
				float value = sv->Value(p0,p1);
				SS_CmdResult = value;
				} else { gPrintErr(" NonSpatial Variable." ); }
			} else if( cmd[1] == 'g' ) {
				if( SS_CurrentVariable->GetF(FisSpatial) ) {
					int s0 = 15, s1 = 15;
					int pl0=0, pl1=0, iproc = 0;
					if( nargs >= 3 ) {  
						pl0 = atoi(SS_Args[1]); pl1 = atoi(SS_Args[2]);
						Point2 pt(pl0,pl1);
						CVariable* sv =  (CVariable*)SS_CurrentVariable;
						iproc = sv->Grid().GetOwner( pt );
					} 
					if( nargs >= 5 ) {  s0 = atoi(SS_Args[3]); s1 = atoi(SS_Args[4]); } 
					int layer_index = -1;
					if( nargs >= 6 ) {  layer_index = atoi(SS_Args[5]);  } 
					if( gIProc == iproc ) {
						CVariable* sv =  (CVariable*)SS_CurrentVariable;
						SS_CmdResult = "(  ";
						for ( int p0 = pl0; p0 < pl0+s0; p0++ ) {
							SS_CmdResult += "\n  ";
							for ( int p1 = pl1; p1 < pl1+s1; p1++ ) {
								float value = sv->Value(p0,p1,layer_index);
								if( value == floatVec::ErrorVal() ) {
									sprintf(gMsgStr,"    *     ");
								} else {
									sprintf(gMsgStr,"%9.2e ",value);
								}
								SS_CmdResult += gMsgStr;
							}
						}
					}
					SS_CmdResult += "\n)\n";
					SS_CmdResult.ProcSend(iproc,0); 
				} else { gPrintErr(" NonSpatial Variable." ); }
			} else if( cmd[1] == 'v' ) {
				  byte mode = 1;
			      if(nargs > 1) {
					char cm = SS_Args[1].elem(0);
					if( cm == 'x' ) { mode = 1; }
					else if( cm == 'e' ) { mode = 2; }
					else if( cm == 'f' ) { mode = 3; }
				  } else { gPrintErr("Must specify element index"); break; }
				  TDataDisplayPipe* ddp = TDataDisplayPipe::GetGlobalViewer( (CVariable&)(*SS_CurrentVariable), mode );
				  ddp->Execute(); 
			} else {
				int iEnd = 100, iStart = 0; 
				if( nargs == 2 ) { iEnd = atoi(SS_Args[1]); }
				if( nargs == 3 ) { iStart = atoi(SS_Args[1]); iEnd = atoi(SS_Args[2]); }
				if( SS_CurrentVariable == NULL ) { gPrintErr("Error, Variable Undefined."); break; }
				((CVariable*)SS_CurrentVariable)->PrettyPrint( SS_CmdResult, iStart, iEnd );
			}
    }    break;
    case 'c': {
	  if( cmd[1] == 'i' ) { 
        if( SS_CurrentVariable == NULL ) { gPrintErr("Error, Variable Undefined."); break; }
		if( SS_CurrentVariable->GetF(FisImport) ) {
		  SS_CurrentVariable = SS_CurrentVariable->Connection();
		  SS_CurrentModule = SS_CurrentVariable->GetModule();
		  SS_CurrentCommand = SS_CurrentVariable->GetDefaultCommand();
		  SS_CmdResult = "Setting current Variable to " + SS_CurrentModule->SName() + ":" + SS_CurrentVariable->SName();
		} else {
		  SS_CmdResult = "Current Variable" + SS_CurrentModule->SName() + ":" + SS_CurrentVariable->SName() + " is not an import. ";
		}
		break;
	  }
	  if( cmd[1] == 'p' ) { 
        if( SS_ListVariable == NULL ) { gPrintErr("Error, List Variable Undefined."); break; }
		SS_CurrentVariable = SS_ListVariable;
		SS_CurrentModule = SS_CurrentVariable->GetModule();
		SS_CurrentCommand = SS_CurrentVariable->GetDefaultCommand();
		SS_CmdResult = "Setting current Variable to " + SS_CurrentModule->SName() + ":" + SS_CurrentVariable->SName();
		break;
	  }
     int index=0;	SS_CmdResult = "";	
      if(nargs > 1) {
        index = atoi(SS_Args[1]);
      } else { gPrintErr("Must specify element index"); break; }
      switch(SS_Level) {
      case kSS_Module: {
        Pix p = SS_Pix[index];
				if( p == 0 ) { gPrintErr(" Bad List Access Attempt."); break; }
        SS_CurrentModule = (Module*) &(TModel::I().ModuleList()(p));
        SS_CmdResult = SS_CurrentModule->Name();
        } break;
      case kSS_Variable: {
        Pix p = SS_Pix[index];
				if( p == 0 ) { gPrintErr(" Bad List Access Attempt."); break; }
        if( SS_CurrentModule == NULL ) { gPrintErr("Error, Module Undefined."); break; }
        SS_CurrentVariable = (TVariable*) &(SS_CurrentModule->VarList()(p));
        SS_CurrentCommand = SS_CurrentVariable->GetDefaultCommand();
		SS_CmdResult = "Setting current Variable to " + SS_CurrentModule->SName() + ":" + SS_CurrentVariable->SName();
       } break;
      case kSS_Command: {    
        Pix p = SS_Pix[index];
				if( p == 0 ) { gPrintErr(" Bad List Access Attempt."); break; }
        if( SS_CurrentVariable == NULL ) { gPrintErr("Error, Variable Undefined."); break; }
        SS_CurrentCommand = (TCommand*) &(SS_CurrentVariable->CommandList()(p));
        SS_CmdResult = SS_CurrentCommand->Name();
        } break;
     case kSS_Dependency: {
        Pix p = SS_Pix[index];
		if( p == 0 ) { gPrintErr(" Bad List Access Attempt."); break; }
        if( SS_ListCommand == NULL ) { gPrintErr("Error, List Command Undefined."); break; }
        TDependency& d = (TDependency&) SS_ListCommand->DependencyList()(p);
        SS_CurrentVariable = (TVariable*) d.GetVar();
        SS_CurrentCommand = SS_CurrentVariable->GetDefaultCommand();
		SS_CmdResult = "Setting current Variable to " + SS_CurrentModule->SName() + ":" + SS_CurrentVariable->SName();
        } break;
     case kSS_HDF: {
       char mode = SS_Args[2][0];
       if( mode == 'a' ) {
				 if( nargs > 3 ) {
					 int imag = atoi(SS_Args[3]);
					 if( imag > 0 && imag < 10 ) { HDFM::SetInfo( HDFM::kMag, imag ); /* HDFM::ClearAnim(); */ }
				 }
       }
       HDFM::PushDir( index );
       if( nargs > 2) SS_MakeHDFList(mode);
       else SS_MakeHDFList(0);
     } break;
      }
    } break;
  case 'V': 
 		 if( cmd[1] == 'S' ) { 
			if( SS_CurrentVariable == NULL ) { gPrintErr("Error, Variable Undefined."); break; }
      if(nargs == 2) {
        float value = atof(SS_Args[1]);
        ((CVariable*)SS_CurrentVariable)->SetValue(0,0,value);
			} else if(nargs == 4) {
        int row = atoi(SS_Args[1]);
        int col = atoi(SS_Args[2]);
        float value = atof(SS_Args[3]);
        ((CVariable*)SS_CurrentVariable)->SetValue(row,col,value);
			} else if(nargs == 5) {
        int row = atoi(SS_Args[1]);
        int col = atoi(SS_Args[2]);
        int layer = atoi(SS_Args[3]);
        float value = atof(SS_Args[4]);
        ((CVariable*)SS_CurrentVariable)->SetValue(row,col,value,layer);
			}
		 } else {
			SS_CmdResult = "( "; 
			SS_CmdResult += SS_CurrentModule->Name();
			SS_CmdResult += " "; 
			SS_CmdResult += SS_CurrentVariable->Name();
			SS_CmdResult += " "; 
			MakeVarDataList( SS_CurrentVariable );
			SS_CmdResult += " )"; 
		}
    break;
  case 'D':
		 if( cmd[1] == 'B' ) { 
			const char* display = ( nargs > 1 ) ?  SS_Args[1].chars() : NULL;
			SS_error = Env::AttachDebugger( display ); 
			SS_CmdResult = SME_debug_get_result_string();
		} else {
			if( nargs > 1 ) SS_GoToDebug = atoi(SS_Args[1]);
			else SS_GoToDebug = 1;
			SS_CmdResult = "Debug";
		}
    break;
  case 'X':
#ifdef JAVA
		ViewServer::disconnect( ); 
#endif
    SS_Exit();
    break;
  case 'S': {  
	 if( cmd[1] == 'C' ) { 
		gPrintScreen( " Saving config file " ); 
		if( nargs < 2 ) { 
		  TModel::I().WriteConfigFile(); 
		} 
		else { 
		  TModel::I().WriteConfigFile(SS_Args[1]); 
		}	      
 	 } else {    	  
		SS_Close();
		SS_CmdResult = "Stop";
		int& iSim = Env::BatchIndex();
		if( iSim < 0 ) iSim = 1;
		else iSim++;
	  }
    } break;
  case 'C':
    if( nargs < 3 ) { gPrintErr( " Must Specify Configuration Level (a,m,v,*) " ); }
    else {
      char tch = SS_Args[1][0];
	  int send_cdata = 0;
	  char format = 'd';
	  if(nargs > 2) {
        format = SS_Args[2][0];
	  }
	  if( cmd[1] == 'S' ) {
		int rv = -1;
		switch ( tch ) {
		case 'a': rv = SS_GetConfigData(kSS_Model, SS_CmdResult ); break;
		case 'm': {
		  rv = SS_GetConfigData(kSS_Module, SS_CmdResult ); 
		  if( format == 'c' ) { clearComments(SS_CmdResult);  }
		  } break;
		case 'v': {
		  rv = SS_GetConfigData(kSS_Variable, SS_CmdResult );
		  if( format == 'c' ) { clearComments(SS_CmdResult); }
		  if( send_cdata ) {
			SS_CmdResult += " cdata(";
			MakeVarDataString( SS_CmdResult, True );  
			SS_CmdResult+=")";
		  }
		  } break;
		}
	  } else {
		switch ( tch ) {
		case 'a': SS_ConfigArgs(kSS_Model, nargs); break;
		case 'm': SS_ConfigArgs(kSS_Module, nargs); break;
		case 'v': SS_ConfigArgs(kSS_Variable, nargs); break;
		}
		if( isupper(tch) ) {
		  SS_CD.SName() = SS_Args[1];
		  SS_ConfigArgs( kSS_External, nargs ); 
		}
		SS_CmdResult = "Config";
	  }
	}
    break;
  case 'N': {
		if( cmd[1] == 'V' ) {
#ifdef JAVA
			const char* host = (nargs > 1) ? (const char*) SS_Args.elem(1) : "localhost";
			const char* port = (nargs > 2) ? (const char*) SS_Args.elem(2) : "-1";
			ViewServer::config( host, atoi(port) ); 
#endif
		} else {
			// fv initialize network; bind socket, return localport
			if (nargs < 2)  {  gPrintErr("Must specify at least <hostname>"); }
			else if( gIProc == 0 ) {
					if( cmd[1] == 'J' ) {    // fire up java shell 
						SS_InitializeNetworkJava(
							 nargs > 1 ? (const char*) SS_Args.elem(1) : "localhost",
							 nargs > 2 ? (const char*) SS_Args.elem(2) : "-1",
							 nargs > 3 ? (const char*) SS_Args.elem(3) : "0" );
						SS_CmdResult =  Env::Host();
						SS_CmdResult += ' ';
						SS_CmdResult.appendIndex(Env::Port());
						SS_WaitForJavaConnection();
					} else {
						SS_InitializeNetwork( 
							 nargs > 1 ? (const char*) SS_Args.elem(1) : "localhost",
							 nargs > 2 ? (const char*) SS_Args.elem(2) : "0",
							 nargs > 3 ? (const char*) SS_Args.elem(3) : "0",
							 nargs > 4 ? (const char*) SS_Args.elem(4) : "0",
							 nargs > 5 ? (const char*) SS_Args.elem(5) : "0");
						SS_CmdResult =  Env::Host();
						SS_CmdResult += ' ';
						SS_CmdResult.appendIndex(Env::Port());
					}
			}
		}
    } break; 
  case 'R':
    // fv set replication index
    if (nargs < 2) 
      gPrintErr("Must specify replication index");
    else
      {
	Env::ReplicationIndex() = atoi(SS_Args[1]);
	SS_CmdResult = SS_Args[1];
      }
    break;
  case 'I':
    // fv change HDF input dataset
    if (nargs < 2) 
      gPrintErr("Must specify HDF input file name");
    else
      {
	Env::HDFInputFile() = SS_Args[1];
	SS_CmdResult = SS_Args[1];
      }
    break;
  case 'W':
    int exe_cmd = 0;
    if(nargs > 1) { exe_cmd = atoi(SS_Args[1]); } 
    if( SS_CurrentVariable == NULL ) { gPrintErr("Error, Variable Undefined."); break; }			
    if(SS_CurrentSocketVariable && SS_CurrentSocketVariable != SS_CurrentVariable) {
      TCommand* c = SS_CurrentSocketVariable->ProcessPipes( Pipe::kSocket, Variable::kReturn );
      c->SetObjInfo(TCommand::kStatus,TCommand::kInActive);		
    }
    SS_CurrentSocketVariable = SS_CurrentVariable;
    TCommand* c = SS_CurrentSocketVariable->ProcessPipes( Pipe::kSocket, Variable::kReturn );	
    if( c == NULL ) { 
      gPrintErr( "No Socket Pipe Defined for Current Variable."); 
      SS_CurrentSocketVariable = NULL;
    }	
    else { 
      c->SetObjInfo(TCommand::kStatus,TCommand::kActive);	
      if(exe_cmd) c->Execute(); 
    }	
    break;
  }
	if(repeat) { *repeat = SS_repeat_index; }
  return SS_CmdResult.chars();
}

void MakeVarDataString( CString& str, Bool append ) {
  if( SS_CurrentVariable == NULL ) { gPrintErr(" Variable Undefined. "); return; }
  ((CVariable*)SS_CurrentVariable)->MakeStatusDataString( str, append );
}

  
void MakeVarDataList( Variable* v ) {
  if( v->GetF(FisSpatial) ) SS_CmdResult += " s "; else  SS_CmdResult += "   "; 
  if( v->GetF(FisImport) ) SS_CmdResult += "i "; else  SS_CmdResult += "  "; 
  if( !v->GetF(FisConstant) ) SS_CmdResult += "t "; else  SS_CmdResult += "  "; 
  switch( v->GetCInfo(Variable::kType) ) {
  case Variable::kNULL: SS_CmdResult += "N "; break;
  case Variable::kAux: SS_CmdResult += "A "; break;
  case Variable::kFlux: SS_CmdResult += "F "; break;
  case Variable::kState: SS_CmdResult += "S "; break; 
  case Variable::kData: SS_CmdResult += "D "; break;
  case Variable::kTime: SS_CmdResult += "T "; break;
  }
  switch( v->GetCInfo(Variable::kInitMode) ) {
  case Variable::kDefault: SS_CmdResult += "DV "; break;
  case Variable::kMap: SS_CmdResult += "Mp "; break;
  case Variable::kTimeSeries: SS_CmdResult += "TS "; break;
  case Variable::kPtTimeSeries: SS_CmdResult += "PTS "; break; 
  case Variable::kDBase: SS_CmdResult += "DB "; break;
  case Variable::kParameter: SS_CmdResult += "Pm "; break;
  }
}

void MakeCompactVarDataList( Variable* v ) {
  if( v->GetF(FisSpatial) ) SS_CmdResult += " s"; else  SS_CmdResult += " n"; 
  if( v->GetF(FisImport) ) SS_CmdResult += "i"; else  SS_CmdResult += "n"; 
  if( !v->GetF(FisConstant) ) SS_CmdResult += "t"; else  SS_CmdResult += "n"; 
  switch( v->GetCInfo(Variable::kType) ) {
		case Variable::kNULL: SS_CmdResult += "d"; break;
		case Variable::kAux: SS_CmdResult += "A"; break;
		case Variable::kFlux: SS_CmdResult += "F"; break;
		case Variable::kState: SS_CmdResult += "S"; break; 
		case Variable::kData: SS_CmdResult += "D"; break;
		case Variable::kTime: SS_CmdResult += "T"; break;
		case Variable::kConstant: SS_CmdResult += "C"; break;
		default: SS_CmdResult += "N"; break;
  }
  switch( v->GetCInfo(Variable::kInitMode) ) {
		case Variable::kDefault: SS_CmdResult += "d"; break;
		case Variable::kMap: SS_CmdResult += "M"; break;
		case Variable::kTimeSeries: SS_CmdResult += "t"; break;
		case Variable::kPtTimeSeries: SS_CmdResult += "T"; break; 
		case Variable::kDBase: SS_CmdResult += "D"; break;
		case Variable::kParameter: SS_CmdResult += "P"; break;
		default: SS_CmdResult += "N"; break;
  }
}

int SS_MakeHDFList(char mode) {
  if( mode == 'M' ) {
    int index = 0;
    SS_CmdResult = "\n";
    TNamedObjectList& ml = TModel::I().ModuleList();
    for( Pix mp = ml.first(); mp; ml.next(mp) ) {
      TModule& m = (TModule&)ml(mp);
      TNamedObjectList& vl = m.VariableList();
      for( Pix vp = vl.first(); vp; vl.next(vp) ) {
	TVariable& v = (TVariable&)vl(vp);
	TNamedObjectList& cl = v.CommandList();
	for( Pix p = cl.first(); p; cl.next(p) ) {
	  TCommand& c = (TCommand&)cl(p);
	  if( c.GetObjInfo(TCommand::kExType) == TCommand::kPipe ) {
	    Pipe* p = ((TPipeCommand&)c).GetPipe();
	    if( p->GetObjInfo(Pipe::kType)==Pipe::kMapArchive) {
	      MapArchivePipe* mp = (MapArchivePipe*)p;
	      if( mp->GetObjInfo(MapArchivePipe::kMapType) == kIOex) {
		SS_CmdResult += "\t\t(";
		SS_CmdResult.appendIndex(index++); 
		SS_CmdResult += " ";
		SS_CmdResult += m.Name(); 
		SS_CmdResult += " ";   
		SS_CmdResult += v.Name();    
		SS_CmdResult += " m ";
		SS_CmdResult += mp->MapName();
		SS_CmdResult += " ";
		SS_CmdResult += ")\n";					
	      }
	    }
	  }		
	}	
      }
    }	
  } else	if( mode == 'd' ) {
    SS_CmdResult = "\n";
    TNamedObjectList& cl = SS_CurrentVariable->CommandList();
    int index = 0;
    for( Pix p = cl.first(); p; cl.next(p) ) {
      TCommand& c = (TCommand&)cl(p);
      if( c.GetObjInfo(TCommand::kExType) == TCommand::kPipe ) {
	Pipe* p = ((TPipeCommand&)c).GetPipe();
	if( p->GetObjInfo(Pipe::kType)==Pipe::kMapArchive) {
	  MapArchivePipe* mp = (MapArchivePipe*)p;
	  if( mp->GetObjInfo(MapArchivePipe::kMapType) == kIOex) {
	    SS_CmdResult += "\t\t(";
	    SS_CmdResult.appendIndex(index++);     
	    SS_CmdResult += " m ";
	    SS_CmdResult += mp->MapName();
	    SS_CmdResult += " ";
	    SS_CmdResult += ")\n";					
	  }
	}
      }		
    }			
  }
#ifdef HAS_HDF	
  else {
    if( HDFM::Open (HDFM::kFReadArchive) == FAIL ) return FAIL;
    if( mode == 't' || HDFM::GetInfo(HDFM::kHLevel)==0 ) HDFM::TopDir();
    int nargs = HDFM::NStrArgs();
    SS_Level=kSS_HDF;
    SS_CmdResult = "\n";
    for( int i=0; i<nargs; i++ ) {
      SS_CmdResult += "\t\t(";
      SS_CmdResult.appendIndex(i);     
      SS_CmdResult += " ";
      SS_CmdResult += HDFM::Arg(i);
      SS_CmdResult += " ";
      SS_CmdResult += ")\n";
    }
  }
#endif
  return 0;
}

void MakeList( TNamedObjectList&  mList, const char* hint, int  addVarData ) {
  int index = 0;  SS_Pix.fill(0);
  SS_CmdResult = "(\n";
  for( Pix p = mList.first(); p; mList.next(p) ) {
    TNamedObject& obj = (TNamedObject&) mList(p);
    if( ( hint == NULL ) || (obj.SName().starts_with(hint)) ) {
	  SS_Pix.elem(index) = p;
	  SS_CmdResult += "\t\t(";
	  SS_CmdResult.appendIndex(index++);     
	  SS_CmdResult += " ";
	  SS_CmdResult += obj.SName();
	  SS_CmdResult += " ";
	  if(addVarData) MakeVarDataList((Variable*)&obj);
	  SS_CmdResult += ")\n";
	}
  }  
  SS_CmdResult += ")\n";
}

void MakeModuleList(int& index) {
	SS_CmdResult = "";
	CString dname;
	int mindex=1;
  TNamedObjectList&  mList = TModel::I().ModuleList();
  for( Pix p = mList.first(); p; mList.next(p) ) {
		if( mindex++ == index ) {
			Module& m = (TModule&) mList(p);
			SS_CmdResult += m.SName();
			SS_CmdResult += "{";
			TNamedObjectList&  vList = m.VarList();
			for( Pix pv = vList.first(); pv; vList.next(pv) ) {
				Variable& v = (Variable&) vList(pv);
				SS_CmdResult += "(";
				SS_CmdResult += v.SName();
				MakeCompactVarDataList(&v);
				TNamedObjectList& cl = v.CommandList();
				SS_CmdResult += "(";
				for( Pix cv = cl.first(); cv; cl.next(cv) ) {
					TCommand& c = (TCommand&) cl(cv);
					SS_CmdResult += c.SName();
					SS_CmdResult += "{";
					SS_CmdResult += c.Code();
					SS_CmdResult += "}";
					TNamedObjectList& dl = c.DependencyList();
					SS_CmdResult += "(";
					for( Pix dv = dl.first(); dv; dl.next(dv) ) {
						TDependency& d = (TDependency&) dl(dv);
						if(dv != dl.first()) { SS_CmdResult += " "; }
						dname = d.SName();
						dname.del("_Dep", dname.size()-5 );
						SS_CmdResult += dname;
					}
					SS_CmdResult += ")";
				}
				SS_CmdResult += ")";
				SS_CmdResult += ")";
			}
			SS_CmdResult += "}";
			SS_CmdResult += (char)0;
			if( p ==  mList.last() ) { index = 0; }
			else { index++; }
			return;
		}  
	}
}

void SS_MakeList( ECnfgLevel level, const char* hint ) {
  if( hint[0] == '\0' ) hint = NULL;
  switch(level) {
  case kSS_Model:
		MakeModuleList( SS_repeat_index );
		break;
  case kSS_Module:
    MakeList(TModel::I().ModuleList(),hint);
    SS_Level = kSS_Module;
    break;
  case kSS_Variable:
    if( SS_CurrentModule == NULL ) { gPrintErr("Error, Module Undefined."); break; }
    MakeList( SS_CurrentModule->VarList(),hint,True );
    SS_Level = kSS_Variable;
   break;
  case kSS_Command: {
    if( SS_CurrentVariable == NULL ) { gPrintErr("Error, Variable Undefined."); break;}
    SS_ListVariable = SS_CurrentVariable;
    MakeList( SS_CurrentVariable->CommandList(),hint );
    SS_Level = kSS_Command;
    } break;
  case kSS_Parameter: {
#ifdef HAS_MPE
    MakeList( GOFCalc::ParameterList(),hint );
    SS_Level = kSS_Parameter;
#endif
    } break;
  case kSS_Dependency: {
    if( SS_CurrentCommand == NULL ) { gPrintErr("Error, Command Undefined."); break; }
    SS_ListVariable = SS_CurrentVariable;
    MakeList( SS_CurrentCommand->DependencyList(),hint );
    SS_ListCommand = SS_CurrentCommand;
    SS_Level = kSS_Dependency;
    } break;
  } 
}

void SS_MakeBreakList() {
  if( SS_CurrentModule == NULL ) { gPrintErr("Error, Module Undefined."); }
  SS_CmdResult = "( ";
  TNamedObjectList&  vList = SS_CurrentModule->VarList();
  for( Pix p = vList.first(); p; vList.next(p) ) {
    Variable& v = (Variable&) vList(p);
    TNamedObjectList&  cList = v.CommandList();
    for( Pix pc = cList.first(); pc; cList.next(pc) ) {
      TCommand& c = (TCommand&) cList(pc);
      if( c.Break() ) {
        SS_CmdResult += "(";
        SS_CmdResult += v.SName();     
        SS_CmdResult += " ";
        SS_CmdResult += c.SName();
        SS_CmdResult += ") ";
      }
    }  
  }
	SS_CmdResult += " )";
}


//----------------------------------------------------------------------------------------
// SS_Init 
//----------------------------------------------------------------------------------------

int SS_Init (int argc, char *argv[])
{  
  Env::Startup(argc,argv,Env::kDriver);
  
  if( Env::UseJavaSocket() ) {
	SS_WaitForJavaConnection();
//	CString cmd("NJ "); cmd += Env::Host(); 
//	SS_ExecuteCommand ((char*)(cmd.chars()), NULL, NULL, NULL, 0 ); 	
  }
  
//  Env::BCastTest(1);
	  
  TModel::I().Startup();
    
  if( Env::BatchIndex() == 0 ) {
		gDebug = 0;
		TModel::I().Run();
		SS_Exit();
	}	
  return gIProc;
}  

int SS_Open() {
  if( !Env::GetInfo(Env::kSimOpen) ) { 
    TModel::I().InitSimulation(); 
    Env::SetInfo(Env::kSimOpen,1); 
    return 1;
  }
  return 0;
}

void SS_Close ()
{
  if( Env::GetInfo(Env::kSimOpen)  ) TModel::I().CloseSimulation(SS_iSim++);
  if( !Env::UseMPE() ) { 
	gPrintScreen("Simulation Reset To Initial Time Condition (Spatial Frames Retained).");
  }
  SS_CurrentModule = 0; 
  SS_CurrentVariable = 0; 
  SS_CurrentCommand = 0;
  Env::SetInfo(Env::kSimOpen,0);
}

int SS_Update ( float end_time, int useTimer )
{
  SS_Open();
//  if( useTimer ) { Env::StartTimer(); }
  
  int rv = TModel::I().RunModel(end_time);
//  if( useTimer ) { Env::StopTimer(); }
//  TModel::I().SetCurrentContext(SS_CurrentModule,SS_CurrentVariable,SS_CurrentCommand);
  return  rv;
}

int SS_Run ( int useTimer )
{
  SS_Open();
//  if( useTimer ) { Env::StartTimer(); }  
  int rv = TModel::I().RunModel( FLT_MAX, SS_DumpToDebug() );
//  if( useTimer ) { Env::StopTimer(); }
//	TModel::I().SetCurrentContext(SS_CurrentModule,SS_CurrentVariable,SS_CurrentCommand);
  return  rv;

}

int SS_RunMPE( int max_runs ) {
#ifdef HAS_MPE
   Bool debug = False;
   if( Env::UseMPE() ) {
	 GOFCalc::verbose(debug);
	 GOFCalc::readReferenceData();	
	 if( GOFCalc::getSearchType() == kEvol ) {
		GOFCalc::evolve(max_runs);
	 } else {
	   GOFCalc::initGradOptimizer();
	   for( int iter=0; iter<max_runs; iter++ ) {
		  int start = GOFCalc::getProcOffset(iter);		
		  int stop = start + GOFCalc::getNCells();
		  for( int ic=start; ic<stop; ic++ ) {
			SS_Open();
			GOFCalc::updateParameters(ic);
			int rv = TModel::I().RunModel( FLT_MAX, SS_DumpToDebug() );
			GOFCalc::updateData();
			GOFCalc::calcGOF(ic,iter);
			SS_Close();
		  }
		  if( !GOFCalc::optimize(iter) ) break;
		}
	  }
	  TModel::I().SetCurrentContext(SS_CurrentModule,SS_CurrentVariable,SS_CurrentCommand);	
   }
   SS_Open();       // Finalization run.
   int rv = TModel::I().RunModel( FLT_MAX, SS_DumpToDebug() );
   GOFCalc::plotVariables( ".final" );
   SS_Exit();
#endif
   return 0;
}

float SS_Step ( EStepMode sm )
{
  SS_Open();
  float rv;
  switch (sm) {
		case kSS_EventS:
			rv = TModel::I().StepModel(TModel::kEvent, SS_DumpToDebug());
			break;
		case kSS_CommandS:
			rv = TModel::I().StepModel(TModel::kCommand, SS_DumpToDebug());
			break;
		case kSS_UpdateD:
			rv = TModel::I().StepModel(TModel::kUpdate, SS_DumpToDebug());
			break;
	}
	TModel::I().SetCurrentContext(SS_CurrentModule,SS_CurrentVariable,SS_CurrentCommand);
  return  rv;

}



void SS_Exit( int doExit ) 
{
   Externals::finalize();
   TModel::I().Shutdown();
   if( doExit ) { exit(99); }
}

float SS_GetValue(int index, int row, int col)
{
  return TModel::I().GetValue(index,row,col);
}

void SS_SetValue(int index, int row, int col, float value) {
  TModel::I().SetValue(index,row,col,value);
}

int SS_SetCurrentModule(const char* name)
{
  SS_CurrentModule = TModel::I().GetModule( name, False );
  SS_Level = kSS_Module;
  if(SS_CurrentModule) { sprintf(gMsgStr,"SetCurrentModule: %s",SS_CurrentModule->Name()); gPrintLog(); }
  return ( SS_CurrentModule == NULL ) ? 0 : 1;
}

int SS_SetCurrentVariable(const char* name)
{
  if( SS_CurrentModule == NULL ) { gPrintErr("Error, Module Undefined."); return 0; }
  SS_CurrentVariable = SS_CurrentModule->GetVariable( name, False );
  SS_Level = kSS_Variable;
  if(SS_CurrentVariable) { sprintf(gMsgStr,"\nSetCurrentVariable: %s",SS_CurrentVariable->Name()); gPrintLog(); }
  return ( SS_CurrentVariable == NULL ) ? 0 : 1;
}

int SS_SetCurrentVariable( void* v ) {
  SS_CurrentVariable = (CVariable*)v;
  if( SS_CurrentVariable == NULL ) return 0;
  SS_Level = kSS_Variable;
  SS_CurrentModule = SS_CurrentVariable->GetModule();
  SS_CurrentCommand = SS_CurrentVariable->GetDefaultCommand();
  if(SS_CurrentVariable) { 
	sprintf(gMsgStr,"Setting current Variable to " + SS_CurrentModule->SName() + ":" + SS_CurrentVariable->SName()); 
	gPrintScreen(); 
  }
  return  1;
}

int SS_SetCurrentCommand(const char* name)
{
  if( SS_CurrentVariable == NULL ) { gPrintErr("Error, Variable Undefined."); return 0; }
  SS_CurrentCommand = SS_CurrentVariable->GetCommand( name, TCommand::kUndefined, TCommand::kCode, False );
  SS_Level = kSS_Command;
  if( SS_CurrentCommand && gDebug ) { sprintf(gMsgStr,"\nSetCurrentCommand: %s",SS_CurrentCommand->Name()); gPrintLog(); }
  return ( SS_CurrentCommand == NULL ) ? 0 : 1;
}

int SS_Config( ECnfgLevel level, char* cmd, ... ) {
  va_list arg_ptr;
  char* arg; 
  int i=0;
  SS_CD.SetCmd(cmd);
  va_start(arg_ptr, cmd); 
  while( (arg = va_arg( arg_ptr, char* )) != NULL ) {
    SS_CD.SetArg(i++,arg);
  }
  va_end(arg_ptr);
  
  return SS_RegisterConfigData(level);
}

int SS_ConfigArgs( ECnfgLevel level, int nargs ) {
  SS_CD.SetCmd(SS_Args[2]);
  for( int i=0; i < (nargs-3); i++ ) {
    SS_CD.SetArg(i,SS_Args[i+3]);
  }
  return SS_RegisterConfigData(level);
}

      

int SS_GetConfigData(ECnfgLevel level, CString& config_str ) {
  CStream cStream(&config_str);
  switch(level) {
   case kSS_Model:
    TModel::I().WriteDataToConfig( cStream ) ;
    return 1;
   case kSS_Module:
    if( SS_CurrentModule == NULL ) { gPrintErr("Error, Module Undefined."); return 0; }
    SS_CurrentModule->WriteDataToConfig( cStream );
    return 1;
   case kSS_Variable:
    if( SS_CurrentVariable == NULL ) { gPrintErr("Error, Variable Undefined."); return 0; }
    SS_CurrentVariable->WriteDataToConfig( cStream );
    return 1;
   case kSS_External:
	return 0;
  }
  return 0;
}

int SS_RegisterConfigData(ECnfgLevel level) {
	sprintf(gMsgStr,"\nRegister Config(%x): %s, %x %x",
		level, SS_CD.Cmd().chars(), SS_CurrentModule, SS_CurrentVariable ); 
	gPrintLog(); 
  switch(level) {
  case kSS_Model:
    return TModel::I().Config(SS_CD);
  case kSS_Module:
    if( SS_CurrentModule == NULL ) { gPrintErr("Error, Module Undefined."); return 0; }
    return SS_CurrentModule->Config(SS_CD);
  case kSS_Variable:
    if( SS_CurrentVariable == NULL ) { gPrintErr("Error, Variable Undefined."); return 0; }
    return SS_CurrentVariable->Config(SS_CD);
	case kSS_External:
		return TModel::I().ConfigExternal(SS_CD,SS_CurrentModule);
  }
  return 0;
}

const float* SS_GetCurrentData( int* size ) {
	if( SS_Level == kSS_HDF ) {
		SDSObject* sds = HDFM::SDS();
		HDFM::SDRead(0);
	} else {
		if( SS_CurrentVariable == NULL ) { gPrintErr("Error, Variable Undefined."); return NULL; }
		CVariable* cv = (CVariable*)SS_CurrentVariable;
		*size = cv->DataSize();
		return cv->Data();
	}
}

const int SS_SetCurrentData( float* data, int layOut, int size ) {
		if( SS_CurrentVariable == NULL ) { gPrintErr("Error, Variable Undefined."); return 0; }
		CVariable* cv = (CVariable*)SS_CurrentVariable;
		return cv->SetDataValues( data, layOut, size );
}

int SS_Proc() { return gIProc; }

