//----------------------------------------------------------------------------------------
//	TSDArray.cc
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//	Change History:
//----------------------------------------------------------------------------------------

#include "TArray.h"
#include "HDFMgr.h"

int32 TSDArray::SDWrite( const char* info, const char* units, const char* format ) 
{
	int32 istat = -1;
#ifdef HAS_HDF
  if( empty() ) return 0;
  fSD_id = HDFM::Open(HDFM::kFOutput);
  if( fSDS_id == 0 ) {
    int32 dims[3];
    dims[0] = SD_UNLIMITED;
    dims[1] = extents(0);
    dims[2] = extents(1);
    fSDS_id = SDcreate(fSD_id, (char*)DSetName(),DFNT_FLOAT32,3,dims);
    SDsetdatastrs(fSDS_id, (char*)DSetName(), (char*)units,(char*)format,NULL);
    SDsetattr(fSDS_id,"info",DFNT_CHAR8,sizeof(info),(void*)info);
    int R[4] = { lower(0), lower(1), upper(0), upper(1) };
    SDsetattr( fSDS_id, "region", DFNT_INT32, 4, (char*)R );
    HDFM::AddSDS(fSDS_id,HDFM::kFArray,VariableName());
  }

  int32 start[3], edges[3];
  edges[0] = 1;
  edges[1] = extents(0);
  edges[2] = extents(1);

	int32 size, dim_id = SDgetdimid( fSDS_id, 0 );
  SDdiminfo( dim_id, NULL, &size, NULL, NULL);
  start[0] = size;
  start[1] = 0;
  start[2] = 0;
  istat = SDwritedata(fSDS_id,start,NULL,edges,Data());

  if(istat != 0) printf("\nerror writing SDS file: %s\n",fDSetName.chars());
#endif
  return istat;
}

int32 TSDArray::ExportSetAttributes( int32 sds_id  ) {
#ifdef HAS_HDF
  char info[kAttrStrLen];
  ReadAttr("info",info);
  SDsetattr(sds_id,"info",DFNT_CHAR8,sizeof(info),info);
  int R[4];
  int32 rindex = SDfindattr(fSDS_id,"region");
  SDreadattr(fSDS_id,rindex,R);
  SDsetattr( sds_id, "region", DFNT_INT32, 4, R );
#endif
	return 0;
}

int32 TSDArray::SDRead( int index, int32 ref, char* info, char* units, char* format   ) {
	int32 istat = -1;
#ifdef HAS_HDF
	fSD_id = HDFM::Open(HDFM::kFInput);
  if( fSDS_id == 0 ) {
		int32 sds_index; 
    if( ref < 0 ) sds_index = HDFM::GetSDS( HDFM::kFArray, DSetName(), VariableName() );
    else sds_index = SDreftoindex(fSD_id,ref); 
    fSDS_id = SDselect(fSD_id,sds_index);
    if( fSDS_id == FAIL ) { 
			sprintf(gMsgStr," Can't find SDS file %s : %s",VariableName(),DSetName()); 
			gPrintErr(); return FAIL;
		}
    int R[4];
    int32 rindex = SDfindattr(fSDS_id,"region");
    SDreadattr(fSDS_id,rindex,R);
    setregion(R[0],R[1],R[2],R[3]);
    Alloc();
    char name[kAttrStrLen];
    SDgetdatastrs(fSDS_id, name, units, format, NULL, kAttrStrLen );
    DSetName(name);
  } else SDgetdatastrs(fSDS_id, NULL, units, format, NULL, kAttrStrLen );
  ReadAttr("info",info);

  int32 start[3], edges[3];
  edges[0] = 1;
  start[0] = index;
  edges[1] = extents(0);
  edges[2] = extents(1);
  start[1] = 0;
  start[2] = 0;
	istat = SDreaddata(fSDS_id,start,NULL,edges,Data()); 
#endif
  return istat;
}

int TSDArray::TextDump( const CString& path, int index, int nData, const char* info, const char* units, const char* format ) const {

  FILE* outFile;
  CString filepath = path + fVariableName.chars() + "_" + fDSetName.chars() + ".TEXT";
  if(index==0) outFile = fopen(filepath.chars(),"w");
  else outFile = fopen(filepath.chars(),"a");
  if(outFile==NULL) { fprintf(stderr,"Can't open file: %s\n",filepath.chars()); fflush(stderr); return 0; }
  Dump(outFile);
  if(info) fprintf(outFile,"Info = %s\n",info);
  if(units) fprintf(outFile,"Units = %s\n",units);
  if(format) fprintf(outFile,"Format = %s\n",format);
  return 0;
}

void TSDArray::Dump( FILE* outFile ) const { 
  if(outFile==NULL) return;
  fprintf(outFile,"\n\n_________Text Dump for TSDArray %s_%s__________\n",
	  fVariableName.chars(),fDSetName.chars());
  for(int ir=0; ir<extents(0); ir++) {
    for(int ic=0; ic<extents(1); ic++) {
      float value = (*this)(ir,ic);
      fprintf(outFile,"%f ",value);
    }
    fprintf(outFile,"\n");
  }
  fclose(outFile);
}

#ifdef HAS_X
int TSDArray::ViewData( TSheet* aSheet, int index, int nData, char* info ) {
  float fmin, fmax;
  int cols = extents(1);
  int rows = extents(0);
  aSheet->SetDomain(0.0,1.0);
  CString Name = fVariableName + " " + fDSetName;
  aSheet->SetName(Name.chars());
  aSheet->SetData( Data(), rows, cols );
 
  aSheet->Flush();
}
#endif

void TSDArray::CopyFromCoverage ( PointSetCoverage cov, OrderedPointSet& set ) {
  for (Pix i = set.first(); i; set.next(i)) {
    const OrderedPoint& point = set.GetPoint(i);
    if( inside(point) ) { (*this)[point] = cov.Value(point); }
  }
}

void TSDArray::CopyFromCoverage ( PointSetCoverage cov, Region2& r ) {
  OrderedPointSet set;
  Region2 lr = r * (*this); 
  set.CopyRegion( cov.Grid(), lr );
  for (Pix i = set.first(); i; set.next(i)) {
    const OrderedPoint& point = set.GetPoint(i);
    (*this)[point] = cov.Value(point);
  }
}

void TSDArray::Condense( int targetProc ) {
  if( targetProc < 0 ) targetProc = GetOwner();  
#ifdef USE_MPI
  if( gIProc == targetProc ) {
    MPI_Status status; 
    if( fGlobalRegion != *this ) ReAlloc(fGlobalRegion);
    float* dataTmp = new float[ fGlobalRegion.nelem() ];
    for(int ip=0; ip<gNProcs(); ip++) {
      Region2 srcRegion = Net::LocalRegion(ip) *  fGlobalRegion;
      int size = srcRegion.nelem();
      if( size > 0 ) {
				MPI_Recv(dataTmp,size,MPI_FLOAT,ip,kCondenseTag,MPI_COMM_WORLD,&status);	
				CopyDataFrom(dataTmp,srcRegion);
      }
    }
    delete[] dataTmp;
  } else {
    int size = nelem();
    if(size>0) MPI_Send(Data(),size,MPI_FLOAT,targetProc,kCondenseTag,MPI_COMM_WORLD);
  }
#endif  
}




  

 


