//----------------------------------------------------------------------------------------
//	TSliceSeries.cc
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//----------------------------------------------------------------------------------------

#include "TSeries.h"
#include "HDFMgr.h"
#include "MML_Time.h"
#include "limits.h"
#include "Utilities.h"

#ifdef USE_MPI
MPI_Comm TSliceSeries::fComm = 0;
#endif

int TSliceSeries::fSeriesCount = 0;
float TSliceSeries::fGarbage = 0.0;


/*
*************************************************************************
*									*
* TSliceSeries								*
*									*
*************************************************************************
*/


int32 TSliceSeries::SDFinalize() {
  if( SDSObject::SDFinalize() ) { return 1; }
  else return 0;
} 


TSliceSeries&  TSliceSeries::operator= ( TSliceSeries& ts ) { 
	memcpy(fSeriesInfo,ts.fSeriesInfo,8); 
	fSeriesInfo[kTS_DataType] = ts.fSeriesInfo[kTS_DataType]; 
	fCurrentIndex = ts.fCurrentIndex; 
	fCurrentTime = ts.fCurrentTime;  
	fReadIndex = ts.fReadIndex; 
	fSeriesIndex = ts.fSeriesIndex;
	fSeriesCount = ts.fSeriesCount; 
	return *this;
}


/*
*************************************************************************
*									*
* TTemporalSeries								*
*									*
*************************************************************************
*/

int TTemporalSeries::_days_in_month[] = { 0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

void TTemporalSeries::PrettyPrint( CString& result, int start, int end ) {
	result = "\n(\n    Time     Value";
	end = ( (end<0) || (end>fence()) ) ? fence() : end;
    for (int i = start; i < end; ++i) {
      float time = (*this)[i].Time();
      float value = elem(i).Value();
      result += format("\n%12.3f ",time);
	  result += format(" %12.4e",value);
	} 
	result += "\n)\n";
}

void TTemporalSeries::PrettyPrint( FILE* ofile ) {
	fprintf(ofile,"\n\n\tTime     Value");
    for (int i = 0; i < fence(); ++i) {
      float time = (*this)[i].Time();
      float value = elem(i).Value();
      fprintf(ofile,"\n\t%8.2f %e",time,value);
	} 
	fflush(ofile);     
}

float TTemporalSeries::InterpValue(float time) {
  if( empty() ) return kSeriesNullValue;
  if( time == fCurrentTime ) return fCurrentValue;
  if ( time > fCurrentTime ) {
    float cur_time, prev_time = fCurrentTime;
		fCurrentTime = time;
    for (int i = fCurrentIndex+1; i < fence(); ++i) {
      cur_time = (*this)[i].Time();
      if( cur_time == time ) { 
				fCurrentIndex = i;
				fCurrentTimeSlice = &elem(fCurrentIndex);
				return fCurrentValue = fCurrentTimeSlice->Value(); 
			}
      else if( cur_time > time ) {
				float dt = (time-prev_time)/(cur_time - prev_time);
				float V0 = (*this)(i-1);
				if(fCurrentIndex != i-1) { 
					fCurrentIndex = i-1;
					fCurrentTimeSlice = &elem(fCurrentIndex);
				}
				fCurrentValue = V0 + dt * ( elem(i).Value() - V0 ); 
				return fCurrentValue; 
      }
      prev_time = cur_time;
    }
    return fCurrentValue = LastValue();
  } else {
    float cur_time, prev_time = fCurrentTime;
		fCurrentTime = time;
		for (int i = fCurrentIndex-1; i > ecnef(); --i) {
			cur_time = (*this)[i].Time();
			if( cur_time == time )  { 
				fCurrentIndex = i;
				fCurrentTimeSlice = &elem(fCurrentIndex);
				return fCurrentValue = fCurrentTimeSlice->Value(); 
			}
			else if( cur_time < time ) {
				float dt = ( time - cur_time )/( prev_time - cur_time );
				float V1 = (*this)(i+1);
				float V0 = (*this)(i);
				if(fCurrentIndex != i) { 
					fCurrentIndex = i;
					fCurrentTime = cur_time;
					fCurrentTimeSlice = &elem(fCurrentIndex);
				}
				fCurrentValue = V0 + dt * ( V1 - V0 );
				return fCurrentValue ;  
			}
			prev_time = cur_time;
		}
		return fCurrentValue = FirstValue();
  }
}

int TTemporalSeries::Add(float time, float value) {
  if( fSeriesInfo[kTS_DataType] == kTS_NULL) fSeriesInfo[kTS_DataType] = kTS_Value;
  else if ( fSeriesInfo[kTS_DataType] == kTS_Array ) gPrintErr("Mixed Datatypes in TSliceSeries.");
	fCurrentTimeSlice = grow_high(fCurrentIndex);
	fCurrentTimeSlice->SetData(time,value);
  fCurrentTime = time;
  fCurrentValue = value;
	return fCurrentIndex;
}

#ifdef HAS_X

int TTemporalSeries::GraphData( TGraph* aGraph ) {
  float fmin, fmax;
  if( fSeriesInfo[kTS_DataType] != kTS_Value) {
    gPrintErr( " Can't graph spatial or unallocated TimeSeries " ); 
    return 0;
  }   
  floatVec data(Size());
  floatVec time(Size());

  for(int i=0; i<Size(); i++) {
    data[i] = (*this)[i].Value();
    time[i] = TimeStamp(i);
    if( data[i]<fmin || i==0 ) fmin = data[i];
    if( data[i]>fmax || i==0 ) fmax = data[i];
  }
  float t0 = time[0];
  float tf = time[Size()-1];
  float dt = (tf-t0)/Size();
  if( fmin == fmax ) { fmax += 1.0; fmin -= 1.0; }
  aGraph->SetRange(fmin,fmax);
  aGraph->SetDomain(t0,tf,dt);
  CString Name = fVariableName + " " + fDSetName;
  aGraph->SetName(Name.chars());
  aGraph->SetData( data.data(), Size() );
 
  aGraph->Flush();

}

int TTemporalSeries::ViewData( TSheet* aSheet ) {
  float fmin, fmax;
  if( fSeriesInfo[kTS_DataType] != kTS_Value) {
    gPrintErr( " Can't graph spatial or unallocated TimeSeries " ); 
    return 0;
  }   
  floatVec data(2*Size());

  for(int i=0; i<Size(); i++) {
    data[2*i] = (*this)[i].Value();
    data[2*i+1] = TimeStamp(i);
    if( data[i]<fmin || i==0 ) fmin = data[i];
    if( data[i]>fmax || i==0 ) fmax = data[i];
  }
  int cols = 2;
  int rows = Size();
  aSheet->SetDomain(0.0,1.0);
  CString Name = fVariableName + " " + fDSetName;
  aSheet->SetName(Name.chars());
  aSheet->SetData( data.data(), rows, cols );
 
  aSheet->Flush();
}

#endif

inline int isNonDay( int month, int line ) {
	return ( ( ( month == 2 ) && ( line > 28 ) ) || 
				 ( ( ( month == 4 ) || ( month == 6 ) || ( month == 9 ) || ( month == 11 ) ) && ( line > 30 ) ) );
}

inline int get_field( FILE *infile,  int length, char sep = 0, char* data = NULL ) {
  int count = 0, test;
  while( ( test = fgetc(infile) ) != EOF ) {
	if( (sep > 0) && (test == sep) ) { return -2; }
	if( data != NULL ) { data[count] = test; }
	if( ++count == length ) break;
  }
  if( (count < length) ) return 1;
  if( sep > 0 ) {
	if( (( test = fgetc(infile) ) != sep) && ( test != '\n' ) && ( test != EOF )) { return -1; }
  }
  return 0;
}

inline int skip_line( FILE *infile, CString& buff ) {
  int count = 0, test; buff.clear();
  while( ( test = fgetc(infile) ) != EOF ) {
	buff += (char) test;
	if( test == '\n' ) return 0;
  }
  return -1;
}

inline void syntax_error(const char* file, const char* msg, const char* loc = NULL, int i1=0, int i2=0, int i3=0, int i4=0 ) {
  const char* sloc = (loc == NULL) ? "" : loc;
  sprintf(gMsgStr, "Syntax error in timeseries file %s: %s (%s,%d,%d,%d,%d)", file, msg, sloc, i1, i2, i3, i4 ); gFatal();
}

int TTemporalSeries::Import( const char* pathName, int format )
{
  FILE *cfile = fopen(pathName,"r");
  int local_debug = 0, itmp=0;
  if(cfile==NULL) { 
    sprintf(gMsgStr,"Unable to open timeseries file (%s)\n",pathName); gPrintErr();
    Add( LastTimeStamp(), kSeriesNullValue );
    return 1; 
  }
  else { sprintf(gMsgStr,"Reading series file %s",pathName); gPrintLog(); }

  switch (format) {

  case 1: {
		static char* fileInput = NULL; 
		static int  buffLen = 0, test_index=0, start_title_line=0, title_line = 0;
		int pos=-1, filePtr = 0, in_dashline = 0, isCR = 0;
		while(1) {  			
			itmp=fgetc(cfile); 
			if( (itmp == EOF) ) { break; }
			if( pos<0 ) {
				if( itmp == '-' ) { in_dashline++; }
				else if( itmp != ' ' ) {
					if( itmp == '\n' ) { 
						if( in_dashline > 10 ) { 
							start_title_line = 1;
						} else if( title_line > 0 ) {
							pos = filePtr + 1;
							title_line = 0;
						}
					}	else { 
						in_dashline = 0; 
					}
				}
			}
			if( filePtr >= buffLen ) {
				int oldLen = buffLen;
				char* ctmp = new char[ buffLen = (filePtr + 256) ];
				if( fileInput != NULL ) {
					strncpy(ctmp,fileInput,oldLen);
					delete[] fileInput;
				}
				fileInput = ctmp;
			}
			if( itmp != '\r' ) { 
				if( itmp == '\n' ) {
					if( isCR == 0 ) { 
						fileInput[filePtr++] = '\n'; 
						isCR = 1; 
					}
				} else {
					if( start_title_line ) {
						if( itmp == ' ' ) {
							pos = filePtr;
						} else {
							title_line = 1;
						}
						start_title_line = 0;
					}
					fileInput[filePtr++] = (char) itmp; 
					isCR = 0;
				}
			} else { 
				if( isCR == 0 ) { 
					fileInput[filePtr++] = '\n'; 
					isCR = 1; 
				}
			}
		}
    int sLen = 372, line = 1; 
    int month = 0, jday=0, itmp=0; 
    char  ctest = '\n', cmark[373], ret = '\n', fnum[20];
    unsigned char cmove[373], cmv=0;
    floatVec tvar(380);
    sprintf(gMsgStr,"Reading file %s, size = %d, start pos = %d",pathName,filePtr,pos); gPrintScreen();
    if( pos < 0 )  {
			sprintf(gMsgStr,"\n\nerror: Can't find start position in: \n\n %s", fileInput ); gPrintScreen();
			exit(-1);
		}
    while( pos < filePtr ) {  			
      if( ctest == '\n' ) {
				if(line<31) { 
					while( isspace( ctest=fileInput[pos] ) ) { ++pos; }
					int j=0;
					while ( isdigit( ctest=fileInput[pos] ) ) {  fnum[j++] = ctest; pos++; }
					fnum[j++] = '\0';
					line = (int) atoi(fnum); 
					month = 0;
				} 
				else break;
      } else if( ctest == '\t' || ctest == ' ' )  { 
				month++; 
				while( ( ctest=fileInput[++pos] ) == ' ' || ctest == '\t' ) {;} 
      }
      else if (ctest == '*') { 
				tvar.elem( jday ) = -1; 
				cmark[ jday ] = '*'; 
				while( (ctest=fileInput[++pos] ) == '*' ) {;}
      }
      else if (ctest == 'T') { 
				tvar.elem(  jday ) = 0.0; 
				cmark[ jday ] = 'T'; 
				while( isalpha( ctest=fileInput[++pos]  ) ) {;}
      }
      else if (ctest == '-') { 
				if( ( ctest=fileInput[++pos] ) == '-' ) {
					if( isNonDay( month, line ) ) {
						tvar.elem( jday ) = -1; 
						cmark[ jday ] = '*'; 
					} else {
						tvar.elem(  jday ) = kSeriesNullValue; 
						cmark[ jday ] = '1'; 
					}
					while( ( ctest=fileInput[++pos] ) == '-' ) {;}
				}
				else {
					while( ( ctest=fileInput[++pos] ) == ' ' ) {;}
					if( isdigit(ctest) || ctest == '.' ) {
						int j=1;
						fnum[0] = '-';
						fnum[j] = ctest;
						while ( isdigit( ctest=fileInput[++pos] ) || ctest == '.' )  { fnum[++j] = ctest; }
						fnum[++j] = '\0';
						if( isNonDay( month, line ) ) {  //  ignore leap years
							tvar.elem( jday ) = -1; 
							cmark[ jday ] = '*'; 
						} else {
							tvar.elem(  jday ) = (float) atof(fnum); 
							cmark[ jday ] = '1'; 
						}
						if( isalpha(ctest) ) { ctest=fileInput[++pos]; }
					}
					else { sprintf(gMsgStr,"Format error in TS File %s, day = %d, month = %d\n",pathName,jday,month); gFatal(); } 
				}
      }
      else if( isdigit(ctest) || ctest == '.'  ) {
				int j=0;
				fnum[j] = ctest;
				while ( isdigit( ctest=fileInput[++pos] ) || ctest == '.' )  { fnum[++j] = ctest; }
				fnum[++j] = '\0';
				if( isNonDay( month, line ) ) {   //  ignore leap years
					tvar.elem( jday ) = -1; 
					cmark[ jday ] = '*'; 
				} else {
					tvar.elem(  jday ) = (float) atof(fnum); 
					cmark[ jday ] = '1'; 
				}
				if( isalpha(ctest) ) { ctest=fileInput[++pos]; }
      }	
      if(month<=12) jday = 31*(month-1) +line - 1 ;
      else jday = sLen;
    }
		 
    for(jday=0; jday<sLen; jday++) {
      if( cmark[jday] == '*') cmv++;
      cmove[jday] = cmv;
    }
    for(jday=0; jday<sLen; jday++) {
      if( cmark[ jday ] != '*') tvar.elem(  jday - cmove[jday] )  =  tvar[ jday  ];
    }
    for( jday=0; jday<365; jday++) Add( gTime()+jday, tvar[jday] );
    if( local_debug ) {
			sprintf(gMsgStr,"\n\n Reading Timeseries file %s, pos = %d",pathName,pos); gPrintLog();
			for( jday=0; jday<365; jday++) { 
				if( jday % 12 == 0 ) { sprintf( gMsgStr, "\n %f ", tvar[jday] ); } 
				else { sprintf( gMsgStr, " %f ", tvar[jday] ); } 
				gPrintLog(); 
			}
    }
    fTS = 1.0;
    break; }
  case 2: {
    float time=0, value, dt;
    int sLen;
    int itest = Util::scan_forward(cfile,"@>"); if(itest==-1)  { return 0; }  
    itest = Util::scan_forward(cfile,"="); if(itest==-1)  { return 0; }  
    fscanf(cfile,"%f",&dt);
    itest = Util::scan_forward(cfile,"@>"); if(itest==-1)  { return 0; }   
    fscanf(cfile,"%d",&sLen);
    itest = Util::scan_forward(cfile,"@>"); if(itest==-1)  { return 0; }    
    for (int line = 0; line<sLen; line++) {
      itest = Util::scan_forward(cfile,">"); if(itest==-1)  { return 0; }   
      fscanf(cfile,"%f",&value);
      Add( time, value );
      time += dt;
    }
    sprintf(gMsgStr,"Reading file %s, status = %d",pathName,itest); gPrintScreen();
    fTS = dt;
    } break;
  case 3: {
    float time=0, value, dt;
    int sLen;
    int itest = Util::scan_forward(cfile,"@>"); if(itest==-1)  { return 0; }  
    itest = Util::scan_forward(cfile,"="); if(itest==-1)  { return 0; }  
    fscanf(cfile,"%f",&dt);
    itest = Util::scan_forward(cfile,"@>"); if(itest==-1)  { return 0; }   
    itest = Util::scan_forward(cfile,"="); if(itest==-1)  { return 0; }  
    fscanf(cfile,"%d",&sLen);
    itest = Util::scan_forward(cfile,"@>"); if(itest==-1)  { return 0; }    
    for (int line = 0; line<sLen; line++) {
      itest = Util::scan_forward(cfile,"\n"); if(itest==-1)  { return 0; }   
      fscanf(cfile,"%f",&value);
      Add( time, value );
      time += dt;
    }
    sprintf(gMsgStr,"Reading file %s, status = %d, lines read = %d",pathName,itest,sLen); gPrintScreen();
    fTS = dt;
    } break;
  case 5: {
		float time=((empty()) ? 0.0 : high_element().Time()), value, value1, dt;
		int test, incomment = 0, index = 0, max_index = INT_MAX, min_index=0, go=1;
		int first_read = 1, year_index = ((int)TTime::Year()), dt_year = TTime::DTYear();
		char comment_char = '#'; 
		char nodata_char = '~'; 
		byte has_dt = 0;
		CString label, svalue;
		if( gDebug) { sprintf(gMsgStr,"Reading TSformat-5 file %s: year %d",pathName,year_index); gPrintScreen(); gPrintLog(); }
		while( go && (( test = fgetc(cfile) ) != EOF) ) {
			if( test == comment_char ) { incomment = 1; }
			if( incomment && test == '\n' ) { incomment = 0; }
			if( !isspace(test) && (incomment == 0) ) {
				if( isalpha(test) ) {
				  if( test == nodata_char ) {
					if( has_dt ) { time += dt; index++; }
				  } else {
					label = (char)test; 
					while( isalpha( test = fgetc(cfile) ) ) { label += (char)test; }
					while( isspace( test = fgetc(cfile) ) || (test == '=') ) { if( test == EOF ) { go = 0; break; } }
					svalue = (char)test; 
					while( Util::isEnum( test = fgetc(cfile) ) ) { svalue += (char)test; }
					if( label == "dt" ) { 
						if( Util::is_float(svalue) ) { 
						  dt = atof(svalue); has_dt = 1;
						  if( gDebug) { sprintf(gMsgStr,"Set dt = %f",dt); gPrintLog(); }
						}
						else { sprintf(gMsgStr, "Ill formed value in timeseries Import: %s", svalue.chars() ); gPrintErr(); }				
						if( fStartYear >= 0 ) { 	
						  if( fStartYear < year_index ) { 
							min_index = (int) ((year_index-fStartYear)*(365/dt)); 
							if( gDebug) { 
							  sprintf(gMsgStr,"Set start year %d: start index = %d (current year = %d)",fStartYear,min_index,year_index); 
							  gPrintLog(); 
							}
						  }
						}		
					}
					else if( label == "year" ) { 
						if( Util::is_float(svalue) ) { fStartYear = atof(svalue); }
						else { sprintf(gMsgStr, "Ill formed value in timeseries Import: %s", svalue.chars() ); gPrintErr(); }
						if( has_dt ) { 	
						  min_index = (int) ((year_index-fStartYear)*(365/dt)); 
						  if( gDebug) { sprintf(gMsgStr,"Set start year %d: start index %d (current year %d)",fStartYear,min_index,year_index); gPrintLog(); }
						}		
					}
					else if( label == "start_time" ) { 
						if( Util::is_float(svalue) ) { time = atof(svalue); }
						else { sprintf(gMsgStr, "Ill formed value in timeseries Import: %s", svalue.chars() ); gPrintErr(); }				
					}
					else if( label == "last_index" ) { 
						if( Util::is_integer(svalue) ) { max_index = atoi(svalue); }
						else { sprintf(gMsgStr, "Ill formed value in timeseries Import: %s", svalue.chars() ); gPrintErr(); }				
					}
					else if( label == "first_index" ) { 
						if( Util::is_integer(svalue) ) { min_index = atoi(svalue); }
						else { sprintf(gMsgStr, "Ill formed value in timeseries Import: %s", svalue.chars() ); gPrintErr(); }				
					}
					else { sprintf(gMsgStr, "Unknown label in timeseries Import: %s", label.chars() ); gPrintErr();  }
					while( test != '\n' ) { if( ( test = fgetc(cfile) ) == EOF ) { go = 0; break; } }
				  }
				} else if ( Util::isFnum(test) ) {
					svalue = (char)test; 
					while( Util::isEnum( test = fgetc(cfile) ) ) { svalue += (char)test; }
					if( Util::is_float(svalue) ) { value = atof(svalue); }
					else { sprintf(gMsgStr, "Ill formed value in timeseries Import: %s", svalue.chars() ); gPrintErr(); }	
					if( has_dt ) {			
						if( (index < max_index) && (index >= min_index) ) { 
						  float time1 = time - min_index*dt;
						  Add( time1, value ); 
						  if( gDebug) { sprintf(gMsgStr,"Add time %f: value %f",time1,value); gPrintLog(); }
						} else if( gDebug) { sprintf(gMsgStr,"Skipping time %f",time); gPrintLog(); }
						time += dt; index++;
						while( test != '\n' ) { if( ( test = fgetc(cfile) ) == EOF ) { go = 0; break; } }
					} else {
						while( isspace(test) ) { 
							if( test == '\n' )  { sprintf(gMsgStr, "Missing value in timeseries Import (no dt): time %f -> (?)", value ); 
								gPrintErr(); go = 0; break; 
							}	
							if( ( test = fgetc(cfile) ) == EOF ) { go = 0; break; } 
						}
						if ( Util::isFnum(test) ) {
							svalue = (char)test; 
							while( Util::isEnum( test = fgetc(cfile) ) ) { svalue += (char)test; }
							if( Util::is_float(svalue) ) { value1 = atof(svalue); }
							else { sprintf(gMsgStr, "Ill formed value in timeseries Import: %s", svalue.chars() ); gPrintErr(); }						
							float time1 = time + value;
							if( index < max_index && index >= min_index ) { 
							  float time2 = time1 - min_index*dt;
							  Add( time2, value1 ); 
							  if( gDebug) { sprintf(gMsgStr,"Add time %f: value %f",time2,value1); gPrintLog(); }
							} else if( gDebug) { sprintf(gMsgStr,"Skipping time %f",time1); gPrintLog(); }
							index++;
							while( test != '\n' ) { if( ( test = fgetc(cfile) ) == EOF ) { go = 0; break; } }
						} else  { 
							sprintf(gMsgStr, "Syntax error in timeseries Import at line %d, (expecting float): %c (%x) ", index, test, test );
							gPrintErr(); go = 0; 
							break; 
						}
					}
				}
			}
		}
    sprintf(gMsgStr,"Reading file %s, lines read = %d",pathName,index); gPrintScreen();
    first_read = 0;
    fTS = dt;
    } break;
  case 6: {
	  TTemporalSeries buff;
	  int first_read = 1, year_index = ((int)TTime::Year());
	  float first_simtime = 0.0, simtime_lag = 0.0;
	  int month_index = ((int) ( (TTime::Year() - year_index ) * 12 )) + 1;
	  if( gDebug) { sprintf(gMsgStr,"Reading TSformat-6 file %s: year %d, month %d ",pathName,year_index,month_index); gPrintScreen(); gPrintLog(); }
	  char ctmp[7], date[7], ctest[2], cmo[3]; int ival = 0, month = 0, has_station_name = 0; 
	  ctmp[6] = '\0'; date[6] = '\0';  ctest[1] = '\0'; cmo[2] = '\0';
	  CString header, field_sizes;
	  for( int i=0; i<6; i++ ) { date[i] = '0'; }
      if( skip_line(cfile,header) != 0 ) { syntax_error(pathName, "no header" ) ; }   
      if( skip_line(cfile,field_sizes) != 0 ) { syntax_error(pathName, "no data" ) ; }  
      if(  header.contains("STATION NAME") ) { has_station_name = 1; }
      while( 1 ) {
		int rv = get_field( cfile,  4, ',' );
		if( rv == 1 ) break;
		if( rv != 0 )  { syntax_error(pathName, "DSET" ) ;  }  
		if( get_field( cfile,  6, ',' ) != 0 )  { syntax_error(pathName, "COOPID", date ) ;  }  
		if( get_field( cfile,  5, ',' ) != 0 )  { syntax_error(pathName, "WBNID", date ) ;  } 
		if( has_station_name ) { 
		  if( get_field( cfile,  30, ',' ) != 0 )  { syntax_error(pathName, "STATION NAME", date ) ;  }  
		}
		if( get_field( cfile,  2, ',' ) != 0 )  { syntax_error(pathName, "CD", date ) ;  }  
		if( get_field( cfile,  4, ',' ) != 0 )  { syntax_error(pathName, "ELEM", date ) ;  }  
		if( get_field( cfile,  2, ',' ) != 0 )  { syntax_error(pathName, "UN", date ) ;  }  
		if( get_field( cfile,  6, ',', date ) != 0 )  { syntax_error(pathName, "YEARMO", date ) ;  } 
		cmo[0] = date[4]; cmo[1] = date[5]; month = atoi(cmo); date[4] = 0;
		int year = atoi(date), add_data = 0;
		if( ( year >= year_index )  && ( month >= month_index ) ) { add_data = 1; }
		else if( gDebug) { sprintf(gMsgStr,"Skipping data: year %d, month %d",year,month); gPrintLog();	}
		for( int jday=1; jday<32; jday++ ) {
		  if( ( rv = get_field( cfile,  4, ',' )) != 0 )  { syntax_error(pathName, "DH", date, rv, jday ) ;  } 
		  if( ( rv = get_field( cfile,  6, ',', ctmp )) != 0 )  { syntax_error(pathName, "ELEM", date, rv, jday  ) ;  } 
		  if( ( rv = get_field( cfile,  1, ',', ctest )) != 0 )  { syntax_error(pathName, "M", date, rv, jday  ) ;  } 
		  if( ( rv = get_field( cfile,  1, ',' )) != 0 )  { syntax_error(pathName, "Q", date, rv, jday ) ;  } 
		  if( is_valid_date( month, jday ) && add_data ) {
			if( ctest[0] != 'M' ) {
			  ival = atoi(ctmp);
			}
			if( ival >= 0 ) {
			  float fyear = year + get_year_fraction( month, jday ) ;
			  float simtime = TTime::SimTimeFromYear( fyear );
			  if( first_read ) { 
				first_simtime = simtime; 
				simtime_lag = simtime - gTime();
				first_read = 0; 
			  }
			  Add( simtime, (float) ival );
			  if( gDebug) { sprintf(gMsgStr,"Add time %d.%d.%d (%f): %d",year,month,jday,simtime,ival); gPrintLog(); }
			  float lagged_time = simtime - simtime_lag;
			  if( lagged_time < first_simtime ) {
				buff.Add( lagged_time, (float) ival ); 
				if( gDebug) { sprintf(gMsgStr,"Add lagged time (%f): %d",lagged_time,ival); gPrintLog(); }
			  }
			}
		  }
		}
	  }
	  if( buff.length() > 0 ) {
		for( Pix p = buff.last(); p; buff.prev(p) ) {
		   TTemporalTimeSlice&  ts  = buff.elem (p); 
		   add_low(ts);
		}
	  }
	} break;
  default:
    sprintf(gMsgStr,"ERROR: unknown format in read_timeseries: %d\n",format); gFatal();
  }
  fclose(cfile);
  if( (fCurrentTimeSlice == NULL) && !empty() ) {
	 fCurrentTimeSlice =  &low_elem();
	 if(fCurrentTimeSlice) { fCurrentTime = fCurrentTimeSlice->Time(); fCurrentValue = fCurrentTimeSlice->Value();}
  }
  return 1;
}


int TTemporalSeries::AddPointInformation(int index, Point2& point) {
#ifdef HAS_HDF
  if(  fSDS_id == 0 ) return 0;
  char name[20];
  sprintf(name,"point%d",index);
  int p[2]; p[0] = point(0); p[1] = point(1);
  SDsetattr(fSDS_id,name,DFNT_INT32,2,p);
#endif
  return 0;
}

int TTemporalSeries::GetPointInformation(int index, Point2& point) {
#ifdef HAS_HDF
  if( fSDS_id == 0 ) return 0;
  char name[20];
  sprintf(name,"point%d",index);
  int p[2];
  ReadAttr(name,p); 
  point.elem(0) = p[0]; point.elem(1) = p[1];
#endif
  return 0;
}

void TTemporalSeries::AddData( float* data, int np, float t0, float ts ) 
{  
  for( int i=0; i<np; i++) Add(t0+i*ts, data[i]);
  fSeriesInfo[kTS_ConstantStep] = True;
  fTS = ts;
}

int TTemporalSeries::GetData( float*& data ) 
{
  if(data) delete [] data;
  int np = length();
  data = new float[np+1];
  for( int i=0; i<np; i++) data[i] = Value(i);
  data[np] = fTS;
  return np;
}

int32 TTemporalSeries::SDWrite( const char* info, const char* units, const char* format ) 
{
	int32 istat = -1;
#ifdef HAS_HDF
	fSD_id = HDFM::Open(HDFM::kFOutput);
  if( fSDS_id == 0 ) {
    int32 dims[3];
    dims[0] = SD_UNLIMITED;
    dims[1] = Size();
    dims[2] = 1;
    fSDS_id = SDcreate( fSD_id, (char*)DSetName(), DFNT_FLOAT32, 3, dims );
    SDsetdatastrs( fSDS_id, (char*)DSetName(), (char*)units, (char*)format, NULL );
    SDsetattr(fSDS_id,"info",DFNT_CHAR8,sizeof(info),(void*)info);
    SDsetattr(fSDS_id,"length",DFNT_INT32,1,(void*)Size());
    SDsetattr(fSDS_id,"uniform_DT",DFNT_CHAR8,1, (void*)(&fSeriesInfo[kTS_ConstantStep]) );
    SDsetattr(fSDS_id, "DT", DFNT_FLOAT32, 1, (void*)(&fTS));
    int dim_id = SDgetdimid(fSDS_id,0); 
    SDsetdimname(dim_id,"Series");
    dim_id = SDgetdimid(fSDS_id,1); 
    SDsetdimname(dim_id,"index");
    dim_id = SDgetdimid(fSDS_id,2); 
    SDsetdimname(dim_id,"Time");
    HDFM::AddSDS(fSDS_id,HDFM::kFSeries,VariableName());
  }

  int32 start[3], edges[3];
  floatVec tval(Size()), fval(Size());
  edges[0] = 1;
  int32 size, dim_id = SDgetdimid( fSDS_id, 0 );
  SDdiminfo( dim_id, NULL, &size, NULL, NULL);
  start[0] = size;

	edges[1] = Size();
	edges[2] = 1;
	start[1] = 0;
	start[2] = fSeriesIndex;
	for( int i=0; i< Size(); i++) {
		 tval.elem( i ) = TimeStamp(i);
		 fval.elem( i ) = (*this)(i);
	}
	istat = SDwritedata(fSDS_id,start,NULL,edges,(void*)fval.data());

  dim_id = SDgetdimid(fSDS_id,1);  
  if( GetSeriesInfo(kTS_ConstantStep)) {
    tval.elem( 1 ) = tval[ Size()-1];
    SDsetdimscale(dim_id,2,DFNT_FLOAT32,&(tval.elem(0)));
  } else {
    SDsetdimscale(dim_id,Size(),DFNT_FLOAT32,&(tval.elem(0)));
  }
  if(istat != 0) printf("\nerror writing SDS file: %s\n",SeriesName());
#endif
  return istat;
}

int32 TTemporalSeries::ExportSetAttributes( int32 sds_id  )
{
#ifdef HAS_HDF
  int32 length;
  char info[ kAttrStrLen ];
  ReadAttr("info",info);
  ReadAttr("uniform_DT",&fSeriesInfo[kTS_ConstantStep]);
  ReadAttr("DT",&fTS);
  ReadAttr("length",&length);

  SDsetattr(sds_id,"info",DFNT_CHAR8,sizeof(info),info);
  SDsetattr(sds_id,"length",DFNT_INT32,1,&length);
  SDsetattr(sds_id,"uniform_DT",DFNT_CHAR8,1, &fSeriesInfo[kTS_ConstantStep] );
  SDsetattr(sds_id, "DT", DFNT_FLOAT32, 1, &fTS );

  int32 dim_id = SDgetdimid(sds_id,0); 
  SDsetdimname(dim_id,"Series");
  dim_id = SDgetdimid(sds_id,1); 
  SDsetdimname(dim_id,"index");
  dim_id = SDgetdimid(sds_id,2); 
  SDsetdimname(dim_id,"Time");

  floatVec time( Size() );
  dim_id = SDgetdimid(fSDS_id,1);
  SDgetdimscale( dim_id, (void*)time.data() ); 
  dim_id = SDgetdimid(sds_id,1);
  if( GetSeriesInfo(kTS_ConstantStep)) {
    SDsetdimscale(dim_id,2,DFNT_FLOAT32,(void*)time.data());
  } else {
    SDsetdimscale(dim_id,Size(),DFNT_FLOAT32,(void*)time.data());
  }
#endif
	return 0;
}


int32 TTemporalSeries::SDRead( int index, int32 ref, char* info, char* units, char* format   ) {
	int32 istat = -1; 
#ifdef HAS_HDF
	fSD_id = HDFM::Open(HDFM::kFInput);
  if( fSDS_id == 0 ) {
		int32 sds_index; 
    if( ref < 0 ) sds_index = HDFM::GetSDS( HDFM::kFSeries, DSetName(), VariableName() );
    else sds_index = SDreftoindex(fSD_id,ref); 
    fSDS_id = SDselect(fSD_id,sds_index);
    if( fSDS_id == FAIL ) { 
			sprintf(gMsgStr," Can't find SDS file %s : %s",VariableName(),DSetName()); 
			gPrintErr(); return FAIL;
		}
    ReadAttr("info",info);  
    ReadAttr("uniform_DT",&fSeriesInfo[kTS_ConstantStep]);
    ReadAttr("DT",&fTS);
    int32 length;
    ReadAttr("length",&length);
    char name[kAttrStrLen];
    SDgetdatastrs(fSDS_id, name, units, format, NULL, kAttrStrLen );
    DSetName(name);
  } else SDgetdatastrs(fSDS_id, NULL, units, format, NULL, kAttrStrLen  );

  clear();
  int32 start[3], edges[3];
  edges[0] = 1;
  start[0] = index;

  edges[1] = 1;
  edges[2] = 1;
  start[2] = 0;

  floatVec time( Size() );
  int32 dim_id = SDgetdimid(fSDS_id,1); 
  SDgetdimscale( dim_id, &(time.elem(0)) ); 
  if( GetSeriesInfo(kTS_ConstantStep)) {
    for(int i=1; i<Size(); i++ ) time.elem(i) = time.elem(0) + i*fTS;
  }
	float value;
	for( int i=0; i<Size(); i++) {
		start[1] = i;
		int32 istat = SDreaddata(fSDS_id,start,NULL,edges,&value);
		Add(time[i],value);
	}
  if( fCurrentTimeSlice == NULL ) {
		fCurrentTimeSlice =  &low_elem();
		if(fCurrentTimeSlice) { fCurrentTime = fCurrentTimeSlice->Time();  fCurrentValue = fCurrentTimeSlice->Value(); }
	}
#endif
  return istat;
}
  

int TTemporalSeries::TextDump( CString& path, int index, int nData, const char* info, const char* units, const char* format ) const  {

  FILE* outFile;
  path += fVariableName + "_" + fDSetName.chars() + ".TEXT";
  if(index==0) outFile = fopen(path.chars(),"w");
  else outFile = fopen(path.chars(),"a");
  if(outFile==NULL) { sprintf(gMsgStr,"Can't open file: %s\n",path.chars()); gPrintErr(); return 0; }
  fprintf(outFile,"\n\n_________Text Dump for TSeries %s_%s__________\n",
	  fVariableName.chars(),fDSetName.chars());
  if(info) fprintf(outFile,"Info = %s\n",info);
  if(units) fprintf(outFile,"Units = %s\n",units);
  if(format) fprintf(outFile,"Format = %s\n",format);
  fprintf(outFile,"Index \tTime \tValues");
  for( int time_index=0; time_index<Size(); time_index++) { 
    float time = TimeStamp(time_index);   
		float value = (*this)[time_index]();
		fprintf(outFile,"\n%d: \t%f \t%f",time,value);
  }
  fclose(outFile);
}

TTemporalSeries&  TTemporalSeries::operator= ( TTemporalSeries& ts )  {
  TSliceSeries::operator= ( ts );
	TTemporalTimeSliceRPlex::operator=(ts);
  fCurrentTimeSlice = ts.fCurrentTimeSlice;
  return *this;
}

/*
*************************************************************************
*									*
* TSpatialSeries								*
*									*
*************************************************************************
*/

float* TSpatialSeries::Data(float time) {
  if( time == fCurrentTime ) return (float*)fCurrentTimeSlice->Data();
  else if ( time > fCurrentTime ) {
    float cur_time, prev_time = fCurrentTime;
    for (int i = fCurrentIndex+1; i < fence(); ++i) {
      cur_time = (*this)[i].Time();
      if( cur_time == time ) return Data(i);
      else if( cur_time > time ) {
				if( (cur_time - time) < (cur_time-prev_time)/2 ) return Data(i);
				else return Data(i-1);
      }
      prev_time = cur_time;
    }
    TSpatialTimeSlice& ts = (TSpatialTimeSlice&)high_element();
    return (float*)ts.Data();
  } else {
    float cur_time, prev_time = fCurrentTime;
      for (int i = fCurrentIndex-1; i > ecnef(); --i) {
				cur_time = (*this)[i].Time();
				if( cur_time == time ) return Data(i);
				else if( cur_time < time ) {
					if( ( time - cur_time ) < (prev_time-cur_time)/2 ) return Data(i);
					else return Data(i+1);
				}
				prev_time = cur_time;
      }
    TSpatialTimeSlice& ts = (TSpatialTimeSlice&)low_element();
    return (float*)ts.Data();
  }
}

TSpatialSeries&  TSpatialSeries::operator= ( TSpatialSeries& ts )  {
  TSliceSeries::operator=(ts);
  TSpatialTimeSliceRPlex::operator=(ts);
  fArraySize = ts.fArraySize ;
  fCurrentTimeSlice = ts.fCurrentTimeSlice;
  fPixStack = ts.fPixStack;
  return *this;
}


int TSpatialSeries::Add(float time, float* valp, int arraySize, ETS_CopyType cType  ) {
  if( fSeriesInfo[kTS_DataType] == kTS_NULL) { fSeriesInfo[kTS_DataType] = kTS_Array; fArraySize = arraySize; }
  else if ( fSeriesInfo[kTS_DataType] == kTS_Value ) gPrintErr("Mixed Datatypes in TSliceSeries.");
#ifdef DEBUG
  if(arraySize !=  fArraySize ) gPrintErr( " Uneven Array sizes in TSliceSeries " );
#endif
  float* valPtr;
  if( cType == kTS_Deep ) {
		if( fPixStack && !fPixStack->empty() ) { valPtr = (float*)fPixStack->pop(); }
    else { valPtr = new float[arraySize]; }
    memcpy(valPtr,valp,arraySize*sizeof(float));
  } else valPtr = valp;
	fCurrentTimeSlice = grow_high(fCurrentIndex);
	fCurrentTimeSlice->SetData(time,valPtr,cType,fPixStack);
	fCurrentTime = time;
	return fCurrentIndex;
}

int TSpatialSeries::Allocate( float time, int arraySize  ) {
  if( fSeriesInfo[kTS_DataType] == kTS_NULL) { 
    fSeriesInfo[kTS_DataType] = kTS_Array;
    if( arraySize > 0 ) fArraySize = arraySize;  
  }
  else if ( fSeriesInfo[kTS_DataType] == kTS_Value ) gPrintErr("Mixed Datatypes in TSliceSeries.");
#ifdef DEBUG
  if( arraySize > 0 && arraySize !=  fArraySize ) gPrintErr( " Uneven Array sizes in TSliceSeries " );
#endif
	float* valPtr;
	if( fPixStack && !fPixStack->empty() ) { valPtr = (float*)fPixStack->pop(); }
	else { valPtr = new float[arraySize]; }
	fCurrentTimeSlice = grow_high(fCurrentIndex);
	fCurrentTimeSlice->SetData(time,valPtr,kTS_Deep,fPixStack);
	fCurrentTime = time;
	return fCurrentIndex;
}

void TSpatialSeries::DumpToFile( FILE* oFile ) {
  float ftmp;
  for( int i=lo; i<fnc; i++ ) { 
	TSpatialTimeSlice& ts = elem (i);
    fprintf(oFile,"\n%f",ts.Time());
	for( int i=0; i<fArraySize; i++ ) {
	  ftmp =  ts.Value(i);
	  fprintf(oFile," %f",ftmp);
	}
  }
}

int32 TSpatialSeries::SDWrite( const char* info, const char* units, const char* format ) 
{
	int32 istat = -1;
#ifdef HAS_HDF
	fSD_id = HDFM::Open(HDFM::kFOutput);
  if( fSDS_id == 0 ) {
    int32 dims[3];
    dims[0] = SD_UNLIMITED;
    dims[1] = Size();
    dims[2] = fArraySize;
    fSDS_id = SDcreate( fSD_id, (char*)DSetName(), DFNT_FLOAT32, 3, dims );
    SDsetdatastrs( fSDS_id, (char*)DSetName(), (char*)units, (char*)format, NULL );
    SDsetattr(fSDS_id,"info",DFNT_CHAR8,sizeof(info),(void*)info);
    SDsetattr(fSDS_id,"length",DFNT_INT32,1,(void*)Size());
    SDsetattr(fSDS_id,"array_size",DFNT_INT32,1,(void*)(&fArraySize));
    SDsetattr(fSDS_id,"uniform_DT",DFNT_CHAR8,1, (void*)(&fSeriesInfo[kTS_ConstantStep]) );
    SDsetattr(fSDS_id, "DT", DFNT_FLOAT32, 1, (void*)(&fTS));
    int dim_id = SDgetdimid(fSDS_id,0); 
    SDsetdimname(dim_id,"Series");
    dim_id = SDgetdimid(fSDS_id,1); 
    SDsetdimname(dim_id,"Space");
    dim_id = SDgetdimid(fSDS_id,2); 
    SDsetdimname(dim_id,"Time");
    HDFM::AddSDS(fSDS_id,HDFM::kFSeries,VariableName());
  }

  int32 start[3], edges[3];
  floatVec tval(Size()), fval(Size());
  edges[0] = 1;
  int32 size, dim_id = SDgetdimid( fSDS_id, 0 );
  SDdiminfo( dim_id, NULL, &size, NULL, NULL);
  start[0] = size;

	edges[2] = fArraySize;
	edges[1] = 1;
	for( int i=0; i< Size(); i++) {
		start[1] = i;
		start[2] = 0;
		int32 istat = SDwritedata(fSDS_id,start,NULL,edges,(void*)Data(i));
		tval.elem( i ) = TimeStamp(i);
	}

  dim_id = SDgetdimid(fSDS_id,1);  
  if( GetSeriesInfo(kTS_ConstantStep)) {
    tval.elem( 1 ) = tval[ Size()-1];
    SDsetdimscale(dim_id,2,DFNT_FLOAT32,&(tval.elem(0)));
  } else {
    SDsetdimscale(dim_id,Size(),DFNT_FLOAT32,&(tval.elem(0)));
  }
  if(istat != 0) printf("\nerror writing SDS file: %s\n",SeriesName());
#endif
  return istat;
}

int32 TSpatialSeries::ExportSetAttributes( int32 sds_id  )
{
#ifdef HAS_HDF
  int32 length;
  char info[ kAttrStrLen ];
  ReadAttr("info",info);
  ReadAttr("uniform_DT",&fSeriesInfo[kTS_ConstantStep]);
  ReadAttr("DT",&fTS);
  ReadAttr("length",&length);
  ReadAttr("array_size",&fArraySize);

  SDsetattr(sds_id,"info",DFNT_CHAR8,sizeof(info),info);
  SDsetattr(sds_id,"length",DFNT_INT32,1,&length);
  SDsetattr(sds_id,"array_size",DFNT_INT32,1,&fArraySize);
  SDsetattr(sds_id,"uniform_DT",DFNT_CHAR8,1, &fSeriesInfo[kTS_ConstantStep] );
  SDsetattr(sds_id, "DT", DFNT_FLOAT32, 1, &fTS );

  int32 dim_id = SDgetdimid(sds_id,0); 
  SDsetdimname(dim_id,"Series");
  dim_id = SDgetdimid(sds_id,1); 
  SDsetdimname(dim_id,"Space");
  dim_id = SDgetdimid(sds_id,2); 
  SDsetdimname(dim_id,"Time");

  floatVec time( Size() );
  dim_id = SDgetdimid(fSDS_id,1);
  SDgetdimscale( dim_id, (void*)time.data() ); 
  dim_id = SDgetdimid(sds_id,1);
  if( GetSeriesInfo(kTS_ConstantStep)) {
    SDsetdimscale(dim_id,2,DFNT_FLOAT32,(void*)time.data());
  } else {
    SDsetdimscale(dim_id,Size(),DFNT_FLOAT32,(void*)time.data());
  }
#endif
	return 0;
}


int32 TSpatialSeries::SDRead( int index, int32 ref, char* info, char* units, char* format  ) {
	int32 istat = -1; 
#ifdef HAS_HDF
	fSD_id = HDFM::Open(HDFM::kFInput);
  if( fSDS_id == 0 ) {
		int32 sds_index; 
    if( ref < 0 ) sds_index = HDFM::GetSDS( HDFM::kFSeries, DSetName(), VariableName() );
    else sds_index = SDreftoindex(fSD_id,ref); 
    fSDS_id = SDselect(fSD_id,sds_index);
    if( fSDS_id == FAIL ) { 
			sprintf(gMsgStr," Can't find SDS file %s : %s",VariableName(),DSetName()); 
			gPrintErr(); return FAIL;
		}
    ReadAttr("info",info);  
    ReadAttr("uniform_DT",&fSeriesInfo[kTS_ConstantStep]);
    ReadAttr("DT",&fTS);
    int32 length;
    ReadAttr("length",&length);
    ReadAttr("array_size",&fArraySize);
    char name[kAttrStrLen];
    SDgetdatastrs(fSDS_id, name, units, format, NULL, kAttrStrLen );
    DSetName(name);
  } else SDgetdatastrs(fSDS_id, NULL, units, format, NULL, kAttrStrLen  );

  clear();
  int32 start[3], edges[3];
  edges[0] = 1;
  start[0] = index;

  edges[1] = 1;
  edges[2] = fArraySize;
  start[2] = 0;

  floatVec time( Size() );
  int32 dim_id = SDgetdimid(fSDS_id,1); 
  SDgetdimscale( dim_id, &(time.elem(0)) ); 
  if( GetSeriesInfo(kTS_ConstantStep)) {
    for(int i=1; i<Size(); i++ ) time.elem(i) = time.elem(0) + i*fTS;
  }
	for( int i=0; i<Size(); i++) {
		start[1] = i;
		Allocate(time[i]);
		istat = SDreaddata(fSDS_id,start,NULL,edges,&(LastValue(0)));
	}
#endif
  return istat;
}
  

int TSpatialSeries::TextDump( CString& path, int index, int nData, const char* info, const char* units, const char* format ) const  {

  FILE* outFile;
  path += fVariableName + "_" + fDSetName.chars() + ".TEXT";
  if(index==0) outFile = fopen(path.chars(),"w");
  else outFile = fopen(path.chars(),"a");
  if(outFile==NULL) { sprintf(gMsgStr,"Can't open file: %s\n",path.chars()); gPrintErr(); return 0; }
  fprintf(outFile,"\n\n_________Text Dump for TSeries %s_%s__________\n",
	  fVariableName.chars(),fDSetName.chars());
  if(info) fprintf(outFile,"Info = %s\n",info);
  if(units) fprintf(outFile,"Units = %s\n",units);
  if(format) fprintf(outFile,"Format = %s\n",format);
  fprintf(outFile,"Index \tTime \tValues");
  for( int time_index=0; time_index<Size(); time_index++) { 
    float time = TimeStamp(time_index);   
      fprintf(outFile,"\n%d: \t%f ",time);
      for( int array_index=0; array_index < fArraySize; array_index++) {
				float value = (*this)[time_index](array_index);
				fprintf(outFile,"%f ",value);
      }
  }
  fclose(outFile);
}

void TSpatialSeries::PrettyPrint( CString& result, int start, int end ) {
	result = "\n(\n    Time         Values";
	end = ( (end<0) || (end>fence()) ) ? fence() : end;
    for (int i = start; i < end; ++i) {
      float time = (*this)[i].Time();
      const float* data = elem(i).Data();
      result += format("\n%12.3f ",time);
      for( int i=0; i<fArraySize; i++ ) {
		result += format(" %12.4e",data[i]);
	  }
	} 
	result += "\n)\n";
}



float TSpatialSeries::InterpValue(float time, int array_index) {
  if( empty() ) return kSeriesNullValue;
  if( time == fCurrentTime ) return fCurrentTimeSlice->Value(array_index);
  else if ( time > fCurrentTime ) {
    float cur_time, prev_time = fCurrentTime;
    for (int i = fCurrentIndex+1; i < fence(); ++i) {
      cur_time = elem(i).Time();
      if( cur_time == time ) { 
				fCurrentIndex = i;
				fCurrentTimeSlice = &elem(fCurrentIndex);
				return fCurrentTimeSlice->Value(array_index); 
			}
      else if( cur_time > time ) {
				float dt = (time-prev_time)/(cur_time - prev_time);
				float V0 = elem(i-1).Value(array_index);
				if(fCurrentIndex != i-1) { 
					fCurrentTime = prev_time;
					fCurrentIndex = i-1;
					fCurrentTimeSlice = &elem(fCurrentIndex);
				}
				return V0 + dt * ( elem(i).Value(array_index) - V0 );  
      }
      prev_time = cur_time;
    }
    return LastValue(array_index);
  } else {
    float cur_time, prev_time = fCurrentTime;
      for (int i = fCurrentIndex; i > ecnef(); --i) {
				cur_time = elem(i).Time();
				if( cur_time == time ) { 
					fCurrentIndex = i;
					fCurrentTimeSlice = &elem(fCurrentIndex);
					return fCurrentTimeSlice->Value(array_index); 
				}
				else if( cur_time < time ) {
					float dt = (time-prev_time)/(cur_time - prev_time);
					float V0 = elem(i-1).Value(array_index);
					if(fCurrentIndex != i) { 
						fCurrentTime = cur_time;
						fCurrentIndex = i;
						fCurrentTimeSlice = &elem(fCurrentIndex);
					}
					return V0 + dt * (  elem(i).Value(array_index) - V0 );  
				}
				prev_time = cur_time;
      }
      return FirstValue(array_index);
  }
}

void TSpatialSeries::SetCurrentTime(float time) {
  float cur_time;
  if ( time > fCurrentTime ) {
    for (int i = fCurrentIndex+1; i < fence(); ++i) {
      cur_time = (*this)[i].Time();
      if( (cur_time >= time) || (i == (fence()-1))) {
				fCurrentTime = cur_time;
				fCurrentIndex = i;
				fCurrentTimeSlice = &elem(fCurrentIndex);
				return;
			}
		}
	} else if( time < fCurrentTime ) {
		for (int i = fCurrentIndex-1; i > ecnef(); --i) {
			cur_time = (*this)[i].Time();
			if( cur_time <= time || i == ecnef()+1) {
				fCurrentTime = cur_time;
				fCurrentIndex = i;
				fCurrentTimeSlice = &elem(fCurrentIndex);
      }
    }
  }
}



