//----------------------------------------------------------------------------------------
//	TSliceSeries.h
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//	Change History:
//----------------------------------------------------------------------------------------

#ifndef __TSliceSeries__
#define __TSliceSeries__

#include "TimePlex.h"
#include "MultiCoverage.h"
#include "SDSObject.h"
#include "PixPlex.h"
#include "TMap.h"

#define kSeriesNullValue FLT_MIN

/*
*************************************************************************
*									*
* TSliceSeries								*
*									*
*************************************************************************
*/

enum ETSliceSeriesInfo { kTS_DataType, kTS_ConstantStep, kTS_SpatialReferenced };

class TSliceSeries : public SDSObject {

protected:

  int fCurrentIndex;
  float fCurrentTime;
  static float fGarbage;

  byte fSeriesInfo[8];
  int fSeriesIndex;
  int fReadIndex;
  static int fSeriesCount;
  float fTS;

#ifdef USE_MPI
  static MPI_Comm fComm;
#endif

public:

  TSliceSeries() : SDSObject() 
    { memset(fSeriesInfo,0,8); fSeriesInfo[kTS_DataType] = kTS_NULL; 
			fCurrentIndex = 0; fCurrentTime = -FLT_MAX;  fReadIndex = 0; 
			fSeriesIndex = fSeriesCount = 0; 
		}

  inline void SeriesName( CString name ) { fDSetName = name; }
  inline const char* SeriesName() { return fDSetName; }
  inline void Reset() { fCurrentIndex = 0; fCurrentTime = -FLT_MAX;   fReadIndex = 0; }

  byte GetSeriesInfo(ETSliceSeriesInfo index) const { return fSeriesInfo[index]; } 
  void SetSeriesInfo(ETSliceSeriesInfo index, byte value) { fSeriesInfo[index] = value; } 

  inline void SetupComm() {
#ifdef USE_MPI
  if( fComm == 0 )  { 			
		if( Env::GetComm(&fComm) ) {  gFatal("Error in TSliceSeries::SetupComm");  }   
	}
#endif
  }

  inline float DT() const { return fTS; }
  inline int SeriesIndex(int index) { return fSeriesIndex = index; }
	TSliceSeries& operator= ( TSliceSeries& ts );

  virtual int32 SDFinalize();
  
};

class TTemporalSeries : public TTemporalTimeSliceRPlex, public TSliceSeries {

  TTemporalTimeSlice* fCurrentTimeSlice;
  float fCurrentValue;
  int fStartYear;
  static int _days_in_month[];

public:

  TTemporalSeries(int chunkSize) : TTemporalTimeSliceRPlex(chunkSize)  {  fCurrentTimeSlice = NULL; fStartYear= -1; }
  TTemporalSeries() {  fCurrentTimeSlice = NULL;  fStartYear = -1; }

  inline void Reset() { fCurrentTimeSlice = NULL; TSliceSeries::Reset(); TTemporalTimeSliceRPlex::Reset();  }

  virtual int32 SDWrite( const char* info=NULL, const char* units=NULL, const char* format=NULL );
  virtual int32 SDRead( int index=0, int32 group_ref=-1, char* info=NULL, char* units=NULL, char* format=NULL );
  virtual int Import( const char* pathName, int format );
  
  virtual int32 ExportSetAttributes(int32 sds_id);
  virtual int TextDump( CString& path, int index, int nData, const char* info=NULL, const char* units=NULL, const char* format=NULL ) const ;

#ifdef HAS_X
  virtual int GraphData( TGraph* aGraph );
  virtual int ViewData( TSheet* aSheet );
#endif

  int Add(float time, float value);
  inline void SetStartYear(int startYear) { fStartYear = startYear; }
  void AddData( float* data, int np, float t0, float ts );
  int GetData( float*& data );
  inline int empty() const { return (fCurrentTimeSlice == NULL); }

  int Allocate( float time );
  
  static inline int is_valid_date( int month, int day ) {
	if( (month > 0) && (month <= 12) ) {
	  return ( ( day > 0 ) && ( day <= _days_in_month[month] ) ); 
	} else { return 0; }
  }

  static inline float get_year_fraction( int month, int day ) {
	int days = 0;
	for( int im = 1; im < month; im++ ) { days += _days_in_month[im]; }
	days += (day-1);
	return days/365.0;
  }

  TTemporalSeries& operator= ( TTemporalSeries& ts );

  inline float LastValue() const {  if( empty() ) { return kSeriesNullValue; } else return high_element()(); }
  inline float LastValueOffset(int offset) const {  if( empty() ) { return kSeriesNullValue; } else return high_element(offset)(); }
  inline float& LastValue() {  
	if( empty() ) { fCurrentValue = kSeriesNullValue; return fCurrentValue; } 
	else return high_elem().Value(); 
  }
  inline float FirstValue() const {   if( empty() ) { return kSeriesNullValue; } else return low_element()(); }
  
  inline float FirstTime() const {   if( empty() ) { return kSeriesNullValue; } else return low_element().Time(); }
  inline float LastTime() const {   if( empty() ) { return kSeriesNullValue; } else return high_element().Time(); }

  inline float Value( int time_index ) {
#ifdef DEBUG
      if( !valid(time_index) ) { gPrintErr( " Bad time index in TSliceSeries " ); return 0.0; }
#endif
	  if( empty() ) return kSeriesNullValue;
      return ((TTemporalTimeSlice)(*this)[time_index]).Value();
  }

  inline float operator() ( int time_index ) const {
#ifdef DEBUG
      if( !valid(time_index) ) { gPrintErr( " Bad time index in TSliceSeries " ); return 0.0; }
#endif
	  if( empty() ) return kSeriesNullValue;
      return ((TTemporalTimeSlice)(*this)[time_index]).Value();
  }

  inline float Value(float time) { return InterpValue(time); }

  inline float CurValue() {
	if( empty() ) return kSeriesNullValue;
    return fCurrentValue; 
  }

  float InterpValue(float time);
  void PrettyPrint( FILE* ofile = stdout );
  void PrettyPrint( CString& result, int start=0, int end = -1 );

  virtual void Dump(FILE* file = NULL) const {;}
  int AddPointInformation(int index, Point2& point);
  int GetPointInformation(int index, Point2& point);

	inline const float* TimeSeriesData() {  
		return TTemporalTimeSliceRPlex::TimeSeriesData();
	}
  inline int Size() const { return length(); }

  TTemporalSeries& operator= ( TSliceSeries& ts );

  inline float LastTimeStamp() const { return (empty()) ? -FLT_MAX : high_element().Time(); }
  inline float FirstTimeStamp() const { return (empty()) ? -FLT_MAX : low_element().Time(); }

	inline void Update( float time, float value  ) {
		if( fnc && fCurrentTime >= time  ) {
			LastValue() = value;
		} else {
			Add( time, value );
		}		
	}  
	
  inline float TimeStamp(int time_index)  const { return (*this)[time_index].Time(); }
  
  inline void DeleteBelow( float time ) {;}   // had trouble with this method so currently disabled.
  
};

class TSpatialSeries : public TSpatialTimeSliceRPlex, public TSliceSeries {

  TSpatialTimeSlice* fCurrentTimeSlice;
  int fArraySize;
  PixStack* fPixStack;

public:

  TSpatialSeries(int chunkSize) : TSpatialTimeSliceRPlex(chunkSize)  {  fCurrentTimeSlice = NULL;   fPixStack = NULL; }
  TSpatialSeries()  {  fCurrentTimeSlice = NULL;   fPixStack = NULL; }

  inline void Reset() {  clear();   fCurrentIndex = 0; fCurrentTime = -FLT_MAX; fCurrentTimeSlice = NULL;   fReadIndex = 0; }
  inline int empty() const { return (fCurrentTimeSlice == NULL); }

  virtual int32 SDWrite( const char* info=NULL, const char* units=NULL, const char* format=NULL );
  virtual int32 SDRead( int index=0, int32 group_ref=-1, char* info=NULL, char* units=NULL, char* format=NULL );  
  virtual int32 ExportSetAttributes(int32 sds_id);
  virtual int TextDump( CString& path, int index, int nData, const char* info=NULL, const char* units=NULL, const char* format=NULL ) const ;
  void PrettyPrint( CString& result, int start, int end );
  void DumpToFile( FILE* oFile );
    
  inline int ArraySize() const { return fArraySize; }
	inline void SetStack( PixStack& ps ) { fPixStack = &ps; }

  int Add(float time, float* valp, int arraySize, ETS_CopyType cType = kTS_Deep );
  int Allocate( float time, int arraySize = -1  );

  TSpatialSeries& operator= ( TSpatialSeries& ts );

  inline int Add(float time, MultiCoverage& cov, ETS_CopyType cType = kTS_Deep ) {
    return Add( time, cov.DataStart(), cov.Size(), cType );
  }

  inline void Extract( float time, MultiCoverage& cov  ) const {
    const float* data = Data(time);
    cov.CopyData( data , fArraySize ); 
  }

  inline float LastValue( int array_index ) const { 
#ifdef DEBUG
      if( array_index < 0 || array_index >= fArraySize ) { gPrintErr( " Bad Array index in TSliceSeries " ); return 0.0; }
#endif
      return ((TSpatialTimeSlice&)high_element()).Value( array_index );
  }

  inline float& LastValue( int array_index ) { 
#ifdef DEBUG
      if( array_index < 0 || array_index >= fArraySize ) 
	{ gPrintErr( " Bad Array index in TSliceSeries " ); return fGarbage; }
#endif
      return ((TSpatialTimeSlice&)high_element()).Value( array_index );
  }
  
  inline float& LastValueOffset( int offset, int array_index ) { 
#ifdef DEBUG
      if( array_index < 0 || array_index >= fArraySize ) 
	{ gPrintErr( " Bad Array index in TSliceSeries " ); return fGarbage; }
#endif
      return ((TSpatialTimeSlice&)high_element(offset)).Value( array_index );
  }
  
  inline float* LastDataOffset( int offset ) { 
      return (float*)((TSpatialTimeSlice&)high_element(offset)).Data();
  }

  inline float FirstValue( int array_index ) const { 
#ifdef DEBUG
      if( array_index < 0 || array_index >= fArraySize ) 
	{ gPrintErr( " Bad Array index in TSliceSeries " ); return 0.0; }
#endif
    return ((TSpatialTimeSlice&)low_element()).Value( array_index ); 
  }

  inline float Value( int time_index, int array_index = -1) {
#ifdef DEBUG
      if( !valid(time_index) ) { gPrintErr( " Bad time index in TSliceSeries " ); return 0.0; }
#endif
#ifdef DEBUG
      if( array_index < 0 || array_index >= fArraySize ) { gPrintErr( " Bad Array index in TSliceSeries " ); return 0.0; }
#endif
      return ((TSpatialTimeSlice&)(*this)[time_index]).Value(array_index);
  }

  inline float operator() ( int time_index, int array_index) const {
#ifdef DEBUG
      if( !valid(time_index) ) { gPrintErr( " Bad time index in TSliceSeries " ); return 0.0; }
      if( array_index < 0 || array_index >= fArraySize )  { gPrintErr( " Bad Array index in TSliceSeries " ); return 0.0; }
#endif
      return ((TSpatialTimeSlice&)(*this)[time_index]).Value(array_index);
  }

  inline const float* Data( int time_index ) const {
#ifdef DEBUG
      if( !valid(time_index) ) { gPrintErr( " Bad time index in TSliceSeries " ); return NULL; }
#endif
      return ((TSpatialTimeSlice&)(*this)[time_index]).Data();
  }

  inline float* Data( int time_index  ) {
#ifdef DEBUG
      if( !valid(time_index) ) { gPrintErr( " Bad time index in TSliceSeries " ); return NULL; }
#endif
      return ((TSpatialTimeSlice&)(*this)[time_index]).DataStart();
  }

  inline float* LastData() {
      return ((TSpatialTimeSlice&)high_element()).DataStart();
  }

  inline float* FirstData() {
      return ((TSpatialTimeSlice&)low_element()).DataStart();
  }

  float* Data(float time);
  
  inline float Value(float time, int array_index) {
    if( time == fCurrentTime ) { return fCurrentTimeSlice->Value(array_index); }
    else return InterpValue(time, array_index);
  }

  inline float CurValue(int array_index) {
    return fCurrentTimeSlice->Value(array_index); 
  }

  inline void DeleteBelow( float time ) {
/*
    for( Pix p = first(); p; next(p) ) {
			TSpatialTimeSlice* ts = (TSpatialTimeSlice*)p;
			if( ts->Time() < time ) {
				ts->Free();
			} else break;
    }
*/
    for( int time_index = lo; time_index<fnc; time_index++ ) {
			TSpatialTimeSlice& ts = elem(time_index);
			if( ts.Time() < time ) {
				ts.Free();
			} else break;
    }
  }
    
  inline void DeleteExtraSlicesBelow( int nSlicesToKeep ) {
		int high = fnc-nSlicesToKeep;
    while( lo < high ) {
      del_low();
    }
  }

  float InterpValue(float time, int array_index);

  virtual void Dump(FILE* file = NULL) const {;}

	inline const float* TimeSeriesData(int array_index) {  
#ifdef DEBUG
      if( array_index >= fArraySize )  { 
				gPrintErr( " Bad Array index in TSliceSeries " ); return NULL; 
			}
#endif
		return TSpatialTimeSliceRPlex::TimeSeriesData(array_index);
	}
  inline int Size() const { return length(); }

  inline float LastTimeStamp() const { return (empty()) ? -FLT_MAX : high_element().Time(); }
  inline float FirstTimeStamp() const { return (empty()) ? -FLT_MAX : low_element().Time(); }
  inline float TimeStamp(int time_index)  const { return (*this)[time_index].Time(); }
  
  inline void SetTime(float time) {
    if( time == fCurrentTime ) return;
    SetCurrentTime(time);
  }
  void SetCurrentTime(float time);

};


#endif
