/* A C Program to run GRAM, submitting a simple job, and 
 * waiting until it executes
 */

/*************************************************
                Included Files
**************************************************/

#include <stdio.h> 
#include <string.h> 
#include "globus_gram_client.h"

int scmp(const char* a, const char* b) {
  int diff = 0;
  while ((diff = *a - *b++) == 0 && *a++ != 0);
  return diff;
}

#define SAME( a, b ) (scmp(a,b) == 0)

/* The header file which includes all the GRAM functions */

/**************************************************
                  Functions
***************************************************/

/* This is a function that you write..later...but which must 
 * have this definition. It is the function used when the remote 
 * Job Manager needs to contact your local program to inform it 
 * about the status of the remote  program. It is passed along 
 * with the the job_request to the remote computer
 */

static void callback_func(void * user_callback_arg,
                          char * job_contact,
                          int state,
                          int errorcode);

/**************************************************
                 Type Definitions
*************************************************/

/* Setting up the GRAM monitor. The monitor will stall
 * this program until the remote program is terminated, 
 * either through failure or naturally. Without the monitor,
 * this program would submit the job, and end before the job
 * completed. The monitor works with a lock. Only one function
 * may access the Done flag at a time, so in order to access it,
 * the gram must set a lock on the monitor variable, so that 
 * nothing else may access it, then change it, and finally 
 * unlock it. This is seen later in the code.
 */ 


/* This whole structure is the monitor */

typedef struct
{
    globus_mutex_t mutex;
    globus_cond_t cond;
    globus_bool_t done;
} my_monitor_t;


/***************************************************
                   Main Code
***************************************************/

int main(int argc, char ** argv)
{
    int callback_fd;
    int job_state_mask;
    int rc; /* The return value of the request function. 
             * If successful, it should be 0 */

    char * callback_contact; /* This is the identifier for 
                              * the callback, returned by 
                              * globus_gram_job_request
                              */

    char * job_contact; /* This is the identifier for the job,
                         * returned by globus_gram_job_request
                         */

    char * rm_contact;
    char specification[800];
    float confidence;
    globus_bool_t done;
    globus_gram_client_time_t estimate;
    globus_gram_client_time_t interval_size;
    my_monitor_t Monitor;
    char* queue = "";

    /* Retrieve relevant parameters from the command line */ 
    if (argc != 6)
    {
        /* invalid parameters passed */
        printf("\n Usage: %s <contact> <task> <host> <project> <model>\n",argv[0]);    
        printf("\tContacts: modi4.ncsa.uiuc.edu-fork modi4.ncsa.uiuc.edu-lsf\n");    
        printf("\tTasks: test-ls project run \n");    
        return(1);
    }

    if ((rc = globus_module_activate(GLOBUS_GRAM_CLIENT_MODULE))
	!= GLOBUS_SUCCESS)
    {
	printf("\tERROR: gram module activation failed\n");
	return(1);
    }
    
    if( SAME(argv[1],"modi4.ncsa.uiuc.edu-fork") ) {
	  rm_contact = "modi4.ncsa.uiuc.edu:4002/jobmanager-fork:/C=US/O=Globus/O=The University of Illinois Urbana-Champaign/OU=The National Center for Supercomputing Applications/CN=modi4.ncsa.uiuc.edu";
    } else  if( SAME(argv[1],"modi4.ncsa.uiuc.edu-lsf")  ) {
	  rm_contact = "modi4.ncsa.uiuc.edu:4002/jobmanager-lsf:/C=US/O=Globus/O=The University of Illinois Urbana-Champaign/OU=The National Center for Supercomputing Applications/CN=modi4.ncsa.uiuc.edu";
	  queue = "(queue=globus_htb)";
    } else {
	  printf("Gram Contact error: Unknown Host: %s",argv[1]);
	  return(2);
    }

    if( SAME(argv[2],"test-ls") ) {    
	  sprintf(specification,"&(executable=/bin/ls)( directory = $(HOME) )( stdout = $(HOME)/gram-ls.stdout )( stderr = $(HOME)/gram-ls.stderr )" );
	  
	} else if ( SAME(argv[2],"run") ) {	
//	  sprintf(specification,"&(executable = SME )(directory = /scratch-res4/maxwell/bin/ )(stdout = ~/SME.run.stdout)(stdout = ~/SME.run.stderr)(arguments = -p -java %s run %s )",argv[3],argv[5]);
	  sprintf(specification,"&(executable = SME )(directory = /scratch-res4/maxwell/bin/ )( stdout = $(HOME)/SME-run.stdout )( stderr = $(HOME)/SME-run.stderr )(environment = (NPROCS 4))(environment = (MPI_TYPE_MAX 2048))%s(arguments = -p -java %s run %s )",queue,argv[3],argv[5]);
//	  specification += "(directory = SME )";
//	   &(executable = a.out )
//		(directory  = /home/nobody )
//      (environment = (NPROCS 4))
//      (queue=globus_htb)

	} else if ( SAME(argv[2],"project") ) {	
	  sprintf(specification,"&(executable = SME )(directory = /scratch-res4/maxwell/bin/ )(arguments = project %s )",argv[4]);
	  
    } else {
	  printf("Gram Specification error: Unknown Task: %s",argv[2]);
	  return(3);
    }

    job_state_mask = GLOBUS_GRAM_CLIENT_JOB_STATE_ALL;

    /* Initialize the monitor function to look for callbacks.  It 
     * initializes the locking mechanism, and then the condition 
     * variable
     */

    globus_mutex_init(&Monitor.mutex, (globus_mutexattr_t *) NULL);
    globus_cond_init(&Monitor.cond, (globus_condattr_t *) NULL);

    /* entering the monitor and clearing the flag. Locking the 
     * Monitor to prevent anything else from changing the value of
     * Monitor.done
     */
    globus_mutex_lock(&Monitor.mutex);

    /* Change the value of Monitor.done to false, initializing it */	
    Monitor.done = GLOBUS_FALSE;

    /* Releasing the lock on the monitor, letting anything else access it */
    globus_mutex_unlock(&Monitor.mutex);

    /* Setting up the communications port for returning the callback. 
     * You pass it the callback function.  The callback_contact is the
     * callback identifier returned by the function
     */

    globus_gram_client_callback_allow(callback_func,
				      (void *) &Monitor,
				      &callback_contact);

    printf("\n  TEST: submitting to resource manager...\n");
    printf("\n\tremote contact = %s",rm_contact);
    printf("\n\trsl spec = %s\n\n",specification);

    /* Send the GRAM request.  The rm_contact, specification, and
     * job_state_mask were retrieved earlier from the command line    
     * The callback_contact was just returned by 
     * globus_gram_client_callback_allow.  The job_request is returned by
     * this function
     */

    rc = globus_gram_client_job_request(rm_contact,
					specification,
					job_state_mask,
					callback_contact,
					&job_contact);

    if (rc != 0) /* if there is an error */
    {
        printf("TEST: gram error: %d - %s\n", 
                rc, 
                /* translate the error into english */
                globus_gram_client_error_string(rc));
        return(1);
    }

    /* Wait until there is a callback saying there was a termination, either
     * successful or failed.  We lock the Monitor again so as to ensure that
     * no one else tampers with it. Then we wait until the condition is
     * signaled by the callback_function. When it is signaled, and
     * Monitor.done is set to GRAM_TRUE - (these two things always happen
     * in conjunction in our callback_func) Then we unlock the monitor and
     * continue the program.
     */

    globus_mutex_lock(&Monitor.mutex);
    while (!Monitor.done)
    {
       /* Within the cond_wait function, it unlocks the monitor,
        * allowing the callback_func to take the lock. When it gets a
        * cond_signal, it re-locks the monitor, and returns to this
        * program.  But DO NOT unlock the monitor yourself- use the
        * globus_gram_cond_wait function, as it insures safe
        * unlocking.
        */
        globus_cond_wait(&Monitor.cond, &Monitor.mutex);
    } /* endwhile */

    globus_mutex_unlock(&Monitor.mutex);

    /* Remove Monitor.  Given that we are done with our monitor, (it has
     * already held the program until the job completed) we can now dispose
     * of it. We destroy both the mutex and the condition.  This frees up any
     * space it may have occupied.
     */

    globus_mutex_destroy(&Monitor.mutex);
    globus_cond_destroy(&Monitor.cond);

    /* Free up the resources of the job_contact, as the job is over, and
     * the contact is now useless.
     */
    globus_gram_client_job_contact_free(job_contact);

    /* Deactivate GRAM */
    globus_module_deactivate(GLOBUS_GRAM_CLIENT_MODULE);

    printf("\tTEST completed\n");

    return 0;
}

/******************************************************************
 * This is the callback function, as per the definition. We can write
 * whatever we want into the function, but remember that the 
 * cond_signal must be triggered and Monitor.done must also be set  to
 * true to exit the waiting loop in the main code. The function is called
 * from the job manager, which provides values for state and errorcode
 */

static void
callback_func(void * user_callback_arg,
              char * job_contact,
              int state,
              int errorcode)
{
    my_monitor_t * Monitor = (my_monitor_t *) user_callback_arg;

    switch(state)
    {
    case GLOBUS_GRAM_CLIENT_JOB_STATE_PENDING:
	printf("\tTEST: Got GLOBUS_GRAM_CLIENT_JOB_STATE_PENDING"
	       " from job manager\n");
	break; 

    case GLOBUS_GRAM_CLIENT_JOB_STATE_ACTIVE:
	printf("\tTEST: Got GLOBUS_GRAM_CLIENT_JOB_STATE_ACTIVE"
	       " from job manager\n");
	break; 
   
    case GLOBUS_GRAM_CLIENT_JOB_STATE_FAILED:
	printf("\tTEST: Got GLOBUS_GRAM_CLIENT_JOB_STATE_FAILED"
	       " from job manager\n");
        globus_mutex_lock(&Monitor->mutex);
        Monitor->done = GLOBUS_TRUE;
        globus_cond_signal(&Monitor->cond);
        globus_mutex_unlock(&Monitor->mutex);
	break; 

    case GLOBUS_GRAM_CLIENT_JOB_STATE_DONE:
	printf("\tTEST: Got GLOBUS_GRAM_CLIENT_JOB_STATE_DONE"
	       " from job manager\n");
        globus_mutex_lock(&Monitor->mutex);
        Monitor->done = GLOBUS_TRUE;
        globus_cond_signal(&Monitor->cond);
        globus_mutex_unlock(&Monitor->mutex);
	break; 
    }
}
