
package miiee.TGraph;

import javax.swing.*;
import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import java.util.Vector;
import miiee.util.*;

class GraphCanvas extends JComponent implements MouseListener, MouseMotionListener
{
    protected Color _Ccolor[] = {
            Color.red, Color.blue, Color.green, Color.pink, Color.orange, Color.yellow, Color.magenta, Color.cyan
    };
    static final boolean DEBUG = false;
    static final boolean MOREDEBUG = false;
    static final int _precision = 4;
    protected int left;
    protected int top;
    protected int right;
    protected int bot;
    protected int strHeight;
    protected int strWidth;
    protected FontMetrics fm;
    public boolean abbreviate = false;
    protected Color gridColour;
    protected Color gridHalfColour;
    protected Color gridQtrColour;
    protected int xOrigMin = -Integer.MAX_VALUE;
    protected int xOrigMax = -Integer.MAX_VALUE;
    protected int xMin = -9999;
    protected int xOffset = 0;
    protected int xMax;
    protected int xRange;
    protected float yOrigMin = 0f;
    protected float yOrigMax = 1f;
    protected float yMin;
    protected float yMax;
    protected float yRange;
    protected float yScale;
    protected float xScale;
    protected float xInc = -1.0f;
    protected float yScaleRatio;
    protected float xScaleRatio;
    protected int zoomJumpX;
    protected int zoomJumpY;
    protected int zX1;
    protected int zX2;
    protected int lzX;
    protected int zY1;
    protected int zY2;
    protected int lzY;
    public boolean zooming;
    protected boolean initialized;
    protected Color graphColour[];
    protected int numGraphs;
    protected int numComponents;
    String param[];
    String label[];
    String scale[];
    int flags[];
    static final int DELTA = 1;
    public String brokenMsg;
    protected Vector _data[];
    protected int time_units;
    public static final int SECONDS = 1;
    public static final int DAYS = 2;
    public static final int MONTHS = 3;
    public static final int YEARS = 4;
    String lastXTick;
    int lastXTickPos;
	private BufferedImage _offImg;
	private Color _background = Color.white;

    public GraphCanvas(int num)
    {
        super();
//        _Ccolor = {
//            Color.red, Color.blue, Color.green, Color.pink, Color.orange, Color.yellow, Color.magenta, Color.cyan
//        };
        zX1 = -1;
        zY1 = -1;
        lastXTick = null;
        fullReset(num);
        _data = new Vector[num];
    }

    public void init()
    {
        if(!initialized)
        {
            addMouseListener(this);
            addMouseMotionListener(this);
            initialized = true;
        }
    }
  
    public void interpolateDataValues(int component_index, int x0, int x1 ) {
        if(_data[component_index] == null) return;        
		if( x1 <= (x0+1) ) return ;
		Vector v = _data[component_index];
		float v0 = ((Float)v.get(x0)).floatValue();
		float v1 = ((Float)v.get(x1)).floatValue();
		int nelem = x1-x0;
		float s = (v1-v0)/nelem;
        for( int i = 1; i<nelem; i++ ) {
            int index = i + x0;
            Float val = new Float( v0 + i*s );
			if( index >= v.size() ) { v.setSize(index+16); } 
			v.setElementAt( val, index );
        }
        repaint();
    }

    public void setDataValues(int component_index, int x0, int npoints, float data[]) {
        if(_data[component_index] == null) return;  
        String outs =   "Set data values: x0 = " + x0 + ", npoints = " + npoints +", data = ";
		Vector v = _data[component_index];
        for( int i = 0; i < npoints; i++ ) {
            Float val = new Float(data[i]);
            int index = i + x0;
			if( index >= 0 ) {
			  if( index >= v.size() ) { 
				for( int j = v.size(); j <= index; j++ ) { v.add(null); } 
			  } 
			  v.setElementAt( val, index );
			  outs += val + ", ";
			}
        }
		SimIO.print(outs);
        repaint();
    }

    public void setDataValues( int component_index, DataSet d, int array_index0, int array_index1, int array_index2 ) {
	  init();
	  int start=0, dsize = d.size();
	  float[] data = new float[dsize];
	  for( int i= 0; i <= dsize; i++ ) { 
		DataEntry de = d.getEntry(i);
		if( de != null ) { 
		  float value = de.getValue(array_index0, array_index1, array_index2);
		  float dt = de.dt(), time = de.time();
		  if( dt == 0f ) { dt = 1f; }
		  int xAbsolute = (int) (time/dt);
		  SimIO.print("Adding data entry: time = " + time + ",  dt = " + dt + ",  index0 = " + array_index0 + ", val = " +  value );
		  if( xOrigMin == -Integer.MAX_VALUE ) { 
			xInc = dt;
			xOffset = xAbsolute;
			xOrigMin = xMin = 0;
			yOrigMin = yMin = value;
			yOrigMax = yMax = value;
		  }
		  else if( xInc != dt ) { SimIO.print("Graph viewer can't handle variable timesteps: " + xInc + " vs " + dt ); return; }
		  int xRel = xAbsolute - xOffset;
		  if( i == 0 ) { start = xRel; }
		  xOrigMax = xMax = xRel;
		  if( yOrigMin > value ) {  yMin = yOrigMin = value; }
		  if( yOrigMax < value ) {  yMax = yOrigMax = value; }
		  data[i] = value;
		}
	  }
	  setDataValues( component_index, start, dsize, data );
	}

   public void setDataValues( int component_index, DataSet d ) {
	  init();
	  for( int i= 0; i <= d.size(); i++ ) { 
		DataEntry de = d.getEntry(i);
		if( de != null ) { 
		  SimIO.print("Adding data entry: time = " + de.time() + ",  dt = " + de.dt() );
		  int start = (int) (de.time()/de.dt());
		  if( xInc < 0.0f ) {
			xInc = de.dt();
			xOrigMin = xMin = start;
			yOrigMin = yMin = de.min();
			yOrigMax = yMax = de.max();
		  }
		  else if( xInc != de.dt() ) { SimIO.print("Graph viewer can't handle variable timesteps: " + xInc + " vs " + de.dt() ); return; }
		  if( start >= xOrigMax ) {
			float[] data = de.getValues();
			xOrigMax = xMax = start + de.dataSize() - 1;
			if( yOrigMin > de.min() ) {  yMin = yOrigMin = de.min(); }
			if( yOrigMax < de.max() ) {  yMax = yOrigMax = de.max(); }
			setDataValues( component_index, start, de.dataSize(), data );
		  }
		}
	  }
	}

    public void setDataValues( int component_index, DataSet d, int row, int col  ) {
	  init();
	  float[] data = new float[1];
	  for( int i= 0; i <= d.size(); i++ ) { 
		DataEntry de = d.getEntry(i);
		if( de != null ) { 
		  SimIO.print("Adding data entry: time = " + de.time() + ",  dt = " + de.dt() );
		  int time_index = (int) (de.time()/de.dt());
		  data[0] =  d.getValue( row, col, de );
		  if( xInc < 0.0f ) {
			xInc = de.dt();
			xOrigMin = xMin = time_index;
			yOrigMin = yMin = data[0];
			yOrigMax = yMax = data[0];
		  }
		  else if( xInc != de.dt() ) { SimIO.print("Graph viewer can't handle variable timesteps: " + xInc + " vs " + de.dt() ); return; }
		  int old_time_index = xOrigMax;
		  xOrigMax = xMax = time_index;
		  if( yOrigMin > data[0] ) {  yMin = yOrigMin = data[0]; }
		  if( yOrigMax < data[0] ) {  yMax = yOrigMax = data[0]; }
		  setDataValues( component_index, time_index, 1, data );
		  if( old_time_index >= 0 ) {
			interpolateDataValues( component_index, old_time_index, time_index );
		  }
		}
	  }
	}

	  
    public void setDimensions(int t0, int tf, float dt, float ymin, float ymax, int units)
    {
        init();
        xOrigMin = xMin = t0;
        xOrigMax = xMax = tf;
        yOrigMin = yMin = ymin;
        yOrigMax = yMax = ymax;
        xInc = dt;
        time_units = units;
        repaint();
    }

    public void setComponent(int component_index, String color, String name) {
        if(component_index <= numGraphs) {
            int dataSize = xOrigMax - xOrigMin;
            _data[component_index] = new Vector();  
            label[component_index] = name;
            if(color == "red")
                graphColour[component_index] = Color.red;
            else
            if(color == "blue")
                graphColour[component_index] = Color.blue;
            else
            if(color == "cyan")
                graphColour[component_index] = Color.cyan;
            else
            if(color == "green")
                graphColour[component_index] = Color.green;
            else
            if(color == "pink")
                graphColour[component_index] = Color.pink;
            else
            if(color == "orange")
                graphColour[component_index] = Color.orange;
            else
            if(color == "yellow")
                graphColour[component_index] = Color.yellow;
            else
            if(color == "magenta")
                graphColour[component_index] = Color.magenta;
            if(component_index >= numComponents)
                numComponents = component_index + 1;
        }
    }

    public void fullReset(int num)
    {
        numGraphs = num;
        label = new String[numGraphs];
        param = new String[numGraphs];
        scale = new String[numGraphs];
        flags = new int[numGraphs];
        graphColour = new Color[numGraphs];
        for(int i = 0; i < num; i++)
        {
            label[i] = "(unused)";
            param[i] = Integer.toString(i);
            scale[i] = "1";
            flags[i] = 0;
            graphColour[i] = _Ccolor[i];
        }
		xOrigMin = -Integer.MAX_VALUE;
		xOrigMax = -Integer.MAX_VALUE;
		yOrigMin = 0f;
		yOrigMax = 1f;
        reset();
    }

    public void renderImage(Graphics2D g)
    {
		if(DEBUG) SimIO.print( "Rendering Image" );
        g.setColor(Color.black);
        fm = getFontMetrics(getFont());
        if(fm == null)
        {
            strWidth = 30;
            strHeight = 20;
        }
        else
        {
            Float F = new Float(yMax / 3D);
            String s = F.toString();
            strHeight = fm.getHeight();
            strWidth = fm.stringWidth(s);
        }
        right = getSize().width - 10;
        bot = getSize().height - 30;
        top = 10 + strHeight / 3;
        left = 15 + strWidth;
        g.drawLine(left, top, left, bot);
        g.drawLine(left, bot, right, bot);
        g.drawLine(left, top, right, top);
        g.drawLine(right, top, right, bot);
        Color bg = getBackground();
        if(bg == null)
            bg = Color.gray;
        gridColour = bg.darker();
        gridHalfColour = new Color((gridColour.getRed() + 2 * bg.getRed()) / 3, (gridColour.getGreen() + 2 * bg.getGreen()) / 3, (gridColour.getBlue() + 2 * bg.getBlue()) / 3);
        gridQtrColour = new Color((gridHalfColour.getRed() + 2 * bg.getRed()) / 3, (gridHalfColour.getGreen() + 2 * bg.getGreen()) / 3, (gridHalfColour.getBlue() + 2 * bg.getBlue()) / 3);
		if( yMax == yMin ) {
			yMax = yMax + 0.5f; 
			yMin = yMin - 0.5f;
		}
        yRange = yMax - yMin;
        yScaleRatio = (bot - top) / yRange;
        yScale = determineYTickInterval(yRange);
        if(yScale == 0.0F) {
			SimIO.print( "Uncommensible Y scale: ymax = " + yMax + " ymin = " + yMin );
			return;
		}
        xRange = xMax - xMin;
        xScaleRatio = (right - left) / xRange;
        xScale = determineXTickInterval(xRange);
        zoomJumpX = (right - left) / 10;
        zoomJumpY = (bot - top) / 10;
        try
        {
			if(DEBUG) SimIO.print( "Generating x tick marks ( " + Integer.toString(xMin) + ", "  + Integer.toString(xMax) + ", "  + Float.toString(xScale) + ") "  );
            if(!Float.isInfinite(xScaleRatio)) {
                for(float x = xMin; x <= xMax; x += xScale) {
                    int pix = xPlotToPixel(x);
                    float value = ( x + xOffset) * xInc;
                    xTickMark(g, pix, value, 0);
                    if(DEBUG) System.out.print(".");
                }

                for(float x = xMin + xScale / 2.0F; x <= xMax; x += xScale) {
                    int pix = xPlotToPixel(x);
                    float value = ( x + xOffset) * xInc;
                    xTickMark(g, pix, value, 1);
                    if(DEBUG) System.out.print(".");
                }

            }
        }
        catch(Exception ex)
        {
            SimIO.print("X tick mark exception " + ex);
            ex.printStackTrace();
        }
		SimIO.print( "Generating y tick marks ( " + Float.toString(yMin) + ", "  + Float.toString(yMax) + ", "  + Float.toString(yScale) + ") "  );

		if( yScale < 0 ) { SimIO.print( " error: Illegal yScale: " + Float.toString(yScale) ); return; }
		
        for(float y = yMin; y <= yMax; y += yScale) {
            yTickMark(g, yPlotToPixel(y), y, false);
			if(DEBUG) System.out.print(".");
		}
		
        for(float y = yMin + yScale / 2.0F; y <= yMax; y += yScale) {
            yTickMark(g, yPlotToPixel(y), y, true);
			if(DEBUG) System.out.print(".");
		}
		
	   if(DEBUG) SimIO.print( "Generating center line ( " + Float.toString(yMin) + ", "  + Float.toString(yMax) + ") "  );
       if(yMin != 0.0F && yMax != 0.0F)
        {
            g.setColor(Color.black);
            int zeroLine = yPlotToPixel(0.0F);
            if(zeroLine > top && zeroLine < bot)
                g.drawLine(left, zeroLine, right, zeroLine);
        }
        
		if(DEBUG) SimIO.print( "Generating graphs ( " + Integer.toString(numGraphs)  + ") "  );
	   
        for(int i = 0; i < numGraphs; i++)
            drawGraph(g, i);

    }

    private void drawGraph(Graphics2D g, int which)
    {
        int lastX = -1;
        int lastY = -1;
        boolean limitsChanged = false;
        float lastYDatum = 0.0F;
        if(label[which].equals("(unused)"))
            return;
        boolean graphDeltas;
        try
        {
            if((flags[which] & 0x1) != 0)
                graphDeltas = true;
            else
                graphDeltas = false;
        }
        catch(NullPointerException ex)
        {
            return;
        }
        int myScale;
        try
        {
            myScale = Integer.parseInt(scale[which]);
        }
        catch(NumberFormatException ex)
        {
            myScale = 1;
        }
        g.setColor(graphColour[which]);
        if(graphDeltas)
        {
            Float f;
            try
            {
                f = (Float)_data[which].elementAt(xMin - 1);
            }
            catch(Exception ex)
            {
                try
                {
                    f = (Float)_data[which].elementAt(xMin);
                }
                catch(Exception ex1)
                {
                    try
                    {
                        f = (Float)_data[which].elementAt(1);
                    }
                    catch(Exception ex2)
                    {
                        return;
                    }
                }
            }
            lastYDatum = f.floatValue();
        }
        for(int x = xMin; x <= xMax; x++)
        {
            float y;
            try
            {
                Float F = (Float)_data[which].elementAt(x);
                float f = F.floatValue();
                if(graphDeltas)
                {
                    float tmp = f;
                    f -= lastYDatum;
                    lastYDatum = tmp;
                }
                y = f * myScale;
            }
            catch(Exception ex)
            {
                y = yOrigMin;
            }
            if(y < yOrigMin)
            {
                float py = prettifyYRange(y);
                if(yMin == yOrigMin)
                    yMin = py;
                yOrigMin = py;
                limitsChanged = true;
            }
            if(y > yOrigMax)
            {
                float py = prettifyYRange(y);
                if(yMax == yOrigMax)
                    yMax = py;
                yOrigMax = py;
                limitsChanged = true;
            }
            int plotX = xPlotToPixel(x);
            int plotY = yPlotToPixel(y);
            if(plotY < top)
                plotY = top;
            if(plotY > bot)
                plotY = bot;
            if(lastX == -1)
            {
                lastX = plotX;
                lastY = plotY;
            }
            g.drawLine(lastX, lastY, plotX, plotY);
            lastX = plotX;
            lastY = plotY;
        }

        String s = "[" + label[which] + "]";
        int h = fm.getHeight() + 1;
        int w = fm.stringWidth(s);
        g.drawString(s, right - w, top + h * which);
        if(limitsChanged)  repaint();
    }

    float determineXTickInterval(float xRange)
    {
        return xRange / Math.round(xRange / (60F / xScaleRatio));
    }

    float determineYTickInterval(float yRange)
    {
        int i;
        for(i = Math.round(yRange / (60F / yScaleRatio)); i > Math.round(yRange / (((bot - top) / 5) / yScaleRatio)) && i > 0; i--)
            if(yRange % i == 0.0F)
                break;

        return yRange / i;
    }

    int yPlotToPixel(float plotY)
    {
        return (int)(bot - (plotY - yMin) * yScaleRatio);
    }

    int xPlotToPixel(float plotX)
    {
        return (int)(left + (plotX - xMin) * xScaleRatio);
    }

    float prettifyYRange(float y)
    {
        float newy = 1.0F;
        boolean negative;
        if(y < 0.0F)
        {
            negative = true;
            y = -y;
        }
        else
        {
            negative = false;
        }
        do
        {
            while(newy < y) 
            {
                if(newy * 10F > y)
                    return (negative ? -1 : 1) * (newy * 10F);
                newy *= 10F;
            }

            if(newy / 10F < y)
                return (negative ? -1 : 1) * newy;
            newy /= 10F;
        }
        while(true);
    }

    void yTickMark(Graphics g, int y, float val, boolean isSubTick)
    {
        String s;
        if(abbreviate)
        {
            if(val > 1000000F)
            {
                FormattedFloat f = new FormattedFloat(Math.round(val / 100000D) / 10D);
                s = f.toJavaFormatString(4) + "M";
            }
            else
            if(val > 1000F)
            {
                FormattedFloat f = new FormattedFloat(Math.round(val / 100D) / 10D);
                s = f.toJavaFormatString(4) + "K";
            }
            else
            {
                FormattedFloat i = new FormattedFloat(val);
                s = i.toJavaFormatString(4);
            }
        }
        else
        {
            FormattedFloat i = new FormattedFloat(val);
            s = i.toJavaFormatString(4);
        }
        if(y != top && y != bot)
        {
            if(isSubTick)
                g.setColor(gridHalfColour);
            else
                g.setColor(gridColour);
            g.drawLine(left + 1, y, right - 1, y);
        }
        g.setColor(Color.black);
        if(!isSubTick)
        {
            g.drawLine(left - 5, y, left, y);
            g.drawString(s, 5, y + strHeight / 2);
        }
    }

    public void paint(Graphics g) {
        
        Graphics2D g2 = createGraphics2D(g);
        if( g2 == null ) return;
        
		renderImage(g2);
		g2.dispose();
		
        if ( _offImg != null && isShowing() )  {
            g.drawImage( _offImg, 0, 0, this );
        }
    }

    public Graphics2D createGraphics2D(Graphics g) {
        Graphics2D g2 = null;
        int width = getSize().width; 
        int height = getSize().height; 
        if ( width <= 0 || height <= 0) return null;

        if ( _offImg == null || _offImg.getWidth() != width || _offImg.getHeight() != height ) {
            _offImg = (BufferedImage) createImage(width, height);
        } 
        if ( _offImg != null ) {
            g2 = _offImg.createGraphics();
//			  g2.setRenderingHints(Graphics2D.ANTIALIASING, Graphics2D.ANTIALIAS_ON);
//			  g2.setRenderingHints(Graphics2D.RENDERING, Graphics2D.RENDER_QUALITY); 
//			g2.setRenderingHints(Graphics2D.RENDERING, Graphics2D.RENDER_SPEED ); 
			g2.setBackground( _background );
			g2.clearRect(0, 0, width, height);
        }
        return g2;
    }

    void clearLastXTick()
    {
        lastXTick = "";
        lastXTickPos = 0;
    }

    void xTickMark(Graphics g, int x, float value, int subTickLevel)  
    {
        FormattedFloat f = new FormattedFloat(value);
        String s = f.toJavaFormatString(4);
//        String s = f.toString();
        if(lastXTickPos > 0 && lastXTickPos < x + 10 && lastXTickPos > x - 10 || lastXTick != null && lastXTick.equals(s))
            return;
        lastXTick = s;
        lastXTickPos = x;
        if(fm == null)
            strWidth = 30;
        else
            strWidth = fm.stringWidth(s);
        if(x != left && x != right)
        {
            if(subTickLevel > 0)
                g.setColor(gridHalfColour);
            else
                g.setColor(gridColour);
            g.drawLine(x, top + 1, x, bot - 1);
        }
        if(subTickLevel > 0)
            g.setColor(gridColour);
        else
            g.setColor(Color.black);
        g.drawLine(x, bot, x, bot + 5);
        if(subTickLevel == 0)
            g.drawString(s, x - strWidth / 2, bot + 20);
        g.setColor(Color.black);
    }

    public synchronized void Break(String msg)
    {
        brokenMsg = msg;
    }

    protected void xorZoomRect(int x1, int y1, int x2, int y2)
    {
        Graphics g = getGraphics();
        g.setXORMode(getBackground());
        int x = x1;
        int xw = x2 - x1;
        if(x2 < x1)
        {
            x = x2;
            xw = x1 - x2;
        }
        int y = y1;
        int yw = y2 - y1;
        if(y2 < y1)
        {
            y = y2;
            yw = y1 - y2;
        }
        g.drawRect(x, y, xw, yw);
        g.setColor(getForeground());
        g.setPaintMode();
    }

    public void zoomIn()
    {
        if(zX1 == -1 || zY1 == -1)
        {
            zX1 = left + zoomJumpX;
            zX2 = right - zoomJumpX;
            zY1 = top + zoomJumpY;
            zY2 = bot - zoomJumpY;
        }
        if(zX2 < zX1)
        {
            int tmp = zX2;
            zX2 = zX1;
            zX1 = tmp;
        }
        if(zY2 < zY1)
        {
            int tmp = zY2;
            zY2 = zY1;
            zY1 = tmp;
        }
        xMax = (int)((zX2 - left) / xScaleRatio + xMin);
        xMin = (int)((zX1 - left) / xScaleRatio + xMin);
        yMax = (bot - zY1) / yScaleRatio + yMin;
        yMin = (bot - zY2) / yScaleRatio + yMin;
        lzX = zX1 = lzY = zY1 = -1;
        zooming = false;
        repaint();
    }

    public void zoomOut()
    {
        xMin = (int)Math.max(0.0F, xMin - zoomJumpX / xScaleRatio);
        xMax = xMax + (int)(zoomJumpX / xScaleRatio);
        if(xMin == xMax)
            if(xMin > 0)
                xMin--;
            else
                xMax++;
        yMin = Math.max(0.0F, yMin - zoomJumpY / yScaleRatio);
        yMax = yMax + zoomJumpY / yScaleRatio;
        zooming = false;
        repaint();
    }

    public void reset() {
		if( xOrigMin == xOrigMax ) {
		  xMin = 0;
		  xMax = 1;
		} else {
		  xMin = xOrigMin;
		  xMax = xOrigMax;
		}
        yMin = yOrigMin;
        yMax = yOrigMax;
		repaint();
    }

    public void mouseDragged(MouseEvent e)
    {
        int x = e.getX();
        int y = e.getY();
        xorZoomRect(zX1, zY1, lzX, lzY);
        zooming = true;
        x = Math.max(x, left);
        x = Math.min(x, right);
        y = Math.max(y, top);
        y = Math.min(y, bot);
        xorZoomRect(zX1, zY1, x, y);
        lzX = x;
        lzY = y;
    }

    public void mouseMoved(MouseEvent mouseevent)
    {
    }

    public void mouseClicked(MouseEvent mouseevent)
    {
    }

    public void mousePressed(MouseEvent e)
    {
        int x = e.getX();
        int y = e.getY();
        if(zX1 != -1 && zY1 != -1)
            xorZoomRect(zX1, zY1, zX2, zY2);
        zX2 = -1;
        zY2 = -1;
        zooming = false;
        x = Math.max(x, left);
        x = Math.min(x, right);
        y = Math.max(y, top);
        y = Math.min(y, bot);
        lzX = zX1 = x;
        lzY = zY1 = y;
    }

    public void mouseReleased(MouseEvent e)
    {
        int x = e.getX();
        int y = e.getY();
        Graphics g = getGraphics();
        if(x == zX1 || y == zY1)
        {
            zX1 = zX2 = -1;
            zY1 = zY2 = -1;
        }
        else
        {
            x = Math.max(x, left);
            x = Math.min(x, right);
            y = Math.max(y, top);
            y = Math.min(y, bot);
            zX2 = x;
            zY2 = y;
        }
    }

    public void mouseEntered(MouseEvent mouseevent)
    {
    }

    public void mouseExited(MouseEvent mouseevent)
    {
    }

    
}
