package miiee.client;

import java.net.*;
import javax.swing.*;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.util.StringTokenizer;
import java.util.Vector;
import javax.swing.JOptionPane;
import miiee.util.*;

class MessageList extends StringVector  {
	public MessageList( int size, int incr ) { super(size,incr); }
	protected String getPreSeparator( int i ) { return "\n"; }
}

//---- client ---------------------------------------------------------

public abstract class SNPClient extends Object {  

  public final static int  SNP_OK =(0);
  public final static int  SNP_OK_WEI =(1);
  public final static int  SNP_ERROR =(-1);

  protected boolean gLoggedIn = false;
  protected boolean gReadingStartup = false;
  protected boolean gDoneApplication = false;
  protected int gNetworkTimeout = 30;
  protected String _host;
  protected int _port;
  protected String _model;
  protected String _project;
  protected float _current_time = 0f;
  protected float _current_timestep = 1f;
  
  protected TelnetStream _stream = new TelnetStream();
  protected SNPCommand _command = new SNPCommand();
  protected Response _rp = new Response(); 
  protected MessageReader _msg_reader = new MessageReader("Messages");

	public SNPClient( String host, int port ) { 
	  _host = host; 
	  _port = port;
	}

	public String getLocalHostName() { return _stream.getLocalHostName(); }
	public String getRemoteHostName() { return _stream.getRemoteHostName(); }
	public MessageReader getMessageReader() { return _msg_reader; }
	
	public void  open( ) throws UnknownHostException, SNPException {
	  _stream.open(_host,_port);
		
	  Response r = new Response();
	  /* Read the startup message from the server. */
	  gReadingStartup = true;

	  try {
	  
		_stream.getResponse(_rp);
		
	  } catch ( SNPException err ) {
	/* They probably hung up on us right away.  That's too bad,
	 * but we can tell the caller that they can call back later
	 * and try again.
	 */
		  gReadingStartup = false;
		  try {
			_stream.close();
		  } catch ( IOException ex1 ) {;}
		  throw new SNPException( SNPException.HANGUP, err.getException() );
	  }
	  gReadingStartup = false;
	}

  public void close_connection()  {
	try {
	  _stream.close();
	} catch ( IOException ex1 ) {;}
  }
 
  public  int send_command() throws SNPException {
  
	try { 
	  _stream.writeLine( _command.getBytes() );
	} catch ( IOException err ) {
	  throw new SNPException( SNPException.WRITE, err );
	} finally {
	  _command.clear();
	}
	return SNP_OK;
  }

  int driver_command() throws SNPException {  
	
	_command.setCode("COMM");
	send_command();
	return SNP_OK;
  }
  
  int xml_command() throws SNPException {  
	
	_command.setCode("CNFG");
	send_command();
	return SNP_OK;
  }

  public int rcmd () throws SNPException {

	send_command();	
	return _stream.getResponse( _rp );  // read response from server
  } 
  
  abstract public void refresh() ;
  
  public int set_timeout (int timeout) throws SNPException {
	int oldt; 
	oldt = gNetworkTimeout;
	gNetworkTimeout = timeout;
	return oldt;
  }

  public  String get_line_response(Response rp) throws SNPException 
  {
	if ( _rp.size() == 0 ) return null;
	return (String) _rp.get(0);
  }

  public int login( String user, String password )  throws SNPException {
  
	_command.setCode("USER");
	_command.addArg(user);
	int code;
	
	if ( (code = rcmd()) != 3 ) { 
	  throw new SNPException( _rp.code, "Login USER error: " + _stream.getCurrentTelnetString() + ", user: " + user + ", pw: " + password );
	}

	_command.setCode("PASS");
	_command.addArg(password);

	if ( (code = rcmd()) != 2 ) {
	  throw new SNPException( _rp.code, "Login PASS error: " + _stream.getCurrentTelnetString() + ", user: " + user + ", pw: " + password  );
	}

	/* update stats */
	gLoggedIn = true;
	return code;
  }

  public int close_model() throws SNPException  {
  
	_rp.eofOkay = true;
	_command.setCode("QUIT");
	int code = rcmd();

	/* free up connection slots */
	close_connection();
	_model = null;

	return SNP_OK;
  }

  public String get_project() { return _project; }
  public String get_model() { return _model; }
  public void set_model( String model ) { _model = model; }

  public int select_project ( String project) throws SNPException  {
	_project = 	project;
	_command.setCode("PROJ");
	_command.addArg(project);
		  int code = rcmd();
	if ( code != 2 ) { 
	  throw new SNPException( _rp.code, "Project selection error: " + _stream.getCurrentTelnetString() );
	}
	return SNP_OK;
  }

  public int set_server_timeout ( int timeout ) throws SNPException  {
	_command.setCode("TOUT");
	_command.addArg( Integer.toString(timeout) );
	int code = rcmd();
	if ( code != 2 ) { 
	  throw new SNPException( _rp.code, "Set-timeout error: " + _stream.getCurrentTelnetString() );
	}
	return SNP_OK;
  }

  public int select_model (String model) throws SNPException 
  {
	_model = model;
	_command.setCode("MODL");
	_command.addArg(model);
	int code = rcmd();
	if ( code != 2 ) { 
	  throw new SNPException( _rp.code, "Model selection error: " + _stream.getCurrentTelnetString() );
	}
	return SNP_OK;
  }

  public int select_interface (String intf ) throws SNPException 
  {
	_command.setCode("INTF");
	_command.addArg(intf);
	int code = rcmd();
	if ( code != 2 ) { 
	  throw new SNPException( _rp.code, "Interface selection error: " + _stream.getCurrentTelnetString() );
	}
	return SNP_OK;
  }

  abstract  public  Variable getVariable(  String full_name );

  public int run_model () throws SNPException 
  {
	_command.setCode("RUNM");
	int code = rcmd();
	if ( code != 2 ) { 
	  throw new SNPException( _rp.code, "Model run error: " + _stream.getCurrentTelnetString() );
	}
	return  SNP_OK;  
  }

  public int stop_model () throws SNPException  {  
	_command.setCommand( "exit" );
	return submit_command ();
 }

  public int submit_command () throws SNPException  {  
	driver_command();
	_stream.getResponse(_rp);

	return _rp.code > 300 ? SNP_OK_WEI : SNP_OK;
  } 

  public String get_result () throws SNPException  {
  
	_command.setCode("ANSW");
	int code = rcmd();
	return _rp.toUnpaddedString();
  }

  public String get_previous_result () throws SNPException {
	return _rp.toUnpaddedString();
  }
    
  public void retrieve_messages(String label ) throws SNPException {
	StringVector info = retrieve_info_messages();
//	StringVector warn = retrieve_warning_messages();
//	StringVector error = retrieve_error_messages();
	_msg_reader.addMessages(label,info);
  }
  
   public StringVector retrieve_info_messages() throws SNPException {
	int nw = get_n_info ();
	MessageList sv = null;
	SimIO.print( " retrieve " + nw + " info_messages " );
	if( nw > 0 ) {
	  sv = new MessageList(nw,10);
	  for( int i=0; i<nw; i++ ) {
		String s = get_info(i);
		sv.addArg(s);
	  }
	}
  	return sv; 
  }

   public StringVector retrieve_warning_messages() throws SNPException {
	int  nw = get_n_warnings ();
	MessageList sv = null;
	if( nw > 0 ) {
	  SimIO.print( "retrieve_info_messages: " );
	  sv = new MessageList(nw,10);
	  for( int i=0; i<nw; i++ ) {
		String s = get_warning(i);
		SimIO.print( s );
		sv.addArg(s);
	  } 
	}
  	return sv; 
  }

   public StringVector retrieve_error_messages() throws SNPException {
	int nw = get_n_errors ();
	MessageList sv = null;
	if( nw > 0 ) {
	  SimIO.print( "retrieve_info_messages: " );
	  sv = new MessageList(nw,10);
	  for( int i=0; i<nw; i++ ) {
		String s = get_error(i);
		SimIO.print( s );
		sv.addArg(s);
	  } 
	}
	return sv; 
  }
 
  public void dump_info() throws SNPException {
	int nw = get_n_info ();
	for( int i=0; i<nw; i++ ) {
	  String s = get_info(i);
	  SimIO.print(s);
	} 
  }
  public void dump_warnings() throws SNPException {
	int nw = get_n_warnings ();
	for( int i=0; i<nw; i++ ) {
	  String s = get_warning(i);
	  SimIO.print(s);
	} 
  }
  public void dump_errors() throws SNPException {
	int nw = get_n_errors ();
	for( int i=0; i<nw; i++ ) {
	  String s = get_error(i);
	  SimIO.print(s);
	} 
  }

  public int get_n_errors () throws SNPException  {
	_command.setCode("NERR");
	int rv=0, code = rcmd();

	if (  _rp.size() == 0 ) {
	  throw new SNPException(SNPException.SERVER,"Empty response from server");
	}
	String s = (String)_rp.get(0);
	try {
	  rv = Integer.parseInt( s.trim() ); 
	} catch ( java.lang.NumberFormatException err ) {
	  SimIO.print( "get_n_errors response: " + s );
	}
	return rv;  
  }

  public int get_n_warnings ()  throws SNPException  {
	_command.setCode("NWAR");
	int rv=0, code = rcmd();

	if (  _rp.size() == 0 ) {
	  throw new SNPException(SNPException.SERVER,"Empty response from server");
	}
	String s = (String)_rp.get(0);
	try {
	  rv = Integer.parseInt( s.trim() ); 
	} catch ( java.lang.NumberFormatException err ) {
	  SimIO.print( "get_n_warnings response: " + s );
	}
	return rv;  
  }

  public int get_procn ()  throws SNPException  {
	_command.setCode("PNUM");
	int rv=0, code = rcmd();

	if (  _rp.size() == 0 ) {
	  throw new SNPException(SNPException.SERVER,"Empty response from server");
	}
	String s = (String)_rp.get(0);
	try {
	  rv = Integer.parseInt( s.trim() ); 
	} catch ( java.lang.NumberFormatException err ) {
	  SimIO.print( "get_procn response: " + s );
	}
	return rv;  
  }

  public int get_n_info ()  throws SNPException  {
	_command.setCode("NINF");
	int rv=0, code = rcmd();

	if (  _rp.size() == 0 ) {
	  throw new SNPException(SNPException.SERVER,"Empty response from server");
	}
	String s = (String)_rp.get(0);
	try {
	  rv = Integer.parseInt( s.trim() ); 
	} catch ( java.lang.NumberFormatException err ) {
	  SimIO.print( "get_n_info response: " + s );
	} catch ( NullPointerException err ) {
	  return 0;
	}
	return rv;  
  }

  public  String get_error (int n) throws SNPException  {
	_command.setCode("ERRO");
	_command.addArg( n );
	int code = 0;
	
	try {
	  code = rcmd();
	} catch ( NumberFormatException err ) {
	  SimIO.show_warning( null, " Number format exception in get_error" );
	}
	if (  _rp.size() == 0 ) {
	  SimIO.show_warning( null,  "Empty response from server in get_error" );
	}
	return (String)_rp.toUnpaddedString();    
  }

  public  String get_warning (int n) throws SNPException  {
	_command.setCode("WARN");
	_command.addArg( n );
	int code = 0;
	
	try {
	  code = rcmd();
	} catch ( NumberFormatException err ) {
	  SimIO.show_warning( null, " Number format exception in get_warning" );
	}
	if (  _rp.size() == 0 ) {
	  SimIO.show_warning( null, "Empty response from server in get_warning" );
	}
	return (String)_rp.toUnpaddedString();        
  }

  public  String get_info ( int n )  throws SNPException   {
	_command.setCode("INFO");
	_command.addArg( n );
	int code = 0;
	
	code = rcmd();
	if (  _rp.size() == 0 ) {
	  SimIO.show_warning( null,  "Empty response from server in get_info" );
	  return "";
	}
	return (String)_rp.toUnpaddedString();    
  }

  private int split_list( String list,  Vector result,  String delim ) {
	if( ( result == null ) || ( list == null ) ) return 0;
	if( delim == null ) delim = " ";
	result.setSize(0);
    StringTokenizer st = new StringTokenizer(list,delim);
    int size = 0;
	while (st.hasMoreTokens()) {
	  result.add( st.nextToken() );
	  size++;
	}
	return size;
  }

  public String get_project_info()  throws SNPException  {
	_command.setCode("PINF");
	int code = rcmd();

	if (  _rp.size() == 0 ) {
	  throw new SNPException(SNPException.SERVER,"Empty response from server");
	}

	return _rp.toString();
  }

  public String get_model_info()  throws SNPException  {
	_command.setCode("MINF");
	int code = rcmd();

	if (  _rp.size() == 0 ) {
	  throw new SNPException(SNPException.SERVER,"Empty response from server");
	}
	
	return (String) _rp.get(0);
  }

  public int project_list ( Vector v ) throws SNPException  {

	_command.setCode("LISP");
	int code = rcmd();

	if (  _rp.size() == 0 ) {
	  throw new SNPException(SNPException.SERVER,"Empty response from server");
	}
	String s = (String)_rp.get(0);    
	return split_list( s, v, " " );
  }

  public int model_list ( Vector v ) throws SNPException {

	_command.setCode("LISM");
	int code = rcmd();

	if (  _rp.size() == 0 ) {
	  throw new SNPException(SNPException.SERVER,"Empty response from server");
	}
	String s = (String)_rp.get(0); 
	return split_list( s, v, " " );
  }


  /* private retrieval functions with translation of result in
	 different formats; all return SNP_OK or SNP_ERROR */

  int driver_command_clist ( Vector v ) throws SNPException  {

	driver_command ();
	_stream.getResponse( _rp );
	int rv = _rp.code > 300 ? SNP_OK_WEI : SNP_OK;
	
	String s = get_result();	
	split_list( s, v, " " );

	return rv;
  }

  int driver_command_string ( String[] s ) throws SNPException  {
	driver_command ();
	_stream.getResponse( _rp );
	int rv = _rp.code > 300 ? SNP_OK_WEI : SNP_OK;
	s[0] = get_result(); 	
	return rv;
  }

  int driver_command_int ( int[] ival ) throws SNPException  {
	driver_command ();
	_stream.getResponse( _rp );
	int rv = _rp.code > 300 ? SNP_OK_WEI : SNP_OK;

	String s = get_result(); 

	try { 	
	  ival[0] = Integer.parseInt( s.trim() );
	} catch( NumberFormatException err) {
	  warning( "NumberFormatException, Server returned: " + s );
	  ival[0] = -1;
	}
	return rv;
  }

	int driver_command_float( float[] fval ) throws SNPException { 
		driver_command ();
		_stream.getResponse( _rp );
		int rv = _rp.code > 300 ? SNP_OK_WEI : SNP_OK;
		
		String s = get_result();

		try { 	
			Float f = Float.valueOf(s.trim());	
			fval[0] = f.floatValue();
		} catch( NumberFormatException err) {
			warning( "NumberFormatException, Server returned: " + s );
			fval[0] = Float.NaN;
		}
		return rv;
	}

	int xml_command_string( String s ) throws SNPException { 
		xml_command ();
		_stream.getResponse( _rp );
		int rv = _rp.code > 300 ? SNP_OK_WEI : SNP_OK;
		s = get_result();
		return rv;
	}

  int driver_command_byte2d_rle (int id) { return 1; }
  int driver_command_int2d_rle  (int id) { return 1; }
  int driver_command_long2d_rle (int id) { return 1; }
  int driver_command_float2d_rle (int id) { return 1; }
  
  static void error( String msg ) {
	  JOptionPane.showMessageDialog( null, msg, "SNP error", JOptionPane.ERROR_MESSAGE );
  }
  static void warning(  String msg ) {
	  JOptionPane.showMessageDialog( null, msg, "SNP warning", JOptionPane.WARNING_MESSAGE );
  }
  static void info( String msg ) {
	  JOptionPane.showMessageDialog( null, msg, "SNP info", JOptionPane.INFORMATION_MESSAGE );
  }
  
   public int sme_step() throws SNPException {
	_command.addArg( "SME" );
	_command.addArg( "run" );
	_command.addArg( Float.toString( _current_time += _current_timestep ) );
	int result = submit_command();
	return result;
  }

  public int sme_run( float time ) throws SNPException {
	_command.addArg( "SME" );
	_command.addArg( "run" );
	_command.addArg( Float.toString( time ) );
	int result = submit_command();
	_current_time = time;
	return result;
  }

  public int sme_restart() throws SNPException  {
	_command.addArg( "SME" );
	_command.addArg( "restart" );
	int result = submit_command();
	_current_time = 0f;
	return result;
  }

  public float sme_current_time () {
	return _current_time;
  }

  public float sme_current_timestep () {
	return _current_timestep;
  }

  public void  sme_set_timestep (float f) {
	_current_timestep = f;
  }

  /* private varinfo functions */

  public int sme_varinfo_int ( String mod, String var, String info ) throws SNPException {  
	int[] r = new int[1]; 
	_command.addArg( mod + "." + var );
	_command.addArg("i");
	_command.addArg(info);
	driver_command_int( r );
	return r[0];
  }

  public String sme_varinfo_string ( String mod, String var, String info ) throws SNPException {
	String[] s = new String[1]; 
	_command.addArg( mod + "." + var );
	_command.addArg("i");
	_command.addArg(info);
	driver_command_string( s );
	String rv = s[0];
//	if( rv.endsWith("$@$") ) { rv = rv.substring(0,rv.length()-3); }
	return rv;
  }

 public String sme_shell_varinfo_string ( String mod, String var, String info ) throws SNPException {
	String[] s = new String[1]; 
	_command.addArg( mod + "." + var );
	_command.addArg("exec");
	_command.addArg(info);
	driver_command_string( s );
	String rv = s[0];
//	if( rv.endsWith("$@$") ) { rv = rv.substring(0,rv.length()-3); }
	return rv;
  }

  public String sme_var_config_info( String mod, String var, char level ) throws SNPException {
	String[] s = new String[1]; 
	switch ( level ) {
	  case 'a' : _command.addArg( "SME" ); break;
	  case 'm' : _command.addArg( mod ); break;
	  case 'v' : _command.addArg( mod + "." + var ); break;
	}
	_command.addArg("exec");
	String info = "CS " + level + " c ";    // level = a: model, m: module, v: variable
	_command.addArg(info);
	driver_command_string( s );
	return s[0];
  }

  public float sme_varinfo_float ( String mod, String var, String info ) throws SNPException {
	float[] f = new float[1]; 
	_command.addArg( mod + "." + var );
	_command.addArg("i");
	_command.addArg(info);
	driver_command_float( f );
	return f[0];
  }

  public String sme_xml_cmd ( String cmd ) throws SNPException {
	_command.addArg(cmd);
	String s = new String();
	xml_command_string( s ) ;
	return s;
  }


  public int sme_var_browse_data ( String mod, String var, boolean edit, boolean show_data ) throws SNPException {
	return sme_var_browse_data( mod + "." + var, edit, show_data );
  }
  
  public int sme_var_browse_data ( String name, boolean edit, boolean show_data ) throws SNPException {
	_command.addArg( name );
	_command.addArg("exec");
	_command.addArg("dv");
	if( show_data ) {
	  if( edit ) { _command.addArg("e"); }
	  else { _command.addArg("x"); }
	} else {
	  _command.addArg("f"); 
	}
	return submit_command();
  }

  public int sme_start_debugger () throws SNPException {
	if( getLocalHostName().compareTo( getRemoteHostName() ) == 0 ) {    //   assume no permission to open xterm locally
	  String[] s = new String[1]; 										//   if running on remote machine
	  _command.addArg( "SME" );
	  _command.addArg("exec");
	  _command.addArg("DB");
	  _command.addArg( getLocalHostName() + ":0.0" ); 
	  int rv = driver_command_string ( s );
	  SimIO.print(s[0]);
	  return rv;
	} else {
	  SimIO.beep();
	  return -1;
	}
  }
  
    public int sme_var_dump_data ( String mod, String var, int r0, int c0, String s[] ) throws SNPException {									
	  _command.addArg( mod + "." + var ); 	  
	  _command.addArg("exec");
	  _command.addArg( Integer.toString(r0) );
	  _command.addArg( Integer.toString(c0) );
	  int rv = driver_command_string ( s );
	  return rv;
  }

  int sme_set_scalar ( String module, String var, float value) throws SNPException {
	_command.addArg( module + "." + var );
	_command.addArg(" set ");
	_command.addArg( String.valueOf(value) );
	return submit_command();
  }

  int sme_set_spatial_at (  String module, String var, int x, int y, float value ) throws SNPException  {
	_command.addArg( module + "." + var );
	_command.addArg(" set ");
	_command.addArg(" -r " + String.valueOf(x) );
	_command.addArg(" -c " + String.valueOf(y) );
	_command.addArg( String.valueOf(value) );
	return submit_command();
  }

  int sme_set_spatial_at (  String vname, int x, int y, String value ) throws SNPException  {
	_command.addArg( vname );
	_command.addArg(" set ");
	_command.addArg(" -r " + String.valueOf(x) );
	_command.addArg(" -c " + String.valueOf(y) );
	_command.addArg( value );
	return submit_command();
  }

  /* public info functions */
  public float sme_var_min ( String module, String var) throws SNPException
  {
	return sme_varinfo_float( module, var, "min");
  }

  public float sme_var_max ( String module, String var) throws SNPException
  {
	return sme_varinfo_float(module, var, "max");
  }
  
  public float sme_model_timestep ( ) throws SNPException {
	  float[] f = new float[1]; 
	  _command.addArg( "SME" );
	  _command.addArg("dt");
	  driver_command_float( f );
	  _current_timestep = f[0];
	  return f[0];
  }

  public float sme_model_start_time ( ) throws SNPException {
	  float[] f = new float[1]; 
	  _command.addArg( "SME" );
	  _command.addArg("t0");
	  driver_command_float( f );
	  _current_time = f[0];
	  return f[0];
  }

  public int sme_var_size ( String mod, String var, int[] dims ) throws SNPException {
	_command.addArg( mod + "." + var );
	_command.addArg("i");
	_command.addArg("size");
	Vector v = new Vector(6,6);
	
	int res = driver_command_clist ( v );

	if( v.size() != 2 ) {
	  throw new SNPException( SNPException.INTERNAL, "Illegal dimensions in sme_var_size" );
	}
	
	dims[0] = Integer.parseInt( ((String)v.get(0)) );
	dims[1] = Integer.parseInt( ((String)v.get(1)) );
	
	return res;
  }

  public boolean sme_var_isspatial ( String module, String var) throws SNPException {
	return (sme_varinfo_int( module, var, "spatial") > 0);
  }

  public boolean sme_var_isscalar (String module, String var) throws SNPException {
	return (sme_varinfo_int( module, var, "scalar") > 0);
  }

  public boolean sme_var_isdynamic (String module, String var) throws SNPException {
	return (sme_varinfo_int(module, var, "dynamic") > 0);
  }

  public boolean sme_var_isconstant ( String module, String var) throws SNPException {
	return (sme_varinfo_int(module, var, "constant") > 0);
  }

  public boolean sme_var_isimport (String module, String var) throws SNPException {
	return (sme_varinfo_int(module, var, "import") > 0);
  }

  public boolean sme_var_isexport (String module, String var) throws SNPException {
	return (sme_varinfo_int(module, var, "export") > 0);
  }

  public boolean sme_var_isflux (String module, String var) throws SNPException {
	return (sme_varinfo_int(module, var, "flux") > 0);
  }

  public boolean sme_var_isstate (String module, String var) throws SNPException {
	return (sme_varinfo_int(module, var, "state") > 0);
  }

  public boolean sme_var_isignored (String module, String var) throws SNPException  {
	return (sme_varinfo_int(module, var, "ignored") > 0);
  }

  public boolean sme_var_isclamped (String module, String var) throws SNPException {
	return (sme_varinfo_int(module, var, "clamped") > 0);
  }


  public int sme_var_dependencies (String mod, String var, String action,  Vector v ) throws SNPException {
  	_command.addArg( mod + "." + var );
	_command.addArg("info");
	_command.addArg("dependencies");
	if( action != null ) { _command.addArg(action); }
	return driver_command_clist (v);
  }

  public int sme_var_actions (String mod, String var,  Vector v ) throws SNPException {
  	_command.addArg( mod + "." + var );
	_command.addArg("info");
	_command.addArg("actions");

	return driver_command_clist (v);
  }

  public String sme_var_actioninfo (String mod, String var,  String action ) throws SNPException {
 	String[] s = new String[1]; 
 	_command.addArg( mod + "." + var );
	_command.addArg("info");
	_command.addArg("actioninfo");
	_command.addArg(action);

	driver_command_string ( s );
	return s[0];
  }

  public int sme_modules (Vector v) throws SNPException {
	_command.addArg("SME");
	_command.addArg("module_list");
	return driver_command_clist( v );
  }

  public int sme_scenario_parameters (Vector v) throws SNPException {
	_command.addArg("SME");
	_command.addArg("parameter_list");
	return driver_command_clist( v );
  }

  public int sme_variables (String module, Vector v ) throws SNPException {
	_command.addArg(module);
	_command.addArg("variable_list");
	return driver_command_clist( v );  
  }
  
    public int sme_configure_object( String object, ConfigCommand cmd ) throws SNPException {	
	_command.addArg(object);
	_command.addArg("c");
	_command.addArg( cmd.toString() );
	return submit_command();
  }

  public int sme_delete_configuration( String object, ConfigCommand cmd ) throws SNPException {	
	_command.addArg(object);
	_command.addArg("exec");
	_command.addArg("k");
	_command.addArg( cmd.toString() );
	return submit_command();
  }

  public int sme_save_configuration( String name ) throws SNPException {	
	_command.addArg( "SME" );
	_command.addArg("exec");
	_command.addArg("SC");
	_command.addArg( name );
	return submit_command();
  }

  public int sme_configure_variable( String mod, String var, ConfigCommand cmd ) throws SNPException {
	System.out.println( "sme_configure_variable, mod: " + mod + ", var: " + var + ", cmd: " + cmd.toString() );
	_command.addArg( mod + "." + var );
	_command.addArg("c");
	_command.addArg( cmd.ID() );
	for(int i = 0; i < cmd.size(); i++) {
		String arg = (String) cmd.elementAt(i);
		_command.addArg( arg );
	}
	return submit_command();
  }
  
  abstract public int getViewserverPort();
  abstract public boolean initViewserver( int port );
  
  public int sme_configure_viewserver( int port ) throws SNPException {
	if( getViewserverPort() < 0 ) {
	  initViewserver(port);
	  _command.addArg( "SME" );
	  _command.addArg("exec");
	  _command.addArg("NV");
	  _command.addArg( _stream.getLocalHostName() );
	  _command.addArg( getViewserverPort() );
	  return submit_command();
	} 
	return 0;
  }
      
  public int sme_var_display_data( String mod, String var ) throws SNPException {
	sme_configure_viewserver( 17726 );
	if( mod != null ) {
	  ConfigCommand cc = new ConfigCommand("DD");
	  return sme_configure_variable(mod,var,cc);
	}
	return -1;
  }

  public int sme_open() throws SNPException  {
	_command.addArg( "SME" );
	_command.addArg( "open" );
	int rv = submit_command();
	return rv;
  }
  
}
