package miiee.dataview;  

import javax.swing.*;
import java.io.*;
import javax.swing.table.*;
import java.util.Vector;
import java.awt.image.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;
import java.applet.Applet;
import javax.help.*;
import miiee.util.*;

public class DataTable extends JPanel implements IOController, AppletKernel  {

    protected JTable       _table;
    protected DataTableModel _dataModel;
    protected AnimationViewer _animationViewer;
    protected JFrame _frame;
    protected String _name;
    
    protected AttributeValuePair _time;
    protected AttributeValuePair _layer;
    protected String _helpText;
    protected SaveFileControlPanel _saveControlPanel;
    protected DataSet _data;
    protected int _currentFrame = 0;
    protected int _currentLayer = 0;
    private int[] _compressedIndex;
    private int _offset0 = 0;
    private int _offset1 = 0;
    private int _offset2 = 0;
    private float _time_offset = 0;
    private float _dt = -1;
    protected AttributeValuePair _offValue0;
    protected AttributeValuePair _offValue1;
    protected AttributeValuePair _offValue2;
    protected Rectangle _boundsRec = new Rectangle();
    protected boolean _isEditable;
    
   /**
      * Constructs a new instance of DataTable.
      */
    public DataTable( DataSet d, boolean editable ) {
		_data = d;
		_name = d.toString();
		generateHelpText();
		_isEditable = editable;
    }

	public static DataTable getTablePanel( DataSet ds, AnimationViewer av, boolean isEditable ) {
		 DataTable dt = getTable( ds, false, isEditable, false );
		 dt._animationViewer = av;
		 dt.setTime( av.getTime() );
		 return dt;
	}

	public static DataTable getTable( DataSet ds, AnimationViewer av, boolean isEditable ) {
		 DataTable dt = getTable( ds, false, isEditable, true );
		 dt._animationViewer = av;
		 dt.setTime( av.getTime() );
		 return dt;
	}
		
	public static DataTable getTable( DataSet ds, boolean isDemo, boolean isEditable, boolean hasFrame ) {
	  DataTable d = new DataTable(ds,isEditable);
	  
	  JFrame frame = null;
	  if( hasFrame ) {
		frame = new JFrame("Table");
		frame.addWindowListener(new myWindowAdapter(d,isDemo));
		d.setFrame(frame);
	  }
	  
	  d.init();
	  ds.addEntryDataListener(d);
 
	  if( frame != null ) {
		frame.getContentPane().add(d);
		frame.setJMenuBar( d.constructMenuBar() );
		frame.setBackground(Color.lightGray);	
		frame.pack();
		frame.setVisible(true);
	  }
	  
	  return d;
	}
	
	public int offset( int dim ) {
	  switch(dim) {
		case 0: return _offset0;
		case 1: return _offset1;
		default: return 0;
	  }
	}
	
	public int getRowCount() { return _dataModel.getRowCount(); }
	public int getColumnCount() { return _dataModel.getColumnCount(); }

    public void saveAsASCII() {
	  SimulationData sd = DataSet.getCurrentSimulation();
	  String fileName = JOptionPane.showInputDialog("File name:"); 
	  String path = sd.getProjPath();
	  BufferedWriter out = null;
	  File ofile = null;
	  try {
		ofile = new File( path, fileName );
		FileOutputStream fos =  new FileOutputStream( ofile );
		out = new BufferedWriter(new OutputStreamWriter(fos));
		SimIO.print( "Saving data to file: " + ofile.getAbsolutePath() ); 
	  } catch ( Exception e0 ) {
		SimIO.show_error( null, "Error opening file stream: " + ofile.getAbsolutePath() ); 
		SimIO.print(e0.getMessage());
		return;
	  } 

	  try {
		int rows = getRowCount();
		int cols = getColumnCount();
		for ( int ir=0; ir<rows; ir++ ) {
		  out.newLine(); 
		  for ( int ic=1; ic<cols; ic++ ) {
			Object obj = getDataValue( ir, ic );
			out.write( obj.toString() + " " );
		  }
		}
	  } catch ( Exception e1 ) {
		SimIO.show_error( null, "Error writing data: ", e1 ); 
	  } 
	  try {
		out.close();
	  } catch ( Exception e2 ) {;}
	}
    	
    public synchronized void init() {   
		_dataModel = new DataTableModel(18,18,_data,this);
		_dataModel.setEditable( _isEditable ); 
		_table = new JTable(_dataModel);
		formatColumns(); 
		JScrollPane scrollpane = new JScrollPane(_table);
		scrollpane.setPreferredSize(new Dimension(600, 350));
		scrollpane.setBorder( BorderFactory.createLoweredBevelBorder() );
		
		setLayout(new BorderLayout());
		add( "Center", scrollpane );
		add( "South", constructOptionsPanel());				
    }
/*    
    public void repaint() {
	  _table.getBounds(_boundsRec);
	  _table.repaint(_boundsRec.x,_boundsRec.y,_boundsRec.width,_boundsRec.height);
	  super.repaint();
    }
    
    public void reshape(int x, int y, int w, int h) {
	  super.reshape(x, y, w, h);
	  _dataModel.fireTableDataChanged();
	  repaint();
    }
*/  
	public  DataSet getDataSet() { return _data; }
  
	public Object getDataValue( int row, int col ) { 
	  float value = Float.NaN;
	  boolean isInteger = false;
	  int adim = _data.getArrayDimension();
	  int dim = _data.dims();
	  if( (dim < 2) && (adim < 1) ) {
		int display_cols = 5;
		if( _dt < 0f ) { _dt = 1; }
		int col_index = col/2;
		if( col_index < display_cols ) {
		  int offset = (int)(_time_offset/_dt);
		  int data_index =  col_index + row*display_cols + offset;
		  int entry_index = data_index/_data.size(0);
		  DataEntry de = _data.getEntry( entry_index );
		  if( de != null ) { 
			System.out.println("  dbg("+row+","+col+")" + ", di: " + data_index + ", ei: " + 
							  entry_index + ", ds: " + _data.size(0) + ", " + _data.size(1) + ", " + de.dataSize());
			int index = (data_index % _data.size(0));
			_dt = de.dt();
			if( ( col % 2 ) == 0 ) {
			  value  = de.time() + index*de.dt(); 
			} else {
			  value  = de.getValue( index ); 
			}				  				  
		  } 
		}
	  } else if( adim > 0 ) {
		DataEntry de = null;
		if( _dt < 0f ) { de = _data.getEntry( 0 ); _dt = de.dt(); }
		int i0=col-1, i1=_offset0, i2=_offset1; 
		int time_index = row + (int)(_time_offset/_dt);
		if( time_index != 0 ) { de = _data.getEntry( time_index ); }
		if( de != null ) { 
//		  System.out.println("  dbg("+row+","+col+")" + ", di: " + data_index + ", ei: " + 
//							entry_index + ", ds: " + _data.size(0) + ", " + _data.size(1) + ", " + de.dataSize());
		  if( col  == 0 ) {
			value  = de.time(); 
		  } else if( col > _data.size( 0 ) ) {
			value = Float.NaN;
		  } else {
			value  = de.getValue(i0,i1,i2); 
		  }				  				  
		} 
	  } else {
		if( col == 0 ) {
		  return new Integer( _offset1 + row ); 
		} else {
		  DataEntry de = _data.getEntry( _currentFrame );
		  if( de != null ) { 
			int index = getDataIndex(row,col-1);
			if( index >= 0 ) { 
			  value = de.getValue( index );
			  isInteger = (de.getFormat() > 0); 
			}
		  } 
		}
	  }
	  FormattedFloat ff = new FormattedFloat(value); 
	  ff.setFixedDigits(5);
	  ff.setRounding( isInteger );
	  return ff;
   }

	public boolean setDataValue( int row, int col, float value ) { 
	  if( _data.dims() >= 2 ) {
		if( col > 0 ) {
		  DataEntry de = _data.getEntry( _currentFrame );
		  if( de != null ) { 
			int index = getDataIndex(row,col-1);
			if( index >= 0 ) { 
			   de.setValue( index, value); 
			   return true;
			}
		  } 
		}
	  }
	  return false;
   }

    public void addTableModelListener(TableModelListener l) {
	  _dataModel.addTableModelListener(l);
    }
  
   private int  getDataIndex( int row, int col ) {
	 if( ( row >= _data.size(1)) || ( col >= _data.size(0)) ) { return -1; }
	 int base_index = _currentLayer*_data.size(0)*_data.size(1) + _offset0 + _offset1*_data.size(0);
	 int index = base_index + row*_data.size(0) + col;
	 if( _data.format() == 0 ) return index;
	 if( _compressedIndex == null ) {  _compressedIndex = _data.generateCompressionMap(); }
	 int cindex = -1;
	 try {
	   cindex  = _compressedIndex[ index ];
	 } catch( ArrayIndexOutOfBoundsException err ) {;}
	 return cindex;
   }
   
  private JPanel constructOptionsPanel() {
	JPanel           borderPane = new JPanel(false);  
	JButton aButton;

	int dim = _data.dims();
	int adim = _data.getArrayDimension();
	borderPane.setLayout(new GridLayout(dim,1));
	
	if( dim > 0 ) {
	  JPanel   offsetsPanel = new JPanel(false);
	  offsetsPanel.setLayout(new FlowLayout());
	  borderPane.add(offsetsPanel);

	  if( adim > 0 ) {	  
		_offValue0 = AttributeValuePair.New( "time offset:", true, 5 );
		_offValue0.setValue("0");
		offsetsPanel.add(_offValue0);
		if( adim > 1 ) {
		  _offValue1 = AttributeValuePair.New( "array index 2:", true, 5 );
		  _offValue1.setValue("0");
		  offsetsPanel.add(_offValue1);
		}
		if( adim > 2 ) {
		  _offValue2= AttributeValuePair.New( "array index 3:", true, 5 );
		  _offValue2.setValue("0");
		  offsetsPanel.add(_offValue2);	
		}
	  } else  if( dim > 1 ) {
		_offValue0 = AttributeValuePair.New( "col offset:", true, 5 );
		_offValue0.setValue("0");
		offsetsPanel.add(_offValue0);

		_offValue1= AttributeValuePair.New( "row offset:", true, 5 );
		_offValue1.setValue("0");
		offsetsPanel.add(_offValue1);	
	  } else {
		_offValue0 = AttributeValuePair.New( "time offset:", true, 5 );
		_offValue0.setValue("0");
		offsetsPanel.add(_offValue0);
	  }
	  aButton = new JButton("set offset");
	  aButton.addActionListener( new GetOffsetsAction() );
	  offsetsPanel.add( aButton );

	  if( dim > 1 ) {
		JPanel  framePanel = new JPanel(false);
		framePanel.setLayout(new FlowLayout());
		borderPane.add(framePanel);

		aButton = new JButton("next Frame");
		aButton.addActionListener( new NextEntryAction() );
		framePanel.add( aButton );

		aButton = new JButton("prev Frame");
		aButton.addActionListener( new PreviousEntryAction() );
		framePanel.add( aButton );
		
		_time = AttributeValuePair.New( "time:", true, 16 );
		framePanel.add(_time);
		DataEntry de = _data.getEntry( 0 );
		if( de != null ) {	
		  _time.setValue( Float.toString( de.time() ) );
		}

		aButton = new JButton("set Time");
		aButton.addActionListener( new SetTimeAction() );
		framePanel.add( aButton );

		if( dim > 2 ) {
		  JPanel   layerPanel = new JPanel(false);
		  layerPanel.setLayout(new FlowLayout());
		  borderPane.add(layerPanel);
		  
		  aButton = new JButton("next Layer");
		  aButton.addActionListener( new NextLayerAction() );
		  layerPanel.add( aButton );

		  aButton = new JButton("prev Layer");
		  aButton.addActionListener( new PreviousLayerAction() );
		  layerPanel.add( aButton );
		  
		  _layer = AttributeValuePair.New( "layer:", false, 5 );
		  layerPanel.add(_layer);
		  _layer.setValue( Integer.toString( _currentLayer ) );
		}
	  }
	}	
	return borderPane;
    }

	public void setOffsets( int off0, int off1 ) {
	  _offset0 = off0; _offValue0.setValue( Integer.toString(off0) );
	  _offset1 = off1; _offValue1.setValue( Integer.toString(off1) );
	  if( _data.dims() > 1 ) { resetColumnLabels(); }
	  _dataModel.fireTableDataChanged();
	  repaint(); 
	}
	
	public int getOffset( int dim ) { return (dim==0) ? _offset0 : _offset1; }

	private void getOffsets() {
	  String s0 = _offValue0.getValue();
	  int dims = _data.dims();  
	  int adim = _data.getArrayDimension();
	  if( (dims == 1) || (adim > 0) ) {
		try {
		  Float f = Float.valueOf(s0); 
		  _time_offset = f.floatValue();
		} catch ( NumberFormatException err ) {
		  SimIO.print(" NumberFormatException ");
		  _time_offset = 0; _offValue0.setValue("0");
		}
		SimIO.print(" got time offset: " + _time_offset );
	  }
	  if(  adim > 1 ) {
		String s1 = _offValue1.getValue();
		try { 
		  _offset1 = Integer.parseInt(s1);
		} catch ( NumberFormatException err ) {
		  SimIO.print(" NumberFormatException ");
		  _offset1 = 0; _offValue1.setValue("0");
		}
		int ds1 =  _data.size( 1 );
		if( _offset1 >=  ds1 ) {
		  _offset1 = ds1-1; _offValue1.setValue(Integer.toString(_offset1));
		}
		if(  adim > 2 ) {
		  String s2 = _offValue2.getValue();
		  try { 
			_offset2 = Integer.parseInt(s2);
		  } catch ( NumberFormatException err ) {
			SimIO.print(" NumberFormatException ");
			_offset2 = 0; _offValue2.setValue("0");
		  }
		  int ds2 =  _data.size( 2 );
		  if( _offset2 >=  ds2 ) {
			_offset2 = ds2-1; _offValue2.setValue(Integer.toString(_offset2));
		  }
		}
	  } else if( dims > 1 ) {
		try { 
		  _offset0 = Integer.parseInt(s0);
		} catch ( NumberFormatException err ) {
		  SimIO.print(" NumberFormatException ");
		  _offset0 = 0; _offValue0.setValue("0");
		}
		String s1 = _offValue1.getValue();
		try { 
		  _offset1 = Integer.parseInt(s1);
		} catch ( NumberFormatException err ) {
		  SimIO.print(" NumberFormatException ");
		  _offset1 = 0; _offValue1.setValue("0");
		}
	  }
	  if( _data.dims() > 1 ) { resetColumnLabels(); }
	  _dataModel.fireTableDataChanged();
	  repaint(); 
	}
	
    /** Construct a menu. */

    private JMenuBar constructMenuBar() {
	JMenu            menu;
	JMenuBar         menuBar = new JMenuBar();
	JMenuItem        menuItem;

	menu = new JMenu("File");
	menu.setBackground(Color.lightGray);
	menuBar.add(menu);

	JMenu save_menu = new JMenu("Save");
	menuBar.add(save_menu);

	menuItem = save_menu.add(new JMenuItem("binary"));
	menuItem.addActionListener(new SaveAction(this));

	menuItem = save_menu.add(new JMenuItem("ASCII"));
	menuItem.addActionListener(new SaveASCIIAction());

	menuItem = menu.add(new JMenuItem("Quit"));
	menuItem.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		   stop();
		   destroy();
		   if( _frame != null ) { _frame.dispose(); }
	    }});

	menu = new JMenu("View");
	menuBar.add(menu);

//	menuItem = menu.add(new JMenuItem("Manual rescale"));
//	menuItem.addActionListener(new ManualScaleAction());

	menuItem = menu.add(new JMenuItem("1"));
//	menuItem.addActionListener(new ReloadAction());

	menuItem = menu.add(new JMenuItem("2"));
//	menuItem.addActionListener(new RemoveAction());

	menu = new JMenu("Help");
	menuBar.add(menu);

	try {
	  menuItem = menu.add(new JMenuItem("DataTable help"));
	  HelpBroker hb = SimIO.getHelpBroker( "SME", "VdataTable"  );
	  menuItem.addActionListener( new CSH.DisplayHelpFromSource(hb)  );
	} catch( Exception err ) {;}

	return menuBar;
    }

    class SaveAction extends Object implements ActionListener {
	  IOController _ac;
	  public SaveAction( IOController ac ) { super(); _ac = ac; }
	  public void actionPerformed(ActionEvent e) {
		SimulationData sd = DataSet.getCurrentSimulation();
		_saveControlPanel = SaveFileControlPanel.showControlPanel( _ac, _name, sd.getProjPath(), null );
	  }
    } 

    class SaveASCIIAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) { saveAsASCII(); }
    } 
    
    class NextEntryAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) {
		if( (_data.dims() > 1) && (_currentFrame < (_data.size()-1)) ) {
		  DataEntry de = _data.getEntry( ++_currentFrame );	
		  _dataModel.fireTableDataChanged();
		  repaint(); 
		  if( de != null ) {
			String f = 	Float.toString( de.time() );
			_time.setValue( f );
		  }	  
		}
	  }
    } 

    class SetTimeAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) {
		if( _data.dims() > 1 ) {
		  String s0 = _time.getValue();
		  try {
			Float f = Float.valueOf(s0); 
			setTime( f.floatValue() );
		  } catch ( NumberFormatException err ) {
			SimIO.print(" NumberFormatException ");
		  }
		}
	  }
    }
      
	 public void setIndex( int index ) { 
	   _currentFrame = ( index >= 0 ) ? index : 0;
	   int max_index = _data.size() - 1;
	   if( _currentFrame > max_index ) { _currentFrame = max_index; }
	   SimIO.print(" Set current Frame: " + _currentFrame );
	   _dataModel.fireTableDataChanged();
	   DataEntry de =  (DataEntry) _data.getEntry( index );
	   String f = 	Float.toString( de.time() );
	   _time.setValue( f );
	   repaint(); 
	 }
    
    public int setTime( float time ) { 
	  if( _dt < 0f ) { _dt = getTimestep(); }
	  if( _dt < 0f  ) {
		_currentFrame = 0;
	  } else {
		_currentFrame = Math.round(time/_dt);
		int max_index = _data.size() - 1;
		if( _currentFrame > max_index ) { _currentFrame = max_index; }
		if( _currentFrame < 0 ) { _currentFrame = 0; }
	  }
	  SimIO.print(" Set current Frame: " + _currentFrame );
	  _dataModel.fireTableDataChanged();
	  String f = 	Float.toString( time );
	  _time.setValue( f );
	  repaint(); 
	  return _currentFrame;
    }

    public float getTimestep() {
	  DataEntry id0 =  (DataEntry) _data.getEntry( 0 );
	  DataEntry id1 =  (DataEntry) _data.getEntry( 1 );
	  try {
		float t0 = id0.time();
		float t1 = id1.time();
		return t1 - t0;
	  } catch ( NullPointerException e) {
		return -1f;;
	  }
    }


    class PreviousEntryAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) {
		if( (_data.dims() > 1) && (_currentFrame > 0) ) {
		  DataEntry de = _data.getEntry( --_currentFrame );
		  _dataModel.fireTableDataChanged();
		  repaint(); 
		  if( de != null ) {	
			String f = 	Float.toString( de.time() );
			_time.setValue( f );
		  }
		}	  
	  }
    } 
    
    class GetOffsetsAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) {   getOffsets();   }
    } 

    class NextLayerAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) {
		if( (_data.dims() > 2) && (_currentLayer < (_data.size(2)-1)) ) {
		  _layer.setValue( Integer.toString( ++_currentLayer ) );
		}
	  }
    } 

    class PreviousLayerAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) {
		if( (_data.dims() > 2) && (_currentLayer > 0) ) {
			_layer.setValue( Integer.toString( --_currentLayer ) );
		}	  
	  }
    } 
        
	void generateHelpText() {
	  _helpText = 
		"  Displays raw data form DataSets: \n"
	  + "  1D: Displays segment of timeseries, use time offset to move forward in series. \n"
	  + "  2D: Displays array of data from map, use row/col offsets to set location of array in map. \n"
	  + "  3D: Displays array of data from one layer of (3D)map, use Layer buttons to change layer. \n"
	  + "  Use Frame buttons to move between entries in DataSet.  ";
	}
	    
	void print(String msg) {
	  System.out.print("\nJAVA: " + msg);
	}

    public void start() { repaint(); }   
    public void initDemo() { ; }
    public void stop() { ; }
    
    public void destroy() { 
	  _table = null;
	  if( _saveControlPanel != null ) { 
		_saveControlPanel.dispose(); 
		_saveControlPanel = null; 
	  }
	  if( _animationViewer != null ) {
		_animationViewer.invalidate_dataTable();
	  }
	}
	
    public void dispose() {  
	  if( _frame != null )  {
		_frame.setVisible(false);
		_frame.dispose();
	  }
	}
	
    public void setFrame(JFrame f) {  _frame = f; }
	public Container getContentPane() { return this; }


   public void saveData( String dir, String baseName, SimulationData data ) {;} 

   public void saveData( String dir, String baseName, int format, boolean saveAllFrames ) {
   /*
	 try {
	   if( saveAllFrames ) {
		 switch( format ) {
//		   case 0: _table.SaveAnimationPPM(dir,baseName); break;
//		   case 2: _table.SaveAnimationJPEG(dir,baseName); break;
		 }
	   }
	   else {
		 switch( format ) {
//		   case 0: _table.SaveImagePPM(-1,dir,baseName); break;
//		   case 2: _table.SaveImageJPEG(-1,dir,baseName); break;
		 }
	   }
	  } catch ( java.io.IOException err ) {
		SimIO.print( "File write exception: " + err.getMessage() );
	  }
*/
   } 
 
  
	public void intervalAdded(ListDataEvent e) {;}
    public void intervalRemoved(ListDataEvent e) {;}
    public void contentsChanged(ListDataEvent e) {;}


	public static void main(String args[])  {
//	  DataSet ds = DataSet.create2DDemo( 5, 100, 120 ); 
	  DataSet ds = DataSet.create1DDemo( 50, 5 ); 
	  DataTable dt = getTable( ds, true, true, true );
	  
//	  DataTable dt = new DataTable( ds );
//	  AppContainer ac = new AppContainer( ds.toString(), dt, null );
//	  ds.addEntryDataListener(dt);
//	  ac.setVisible(true);
	}
	
	public void formatColumns() {
		_table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		_table.setAutoCreateColumnsFromModel(false); 
        TableColumnModel cm = _table.getColumnModel();
        DefaultTableCellRenderer indexColumnRenderer = new DefaultTableCellRenderer();
        indexColumnRenderer.setBackground(Color.lightGray);
        indexColumnRenderer.setHorizontalAlignment(JLabel.RIGHT);
        DefaultTableCellRenderer dataColumnRenderer = new DefaultTableCellRenderer() {
		  public void setValue(Object value) {
			FormattedFloat ff = (FormattedFloat)value;
			if( value != null ) {
			  if( ff.isNaN() ) { setBackground( Color.black ); setText( "" ); }
			  else if( ff.isINF() ) { setBackground( Color.red ); setText( "INF" ); }
			  else { setBackground(Color.white); setText( value.toString() ); }
	        } else {
			  setBackground( Color.black ); setText( "" );
			}
		  }
        };
		dataColumnRenderer.setForeground( Color.black );
        dataColumnRenderer.setHorizontalAlignment(JLabel.RIGHT);
		int dim = _data.dims();
		int adim = _data.getArrayDimension();
		for( int columnIndex = 0; columnIndex < _dataModel.getColumnCount(); columnIndex++ ) {
		  TableColumn tableColumn =  cm.getColumn(columnIndex);
		  if( (dim < 2) && (adim < 1) ) {
			if( ( columnIndex % 2 ) == 0 ) {
			  tableColumn.setCellRenderer(indexColumnRenderer); 
			  tableColumn.setWidth(60); 
			} else {
			  tableColumn.setCellRenderer(dataColumnRenderer); 
			  tableColumn.setWidth(80); 
			}
		  } else {
			if( columnIndex == 0 ) {
			  tableColumn.setCellRenderer(indexColumnRenderer); 
			  tableColumn.setWidth(30); 
			} else {
			  tableColumn.setCellRenderer(dataColumnRenderer); 
			  tableColumn.setWidth(80); 
			}
		  }
		}
    }

	public void resetColumnLabels() {
        TableColumnModel cm = _table.getColumnModel();
		for( int columnIndex = 1; columnIndex < _dataModel.getColumnCount(); columnIndex++ ) {
		  TableColumn tableColumn =  cm.getColumn(columnIndex);
		  String columnName = _dataModel.getColumnName(columnIndex);
		  tableColumn.setHeaderValue(columnName);
		}
    }
}


class myWindowAdapter extends WindowAdapter {
	private DataTable _dt;
	private boolean _isDemo;
	myWindowAdapter( DataTable dt, boolean isDemo ) { _dt = dt; _isDemo = isDemo; }
	public void windowClosing(WindowEvent e) { 
	  if( _isDemo ) System.exit(0);
	  else _dt.destroy();
	}			  
}

