package miiee.dataview;

import java.awt.*;
import javax.swing.*;
import javax.swing.table.*;
import javax.swing.event.*;
import com.sun.image.codec.jpeg.*;
import java.awt.image.*;
import java.io.FileOutputStream;
import java.io.File;
import java.awt.geom.AffineTransform;
import java.util.Vector;
import java.util.HashMap;
import java.awt.event.*;
import miiee.util.*;
import javax.help.*;
import java.io.DataOutputStream;  

public class ImageTable extends JComponent implements DisposableListDataListener {

    protected JTable       _table;
    protected ImageDataTableModel _dataModel = new ImageDataTableModel();
	
	private int _data_width =  0;
	private int _data_height =  0;
	private boolean _debug =  false;
	private float _mag_factor = 2f;
	private AttributeValuePair _timeOffset;
	private AttributeValuePair _timeStep;
	private AttributeValuePair _magFactor;
	private AttributeValuePair _rescaleMax;
	private AttributeValuePair _rescaleMin;
	private TextPaneTableCellRenderer _indexColumnRenderer;
	private DefaultTableCellRenderer _dataColumnRenderer;
    protected String _helpText;
	private JFrame _frame;

    public ImageTable() {
	   super();
 	   _table = new JTable(_dataModel);
   }

    public void addDataSet( DataSet d ) {
    	if( d.dims() != 2 ) {
		  SimIO.print("This viewer only accepts 2D dataSets, ignoring " + d );
		  return;
		}
				
		if( _data_width <  d.size(0) )  _data_width = d.size(0);
		if( _data_height < d.size(0) )  _data_height= d.size(1);

	   float dt = _dataModel.addDataSet( d );
	   if( _timeStep != null ) {
		 _timeStep.setValue( Float.toString(dt) );
	   }
	   d.addEntryDataListener( this );
    }
            
    public synchronized void init() { 
        _indexColumnRenderer = new TextPaneTableCellRenderer( );
        _indexColumnRenderer.setFontSize( 10f );
        _dataColumnRenderer = new DefaultTableCellRenderer() {
		  public void setValue(Object value) {
			ImageIcon icon = (ImageIcon)value;
			if( icon != null ) {
			  setIcon(icon);
	        } else {
			  setBackground( Color.black ); 
			}
		  }
		  public void paint(Graphics g) {
			ImageIconJ2D image =  (ImageIconJ2D)getIcon();
			image.paintIcon(this,g,0,0);
			super.paint(g);
		  }
        };
        _dataColumnRenderer.setHorizontalAlignment(JLabel.RIGHT);		
        formatColumns(); 
		JScrollPane scrollpane = new JScrollPane(_table);
		scrollpane.setPreferredSize(new Dimension(600, 350));
		scrollpane.setBorder( BorderFactory.createLoweredBevelBorder() );
		
		setLayout(new BorderLayout());
		add( "Center", scrollpane );
		add( "South", constructOptionsPanel() );
		_dataModel.magnifyImages( _mag_factor );		
    }
    
	public Vector dataSets() { return _dataModel.dataSets(); }
	
	public  DataSet getDataSet() { return (DataSet) dataSets().get(0); }
	
	private JPanel constructOptionsPanel() {
	  JPanel           borderPane = new JPanel(false);  
	  JButton aButton;

	  borderPane.setLayout(new GridLayout(2,1));

	  JPanel   offsetsPanel = new JPanel(false);
	  BoxLayout bl = new BoxLayout(offsetsPanel, BoxLayout.X_AXIS ); offsetsPanel.setLayout( bl ); 	 
	  borderPane.add(offsetsPanel);
	  
	  JPanel  magPanel = new JPanel( new FlowLayout() );
	  magPanel.setMaximumSize(new Dimension(Short.MAX_VALUE,  Short.MAX_VALUE)); 
//	  magPanel.setAlignmentY( Component.CENTER_ALIGNMENT );
	  magPanel.setBorder( BorderFactory.createEtchedBorder() );
	  offsetsPanel.add(magPanel);
	  
	  _magFactor = AttributeValuePair.New( "magnification:", true, 5 );
	  _magFactor.setValue( Float.toString( _mag_factor ) );
	  magPanel.add(_magFactor);

	  aButton = new JButton("set");
	  aButton.addActionListener( new GetMagAction() );
	  magPanel.add( aButton );
				
	  JPanel  buttonsPanel = new JPanel( new FlowLayout() );
	  buttonsPanel.setMaximumSize(new Dimension(Short.MAX_VALUE,  Short.MAX_VALUE)); 
//	  buttonsPanel.setAlignmentY( Component.CENTER_ALIGNMENT );
	  buttonsPanel.setBorder( BorderFactory.createEtchedBorder() );
	  offsetsPanel.add(buttonsPanel);
	  
	  aButton = new JButton("rescale");
	  aButton.addActionListener( new RescaleAction() );
	  buttonsPanel.add( aButton );		

	  aButton = new JButton("toggle smoothing");
	  aButton.addActionListener( new SmoothAction() );
	  buttonsPanel.add( aButton );		

	  offsetsPanel = new JPanel(false);
	  bl = new BoxLayout(offsetsPanel, BoxLayout.X_AXIS ); offsetsPanel.setLayout( bl ); 	 
	  borderPane.add(offsetsPanel);

	  JPanel  timePanel = new JPanel( new FlowLayout() );
	  timePanel.setMaximumSize(new Dimension(Short.MAX_VALUE,  Short.MAX_VALUE)); 
//	  timePanel.setAlignmentY( Component.CENTER_ALIGNMENT );
	  timePanel.setBorder( BorderFactory.createEtchedBorder() );
	  offsetsPanel.add(timePanel);

	  _timeOffset = AttributeValuePair.New( "time offset:", true, 5 );
	  _timeOffset.setValue("0");
	  timePanel.add(_timeOffset);

	  aButton = new JButton("set");
	  aButton.addActionListener( new GetOffsetsAction() );
	  timePanel.add( aButton );		

	  JPanel  dtPanel = new JPanel( new FlowLayout() );
	  dtPanel.setMaximumSize(new Dimension(Short.MAX_VALUE,  Short.MAX_VALUE)); 
//	  dtPanel.setAlignmentY( Component.CENTER_ALIGNMENT );
	  dtPanel.setBorder( BorderFactory.createEtchedBorder() );
	  offsetsPanel.add(dtPanel);

	  _timeStep = AttributeValuePair.New( "timestep:", true, 5 );
	  _timeStep.setValue( Float.toString( _dataModel.getTimeStep() ));
	  dtPanel.add( _timeStep );

	  aButton = new JButton("set");
	  aButton.addActionListener( new GetTimeStepAction() );
	  dtPanel.add( aButton );	
	  	
	  return borderPane;
    }

    private JMenuBar constructMenuBar() {
	  JMenu            menu;
	  JMenuBar         menuBar = new JMenuBar();
	  JMenuItem        menuItem;

	  menu = new JMenu("File");
	  menu.setBackground(Color.lightGray);
	  menuBar.add(menu);

	  menuItem = menu.add(new JMenuItem("Add selected DataSet(s)"));
	  menuItem.addActionListener(new AddDataSetAction());

	  menuItem = menu.add(new JMenuItem("Delete row (DataSet)"));
	  menuItem.addActionListener(new DeleteDataSetAction());

	  menuItem = menu.add(new JMenuItem("Save"));
//	  menuItem.addActionListener(new SaveAction(this));

	  menuItem = menu.add(new JMenuItem("Quit"));
	  menuItem.addActionListener(new ActionListener() {
		  public void actionPerformed(ActionEvent e) {
			 if( _frame != null ) { _frame.dispose(); }
		  }});

	  menu = new JMenu("View");
	  menuBar.add(menu);

	  menuItem = menu.add(new JMenuItem("Manual rescale"));
	  menuItem.addActionListener(new ManualScaleAction());

	  menuItem = menu.add(new JMenuItem("Toggle grayscale"));
	  menuItem.addActionListener(new ToggleGrayscaleScaleAction());

	  menuItem = menu.add(new JMenuItem("1"));
  //	menuItem.addActionListener(new InsertAction());

	  menuItem = menu.add(new JMenuItem("2"));
  //	menuItem.addActionListener(new ReloadAction());

	  menuItem = menu.add(new JMenuItem("3"));
  //	menuItem.addActionListener(new RemoveAction());

	  menu = new JMenu("Help");
	  menuBar.add(menu);

	  try {
		menuItem = menu.add(new JMenuItem("ImageTable help"));
		HelpBroker hb = SimIO.getHelpBroker( "SME", "VimageTable"  );
		menuItem.addActionListener( new CSH.DisplayHelpFromSource(hb)  );
	} catch( Exception err ) {;}

	  return menuBar;
    }

        
	void generateHelpText() {
	  _helpText = "  To Scale images: \n"
	  + "  1. Use mouse to select image or row ( click on first (TEXT) panel in row ), \n"
	  + "  2. Click on rescale button to rescale row based on selected image, \n"
	  + "  3. Use 'Manual rescale' under View menu to manually rescale selected image or row \n"
	  + "  4. Select row and click on rescale button to reset default scaling \n"
	  + "  Magnify images by entering mag factor in magnification panel. \n \n "
	  + "  Setting the time offset changes the display window such that the first \n  "
	  + "  column of images corresponds to the time (offset) specified.  ";
	}


 	public void formatColumns() {
		_table.setAutoResizeMode(JTable.AUTO_RESIZE_ALL_COLUMNS);
		_table.setAutoCreateColumnsFromModel(false); 
		_table.setCellSelectionEnabled(true);
		int imageHeight = (int) ( _data_height*_mag_factor + 5 );
		_table.setRowHeight( imageHeight );
		int imageWidth = (int)(_data_width*_mag_factor);
		SimIO.print("format Columns: w: " + imageWidth + " , h : " + imageHeight );
		TableColumnModel cm = _table.getColumnModel();
		for( int columnIndex = 0; columnIndex < _dataModel.getColumnCount(); columnIndex++ ) {
		  TableColumn tableColumn =  cm.getColumn(columnIndex);
		  if( columnIndex == 0 ) {
			tableColumn.setCellRenderer(_indexColumnRenderer); 
			tableColumn.setWidth(120); 
			tableColumn.setMinWidth(120);
			tableColumn.setMaxWidth(125);
			tableColumn.setPreferredWidth(122);
		  } else {
			tableColumn.setCellRenderer(_dataColumnRenderer); 
			tableColumn.setWidth(imageWidth+3);
			tableColumn.setMinWidth(imageWidth);
			tableColumn.setMaxWidth(imageWidth+6);
			tableColumn.setPreferredWidth(imageWidth+3);
		  }
		}
		_table.sizeColumnsToFit(-1);
		_table.revalidate();
        _table.repaint(); 
    } 

	public void resetColumnLabels() {
        TableColumnModel cm = _table.getColumnModel();
		for( int columnIndex = 1; columnIndex < _dataModel.getColumnCount(); columnIndex++ ) {
		  TableColumn tableColumn =  cm.getColumn(columnIndex);
		  String columnName = _dataModel.getColumnName(columnIndex);
		  tableColumn.setHeaderValue(columnName);
		}
    }
     
   class GetMagAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) { 
		String s0 = _magFactor.getValue();
		try {
		  Float f = Float.valueOf(s0); 
		  _mag_factor = f.floatValue();
		  formatColumns();
		  _dataModel.magnifyImages( _mag_factor );
		} catch ( NumberFormatException err ) {
		  SimIO.print(" NumberFormatException ");
		  _magFactor.setValue( Float.toString(_mag_factor) );
		}		  
		repaint(); 
	  }
    } 


   class RescaleAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) { 
		int col = _table.getSelectedColumn();
		if( col < 0 ) { SimIO.beep(); return; }
		int row = _table.getSelectedRow();
		if( row < 0 ) { SimIO.beep(); return; }
		DataSet d = _dataModel.getDataSet(row);
		if( d == null ) { SimIO.beep(); return; }
		float max, min;
		if( col > 0 ) {
		  DataEntry id = _dataModel.getDataEntryAt( d, col );
		  if( id == null ) { SimIO.beep(); return; }
		  max = id.max(); min = id.min();
		} else  {
		  max = d.getDataMax();  min = d.getDataMin();
		}
		_dataModel.scaleImages( d, null, row, max, min );
		repaint(); 
	  }
    } 
	
   class SmoothAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) { 
		int row = _table.getSelectedRow();
		_dataModel.toggleSmoothing(row);
		repaint(); 
	  }
    } 
	
    class GetOffsetsAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) { 
		String s0 = _timeOffset.getValue();
		try {
		  Float f = Float.valueOf(s0); 
		  _dataModel.setTimeOffset( f.floatValue() );
		  resetColumnLabels();
		} catch ( NumberFormatException err ) {
		  SimIO.print(" NumberFormatException ");
		  _timeOffset.setValue("0");
		}
		repaint(); 
	  }
    } 

    class DeleteDataSetAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) { 
		  int row = _table.getSelectedRow();
		  if( row < 0 ) { SimIO.beep(); return; }
		  _dataModel.deleteRow(row);
		  repaint(); 
	  }
    } 

    class AddDataSetAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) { 
		  Vector ds = new Vector(16,16);
		  int nSelections = DataSet.getTreeDataSetSelections( ds, -1 );
		  if( nSelections <= 0 ) { SimIO.beep(); return; }
		  for( int i=0; i < ds.size(); i++ ) {
			DataSet d = (DataSet) ds.elementAt(i);
			_dataModel.addDataSet(d);
		  }
		  repaint(); 
	  }
    } 
 
     class ToggleGrayscaleScaleAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) {
		ImageIconJ2D.toggleGrayscale();
	  }
    } 

     class GetTimeStepAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) { 
		String s0 = _timeStep.getValue();
		try {
		  Float f = Float.valueOf(s0); 
		  SimIO.print(" Setting timestep: " + f.floatValue() );
		  _dataModel.setTimeStep( f.floatValue() );
		  resetColumnLabels();
		} catch ( NumberFormatException err ) {
		  SimIO.print(" NumberFormatException ");
		  _timeStep.setValue("0"); SimIO.beep();
		}
		repaint(); 
	  }
    } 

    class ManualScaleAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) {
		int col = _table.getSelectedColumn();
		if( col < 0 ) { SimIO.beep(); return; }
		int row = _table.getSelectedRow();
		if( row < 0 ) { SimIO.beep(); return; }
		DataSet d = _dataModel.getDataSet(row);
		if( d == null ) { SimIO.beep(); return; }
		DataEntry id = null;
		if( col > 0 ) {
		  id = _dataModel.getDataEntryAt( d, col );
		  if( id == null ) { SimIO.beep(); return; }
		} 
		Scale s = _dataModel.getScale(row);
		float max, min;
		if( s != null ) {
		  max = s._max; min = s._min;
		} else if( id != null ) {		  
		  max = id.max(); min = id.min();
		} else  {
		  max = d.getDataMax();  min = d.getDataMin();
		}
		open_manual_scale_panel( max, min, d, id, row );
	  }
    } 

    class ManualScaleSetAction extends Object implements ActionListener {
	  JFrame _frame;
	  DataSet _d;
	  DataEntry _id;
	  int _row;
	  ManualScaleSetAction(JFrame frame, DataSet d, DataEntry id, int row  ) { 
		_frame = frame; _d = d; _id = id; _row = row;
	  }
	  public void actionPerformed(ActionEvent e) {
		try {
		  String max =  _rescaleMax.getValue();
		  Float fmax = Float.valueOf(max);
		  String min =  _rescaleMin.getValue();	
		  Float fmin = Float.valueOf(min);
		  _dataModel.scaleImages( _d, _id, _row, fmax.floatValue(), fmin.floatValue() );
		  repaint();
		  _frame.setVisible(false);
		  _frame.dispose();
		} catch ( NumberFormatException ex ) { 
		  SimIO.print(" NumberFormatException in ManualScaleSetAction "); 
		  SimIO.beep();
		}
	  }
    } 
   
    class ManualScaleCancelAction extends Object implements ActionListener {
	  JFrame _frame;
	  ManualScaleCancelAction(JFrame frame) { _frame = frame; }
	  public void actionPerformed(ActionEvent e) {
		_frame.setVisible(false);
		_frame.dispose();
	  }
    } 

   	void open_manual_scale_panel( float max, float min, DataSet d, DataEntry id, int row ) {
		int infolen = 12;
		JPanel  panel = new JPanel(true);
		JFrame frame = new JFrame("Manual ReScale");
		frame.getContentPane().add("Center", panel);
		panel.setLayout( new GridLayout(0,1) );

		frame.addWindowListener( 
			new WindowAdapter() {
			  public void windowClosing(WindowEvent e) { ; }
			}
		);

		if( _rescaleMax == null ) {
		  _rescaleMax = AttributeValuePair.New( "Max value:", true, infolen );
		}
		_rescaleMax.setValue( Float.toString(max) );
		panel.add(_rescaleMax);

		if( _rescaleMin == null ) {
		  _rescaleMin = AttributeValuePair.New( "Min value:", true, infolen );
		}
		_rescaleMin.setValue( Float.toString(min) );
		panel.add(_rescaleMin);

		JPanel  buttonPanel = new JPanel(false);
		buttonPanel.setLayout(new FlowLayout());
		panel.add(buttonPanel);

		JButton aButton;
		aButton = new JButton("Set");
		aButton.addActionListener( new ManualScaleSetAction(frame,d,id,row) );
		buttonPanel.add( aButton );

		aButton = new JButton("Cancel");
		aButton.addActionListener( new ManualScaleCancelAction(frame) );
		buttonPanel.add( aButton );

		frame.pack();
		frame.show();
	}
   
	public void dispose() { if( _frame != null ) { _frame.dispose(); } }
	public void destroy() { ; }

    public void intervalAdded(ListDataEvent e) { 
	  if( _dataModel.intervalAdded(e) ) {
		resetColumnLabels();
		_dataModel.fireTableDataChanged();
	  }
	  repaint();
	}
	
    public void intervalRemoved(ListDataEvent e) {;}
    public void contentsChanged(ListDataEvent e) {;}
	
	public JFrame show( String name ) {
	  _frame = new JFrame(name);
	  WindowListener l = new WindowAdapter() {
		  public void windowClosing(WindowEvent e) { dispose(); }
	  };
	  _frame.addWindowListener( l );
            
	  init();
 
	  _frame.getContentPane().add(this);
	  _frame.setJMenuBar( constructMenuBar() );
	  _frame.setBackground(Color.lightGray);	
	  _frame.pack();
	  _frame.setVisible(true);
	  return _frame;
	}
	
    public static void main(String s[]) {

		int width = 150;
		int height = 100;
		int max_images = 5;	
		       
        ImageTable iT = new ImageTable();
        
        DataSet ds = DataSet.create2DDemo( max_images, width, height );
        iT.addDataSet(ds);
        JFrame frame = iT.show( "Image Table" );
        		   		
        WindowListener l = new WindowAdapter() {
            public void windowClosing(WindowEvent e) {System.exit(0);}
            public void windowClosed(WindowEvent e) {System.exit(0);}
        };
        frame.addWindowListener(l);
	  }

}

class ImageIconJ2D extends ImageIcon {

	private AffineTransform _transform;
	private AffineTransformOp _transformOp; 
	private BufferedImage _offImg;
	private DataSet _ds;
	private DataEntry _id;
	private float _scale_z = 1;
	private float _translate_z;
	private float _max;
	private float _min;
	private Color _background = Color.black;
	private RenderingHints _rh = new RenderingHints(RenderingHints.KEY_ANTIALIASING,RenderingHints.VALUE_ANTIALIAS_ON);
	private float _mag = 2.0f;
	private float _max_raster_val;
	private float _max_uns_raster_val;
	private boolean _smooth_image = true;
	private boolean _clear_canvas = true;
	private boolean _reset_transform = true;
	private boolean _recalc_image_data = true;
	private static int _colormapIndex = 1;

	ImageIconJ2D( DataSet d, DataEntry id ) {
	   float byte_max = 256f;	
	   _max_raster_val = Short.MAX_VALUE; 		
	   _max_uns_raster_val = byte_max*byte_max;		
	  _ds = d; _id = id;
	  BufferedImage bi = new BufferedImage ( d.size(0), d.size(1), BufferedImage.TYPE_INT_RGB );
	  setDescription( d.toString() );
	  setImage(bi);
	}

	public static void toggleGrayscale() {
	  _colormapIndex = ( _colormapIndex > 0 ) ? 0 : 1;
	}	 
	public DataSet getDataSet() { return _ds; }
	 
	public void calcTransform() {
	  if( _reset_transform ) {
        _transform = new AffineTransform( _mag, 0, 0, _mag, 0, 0 ); 
        _transformOp = new AffineTransformOp( _transform, _rh );
	  }
	}
	
	public void toggleSmoothing() { _smooth_image = !_smooth_image; }
	
    public int getIconWidth() {
	  BufferedImage bi = (BufferedImage) getImage();
	  return (int) (bi.getWidth()*_mag); 
    }

    public int getIconHeight() {
	  BufferedImage bi = (BufferedImage) getImage();
	  return (int) (bi.getHeight()*_mag);
    }

	public int setMagnification( float mag ) {
	  if(  _mag != mag  ) {
		_mag =  mag;
		_reset_transform = true;
		return 1;
	  } else return 0;
	}

	void resetScaling( ) {
	  _translate_z = 0f;
	  _scale_z = 255f / _max_uns_raster_val;
	  _recalc_image_data = true;
	}
    
	void setScaling( float max, float min ) {
	  _max = max; _min = min;
	  float ds = max - min;
	  if( ds == 0f ) {
		_scale_z =   255f / _max_uns_raster_val;
		_translate_z = 0f;
	  } else {
		_scale_z = ( (_id.max() - _id.min()) / ds ) * ( 255f / _max_uns_raster_val );
		_translate_z = ( _id.min() - min ) * ( 255f / ds );
	  }
	  _recalc_image_data = true;
//	  SimIO.print("Calc scaling: ref( " + max + " , " + min + " ) : cur( " + _id.max() + " , " + _id.min() + " ) -> s: " + _scale_z + " , o: " + _translate_z  );
	}
	
	public float getMax() { return _max; }
	public float getMin() { return _min; }
	
	public void setSmoothing( boolean b ) {
	  _smooth_image = b;
	  _clear_canvas = true;
	}
		
    public synchronized void paintIcon(Component c, Graphics g, int x, int y) {
		try {
		  BufferedImage image = (BufferedImage)getImage();
		  filterData( _ds, _id, image );
		  Graphics2D g2 = createGraphics2D( image, c, g );
		  if( g2 == null ) return;
		  g2.drawImage( (BufferedImage)getImage(), null, 0, 0 );
		  g2.dispose();
		} catch (java.awt.image.ImagingOpException err) {
		  SimIO.print( "In ImageTable.renderImage: " + err.getMessage() );
		  err.printStackTrace();
		}	
        if ( _offImg != null  )  {
            g.drawImage( _offImg, x, y, c );
        }
    }
   
	public Graphics2D createGraphics2D( BufferedImage image, Component c, Graphics g  ) {
        Graphics2D g2 = null;
		calcTransform();		
        int width = getIconWidth();
        int height = getIconHeight();
        if ( width <= 0 || height <= 0) { return null; }

		if (_offImg == null || _offImg.getWidth() != width || _offImg.getHeight() != height) {
			_offImg = (BufferedImage) c.createImage(width, height);
		} 

		if( _smooth_image ) { 
		  _rh.put( RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON );
		  _rh.put( RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY );
//		  _rh.put( RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BILINEAR );
		} else {
		  _rh.put( RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF );
		  _rh.put( RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_SPEED );
//		  _rh.put( RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_NEAREST_NEIGHBOR );
		}
			
		g2 = _offImg.createGraphics();
		g2.setTransform(_transform );
		g2.setRenderingHints( _rh ); 
		g2.setBackground( _background );
		if( _clear_canvas ) { g2.clearRect(0, 0, width, height); }
		
        return g2;
    }
    
	public synchronized void filterData( DataSet ds, DataEntry de, BufferedImage dst )  {  // Assumes default RGB color model.
	  if( _recalc_image_data ) {
		  _recalc_image_data  = false;
		  byte[] mask = ds.getActiveRegion();
		  byte format = ds.format();
		  short[] src_data = de.getData();
		  int width = ds.size(0);				
		  int height = ds.size(1);
		  int  max_src_raster_val = 256*256;	

		  int pixVal, rgb, r, g, b, index, mindex = -1;
		  for (int y=0; y < height; y++ ) {
			  for (int x = 0; x < width; x++ ) {
				  index = x + y*width;
				  if( ( mask == null ) || ( mask[index] > 0 ) ) { 
					if( format == 0 ) { mindex = index; }
					else { mindex++; }
					pixVal = src_data[mindex];
					if( pixVal == 0 ) {						
					  rgb =  (0xff << 24) | ( 0xff << 16) | ( 0xff << 8) | 0xff;
					} else {
					  pixVal = ( pixVal < 0 ) ? pixVal + max_src_raster_val : pixVal;
					  pixVal = (int) ( pixVal*_scale_z + _translate_z );
					  pixVal = ( pixVal  < 0 ) ? 0 : pixVal;
					  pixVal = ( pixVal  > 255 ) ? 255 : pixVal;
					  switch( _colormapIndex ) {
						case 0:       // grayscale
						  r =   g =   b = 255 - pixVal;
						break;
						default:      // rainbow
						  r = pixVal;
						  g = ( pixVal >= 128 ) ? ( 511 - 2*pixVal) : 2*pixVal;
						  b = 255 - pixVal;
						break;					  
					  }
					  rgb =  (0xff << 24) | ( r << 16) | ( g << 8) | b;
					}
				  } else {
					if( _colormapIndex > 0 ) {
					  rgb = (0xff << 24) | ((int)0);
					} else {
					  rgb = (0xff << 24) | ((int)100);
					}
				  }
				  dst.setRGB( x, y, rgb );
			  }
		  }		
	  }
	}
}

class Scale extends Object {
  public float _max;
  public float _min;
  public Scale( float max, float min ) {
	super(); 
	_max = max;
	_min = min;
  }
}

class ImageDataTableModel extends AbstractTableModel {
	private HashMap _images = new HashMap();
	private Vector _rowLabels = new Vector(16,16);
	private Vector _scaleLabels = new Vector(16,16);
	private Vector _dataSets = new Vector(16,16);
	private int _rows;
	private int _cols = 32;
	private float _dt = 0f;
	private float _time_offset = 0f;
	
	ImageDataTableModel() { super(); }
	
	public float addDataSet( DataSet d ) {
	  _dataSets.addElement(d); 
	  _rows = _dataSets.size();
	  int cols = 1500/d.size(0);
	  if( ( cols > 0 ) && ( cols < _cols ) ) { _cols = cols; }
	   if( _rows >= _rowLabels.size() ) { _rowLabels.setSize(_rows+16); }
	   _rowLabels.setElementAt( d.toString(), _rows-1 );
	   _dt = d.getTimestep();
	   SimIO.print( "Adding DataSet:" + d + "setting dt:" + (( _dt == Float.MAX_VALUE ) ? "NaN" : Float.toString(_dt) ) );
	   fireTableStructureChanged();
	   float max = d.getDataMax();
	   float min =  d.getDataMin();
	   scaleImages( d, null, _rows-1, max, min ); 
	   return _dt;
	}

	public void removeDataSet( DataSet ds ) {
	  if( _dataSets.remove(ds) ) { 
		 _rows = _dataSets.size();
		 for( int j=0; j< _dataSets.size(); j++ ) {
			DataSet d = (DataSet)_dataSets.get(j);
			if( d!= null ) {
			   _rowLabels.setElementAt( d.toString(), j );
			} else {
			  _rowLabels.setElementAt( "", j );
			}
		 }
		 for( int i=0; i< ds.size(); i++ ) {
			  DataEntry id = ds.getEntry(i);
			  if( id != null ) { _images.remove( id ); }
		 }
		 fireTableStructureChanged();
	  }
	}
	
	public void deleteRow( int row ) {
	  DataSet d = getDataSet(row);
	  if( d == null ) { SimIO.beep(); return; }
	  removeDataSet( d );
	}
					
	public String getColumnName(int column) {
	  if( column == 0 ) return "";
	  if( _dt == Float.MAX_VALUE ) { return ""; }
	  float time = _time_offset + (column-1)*_dt;
	  FormattedFloat ft = new FormattedFloat( time );
	  return ft.toJavaFormatString(2) ; 
	}

	public int getColumnCount() { 
		return _cols + 1; 
	}
	
	public int getRowCount() { 
		return _rows;
	}
	
	public Vector dataSets() { return _dataSets; }

	public DataSet getDataSet(int row) { 
		DataSet d = null;
		try {
		  d = (DataSet)_dataSets.get(row);
		} catch( ArrayIndexOutOfBoundsException err ) { ; }
		return d;
	}
	
	public Scale getScale(int row) {
		Scale scale = null;
		try {
		  scale = (Scale) _scaleLabels.get(row);
		} catch ( ArrayIndexOutOfBoundsException err ) {; }
		return scale;
	}
		
	public Object getValueAt( int row, int col) {
	  DataSet d = (DataSet)_dataSets.get(row);
	  if( d == null ) { return null; }
	  
	  if( col == 0 ) {
		StringBuffer sb = new StringBuffer(256);
		String s0 = null;
		try {
		  s0 = (String) _rowLabels.get(row);
		} catch ( ArrayIndexOutOfBoundsException err ) {; }
		
		sb.append( (s0==null) ? "unknown" : s0 );
		
		Scale s = getScale(row); 
		if( s != null ) { 
		  FormattedFloat fmax = new FormattedFloat(s._max);
		  FormattedFloat fmin = new FormattedFloat(s._min);
		  sb.append( "\nmax = "  );
		  sb.append(  fmax.toJavaFormatString(2) );
		  sb.append( "\nmin = "  );
		  sb.append(  fmin.toJavaFormatString(2) );
		}
		int nImages = d.size();
		sb.append( "\nNImages = "  );
		sb.append( Integer.toString(nImages)  );
		
		DataEntry id = d.getEntry(0);
		if( id != null ) {
		  FormattedFloat ft = new FormattedFloat(id.time());
		  sb.append( "\nt0 = "  );
		  sb.append(  ft.toJavaFormatString(2) );
		}

		id = d.getEntry(nImages-1);
		if( id != null ) {
		  FormattedFloat ft = new FormattedFloat(id.time());
		  sb.append( "\ntF = " );
		  sb.append( ft.toJavaFormatString(2) );
		}

		return sb.toString();
	  }
	  DataEntry id = getDataEntryAt( d, col);
	  return getImage( d, id, row );
	}

	public DataEntry getDataEntryAt( DataSet d, int col ) {
	  if( _dt == Float.MAX_VALUE ) {
		if( col == 1 ) return d.getEntry(0);
		else return null;
	  }
	  float time = _time_offset + (col-1)*_dt;
	  return d.getEntry(time);
	}
	
	public Class getColumnClass(int columnIndex) {
	  return ( columnIndex == 0 ) ? String.class : ImageIcon.class;
	}
			
	
	ImageIconJ2D getImage(DataSet d, DataEntry id, int row ) {
	  if( id == null ) return null;
	  ImageIconJ2D image = (ImageIconJ2D)_images.get(id);
	  if( image == null ) {
		image = new ImageIconJ2D( d, id );
		_images.put( id, image ); 
		 Scale s = getScale(row);
		 if( s != null ) {
		   image.setScaling( s._max, s._min ); 
		 }		
	  }
	  return image;
	}

	public void scaleImages( DataSet d, DataEntry id, int row, float max, float min ) {
	  if( id == null ) {
		for( int i=0; i< d.size(); i++ ) {
			ImageIconJ2D image = getImage( d, d.getEntry(i), row );
			if( image != null ) { image.setScaling(max,min); }
		}
	  } else {
		  ImageIconJ2D image = getImage( d, id, row );
		  if( image != null ) { image.setScaling(max,min); }
	  }
	  if( row >= _scaleLabels.size() ) { _scaleLabels.setSize(row+16); }
	  Scale s = new Scale(max,min);
	  _scaleLabels.setElementAt( s, row );
	  fireTableDataChanged();
	}

	public void toggleSmoothing(  int row ) {
	  if( row < 0 ) { SimIO.beep(); return; }
	  DataSet d = getDataSet(row);
	  if( d == null ) { SimIO.beep(); return; }
	  for( int i=0; i< d.size(); i++ ) {
		  ImageIconJ2D image = getImage( d, d.getEntry(i), row );
		  if( image != null ) { image.toggleSmoothing(); }
	  }
	  fireTableDataChanged();
	}

	
	public void magnifyImages(float mag) {
	  for( int j=0; j< _dataSets.size(); j++ ) {
		DataSet d = (DataSet)_dataSets.get(j);
		if( d!= null ) {
		  for( int i=0; i< d.size(); i++ ) {
			  ImageIconJ2D image = getImage( d, d.getEntry(i), j );
			  if( image != null ) { image.setMagnification(mag); }
		  }
		}
	  }
	  fireTableDataChanged();
	}
	
	public void setTimeOffset( float time ) { 
	  if( _time_offset != time ) {
		_time_offset = time; 
		fireTableDataChanged();
	  }
	}

	public void setTimeStep( float dt ) { 
	  if( dt != _dt ) {
		_dt = dt; 
		fireTableDataChanged();
	  }
	}
	public float getTimeStep( ) { 
	  return _dt; 
	}
	
	public boolean  intervalAdded(ListDataEvent e) {
	  boolean rv = false;
	  DataSet d = (DataSet) e.getSource();
	  int row = _dataSets.indexOf(d);
	  int index = e.getIndex0();
	  if( (_dt == Float.MAX_VALUE) || ( index < 2 ) ) {
		float dt = d.getTimestep();
		if( dt < _dt ) {
		  _dt = d.getTimestep();
		  if( _dt != Float.MAX_VALUE ) {
			rv = true;
		  }
		}
	  }	
	  fireTableCellUpdated(row, 0); 
	  return rv;
	}
}

