package miiee.dataview;  

import javax.swing.*;
import java.io.*;
import javax.swing.table.*;
import java.util.Vector;
import java.awt.image.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;
import java.applet.Applet;
import javax.help.*;
import miiee.util.*;
import miiee.tree.EditableTextTableCellRenderer;

public class ParameterViewer extends JPanel implements IOController, AppletKernel {

    protected JTable       _table;
    protected ParameterViewerModel _dataModel;
    protected JFrame _frame;
    protected String _name;
    protected int _maxCols;
    
    protected AttributeValuePair _layer;
    protected String _helpText;
    protected SaveFileControlPanel _saveControlPanel;
    protected DataSet _data;
    protected int _currentLayer = 0;
    private float _dt = -1;
    protected Rectangle _boundsRec = new Rectangle();
    protected int _labelColWidth = 200;
    protected int _dataColWidth = 150;
    protected String _labelColHeader;
    protected String _dataColHeader;
    protected static final Color _light_yellow = new Color(255,255,200);
    
   /**
      * Constructs a new instance of ParameterViewer.
      */
    public ParameterViewer( DataSet d, int maxCols ) {
		_data = d;
		_name = d.toString();
		generateHelpText();
		_maxCols = maxCols;
		d.addEntryDataListener( this );
    }
		
	public JFrame showTable() {
	  JFrame frame = new JFrame("Parameter Viewer");
	  WindowListener l = new WindowAdapter() {
		  public void windowClosing(WindowEvent e) { destroy(); }
	  };
	  frame.addWindowListener(l);
            
	  setFrame(frame);
	  init();

	  frame.getContentPane().add(this);
	  frame.setJMenuBar( constructMenuBar() );
	  frame.pack();
	  
 	  Dimension         paneSize = _frame.getSize();
	  Dimension         screenSize = _frame.getToolkit().getScreenSize();
	  _frame.setLocation((screenSize.width - paneSize.width) / 3,
				  (screenSize.height - paneSize.height) / 3);
				  
	  frame.setVisible(true);
	  return frame;
	}
		
	public int getRowCount() { return _dataModel.getRowCount(); }
	public int getColumnCount() { return _dataModel.getColumnCount(); }
	
    public synchronized void init() {   
		_dataModel = new ParameterViewerModel(_data,_maxCols);
		_data.addEntryDataListener(this);
		_table = new JTable(_dataModel);
		formatColumns(); 
		JScrollPane scrollpane = new JScrollPane(_table);
		scrollpane.setPreferredSize(new Dimension(400, 500));
		scrollpane.setBorder( BorderFactory.createLoweredBevelBorder() );
		
		setLayout(new BorderLayout());
		add( "Center", scrollpane );
		add( "South", constructOptionsPanel());				
    }

	public  DataSet getDataSet() { return _data; }
	
	synchronized public void intervalAdded(ListDataEvent e) {
	  _dataModel.intervalAdded(e);
	}
    public void intervalRemoved(ListDataEvent e) {
	  _dataModel.intervalRemoved(e);
	}
    public void contentsChanged(ListDataEvent e) {
	  _dataModel.contentsChanged(e);
	}
	
    public void addTableModelListener(TableModelListener l) {
	  _dataModel.addTableModelListener(l);
    }
     public void saveAsASCII() {
	  SimulationData sd = DataSet.getCurrentSimulation();
	  String fileName = JOptionPane.showInputDialog("File name:"); 
	  String path = sd.getProjPath();
	  BufferedWriter out = null;
	  File ofile = null;
	  try {
		ofile = new File( path, fileName );
		FileOutputStream fos =  new FileOutputStream( ofile );
		out = new BufferedWriter(new OutputStreamWriter(fos));
		SimIO.print( "Saving data to file: " + ofile.getAbsolutePath() ); 
	  } catch ( Exception e0 ) {
		SimIO.show_error( null, "Error opening file stream: " + ofile.getAbsolutePath() ); 
		SimIO.print(e0.getMessage());
		return;
	  } 

	  try {
		int rows = getRowCount();
		int cols = getColumnCount();
		for ( int ir=0; ir<rows; ir++ ) {
		  out.newLine(); 
		  for ( int ic=1; ic<cols; ic++ ) {
			Object obj = _dataModel.getValueAt( ir, ic );
			out.write( obj.toString() + " " );
		  }
		}
	  } catch ( Exception e1 ) {
		SimIO.show_error( null, "Error writing data: ", e1 ); 
	  } 
	  try {
		out.close();
	  } catch ( Exception e2 ) {;}
	}
	
  private JPanel constructOptionsPanel() {
	JPanel           borderPane = new JPanel(false);  
	JButton aButton;

	int dim = _data.dims();
	borderPane.setLayout(new GridLayout(dim,1));
	
	if( dim > 0 ) {
	  JPanel   offsetsPanel = new JPanel(false);
	  offsetsPanel.setLayout(new FlowLayout());
	  borderPane.add(offsetsPanel);
	}	
	return borderPane;
  }
	
    /** Construct a menu. */

    private JMenuBar constructMenuBar() {
	JMenu            menu;
	JMenuBar         menuBar = new JMenuBar();
	JMenuItem        menuItem;

	menu = new JMenu("File");
	menu.setBackground(Color.lightGray);
	menuBar.add(menu);

	JMenu save_menu = new JMenu("Save");
	menuBar.add(save_menu);

	menuItem = save_menu.add(new JMenuItem("binary"));
	menuItem.addActionListener(new SaveAction(this));

	menuItem = save_menu.add(new JMenuItem("ASCII"));
	menuItem.addActionListener(new SaveASCIIAction());

	menuItem = menu.add(new JMenuItem("Quit"));
	menuItem.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		   stop();
		   destroy();
		   if( _frame != null ) { _frame.dispose(); }
	    }});

	menu = new JMenu("View");
	menuBar.add(menu);

//	menuItem = menu.add(new JMenuItem("Manual rescale"));
//	menuItem.addActionListener(new ManualScaleAction());

	menuItem = menu.add(new JMenuItem("1"));
//	menuItem.addActionListener(new ReloadAction());

	menuItem = menu.add(new JMenuItem("2"));
//	menuItem.addActionListener(new RemoveAction());

	menu = new JMenu("Help");
	menuBar.add(menu);

	try {
	  menuItem = menu.add(new JMenuItem("Show ParameterViewer help"));
	  HelpBroker hb = SimIO.getHelpBroker( "SME", "VparameterViewer"  );
	  menuItem.addActionListener( new CSH.DisplayHelpFromSource(hb)  );
	} catch( Exception err ) {;}

	return menuBar;
    }

    class SaveAction extends Object implements ActionListener {
	  IOController _ac;
	  public SaveAction( IOController ac ) { super(); _ac = ac; }
	  public void actionPerformed(ActionEvent e) {
		SimulationData sd = DataSet.getCurrentSimulation();
		_saveControlPanel = SaveFileControlPanel.showControlPanel( _ac, _name, sd.getProjPath(), null );
	  }
    } 

    class SaveASCIIAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) { saveAsASCII(); }
    } 
       
	void generateHelpText() {
	  _helpText = 
		"  Displays raw data form DataSets: \n"
	  + "  1D: Displays segment of timeseries, use time offset to move forward in series. \n"
	  + "  2D: Displays array of data from map, use row/col offsets to set location of array in map. \n"
	  + "  3D: Displays array of data from one layer of (3D)map, use Layer buttons to change layer. \n"
	  + "  Use Frame buttons to move between entries in DataSet.  ";
	}
	    
	void print(String msg) {
	  System.out.print("\nJAVA: " + msg);
	}

    public void start() { repaint(); }   
    public void initDemo() { ; }
    public void stop() { ; }
    
    public void destroy() { 
	  _table = null;
	}
	
    public void dispose() {  
	  if( _frame != null )  {
		_frame.setVisible(false);
		_frame.dispose();
	  }
	}
	
    public void setFrame(JFrame f) {  _frame = f; }
	public Container getContentPane() { return this; }


   public void saveData( String dir, String baseName, SimulationData data ) {;} 

   public void saveData( String dir, String baseName, int format, boolean saveAllFrames ) {
   /*
	 try {
	   if( saveAllFrames ) {
		 switch( format ) {
//		   case 0: _table.SaveAnimationPPM(dir,baseName); break;
//		   case 2: _table.SaveAnimationJPEG(dir,baseName); break;
		 }
	   }
	   else {
		 switch( format ) {
//		   case 0: _table.SaveImagePPM(-1,dir,baseName); break;
//		   case 2: _table.SaveImageJPEG(-1,dir,baseName); break;
		 }
	   }
	  } catch ( java.io.IOException err ) {
		SimIO.print( "File write exception: " + err.getMessage() );
	  }
*/
   } 

	public static void main(String args[])  {
	  DataSet ds = DataSet.create1DDemo( 50, 5 ); 
	  ParameterViewer P = new ParameterViewer( ds, 100 );  
	  P.showTable();	  
	}
	
	public void setColumnWidths(  int labelColWidth,  int dataColWidth) {
	  _labelColWidth = labelColWidth;
	  _dataColWidth = dataColWidth;
	}

	public void setColumnHeaders(  String labelColHeader,  String dataColHeader ) {
	  _labelColHeader = labelColHeader;
	  _dataColHeader = dataColHeader;
	}
	
	public void formatColumns() {
		_table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		_table.setAutoCreateColumnsFromModel(false); 
        TableColumnModel cm = _table.getColumnModel();
        DefaultTableCellRenderer indexColumnRenderer = new DefaultTableCellRenderer(){
		  public void setValue(Object value) {
			String val = (String)value;
			if( val != null ) {
			  if( val.regionMatches(false,0,"**",0,2) ) { 
				setBackground( Color.black ); setForeground( Color.white ); 
			  } else {
				setBackground( _light_yellow ); setForeground( Color.black ); 
			  }
			  setText( val ); 
			} else {
			  setText( "" );
			}
		  }
        };
		indexColumnRenderer.setHorizontalAlignment(JLabel.LEFT);
        DefaultTableCellRenderer dataColumnRenderer = new DefaultTableCellRenderer() {
		  public void setValue(Object value) {
			FormattedFloat ff = (FormattedFloat)value;
			if( value != null ) {
			  if( ff.isNaN() ) { setBackground( Color.black ); setText( "" ); }
			  else if( ff.isINF() ) { setBackground( Color.red ); setText( "INF" ); }
			  else { setBackground(Color.white); setText( ff.toJavaFormatString(4) ); }
	        } else {
			  setBackground( Color.black ); setText( "" );
			}
		  }
        };
		dataColumnRenderer.setForeground( Color.black );
        dataColumnRenderer.setHorizontalAlignment(JLabel.CENTER);
		int dim = _data.dims();
		for( int columnIndex = 0; columnIndex < _dataModel.getColumnCount(); columnIndex++ ) {
		  TableColumn tableColumn =  cm.getColumn(columnIndex);
		  if( columnIndex == 0 ) {
			tableColumn.setCellRenderer(indexColumnRenderer); 
			tableColumn.setPreferredWidth(_labelColWidth);
			if( _labelColHeader != null ) {
			  tableColumn.setHeaderRenderer( new DefaultTableCellRenderer() {
				public void setValue(Object value) { 
				  setBackground( Color.cyan ); setText( _labelColHeader ); 
				  setHorizontalAlignment(JLabel.CENTER); setBorder( BorderFactory.createEtchedBorder() ); }
			  });
			}
		  } else {
			tableColumn.setCellRenderer(dataColumnRenderer); 
			JTextField textField = new JTextField();
			tableColumn.setCellEditor( new DefaultCellEditor(textField) ); 
			tableColumn.setPreferredWidth(_dataColWidth); 
			if( _dataColHeader != null ) {
			  tableColumn.setHeaderRenderer( new DefaultTableCellRenderer() {
				public void setValue(Object value) { 
				  setBackground( Color.cyan ); setText( _dataColHeader ); 
				  setHorizontalAlignment(JLabel.CENTER); setBorder( BorderFactory.createEtchedBorder() );
				}
			  });
			}
		  }
		}
    }

}

class ParameterViewerModel extends AbstractTableModel {
	private DataSet _data;
	private int _rows;
	private int _cols = 0;
	private int _maxCols;
		
	ParameterViewerModel( DataSet d, int maxCols ) { 
	  super();
	  _maxCols = maxCols; 
	  _data = d;
	  resize_table();
	}
						
	public String getColumnName(int column) {
	  if( column == 0 ) return "";
	  return "Data " +  column; 
	}

	public int getColumnCount() { 
		return _cols + 1; 
	}
	
	public int getRowCount() { 
		return _rows;
	}
	
	public DataSet getDataSet() { return _data; }
			
	public Object getValueAt( int row, int col ) {
	  DataEntry de = (DataEntry) _data.getEntry(row);
	  if( de == null ) { return null; }
	  
	  if( col == 0 ) {
//		FormattedFloat ft = new FormattedFloat(de.time());
//		String rv = de.toString() + "(" + ft.toJavaFormatString(2) + ")";
		return de.toString();
	  } else { 
		int index = ( de.dataSize() - 1 )  - ( _cols - col );
		if( index < 0 ) return null;
		return new FormattedFloat( de.getValue(index) );
	  }
	}

    public boolean isCellEditable(int rowIndex, int columnIndex) {
	  return ( columnIndex > 0 );
    }

    public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
	  String sval = aValue.toString();
	  DataEntry de = (DataEntry) _data.getEntry(rowIndex);
	  if( de == null ) { return; }
	  try { 
		float fval = Float.valueOf(sval).floatValue();
		de.setValue( 0, fval );
		fireTableCellUpdated(rowIndex,columnIndex);
	  } catch ( NumberFormatException err ) { SimIO.print("Number format exception: " + sval); }
    }
	
	public Class getColumnClass(int columnIndex) {
	  return  (columnIndex == 0 ) ? String.class : FormattedFloat.class;
	}
	
    synchronized public void intervalAdded(ListDataEvent e) {
	  resize_table();
	}
	public void intervalRemoved(ListDataEvent e) {
	  resize_table();
	}
    public void contentsChanged(ListDataEvent e) {
  		fireTableDataChanged();
	}

	void resize_table() {
	  int cols = _cols;
	  int rows = _rows;
	  _rows = _data.size();
	  for( int i=0; i<_data.size(); i++ ) {
		DataEntry de = (DataEntry) _data.getEntry(i);
		if( de.dataSize() > _cols ) _cols = de.dataSize();	  
	  }
	  if( _cols > _maxCols ) _cols = _maxCols;
	  if( (cols != _cols) || (rows != _rows) ) {
		fireTableStructureChanged();
	  }
	}
}

