package miiee.util;  

import javax.swing.*;
import java.io.*;
import javax.swing.table.*;
import java.util.Vector;
import java.awt.image.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;
import java.applet.Applet;
import javax.help.*;
import miiee.util.*;

public class ColorTableEditor extends JPanel implements ListSelectionListener {

    protected JTable       _table;
    protected ColorTableModel _dataModel;
    protected ColorMap _colormap;
    protected JFrame _frame;
    protected String _name;
    protected JButton _colorSelectionButton = new JButton("    ");
    
    protected String _helpText = "Double click on a color cell to change that color. Clicking the 'fill' button\n"
								+ " will fill the selected rows with the current fill color (shown next to \n"
								+ " the fill button).  Click on the fill color to change it.  The clear button \n"
								+ " fills the table with black. ";
    
   /**
      * Constructs a new instance of DataTable.
      */
    public ColorTableEditor( ColorMap cm ) {
		_colormap = cm;
    }

		
	public static ColorTableEditor getTable( ColorMap cm ) {
	  ColorTableEditor d = new ColorTableEditor(cm);
	  
	  JFrame frame = new JFrame("Color Table Editor");
//	  frame.addWindowListener(new myWindowAdapter(cm));
	  d.setFrame(frame);
	  
	  d.init();
	  frame.getContentPane().add(d);
	  frame.setJMenuBar( d.constructMenuBar() );
	  frame.setBackground(Color.lightGray);	
	  frame.pack();
	  frame.setVisible(true);
	  
	  return d;
	}
		
	public int getRowCount() { return _dataModel.getRowCount(); }
	public int getColumnCount() { return _dataModel.getColumnCount(); }
    	
    public synchronized void init() {   
		_dataModel = new ColorTableModel( _colormap.getColorMap() );
		_table = new JTable(_dataModel);
		formatColumns(); 
		JScrollPane scrollpane = new JScrollPane(_table);
		scrollpane.setPreferredSize(new Dimension(150, 500));
		scrollpane.setBorder( BorderFactory.createLoweredBevelBorder() );
		
		setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));

		add( scrollpane );
		add( constructFillPanel() );				
		add( constructOptionsPanel() );				
		
    }
  

    public void addTableModelListener(TableModelListener l) {
	  _dataModel.addTableModelListener(l);
    }
    
    void setFillColor( Color c ) {
		_colorSelectionButton.setBackground(c);
	}
  
    Color getFillColor() {
		return _colorSelectionButton.getBackground();
	}
	
	public void valueChanged(ListSelectionEvent e) {
//		Color c = getSelectedColor();
//		setFillColor( c );
	}
   
  private JPanel constructOptionsPanel() {
	  JButton aButton;

	  JPanel   offsetsPanel = new JPanel(false);
	  offsetsPanel.setLayout(new FlowLayout());

	  aButton = new JButton("Set");
	  aButton.addActionListener( new SetColormapAction() );
	  offsetsPanel.add( aButton );

	  aButton = new JButton("Clear");
	  aButton.addActionListener( new ClearAction() );
	  offsetsPanel.add( aButton );

	  aButton = new JButton("Close");
	  aButton.addActionListener( new CloseAction() );
	  offsetsPanel.add( aButton );
	
	  return offsetsPanel;
    }

  private JPanel constructFillPanel() {
	  JButton aButton;

	  JPanel   offsetsPanel = new JPanel(false);
	  offsetsPanel.setLayout(new FlowLayout());

	  aButton = new JButton("Fill");
	  aButton.addActionListener( new FillAction() );
	  offsetsPanel.add( aButton );

	  offsetsPanel.add( new JLabel("Fill Color:") );

	  _colorSelectionButton.addActionListener( new SelectFillColorAction() );
	  offsetsPanel.add( _colorSelectionButton );
	
	  return offsetsPanel;
    }
	
    /** Construct a menu. */

    private JMenuBar constructMenuBar() {
	JMenu            menu;
	JMenuBar         menuBar = new JMenuBar();
	JMenuItem        menuItem;

	menu = new JMenu("File");
	menu.setBackground(Color.lightGray);
	menuBar.add(menu);

	menuItem = menu.add(new JMenuItem("Save"));
	menuItem.addActionListener(new SaveAction());

	menuItem = menu.add(new JMenuItem("Quit"));
	menuItem.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		   stop();
		   destroy();
		   if( _frame != null ) { _frame.dispose(); }
	    }});

	menu = new JMenu("View");
	menuBar.add(menu);

//	menuItem = menu.add(new JMenuItem("Manual rescale"));
//	menuItem.addActionListener(new ManualScaleAction());

	menuItem = menu.add(new JMenuItem("1"));
//	menuItem.addActionListener(new ReloadAction());

	menuItem = menu.add(new JMenuItem("2"));
//	menuItem.addActionListener(new RemoveAction());

	menu = new JMenu("Help");
	menuBar.add(menu);
	menuItem = menu.add(new JMenuItem(" Color Editor Help "));
	menuItem.addActionListener(new ShowHelpTextAction());

	try {
	  menuItem = menu.add(new JMenuItem("SME help"));
	  HelpBroker hb = SimIO.getHelpBroker( "SME", "ColorTableEditor"  );
	  menuItem.addActionListener( new CSH.DisplayHelpFromSource(hb)  );
	} catch( Exception err ) {;}

	return menuBar;
    }

	class SaveAction extends Object implements ActionListener  {
	  public void actionPerformed(ActionEvent e) {
		String s = JOptionPane.showInputDialog(" Please give this ColorMap a name: ");
		if( s != null ) {
		  String file = System.getProperty("user.home") + "/.sme/colormaps/" + s + ".cmap";  
		  try {
			_colormap.write(file);
		  } catch( IOException err ) {
			SimIO.show_error( null, "Colormap write error:", err );
		  }
		}
	  }
    } 		

    class SetColormapAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) {  
		float[][]  data = _dataModel.getColorData();
		_colormap.setValues(data);  
	  }
    } 

    class ClearAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) {  
		_dataModel.fill(Color.black);
	  }
    } 

    class FillAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) { 
		 int[] rows = _table.getSelectedRows();
		 Color c = getFillColor();
		_dataModel.fill(c,rows);
	  }
    } 
    
    class SelectFillColorAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) { 
		 Color initialColor = getFillColor();
		 Color c = JColorChooser.showDialog( _frame.getContentPane(), "Color Chooser", initialColor );
		setFillColor(c);
	  }
    } 

    class CloseAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) {  
		dispose();
	  }
    } 

    class ShowHelpTextAction extends Object implements ActionListener {
	  public void actionPerformed(ActionEvent e) {  
		JDialog dialog = new JDialog(_frame,"Help Text");
		JTextArea text = new JTextArea(_helpText);
		dialog.getContentPane().add(text);
		dialog.pack();
		dialog.setVisible(true);
	  }
    } 
    
    public Color getSelectedColor() {
 		 int row = _table.getSelectedRow();
		return _dataModel.getColor(row);   
    }

	void print(String msg) {
	  System.out.print("\nJAVA: " + msg);
	}

    public void start() { repaint(); }   
    public void initDemo() { ; }
    public void stop() { ; }
    
    public void destroy() { 
	  _table = null;
	}
	
    public void dispose() {  
	  if( _frame != null )  {
		_frame.setVisible(false);
		_frame.dispose();
	  }
	}
	
    public void setFrame(JFrame f) {  _frame = f; }
	public Container getContentPane() { return this; }  
	public void intervalAdded(ListDataEvent e) {;}
    public void intervalRemoved(ListDataEvent e) {;}
    public void contentsChanged(ListDataEvent e) {;}


	public static void main(String args[])  {
	  ColorTableEditor dt = getTable( new RGBMap() );	  
	}
	
	public void formatColumns() {
	
		_table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		_table.setAutoCreateColumnsFromModel(false); 
		_table.setRowSelectionAllowed(true);
		_table.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		ListSelectionModel lm = _table.getSelectionModel();
		lm.addListSelectionListener(this);
        TableColumnModel cm = _table.getColumnModel();
        DefaultTableCellRenderer indexColumnRenderer = new DefaultTableCellRenderer();
        indexColumnRenderer.setBackground(Color.lightGray);
        indexColumnRenderer.setHorizontalAlignment(JLabel.CENTER);

        DefaultTableCellRenderer colorColumnRenderer = new DefaultTableCellRenderer() {
		  public void setValue(Object value) {
			setBackground( (Color)value ); 
		  }
        };
		ColorCellEditor color_editor = new ColorCellEditor( colorColumnRenderer, _frame.getContentPane() );
		color_editor.setClickCountToStart(2);
		
		for( int columnIndex = 0; columnIndex < _dataModel.getColumnCount(); columnIndex++ ) {
		  TableColumn tableColumn =  cm.getColumn(columnIndex);
		  if( columnIndex == 0 ) {
			tableColumn.setCellRenderer(indexColumnRenderer); 
			tableColumn.setWidth(60); 
		  } else {
			tableColumn.setCellRenderer(colorColumnRenderer); 
			tableColumn.setCellEditor( color_editor );
			tableColumn.setWidth(60); 
		  } 
		}
    }
    
    class ColorCellEditor extends AbstractCellEditor {
		DefaultTableCellRenderer _renderer;
		Component _component;

		public ColorCellEditor( DefaultTableCellRenderer renderer, Component c ) {
			_renderer = renderer;
			_component = c;
			_renderer.addMouseListener(new MouseAdapter() {
				public void mousePressed(MouseEvent e) {
					SwingUtilities.invokeLater(new Runnable() {
						public void run() {
							Color initialColor = getSelectedColor(); 
							Color c1 = JColorChooser.showDialog(_component, "Color Chooser", initialColor);
							if( c1 != null ) { 
								setCellEditorValue(c1); 
								setFillColor( c1 );
							}
							fireEditingStopped();
						}
					});
				}
			});
		}
		public Component getTableCellEditorComponent( JTable table, Object value, boolean isSelected, int row, int col) {
			return _renderer.getTableCellRendererComponent( table, value, true, true, row, col);
		}
	}	
}

class ColorTableModel extends AbstractTableModel {
	private float[][] _val;
	int _rows;
	int _cols;
	int _offset;
	
	ColorTableModel( float[][] val ) {
	  super();
	  _rows = 256;
	  _cols = 2;
	  if( val == null ) { _val = new float[_rows][_cols]; }
	  else { _val = val; }
	}
				
	public String getColumnName(int column) {
	  String name = "";
	  switch( column ) {
		case 0: name = "Index"; break;
		case 1: name =  "Color"; break;
	  }
	  return name;
	}
	
	public int getColumnCount() { 
		return _cols; 
	}
	public int getRowCount() { 
		return _rows;
	}	
	
	public Object getValueAt(int row, int col) { 
		if( col == 0 ) {
		  return new Integer( row ); 
		} else {
		  return new Color(_val[row][0],_val[row][1],_val[row][2]);
		}
	}

    public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
	  if( columnIndex == 1 ) {
		  Color c = (Color)aValue;
		  c.getColorComponents(_val[rowIndex]);
		  fireTableCellUpdated(rowIndex,columnIndex);
	  }
    }
	
    public boolean isCellEditable(int rowIndex, int columnIndex) {
	  return ( columnIndex == 1 );
    }
    
    public void fill( Color c ) {
		if( c == null ) { SimIO.beep(); return; }
		for( int i=0; i<_rows; i++ ) { c.getColorComponents(_val[i]); }
		fireTableDataChanged();
    }

    public void fill( int row ) {
		if( row < 0 || ( row > 255 ) ) { SimIO.beep(); return; }
		for( int i=0; i<_rows; i++ ) { 
			for( int j=0; j<3; j++ ) { _val[i][j] = _val[row][j]; }
		}
		fireTableDataChanged();
    }

    public void fill( Color c, int[] rows ) {
		if( c == null ) { SimIO.beep(); return; }
		for ( int i=0; i<rows.length; i++ ) {
			int row = rows[i];
			if( row >= 0 && ( row <= 255 ) ) { c.getColorComponents(_val[row]); }
		}
		fireTableDataChanged();
    }
	

	public Class getColumnClass(int columnIndex) {
	  if( columnIndex == 0 ) { return Integer.class; }
	  return Color.class;
	}
	
	public float[][]  getColorData() { return _val; }
	
	public Color getColor( int row ) {
		if( ( row < 0 ) || ( row > 255 )  ) { SimIO.beep(); return null; }
		return (Color) getValueAt( row, 1 );
	}
	
}
