/*
This class was borrowed from VisAD (http://www.ssec.wisc.edu/~billh/visad.html).  
Thanks, Bill & Nick;  Great work!
*/

package miiee.util;

import javax.swing.JComponent;
import java.awt.event.*;
import java.awt.*;
import java.util.Vector;
import java.io.*;

import javax.swing.*;

/** 
 * A color widget that allows users to interactively map numeric data to
 * RGBA tuples based on the Vis5D color widget
 *
 * @author Nick Rasmussen nick@cae.wisc.edu
 * @version $Revision: 1.2 $, $Date: 2001/01/22 20:48:20 $
 * @since Visad Utility Library, 0.5
 */

public class ColorWidget extends JComponent implements ColorChangeListener {

	/** The visibility of the preview panel at the botom of the widget */
	private boolean previewVisible;
	
	/** The ColorMap associsted with this color widget */
	private ColorMap map;
	
	/** The ColorPreview associated with this color widget */
	private ColorPreview colorPreview;
		
	private	JMenu   _fileReadMenu;
	private Vector _colormaps;
	
	private static JFrame _frame;
	
	/** Construct a color widget with a ColorPreview and the default ColorMap */
	public ColorWidget() {
		this(true);
	}
	
	/** Construct a color widget with the default ColorMap
	 * @param preview indicates wether or not the preview bar at the
	 * bottom of the widget should be present
	 */
	public ColorWidget(boolean preview) {
		this(new RGBMap(), preview);
	}

	/** Construct a color widget with a ColorPreview and the specified ColorMap	
	 * @param map the ColorMap for the widget to use
	 */
	public ColorWidget(ColorMap map) {
		this(map, true);
	}
	
	/** Construct a color widget with the desired ColorMap and ColorPreview visibility
	 * @param map the ColorMap for the widget to use
	 * @param preview indicates wether or not the preview bar at the
	 * bottom of the widget should be present
	 */
	public ColorWidget(ColorMap map, boolean preview) {
		previewVisible = preview;
		if (preview) {
			colorPreview = new ColorPreview(this);
		}
		//setLayout(new WidgetLayout(this));
		setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
		setColorMap(map);
	}	
	
	/** main method for standalone testing */
	public static void main(String[] argv) { 
	  display( false );
	}
		
	public static ColorWidget display( boolean hasAlpha ) {
		if( _frame != null ) {
		   _frame.setVisible(false);
		   _frame.dispose();	   
		}
		_frame = new JFrame("VisAD Color Widget");
		_frame.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) { ;}
		});
		ColorWidget w = new ColorWidget( (hasAlpha) ? (ColorMap) (new RGBAMap()) : (ColorMap) (new RGBMap()) );

		_frame.setJMenuBar( w.constructMenuBar() );
		w.getSavedColorMaps();		
		_frame.setContentPane(w);
		_frame.setSize(w.getWidgetSize());
		_frame.setVisible(true);	
		return w;		
	}
	
	void addImportFile( File f ) {
	  String name = f.getName();
	  JMenuItem menuItem = _fileReadMenu.add( new JMenuItem( name.substring( 0, name.length()-5 ) ) );
	  menuItem.addActionListener( new RetrieveAction(f) );
	  _colormaps.addElement(f);
	}

  private JMenuBar constructMenuBar() {
	JMenuBar         menuBar = new JMenuBar();
	JMenuItem        menuItem;
	
	JMenu fileMenu = new JMenu("File");
	menuBar.add(fileMenu);
	menuItem = fileMenu.add(new JMenuItem("Save"));
	menuItem.addActionListener(new SaveAction());

	menuItem = fileMenu.add(new JMenuItem("Close"));
	menuItem.addActionListener(  new ActionListener() {
	  public void actionPerformed(ActionEvent e) { _frame.dispose(); }  
	}  );
	
	_fileReadMenu = new JMenu("Import");
	menuBar.add(_fileReadMenu);

	JMenu editMenu = new JMenu("Edit");
	menuBar.add(editMenu);
	menuItem = editMenu.add(new JMenuItem("Data Table"));
	menuItem.addActionListener(new DataTableAction());
	
	return menuBar;
  }
   
   public void getSavedColorMaps() { 
	if( _colormaps == null ) {
		File dir = new File( System.getProperty("user.home") + "/.sme/colormaps/" );
		_colormaps =  new Vector();
		String[] names = dir.list();

		int nameCount = names == null ? 0 : names.length;	 		   		  
		for (int i = 0; i < nameCount; i++) {
		  if( names[i].endsWith(".cmap")  ) {
			File f = new File( dir.getPath(), names[i] );
			if ( f.isFile() ) {
			  addImportFile( f ); 
			}
		  }
		}	  
		System.out.println("Getting files for " + dir + ", file count = " + nameCount + ", cmap count = " + _colormaps.size()  );
	  }
    }

	class SaveAction extends Object implements ActionListener  {
	  public void actionPerformed(ActionEvent e) {
		String s = JOptionPane.showInputDialog(" Please give this ColorMap a name: ");
		if( s != null ) {
		  String file = System.getProperty("user.home") + "/.sme/colormaps/" + s + ".cmap";  
		  try {
			map.write(file);
			addImportFile(  new File(file) );
		  } catch( IOException err ) {
			SimIO.show_error( null, "Colormap write error:", err );
		  }
		}
	  }
    } 		


	class RetrieveAction extends Object implements ActionListener  {
	  File _file;
	  public RetrieveAction( File f ) { _file = f; }
	  public void actionPerformed(ActionEvent e) {
		try {
		   map.read( new FileInputStream( _file ) );
		   map.repaint();
		} catch ( FileNotFoundException err ) {
		    SimIO.show_error( null, "Can't find Colormap:", err );   
		} catch ( IOException err1 ) {
		    SimIO.show_error( null, "Colormap read error: ", err1 );   
		}
	  }
    } 		
    
	class DataTableAction extends Object implements ActionListener  {
	  public void actionPerformed(ActionEvent e) {
		ColorTableEditor ct = ColorTableEditor.getTable( map );
	  }
    } 		
	/** The vector containing the ColorChangeListeners */
	private Vector listeners = new Vector();
	
	/** Add a ColorChangeListener to the listeners list */
	public synchronized void addColorChangeListener(ColorChangeListener c) {
		if (!listeners.contains(c)) {
			listeners.addElement(c);
		}
	}
	
	/** Remove a ColorChangeListener from the listeners list */
	public synchronized void removeColorChangeListener(ColorChangeListener c) {
		if (listeners.contains(c)) {
			listeners.removeElement(c);
		}
	}
	
	/** Notify the ColorChangeListerers that the color widget has changed */
	protected synchronized void notifyListeners(ColorChangeEvent e) {
		for (int i = 0; i < listeners.size(); i++) {
			ColorChangeListener c = (ColorChangeListener) listeners.elementAt(i);
			c.colorChanged(e);
		}
	}
	
	/** Listen to the ColorMap and re-dispatch the ColorChangeEvents to
	 * the ColorChangeListeners listening to the widget
	 */
	public void colorChanged(ColorChangeEvent e) {
		notifyListeners(e);
	}
	
	/** Set the ColorWidget to listen to a specific ColorMap */
	public void setColorMap(ColorMap map) {
		if (this.map != null) {
			this.map.removeColorChangeListener(this);
		}
		
		this.map = map;
		
		map.addColorChangeListener(this);
	
		removeAll();
		add(map);
		if (previewVisible) {
			if (colorPreview == null) {
				colorPreview = new ColorPreview(this);
			}
			add(colorPreview);
		}
		
	}
		
	/** Make the preview bar at the bottom of the widget visible */
	public void showPreview() {
	
		if (previewVisible) return;	
		previewVisible = true;
		this.setColorMap(map);
	}
	
	/** Hide the preview bar at the bottom of the widget */
	public void hidePreview() {
		
		if (!previewVisible) return;
		
		previewVisible = false;
		this.setColorMap(map);
	}
	
	/** Returns the ColorMap that the color wdget is curently pointing to */
	public ColorMap getColorMap() {
		return map;
	}
	
	/** Analyses the visible components and determines the preferred size */
	public Dimension getWidgetSize() {
		Dimension d = map.getPreferredSize();
		if (previewVisible) {
			Dimension p = colorPreview.getPreferredSize();
			Dimension n = new Dimension(d.width, d.height + p.height);
			d = n;
		}
		return d;
	}
	
}
