package miiee.util;

import java.util.Vector;
import java.awt.Toolkit;
import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.*;
import java.util.ListIterator;
import java.awt.Component;

class SimulationDataTree {

	protected JTree _simtree;
    protected DefaultTreeModel _treeModel;
  	protected RepairedMutableTreeNode _top_simnode = new RepairedMutableTreeNode("SimRuns");
  	protected RepairedMutableTreeNode _currentSimulation;
	protected int simulation_index = 0;
    private Toolkit _toolkit = Toolkit.getDefaultToolkit();
	private static SimulationDataTree _dataTree = new SimulationDataTree();
    
	public SimulationDataTree() {
	  _treeModel = new DefaultTreeModel(_top_simnode);
	  _simtree = new JTree(_treeModel);
	  _simtree.setCellRenderer( new DataTreeCellRenderer() );  
	  _simtree.getSelectionModel().setSelectionMode( TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION );
	}
	
	public JTree getTree() { return _simtree; }
	public RepairedMutableTreeNode getRoot() { return _top_simnode; }
	
	public static SimulationDataTree get() { return _dataTree; }

    public void clear() {
        _top_simnode.removeAllChildren();
        _treeModel.reload();
    }
    
     public void  addTreeModelListener(TreeModelListener l) {
	   _treeModel.addTreeModelListener(l);
     }

    /** Remove the currently selected node. */
    public void removeCurrentNode( TreePath tp ) {
        TreePath currentSelection = ( tp == null ) ? _simtree.getSelectionPath() : tp;
        if (currentSelection != null) {
            RepairedMutableTreeNode currentNode = (RepairedMutableTreeNode)
                         (currentSelection.getLastPathComponent());
            MutableTreeNode parent = (MutableTreeNode)(currentNode.getParent());
            if (parent != null) {
                _treeModel.removeNodeFromParent(currentNode);
                return;
            }
        } 
        // Either there was no selection, or the root was selected.
        _toolkit.beep();
    }
    
    public void deleteSelectedNodes() {
	  int nSelections = _simtree.getSelectionCount();
	  if( nSelections <= 0 ) return;
	  TreePath[] tps = _simtree.getSelectionPaths(); 
	  for( int i=0; i<nSelections; i++ ) {
		TreePath tp = tps[i]; 
		if( tp.getPathCount() == 3 ) { 
		  RepairedMutableTreeNode tn = (RepairedMutableTreeNode)tp.getLastPathComponent();
		  DataSet d = (DataSet) tn.getUserObject();
		  d.destroyDataListeners();
		  d.clear();
		  deleteDataSet( (DataSetNode) tn ); 
		} else if ( tp.getPathCount() == 2 ) {
		  RepairedMutableTreeNode tn = (RepairedMutableTreeNode)tp.getLastPathComponent();
		  int cc = tn.getChildCount();
		  for( int j=0; j<cc ; j++ ) {
			RepairedMutableTreeNode tnc = (RepairedMutableTreeNode)tn.getChildAt(j);
			DataSet d = (DataSet) tnc.getUserObject();
			d.destroyDataListeners();
			d.clear();
		  }
		  deleteSimulation( tn ); 
		} else {
		  _toolkit.beep();
		  return;
		}
	  }
	}
    
	public RepairedMutableTreeNode addSimulation( String proj_name, String model_name, String proj_path ) {
		SimulationData sd = new SimulationData( proj_name, model_name, proj_path );
        _currentSimulation =  new RepairedMutableTreeNode(sd);
        _treeModel.insertNodeInto( _currentSimulation, _top_simnode, sd.simIndex() );
       return _currentSimulation;
	}

	public RepairedMutableTreeNode appendSimulation( SimulationData sd ) {
        RepairedMutableTreeNode tn =  new RepairedMutableTreeNode(sd);
        int simIndex = _top_simnode.getChildCount();
        _treeModel.insertNodeInto( tn, _top_simnode, simIndex );
        sd.setSimIndex( simIndex );
		for( int i=0; i<sd.size(); i++ ) {
		  DataSet d = (DataSet) sd.get(i);
		  if( d != null && d.isValid() ) {
			addDataSet( d, -1, simIndex );
		  }
		}
       return _currentSimulation;
	}

	public RepairedMutableTreeNode deleteSimulation( RepairedMutableTreeNode sn ) {
		SimulationData sd = (SimulationData) sn.getUserObject();
		AnimationControlPanel.closeControlPanel( sd.simIndex() );
        _treeModel.removeNodeFromParent(sn);
		SimulationData sd1 = new SimulationData( sd.simIndex(), "deleted" );
        RepairedMutableTreeNode dn =  new RepairedMutableTreeNode(sd1);
        _treeModel.insertNodeInto( dn, _top_simnode, sd.simIndex() );
        return dn;
	}
	
	public RepairedMutableTreeNode deleteDataSet( DataSetNode dn ) {
		RepairedMutableTreeNode parent = (RepairedMutableTreeNode) dn.getParent(); 
		DataSet d = (DataSet) dn.getUserObject();
		short dsetindex = d.getDSetNodeIndex();
		SimulationData sd = (SimulationData) parent.getUserObject();
		DataSet dd = new DataSet( d.index(), dsetindex, d.getSimNodeIndex(), "deleted" );
        DataSetNode dn1 =  new DataSetNode(dd);
        _treeModel.removeNodeFromParent(dn);
        _treeModel.insertNodeInto( dn1, parent, dsetindex );
        sd.setElementAt( dd, dsetindex );
        d.clear();
        return dn1;
	}
		
	public boolean addDataSet(  DataSet d, int dsetIndex, int simIndex  ) {
        DataSetNode childNode =    new DataSetNode(d);
        RepairedMutableTreeNode sim_node = (simIndex<0) ? _currentSimulation : 
										  (RepairedMutableTreeNode) _top_simnode.getChildAt(simIndex);
		if( sim_node == null ) {
		  SimIO.print( " Attempt to add DataSet to non-existent Simulation: " + simIndex );
		  return false;
		}
		SimulationData sd = (SimulationData) sim_node.getUserObject();
        int nodeIndex = ( dsetIndex < 0 ) ? sim_node.getChildCount() : dsetIndex;
        _treeModel.insertNodeInto( childNode, sim_node, nodeIndex );
        d.setNodeIndex( (short)nodeIndex, (short) sd.simIndex() );
        if( _simtree.isVisible() ) {
			_simtree.expandPath(new TreePath(sim_node.getPath()));
			_simtree.scrollPathToVisible(new TreePath(childNode.getPath()));
		}
		for( int i=0; i< d.size(); i++ ) {
		  DataEntry de = d.getEntry( i );
		  if( de != null ) { addEntry( childNode, de ); } 
		}
        return true;
   }

   public int addEntry( DataSet d, DataEntry de ) {
	 DataSetNode dset_node = getDataSetNode ( d );
	 return addEntry( dset_node, de );
   }
   
   public int addEntry( DataSetNode dset_node, DataEntry de ) {
		if( dset_node == null ) { return -1; } 
        RepairedMutableTreeNode childNode =    new RepairedMutableTreeNode(de);
        int index = dset_node.getChildCount();
        _treeModel.insertNodeInto( childNode, dset_node, index );
		dset_node.fireEntryAdded( index );
        return index;
   } 
   
   public SimulationData getCurrentSimulation() { 
	 try {
	   return (SimulationData) _currentSimulation.getUserObject() ; 
	 } catch ( NullPointerException err ) {
	   return null;
	 }
   } 
   
   public DataSet getDataSet(int dsetIndex) {
	 DataSet d = null;
	 try {
		d = (DataSet) getCurrentSimulation().getDataSet(dsetIndex);
	 } catch ( NullPointerException err ) {
		SimIO.print( "No simulation exists, can't get dataset"  );	 
	 }
	 return d;
   }

   public DataSetNode getDataSetNode ( DataSet d ) {
	 RepairedMutableTreeNode tn = getSimulationNode( d );
	 int dset_index = d.getDSetNodeIndex();
	 return (DataSetNode) tn.getChildAt(dset_index);
   }

   public RepairedMutableTreeNode getSimulationNode ( DataSet d ) {
	 int sim_index = d.getSimNodeIndex();
	 return (RepairedMutableTreeNode) _top_simnode.getChildAt(sim_index);
   }


   public ListIterator getDatasetIterator() {
	 ListIterator l = null;
	 try {
	  l = getCurrentSimulation().getDatasetIterator();
	 } catch ( NullPointerException err ) {
		SimIO.print( "No simulation exists, can't get DatasetIterator"  );	 	 
	 }
	 return l;
   }   
}

class DataSetNode extends RepairedMutableTreeNode {
	protected EventListenerList _listenerList = new EventListenerList();

	public DataSetNode( DataSet d ) { super(d); }
	
	public void addEntryDataListener( DisposableListDataListener l ) {
	  _listenerList.add(ListDataListener.class, l);
	}
	public void removeEntryDataListener( DisposableListDataListener l ) {
		_listenerList.remove(ListDataListener.class, l);
	}

	public void destroyDataListeners() {
	  Object[] listeners = _listenerList.getListenerList();

	  for (int i = listeners.length - 2; i >= 0; i -= 2) {
		  if (listeners[i] == ListDataListener.class) {
			DisposableListDataListener ldl = (DisposableListDataListener)listeners[i+1];
			ldl.dispose();
			removeEntryDataListener( ldl );
		  }	       
	  }
	}

	void fireEntryAdded( int index ) {
	  Object[] listeners = _listenerList.getListenerList();
	  ListDataEvent e = null;

	  for (int i = listeners.length - 2; i >= 0; i -= 2) {
		  if (listeners[i] == ListDataListener.class) {
			if (e == null) {
				e = new ListDataEvent( getUserObject(), ListDataEvent.INTERVAL_ADDED, index, index );
			}
			((ListDataListener)listeners[i+1]).intervalAdded(e);
		  }	       
	  }
	}
	
}

public	class  DataSet extends Vector {  
		
		protected String _info;
		protected String _cats;
		protected byte _dims;
		protected byte _arrayDimension = 0;
		protected byte _format;
		protected byte _mode=0;
		protected byte[] _map;
		protected int _index;
		protected int _L0 = 0;
		protected int _L1 = 0;
		protected int _L2 = 0;
		protected int _current_index = 0;
		protected int _data_size = 0;
		protected short _sim_index;
		protected short _node_index;
		protected final static byte INVALID_FORMAT = (byte)255;
//		protected final static int _dset_index_offset = 256;
//		protected static int _dset_index_count = 0;

		
		private  DataSet( int dsetIndex, int L0, int L1, int L2, byte dims, String info, String cats, byte[] map, byte format, byte mode )  {
		   super(16,16);
		   _info = info;
		   _index = dsetIndex;
		   _dims = dims;
		   _map = map;
		   _L0 = L0;
		   _L1 = L1;
		   _L2 = L2;
		   _format = format;
		   _mode = mode;
		}

		public  DataSet( int L0, int L1, int L2, byte dims, String info, String cats, byte[] map, byte format, byte mode )  {
		  this( -1, L0, L1, L2, dims, info, cats, map, format, mode );
		}

		public  DataSet( int dsetIndex, short node_index, short sim_index, String info ) {
		   super(0,16);
		   _info = info;
		   _index = dsetIndex;
		   _node_index = node_index;
		   _sim_index = sim_index;
		   _format = INVALID_FORMAT;
		}
		
		public int getArrayIndex(int i0, int i1, int i2 ) {
		  switch( _arrayDimension ) {
			case 1: return i0;
			case 2: return i0 + i1*_L0;
			case 3: return i0 + i1*_L0 + i2*_L0*_L1;
		  }
		  return 0;
		}
		
		public void setInfo( String info ) { _info = info; }
		
		public boolean isValid() {
		  return ( _format != INVALID_FORMAT );
		}
		
		public void setArrayDimension( byte adim ) {
		  _arrayDimension = adim;
		}
		
		public int getArrayDimension() {
		  return _arrayDimension;
		}
		
		public boolean isCommensurate( DataSet d1 ) {
		  return ( (_format == d1._format) && (_L0 == d1._L0) && (_L1 == d1._L1) && 
		  (_L2 == d1._L2) && (_dims == d1._dims) );
		}
		
		public int size( int dim ) {
		  switch( dim ) {
			case 0: return _L0;
			case 1: return _L1;
			case 2: return _L2;
		  }
		  return -1;
		}
		
		public int index() { return _index; }
		public byte browse() { return _mode; }
		public boolean edit() { return _mode > 1; }
		public byte[] getActiveRegion() { return _map; }
		public byte dims() { return _dims; }
		public byte format() { return _format; }
		public void setSelection( int index ) { _current_index = index; }
		
		public void addEntry( DataEntry de ) {
		  addElement(de);
		  SimulationDataTree.get().addEntry( this, de );  
		  de.setDataSet(this);
		}
		
		public static JTree getTree() { return SimulationDataTree.get().getTree(); }
		
		public SimulationData getSimulationData() { 
			RepairedMutableTreeNode tn = SimulationDataTree.get().getSimulationNode ( this );
			return (SimulationData) tn.getUserObject();
		}

		public DataEntry getEntry( int index ) {
		  try {
			return (DataEntry) elementAt(index);
		  } catch ( ArrayIndexOutOfBoundsException err ) {
			return null;
		  }
		}

		public int getEntryIndex(DataEntry de) {
		  try {
			int index = 0;
			while( true ) {
			  DataEntry de1 = (DataEntry) elementAt(index++);
			  if( de1 == de ) return index-1;
			}
		  } catch ( ArrayIndexOutOfBoundsException err ) {
			return -1;
		  }
		}
		
		public DataEntry lastEntry() { return (DataEntry) lastElement(); }
		public DataEntry firstEntry() { return (DataEntry) firstElement(); }
		
		public DataEntry getSelectedEntry() { return ( _current_index < 0 ) ? null : getEntry( _current_index ); }		
		
		public static int getTreeDataSetSelections( Vector ds, int viewer_index ) {
		  JTree simTree = SimulationDataTree.get().getTree();
		  int nSelections = simTree.getSelectionCount();
		  if( nSelections > 0 ) { 
			TreePath[] tps = simTree.getSelectionPaths(); 
			for( int i=0; i<nSelections; i++ ) {
			  TreePath tp = tps[i]; 
			  if( tp.getPathCount() == 3 ) { 
				RepairedMutableTreeNode tn = (RepairedMutableTreeNode)tp.getLastPathComponent();
				DataSet d = (DataSet) tn.getUserObject();
				if( ViewerData.checkDataValidity( viewer_index, d.dims() ) ) {
				  if( d.isValid() ) { ds.addElement( d ); }
				}
			  } else if ( tp.getPathCount() == 2 ) {  // List and add components;
				RepairedMutableTreeNode tn = (RepairedMutableTreeNode)tp.getLastPathComponent();
				int cc = tn.getChildCount();
				for( int j=0; j<cc ; j++ ) {
				  RepairedMutableTreeNode tnc = (RepairedMutableTreeNode)tn.getChildAt(j);
				  DataSet d = (DataSet) tnc.getUserObject();
				  if( ViewerData.checkDataValidity( viewer_index, d.dims() ) ) {
					if( d.isValid() ) {  ds.addElement( d ); }
				  }
				}
			  }
			}
		  }
		  return nSelections;
		}

		public DataEntry getEntry( float time ) { 
		  DataEntry rv = null, id1 = null;
		  if( time == 0.0f  ) {
			if( _current_index > 0 ) { _current_index = 0; } 
			rv = getEntry( 0 );
		  } else {
			while( true ) {
			  rv = ( id1 == null ) ? getEntry( _current_index ) : id1;
			  int max_index = size() - 1;
			  if( rv == null ) { return null; }
			  float  tcur = rv.time();
			  if( time == tcur ) { break; }
			  else if( time < tcur ) {            // backwards stepping  
				if( _current_index == 0 ) { break; }
				id1 =  getEntry( --_current_index );
				float tnext = id1.time();
				if( ( time >= tnext ) && ( ( (time-tnext)/(tcur-tnext) ) <= 0.5f ) ) { 
				  rv = id1;
				  break;
				} 
			  } else if( time > tcur ) {                // forward stepping  
				if( _current_index == max_index ) { break; }
				id1 =  getEntry( ++_current_index );
				float tnext = id1.time();
				if( ( time < tnext ) && ( ( (tnext-time)/(tnext-tcur) ) < 0.5f ) ) { 
				  rv = id1;
				  break;
				} 
			  }
			}
		  }
		  return rv;
		}

		public float getDataMax( ) {
		  float data_max_value = -Float.MAX_VALUE;
		  for( int i=0; i<elementCount; i++ ) {
			DataEntry id = (DataEntry) elementData[i];
			if( (id != null ) && (id.max() > data_max_value) ) {
			  data_max_value = id.max();
			}
		  }
		  return data_max_value;
		}
		
		public float getDataMin( ) {
		  float _data_min_value = Float.MAX_VALUE;
		  for( int i=0; i<elementCount; i++ ) {
			DataEntry id = (DataEntry) elementData[i];
			if( id.min() < _data_min_value ) {
			  _data_min_value = id.min();
			}
		  }
		  return _data_min_value;
		}
		
		public void setNodeIndex( short dset_node_index, short sim_node_index ) {
		  _node_index = dset_node_index;
		  _sim_index = sim_node_index;
		}
		
		public short getDSetNodeIndex() {
		  return _node_index;
		}
		public short getSimNodeIndex() {
		  return _sim_index;
		}
		
		public String toString() { return _info; }
		public String Legend() { return _cats; }
		
		public boolean setDataSize( int size ) {
		  boolean rv = (size>0);
		  if( _data_size == 0 ) {
			_data_size = size;
		  } else if( _data_size != size ) {
			rv = false;  
		  }
		  return rv;
		}
		
		public static void initSimulation( String proj_name, String model_name, String proj_path ) { 
		  SimulationDataTree.get().addSimulation( proj_name, model_name, proj_path ); 
		}

	   public static void addSimulation( SimulationData sd ) {
		 try {
			SimulationDataTree.get().appendSimulation( sd ); 
		  } catch ( Exception err ) {
			SimIO.print( " Error unserializing data: " ); 
			err.printStackTrace();
		  }
	   }
	   
	   public static void deleteTreeSelection() {
		  SimulationDataTree.get().deleteSelectedNodes();
	   }

	   public static void clearData() {
		  SimulationDataTree.get().clear();
	   }
	   
	   public static SimulationData getCurrentSimulation() { 
		 return SimulationDataTree.get().getCurrentSimulation(); 
	  }
		
		public static DataSet create( int dsetIndex, int L0, int L1, int L2, byte dims, byte adim, String info, String cats, byte[] map, byte format, byte mode )  {
		  DataSet ds = new DataSet(dsetIndex, L0, L1, L2, dims, info, cats, map, format, mode );
		  ds.setArrayDimension(adim);
		  SimulationData sd = SimulationDataTree.get().getCurrentSimulation();
		  if( sd == null ) {
			SimIO.print( "No simulation exists, can't create dataset"  );
			return null;
		  }
		  if( !sd.addDataSet(dsetIndex,ds) ) {
			SimIO.print("Simulation DataSet " + dsetIndex + " already exists: overwriting.");
		  }
		  if( SimulationDataTree.get().addDataSet( ds, dsetIndex, -1 ) ) {
			SimIO.print("Created Dataset " + info + ", index = " + dsetIndex + ",dims = " + dims );
			return ds;
		  } else return null;
		}
		
		public static DataSet create( int dsetIndex, DataSet ds, String info, byte mode )  {
		  return new DataSet(dsetIndex, ds._L0, ds._L1, ds._L2, ds._dims, info, ds._cats, ds._map, ds._format, mode );
		}
		   		
		public static DataSet create2DDemo( int nEntries, int L0, int L1 ) {
		  float short_max = 256*256-1;
		  initSimulation( "Demo Project", "Transcendental Surfaces", "/" );
		  DataSet d = create( 0, L0, L1, 1, (byte)2, (byte)0, "demo", null, null, (byte)0, (byte)0 );
		  for( int k=0; k< nEntries; k++ ) {
			short[] data = new short[L0*L1];
			float a = 7+k, b = 4+k;
			for( int i=0; i<L1; i++ ) {
			  for( int j=0; j<L0; j++ ) {
				data[ j + i*L0 ] = (short) ( short_max * ( 2 + Math.sin((a*i)/L1) + Math.cos((b*j)/L0) )/4f );
			  }
			}
			DataEntry de = new DataEntry( data, "Surface" + k, 10f*(2*nEntries-k), 0f, 0.0f + k, 1.0f );
			d.addEntry( de );
		  }
		  return d;
		} 

		public static DataSet create1DDemo( int nEntries, int length ) {
		  float short_max = 256*256-1;
		  initSimulation( "Demo Project", "Transcendental Series", "/" );
		  DataSet d = create( 0, length, 1, 1, (byte)1, (byte)0, "demo", null, null, (byte)0, (byte)0 );
		  float series_len = length*nEntries;
		  int time = 0, t0;
		  float a = 7;
		  for( int k=0; k< nEntries; k++ ) {
			short[] data = new short[length];
			t0 = time;
			for( int i=0; i<length; i++ ) {
				data[ i ] = (short) ( short_max * ( time++/(series_len-1f) )  );
			}
			DataEntry de = new DataEntry( data, "Series" + k, 1000*series_len, 0.0f, t0, 1.0f );
			d.addEntry( de );
		  }
		  return d;
		} 
		  
		
		public static DataSet getDataSet(int dsetIndex) {
		  return SimulationDataTree.get().getDataSet(dsetIndex);
		}
				
		public static ListIterator getDatasetIterator() {
		  return SimulationDataTree.get().getDatasetIterator();
		}
			
		public void destroyDataListeners() {
		  if ( _index < 0 ) return;
		  SimulationDataTree.get().getDataSetNode ( this ).destroyDataListeners();
		}
		public void addEntryDataListener( DisposableListDataListener l ) {
		  if ( _index < 0 ) return;
		  SimulationDataTree.get().getDataSetNode ( this ).addEntryDataListener( l );
		}
		public void removeEntryDataListener( DisposableListDataListener l ) {
			if ( _index < 0 ) return;
			SimulationDataTree.get().getDataSetNode ( this ).removeEntryDataListener( l );
		}
		
		public static void  addTreeModelListener(TreeModelListener l) {
		   SimulationDataTree.get().addTreeModelListener(l);
		}
		
		public static RepairedMutableTreeNode getRoot() { return SimulationDataTree.get().getRoot(); }
		
		public float getValue( int row, int col, DataEntry de ) {
		  int width = _L0;
		  if( _format == 0 ) { return de.getValue( row*width + col ); }
		  else {
			return de.getValue( getCompressionConversion( row, col ) ); 
		  }
		}

		public void setValue( int row, int col, DataEntry de, float val ) {
		  int width = _L0;
		  if( _format == 0 ) { de.setValue( row*width + col, val ); }
		  else {
			de.setValue( getCompressionConversion( row, col ), val ); 
		  }
		}

		public float getSourceValue( int row, int col, DataEntry de ) {
		  int width = _L0;
		  if( _format == 0 ) { return de.getSourceValue( row*width + col ); }
		  else {
			return de.getSourceValue( getCompressionConversion( row, col ) ); 
		  }
		}

		public float getValue( int index, DataEntry de ) {
		   return de.getValue( index );
		}
		
		private int getCompressionConversion( int row, int col ) {
		   int width = _L0;
		   int index = row * width + col;
		   if( _map == null ) { return index; }
		   int rv = 0;
		   for( int i=0; i < index; i++ ) {
			  if( _map[ i ] > 0 ) rv++;
		   }
		   return rv; 
		}
		
		public int[] generateCompressionMap() {
		  int size = _L0 * _L1, cnt=0;
		  int[] compressionMap = new int[ size ];
		  for( int i=0; i < size; i++ ) {
			 if( (_map == null ) || ( _map[ i ] > 0 ) ) { compressionMap[i] = cnt++; }
			 else { compressionMap[i] = -1; }
		  } 
		  return compressionMap;
		}
		
		public float getTimestep() {
		  DataEntry id0 =  (DataEntry) getEntry( 0 );
		  DataEntry id1 =  (DataEntry) getEntry( 1 );
		  try {
			float t0 = id0.time();
			float t1 = id1.time();
			return t1 - t0;
		  } catch ( NullPointerException e) {
			return Float.MAX_VALUE;
		  }
		}
		

}


class DataTreeCellRenderer extends DefaultTreeCellRenderer  {
/*
    protected Icon fileIcon = null;
    protected Icon computerIcon = null;
    protected Icon floppyDriveIcon = null;

    protected Icon homeFolderIcon = null;
    protected Icon listViewIcon = null;
    protected Icon detailsViewIcon = null;
*/    
    protected Icon hardDriveIcon = null;
    protected Icon spatialIcon = null;
    protected Icon temporalIcon = null;
    protected Icon parameterIcon = null;
   


	public DataTreeCellRenderer() { 
	  super(); 
/*
	  fileIcon         = UIManager.getIcon("FileView.fileIcon");
	  computerIcon     = UIManager.getIcon("FileView.computerIcon");
	  floppyDriveIcon  = UIManager.getIcon("FileView.floppyDriveIcon");	

	  homeFolderIcon   = UIManager.getIcon("FileChooser.homeFolderIcon");
	  detailsViewIcon  = UIManager.getIcon("FileChooser.detailsViewIcon");
	  listViewIcon     = UIManager.getIcon("FileChooser.listViewIcon");
*/
	  hardDriveIcon    = UIManager.getIcon("FileView.hardDriveIcon");	  
	  spatialIcon      = SimIO.loadIcon( "spatial.gif" );
	  temporalIcon      = SimIO.loadIcon( "temporal.gif" );
	  parameterIcon      = SimIO.loadIcon( "parameter.gif" );

	}
	
	void setDefaultIcon( boolean leaf, boolean expanded ) {
		if (leaf) {
			setIcon(getLeafIcon());
		} else if (expanded) {
			setIcon(getOpenIcon());
		} else {
			setIcon(getClosedIcon());
		}
	}
	
    public Component getTreeCellRendererComponent(JTree tree, Object value,
						  boolean sel,
						  boolean expanded,
						  boolean leaf, int row,
						  boolean hasFocus) {

	  String stringValue = tree.convertValueToText(value, sel, expanded, leaf, row, hasFocus);
	  setText(stringValue);
	  if(sel)
		  setForeground(getTextSelectionColor());
	  else
		  setForeground(getTextNonSelectionColor());
		
	  try {   
		DataSet ds = (DataSet) (((DefaultMutableTreeNode)value).getUserObject());
		int dim = ds.dims();
		switch( dim ) {
		  case 0: setIcon(parameterIcon); break;
		  case 1: setIcon(temporalIcon); break;
		  case 2: setIcon(spatialIcon); break;
		  case 3: setIcon(hardDriveIcon); break;
		  default: setDefaultIcon( leaf, expanded ); break;
		} 
	  } catch ( ClassCastException err ) {
		setDefaultIcon( leaf, expanded );
	  }
	  selected = sel;

	  return this;
    }
}
