#include "UCode.h"
#include <stdlib.h>
#define INF_INC 0.6 // Infiltration increment - how fast inf.coef. grows when no rain (was 1.5 for inches) last=1.2, 0.96, 0.48
#define INF_DEC 280   // Infiltration decrement - how fast inf.coef. drops when rain occurs (was 12) 0.3 last = 525, 480, 380, 280
#define PREC_THRESH 0.08  // Prec. maximum after which it decreases infiltration - was 1.8 and then 2.6 for inches 0.065 for m
#define CPOINT 30   // Defines a center point where average output will be generated

/**************************************************************************/

void Infilt ( CVariable& InfiltRate, CVariable& Precipitation, 
              CVariable& Output1, CVariable& Output2, CVariable& Output3, CVariable& Output4 )
{	
	static FILE *file;
	float DD = 1., PrecipTot, InfiltAv, InfModAv, OutVar1, OutVar2, OutVar3, OutVar4;
	int totpoints = 0;

	DistributedGrid& grid = InfiltRate.Grid();
	grid.SetPointOrdering(0);	                 // Sets grid ordering to default
	
	if (file == NULL) 
	{ // open output file
		CPathString pathName(Env::ArchivePath()); 
    		pathName.Add("Infilter");
		if ( (file = fopen (pathName, "w")) == NULL) gFatal( "Can't open Infilter file! " ) ; 
		fprintf ( file, "Infiltration_av\tInf_modifier\tPrecipit_av\t       OUT1\t      OUT2\t     OUT3\t      OUT4\n");
	  
	}
		
	static CVariable* P_Counter = NULL;	
	if(P_Counter == NULL )
	  { 
	    P_Counter = InfiltRate.GetSimilarVariable("P_Counter");
	                      // starts from root of FrameLink config if Point is NULL
	                      
	    P_Counter -> Set(0.0);
	  }
	                      
	InfiltRate.LinkEdges();	
	
	InfiltAv = 0;
	InfModAv = 0;
	PrecipTot = 0;
	OutVar1 = 0;
	OutVar2 = 0;
	OutVar3 = 0;
	OutVar4 = 0;
	
	for( Pix p = grid.first(); p; grid.next(p) )     // loop over the whole area
		{ 
		  totpoints++;   // calculate total number of points in study area
		  
		  const OrderedPoint& pt = grid.GetPoint(p);     // pt same as p, but ordered

		  if ( (*P_Counter)(pt) > 0.9 ) InfiltRate (pt) *= (*P_Counter)(pt);
		/* was 0.0, but was getting too high peaks */  
//		  if ( Precipitation(pt) > PREC_THRESH ) //decrease inf. when too much rain
//		  	{ DD =  PREC_THRESH/Precipitation(pt);
//		  	  InfiltRate (pt) *= pow(DD,4.);
//		  	}
		  	
		  //increase inf. if no rain
		  if ( Precipitation(pt) < 0.00002 ) (*P_Counter)(pt) += INF_INC;
		  //else decrease inf. if raining
		  else 
		    { DD = Precipitation(pt)*INF_DEC;
		      (*P_Counter)(pt) = ((*P_Counter)(pt) > DD) ? (*P_Counter)(pt) - DD : 0.;
		    }  

//		  int ir0 = pt.row(), ic0 = pt.col();
//		  if (ir0 == CPOINT && ic0 == CPOINT) 
//		    fprintf ( file, "%12.4f\t", Precipitation(pt) );
		  
		  InfiltAv += InfiltRate (pt);
		  InfModAv += (*P_Counter)(pt);
		  PrecipTot += Precipitation(pt);  
		  OutVar1 += Output1(pt);
		  OutVar2 += Output2(pt);
		  OutVar3 += Output3(pt);
		  OutVar4 += Output4(pt);

		}

	InfiltAv /= totpoints;
	InfModAv /= totpoints;
	PrecipTot /= totpoints;
	OutVar1 /= totpoints;
	OutVar2 /= totpoints;
	OutVar3 /= totpoints;
	OutVar4 /= totpoints;
	
	fprintf ( file, "%12.4f\t%10.4f\t%10.4f\t%12.4f\t%10.4f\t%12.4f\t%12.4f\t%12d\n", InfiltAv, InfModAv, PrecipTot, OutVar1, OutVar2, OutVar3, OutVar4, totpoints);
	
	fflush(file); 
}
/**************************************************************************/
#define COUNTY 7
#define SOILS 41



void FertilizerMap( CVariable& FertMap, CVariable& LU, CVariable& Soils, CVariable& Counties)   
// arguments gotten from config file, first arg is always variable being configured.
{
	static int pd = 0;
	
	int f_table[COUNTY][SOILS];
	
	int i, j;

	//	if (pd == 1) return;

	FILE *ffile = NULL;

	if (ffile == NULL) 
	{
		CPathString pathName(Env::DataPath()); 
    		pathName.Add("FertilData");
		if ( (ffile = fopen (pathName, "r")) == NULL)  
		         gFatal( "Can't open FertilData file! " );
// FertilData is in kg/ha
	}

	for (j=0; j<SOILS; j++)
	   for (i=0; i<COUNTY; i++)
	      fscanf ( ffile, "%d\t", &f_table[i][j]);
		
	DistributedGrid& grid = Soils.Grid();
	grid.SetPointOrdering(0);
	// sets grid ordering to default ordering (row-col) (ordering #0)

        for( Pix p = grid.first(); p; grid.next(p) ) 
	     { 
	      const OrderedPoint& pt = grid.GetPoint(p); // sets currentPoint

	      if( LU(pt) > 1 )
       		 FertMap (pt) = f_table[(int)Counties(pt)-1][(int)Soils(pt)-1];
	      else FertMap (pt) = 0.;
             } // end loop over all area
             
         pd = 1;
	 fclose (ffile);
}



/**************************************************************************/
#define LURESID 4
#define LUAGRO 6
#define LUAGRO1 3
#define LUAGRO2 7
#define LUAGRO3 8
#define LUAGRO4 9
#define LUURBAN 5


void DwellMap( CVariable& Dwell, CVariable& LU, CVariable& ZON, CVariable& Bocoef)   
// defines dwelling densities based on the zoning map
{
	static int pd = 0;
		
	int i, j;

	//	if (pd == 1) return;
		
	DistributedGrid& grid = LU.Grid();
	grid.SetPointOrdering(0);
	// sets grid ordering to default ordering (row-col) (ordering #0)

        for( Pix p = grid.first(); p; grid.next(p) ) 
	     { 
	      const OrderedPoint& pt = grid.GetPoint(p); // sets currentPoint

	      if( LU(pt) == LURESID || LU(pt) == LUURBAN || LU(pt) >= LUAGRO )
       		 Dwell (pt) = ZON(pt)*(float)Bocoef.Value();
	      else Dwell (pt) = 0;
             } // end loop over all area
         float Total = Dwell.getSum();
         if( gIProc == 0 )
             printf ("Info:Total Dwelling Units =  %f\n", Total); 
         pd = 1;
}
/**************************************************************************/
#define DENSRES 250
#define DENSURB 900
#define DENSAGR 2

void DwellMap2( CVariable& Dwell, CVariable& LU, CVariable& Bocoef)   
// defines dwelling densities based on some assumed averages for each habitat type
{
	static int pd = 0;
		
	int i, j;

	if (pd == 1) return;
		
	DistributedGrid& grid = LU.Grid();
	grid.SetPointOrdering(0);
	// sets grid ordering to default ordering (row-col) (ordering #0)

        for( Pix p = grid.first(); p; grid.next(p) ) 
	     { 
	      const OrderedPoint& pt = grid.GetPoint(p); // sets currentPoint

	      if( LU(pt) == LURESID )
       	        Dwell (pt) = DENSRES*(float)Bocoef.Value();
	      else if ( LU(pt) == LUURBAN )
       	        Dwell (pt) = DENSURB*(float)Bocoef.Value();
	      else if (  LU(pt) == LUAGRO ||  LU(pt) == LUAGRO1 || LU(pt) == LUAGRO2 || LU(pt) == LUAGRO3 || LU(pt) == LUAGRO4 )
       	        Dwell (pt) = DENSAGR*(float)Bocoef.Value();
	      else Dwell (pt) = 0;
             } // end loop over all area

         float Total = Dwell.getSum();
         if( gIProc == 0 )
             printf ("Info:Total Dwelling Units =  %f\n", Total); 
         pd = 1;
}
/**************************************************************************/
#define PERIOD 10

void DefineMax( CVariable& Max10, CVariable& What)   

{
	static int cur_time = 0;

	if (cur_time == PERIOD) 
	  {  Max10.Set(0.0); 
	     cur_time = 0;
	  }
	
	DistributedGrid& grid = What.Grid();
	grid.SetPointOrdering(0);
	// sets grid ordering to default ordering (row-col) (ordering #0)

        for( Pix p = grid.first(); p; grid.next(p) ) 
	     { 
	      const OrderedPoint& pt = grid.GetPoint(p); // sets currentPoint

	      if( What(pt) > Max10(pt) ) Max10(pt) = What(pt);

             } // end loop over all area
             
         cur_time++;
         
}
/**************************************************************************/
#define TIME1 118
#define TIME2 190
#define TIME3 292
#define TIME4 324

#define FALLOW 6
#define CORN 7
#define WHEAT 8
#define SOY 9

void CropRot( CVariable& LuMap, CVariable& Day )   

{       int DD;

        DD=(int)Day.Value();
	DistributedGrid& grid = LuMap.Grid();
	grid.SetPointOrdering(0);
	// sets grid ordering to default ordering (row-col) (ordering #0)

        for( Pix p = grid.first(); p; grid.next(p) ) 
	     { 
	      const OrderedPoint& pt = grid.GetPoint(p); // sets currentPoint

	      if( DD == TIME1 && LuMap(pt) == FALLOW ) LuMap(pt) = CORN;
	      if( DD == TIME2 && LuMap(pt) == WHEAT ) LuMap(pt) = SOY;
	      if( DD == TIME3 && LuMap(pt) == CORN ) LuMap(pt) = WHEAT;
	      if( DD == TIME4 && LuMap(pt) == SOY ) LuMap(pt) = FALLOW;

             } // end loop over all area
}
/**************************************************************************/
void Output_Func ( CVariable& Map, CVariable& Var1, CVariable& Var2, CVariable& Var3, 
		   CVariable& Var4, CVariable& Var5, CVariable& Var6, 
		   CVariable& Row, CVariable& Col )
{	
	static FILE *file;
	static int ttt;
	float OutVar1, OutVar2, OutVar3, OutVar4, OutVar5, OutVar6;
	int totpoints = 0;

	DistributedGrid& grid = Map.Grid();
	grid.SetPointOrdering(0);	                 // Sets grid ordering to default
	
	if (file == NULL) 
	{ // open output file
		CPathString pathName(Env::ArchivePath()); 
    		pathName.Add("OutPutFile");
		if ( (file = fopen (pathName, "w")) == NULL) gFatal( "Can't open OutPutFile! " ); 
		fprintf ( file, "Time\t    OUT1\t    OUT2\t    OUT3\t    OUT4\t    OUT5\t    OUT6\n");
		ttt = 0;	  
	}

	ttt++;
	
	OutVar1 = 0;
	OutVar2 = 0;
	OutVar3 = 0;
	OutVar4 = 0;
	OutVar5 = 0;
	OutVar6 = 0;
	
	for( Pix p = grid.first(); p; grid.next(p) )     // loop over the whole area
		{ 
		  totpoints++;   // calculate total number of points in study area
		  
		  const OrderedPoint& pt = grid.GetPoint(p);     // pt same as p, but ordered

		  int ir0 = pt.row(), ic0 = pt.col();
		  if ( ir0 == (int)Row.Value() && ic0 == (int)Col.Value() ) 
		    fprintf ( file, "%d\t%14.10f\t%14.10f\t%14.10f\t%14.10f\t%14.10f\t%14.10f", 
		        ttt, Var1(pt), Var2(pt), Var3(pt), Var4(pt), Var5(pt), Var6(pt) );
		  
		  OutVar1 += Var1(pt);
		  OutVar2 += Var2(pt);
		  OutVar3 += Var3(pt);
		  OutVar4 += Var4(pt);
		  OutVar5 += Var5(pt);
		  OutVar6 += Var6(pt);

		}

	OutVar1 /= totpoints;
	OutVar2 /= totpoints;
	OutVar3 /= totpoints;
	OutVar4 /= totpoints;
	OutVar5 /= totpoints;
	OutVar6 /= totpoints;
	
	fprintf ( file, "\t%12.8f\t%12.8f\t%12.8f\t%12.8f\t%12.8f\t%12.8f\t%d\n", 
		  OutVar1, OutVar2, OutVar3, OutVar4, OutVar5, OutVar6, totpoints );
	
	fflush(file); 
}
/**************************************************************************/
void Output_Func_Map ( CVariable& Map, CVariable& OnMap, CVariable& Var1, CVariable& Var2, CVariable& Var3, 
		   CVariable& Var4, CVariable& Var5, CVariable& Var6, 
		   CVariable& Row, CVariable& Col )
{	
	static FILE *file;
	static int ttt;
	float OutVar1, OutVar2, OutVar3, OutVar4, OutVar5, OutVar6;
	int totpoints = 0;

	DistributedGrid& grid = Map.Grid();
	grid.SetPointOrdering(0);	                 // Sets grid ordering to default
	
	if (file == NULL) 
	{ // open output file
		CPathString pathName(Env::ArchivePath()); 
    		pathName.Add("OutPutFile");
		if ( (file = fopen (pathName, "w")) == NULL) gFatal( "Can't open OutPutFile! " ); 
		fprintf ( file, "Time\t    OUT1\t    OUT2\t    OUT3\t    OUT4\t    OUT5\t    OUT6\n");
		ttt = 0;	  
	}

	ttt++;
	
	OutVar1 = 0;
	OutVar2 = 0;
	OutVar3 = 0;
	OutVar4 = 0;
	OutVar5 = 0;
	OutVar6 = 0;
	
	for( Pix p = grid.first(); p; grid.next(p) )     // loop over the whole area
		{ 
		  
		  
		  const OrderedPoint& pt = grid.GetPoint(p);     // pt same as p, but ordered

		  int ir0 = pt.row(), ic0 = pt.col();
		  if ( ir0 == (int)Row.Value() && ic0 == (int)Col.Value() ) 
		   {	
		    fprintf ( file, "%d\t%14.10f\t%14.10f\t%14.10f\t%14.10f\t%14.10f\t%14.10f", ttt, Var1(pt), Var2(pt), Var3(pt), Var4(pt), Var5(pt), Var6(pt) );
		   }
		  totpoints++;   // calculate total number of points in study area

		  if( (int)OnMap(pt) == 3 || (int)OnMap(pt) >= 6 ) continue;   // skip agro units 

		  
		  OutVar1 += Var1(pt);
		  OutVar2 += Var2(pt);
		  OutVar3 += Var3(pt);
		  OutVar4 += Var4(pt);
		  OutVar5 += Var5(pt);
		  OutVar6 += Var6(pt);

		}

	OutVar1 /= totpoints;
	OutVar2 /= totpoints;
	OutVar3 /= totpoints;
	OutVar4 /= totpoints;
	OutVar5 /= totpoints;
	OutVar6 /= totpoints;
	
	fprintf ( file, "\t%12.8f\t%12.8f\t%12.8f\t%12.8f\t%12.8f\t%12.8f\t%d\n", 
		  OutVar1, OutVar2, OutVar3, OutVar4, OutVar5, OutVar6, totpoints );
	
	fflush(file); 
}

/**************************************************************************/

void MakeLuMap ( CVariable& LuMap  )   
{       
  static int once = 0;
  static FILE *ffile;
  int HabFrom, HabTo;
  float T1;
  time_t *timer;
  int count = 0;

  if ( once > 0 ) return;

  if ( ffile == NULL )
    {
      CPathString pathName(Env::DataPath()); 
      pathName.Add("LandUseData");
      if ( (ffile = fopen (pathName, "r")) == NULL)  gFatal( "Can't open LandUseData file! " );
      fscanf ( ffile, "%d", &HabFrom );
      fscanf ( ffile, "%d", &HabTo );
      fscanf ( ffile, "%f", &T1 );

      // HabFrom - from which habitat we change; HabTo - to which habitat we change
      // T1 - proportion of change. Exm: T1=0.1 means that for ~0.1 of cells we want to 
      // make the switch.

      fclose (ffile);
    }

  DistributedGrid& grid = LuMap.Grid();
  grid.SetPointOrdering(0);
	// sets grid ordering to default ordering (row-col) (ordering #0)
  unsigned int ttt = time(timer);

  srand((unsigned int)time(timer));

  for( Pix p = grid.first(); p; grid.next(p) ) 
    { 
      const OrderedPoint& pt = grid.GetPoint(p); // sets currentPoint

      if( (int)LuMap(pt) == HabFrom ) 
	{
	  float rrr = (float)rand();
	  float random =  rrr/RAND_MAX;

	  if ( random > (1-T1) ) 
	    { 
	      LuMap(pt) = HabTo;
	      count++;
	    }
	}
    } // end loop over all area

  printf("\ninfo: number switches = %d", count);

  once = 1;
}
/**************************************************************************/
#define VVV 0

void Road_Dist( CVariable& Dist, CVariable& LU, CVariable& Coef)   
// defines distance along the road network for non-existent roads
{
	static int once = 0;
	int alldone = 0;	
	int i, j;
	int  rr, rc, NN, Nmax = 1;
	Pix p, rp;

	if (once == 1) return;
		
	DistributedGrid& grid = LU.Grid();
	grid.SetPointOrdering(0);

	while (!alldone)
	{
         alldone = 1;
	 // loop over the whole area looking at zero cells	
         printf ("\nInfo: Scanning area"); 

	 for( p = grid.first(); p; grid.next(p) )               
	  { 
	     const OrderedPoint& pt = grid.GetPoint(p);     
	     if (Dist(pt) != VVV) continue;
	     //else there are still unidentified cells

	     if (LU(pt) == 1) 
		{  Dist(pt) = -1;
		   continue;   // if river is in cell, skip it
			// this is to make sure that the river blocks 
			// access and we have to find another way around	
		}

	     alldone = 0;

             // unidentified cell found
	     // look around in a concentric circle to find all non-zero elements
	     float min = 99999999;
             for (NN = 1; NN <= Nmax; NN++)  //NN is number of concentric layer
		  { 
		   for (rr = -NN; rr <= NN; rr++)
		      for (rc = -NN; rc <= NN; rc++)
		        
		      {
		        rp = grid.TranslateByGridCoord( p, rr, rc );

		        if( rp )  
		        {
		          const OrderedPoint& rpt = grid.GetPoint(rp);

		          if (Dist(rpt) > 0) 
			     if (Dist(rpt) < min) min = Dist(rpt);
		             
		        } 
		      }

		   } // end loop over concentric neighborhood 

	     if ( min < 99999999 ) Dist(pt) = min + (float)Coef.Value();

	   }// end loop over cells

	} // end search for unidentified cells
        once = 1;
	int c;

	// scan area once again taking care of -1
	printf ("\nInfo: Scan distances again? (0/1) "); 
	if ((c = getchar()) == '0') return;
 
	for( p = grid.first(); p; grid.next(p) )               
	  { 
	     const OrderedPoint& pt = grid.GetPoint(p);     // pt same as p, but ordered
	     if (Dist(pt) != -1) continue;
	     //else it is a -1

	     float min = 99999999;
             for (NN = 1; NN <= Nmax; NN++)  //NN is number of concentric layer
		  { 
		   for (rr = -NN; rr <= NN; rr++)
		      for (rc = -NN; rc <= NN; rc++)
		        
		      {
		        rp = grid.TranslateByGridCoord( p, rr, rc );

		        if( rp )  
		        {
		          const OrderedPoint& rpt = grid.GetPoint(rp);

		          if (Dist(rpt) > 0 && LU(rpt) != 1) 
			     if (Dist(rpt) < min) min = Dist(rpt);
		             
		        } 
		      }

		   } // end loop over concentric neighborhood 
 
	     if ( min < 99999999 ) Dist(pt) = min + (float)Coef.Value();

	   }// end loop over cells

 
	if( gIProc == 0 )
             printf ("Info: Distances filled"); 


         once = 1;
}
/**************************************************************************/

void MakeLuMapRand ( CVariable& LuMap, CVariable& Open, CVariable& Slope, 
                     CVariable& RoadT, CVariable& G1, CVariable& B1  )   

/* distributes certain categories over the Land Use map according to some
	rules of probability that depend upon the 
	Slope - map of slopes 
	RoadT - map of travel time
	G1    - distance to attractions (goods)
	B1    - distance to distractions (bads)

	Open - map of cells that can be used in the distribution process
*/
{       
  static int once = 0;
  static FILE *ffile;
  int HabFrom, HabTo;
  float T1;

  int n_units;	 	// number of units to distribute
  int new_category; 	// category on Lu Map for the area that is distributed
  float porog;		// threshold for chance of change

  time_t *timer;
  int count = 0;

  if ( once > 0 ) return;

  if ( ffile == NULL )
    {
      CPathString pathName(Env::DataPath()); 
      pathName.Add("LandUseData");
      if ( (ffile = fopen (pathName, "r")) == NULL)  gFatal( "Can't open LandUseData file! " );
      fscanf ( ffile, "%d", &HabFrom );
      fscanf ( ffile, "%d", &HabTo );
      fscanf ( ffile, "%f", &T1 );
      // HabFrom; HabTo; T1 - used in other routines, therefore dummy fscanfs

      fscanf ( ffile, "%d", &n_units );
      fscanf ( ffile, "%d", &new_category );
      fscanf ( ffile, "%f", &porog );

      fclose (ffile);
    }

  DistributedGrid& grid = LuMap.Grid();
  grid.SetPointOrdering(0);
	// sets grid ordering to default ordering (row-col) (ordering #0)

  unsigned int ttt = time(timer);
  srand((unsigned int)time(timer));

	//find maximum of RoadTime map to do normalization
  float MaxRTime = 0.;
  for( Pix p = grid.first(); p; grid.next(p) ) 
    { 
      	const OrderedPoint& pt = grid.GetPoint(p); // sets currentPoint
        if (RoadT(pt) > MaxRTime) MaxRTime = RoadT(pt);
    }

  int number_loops = 1;

  while ( n_units > 0 && number_loops < 1000)
  {

    for( Pix p = grid.first(); p; grid.next(p) ) 
    { 
      	const OrderedPoint& pt = grid.GetPoint(p); // sets currentPoint
      
	if ( !(int)Open(pt) ) continue;	// cell not available for change
 
      	float rrr = (float)rand();
	float random =  rrr/RAND_MAX;

 	float chance = 0;

	chance = random*(1-RoadT(pt)/MaxRTime)*(1/Slope(pt))*(1/(1-G1(pt)))*(1/B1(pt));

	if ( chance > porog ) 
	    { 
	      LuMap(pt) = new_category;
	      n_units--;
	    }
    }// end loop over all area
	number_loops++;
  } // end loop over number of loops

  printf("\nall units distributed in %d loops\n", number_loops);

  once = 1;
}
/**************************************************************************/


