/**************************************************************************/
// Hydrologic functions used to distribute ground water horizontally
//@ Alexey Voinov -- voinov@cbl.umces.edu
/**************************************************************************/


#include "GWater.h"
/**************************************************************************/

#define MAXBOUND 1000 //max number of elements in array to store open boundary values

void GWTransport( CVariable& GWater, CVariable& Porosity, CVariable& Conductivity, 
		  CVariable& MapOut, CVariable& Stuff, CVariable& UnsatW )
		  
//		  , CVariable& F_Cap,
//		  CVariable& Elev, CVariable& Elev_min, CVariable& SWater )    
// arguments gotten from config file, first arg is always variable being configured.
{
	static FILE *ffile;

	int niter = 1;
	static float base_rate; // base_rate is in range ( 0.0 -> 1.0 )
	static float DIF;   // diffusion to slow down the movement of Stuff
	float Delta;	// variable to accumulate the water loss/gain through open boundary
	Pix rPix[10];
	static int pd;
	static int count;
	static float boundary[MAXBOUND]; //array to store open boundary values
	int first_tcall=0;
	Grid_Direction il;
	float TotStuff, TotWMove;
	
	float move, ftmp1;

	Porosity.LinkEdges();
	Conductivity.LinkEdges();
	MapOut.LinkEdges();

	DistributedGrid& grid = GWater.Grid();
	grid.SetPointOrdering(0);
	// sets grid ordering to default ordering (row-col) (ordering #0)

	if ( ffile == NULL )
	{
		CPathString pathName(Env::DataPath()); 
    		pathName.Add("GWData");
		if ( (ffile = fopen (pathName, "r")) == NULL)  gFatal( "Can't open GWData file! " );
		fscanf ( ffile, "%f %f\n", &base_rate, &DIF );
		
	}
	
	static FILE *ffile1 = NULL;
	int j=0;
	if (ffile1 == NULL) 
	{
		CPathString pathName(Env::ArchivePath()); 
    		pathName.Add("GOut");
		if ( (ffile1 = fopen (pathName, "w")) == NULL)  gFatal( "Can't open GOut file! " );
		fprintf ( ffile1, "base_rate=%12.4f DIF=%12.3f\n", base_rate, DIF );
	        fprintf ( ffile1, 
			  "   N  \t  Delta  \t  SumGW  \t  SumUW  \t  Net  \t TotalMoved \t Stuff  \n");

		count = 0;
		
	  // store open boundary values
	  	for( Pix p = grid.first(); p; grid.next(p) )
		{
	        	const OrderedPoint& pt = grid.GetPoint(p);           
			if( MapOut(pt) == 2 || MapOut(pt) == 10 ) boundary[j++] = GWater(pt);
		}
	}
	
	float SumGW = 0.;
	float SumUW = 0.;

	Delta = 0.;
	float fnet = 0.;
	TotStuff = 0.0;
	TotWMove = 0.0;
	
	for(int i=0; i<niter; i++) 
	  {	
	   GWater.LinkEdges();
	   Stuff.LinkEdges();

	   j = 0;   // boundary point index

	   for( Pix p = grid.first(); p; grid.next(p) ) 
	     { 
	      const OrderedPoint& pt = grid.GetPoint(p); // sets currentPoint
	      int ir = pt.row(), ic = pt.col();  // get row & column of point.

	      if( !grid.onGrid(pt) ) continue;  // (onGrid == False) -> Ghost Point
	      
       	      int rr, rc; 

	      float w_sum = GWater(pt)*Conductivity(pt); 
	       	    // get value of variable GWater at point pt
	      float p_sum = Porosity(pt)*Conductivity(pt);

		    // for each point calculate the average weighted groundwater
		    // head in the vicinity
	      for( il = firstGD(); moreGD(il); incrGD(il) ) 
		  { 
		    // enum Grid_Direction { NE=2, EE, SE, SS, SW, WW, NW, NN };

		    Pix rp = grid.NeighborPix( p, il ); 
		    // relative to pt, takes enum Grid_Direction as arg
		    rPix[il] = rp;
		    if( rp )  
		      {
		        const OrderedPoint& rpt = grid.GetPoint(rp); 
		        w_sum += GWater(rpt)*Conductivity(rpt);
		        p_sum += Porosity(rpt)*Conductivity(rpt);
		      } 
		  }

	      float h0 = w_sum / p_sum;
		//h0 is the average weighted groundwater head for this point
				
	      float ftmp = 0.;
	      float fnet1 = 0.;

	      for( il = firstGD(); moreGD(il); incrGD(il) ) 
		  { // scan the vicinity of the point pt once again to calculate the fluxes
		    Pix rp = rPix[il];
		    if( rp ) 
		    {
		      const OrderedPoint& rpt = grid.GetPoint(rp);
		      ftmp = (h0 * Porosity(pt) - GWater(rpt))
		      		* base_rate * (Conductivity(pt)+Conductivity(rpt))/2. / niter;

		      // for exchange of stuff, I assume that the transport is occuring via the center
		      // point pt. So that at each step the stuff is moved from each of the 
		      // vicinity cells to/from the center point. This can potentially drive the
		      // amount of stuff in the center point below zero, but at the end of the loop
		      // over the vicinity it should be back to normal.

		      if ( ftmp > 0. ) move = Stuff(pt)*DIF*ftmp/GWater(pt);
		      else move = Stuff(rpt)*DIF*ftmp/GWater(rpt);
		      
		      Stuff(rpt) += move;
		      Stuff(pt) -= move;

   	              GWater(rpt) += ftmp;  
		      GWater(pt) -= ftmp;
 		      fnet1 += ftmp;
 		      TotWMove += (ftmp >= 0) ? ftmp : -ftmp;
		    }
		  } // end loop over vicinity of the pt point

		fnet += fnet1;
		  
                // restore the value in the center point if it is an open boundary point
		// Delta keeps track of the net disbalance due to the fluxes through
		// the open boundary. If Delta is >0 then there is a net outflux
		// If Delta is <0 then there is a net influx thorugh the boundary
	        
	        if ( MapOut(pt) == 2 || MapOut(pt) == 10) 
		      //MapOut = ONMAP; =2 or 10 where we have an open boundary
		  { ftmp1 = GWater(pt) - boundary[j];

		    Stuff(pt) -= Stuff(pt)*DIF*ftmp1/GWater(pt); 
			  // stuff displaced from or added to the area
	            GWater(pt) = boundary[j++];
	          }
		else ftmp1 = 0.;
		
		if ( i == niter-1 ) 
		  {
	            Delta += ftmp1;  // ftmp1 is the flux from/to area

		    SumGW += GWater (pt);
		    SumUW += UnsatW (pt);

		    TotStuff += Stuff(pt);
		  }// endif last iteration
		
	     }// end area loop

	  }// end iterations loop
	
    printf ("\ninfo: %d", count);
    fprintf ( ffile1, "%d\t%12.4f\t%12.4f\t%12.4f\t%12.4f\t%12.4f\t%12.4f\n", 
	      count++, Delta, SumGW, SumUW, fnet, TotWMove, TotStuff/(SumGW + SumUW) );
    fflush (ffile1);

}
/**************************************************************************/

void GWTrans_I( CVariable& GWater, CVariable& Porosity, CVariable& Conductivity, 
		CVariable& MapOut, CVariable& Stuff, CVariable& UnsatW, CVariable& Out_F_SD )
		  
//		  , CVariable& F_Cap,
//		  CVariable& Elev, CVariable& Elev_min, CVariable& SWater )    
// arguments gotten from config file, first arg is always variable being configured.
{
	static FILE *ffile;

	int niter = 1;
	static float base_rate; // base_rate is in range ( 0.0 -> 1.0 )
	static float DIF;   // diffusion to slow down the movement of Stuff
	float Delta;	// variable to accumulate the water loss/gain through open boundary
	Pix rPix[10];
	static int pd;
	static int count;
	static float boundary[MAXBOUND]; //array to store open boundary values
	int first_tcall=0;
	Grid_Direction il;
	float TotStuff, TotWMove;
	
	float move, ftmp1;

	Porosity.LinkEdges();
	Conductivity.LinkEdges();
	MapOut.LinkEdges();

	DistributedGrid& grid = GWater.Grid();
	grid.SetPointOrdering(0);
	// sets grid ordering to default ordering (row-col) (ordering #0)

	if ( ffile == NULL )
	{
		CPathString pathName(Env::DataPath()); 
    		pathName.Add("GWData");
		if ( (ffile = fopen (pathName, "r")) == NULL)  gFatal( "Can't open GWData file! " );
		fscanf ( ffile, "%f %f\n", &base_rate, &DIF );
		
	}
	
	static FILE *ffile1 = NULL;
	int j=0;
	if (ffile1 == NULL) 
	{
		CPathString pathName(Env::ArchivePath()); 
    		pathName.Add("GOut");
		if ( (ffile1 = fopen (pathName, "w")) == NULL)  gFatal( "Can't open GOut file! " );
		fprintf ( ffile1, "base_rate=%12.4f DIF=%12.3f\n", base_rate, DIF );
	        fprintf ( ffile1, 
			  "   N  \t  Delta  \t  SumGW  \t  SumUW  \t  Net  \t TotalMoved \t Stuff  \n");

		count = 0;
		
	  // store open boundary values
	  	for( Pix p = grid.first(); p; grid.next(p) )
		{
	        	const OrderedPoint& pt = grid.GetPoint(p);           
			if( MapOut(pt) == 2 || MapOut(pt) == 10 ) boundary[j++] = GWater(pt);
		}
	}
	
	float SumGW = 0.;
	float SumUW = 0.;

	Delta = 0.;
	float fnet = 0.;
	TotStuff = 0.0;
	TotWMove = 0.0;
	
	for(int i=0; i<niter; i++) 
	  {	
	   GWater.LinkEdges();
	   Stuff.LinkEdges();

	   j = 0;   // boundary point index

	   for( Pix p = grid.first(); p; grid.next(p) ) 
	     { 
	      const OrderedPoint& pt = grid.GetPoint(p); // sets currentPoint
	      int ir = pt.row(), ic = pt.col();  // get row & column of point.

	      if( !grid.onGrid(pt) ) continue;  // (onGrid == False) -> Ghost Point
	      
       	      int rr, rc; 

	      float w_sum = GWater(pt)*Conductivity(pt); 
	       	    // get value of variable GWater at point pt
	      float p_sum = Porosity(pt)*Conductivity(pt);

		    // for each point calculate the average weighted groundwater
		    // head in the vicinity
	      for( il = firstGD(); moreGD(il); incrGD(il) ) 
		  { 
		    // enum Grid_Direction { NE=2, EE, SE, SS, SW, WW, NW, NN };

		    Pix rp = grid.NeighborPix( p, il ); 
		    // relative to pt, takes enum Grid_Direction as arg
		    rPix[il] = rp;
		    if( rp )  
		      {
		        const OrderedPoint& rpt = grid.GetPoint(rp); 
		        w_sum += GWater(rpt)*Conductivity(rpt);
		        p_sum += Porosity(rpt)*Conductivity(rpt);
		      } 
		  }

	        float h0 = w_sum / p_sum;
		//h0 is the average weighted groundwater head for this point
				
	        float ftmp = 0.;
	        float fnet1 = 0.;

		if (count == 0) Out_F_SD (pt) = 0;

	        float StuffOld = Stuff(pt);

	        for( il = firstGD(); moreGD(il); incrGD(il) ) 
		  { // scan the vicinity of the point pt once again to calculate the fluxes
		    Pix rp = rPix[il];
		    if( rp ) 
		    {
		      const OrderedPoint& rpt = grid.GetPoint(rp);
		      ftmp = (h0 * Porosity(pt) - GWater(rpt))
		      		* base_rate * (Conductivity(pt)+Conductivity(rpt))/2. / niter;

		      // for exchange of stuff, I assume that the transport is occuring via the center
		      // point pt. So that at each step the stuff is moved from each of the 
		      // vicinity cells to/from the center point. This can potentially drive the
		      // amount of stuff in the center point below zero, but at the end of the loop
		      // over the vicinity it should be back to normal.

		      if ( ftmp > 0. ) move = Stuff(pt)*DIF*ftmp/GWater(pt);
		      else move = Stuff(rpt)*DIF*ftmp/GWater(rpt);
		      
		      Stuff(rpt) += move;
		      Stuff(pt) -= move;

   	              GWater(rpt) += ftmp;  
		      GWater(pt) -= ftmp;
 		      fnet1 += ftmp;
 		      TotWMove += (ftmp >= 0) ? ftmp : -ftmp;
		    }
		  } // end loop over vicinity of the pt point

		Out_F_SD (pt) += ( StuffOld > Stuff(pt) ) ? StuffOld - Stuff(pt) : 0;

		fnet += fnet1;
		  
                // restore the value in the center point if it is an open boundary point
		// Delta keeps track of the net disbalance due to the fluxes through
		// the open boundary. If Delta is >0 then there is a net outflux
		// If Delta is <0 then there is a net influx thorugh the boundary
	        
	        if ( MapOut(pt) == 2 || MapOut(pt) == 10) 
		      //MapOut = ONMAP; =2 or 10 where we have an open boundary
		  { ftmp1 = GWater(pt) - boundary[j];

		    Stuff(pt) -= Stuff(pt)*DIF*ftmp1/GWater(pt); 
			  // stuff displaced from or added to the area
	            GWater(pt) = boundary[j++];
	          }
		else ftmp1 = 0.;
		
		if ( i == niter-1 ) 
		  {
	            Delta += ftmp1;  // ftmp1 is the flux from/to area

		    SumGW += GWater (pt);
		    SumUW += UnsatW (pt);

		    TotStuff += Stuff(pt);
		  }// endif last iteration
		
	     }// end area loop

	  }// end iterations loop
	
    printf ("\ninfo: %d", count);
    fprintf ( ffile1, "%d\t%12.4f\t%12.4f\t%12.4f\t%12.4f\t%12.4f\t%12.4f\n", 
	      count++, Delta, SumGW, SumUW, fnet, TotWMove, TotStuff/(SumGW + SumUW) );
    fflush (ffile1);

}


/**************************************************************************/
