/**************************************************************************/
// Hydrologic functions used to distribute surface water horizontally
//@ Alexey Voinov -- voinov@cbl.umces.edu
/**************************************************************************/

#include "SWater.h"

#define MAXCELLS 200 //number of cells that can be considered in open water
/*
#define ROWOUT 118  // row and column for an output cell
#define COLOUT 80
#define OPENW 1  //category for open water cells
#define Nmax 1  //max number of cells engaged in the equilibration algorithm
#define transport_st 0.0001  //stage needed to transport stuff horizontally - was 0.001
#define flux_parm 1.0  //what portion of available water can be fluxed downhill

#define MAXDRUN 40 //was 100(150) for the 200 m resolution; 20 for 1km; 700 for 30 m 
#define HSHEAD 0.0025  //squared half-saturation head 0.00003 - 0.00004
#define WATLEV 20 //water level at outflow from area (Elev+Available water) - 
                  //eventually should be a timeseries - may be set to datum to MSL
#define NDIF 0.1 //portion of N on the surface that is available for horizontal runoff -was 0.5
*/


/***********************************************************************

void SWTransML( CVariable& AvailWater, CVariable& outFluxMap, CVariable& HabMap, 
	CVariable& Elev, CVariable& Nitrogen )

// combines fluxing over steep areas and fluxes in open water
// same as before but modified for concentrations instead of stocks
// never had a chance to actually test it
	
{	
	static FILE *file, *fileN, *fileIn;
	static int date = 0;
	static int numout;
	static float TotalOut = 0.;
	float TotNout = 0.0;   // the amount of stuff that is fluxed out from the last cell
	float Nconc, Ntot;
	
	int niter = 30;
	int pd = 1;
	float flow_rate = 1.;
	float df, w_sum, ftmp, move;
	Pix p, rp;
	Pix rPix[MAXCELLS];
	
	int il;
	int rr, rc, NN;
	float MAXDRUN, HSHEAD, WATLEV, NDIF;


	DistributedGrid& grid = AvailWater.Grid();
	grid.SetPointOrdering(0);	                 // Sets grid ordering to default
	
	if (fileIn == NULL) 
	{
		CPathString pathName(Env::ArchivePath()); 
    		pathName.Add("SWData");
		if ( (fileIn = fopen (pathName, "r")) == NULL)  gFatal( "Can't open SWData file! " );
		fscanf ( fileIn, "%f %f %f %f\n", &MAXDRUN, &HSHEAD, &WATLEV, &NDIF );
	}

	if (fileN == NULL) 
	{ // open output file
		CPathString pathName(Env::ArchivePath()); 
    		pathName.Add("NOut");
		if ( (fileN = fopen (pathName, "w")) == NULL) gFatal( "Can't open NOut file! " );
	}

	if (file == NULL) 
	{ // open output file
		CPathString pathName(Env::ArchivePath()); 
    		pathName.Add("WSOut");
		if ( (file = fopen (pathName, "w")) == NULL) gFatal( "Can't open WSOut file! " ) ; 
	  
	  // prepare output file header
	  	numout = 0;
                fprintf ( file, "%d %f\n", niter, flux_parm );

	  	for( p = grid.first(); p; grid.next(p) )
		{
			const OrderedPoint& pt = grid.GetPoint(p);
			                                      
			if( outFluxMap(pt) == 3 || outFluxMap(pt) == 10 ) 
			{
				int ir0 = pt.row(), ic0 = pt.col();
				fprintf ( file, "\t%5dx%d", ir0, ic0 );
				fprintf ( fileN, "\t%5dx%d", ir0, ic0 );
				numout++;
			}
		}
                fprintf ( file, "\tTotal Outflow" );
		fprintf ( fileN, "\t N Outflux\t N out av.\tTotal N" );
		
	}

	fprintf ( file, "\n%d", date );
	fprintf ( fileN, "\n%d", date++ );
	
	static CVariable* swFlux = NULL;	
	if(swFlux == NULL ) { swFlux = AvailWater.GetSimilarVariable("SWFlux"); }
	    // intermediate increment to stage

	static CVariable* NFlux = NULL;	
	if(NFlux == NULL ) { NFlux = AvailWater.GetSimilarVariable("NFlux"); }
	    // intermediate increment to nutrient
	                      
	static CVariable* swTot = NULL;	
	if(swTot == NULL ) { swTot = AvailWater.GetSimilarVariable("SWTot"); }
	    // array that is used to accumulate water fluxed through gaging points
	
	swTot -> Set(0.0);
	                      
	AvailWater.LinkEdges();
	swFlux->Set(0.0);
	NFlux->Set(0.0);
	
// loop over the whole area, but skipping the open water cells	
	for( p = grid.first(); p; grid.next(p) )               
	  { 
	     const OrderedPoint& pt = grid.GetPoint(p);     // pt same as p, but ordered
	     float& w0 = AvailWater(pt);
	     float nf = 0.;
	     
	     if (HabMap(pt) == OPENW) continue;
	     //water flow in terrestrial areas

	     df = w0*flux_parm;
	     (*swFlux)(pt) -= df;

             Pix rp = p;
	    	
	        // making the length of flow path dependent of the amount of water to move
	        // niter = 1+30*pow(w0,0.05); printf ("\n   %d  %f", niter, w0);
	
	     float stt = w0*w0;
	     niter = MAXDRUN*stt/(stt+HSHEAD);  
	    		    	
	     for(int iter=0; iter<niter && rp; iter++) 
	        { // make niter iterations to run water further downhill
	           rp = grid.LinkedPix( rp );           // gets downstream link rp
		   if ( rp )
		   {
			const OrderedPoint& rpt = grid.GetPoint(rp);
						
			if (outFluxMap(rpt) == 3 || outFluxMap(rpt) == 10) 
          		       (*swTot)(rpt) += df;
                               // 3 and 10 mark the gaging stations on StudyArea map.
                               // this is to sum up all the water that is flushed
                               // through the cells with gaging stations
			
          		if (outFluxMap(rpt) == 10)  TotNout += Nitrogen(pt)*df;  
          			//cell_size weighted amount of N fluxed through the end cell
          		if (HabMap(rpt) == OPENW) break;
                        
		   }
		} // end loop over cells along the path of flow
		   		
                // the following is when everything is dumped into one last recipient cell
	     if ( rp ) 
		{  
		 const OrderedPoint& rpt = grid.GetPoint(rp);
		 (*swFlux)(rpt) += df;
		 if (w0>transport_st) 
		    (*NFlux)(rpt) += (NDIF*Nitrogen(pt)-Nitrogen(rpt))*df/(AvailWater(rpt)+df);
		}
	  } //end loop over all area
			
	AvailWater.AddData(*swFlux);
	Nitrogen.AddData(*NFlux);
	
// another loop over the whole area, but only for open water cells	
	for( p = grid.first(); p; grid.next(p) )               
	  { 
	     const OrderedPoint& pt = grid.GetPoint(p);     // pt same as p, but ordered
	     if (HabMap(pt) != OPENW) continue;

	     // water flow in Open Water regions
             w_sum = Elev(pt)+AvailWater(pt);
	     int count = 0;  // number of open water cells
               
             // calculate the average water level
             for (NN = 1; count < Nmax; NN++)  //NN is number of concentric layer
		  { 
		    for (rr = -NN; rr <= NN; rr++)
		      for (rc = -NN; rc <= NN; rc += 2*NN)
		        
		      {
		        rp = grid.TranslateByGridCoord( p, rr, rc );

		        if( rp )  
		        {
		          const OrderedPoint& rpt = grid.GetPoint(rp);
		          if (HabMap(rpt) == OPENW) 
		            { w_sum += Elev(rpt)+AvailWater(rpt);
		              
  		              rPix[count++] = rp;
		            }  
		        } 
		      }

		    for (rc= -NN+1; rc<NN; rc++)
		      for (rr = -NN; rr<=NN; rr+=2*NN)
		      {  
		        rp = grid.TranslateByGridCoord( p, rr, rc );

		        if( rp )  
		        {
		          const OrderedPoint& rpt = grid.GetPoint(rp);
		          if (HabMap(rpt) == OPENW) 
		            { w_sum += Elev(rpt)+AvailWater(rpt);
		              
  		              rPix[count++] = rp;
		            }  
		        } 
		      }
		   } // end loop over concentric neighborhood 

	      float h0 = w_sum/(count+1);
	        
	        // make sure that the average water level does not expose ground
	      if (h0 < Elev(pt)) h0 -= (Elev(pt)-h0)/count;
	        
	      for( il = 0; il < count; il++ ) 
		 { 
		  rp = rPix[il];
		  const OrderedPoint& rpt = grid.GetPoint(rp);

 	          if (h0 < Elev(rpt)) h0 -= (Elev(rpt)-h0)/(count+1);
		 }
		// we can still potentially generate water in this case !!!
				  
		// equilibrate  
	      for( il = 0; il < count; il++ ) 
		 { 
		  rp = rPix[il];
		  const OrderedPoint& rpt = grid.GetPoint(rp);
		  
		  ftmp = ( h0 - (Elev(rpt)+AvailWater(rpt)) ) * flow_rate;
		  if (-ftmp > AvailWater(rpt)) ftmp = -AvailWater(rpt);
		      
		      // for exchange of stuff, I assume that the transport is occuring via the center
		      // point pt. So that at each step the stuff is moved from each of the 
		      // vicinity cells to/from the center point. This can potentially drive the
		      // amount of stuff in the center point below zero, but at the end of the loop
		      // over the vicinity it should be back to normal.
		     
		  if (ftmp > AvailWater(pt)) ftmp = AvailWater(pt);
		  AvailWater(rpt) += ftmp; 
		  AvailWater(pt) -= ftmp;
		  
		  if (ftmp > 0.)
		     Nitrogen(rpt) += (Nitrogen(pt)-Nitrogen(rpt))*ftmp/AvailWater(rpt);
		  if (ftmp < 0.) 
		     Nitrogen(pt) += (Nitrogen(pt)-Nitrogen(rpt))*ftmp/AvailWater(pt);
         
       		 } // end for equilibration loop
       		        		 
//printf ("\ninfo:  %d  %d %f  %f %f", pt.row(), pt.col(), Elev(pt)+AvailWater(pt), AvailWater(pt));
    
		//take care of the outflow from estuary
	      if (outFluxMap(pt) == 10 && (ftmp = Elev(pt)+AvailWater(pt)-WATLEV) > 0) 
//this should be modified if we are to take care of tides. Then WATLEV becomes variable and 
//the ftmp<0 should be also considered
		 { if (ftmp > AvailWater(pt)) ftmp = AvailWater(pt);
		   //if new level is below the surface level, we remove all the water
		   AvailWater(pt) -= ftmp;
		   (*swTot)(pt) += ftmp;

		   TotNout += Nitrogen(pt)*ftmp;  //cellsize weighed amount of nitrogen removed
		 }  
		  	     
	     } //end loop for OPEN Water areas
	     

	
	
// printing results
	int nout = 0;

	Ntot = 0.0;  // total cell-weighted amount of N in the study area
	float NamountOut = 0.0;
		
	for( p = grid.first(); p; grid.next(p) )
		{
			const OrderedPoint& pt = grid.GetPoint(p);                                      
	                Ntot += Nitrogen(pt)*AvailWater(pt);
			if( outFluxMap(pt) == 3 || outFluxMap(pt) == 10 ) 
				 fprintf ( file, "\t%12.6f", (*swTot)(pt) );
			if( outFluxMap(pt) == 3 ) 
				 fprintf ( fileN, "\t%12.6f", Nitrogen(pt));
			if ( outFluxMap(pt) == 10 ) 
				{ TotalOut += (*swTot)(pt); 
					// total discharge from watershed
				  
				  NamountOut += Nitrogen(pt)*AvailWater(pt);
				  nout++;
				}
		}
	
	NamountOut /= nout;
	fprintf ( fileN, "\t%12.6f\t%12.6f\t%12.6f", TotNout, NamountOut, Ntot );
	fprintf ( file, "\t%12.4f", TotalOut );
	
	fflush(file); 
	fflush(fileN); 
}
**********************************************************************/
