/*
 * file:    compmaps.c
 * project: LOSEEM Library for optimization of spatially explicit ecosystem models
 * author:  r.s.
 * date:    10/2001
 * aim:     
 *    multiscape comparison of grid maps using ArcView Ascii Map
 *    format function for comparision of two grid based maps with a
 *    discrete attribute based on Costanza (1989): Model Goodness of
 *    Fit: A multiple resoloution Procedure. In: Ecological
 *    Modelling 47(1989) 199-215 
 *    ...and a couple of modification by   r.seppelt */


#include "smeopt.h"
#include <math.h>
#include <stdio.h>


/* WindowsDistribution 
 *=================== 
 * calculates distribution of habitat id's for a given window with
 * size w at location i,j return number of cells which are different
 * from nodata cells in given windows. */

int window_distribution(hab, i, j, ncols, nrows, w, nodata, a)
     int hab[MAX_ROWS][MAX_COLS];     
     int i,j, w, nodata;
     int nrows, ncols;
     int a[MAX_CLASSES];
{
  int x, y;
  int no;

  no = 0;

  for(x=0; x<MAX_CLASSES; x++)
    a[x] = 0;			/* reset a */

  /* comment: arithmetic if used: 
   *  ( cond ? return-value on true : return value on false) */
 
  for(x=i; x<(i+w<nrows ? i+w : nrows); x++)
    for(y=j; y<(j+w<ncols ? j+w : ncols); y++)
      if (hab[x][y] != nodata)
	{
	  a[hab[x][y]]++;	       
	  no++;
	}
  return no;
}


/* CompareMaps
 * ===========
 * Compares two grid based based using a moving window of size w,
 * returns overall distance of the maps in the function value and a
 * comparison map in map comp.  In: hab0, hab1 integer maps
 * (categorized data) nrows, ncols integer number of rows and columns
 * nodata nodata tag (integer) w window size (int) comp float map */

float compare_maps(hab0, hab1, ncols, nrows, nodata, w, comp)
     int hab0[MAX_ROWS][MAX_COLS];     
     int hab1[MAX_ROWS][MAX_COLS];     
     float comp[MAX_ROWS][MAX_COLS];     
     int nrows;
     int ncols;
     int nodata;
     int w;
{
  int i,j,k,no1,no2;
  int a0[MAX_CLASSES];
  int a1[MAX_CLASSES];
  float sum, cnt;
  int aux1;
  int cell_valid;

  for(i=0; i<nrows; i++)
    for(j=0; j<ncols; j++)
      {
	no1 = window_distribution(hab0, i, j, ncols, nrows, w, nodata, a0);
	no2 = window_distribution(hab1, i, j, ncols, nrows, w, nodata, a1);

	if (no1 != no2)
	  {
	    printf("maps differ in nodata entries, aboart.\n");
	    exit(1);
	  }
	
	aux1 = 0;
	cell_valid = 0;
	for (k=0; k<MAX_CLASSES; k++)
	  {
	    cell_valid = ((a0[k]>0) || (a1[k]>0) || (cell_valid));
	    aux1 += abs(a0[k] - a1[k]);
	  }
	if (cell_valid)
	  comp[i][j] = (float)aux1/(2.0*(float)no1);
	else
	  comp[i][j] = NODATA;
	/* comp may be used as float Ascii-ArcView grid for visualsation
	 * of certain window comparision. */
      }
  
  sum = 0.0;
  cnt = 0.0;
  
  for(i=0; i<nrows; i++)
    for(j=0; j<ncols; j++)
      if (fabs(comp[i][j])<=1.0) /* ignore invalid or nodata cells */
	{
	  sum += comp[i][j];
	  cnt += 1.0;
	}
  return (sum/cnt);
}



/* map_distance
 * ===========
 * return thre scalar map distance values rho_o, rho_inf, rho_int and a function of multiscale map
 * comparision rho(w). w denotes the size of the movin window scanning the two maps given for
 * comparison hab0, hab1. hab0 and hab1 are assume to be discrete (integer) maps.
 * in: hab0, hab1 maps to comapere
 *     ncols, nrow number of columns and rows
 *     nodata identufer for no data entries in hab0 and hab1
 *     frac fraction of map size to be used for maximum windows size max_w. frac =  1
 *         compares entire distribution with w=max_w. For less computational effort one can chose frac<1
 * out: rho
 *      rho_0 = rho[0]
 *      rho_inf = rho[max_w]
 *      rho_int = sum rho(w)/max_w
 *      max_w = min(ncols, nrows)*frac
 *      rho_min = rho[w_min], rho[w_min] < rho[w] for all 0<w<w_max*frac
 *
 * assumption:
 * - hab0 and hab1 are identical with respect to nodata entries
 */

void map_distance(hab0, hab1, ncols, nrows, nodata, rho, max_w, rho_0, rho_inf, rho_int, rho_min, min_w, frac)
     int hab0[MAX_ROWS][MAX_COLS];     
     int hab1[MAX_ROWS][MAX_COLS];     
     int nrows;
     int ncols;
     int nodata;
     float rho[MAX_MAP_SIZE];
     int *max_w, *min_w;
     float *rho_0, *rho_inf, *rho_int, *rho_min;
     float frac;
{
  int w;
  float sum;
  float comp[MAX_ROWS][MAX_COLS];

  /* Estimate maximum length/width of map, return it as max_w */
  *max_w = nrows;
  if (*max_w>ncols)
    *max_w = ncols;
  
  *max_w = (int)((float)*max_w * frac);
  /* generate max_w moving windows, estimate rho_w(map0, map1) */
  sum = 0.0;
  *rho_min = 1.0;
  for (w=1; w<*max_w; w++)
    {
      rho[w]= compare_maps(hab0, hab1, ncols, nrows, 
			   nodata, w, comp);
      sum += rho[w];
      if (rho[w] < *rho_min)
	{
	  *rho_min = rho[w];
	  *min_w = w;
	}
    }

  /* derive characteristic values from rho(w) */
  *rho_0 = rho[1];
  *rho_inf = rho[*max_w-1];
  *rho_int = sum/(*max_w-1);

}
