/**************************************************************************/
// Hydrologic functions used to distribute surface water horizontally
//@ Alexey Voinov -- voinov@cbl.umces.edu
/**************************************************************************/

#include "SWater.h"

//*************************************************************************
// Simplest fluxing procedure over steep areas, assuming that all the water 
// is moved downhill. Also moves 'Stuff' together with water

// Called from SURFACE_WATER variable. AvailWater is SURFACE_WATER
// HydMap - map of streams (usually HYDRO). It has the mouth of the river
//   marked by MOUTH (read from SWdata file) and stations where to monitor 
//   marked by STAT (read from SWdata file).
// Stuff - is the stuff to be moved with water

void SWTransport2( CVariable& AvailWater, CVariable& HydMap, CVariable& Stuff )
	
{	
	static FILE *file, *fileN, *fileIn;
	static int date = 0;
	float TotalOut = 0.;
	float TotNout = 0.;   // the amount of stuff that is fluxed out from the last cell
	float Nconc, Ntot;
	
	float w_sum, ftmp, move;
	Pix p, rp;

	static float MAXDRUN, HSHEAD, WATLEV, NDIF, transport_st, flux_parm;	
	static int  Nmax, OPENW, GSTAT, MOUTH, ROWOUT, COLOUT, ONOFF;

	char ccc[1000];


	DistributedGrid& grid = AvailWater.Grid();
	grid.SetPointOrdering(0);	                 // Sets grid ordering to default

// Read control Data from file SWData
	if (fileIn == NULL) 
	{
		CPathString pathName(Env::DataPath()); 
    		pathName.Add("SWData");
		if ( (fileIn = fopen (pathName, "r")) == NULL)  gFatal( "Can't open SWData file! " );
		
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &MAXDRUN );	//number of iterations across the area over 1 time step
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &HSHEAD );	//dummy parm here
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &WATLEV );	//dummy parm here	
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &NDIF );

		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &OPENW );	//dummy parm here
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &GSTAT );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &MOUTH );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &Nmax );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &transport_st );

		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &flux_parm );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &ROWOUT );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &COLOUT );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &ONOFF );	//switches function on/off
	}

	if (!ONOFF ) return;

	if (fileN == NULL) 
	{ // open output file for stuff
		CPathString pathName(Env::ArchivePath()); 
		pathName.Add("NOut");
		if ( (fileN = fopen (pathName, "w")) == NULL) gFatal( "Can't open NOut file! " ); 
	}

	if (file == NULL) 
	{ // open output file for water
		CPathString pathName(Env::ArchivePath()); 
		pathName.Add("WSOut");
		if ( (file = fopen (pathName, "w")) == NULL) gFatal( "Can't open WSOut file! " ); 
	  
	  // prepare output file header
		fprintf ( file, "transport_st=%f flux_parm=%f MAXDRUN=%f HSHEAD=%f WATLEV=%f NDIF=%f\n", 
			      transport_st, flux_parm, MAXDRUN, HSHEAD, WATLEV, NDIF );

		for( p = grid.first(); p; grid.next(p) )
		{
			const OrderedPoint& pt = grid.GetPoint(p);
			                                      
			if( HydMap(pt) == GSTAT ) 
			{
				int ir0 = pt.row(), ic0 = pt.col();
				fprintf ( file, "\t%10dx%d\t", ir0, ic0 );
				fprintf ( fileN, "\t%10dx%d\t", ir0, ic0 );
			}
	        
			if ( pt.row()==ROWOUT && pt.col()==COLOUT)
               			fprintf ( fileN, "\t%10dx%d ", ROWOUT, COLOUT );
			
		}
        
        fprintf ( file, "\tTotal Outflow from area" );
        fprintf ( fileN, "\t Stuff flux from area\t Conc at outflow\t Total Stuff in area" );
		
	}

	fprintf ( file, "\n%d", date );
	fprintf ( fileN, "\n%d", date++ );
	
	static CVariable* swFlux = NULL;	
	if(swFlux == NULL ) { swFlux = AvailWater.GetSimilarVariable("SWFlux"); }
	    // intermediate increment to stage

	static CVariable* NFlux = NULL;	
	if(NFlux == NULL ) { NFlux = AvailWater.GetSimilarVariable("NFlux"); }
	    // intermediate increment to stuff
	                      
	static CVariable* NTot = NULL;	
	if(NTot == NULL ) { NTot = AvailWater.GetSimilarVariable("NTot"); }
	    // array that is used to accumulate stuff fluxed through gaging points

	static CVariable* swTot = NULL;	
	if(swTot == NULL ) { swTot = AvailWater.GetSimilarVariable("SWTot"); }
	    // array that is used to accumulate water fluxed through gaging points
	
	swTot -> Set(0.0);
	NTot -> Set(0.0);
	                      
	AvailWater.LinkEdges();
	Stuff.LinkEdges();

// make MAXDRUN iterations per function call to run water further downhill
	
	for (int iter=0; iter < MAXDRUN; iter++) 
	{ 
	  swFlux->Set(0.0);
	  NFlux->Set(0.0);

      // loop over the whole area
	  for ( p = grid.first(); p; grid.next(p) )               
	  { 
		const OrderedPoint& pt = grid.GetPoint(p);     // pt same as p, but ordered
		float& w0 = AvailWater(pt);
		float nf = 0.;

		if (w0 > transport_st)  df = w0*flux_parm;

		(*swFlux)(pt) -= df;

		if (w0 > transport_st)  nf = NDIF*Stuff(pt)*df/w0;
		(*NFlux)(pt) -= nf;
	           
		rp = grid.LinkedPix( p );           // gets downstream link rp

		if ( rp )		// if next cell is on map
		{
			const OrderedPoint& rpt = grid.GetPoint(rp);
						
			(*swFlux)(rpt) += df;
		 	(*NFlux)(rpt) += nf;

			if (HydMap(rpt) == GSTAT)
                   // GSTAT marks the gaging stations on River map.
                   // This is to sum up all the water and all the material
			       // that is moved through the cells with gaging stations
			{
				(*swTot)(rpt) += df;
				(*NTot)(rpt) += nf;
			}
          		if (HydMap(rpt) == MOUTH)  
                   // MOUTH marks the mouth on River map
         		{
          			TotNout += nf;	// amount removed through the MOUTH point
          			TotalOut += df;
          		}                        
		 }

	  } //end loop over all area

	  AvailWater.AddData(*swFlux);
	  Stuff.AddData(*NFlux);

	} //end loop over number of iterations
	
// printing results
	Ntot = 0;
		
	for( p = grid.first(); p; grid.next(p) )
	{
		const OrderedPoint& pt = grid.GetPoint(p);  
		Ntot += Stuff(pt);	// total amount of stuff in the area

		if( HydMap(pt) == GSTAT )
		{ 
			fprintf ( file, "\t%12.6f", (*swTot)(pt) );
			fprintf ( file, "\t%12.6f", AvailWater(pt) );
			fprintf ( fileN, "\t%12.6f", (*NTot)(pt) );
			fprintf ( fileN, "\t%12.6f", Stuff(pt) );
		}
		
		if ( pt.row()==ROWOUT && pt.col()==COLOUT)
	                	fprintf ( fileN, "\t%12.6f", Stuff(pt) );
	}
	Nconc = (TotalOut > 0) ? TotNout/TotalOut : 0;

	fprintf ( fileN, "\t%12.6f\t%12.6f\t%12.6f", TotNout, Nconc, Ntot );
	fprintf ( file, "\t%12.4f", TotalOut );
	
	fflush(file); 
	fflush(fileN); 
}


/***********************************************************************/

