/**************************************************************************/
// Hydrologic functions used to distribute surface water horizontally
//@ Alexey Voinov -- voinov@cbl.umces.edu
/**************************************************************************/

#include "SWater.h"

#define MAXCELLS 200 //number of cells that can be considered in open water

float n_out_sum, n_out_conc;

void SWTransport2( CVariable& AvailWater, CVariable& HydMap, CVariable& HabMap, 
	CVariable& Elev, CVariable& Slop, CVariable& Nitrogen )

// combines fluxing over steep areas and fluxes in open water
	
{	
	static FILE *file, *fileN, *fileIn;
	static int date = 0;
	static int numout;
	float TotalOut = 0.;
	float TotNout = 0.0;   // the amount of stuff that is fluxed out from the last cell
	float Nconc, Ntot;
	
	int niter = 30;
	int pd = 1;
	float flow_rate = 1.;
	float df, w_sum, ftmp, move;
	Pix p, rp;
	Pix rPix[MAXCELLS];

	static float MAXDRUN, HSHEAD, WATLEV, NDIF, transport_st, flux_parm;	
	static int  Nmax, OPENW, GSTAT, MOUTH, ROWOUT, COLOUT, ONOFF;

	int il;
	int rr, rc, NN;
	char ccc[1000];


	DistributedGrid& grid = AvailWater.Grid();
	grid.SetPointOrdering(0);	                 // Sets grid ordering to default

	if (fileIn == NULL)   // Read Data from file /data/SWData
	{
		CPathString pathName(Env::DataPath()); 
    		pathName.Add("SWData");
		if ( (fileIn = fopen (pathName, "r")) == NULL)  gFatal( "Can't open SWData file! " );
		
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &MAXDRUN );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &HSHEAD );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &WATLEV );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &NDIF );

		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &OPENW );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &GSTAT );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &MOUTH );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &Nmax );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &transport_st );

		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &flux_parm );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &ROWOUT );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &COLOUT );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &ONOFF );
	}

	if (!ONOFF ) return;

	if (fileN == NULL) 
	{ // open output file
		CPathString pathName(Env::ArchivePath()); 
    		pathName.Add("NOut");
		if ( (fileN = fopen (pathName, "w")) == NULL) gFatal( "Can't open NOut file! " ) ; 
	}

	if (file == NULL) 
	{ // open output file
		CPathString pathName(Env::ArchivePath()); 
    		pathName.Add("WSOut");
		if ( (file = fopen (pathName, "w")) == NULL) gFatal( "Can't open WSOut file! " ) ; 
	  
	  // prepare output file header
	  	numout = 0;
                fprintf ( file, "transport_st=%f flux_parm=%f MAXDRUN=%f HSHEAD=%f WATLEV=%f NDIF=%f\n", 
			  transport_st, flux_parm, MAXDRUN, HSHEAD, WATLEV, NDIF );

	  	for( p = grid.first(); p; grid.next(p) )
		{
			const OrderedPoint& pt = grid.GetPoint(p);
			                                      
			if( HydMap(pt) == GSTAT ) 
			{
				int ir0 = pt.row(), ic0 = pt.col();
				fprintf ( file, "\t%5dx%d\t", ir0, ic0 );
				fprintf ( fileN, "\t%5dx%d\t", ir0, ic0 );
				numout++;
			}
	                if ( pt.row()==ROWOUT && pt.col()==COLOUT)
               			fprintf ( fileN, "\t%5dx%d\t ", ROWOUT, COLOUT );
			
		}
                fprintf ( file, "\tTotal Outflow" );
                fprintf ( fileN, "\t N Outflux\t N conc out\t Ntotarea" );
		
	}

	fprintf ( file, "\n%d", date );
	fprintf ( fileN, "\n%d", date++ );
	
	static CVariable* swFlux = NULL;	
	if(swFlux == NULL ) { swFlux = AvailWater.GetSimilarVariable("SWFlux"); }
	    // intermediate increment to stage

	static CVariable* NFlux = NULL;	
	if(NFlux == NULL ) { NFlux = AvailWater.GetSimilarVariable("NFlux"); }
	    // intermediate increment to nutrient
	                      
	static CVariable* NTot = NULL;	
	if(NTot == NULL ) { NTot = AvailWater.GetSimilarVariable("NTot"); }
	    // intermediate increment to nutrient

	static CVariable* swTot = NULL;	
	if(swTot == NULL ) { swTot = AvailWater.GetSimilarVariable("SWTot"); }
	    // array that is used to accumulate water fluxed through gaging points
	
	swTot -> Set(0.0);
	NTot -> Set(0.0);
	                      
	AvailWater.LinkEdges();
	swFlux->Set(0.0);
	NFlux->Set(0.0);
	
// loop over the whole area, but skipping the open water cells	
	for( p = grid.first(); p; grid.next(p) )               
	  { 
	     const OrderedPoint& pt = grid.GetPoint(p);     // pt same as p, but ordered
	     float& w0 = AvailWater(pt);
	     float nf = 0.;

	     if (HabMap(pt) == OPENW && HydMap(pt) != 4) continue;
	     //water flow in terrestrial areas and those that are outlets from reservoirs

	     df = w0*flux_parm;

	     (*swFlux)(pt) -= df;

	     if (w0>transport_st)  nf = NDIF*Nitrogen(pt)*flux_parm;
			// it should be *df/w0, which is in fact = flux_parm
		
	     (*NFlux)(pt) -= nf;

             Pix rp = p;
	    	
	        // making the length of flow path dependent of the amount of water to move
	        // niter = 1+30*pow(w0,0.05); printf ("\n   %d  %f", niter, w0);
	
	     float stt = w0*w0; 
	     float godo = MAXDRUN*stt/(stt+HSHEAD);
	     niter = godo;
	    		    	
	     for(int iter=0; iter<niter && rp; iter++) 
	        { // make niter iterations to run water further downhill
	           rp = grid.LinkedPix( rp );           // gets downstream link rp

		   if ( rp )
		   {
			const OrderedPoint& rpt = grid.GetPoint(rp);
						
			if (HydMap(rpt) == GSTAT)
                               // GSTAT mark the gaging stations on River map.
                               // this is to sum up all the water and all the 
			       // materialthat is moved
                               // through the cells with gaging stations
				{
				  (*swTot)(rpt) += df;
				  (*NTot)(rpt) += nf;
				}
          		if (HydMap(rpt) == MOUTH)  
                                // MOUTH mark the mouth on River map.
         			{
          			  TotNout += nf;
          			  TotalOut += df;
          			}
          		if (HabMap(rpt) == OPENW) break;
                        
		   }
		} // end loop over cells along the path of flow
		   		
                // the following is when everything is dumped into one last recipient cell
		if ( rp ) 
		{  
		 	const OrderedPoint& rpt = grid.GetPoint(rp);
		 	(*swFlux)(rpt) += df;
		 	if (w0>transport_st) (*NFlux)(rpt) += nf;
		}
	  } //end loop over all area
			
	AvailWater.AddData(*swFlux);
	Nitrogen.AddData(*NFlux);

// another loop over the whole area, but only for open water cells	
	for( p = grid.first(); p; grid.next(p) )               
	  { 
	     const OrderedPoint& pt = grid.GetPoint(p);     // pt same as p, but ordered
	     if (HabMap(pt) != OPENW) continue;

	     // water flow in Open Water regions
             w_sum = Elev(pt)+AvailWater(pt);
	     int count = 0, ow = 1;  // number of open water cells, presence of open water in the vicinity

             // calculate the average water level
	     // loops until either the max allowed number of cells reached, 
	     // or no adjacent water cell present in the vicinity
             for (NN = 1; count < Nmax-1 && ow; NN++)  //NN is number of concentric layer
		  { 
		   ow = 0; 
		   for (rr = -NN; rr <= NN; rr++)
		      for (rc = -NN; rc <= NN; rc += 2*NN)
		        
		      {
		        rp = grid.TranslateByGridCoord( p, rr, rc );

		        if( rp )  
		        {
		          const OrderedPoint& rpt = grid.GetPoint(rp);
		          if (HabMap(rpt) == OPENW) 
		            { w_sum += Elev(rpt)+AvailWater(rpt);
		              ow = 1;
  		              rPix[count++] = rp;
		            }  
		        } 
		      }

		    for (rc= -NN+1; rc<NN; rc++)
		      for (rr = -NN; rr<=NN; rr+=2*NN)
		      {  
		        rp = grid.TranslateByGridCoord( p, rr, rc );

		        if( rp )  
		        {
		          const OrderedPoint& rpt = grid.GetPoint(rp);
		          if (HabMap(rpt) == OPENW) 
		            { w_sum += Elev(rpt)+AvailWater(rpt);
		              ow = 1;
  		              rPix[count++] = rp;
		            }  
		        } 
		      }
		   } // end loop over concentric neighborhood 

	      float h0 = w_sum/(count+1);
	        
	        // make sure that the average water level does not expose ground
	      if (h0 < Elev(pt)) h0 -= (Elev(pt)-h0)/count;
	        
	      for( il = 0; il < count-1; il++ ) 
		 { 
		  rp = rPix[il];
		  const OrderedPoint& rpt = grid.GetPoint(rp);

 	          if (h0 < Elev(rpt)) h0 -= (Elev(rpt)-h0)/count;
		 }
		// we can still potentially generate water in this case !!!	
				  
		// equilibrate  
	      for( il = 0; il < count; il++ ) 
		 { 
		  rp = rPix[il];
		  const OrderedPoint& rpt = grid.GetPoint(rp);
		  
		  ftmp = ( h0 - (Elev(rpt)+AvailWater(rpt)) ) * flow_rate;
		  if (-ftmp > AvailWater(rpt)) ftmp = -AvailWater(rpt);
		      
		  // for exchange of stuff, I assume that the transport is occuring via the center
		  // point pt. So that at each step the stuff is moved from each of the 
		  // vicinity cells to/from the center point. This can potentially drive the
		  // amount of stuff in the center point below zero, but at the end of the loop
		  // over the vicinity it should be back to normal.
		     
		  if (ftmp > AvailWater(pt)) 
		      { ftmp = AvailWater(pt);
		        move = Nitrogen(pt);
		      }
		  else 
		      { if (ftmp > 0.)
		          move = (AvailWater(pt) > 0.) ? Nitrogen(pt)*ftmp/AvailWater(pt) : 0.;
		        else 
		          move = (AvailWater(rpt) > 0.) ? Nitrogen(rpt)*ftmp/AvailWater(rpt) : 0.;
		      }
		          
	          Nitrogen(rpt) += move;
		  AvailWater(rpt) += ftmp; 

		  Nitrogen(pt) -= move;
		  AvailWater(pt) -= ftmp;
		  
		 } // end for equilibration loop
// printf ("\n%d  %d %f  %f %f", pt.row(), pt.col(), Elev(pt)+AvailWater(pt), AvailWater(pt));
    
		//take care of the outflow from estuary
		if (HydMap(pt) == MOUTH && (ftmp = Elev(pt)+AvailWater(pt)-WATLEV) > 0) 
			{ if (ftmp > AvailWater(pt)) ftmp = AvailWater(pt);
			  //if new level is below the water level, we remove all the water
			  if (AvailWater(pt)>0)
			       move = Nitrogen(pt)*ftmp/AvailWater(pt);
			  else move = 0;  
			  Nitrogen(pt) -= move;
			  AvailWater(pt) -= ftmp;
			  TotalOut += ftmp;
			  TotNout += move;
			}  
		  	     
	     } //end loop for OPEN Water areas
	
// printing results
	Ntot = 0;
		
	for( p = grid.first(); p; grid.next(p) )
		{
			const OrderedPoint& pt = grid.GetPoint(p);  
	                Ntot += Nitrogen(pt);

			if( HydMap(pt) == GSTAT )
				{ 
				  fprintf ( file, "\t%12.6f", (*swTot)(pt) );
				  fprintf ( file, "\t%12.6f", AvailWater(pt) );
				  fprintf ( fileN, "\t%12.6f", Nitrogen(pt) );
// think more
				  if (AvailWater(pt)>0 )
				  fprintf ( fileN, "\t%12.6f", Nitrogen(pt)/AvailWater(pt)/15 );
				  else
				  fprintf ( fileN, "\tNaN" );
				}
		         if ( pt.row()==ROWOUT && pt.col()==COLOUT)
	                	fprintf ( fileN, "\t%12.6f", Nitrogen(pt) );
		}
	Nconc = (TotalOut > 0) ? TotNout/TotalOut : 0;

	fprintf ( fileN, "\t%12.6f\t%12.6f\t%12.6f", TotNout, Nconc, Ntot );
	fprintf ( file, "\t%12.4f", TotalOut );
	
	fflush(file); 
	fflush(fileN); 
}
/***********************************************************************/
/***********************************************************************/
void SWTransML_I( CVariable& AvailWater, CVariable& HydMap, CVariable& HabMap, 
	CVariable& Elev, CVariable& Slop, CVariable& Nitrogen, CVariable& outFlux)

// combines fluxing over steep areas and fluxes in open water
// keeps track of the amount of nutrient leaked from each cell
	
{	
	static FILE *file, *fileN, *fileIn;
	static int date = 0;
	static int numout;
	float TotalOut = 0.;
	float TotNout = 0.0;   // the amount of stuff that is fluxed out from the last cell
	float Nconc, Ntot;
	
	int niter = 30;
	int pd = 1;
	float flow_rate = 1.;
	float df, w_sum, ftmp, move;
	Pix p, rp;
	Pix rPix[MAXCELLS];

	static float MAXDRUN, HSHEAD, WATLEV, NDIF, transport_st, flux_parm;	
	static int  Nmax, OPENW, GSTAT, MOUTH, ROWOUT, COLOUT, ONOFF;

	int il;
	int rr, rc, NN;
	char ccc[1000];


	DistributedGrid& grid = AvailWater.Grid();
	grid.SetPointOrdering(0);	                 // Sets grid ordering to default

	if (fileIn == NULL)   // Read Data from file /data/SWData
	{
		CPathString pathName(Env::DataPath()); 
    		pathName.Add("SWData");
		if ( (fileIn = fopen (pathName, "r")) == NULL)  gFatal( "Can't open SWData file! " );
		
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &MAXDRUN );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &HSHEAD );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &WATLEV );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &NDIF );

		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &OPENW );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &GSTAT );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &MOUTH );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &Nmax );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &transport_st );

		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &flux_parm );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &ROWOUT );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &COLOUT );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &ONOFF );
	}

	if (!ONOFF ) return;

	if (fileN == NULL) 
	{ // open output file
		CPathString pathName(Env::ArchivePath()); 
    		pathName.Add("NOut");
		if ( (fileN = fopen (pathName, "w")) == NULL) gFatal( "Can't open NOut file! " ) ; 
	}

	if (file == NULL) 
	{ // open output file
		CPathString pathName(Env::ArchivePath()); 
    		pathName.Add("WSOut");
		if ( (file = fopen (pathName, "w")) == NULL) gFatal( "Can't open WSOut file! " ) ; 
	  
	  // prepare output file header
	  	numout = 0;
                fprintf ( file, "transport_st=%f flux_parm=%f MAXDRUN=%f HSHEAD=%f WATLEV=%f NDIF=%f\n", 
			  transport_st, flux_parm, MAXDRUN, HSHEAD, WATLEV, NDIF );

	  	for( p = grid.first(); p; grid.next(p) )
		{
			const OrderedPoint& pt = grid.GetPoint(p);
			                                      
			if( HydMap(pt) == GSTAT ) 
			{
				int ir0 = pt.row(), ic0 = pt.col();
				fprintf ( file, "\t%5dx%d\t", ir0, ic0 );
				fprintf ( fileN, "\t%5dx%d\t", ir0, ic0 );
				numout++;
			}
	                if ( pt.row()==ROWOUT && pt.col()==COLOUT)
               			fprintf ( fileN, "\t%5dx%d\t ", ROWOUT, COLOUT );
			
		}
                fprintf ( file, "\tTotal Outflow" );
                fprintf ( fileN, "\t N Outflux\t N conc out\t Ntotarea" );
		
	}

	fprintf ( file, "\n%d", date );
	fprintf ( fileN, "\n%d", date++ );
	
	static CVariable* swFlux = NULL;	
	if(swFlux == NULL ) { swFlux = AvailWater.GetSimilarVariable("SWFlux"); }
	    // intermediate increment to stage

	static CVariable* NFlux = NULL;	
	if(NFlux == NULL ) { NFlux = AvailWater.GetSimilarVariable("NFlux"); }
	    // intermediate increment to nutrient
	                      
	static CVariable* NTot = NULL;	
	if(NTot == NULL ) { NTot = AvailWater.GetSimilarVariable("NTot"); }
	    // intermediate increment to nutrient

	static CVariable* swTot = NULL;	
	if(swTot == NULL ) { swTot = AvailWater.GetSimilarVariable("SWTot"); }
	    // array that is used to accumulate water fluxed through gaging points
	
	swTot -> Set(0.0);
	NTot -> Set(0.0);
	                      
	AvailWater.LinkEdges();
	swFlux->Set(0.0);
	NFlux->Set(0.0);
	
// loop over the whole area, but skipping the open water cells	
	for( p = grid.first(); p; grid.next(p) )               
	  { 
	     const OrderedPoint& pt = grid.GetPoint(p);     // pt same as p, but ordered
int irrr=pt.row();
int iccc=pt.col();
	     float& w0 = AvailWater(pt);
	     float nf = 0.;

	     if (date == 0) outFlux(pt) = 0;

	     if (HabMap(pt) == OPENW && HydMap(pt) != 4) continue;
	     //water flow in terrestrial areas and those that are outlets from reservoirs

	     df = w0*flux_parm;

	     (*swFlux)(pt) -= df;

	     if (w0>transport_st)  nf = NDIF*Nitrogen(pt)*flux_parm;
			// it should be *df/w0, which is in fact = flux_parm
		
	     (*NFlux)(pt) -= nf;
	     outFlux(pt) += nf; //accumulate outflow of nutrient from cell

             Pix rp = p;
	    	
	        // making the length of flow path dependent of the amount of water to move
	        // niter = 1+30*pow(w0,0.05); printf ("\n   %d  %f", niter, w0);
	
	     float stt = w0*w0; 
	     float godo = 1 + MAXDRUN*stt/(stt+HSHEAD);
	     niter = godo;
//	     niter = godo*Slop(pt)*Slop(pt);
	     
//  printf("\nniter = %d   godo = %f", niter, godo );
	    		    	
	     for(int iter=0; iter<niter && rp; iter++) 
	        { // make niter iterations to run water further downhill
	           rp = grid.LinkedPix( rp );           // gets downstream link rp

		   if ( rp )
		   {
			const OrderedPoint& rpt = grid.GetPoint(rp);
			
//printf ("\n%d  %d", rpt.row(), rpt.col());
						
			if (HydMap(rpt) == GSTAT)
                               // GSTAT mark the gaging stations on River map.
                               // this is to sum up all the water and all the 
			       // materialthat is moved
                               // through the cells with gaging stations
				{
				  (*swTot)(rpt) += df;
				  (*NTot)(rpt) += nf;
//printf("At GSTAT df = %f coming from %d %d\n", df, irrr, iccc );
					}
          		if (HydMap(rpt) == MOUTH)  
                                // MOUTH mark the mouth on River map.
         			{
          			  TotNout += nf;
          			  TotalOut += df;
//printf("At MOUTH df = %f coming from %d %d\n", df, irrr, iccc );
          			}
          		if (HabMap(rpt) == OPENW) break;
                        
		   }
		} // end loop over cells along the path of flow
		   		
                // the following is when everything is dumped into one last recipient cell
		if ( rp ) 
		{  
		 	const OrderedPoint& rpt = grid.GetPoint(rp);
		 	(*swFlux)(rpt) += df;
		 	if (w0>transport_st) (*NFlux)(rpt) += nf;
		}
	  } //end loop over all area
			
	AvailWater.AddData(*swFlux);
	Nitrogen.AddData(*NFlux);

// another loop over the whole area, but only for open water cells	
	for( p = grid.first(); p; grid.next(p) )               
	  { 
	     const OrderedPoint& pt = grid.GetPoint(p);     // pt same as p, but ordered
	     if (HabMap(pt) != OPENW) continue;

	     // water flow in Open Water regions
             w_sum = Elev(pt)+AvailWater(pt);
	     int count = 0, ow = 1;  // number of open water cells, presence of open water in the vicinity

             // calculate the average water level
	     // loops until either the max allowed number of cells reached, 
	     // or no adjacent water cell present in the vicinity
             for (NN = 1; count < Nmax-1 && ow; NN++)  //NN is number of concentric layer
		  { 
		   ow = 0; 
		   for (rr = -NN; rr <= NN; rr++)
		      for (rc = -NN; rc <= NN; rc += 2*NN)
		        
		      {
		        rp = grid.TranslateByGridCoord( p, rr, rc );

		        if( rp )  
		        {
		          const OrderedPoint& rpt = grid.GetPoint(rp);
		          if (HabMap(rpt) == OPENW) 
		            { w_sum += Elev(rpt)+AvailWater(rpt);
		              ow = 1;
  		              rPix[count++] = rp;
		            }  
		        } 
		      }

		    for (rc= -NN+1; rc<NN; rc++)
		      for (rr = -NN; rr<=NN; rr+=2*NN)
		      {  
		        rp = grid.TranslateByGridCoord( p, rr, rc );

		        if( rp )  
		        {
		          const OrderedPoint& rpt = grid.GetPoint(rp);
		          if (HabMap(rpt) == OPENW) 
		            { w_sum += Elev(rpt)+AvailWater(rpt);
		              ow = 1;
  		              rPix[count++] = rp;
		            }  
		        } 
		      }
		   } // end loop over concentric neighborhood 

	      float h0 = w_sum/(count+1);
	        
	        // make sure that the average water level does not expose ground
	      if (h0 < Elev(pt)) h0 -= (Elev(pt)-h0)/count;
	        
	      for( il = 0; il < count-1; il++ ) 
		 { 
		  rp = rPix[il];
		  const OrderedPoint& rpt = grid.GetPoint(rp);

 	          if (h0 < Elev(rpt)) h0 -= (Elev(rpt)-h0)/count;
		 }
		// we can still potentially generate water in this case !!!	
				  
		// equilibrate  
	      for( il = 0; il < count; il++ ) 
		 { 
		  rp = rPix[il];
		  const OrderedPoint& rpt = grid.GetPoint(rp);
		  
		  ftmp = ( h0 - (Elev(rpt)+AvailWater(rpt)) ) * flow_rate;
		  if (-ftmp > AvailWater(rpt)) ftmp = -AvailWater(rpt);
		      
		  // for exchange of stuff, I assume that the transport is occuring via the center
		  // point pt. So that at each step the stuff is moved from each of the 
		  // vicinity cells to/from the center point. This can potentially drive the
		  // amount of stuff in the center point below zero, but at the end of the loop
		  // over the vicinity it should be back to normal.
		     
		  if (ftmp > AvailWater(pt)) 
		      { ftmp = AvailWater(pt);
		        move = Nitrogen(pt);
		      }
		  else 
		      { if (ftmp > 0.)
		          move = (AvailWater(pt) > 0.) ? Nitrogen(pt)*ftmp/AvailWater(pt) : 0.;
		        else 
		          move = (AvailWater(rpt) > 0.) ? Nitrogen(rpt)*ftmp/AvailWater(rpt) : 0.;
		      }
		          
	          Nitrogen(rpt) += move;
		  AvailWater(rpt) += ftmp; 

		  Nitrogen(pt) -= move;
		  AvailWater(pt) -= ftmp;
		  
		 } // end for equilibration loop
// printf ("\n%d  %d %f  %f %f", pt.row(), pt.col(), Elev(pt)+AvailWater(pt), AvailWater(pt));
    
		//take care of the outflow from estuary
		if (HydMap(pt) == MOUTH && (ftmp = Elev(pt)+AvailWater(pt)-WATLEV) > 0) 
			{ if (ftmp > AvailWater(pt)) ftmp = AvailWater(pt);
			  //if new level is below the water level, we remove all the water
			  if (AvailWater(pt)>0)
			       move = Nitrogen(pt)*ftmp/AvailWater(pt);
			  else move = 0;  
			  Nitrogen(pt) -= move;
			  AvailWater(pt) -= ftmp;
			  TotalOut += ftmp;
			  TotNout += move;
			}  
		  	     
	     } //end loop for OPEN Water areas
	
// printing results
	Ntot = 0;
		
	for( p = grid.first(); p; grid.next(p) )
		{
			const OrderedPoint& pt = grid.GetPoint(p);  
	                Ntot += Nitrogen(pt);
			if( HydMap(pt) == GSTAT )
				{ 
				  fprintf ( file, "\t%12.6f", (*swTot)(pt) );

// this doesn't look quite right. I'm not sure what Ralf calculates here:
//	 			  n_out_sum  = (*swTot)(pt);

				  fprintf ( file, "\t%12.6f", AvailWater(pt) );
				  fprintf ( fileN, "\t%12.6f", Nitrogen(pt) );
// think more
				  if (AvailWater(pt)>0 )
				  fprintf ( fileN, "\t%12.6f", Nitrogen(pt)/AvailWater(pt) );
				  else
				  fprintf ( fileN, "\t        NaN" );


				}
	                if ( pt.row()==ROWOUT && pt.col()==COLOUT)
	                	fprintf ( fileN, "\t%12.6f", Nitrogen(pt) );
		}
	Nconc = (TotalOut > 0) ? TotNout/TotalOut : 0;

	fprintf ( fileN, "\t%12.6f\t%12.6f\t%12.6f", TotNout, Nconc, Ntot );
	fprintf ( file, "\t%12.4f", TotalOut );

  // Handover Values of SWater to global variables (sorry) for printout
  // and further analysis in UCode.cc(Goalfunction)

  n_out_sum = TotNout;
  n_out_conc = Nconc;
	
	fflush(file); 
	fflush(fileN); 
}
/***********************************************************************/

/***********************************************************************/
void SWTransML_I_b( CVariable& AvailWater, CVariable& HydMap, CVariable& HabMap, 
	CVariable& Elev, CVariable& Slop, CVariable& Nitrogen, CVariable& outFlux)

// combines fluxing over steep areas and fluxes in open water
// keeps track of the amount of nutrient leaked from each cell
// calculates the budgets from different areas, designated on the 
// HYDRO map by numbers > 10
	
{	
	static FILE *file, *fileN, *fileIn;
	static int date = 0;
	static int numout;
	float TotalOut = 0.;
	float TotNout = 0.0;   // the amount of stuff that is fluxed out from the last cell
	float Nconc, Ntot;

	float N1 = 0.0, N2 = 0.0, N3 = 0.0;  // to store the amounts coming from the two lower parts of the WShed
	int num00 = 0, num10 = 0, num20 = 0;

	int niter = 30;
	int pd = 1;
	float flow_rate = 1.;
	float df, w_sum, ftmp, move;
	Pix p, rp;
	Pix rPix[MAXCELLS];

	static float MAXDRUN, HSHEAD, WATLEV, NDIF, transport_st, flux_parm;	
	static int  Nmax, OPENW, GSTAT, MOUTH, ROWOUT, COLOUT, ONOFF;

	int il;
	int rr, rc, NN;
	char ccc[1000];


	DistributedGrid& grid = AvailWater.Grid();
	grid.SetPointOrdering(0);	                 // Sets grid ordering to default

	if (fileIn == NULL)   // Read Data from file /data/SWData
	{
		CPathString pathName(Env::DataPath()); 
    		pathName.Add("SWData");
		if ( (fileIn = fopen (pathName, "r")) == NULL)  gFatal( "Can't open SWData file! " );
		
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &MAXDRUN );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &HSHEAD );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &WATLEV );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &NDIF );

		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &OPENW );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &GSTAT );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &MOUTH );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &Nmax );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &transport_st );

		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%f", &flux_parm );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &ROWOUT );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &COLOUT );
		fgets ( ccc, 1000, fileIn); 
		sscanf ( ccc, "%d", &ONOFF );
	}

	if (!ONOFF ) return;

	if (fileN == NULL) 
	{ // open output file
		CPathString pathName(Env::ArchivePath()); 
    		pathName.Add("NOut");
		if ( (fileN = fopen (pathName, "w")) == NULL) gFatal( "Can't open NOut file! " ) ; 
	}

	if (file == NULL) 
	{ // open output file
		CPathString pathName(Env::ArchivePath()); 
    		pathName.Add("WSOut");
		if ( (file = fopen (pathName, "w")) == NULL) gFatal( "Can't open WSOut file! " ) ; 
	  
	  // prepare output file header
	  	numout = 0;
                fprintf ( file, "transport_st=%f flux_parm=%f MAXDRUN=%f HSHEAD=%f WATLEV=%f NDIF=%f\n", 
			  transport_st, flux_parm, MAXDRUN, HSHEAD, WATLEV, NDIF );

	  	for( p = grid.first(); p; grid.next(p) )
		{
			const OrderedPoint& pt = grid.GetPoint(p);
			                                      
			if( HydMap(pt) == GSTAT ) 
			{
				int ir0 = pt.row(), ic0 = pt.col();
				fprintf ( file, "\t%5dx%d\t", ir0, ic0 );
				fprintf ( fileN, "\t%5dx%d\t", ir0, ic0 );
				numout++;
			}
	                if ( pt.row()==ROWOUT && pt.col()==COLOUT)
               			fprintf ( fileN, "\t%5dx%d\t ", ROWOUT, COLOUT );
			
		}
                fprintf ( file, "\tTotal Outflow" );
                fprintf ( fileN, "\t N Outflux\t N conc out\t Ntotarea\t    N2\t    N3" );
		
	}

	fprintf ( file, "\n%d", date );
	fprintf ( fileN, "\n%d", date++ );
	
	static CVariable* swFlux = NULL;	
	if(swFlux == NULL ) { swFlux = AvailWater.GetSimilarVariable("SWFlux"); }
	    // intermediate increment to stage

	static CVariable* NFlux = NULL;	
	if(NFlux == NULL ) { NFlux = AvailWater.GetSimilarVariable("NFlux"); }
	    // intermediate increment to nutrient
	                      
	static CVariable* NTot = NULL;	
	if(NTot == NULL ) { NTot = AvailWater.GetSimilarVariable("NTot"); }
	    // intermediate increment to nutrient

	static CVariable* swTot = NULL;	
	if(swTot == NULL ) { swTot = AvailWater.GetSimilarVariable("SWTot"); }
	    // array that is used to accumulate water fluxed through gaging points
	
	swTot -> Set(0.0);
	NTot -> Set(0.0);
	                      
	AvailWater.LinkEdges();
	swFlux->Set(0.0);
	NFlux->Set(0.0);
	
// loop over the whole area, but skipping the open water cells	
	for( p = grid.first(); p; grid.next(p) )               
	  { 
	     const OrderedPoint& pt = grid.GetPoint(p);     // pt same as p, but ordered
int irrr=pt.row();
int iccc=pt.col();
	     float& w0 = AvailWater(pt);
	     float nf = 0.;

	     if (date == 0) outFlux(pt) = 0;

	     if (HabMap(pt) == OPENW && HydMap(pt) != 4) continue;
	     //water flow in terrestrial areas and those that are outlets from reservoirs

	     df = w0*flux_parm;

	     (*swFlux)(pt) -= df;

	     if (w0>transport_st)  nf = NDIF*Nitrogen(pt)*flux_parm;
			// it should be *df/w0, which is in fact = flux_parm
		
	     (*NFlux)(pt) -= nf;
	     outFlux(pt) += nf; //accumulate outflow of nutrient from cell

             Pix rp = p;
	    	
	        // making the length of flow path dependent of the amount of water to move
	        // niter = 1+30*pow(w0,0.05); printf ("\n   %d  %f", niter, w0);
	
	     float stt = w0*w0; 
	     float godo = 1 + MAXDRUN*stt/(stt+HSHEAD);
	     niter = godo*Slop(pt)*Slop(pt);
	     
//  printf("\nniter = %d   godo = %f", niter, godo );
	    		    	
	     for(int iter=0; iter<niter && rp; iter++) 
	        { // make niter iterations to run water further downhill
	           rp = grid.LinkedPix( rp );           // gets downstream link rp

		   if ( rp )
		   {
			const OrderedPoint& rpt = grid.GetPoint(rp);
			
//printf ("\n%d  %d", rpt.row(), rpt.col());
						
			if (HydMap(rpt) == GSTAT)
                               // GSTAT mark the gaging stations on River map.
                               // this is to sum up all the water and all the 
			       // materialthat is moved
                               // through the cells with gaging stations
				{
				  (*swTot)(rpt) += df;
				  (*NTot)(rpt) += nf;
				  N1 += nf;
				  num00++;
//printf("At GSTAT df = %f coming from %d %d\n", df, irrr, iccc );
				}
          		if (HydMap(rpt) == 10)  
                                // 10 marks the upper part of the lower SWShed
         			{
				  N2 += nf;
				  num10++;
				}

          		if (HydMap(rpt) == 20)  
                                // 20 marks the upper part of the lower SWShed
         			{
				  N3 += nf;
				  num20++;
				}

          		if (HydMap(rpt) == MOUTH)  
                                // MOUTH mark the mouth on River map.
         			{
          			  TotNout += nf;
          			  TotalOut += df;
//printf("At MOUTH df = %f coming from %d %d\n", df, irrr, iccc );
          			}
          		if (HabMap(rpt) == OPENW) break;
                        
		   }
		} // end loop over cells along the path of flow
		   		
                // the following is when everything is dumped into one last recipient cell
		if ( rp ) 
		{  
		 	const OrderedPoint& rpt = grid.GetPoint(rp);
		 	(*swFlux)(rpt) += df;
		 	if (w0>transport_st) (*NFlux)(rpt) += nf;
		}
	  } //end loop over all area
			
	AvailWater.AddData(*swFlux);
	Nitrogen.AddData(*NFlux);

// another loop over the whole area, but only for open water cells	
	for( p = grid.first(); p; grid.next(p) )               
	  { 
	     const OrderedPoint& pt = grid.GetPoint(p);     // pt same as p, but ordered
	     if (HabMap(pt) != OPENW) continue;

	     // water flow in Open Water regions
             w_sum = Elev(pt)+AvailWater(pt);
	     int count = 0, ow = 1;  // number of open water cells, presence of open water in the vicinity

             // calculate the average water level
	     // loops until either the max allowed number of cells reached, 
	     // or no adjacent water cell present in the vicinity
             for (NN = 1; count < Nmax-1 && ow; NN++)  //NN is number of concentric layer
		  { 
		   ow = 0; 
		   for (rr = -NN; rr <= NN; rr++)
		      for (rc = -NN; rc <= NN; rc += 2*NN)
		        
		      {
		        rp = grid.TranslateByGridCoord( p, rr, rc );

		        if( rp )  
		        {
		          const OrderedPoint& rpt = grid.GetPoint(rp);
		          if (HabMap(rpt) == OPENW) 
		            { w_sum += Elev(rpt)+AvailWater(rpt);
		              ow = 1;
  		              rPix[count++] = rp;
		            }  
		        } 
		      }

		    for (rc= -NN+1; rc<NN; rc++)
		      for (rr = -NN; rr<=NN; rr+=2*NN)
		      {  
		        rp = grid.TranslateByGridCoord( p, rr, rc );

		        if( rp )  
		        {
		          const OrderedPoint& rpt = grid.GetPoint(rp);
		          if (HabMap(rpt) == OPENW) 
		            { w_sum += Elev(rpt)+AvailWater(rpt);
		              ow = 1;
  		              rPix[count++] = rp;
		            }  
		        } 
		      }
		   } // end loop over concentric neighborhood 

	      float h0 = w_sum/(count+1);
	        
	        // make sure that the average water level does not expose ground
	      if (h0 < Elev(pt)) h0 -= (Elev(pt)-h0)/count;
	        
	      for( il = 0; il < count-1; il++ ) 
		 { 
		  rp = rPix[il];
		  const OrderedPoint& rpt = grid.GetPoint(rp);

 	          if (h0 < Elev(rpt)) h0 -= (Elev(rpt)-h0)/count;
		 }
		// we can still potentially generate water in this case !!!	
				  
		// equilibrate  
	      for( il = 0; il < count; il++ ) 
		 { 
		  rp = rPix[il];
		  const OrderedPoint& rpt = grid.GetPoint(rp);
		  
		  ftmp = ( h0 - (Elev(rpt)+AvailWater(rpt)) ) * flow_rate;
		  if (-ftmp > AvailWater(rpt)) ftmp = -AvailWater(rpt);
		      
		  // for exchange of stuff, I assume that the transport is occuring via the center
		  // point pt. So that at each step the stuff is moved from each of the 
		  // vicinity cells to/from the center point. This can potentially drive the
		  // amount of stuff in the center point below zero, but at the end of the loop
		  // over the vicinity it should be back to normal.
		     
		  if (ftmp > AvailWater(pt)) 
		      { ftmp = AvailWater(pt);
		        move = Nitrogen(pt);
		      }
		  else 
		      { if (ftmp > 0.)
		          move = (AvailWater(pt) > 0.) ? Nitrogen(pt)*ftmp/AvailWater(pt) : 0.;
		        else 
		          move = (AvailWater(rpt) > 0.) ? Nitrogen(rpt)*ftmp/AvailWater(rpt) : 0.;
		      }
		          
	          Nitrogen(rpt) += move;
		  AvailWater(rpt) += ftmp; 

		  Nitrogen(pt) -= move;
		  AvailWater(pt) -= ftmp;
		  
		 } // end for equilibration loop
// printf ("\n%d  %d %f  %f %f", pt.row(), pt.col(), Elev(pt)+AvailWater(pt), AvailWater(pt));
    
		//take care of the outflow from estuary
		if (HydMap(pt) == MOUTH && (ftmp = Elev(pt)+AvailWater(pt)-WATLEV) > 0) 
			{ if (ftmp > AvailWater(pt)) ftmp = AvailWater(pt);
			  //if new level is below the water level, we remove all the water
			  if (AvailWater(pt)>0)
			       move = Nitrogen(pt)*ftmp/AvailWater(pt);
			  else move = 0;  
			  Nitrogen(pt) -= move;
			  AvailWater(pt) -= ftmp;
			  TotalOut += ftmp;
			  TotNout += move;
			}  
		  	     
	     } //end loop for OPEN Water areas
	
// printing results
	Ntot = 0;
		
	for( p = grid.first(); p; grid.next(p) )
		{
			const OrderedPoint& pt = grid.GetPoint(p);  
	                Ntot += Nitrogen(pt);
			if( HydMap(pt) == GSTAT )
				{ 
				  fprintf ( file, "\t%12.6f", (*swTot)(pt) );

// this doesn't look quite right. I'm not sure what Ralf calculates here:
//	 			  n_out_sum  = (*swTot)(pt);

				  fprintf ( file, "\t%12.6f", AvailWater(pt) );
				  fprintf ( fileN, "\t%12.6f", Nitrogen(pt) );

				}
		         if ( pt.row()==ROWOUT && pt.col()==COLOUT)
	                	fprintf ( fileN, "\t%12.6f", Nitrogen(pt) );
		}
	Nconc = (TotalOut > 0) ? TotNout/TotalOut : 0;

	fprintf ( fileN, "\t%12.6f\t%12.6f\t%12.6f\t%12.6f\t%12.6f\t%12.6f\t%10d\t%10d\t%10d", 
			TotNout, Nconc, Ntot, N1, N2, N3, num00, num10, num20 );
	fprintf ( file, "\t%12.4f", TotalOut );

  // Handover Values of SWater to global variables (sorry) for printout
  // and further analysis in UCode.cc(Goalfunction)

  n_out_sum = TotNout;
  n_out_conc = Nconc;
	
	fflush(file); 
	fflush(fileN); 
}
/***********************************************************************


/***********************************************************************/


