#include "GeneratorServantC.h"
#include "Debug.h"
#include <map>


//using namespace std;


namespace QEDO_CIDL_Generator {


GeneratorServantC::GeneratorServantC
( QEDO_ComponentRepository::CIDLRepository_impl *repository)
: CPPBase(repository)
{
}


GeneratorServantC::~GeneratorServantC
()
{
}


void
GeneratorServantC::generate(std::string target, std::string fileprefix)
{
	try { initialize(target, fileprefix); }
	catch (InitializeError) { return; }

	//
	// open file
	//
	std::string header_name = file_prefix_ + "_SERVANT";
	std::string filename = header_name + ".cpp";
	out.open(filename.c_str());
	out << "//\n";
	out << "// generated by Qedo\n";
	out << "//\n\n";
	out << "#include \"" << header_name << ".h\"\n";
	out << "#include \"CDRTransportCoDec.h\"\n";
	out << "#include \"MarshalBuffer.h\"\n";
	out << "#include \"Output.h\"\n\n";
	out << "#include \"cstring\"\n\n\n";

	//
	// dynamic library identifier
	//
	out << "\n//\n// dynamic library identifier\n//\n";
	out << "static CORBA::Long s_library_id = 0;\n\n";
	out << "void\n set_library_id( CORBA::Long id )\n{\n";
	out.indent();
	out << "s_library_id = id;\n";
	out.unindent();
	out << "}\n\n";
	out << "CORBA::Long\n get_library_id()\n{\n";
	out.indent();
	out << "return s_library_id;\n";
	out.unindent();
	out << "}\n\n";

	doGenerate();

	//
	// close file
	//
	out.close();
}
	

CORBA::ULong
GeneratorServantC::calculate_align (CORBA::ULong cur_len, CORBA::ULong align_val)
{
	CORBA::ULong modulo = cur_len % align_val;
	if (modulo)
		return align_val - modulo;
	else
		return 0;
}


void 
GeneratorServantC::generate_marshal_code (IR__::IDLType_ptr idl_type, std::string& parameter_name, CORBA::ULong rec_depth)
{
	switch (idl_type->def_kind())
	{
	case CORBA__::dk_Primitive:
		{
			IR__::PrimitiveDef_var prim = IR__::PrimitiveDef::_narrow (idl_type);
			assert (!CORBA::is_nil (prim));
			switch (prim->kind())
			{
			case IR__::pk_char:
				out << "Qedo::CDRTransportCoDec::marshal_char (marshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_octet:
				out << "Qedo::CDRTransportCoDec::marshal_octet (marshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_boolean:
				out << "Qedo::CDRTransportCoDec::marshal_boolean (marshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_short:
				out << "Qedo::CDRTransportCoDec::marshal_short (marshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_ushort:
				out << "Qedo::CDRTransportCoDec::marshal_ushort (marshal_buffer, " << parameter_name << ");\n";
				return;;
			case IR__::pk_long:
				out << "Qedo::CDRTransportCoDec::marshal_long (marshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_ulong:
				out << "Qedo::CDRTransportCoDec::marshal_ulong (marshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_float:
				out << "Qedo::CDRTransportCoDec::marshal_float (marshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_longlong:
				out << "Qedo::CDRTransportCoDec::marshal_longlong (marshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_ulonglong:
				out << "Qedo::CDRTransportCoDec::marshal_ulonglong (marshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_double:
				out << "Qedo::CDRTransportCoDec::marshal_double (marshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_longdouble:
				out << "Qedo::CDRTransportCoDec::marshal_longdouble (marshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_string:
				out << "Qedo::CDRTransportCoDec::marshal_string (marshal_buffer, " << parameter_name << ");\n";
				return;
			default:
				// Should be never reached
				assert (0);
				return;
			}
		}
	case CORBA__::dk_Struct:
		{
			IR__::StructDef_var struct_def = IR__::StructDef::_narrow (idl_type);
			assert (!CORBA::is_nil (struct_def));

			IR__::StructMemberSeq_var members = struct_def->members();

			std::string new_parameter_name;

			for (unsigned int i = 0; i < members->length(); i++)
			{
				new_parameter_name = "(" + parameter_name + "." + members.in()[i].name.in() + ")";
				this->generate_marshal_code (const_cast<IR__::StructMember&>(members.in()[i]).type_def, new_parameter_name, rec_depth);
			}
			return;
		}
	case CORBA__::dk_Sequence:
		{
			++rec_depth;

			IR__::SequenceDef_var sequence_def = IR__::SequenceDef::_narrow (idl_type);
			assert (!CORBA::is_nil (sequence_def));

			IR__::IDLType_var idl_type = sequence_def->element_type_def();

			std::string new_parameter_name;
			char rec_depth_str[33];
			sprintf(rec_depth_str, "%ld", rec_depth);

			new_parameter_name = "(" + parameter_name + "[seq_count" + rec_depth_str + "])";

			out << "CORBA::ULong seq_len" << rec_depth << " = " << parameter_name;
			//if (hasVariableLength (idl_type))
			//	out << "->length();\n";
			//else
				out << ".length();\n";

			out << "Qedo::CDRTransportCoDec::marshal_ulong (marshal_buffer, seq_len" << rec_depth << ");\n\n";

			out << "for (unsigned int seq_count" << rec_depth << " = 0; ";
			out << "seq_count" << rec_depth << " < seq_len" << rec_depth;
			out << "; seq_count" << rec_depth << "++)\n";
			out << "{\n", out.indent();

			generate_marshal_code (idl_type, new_parameter_name, rec_depth); out.unindent();
			out << "}\n";

			return;
		}
	case CORBA__::dk_Alias:
		{
			IR__::AliasDef_var alias_def = IR__::AliasDef::_narrow (idl_type);
			assert (!CORBA::is_nil (alias_def));

			IR__::IDLType_var idl_type = alias_def->original_type_def();

			generate_marshal_code (idl_type, parameter_name, rec_depth);

			return;
		}
	default:
		// Not implemented
		assert (0);
		return;
	}
}


void
GeneratorServantC::generate_unmarshal_code (IR__::IDLType_ptr idl_type, std::string& parameter_name, CORBA::ULong rec_depth)
{
	switch (idl_type->def_kind())
	{
	case CORBA__::dk_Primitive:
		{
			IR__::PrimitiveDef_var prim = IR__::PrimitiveDef::_narrow (idl_type);
			assert (!CORBA::is_nil (prim));
			switch (prim->kind())
			{
			case IR__::pk_char:
				out << "Qedo::CDRTransportCoDec::unmarshal_char (unmarshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_octet:
				out << "Qedo::CDRTransportCoDec::unmarshal_octet (unmarshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_boolean:
				out << "Qedo::CDRTransportCoDec::unmarshal_boolean (unmarshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_short:
				out << "Qedo::CDRTransportCoDec::unmarshal_short (unmarshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_ushort:
				out << "Qedo::CDRTransportCoDec::unmarshal_ushort (unmarshal_buffer, " << parameter_name << ");\n";
				return;;
			case IR__::pk_long:
				out << "Qedo::CDRTransportCoDec::unmarshal_long (unmarshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_ulong:
				out << "Qedo::CDRTransportCoDec::unmarshal_ulong (unmarshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_float:
				out << "Qedo::CDRTransportCoDec::unmarshal_float (unmarshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_longlong:
				out << "Qedo::CDRTransportCoDec::unmarshal_longlong (unmarshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_ulonglong:
				out << "Qedo::CDRTransportCoDec::unmarshal_ulonglong (unmarshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_double:
				out << "Qedo::CDRTransportCoDec::unmarshal_double (unmarshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_longdouble:
				out << "Qedo::CDRTransportCoDec::unmarshal_longdouble (unmarshal_buffer, " << parameter_name << ");\n";
				return;
			case IR__::pk_string:
				out << "Qedo::CDRTransportCoDec::unmarshal_string (unmarshal_buffer, " << parameter_name << ");\n";
				return;
			default:
				// Should be never reached
				assert (0);
				return;
			}
		}
	case CORBA__::dk_Struct:
		{
			IR__::StructDef_var struct_def = IR__::StructDef::_narrow (idl_type);
			assert (!CORBA::is_nil (struct_def));

			IR__::StructMemberSeq_var members = struct_def->members();

			std::string new_parameter_name;

			for (unsigned int i = 0; i < members->length(); i++)
			{
				new_parameter_name = "(" + parameter_name + "." + members.in()[i].name.in() + ")";
				generate_unmarshal_code (const_cast<IR__::StructMember&>(members.in()[i]).type_def, new_parameter_name, rec_depth);
			}
			return;
		}
	case CORBA__::dk_Sequence:
		{
			++rec_depth;

			IR__::SequenceDef_var seq_def = IR__::SequenceDef::_narrow (idl_type);
			assert (!CORBA::is_nil (seq_def));

			IR__::IDLType_var seq_idl_type = seq_def->element_type_def();

			out << "CORBA::ULong seq_len" << rec_depth << ";\n";

			out << "Qedo::CDRTransportCoDec::unmarshal_ulong (unmarshal_buffer, seq_len" << rec_depth << ");\n\n";

			//if (hasVariableLength (seq_idl_type))
			//	out << parameter_name << "->length (seq_len" << rec_depth << ");\n\n";
			//else
				out << parameter_name << ".length (seq_len" << rec_depth << ");\n\n";

			std::string new_parameter_name;
			char rec_depth_str[33];
			sprintf(rec_depth_str, "%ld", rec_depth);

			new_parameter_name = "(" + parameter_name + "[seq_count" + rec_depth_str + "])";

			out << "for (unsigned int seq_count" << rec_depth << " = 0; ";
			out << "seq_count" << rec_depth << " < seq_len" << rec_depth << "; ";
			out << "seq_count" << rec_depth << "++)\n";
			out << "{\n"; out.indent();
			generate_unmarshal_code (seq_idl_type, new_parameter_name, rec_depth); out.unindent();
			out << "}\n";

			return;
		}
	case CORBA__::dk_Alias:
		{
			IR__::AliasDef_var alias_def = IR__::AliasDef::_narrow (idl_type);
			assert (!CORBA::is_nil (alias_def));

			IR__::IDLType_var original_idl_type = alias_def->original_type_def();

			generate_unmarshal_code (original_idl_type, parameter_name, rec_depth);

			return;
		}
	default:
		// Not implemented
		assert (0);
		return;
	}
}


void
GeneratorServantC::check_for_generation(IR__::Contained_ptr item)
{
	//
	// check if item is already known
	//
	if (item_well_known(item))
	{
		return;
	}

	//
	// check if item is already in the list or currently processed
	//
	if ((this->m_recursion_set.find(item->id())) != m_recursion_set.end() || this->already_included (item)) {
		return;
	} 
	else {
		m_recursion_set.insert(item->id());
	}

	CORBA::ULong len;
	CORBA::ULong i;
	switch (item->def_kind()) {
	case CORBA__::dk_Module : {
		IR__::ModuleDef_var a_module = IR__::ModuleDef::_narrow(item);

		// modules
		IR__::ContainedSeq_var contained_seq = a_module->contents(CORBA__::dk_Module, true);
		len = contained_seq->length();
		for(i = 0; i < len; i++)
		{
			check_for_generation((*contained_seq)[i]);
		}

		// homes
		contained_seq = a_module->contents(CORBA__::dk_Home, true);
		len = contained_seq->length();
		for(i = 0; i < len; i++)
		{
			check_for_generation((*contained_seq)[i]);
		}

		// compositions
		contained_seq = repository_->contents(CORBA__::dk_Composition, true);
		len = contained_seq->length();
		CIDL::CompositionDef_var a_composition;
		for(i = 0; i < len; i++)
		{
			a_composition = CIDL::CompositionDef::_narrow((*contained_seq)[i]);
			std::string id = a_composition->id();
			std::string::size_type pos = id.find_last_of("/");
			if(pos != std::string::npos) 
			{
				id.replace(pos, std::string::npos, ":1.0");
				if(!id.compare(a_module->id())) 
				{
					check_for_generation(a_composition);
				}
			}
		}

		break; }
	case CORBA__::dk_Composition : {
		CIDL::CompositionDef_var a_composition = CIDL::CompositionDef::_narrow(item);

		// home
		check_for_generation(a_composition->ccm_home());
		insert_to_generate(item);
		break; }
	case CORBA__::dk_Home : {
		IR__::HomeDef_var a_home = IR__::HomeDef::_narrow(item);
		
		// insert this interface in generation list
		this->insert_to_generate(item);

		// managed component
		IR__::ComponentDef_var a_component = a_home->managed_component();
		this->check_for_generation(a_component);

		break; }
	case CORBA__::dk_Component : {
		IR__::ComponentDef_var a_component = IR__::ComponentDef::_narrow(item);

		// insert this interface in generation list
		this->insert_to_generate(item);

		break; }
	default:
		break;
	};

	m_recursion_set.erase(item->id());
};


void 
GeneratorServantC::resolve_atomic_streamtypes(IR__::StreamTypeDef_ptr streamtype, IR__::StreamTypeDefSeq& atomic_streamtypes)
{
	// Test for atomic streamtype
	IR__::StreamTypeDefSeq_var grouped_types = streamtype->grouped_types();

	if (! grouped_types->length())
	{
		atomic_streamtypes.length (atomic_streamtypes.length() + 1);
		atomic_streamtypes[atomic_streamtypes.length() - 1] = IR__::StreamTypeDef::_duplicate (streamtype);
		return;
	}

	for (unsigned int i = 0; i < grouped_types->length(); i++)
	{
		resolve_atomic_streamtypes (grouped_types.inout()[i], atomic_streamtypes);
	}
}


void
GeneratorServantC::doAttribute(IR__::AttributeDef_ptr attribute)
{
	std::string attribute_name = mapName(attribute);

	// not read only
	if(attribute->mode() == IR__::ATTR_NORMAL)
	{
		out << "void" << "\n";
		out << class_name_ << "::" << attribute_name << "(";
		out << map_in_parameter_type(attribute->type_def()) << " param)\n";
		out << "throw(CORBA::SystemException";
		handleException(attribute);
		out << ")\n{\n";
		out.indent();
		out << "try\n{\n";
		out.indent();
		out << "current_executor_ = executor_locator_->obtain_executor(\"" << executor_name_ << "\");\n";
		out.unindent();
		out << "}\ncatch (...)\n{\n";
		out.indent();
		out << "throw CORBA::INTERNAL (42, CORBA::COMPLETED_NO);\n";
		out.unindent();
		out << "}\n\n";
		out << "#ifdef TAO_ORB\n";
		out << interface_name_ << "_ptr facet = dynamic_cast< ";
		out << interface_name_ << "_ptr >(current_executor_);\n";
		out << "#else\n";
		out << interface_name_ << "_var facet = ";
		out << interface_name_ << "::_narrow (current_executor_);\n";
		out << "#endif\n";
		out << "if (CORBA::is_nil (facet))\n{\n";
		out.indent();
		out << "throw CORBA::INTERNAL (42, CORBA::COMPLETED_NO);\n";
		out.unindent();
		out << "}\n\n";
		out << "facet->" << attribute_name << "(param);\n";
		out.unindent();
		out << "}\n\n\n";
	}

	out << map_return_type(attribute->type_def()) << "\n";
	out << class_name_ << "::" << attribute_name << "()\n";
	out << "throw(CORBA::SystemException";
	handleException(attribute);
	out << ")\n{\n";
	out.indent();
	out << "try\n{\n";
	out.indent();
	out << "current_executor_ = executor_locator_->obtain_executor(\"" << executor_name_ << "\");\n";
	out.unindent();
	out << "}\ncatch (...)\n{\n";
	out.indent();
	out << "throw CORBA::INTERNAL (42, CORBA::COMPLETED_NO);\n";
	out.unindent();
	out << "}\n\n";
	out << "#ifdef TAO_ORB\n";
	out << interface_name_ << "_ptr facet = dynamic_cast< ";
	out << interface_name_ << "_ptr >(current_executor_);\n";
	out << "#else\n";
	out << interface_name_ << "_var facet = ";
	out << interface_name_ << "::_narrow(current_executor_);\n";
	out << "#endif\n";
	out << "if (CORBA::is_nil (facet))\n{\n";
	out.indent();
	out << "throw CORBA::INTERNAL (42, CORBA::COMPLETED_NO);\n";
	out.unindent();
	out << "}\n\n";
	out << "return facet->" << attribute_name << "();\n";
	out.unindent();
	out << "}\n\n\n";
}


void 
GeneratorServantC::doException(IR__::ExceptionDef_ptr except)
{
	out << ", " << mapFullName(except);
}


void
GeneratorServantC::doOperation(IR__::OperationDef_ptr operation)
{
	bool is_void = false;
	std::string operation_name = mapName(operation);

	if(operation->result_def()->type()->kind() == CORBA::tk_void)
	{
		is_void = true;
	}

	out << map_return_type(operation->result_def()) << "\n";
	out << class_name_ << "::" << operation_name << "(";
	IR__::ParDescriptionSeq* pards = operation->params();
	CORBA::ULong len = pards->length();
	CORBA::ULong i;
	for( i= len; i > 0; i--)
	{
		if(i < len)
		{
			out << ", ";
		}
		IR__::ParameterDescription pardescr = (*pards)[i - 1];
		if (pardescr.mode == IR__::PARAM_IN) {
			out << map_in_parameter_type (pardescr.type_def) << " " << mapName(string(pardescr.name));
		}
		if (pardescr.mode == IR__::PARAM_OUT) {
			out << map_out_parameter_type (pardescr.type_def) << " " << mapName(string(pardescr.name));
		}
		if (pardescr.mode == IR__::PARAM_INOUT) {
			out << map_inout_parameter_type (pardescr.type_def) << " " << mapName(string(pardescr.name));
		}
	}
	out << ")\n";
	out << "throw(CORBA::SystemException";
	handleException(operation);
	out << ")\n{\n";
	out.indent();
	out << "try\n{\n";
	out.indent();
	out << "current_executor_ = executor_locator_->obtain_executor(\"" << executor_name_ << "\");\n";
	out.unindent();
	out << "}\ncatch (...)\n{\n";
	out.indent();
	out << "throw CORBA::INTERNAL (42, CORBA::COMPLETED_NO);\n";
	out.unindent();
	out << "}\n\n";
	out << "#ifdef TAO_ORB\n";
	out << interface_name_ << "_ptr facet = dynamic_cast< ";
	out << interface_name_ << "_ptr >(current_executor_);\n";
	out << "#else\n";
	out << interface_name_ << "_var facet = ";
	out << interface_name_ << "::_narrow(current_executor_);\n";
	out << "#endif\n";
	out << "if (CORBA::is_nil (facet))\n{\n";
	out.indent();
	out << "throw CORBA::INTERNAL (42, CORBA::COMPLETED_NO);\n";
	out.unindent();
	out << "}\n\n";
	if(!is_void)
	{
		out << "return ";
	}
	out << "facet->" << operation_name << "(";
	for(i = len; i > 0; i--)
	{
		if(i < len)
		{
			out << ", ";
		}
		IR__::ParameterDescription pardescr = (*pards)[i - 1];
		out << mapName(string(pardescr.name));
	}
	out << ");\n";
	out.unindent();
	out << "}\n\n\n";
}


void
GeneratorServantC::doFactory(IR__::FactoryDef_ptr factory)
{
	std::string factory_name = mapName(factory);
	std::string home_name = mapFullNameLocal(home_);
	IR__::ComponentDef_var comp = home_->managed_component();
	std::string component_name = mapFullName(comp);

	out << map_return_type(component_) << "\n";
	out << class_name_ << "::" << factory_name << "(";
	IR__::ParDescriptionSeq* pards = factory->params();
	CORBA::ULong len = pards->length();
	CORBA::ULong i;
	for( i= len; i > 0; i--)
	{
		if(i < len)
		{
			out << ", ";
		}
		IR__::ParameterDescription pardescr = (*pards)[i - 1];
		out << map_in_parameter_type (pardescr.type_def) << " " << mapName(string(pardescr.name));
	};
	out << ")\n";
	out << "throw(CORBA::SystemException";
	handleException(factory);
	out << ")\n{\n";
	out.indent();
	out << "#ifdef TA_ORB\n";
	out << home_name << "_ptr home_executor = dynamic_cast < ";
	out << home_name << "_ptr > (home_executor_.in());\n";
	out << "#else\n";
	out << home_name << "_ptr home_executor = ";
	out << home_name << "::_narrow(home_executor_.in());\n";
	out << "#endif\n";
	out << "if (! home_executor)\n{\n";
	out.indent();
	out << "NORMAL_ERR (\"Home_servant: Cannot cast my executor\");\n";
	out << "throw Components::CreateFailure();\n";
	out.unindent();
	out << "}\n\n";
	out << "Components::EnterpriseComponent_var enterprise_component;\n\n";
	out << "try\n{\n";
	out.indent();
	out << "enterprise_component = home_executor->" << factory_name << "(";
	for(i = len; i > 0; i--)
	{
		if(i < len)
		{
			out << ", ";
		}
		IR__::ParameterDescription pardescr = (*pards)[i - 1];
		out << mapName(string(pardescr.name));
	};
	out << ");\n";
	out.unindent();
	out << "}\n";
	out << "catch (Components::CCMException&)\n{\n";
	out.indent();
	out << "throw Components::CreateFailure();\n";
	out.unindent();
	out << "}\n\n";
	out << "Components::ExecutorLocator_var executor_locator;\n\n";
	out << "try\n{\n";
	out.indent();
	out << "executor_locator = Components::ExecutorLocator::_narrow (enterprise_component);\n";
	out.unindent();
	out << "}\n";
	out << "catch (CORBA::SystemException&)\n{\n";
	out.indent();
	out << "NORMAL_ERR (\"Home_servant: This container can only handle locator-based implementations\");\n";
	out << "throw Components::CreateFailure();\n";
	out.unindent();
	out << "}\n\n";
	out << "Components::SessionComponent_var session_component;\n\n";
	out << "try\n{\n";
	out.indent();
	out << "session_component = Components::SessionComponent::_narrow (enterprise_component);\n";
	out.unindent();
	out << "}\n";
	out << "catch (CORBA::SystemException&)\n{\n";
	out.indent();
	out << "NORMAL_ERR (\"Home_servant: This is a session container, but created component is not a session component\");\n";
	out << "throw Components::CreateFailure();\n";
	out.unindent();
	out << "}\n\n";
	out << "// Create a new context\n";
	out << mapFullNameLocal(comp) << "_ContextImpl_var new_context = new ";
	out << comp->name() << "_Context_callback();\n\n";
	out << "// Set context on component\n";
	out << "session_component->set_session_context (new_context.in());\n\n";
	out << "// Incarnate our component instance (create reference, register servant factories, ...\n";
	out << "Qedo::ComponentInstance& component_instance = this->incarnate_component\n";
	out << "	(executor_locator, dynamic_cast < Qedo::CCMContext* >(new_context.in()));\n\n";
	out << "// use of servant factories\n";
	out << "servant_registry_->register_servant_factory(component_instance.object_id_, ";
	out << mapFullNameServant(comp) << "::cleaner_.factory_);\n\n";
	out << "// Extract our Key out of the object reference\n";
	out << "#ifdef TAO_ORB\n";
	out << "CORBA::OctetSeq* key = Qedo::Key::key_value_from_object_id(component_instance.object_id_);\n\n";
	out << "#else\n";
	out << "CORBA::OctetSeq_var key = Qedo::Key::key_value_from_object_id(component_instance.object_id_);\n\n";
	out << "#endif\n";
	out << "// register all ports\n";
	genFacetRegistration(comp);
	genReceptacleRegistration(comp);
	genEmitterRegistration(comp);
	genPublisherRegistration(comp);
	genConsumerRegistration(comp);
	out << "\nthis->finalize_component_incarnation(component_instance.object_id_);\n\n";
	out << component_name << "_var servant = ";
	out << component_name << "::_narrow (component_instance.component_ref());\n\n";
	out << "return servant._retn();\n";
	out.unindent();
	out << "}\n\n\n";
}


void
GeneratorServantC::doFinder(IR__::FinderDef_ptr finder)
{
	std::string finder_name = mapName(finder);
	std::string home_name = mapFullNameLocal(home_);
	IR__::ComponentDef_var comp = home_->managed_component();
	std::string component_name = mapFullName(comp);

	out << map_return_type(component_) << "\n";
	out << class_name_ << "::" << finder_name << "(";
	IR__::ParDescriptionSeq* pards = finder->params();
	CORBA::ULong len = pards->length();
	CORBA::ULong i;
	for( i= len; i > 0; i--)
	{
		if(i < len)
		{
			out << ", ";
		}
		IR__::ParameterDescription pardescr = (*pards)[i - 1];
		out << map_in_parameter_type (pardescr.type_def) << " " << mapName(string(pardescr.name));
	};
	out << ")\n";
	out << "throw(CORBA::SystemException";
	handleException(finder);
	out << ")\n{\n";
	out.indent();
	out << "#ifdef TAO_ORB\n";
	out << home_name << "_ptr home_executor = dynamic_cast < ";
	out << home_name << "_ptr > (home_executor_.in());\n";
	out << "#else\n";
	out << home_name << "_var home_executor = ";
	out << home_name << "::_narrow (home_executor_.in());\n";
	out << "#endif\n";
	out << "if (CORBA::is_nil (home_executor))\n{\n";
	out.indent();
	out << "NORMAL_ERR (\"Home_servant: Cannot cast my executor\");\n";
	out << "throw Components::CreateFailure();\n";
	out.unindent();
	out << "}\n\n";
	out << "Components::EnterpriseComponent_var enterprise_component;\n\n";
	out << "try\n{\n";
	out.indent();
	out << "enterprise_component = home_executor->" << finder_name << "(";
	for(i = len; i > 0; i--)
	{
		if(i < len)
		{
			out << ", ";
		}
		IR__::ParameterDescription pardescr = (*pards)[i - 1];
		out << mapName(string(pardescr.name));
	};
	out << ");\n";
	out.unindent();
	out << "}\n";
	out << "catch (Components::CCMException&)\n{\n";
	out.indent();
	out << "throw Components::CreateFailure();\n";
	out.unindent();
	out << "}\n\n";
	out << "Components::ExecutorLocator_var executor_locator;\n\n";
	out << "try\n{\n";
	out.indent();
	out << "executor_locator = Components::ExecutorLocator::_narrow (enterprise_component);\n";
	out.unindent();
	out << "}\n";
	out << "catch (CORBA::SystemException&)\n{\n";
	out.indent();
	out << "NORMAL_ERR (\"Home_servant: This container can only handle locator-based implementations\");\n";
	out << "throw Components::CreateFailure();\n";
	out.unindent();
	out << "}\n\n";
	out << "Components::SessionComponent_var session_component;\n\n";
	out << "try\n{\n";
	out.indent();
	out << "session_component = Components::SessionComponent::_narrow (enterprise_component);\n";
	out.unindent();
	out << "}\n";
	out << "catch (CORBA::SystemException&)\n{\n";
	out.indent();
	out << "NORMAL_ERR (\"Home_servant: This is a session container, but created component is not a session component\");\n";
	out << "throw Components::CreateFailure();\n";
	out.unindent();
	out << "}\n\n";
	out << "// Create a new context\n";
	out << mapFullNameLocal(comp) << "_ContextImpl_var new_context = new ";
	out << comp->name() << "_Context_callback();\n\n";
	out << "// Set context on component\n";
	out << "session_component->set_session_context (new_context.in());\n\n";
	out << "// Incarnate our component instance (create reference, register servant factories, ...\n";
	out << "Qedo::ComponentInstance& component_instance = this->incarnate_component\n";
	out << "	(executor_locator, dynamic_cast < Qedo::CCMContext* >(new_context.in()));\n\n";
	out << "// use of servant factories\n";
	out << "servant_registry_->register_servant_factory(component_instance.object_id_, ";
	out << mapFullNameServant(comp) << "::cleaner_.factory_);\n\n";
	out << "// Extract our Key out of the object reference\n";
	out << "#ifdef TAO_ORB\n";
	out << "CORBA::OctetSeq* key = Qedo::Key::key_value_from_object_id(component_instance.object_id_);\n\n";
	out << "#else\n";
	out << "CORBA::OctetSeq_var key = Qedo::Key::key_value_from_object_id(component_instance.object_id_);\n\n";
	out << "#endif\n";
	out << "// register all ports\n";
	genFacetRegistration(comp);
	genReceptacleRegistration(comp);
	genEmitterRegistration(comp);
	genPublisherRegistration(comp);
	genConsumerRegistration(comp);
	out << "\nthis->finalize_component_incarnation(component_instance.object_id_);\n\n";
	out << component_name << "_var servant = ";
	out << component_name << "::_narrow (component_instance.component_ref());\n\n";
	out << "return servant._retn();\n";
	out.unindent();
	out << "}\n\n\n";
}


void
GeneratorServantC::doInterface(IR__::InterfaceDef_ptr intf)
{
	// base interfaces
	IR__::InterfaceDefSeq_var base_seq = intf->base_interfaces();
	CORBA::ULong len = base_seq->length();
	CORBA::ULong i;
	for( i= 0; i < len; i++)
	{
		doInterface((*base_seq)[i]);
	}

	handleAttribute(intf);
	handleOperation(intf);
}

void
GeneratorServantC::doComposition (CIDL::CompositionDef_ptr composition)
{
	//
	// determine componentDef and HomeDef
	//
	component_ = composition->ccm_component();
	IR__::HomeDef_var home = composition->ccm_home();
    
	//
	// determine lifecycle
	//
	CIDL::LifecycleCategory lc = composition->lifecycle();

	//
	// generate Home
	//
	home_ = home;
	open_module(out, component_, "SERVANT_");
	out << "\n\n";
	genHomeServantBegin(home, lc);
	genHomeServant(home, lc);
	close_module(out, component_);

	//
	// entry point
	//
	out << "\n\n//\n// entry point\n//\n";
	out << "Qedo::HomeServantBase*\n";
	out << "create_" << home->name() << "S(void)\n{\n";
	out.indent();
	out << "return new " << mapFullNameServant(home) << "_servant();\n";
	out.unindent();
	out << "}\n\n";


	/*
	//
	// generat Component
	//
	open_module(out, component_, "SERVANT_");
	out << "\n\n";

	genFacetServants(component_);
	genSourceServants(component_);
	genConsumerServants(component_);
	genContextServantBegin(component_);
	genContextServant(component_);
	genComponentServantBegin(component_);
	genComponentServant(component_);

	close_module(out, component_);
*/
	this -> generate_component(component_);
}


void 
GeneratorServantC::generate_component(IR__::ComponentDef* a_component) {
/*
	// base component
	IR__::ComponentDef_var base_component = a_component->base_component();
	if ( ! CORBA::is_nil(base_component)) {
		this->generate_component(base_component);
	}
*/
	//
	// generat Component
	//	
	open_module(out, component_, "SERVANT_");
	out << "\n\n";

	genFacetServants(component_);
	genSourceServants(component_);
	genConsumerServants(component_);
	genContextServantBegin(component_);
	genContextServant(component_);
	genComponentServantBegin(component_);
	genComponentServant(component_);

	close_module(out, component_);


}

void
GeneratorServantC::doComponent(IR__::ComponentDef_ptr component)
{
	/*std::string header_name = std::string(getAbsoluteName(component, "_")) + "_SERVANT";
	std::string filename = header_name + ".cpp";
	out.open(filename.c_str());

	out << "//\n";
	out << "// generated by Qedo\n";
	out << "//\n\n";
	out << "#include \"" << header_name << ".h\"\n";
	out << "#include \"Output.h\"\n\n\n";*/

	/*
	component_ = component;
	
	open_module(out, component, "SERVANT_");
	out << "\n\n";

	genFacetServants(component);
	genSourceServants(component);
	genConsumerServants(component);
	genContextServantBegin(component);
	genContextServant(component);
	genComponentServantBegin(component);
	genComponentServant(component);

	close_module(out, component);
*/
	/*out.close();*/
}


void
GeneratorServantC::doProvides(IR__::ProvidesDef_ptr provides, IR__::ComponentDef_ptr component)
{
	std::string type = mapFullName_(provides->interface_type());

	//
	// provide_...
	//
	out << type << "_ptr\n";
	out << class_name_ << "::provide_" << provides->name() << "()\n";
	out << "throw (CORBA::SystemException)\n{\n";
	out.indent();
	out << type << "_var prov = ";
	if( type.compare( "CORBA::Object" ) == 0 )
	{
		out << "ccm_object_executor_->provide_facet(\"" << provides->name() << "\");\n\n";
	}
	else
	{
		out << type << "::_narrow( ccm_object_executor_->provide_facet(\"" << provides->name() << "\"));\n\n";
	}
	out	<< "return prov._retn();\n";
	out.unindent();
	out << "}\n\n\n";
}


void 
GeneratorServantC::doUses(IR__::UsesDef_ptr uses, IR__::ComponentDef_ptr component)
{
	std::string interface_name = mapFullName_(uses->interface_type());

	//
	// multiple
	//
	if(uses->is_multiple())
	{
		std::string mult_conn = mapFullName(component) + "::" + uses->name() + "Connections";

		// connect_...
		out << "Components::Cookie*\n";
		out << class_name_ << "::connect_" << uses->name() << "(" << interface_name << "_ptr conx)\n";
		out << "throw (Components::ExceededConnectionLimit, Components::InvalidConnection, CORBA::SystemException)\n{\n";
		out.indent();
		out << "return ccm_object_executor_->connect(\"" << uses->name() << "\", conx);\n";
		out.unindent();
		out << "}\n\n\n";
			
		// disconnect_...
		out << interface_name << "_ptr\n";
		out << class_name_ << "::disconnect_" << uses->name() << "(Components::Cookie* ck)\n";
		out << "throw (Components::InvalidConnection, CORBA::SystemException)\n{\n";
		out.indent();
		out << "ccm_object_executor_->disconnect(\"" << uses->name() << "\", ck);\n\n";
		out << "// TODO\n";
		out << "return 0;\n";
		out.unindent();
		out << "}\n\n\n";

		// get_connections_...
		out << mult_conn << "*\n";
		out << class_name_ << "::get_connections_" << uses->name() << "()\n";
		out << "throw (CORBA::SystemException)\n{\n";
		out.indent();
		out << "Components::ConnectedDescriptions* connections = ccm_object_executor_->get_connections(\"";
		out << uses->name() << "\");\n\n";
		out << mult_conn << "_var conn = new(" << mult_conn << ");\n";
		out << "conn->length(connections->length());\n";
		out << "for(unsigned int i = 0; i < connections->length(); i++)\n{\n";
		out.indent();
		out << "conn.inout()[i].objref = " << interface_name << "::_narrow((*connections)[i]->objref());\n";
		out << "conn.inout()[i].ck = (*connections)[i]->ck();\n";
		out << "}\n\n";
		out.unindent();
		out << "return conn._retn();\n";
		out.unindent();
		out << "}\n\n\n";
	}
	//
	// not multiple
	//
	else
	{
		// get_connection_...
		out << interface_name << "_ptr\n";
		out << class_name_ << "::get_connection_" << uses->name() << "()\n";
		out << "throw (CORBA::SystemException)\n{\n";
		out.indent();
		out << "Components::ConnectedDescriptions* connections = ccm_object_executor_->get_connections(\"";
		out << uses->name() << "\");\n\n";
		out << interface_name << "_var use = ";
		if( interface_name.compare( "CORBA::Object" ) == 0 )
		{
			out << "(*connections)[0]->objref();\n\n";
		}
		else
		{
			out << interface_name << "::_narrow ((*connections)[0]->objref());\n\n";
		}
		out << "return " << interface_name << "::_duplicate(use);\n";
		out.unindent();
		out << "}\n\n\n";

		// disconnect_...
		out << interface_name << "_ptr\n";
		out << class_name_ << "::disconnect_" << uses->name() << "()\n";
		out << "throw(Components::NoConnection, CORBA::SystemException)\n{\n";
		out.indent();
		out << interface_name << "_var use = get_connection_" << uses->name() << "();\n\n";
		out << "ccm_object_executor_->disconnect(\"" << uses->name() << "\", (Components::Cookie*)0);\n\n";
		out << "return use._retn();\n";
		out.unindent();
		out << "}\n\n\n";

		// connect_...
		out << "void\n";
		out << class_name_ << "::connect_" << uses->name() << "(";
		out << interface_name << "_ptr conxn)\n";
		out << "throw (Components::AlreadyConnected, Components::InvalidConnection, CORBA::SystemException)\n{\n";
		out.indent();
		out << "ccm_object_executor_->connect(\"" << uses->name() << "\", conxn);\n";
		out.unindent();
		out << "}\n\n\n";
	}
}

void 
GeneratorServantC::doSink(IR__::SinkDef_ptr sink, IR__::ComponentDef_ptr component)
{
	std::string sink_name = sink->name();
	std::string comp_name = component->name();

	// sink port servant constructor
	out << "//\n// " << sink_name << "\n//\n";
	out << comp_name << "::" << sink_name << "::" << sink_name << "()\n";
	out << ": SinkStreamPortServant (\"" << sink_name << "\")\n";
	out << "{\n";
	out << "}\n\n\n";

	// sink port servant destructor
	out << comp_name << "::" << sink_name << "::~" << sink_name << "()\n";
	out << "{\n";
	out << "}\n\n\n";

	// sink port servants begin_stream_* operation
	out << "void\n";
	out << comp_name << "::" << sink_name << "::begin_stream (const char* repos_id, const Components::ConfigValues& meta_data)\n";
	out << "throw(StreamComponents::UnsupportedStreamtype,\n"; out.indent();
	out << "StreamComponents::DuplicateStream,\n";
	out << "CORBA::SystemException)\n"; out.unindent();
	out << "{\n"; out.indent();
	out << "stream_ccm_object_executor_->begin_stream_sink (\"" << sink_name << "\", repos_id, meta_data);\n"; out.unindent();
	out << "}\n\n\n";	

	// sink port servants end_stream_* operation
	out << "void\n";
	out << comp_name << "::" << sink_name << "::end_stream()\n";
	out << "throw(StreamComponents::NoStream,\n"; out.indent();
	out << "CORBA::SystemException)\n"; out.unindent();
	out << "{\n"; out.indent();
	out << "stream_ccm_object_executor_->end_stream_sink (\"" << sink_name << "\");\n"; out.unindent();
	out << "}\n\n\n";

	// sink port dispatcher constructor
	out << comp_name << "::" << sink_name << "_dispatcher::" << sink_name << "_dispatcher (Components::ExecutorLocator_ptr executor_locator)\n";
	out << "{\n"; out.indent();
	out << "CORBA::Object_var the_sink_obj = executor_locator->obtain_executor (\"" << sink_name << "\");\n";
	out << "the_sink_ = " << mapFullName (IR__::Contained::_narrow(component->defined_in()));
	out << "::CCM_" << component->name() << "_" << sink_name << "_Sink::_narrow (the_sink_obj);\n\n";
	out << "if (CORBA::is_nil (the_sink_))\n"; 
	out << "{\n"; out.indent();
	out << "NORMAL_ERR (\"" << comp_name << "::" << sink_name << "_dispatcher: Cannot narrow sink port executor\");\n";
	out << "return;\n"; out.unindent();
	out << "}\n"; out.unindent();
	out << "}\n\n\n";

	// sink port dispatcher destructor
	out << comp_name << "::" << sink_name << "_dispatcher::~" << sink_name << "_dispatcher()\n";
	out << "{\n"; out.indent();
	out << "DEBUG_OUT (\"" << comp_name << sink_name << "_dispatcher: Destructor called\");\n"; out.unindent();
	out << "}\n\n\n";

	// sink port dispatchers begin_stream
	out << "void\n";
	out << comp_name << "::" << sink_name << "_dispatcher::begin_stream (const char* repos_id, const Components::ConfigValues& meta_data)\n";
	out << "{\n"; out.indent();
	out << "the_sink_->begin_stream_" << sink_name << " (repos_id, meta_data);\n"; out.unindent();
	out << "}\n\n\n";

	// sink port dispatchers end_stream
	out << "void\n";
	out << comp_name << "::" << sink_name << "_dispatcher::end_stream()\n";
	out << "{\n"; out.indent();
	out << "the_sink_->end_stream_" << sink_name << "();\n"; out.unindent();
	out << "}\n\n\n";

	// sink port dispatchers failed_stream
	out << "void\n";
	out << comp_name << "::" << sink_name << "_dispatcher::failed_stream()\n";
	out << "{\n"; out.indent();
	out << "the_sink_->failed_stream_" << sink_name << "();\n"; out.unindent();
	out << "}\n\n\n";

	// sink port dispatchers receive_stream
	out << "void\n";
	out << comp_name << "::" << sink_name << "_dispatcher::receive_stream (Qedo::UnmarshalBuffer* unmarshal_buffer)\n";
	out << "{\n"; out.indent();

	IR__::StreamTypeDef_var stream_type = sink->stream_type();
	IR__::IDLType_var transported_type = stream_type->transported_type();

	if (CORBA::is_nil (transported_type))
	{
		out << "the_sink_->receive_stream_" << sink_name << " (unmarshal_buffer);\n";
	}
	else
	{
		out << map_idl_type (transported_type) << " data;\n\n";

		std::string parameter_name = "data";

		generate_unmarshal_code (transported_type, parameter_name, 0);

		out << "\nthe_sink_->receive_stream_" << sink_name << " (data);\n";
	}

	out.unindent();
	out << "}\n\n\n";

	// the sink ports servant factory cleaner variable
	out << "Qedo::ServantFactoryCleaner " << component->name() << "::" << sink->name();
	out << "::cleaner_ (new " << component->name() << "::" << sink->name() << "::ServantFactory());\n";
};

void 
GeneratorServantC::doSource(IR__::SourceDef_ptr source, IR__::ComponentDef_ptr component)
{
}

void 
GeneratorServantC::doEmits(IR__::EmitsDef_ptr emits, IR__::ComponentDef_ptr component)
{


	out << "\n//\n// " << emits->id() << "\n//\n";
	std::string event_name = mapFullName(emits->event());

	// disconnect_...
	out << event_name << "Consumer*\n";
	out << class_name_ << "::disconnect_" << emits->name() << "()\n";
    out << "throw (Components::NoConnection, CORBA::SystemException)\n{\n";
	out.indent();
	out << "CORBA::Object_var emi = ccm_object_executor_->disconnect_consumer(\"";
	out << emits->name() << "\");\n\n";
	out << "return " << event_name << "Consumer::_narrow(emi);\n";
	out.unindent();
	out << "}\n\n\n";

	// connect_...
    out << "void\n";
	out << class_name_ << "::connect_" << emits->name() << "(";
	out << event_name << "Consumer_ptr consumer)\n";
    out << "throw (Components::AlreadyConnected, CORBA::SystemException)\n{\n";
	out.indent();
	out << "ccm_object_executor_->connect_consumer(\"" << emits->name() << "\", consumer);\n";
	out.unindent();
	out << "}\n\n\n";
}


void 
GeneratorServantC::doPublishes(IR__::PublishesDef_ptr publishes, IR__::ComponentDef_ptr component)
{

	out << "\n//\n// " << publishes->id() << "\n//\n";
	std::string event_name = mapFullName(publishes->event());

	// subscribe_...
	out << "::Components::Cookie*\n";
	out << class_name_ << "::subscribe_" << publishes->name() << "(";
	out << event_name << "Consumer_ptr consumer)\n";
	out << "throw (Components::ExceededConnectionLimit, CORBA::SystemException)\n{\n";
	out.indent();
	out << "return ccm_object_executor_->subscribe(\"" << publishes->name() << "\", consumer);\n";
	out.unindent();
	out << "}\n\n\n";

	// unsubscribe_...
	out << event_name << "Consumer_ptr\n";
	out << class_name_ << "::unsubscribe_" << publishes->name() << "(";
	out << "::Components::Cookie* ck)\n";
	out << "throw (Components::InvalidConnection, CORBA::SystemException)\n{\n";
	out.indent();
	out << "return " << event_name << "Consumer::_narrow(ccm_object_executor_->unsubscribe(\"" << publishes->name() << "\", ck));\n";
	out.unindent();
	out << "}\n\n\n";
}


void
GeneratorServantC::doConsumes(IR__::ConsumesDef_ptr consumes, IR__::ComponentDef_ptr component)
{

	std::string event_name = mapFullName(consumes->event());

	// get_consumer_...
	out << event_name << "Consumer_ptr\n";
	out << class_name_ << "::get_consumer_" << consumes->name() << "()\n";
	out << "throw(CORBA::SystemException)\n{\n";
	out.indent();
	out << "Components::EventConsumerBase_var base = ccm_object_executor_->get_consumer(\"";
	out << consumes->name() << "\");\n";
	out << event_name << "Consumer_var consumer = " << event_name << "Consumer::_narrow(base);\n\n";
	out << "return consumer._retn();\n";
	out.unindent();
	out << "}\n\n\n";
}


void
GeneratorServantC::doHome(IR__::HomeDef_ptr home)
{
	/*
	component_ = IR__::ComponentDef::_duplicate(home->managed_component());
	
	open_module(out, component_, "SERVANT_");
	out << "\n\n";
	genHomeServantBegin(home);
	genHomeServant(home);
	close_module(out, component_);

	//
	// entry point
	//
	out << "\n\n//\n// entry point\n//\n";
	out << "Qedo::HomeServantBase*\n";
	out << "create_" << home->name() << "S(void)\n{\n";
	out.indent();
	out << "return new " << mapFullNameServant(home) << "_servant();\n";
	out.unindent();
	out << "}\n\n";

	//out.close();
	*/
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////


void
GeneratorServantC::genFacetServants(IR__::ComponentDef_ptr component)
{
	//
	// handle base component
	//
	IR__::ComponentDef_var base = component->base_component();
	if(!CORBA::is_nil(base))
	{ 
		genFacetServants(base);
	}

	IR__::ContainedSeq_var contained_seq = component->contents(CORBA__::dk_Provides, false);
	CORBA::ULong len = contained_seq->length();
	CORBA::ULong i;
	for( i= 0; i < len; i++)
	{
		//
		// facet servant
		//
		IR__::ProvidesDef_var provides = IR__::ProvidesDef::_narrow(((*contained_seq)[i]));
		std::string class_name = provides->name();
		class_name_ = string(component_->name()) + "::" + class_name;

		// header
		out << "// ================================================\n";
		out << "// " << class_name_ << "\n";
		out << "// ================================================\n\n";
		
		// constructor
		out << class_name_ << "::" << class_name << "()\n{\n}\n\n\n";
		
		// destructor
		out << class_name_ << "::~" << class_name << "()\n{\n}\n\n\n";

		IR__::InterfaceDef_var intf = IR__::InterfaceDef::_narrow(provides->interface_type());
		if( !CORBA::is_nil(intf) )
		{
			executor_name_ = provides->name();
			interface_name_ = mapFullNameLocal(intf);
			doInterface(intf);
		}

		//
		// facet servant factory
		//
		out << "// ================================================\n";
		out << "// " << class_name_ << "::cleaner_\n";
		out << "// ================================================\n\n";
		out << "Qedo::ServantFactoryCleaner " << class_name_ << "::cleaner_ (new ";
		out << class_name_ << "::ServantFactory());\n\n\n";
	}
}

void
GeneratorServantC::genSourceServants(IR__::ComponentDef_ptr component)
{
	IR__::ContainedSeq_var contained_seq = component->contents(CORBA__::dk_Source, false);
	CORBA::ULong len = contained_seq->length();
	CORBA::ULong i;
	for( i= 0; i < len; i++)
	{
	}
}


void
GeneratorServantC::genConsumerServants(IR__::ComponentDef_ptr component)
{
	// handle base component
	IR__::ComponentDef_var base = component->base_component();
	if(!CORBA::is_nil(base))
	{ 
		genConsumerServants(base);
	}

	IR__::ContainedSeq_var contained_seq = component->contents(CORBA__::dk_Consumes, false);
	CORBA::ULong len = contained_seq->length();
	CORBA::ULong i;
	for( i= 0; i < len; i++)
	{
		//
		// consumer servant
		//
		IR__::ConsumesDef_var consumes = IR__::ConsumesDef::_narrow(((*contained_seq)[i]));
		std::string class_name = consumes->name();
		class_name_ = string(component_->name()) + "::" + class_name;
		std::string event_consumer = mapFullNameLocal(consumes->event()) + "Consumer";
		std::string event_name = mapFullName(consumes->event());

		// header
		out << "// ================================================\n";
		out << "// " << class_name_ << "\n";
		out << "// ================================================\n\n";
		
		// constructor
		out << class_name_ << "::" << class_name << "()\n{\n}\n\n\n";
		
		// destructor
		out << class_name_ << "::~" << class_name << "()\n{\n}\n\n\n";

		// push_event
		out << "void\n";
		out << class_name_ << "::push_event (Components::EventBase* ev) throw (CORBA::SystemException)\n{\n";
		out.indent();
		out << event_name << "* event = " << event_name << "::_downcast (ev);\n\n";
		out << "if (!event)\n{\n";
		out.indent();
        out << "// handle error\n";
        out << "throw ::CORBA::INTERNAL(42,::CORBA::COMPLETED_NO);\n";
		out.unindent();
		out << "}\n\n";
		out << "push_" << consumes->event()->name() << "(event);\n";
		out.unindent();
		out << "}\n\n\n";

		// push_...
		out << "void\n";
		out << class_name_ << "::push_" << consumes->event()->name() << "(" << event_name;
		out << "* ev)\n{\n";
		out.indent();
		out << "try\n{\n";
		out.indent();
        out << "current_executor_ = executor_locator_->obtain_executor(\"component\");\n";
		out.unindent();
		out << "}\n";
		out << "catch (...)\n{\n";
		out.indent();
        out << "// handle error\n";
		out << "DEBUG_OUT (\"servantContext: can not obtain executor\");\n";
        out << "throw CORBA::INTERNAL (42, CORBA::COMPLETED_NO);\n";
		out.unindent();
		out << "}\n\n";
		out << event_consumer << "_ptr consumer_ptr = ";
		out << "dynamic_cast < " << event_consumer << "_ptr >(current_executor_);\n\n";
		out << "if (CORBA::is_nil (consumer_ptr))\n{\n";
		out.indent();
        out << "// handle error\n";
		out << "DEBUG_OUT (\"servantContext: can not cast consumer\");\n";
        out << "throw CORBA::INTERNAL (42, CORBA::COMPLETED_NO);\n";
		out.unindent();	
		out << "}\n\n";
	    out << "consumer_ptr->push_" << consumes->event()->name() << "(ev);\n";
		out.unindent();
		out << "}\n\n\n";

		//
		// consumer servant factory
		//
		out << "// ================================================\n";
		out << "// " << class_name_ << "::cleaner_\n";
		out << "// ================================================\n\n";
		out << "Qedo::ServantFactoryCleaner " << class_name_ << "::cleaner_ (new ";
		out << class_name_ << "::ServantFactory());\n\n\n";
	}
}


void
GeneratorServantC::genComponentServantBegin(IR__::ComponentDef_ptr component)
{
	// TODO if not defined in module, use prefix SERVANT_
	class_name_ = component->name();

	//
	// component servant factory
	//
	out << "// ================================================\n";
	out << "// " << class_name_ << "::cleaner_\n";
	out << "// ================================================\n\n";
	out << "Qedo::ServantFactoryCleaner " << class_name_ << "::cleaner_ (new ";
	out << class_name_ << "::ServantFactory());\n\n\n";

	// header
	out << "// ================================================\n";
	out << "// " << class_name_ << "\n";
	out << "// ================================================\n\n";

	// constructor
	out << class_name_ << "::" << class_name_ << "()\n{\n";
	out.indent();
	out << "DEBUG_OUT (\"" << class_name_ << " (component servant) : Constructor called\");\n";
	out.unindent();
	out << "}\n\n\n";

	// desctructor
	out << class_name_ << "::~" << class_name_ << "()\n{\n";
	out.indent();
	out << "DEBUG_OUT (\"" << class_name_ << " (component servant) : Destructor called\");\n";
	out.unindent();
	out << "}\n\n\n";

	executor_name_ = "component";
	interface_name_ = mapFullNameLocal(component) + "_Executor";
}


void
GeneratorServantC::genComponentServant(IR__::ComponentDef_ptr component)
{
//	class_name_ = string (component->name());
	// handle base component
	IR__::ComponentDef_var base = component->base_component();
	if(!CORBA::is_nil(base))
	{ 
		genComponentServant(base);
	}

	handleAttribute(component);
	handleSupportedInterface(component);
	handleProvides(component);
	handleUses(component);
	handleEmits(component);
	handlePublishes(component);
	if(component->contents(CORBA__::dk_Consumes, false)->length())
	{
		// push_event
		out << "void\n";
		out << class_name_ << "::push_event (Components::EventBase* ev) throw (CORBA::SystemException)\n{\n";
		out.indent();
		out << "// not implemented yet, please report : 274772\n";
		out << "throw CORBA::NO_IMPLEMENT();\n";
		out.unindent();
		out << "}\n\n\n";

		handleConsumes(component);
	}
	handleSink(component);
	handleSource(component);
}


void
GeneratorServantC::genContextServantBegin(IR__::ComponentDef_ptr component)
{
	class_name_ = string(component->name()) + "_Context_callback";

	// header
	out << "// ================================================\n";
	out << "// " << class_name_ << "\n";
	out << "// ================================================\n\n";

	// constructor
	out << class_name_ << "::" << class_name_ << "()\n{\n";
	out.indent();
	out << "DEBUG_OUT (\"" << class_name_ << " (context) : Constructor called\");\n";
	out.unindent();
	out << "}\n\n\n";

	// destructor
	out << class_name_ << "::~" << class_name_ << "()\n{\n";
	out.indent();
	out << "DEBUG_OUT (\"" << class_name_ << " (context) : Destructor called\");\n";
	out.unindent();
	out << "}\n\n\n";
}


void
GeneratorServantC::genContextServant(IR__::ComponentDef_ptr component)
{
	// handle base component
	IR__::ComponentDef_var base = component->base_component();
	if(!CORBA::is_nil(base))
	{ 
		genContextServant(base);
	}

	// uses ports
	IR__::ContainedSeq_var contained_seq = component->contents(CORBA__::dk_Uses, false);
	CORBA::ULong len = contained_seq->length();
	CORBA::ULong i;
	for( i= 0; i < len; i++)
	{
		IR__::UsesDef_var a_uses = IR__::UsesDef::_narrow(((*contained_seq)[i]));
		std::string interface_name = mapFullName_(a_uses->interface_type());
		
		//
		// multiple
		//
		if(a_uses->is_multiple())
		{
			std::string mult_conn = mapFullName(component) + "::" + a_uses->name() + "Connections";

			// get_connections_...
			out << mult_conn << "*\n";
			out << class_name_ << "::get_connections_" << a_uses->name() << "()\n";
			out << "throw (CORBA::SystemException)\n{\n";
			out.indent();
			out << "Components::ConnectedDescriptions_var connections;\n";
			out << "connections = ccm_object_executor_->get_connections(\"" << a_uses->name() << "\");\n\n";
			out << mult_conn << "_var conn = new(" << mult_conn << ");\n";
			out << "conn->length(connections->length());\n";
			out << "for(unsigned int i = 0; i < connections->length(); i++)\n{\n";
			out.indent();
			out << "conn.inout()[i].objref = " << interface_name << "::_narrow((*connections)[i]->objref());\n";
			out << "conn.inout()[i].ck = (*connections)[i]->ck();\n";
			out << "}\n\n";
			out.unindent();
			out << "return conn._retn();\n";
			out.unindent();
			out << "}\n\n\n";
		}
		//
		// not multiple
		//
		else
		{
			// get_connection_...
			out << interface_name << "_ptr\n";
			out << class_name_ << "::get_connection_" << a_uses->name() << "()\n{\n";
			out.indent();
			out << "Components::ConnectedDescriptions_var connections;\n";
			out << "connections = ccm_object_executor_->get_connections(\"";
			out << a_uses->name() << "\");\n\n";
			out << "if (! connections->length())\n{\n";
			out.indent();
			out << "return " << interface_name << "::_nil();\n";
			out.unindent();
			out << "}\n\n";
			out << interface_name << "_var use = ";
			if( interface_name.compare( "CORBA::Object" ) == 0 )
			{
				out << "(*connections)[0]->objref();\n\n";
			}
			else
			{
				out << interface_name << "::_narrow ((*connections)[0]->objref());\n\n";
			}
			out << "return use._retn();\n";
			out.unindent();
			out << "}\n\n\n";
		}
	}

	// emits ports
	contained_seq = component->contents(CORBA__::dk_Emits, false);
	len = contained_seq->length();
	for(i = 0; i < len; i++) 
	{
		IR__::EmitsDef_var a_emits = IR__::EmitsDef::_narrow(((*contained_seq)[i]));
		std::string event_name = mapFullName(a_emits->event());
		
		// push_...
		out << "void\n";
		out << class_name_ << "::push_" << a_emits->name() << "(" << event_name << "* ev)\n{\n";
		out.indent();
		out << event_name << "Consumer_var consumer = " << event_name;
		out << "Consumer::_narrow(ccm_object_executor_->get_consumer_for_emitter(\"";
		out << a_emits->name() << "\"));\n\n";
		out << "if (! CORBA::is_nil (consumer))\n{\n";
		out.indent();
		out << "try\n{\n";
		out.indent();
		out << "queue_event(consumer, ev, s_library_id);\n";
		out.unindent();
		out << "}\n";
		out << "catch (CORBA::SystemException& ex)\n{\n";
		out.indent();
		out << "std::cerr << \"exception when pushing event : \" << ex << std::endl;\n";
		out.unindent();
		out << "}\n";
		out.unindent();
		out << "}\n";
		out.unindent();
		out << "}\n\n\n";
	}

	// publishes ports
	contained_seq = component->contents(CORBA__::dk_Publishes, false);
	len = contained_seq->length();
	for(i = 0; i < len; i++) 
	{
		IR__::PublishesDef_var a_publishes = IR__::PublishesDef::_narrow(((*contained_seq)[i]));
		std::string event_name = mapFullName(a_publishes->event());
		
		// push_...
		out << "void\n";
		out << class_name_ << "::push_" << a_publishes->name() << "(" << mapFullName(a_publishes->event());
		out << "* ev)\n{\n";
		out.indent();
		out << "const Qedo::SubscribedConsumerVector& consumers = "; 
		out << "ccm_object_executor_->get_consumers_for_publisher (\"" << a_publishes->name() << "\");\n\n";
		out << "queue_event(consumers, ev, s_library_id);\n";
		out.unindent();
		out << "}\n\n\n";
	}

	// source ports
	contained_seq = component->contents (CORBA__::dk_Source, false);
	len = contained_seq->length();
	for(i = 0; i < len; i++) 
	{
		IR__::SourceDef_var a_source = IR__::SourceDef::_narrow(((*contained_seq)[i]));

		// begin_stream_*
		out << "void\n";
		out << class_name_ << "::begin_stream_" << a_source->name() << " (const char* repos_id,\n";
		out.indent(); out.indent();
		out << "const ::Components::ConfigValues& meta_data)\n";
		out.unindent(); out.unindent();
		out << "throw (StreamComponents::UnsupportedStreamtype, StreamComponents::DuplicateStream)\n";
		out << "{\n";
		out.indent();
		out << "this->begin_stream (\"" << a_source->name() << "\", repos_id, meta_data);\n";
		out.unindent();
		out << "}\n\n";

		// end_stream_*
		out << "void\n";
		out << class_name_ << "::end_stream_" << a_source->name() << "()\n";
		out << "throw (StreamComponents::NoStream)\n";
		out << "{\n";
		out.indent();
		out << "this->end_stream (\"" << a_source->name() << "\");\n";
		out.unindent();
		out << "}\n\n";

		// send_stream_*
		out << "void\n";

		IR__::StreamTypeDef_var stream_type = a_source->stream_type();
		IR__::IDLType_var transported_type = stream_type->transported_type();

		if (CORBA::is_nil (transported_type))
		{
			out << class_name_ << "::send_stream_" << a_source->name() << " (StreamComponents::StreamingBuffer_ptr buffer)\n";
		}
		else
		{
			out << class_name_ << "::send_stream_" << a_source->name() << " (" << map_in_parameter_type (transported_type) << " data)\n";
		}

		out << "throw (StreamComponents::NoStream)\n";
		out << "{\n";
		out.indent();

		if (CORBA::is_nil (transported_type))
		{
			out << "this->send_buffer (\"" << a_source->name() << "\", buffer);\n";
		}
		else
		{
			out << "Qedo::MarshalBuffer* marshal_buffer = new Qedo::MarshalBuffer (100);\n";

			std::string parameter_name = "data";

			generate_marshal_code (transported_type, parameter_name, 0);

			out << "\nthis->send_buffer (\"" << a_source->name() << "\", marshal_buffer);\n\n";

			out << "marshal_buffer->_remove_ref();\n";
		}

		out.unindent();
		out << "}\n\n";
	}
}


void
GeneratorServantC::genHomeServantBegin(IR__::HomeDef_ptr home, CIDL::LifecycleCategory lc)
{
	class_name_ = string(home->name()) + "_servant";
	IR__::ComponentDef_var comp = home->managed_component();

	// header
	out << "// ================================================\n";
	out << "// " << class_name_ << "\n";
	out << "// ================================================\n\n";

	// constructor
	out << class_name_ << "::" << class_name_ << "()\n";
	out << ": HomeServantBase(\"" << home->id() << "\", \"" << comp->id() << "\")\n{\n";
	out.indent();
	out << "DEBUG_OUT (\"" << class_name_ << " (home servant) : Constructor called\");\n";
	out.unindent();
	out << "}\n\n\n";

	// destructor
	out << class_name_ << "::~" << class_name_ << "()\n{\n";
	out.indent();
	out << "DEBUG_OUT (\"" << class_name_ << " (home servant) : Destructor called\");\n";
	out.unindent();
	out << "}\n\n\n";

	// create
	out << mapFullName(component_) << "_ptr\n";
	out << class_name_ << "::create()\n";
	out << "throw(CORBA::SystemException, Components::CreateFailure)\n{\n";
	out.indent();
	out << "DEBUG_OUT (\"Home_servant: create() called\");\n\n";
	out << "#ifdef TAO_ORB\n";
	out << mapFullNameLocal(home) << "_ptr home_executor = dynamic_cast < ";
	out << mapFullNameLocal(home) << "_ptr > (home_executor_.in());\n";
	out << "#else\n";
	out << mapFullNameLocal(home) << "_var home_executor = ";
	out << mapFullNameLocal(home) << "::_narrow (home_executor_.in());\n";
	out << "#endif\n";
	out << "if (CORBA::is_nil (home_executor))\n{\n";
	out.indent();
	out << "NORMAL_ERR (\"Home_servant: Cannot cast my executor\");\n";
	out << "throw Components::CreateFailure();\n";
	out.unindent();
	out << "}\n\n";
	out << "Components::EnterpriseComponent_var enterprise_component;\n\n";
	out << "try\n{\n";
	out.indent();
	out << "enterprise_component = home_executor->create();\n";
	out.unindent();
	out << "}\n";
	out << "catch (Components::CCMException&)\n{\n";
	out.indent();
	out << "NORMAL_ERR (\"Home_servant: cannot create component\");\n";
	out << "throw Components::CreateFailure();\n";
	out.unindent();
	out << "}\n\n";
	out << "Components::ExecutorLocator_var executor_locator;\n\n";
	out << "try\n{\n";
	out.indent();
	out << "executor_locator = Components::ExecutorLocator::_narrow (enterprise_component);\n";
	out.unindent();
	out << "}\n";
	out << "catch (CORBA::SystemException&)\n{\n";
	out.indent();
	out << "NORMAL_ERR (\"Home_servant: This container can only handle locator-based implementations\");\n";
	out << "throw Components::CreateFailure();\n";
	out.unindent();
	out << "}\n\n";
	switch (lc) {
		case (CIDL::lc_Session):
		{
            out << "Components::SessionComponent_var session_component;\n\n";
			break;
		}
		case (CIDL::lc_Extension) :
		{
			out << "Components::ExtensionComponent_var extension_component;\n\n";
			break;
		}
		default:
		{
			// not supported lifecycle
		}
	}
	out << "try\n{\n";
	out.indent();
	switch (lc) {
		case (CIDL::lc_Session):
		{
			out << "session_component = Components::SessionComponent::_narrow (enterprise_component);\n";
			break;
		}
		case (CIDL::lc_Extension) :
		{
			out << "extension_component = Components::ExtensionComponent::_narrow (enterprise_component);\n";
			break;
		}
		default:
		{
			// not supported lifecycle
		}
	}
	out.unindent();
	out << "}\n";
	out << "catch (CORBA::SystemException&)\n{\n";
	out.indent();
	out << "NORMAL_ERR (\"Home_servant: This is a session container, but created component is not a session component\");\n";
	out << "throw Components::CreateFailure();\n";
	out.unindent();
	out << "}\n\n";
	out << "// Create a new context\n";
	out << mapFullNameLocal(comp) << "_ContextImpl_var new_context = new ";
	out << mapFullNameServant(comp) << "_Context_callback();\n\n";
	if (lc==CIDL::lc_Extension) {
		out << "// Set container interceptor registration on context\n";
		out << "new_context -> set_server_interceptor_dispatcher_registration(server_dispatcher_.in());\n";
		out << "new_context -> set_client_interceptor_dispatcher_registration(client_dispatcher_.in());\n";
   	};	
	out << "// Set context on component\n";
	switch (lc) {
		case (CIDL::lc_Session):
		{
			out << "session_component->set_session_context (new_context.in());\n\n";
			break;
		}
		case (CIDL::lc_Extension) :
		{
			out << "extension_component->set_extension_context (new_context.in());\n\n";
			break;
		}
		default:
		{
			// not supported lifecycle
		}
	}
	out << "// Incarnate our component instance (create reference, register servant factories, ...\n";
	out << "Qedo::ComponentInstance& component_instance = this->incarnate_component\n";
	out << "	(executor_locator, dynamic_cast < Qedo::CCMContext* >(new_context.in()));\n\n";
	out << "// register servant factory\n";
	out << "servant_registry_->register_servant_factory(component_instance.object_id_, ";
	out << mapFullNameServant(comp) << "::cleaner_.factory_);\n\n";
	out << "// Extract our Key out of the object reference\n";
	out << "#ifdef TAO_ORB\n";
	out << "CORBA::OctetSeq* key = Qedo::Key::key_value_from_object_id(component_instance.object_id_);\n\n";
	out << "#else\n";
	out << "CORBA::OctetSeq_var key = Qedo::Key::key_value_from_object_id(component_instance.object_id_);\n\n";
	out << "#endif\n";
	out << "// register all ports\n";
	genFacetRegistration(comp);
	genReceptacleRegistration(comp);
	genEmitterRegistration(comp);
	genPublisherRegistration(comp);
	genConsumerRegistration(comp);

	out << "CORBA::RepositoryIdSeq streamtypes;\n\n";
	genSinkRegistration(comp);
	genSourceRegistration(comp);

	out << "\nthis->finalize_component_incarnation(component_instance.object_id_);\n\n";
	out << mapFullName(comp) << "_var servant = ";
	out << mapFullName(comp) << "::_narrow (component_instance.component_ref());\n\n";
	out << "return servant._retn();\n";
	out.unindent();
	out << "}\n\n\n";

	// create_component
	out << "Components::CCMObject_ptr\n";
	out << class_name_ << "::create_component()\n";
	out << "throw(CORBA::SystemException,Components::CreateFailure)\n{\n";
	out.indent();
	out << "return this->create();\n";
	out.unindent();
	out << "}\n\n\n";

	// create_component_with_config
	out << "Components::CCMObject_ptr\n";
	out << class_name_ << "::create_component_with_config(const Components::ConfigValues& config)\n";
	out << "throw(CORBA::SystemException, Components::CreateFailure)\n{\n";
	out.indent();
	out << "DEBUG_OUT (\"Home_servant: create_component_with_config() called\");\n\n";
	out << "#ifdef TAO_ORB\n";
	out << mapFullNameLocal(home) << "_ptr home_executor = dynamic_cast < ";
	out << mapFullNameLocal(home) << "_ptr > (home_executor_.in());\n";
	out << "#else\n";
	out << mapFullNameLocal(home) << "_var home_executor = ";
	out << mapFullNameLocal(home) << "::_narrow (home_executor_.in());\n";
	out << "#endif\n";
	out << "if (CORBA::is_nil (home_executor))\n{\n";
	out.indent();
	out << "NORMAL_ERR (\"Home_servant: Cannot cast my executor\");\n";
	out << "throw Components::CreateFailure();\n";
	out.unindent();
	out << "}\n\n";
	out << "Components::EnterpriseComponent_var enterprise_component;\n\n";
	out << "try\n{\n";
	out.indent();
	out << "enterprise_component = home_executor->create();\n";
	out.unindent();
	out << "}\n";
	out << "catch (Components::CCMException&)\n{\n";
	out.indent();
	out << "throw Components::CreateFailure();\n";
	out.unindent();
	out << "}\n\n";
	out << "Components::ExecutorLocator_var executor_locator;\n\n";
	out << "try\n{\n";
	out.indent();
	out << "executor_locator = Components::ExecutorLocator::_narrow (enterprise_component);\n";
	out.unindent();
	out << "}\n";
	out << "catch (CORBA::SystemException&)\n{\n";
	out.indent();
	out << "NORMAL_ERR (\"Home_servant: This container can only handle locator-based implementations\");\n";
	out << "throw Components::CreateFailure();\n";
	out.unindent();
	out << "}\n\n";
	switch (lc) {
		case (CIDL::lc_Session):
		{
            out << "Components::SessionComponent_var session_component;\n\n";
			break;
		}
		case (CIDL::lc_Extension) :
		{
			out << "Components::ExtensionComponent_var extension_component;\n\n";
			break;
		}
		default:
		{
			// not supported lifecycle
		}
	}
	out << "try\n{\n";
	out.indent();
	switch (lc) {
		case (CIDL::lc_Session):
		{
			out << "session_component = Components::SessionComponent::_narrow (enterprise_component);\n";
			break;
		}
		case (CIDL::lc_Extension) :
		{
			out << "extension_component = Components::ExtensionComponent::_narrow (enterprise_component);\n";
			break;
		}
		default:
		{
			// not supported lifecycle
		}
	}
	out.unindent();
	out << "}\n";
	out << "catch (CORBA::SystemException&)\n{\n";
	out.indent();
	out << "NORMAL_ERR (\"Home_servant: This is a session container, but created component is not a session component\");\n";
	out << "throw Components::CreateFailure();\n";
	out.unindent();
	out << "}\n\n";
	out << "// Create a new context\n";
	out << mapFullNameLocal(comp) << "_ContextImpl_var new_context = new ";
	out << mapFullNameServant(comp) << "_Context_callback();\n\n";
	if (lc==CIDL::lc_Extension) {
		out << "// Set container interceptor registration on context\n";
		out << "new_context -> set_server_interceptor_dispatcher_registration(server_dispatcher_.in());\n";
		out << "new_context -> set_client_interceptor_dispatcher_registration(client_dispatcher_.in());\n";
		out << "\n";
		out << "new_context-> set_contract_data (config);\n\n";

   	};	

	out << "// Set context on component\n";
	switch (lc) {
		case (CIDL::lc_Session):
		{
			out << "session_component->set_session_context (new_context.in());\n\n";
			break;
		}
		case (CIDL::lc_Extension) :
		{
			out << "extension_component->set_extension_context (new_context.in());\n\n";
			break;
		}
		default:
		{
			// not supported lifecycle
		}
	}
	out << "// Incarnate our component instance (create reference, register servant factories, ...\n";
	out << "Qedo::ComponentInstance& component_instance = this->incarnate_component\n";
	out << "	(executor_locator, dynamic_cast < Qedo::CCMContext* >(new_context.in()), config);\n\n";
	out << "// register servant factory\n";
	out << "servant_registry_->register_servant_factory(component_instance.object_id_, ";
	out << mapFullNameServant(comp) << "::cleaner_.factory_);\n\n";
	out << "// Extract our Key out of the object reference\n";
	out << "#ifdef TAO_ORB\n";
	out << "CORBA::OctetSeq* key = Qedo::Key::key_value_from_object_id(component_instance.object_id_);\n\n";
	out << "#else\n";
	out << "CORBA::OctetSeq_var key = Qedo::Key::key_value_from_object_id(component_instance.object_id_);\n\n";
	out << "#endif\n";
	out << "// register all ports\n";
	genFacetRegistration(comp);
	genReceptacleRegistration(comp);
	genEmitterRegistration(comp);
	genPublisherRegistration(comp);
	genConsumerRegistration(comp);

	out << "CORBA::RepositoryIdSeq streamtypes;\n\n";
	genSinkRegistration(comp);
	genSourceRegistration(comp);
	out << "\nthis->finalize_component_incarnation(component_instance.object_id_);\n\n";
	out << mapFullName(comp) << "_var servant = ";
	out << mapFullName(comp) << "::_narrow (component_instance.component_ref());\n\n";
	out << "return servant._retn();\n";
	out.unindent();
	out << "}\n\n\n";
}


void
GeneratorServantC::genFacetRegistration(IR__::ComponentDef_ptr comp)
{
	// handle base component
	IR__::ComponentDef_var base = comp->base_component();
	if(base)
	{ 
		genFacetRegistration(base);
	}

	IR__::ProvidesDefSeq_var facets = comp->provides_interfaces();
	CORBA::ULong len = facets->length();
	CORBA::ULong i;
	std::string rep_id;
	std::string name;
	for( i= 0; i < len; i++)
	{
		name = (*facets)[i]->name();
		IR__::InterfaceDef_var intf = IR__::InterfaceDef::_narrow((*facets)[i]->interface_type());
		if( !CORBA::is_nil(intf) )
		{
			rep_id = intf->id();
		}
		else
		{
			rep_id = "IDL:omg.org/CORBA/Object:1.0";
		}

		out << "CORBA::Object_var "	<< name << "_ref = this->create_object_reference(key, \"" << rep_id << "\");\n";
		out << "PortableServer::ObjectId_var " << name << "_object_id = this->reference_to_oid (";
		out << name << "_ref);\n";
		out << "servant_registry_->register_servant_factory (";
//		out << name << "_object_id, " << mapFullNameServant((*facets)[i]) << "::cleaner_.factory_);\n";
		out << name << "_object_id, " << mapFullNameServant(component_) <<"::" << name << "::cleaner_.factory_);\n";
		out << "component_instance.ccm_object_executor_->add_facet(\"";
		out << name << "\", \"" << rep_id << "\", " << name << "_ref);\n\n";
	}
}


void
GeneratorServantC::genReceptacleRegistration(IR__::ComponentDef_ptr comp)
{
	// handle base component
	IR__::ComponentDef_var base = comp->base_component();
	if(base)
	{ 
		genReceptacleRegistration(base);
	}

	IR__::UsesDefSeq_var receptacles = comp->uses_interfaces();
	CORBA::ULong len = receptacles->length();
	CORBA::ULong i;
	std::string rep_id;
	for( i= 0; i < len; i++)
	{
		IR__::InterfaceDef_var intf = IR__::InterfaceDef::_narrow((*receptacles)[i]->interface_type());
		if( !CORBA::is_nil(intf) )
		{
			rep_id = intf->id();
		}
		else
		{
			rep_id = "IDL:omg.org/CORBA/Object:1.0";
		}

		out << "component_instance.ccm_object_executor_->add_receptacle(\"";
		out << (*receptacles)[i]->name() << "\", \"" << rep_id << "\", ";
		if((*receptacles)[i]->is_multiple()) { 
			out << "true);\n\n";
		}
		else { 
			out << "false);\n\n";
		}
	}
}


void
GeneratorServantC::genEmitterRegistration(IR__::ComponentDef_ptr comp)
{
	// handle base component
	IR__::ComponentDef_var base = comp->base_component();
	if(base)
	{ 
		genEmitterRegistration(base);
	}

	IR__::EmitsDefSeq_var emits = comp->emits_events();
	CORBA::ULong len = emits->length();
	CORBA::ULong i;
	for( i= 0; i < len; i++)
	{
		std::string id = (*emits)[i]->event()->id();
		id.insert(id.find_last_of(":"), "Consumer");

		out << "component_instance.ccm_object_executor_->add_emitter(\"";
		out << (*emits)[i]->name() << "\", \"" << id << "\");\n\n";
	}
}


void
GeneratorServantC::genPublisherRegistration(IR__::ComponentDef_ptr comp)
{
	// handle base component
	IR__::ComponentDef_var base = comp->base_component();
	if(base)
	{ 
		genPublisherRegistration(base);
	}

	IR__::PublishesDefSeq_var publishes = comp->publishes_events();
	CORBA::ULong len = publishes->length();
	CORBA::ULong i;
	for( i= 0; i < len; i++)
	{
		std::string id = (*publishes)[i]->event()->id();
		id.insert(id.find_last_of(":"), "Consumer");

		out << "component_instance.ccm_object_executor_->add_publisher(\"";
		out << (*publishes)[i]->name() << "\", \"" << id << "\");\n\n";
	}
}


void
GeneratorServantC::genConsumerRegistration(IR__::ComponentDef_ptr comp)
{
	// handle base component
	IR__::ComponentDef_var base = comp->base_component();
	if(base)
	{ 
		genConsumerRegistration(base);
	}

	IR__::ConsumesDefSeq_var consumes = comp->consumes_events();
	CORBA::ULong len = consumes->length();
	CORBA::ULong i;
	for( i= 0; i < len; i++)
	{
		std::string id = (*consumes)[i]->event()->id();
		id.insert(id.find_last_of(":"), "Consumer");
		std::string name = (*consumes)[i]->name();
		out << "CORBA::Object_var " << name << "_ref = this->create_object_reference (key, \"" << id << "\");\n";
		out << "PortableServer::ObjectId_var " << name << "_id = this->reference_to_oid(" << name << "_ref);\n";
		out << "servant_registry_->register_servant_factory (" << name << "_id, ";
//		out << mapFullNameServant((*consumes)[i]) << "::cleaner_.factory_);\n";
		out << mapFullNameServant(component_) << "::" << name << "::cleaner_.factory_);\n";
		out << "Components::EventConsumerBase_var " << name << "_sink = Components::EventConsumerBase::_narrow(";
		out << name << "_ref);\n";
		out << "component_instance.ccm_object_executor_->add_consumer(\"";
		out << name << "\", \"" << id << "\", " << name << "_sink);\n\n";
	}
}



void
GeneratorServantC::genSinkRegistration(IR__::ComponentDef_ptr comp)
{
	// handle base component
	IR__::ComponentDef_var base = comp->base_component();
	if(base)
	{ 
		genSinkRegistration(base);
	}

	IR__::SinkDefSeq_var sinks = comp->sinks();
	CORBA::ULong len = sinks->length();
	CORBA::ULong i;
	for( i= 0; i < len; i++)
	{
		std::string sink_name = sinks[i]->name();

		out << "//\n// the sink port " << sink_name << "\n//\n";
		out << "CORBA::Object_var " << sink_name << "_ref = this->create_object_reference (key, \"IDL:StreamComponents/SinkStreamPort:1.0\");\n";
		out << "PortableServer::ObjectId_var " << sink_name << "_object_id = this->reference_to_oid (" << sink_name << "_ref);\n";
		out << "servant_registry_->register_servant_factory (" << sink_name << "_object_id,";
		out << mapFullNameServant(sinks[i]) << "::cleaner_.factory_);\n";
		out << "StreamComponents::SinkStreamPort_var " << sink_name << " = StreamComponents::SinkStreamPort::_narrow (" << sink_name << "_ref);\n\n";

		// Resolve atomic stream types from logical stream type and generate code for it
		IR__::StreamTypeDefSeq atomic_types;
		atomic_types.length(0);
		resolve_atomic_streamtypes (sinks[i]->stream_type(), atomic_types);
		out << "streamtypes.length (" << atomic_types.length() << ");\n";
		for (CORBA::ULong j = 0; j < atomic_types.length(); j++)
		{
			out << "streamtypes[" << j << "] = CORBA::string_dup (\"";
			out << atomic_types[j]->id() << "\");\n";
		}
		out << "\n";

		out << "// Initialize the dispatcher\n";
		out << "Qedo::StreamDataDispatcher* " << sink_name << "_dispatcher =\n"; out.indent();
		out << "new " << mapFullNameServant(sinks[i]) << "_dispatcher (executor_locator.in());\n\n"; out.unindent();
		
		out << "component_instance.stream_ccm_object_executor_->add_sink (\"" << sink_name << "\",\n"; out.indent();
		out << "\"" << sinks[i]->id() << "\"" << ", streamtypes, " << sink_name << ", " << sink_name << "_dispatcher);\n\n"; out.unindent();

		out << sink_name << "_dispatcher->_remove_ref();\n";
	}
}

void
GeneratorServantC::genSourceRegistration(IR__::ComponentDef_ptr comp)
{
	// handle base component
	IR__::ComponentDef_var base = comp->base_component();
	if(base)
	{ 
		genSourceRegistration(base);
	}

	IR__::SourceDefSeq_var sources = comp->sources();
	CORBA::ULong len = sources->length();
	CORBA::ULong i;
	for( i= 0; i < len; i++)
	{
		// Resolve atomic stream types from logical stream type and generate code for it
		IR__::StreamTypeDefSeq atomic_types;
		atomic_types.length(0);
		resolve_atomic_streamtypes (sources[i]->stream_type(), atomic_types);
		out << "streamtypes.length (" << atomic_types.length() << ");\n";
		for (CORBA::ULong j = 0; j < atomic_types.length(); j++)
		{
			out << "streamtypes[" << j << "] = CORBA::string_dup (\"";
			out << atomic_types[j]->id() << "\");\n";
		}
		out << "\n";

		out << "component_instance.stream_ccm_object_executor_->add_source (\"";
		out << sources[i]->name() << "\",\n";
		out.indent(); out.indent();
		out << "\"" << sources[i]->id() << "\",\n";
		if (sources[i]->is_multiple())
			out << "true,\n";
		else
			out << "false,\n";
		out << "streamtypes,\n";
		out << "false /* sync buffer dispatcher */);\n";
		out.unindent(); out.unindent();
	}
}


void
GeneratorServantC::genHomeServant(IR__::HomeDef_ptr home, CIDL::LifecycleCategory lc)
{
	CORBA::ULong i;
	CORBA::ULong ii;

	// handle base home
	IR__::HomeDef_var base = home->base_home();
	if(base)
	{ 
		genHomeServant(base, lc);
	}
	
//	home_ = IR__::HomeDef::_duplicate(home);
	std::string home_name = mapFullNameLocal(home_);

	// attributes
	IR__::ContainedSeq_var contained_seq = home->contents(CORBA__::dk_Attribute, false);
	CORBA::ULong len = contained_seq->length();
	for(i = 0; i < len; i++)
	{
		IR__::AttributeDef_var attribute = IR__::AttributeDef::_narrow(((*contained_seq)[i]));
		std::string attribute_name = mapName(attribute);
		
		// not read only
		if(attribute->mode() == IR__::ATTR_NORMAL)
		{
			out << "void" << "\n";
			out << class_name_ << "::" << attribute_name << "(";
			out << map_in_parameter_type(attribute->type_def()) << " param)\n";
			out << "throw(CORBA::SystemException";
			handleException(attribute);
			out << ")\n{\n";
			out.indent();
			out << "#ifdef TAO_ORB\n";
			out << home_name << "_ptr home_executor = dynamic_cast < ";
			out << home_name << "_ptr> (home_executor_.in());\n";
			out << "#else\n";
			out << home_name << "_var home_executor = ";
			out << home_name << "::_narrow (home_executor_.in());\n";
			out << "#endif\n";
			out << "if (CORBA::is_nil (home_executor))\n{\n";
			out.indent();
			out << "NORMAL_ERR (\"Home_servant: Cannot cast my executor\");\n";
			out << "throw Components::CreateFailure();\n";
			out.unindent();
			out << "}\n\n";
			out << "home_executor->" << attribute_name << "(param);\n";
			out.unindent();
			out << "}\n\n\n";
		}

		out << map_return_type(attribute->type_def()) << "\n";
		out << class_name_ << "::" << attribute_name << "()\n";
		out << "throw(CORBA::SystemException";
		handleException(attribute);
		out << ")\n{\n";
		out.indent();
		out << "#ifdef TAO_ORB\n";
		out << home_name << "_ptr home_executor = dynamic_cast < ";
		out << home_name << "_ptr > (home_executor_.in());\n";
		out << "#else\n";
		out << home_name << "_var home_executor = ";
		out << home_name << "::_narrow (home_executor_.in());\n";
		out << "#endif\n";
		out << "if (CORBA::is_nil (home_executor))\n{\n";
		out.indent();
		out << "NORMAL_ERR (\"Home_servant: Cannot cast my executor\");\n";
		out << "throw Components::CreateFailure();\n";
		out.unindent();
		out << "}\n\n";
		out << "return home_executor->" << attribute_name << "();\n";
		out.unindent();
		out << "}\n\n\n";
	}

	// operations
	contained_seq = home->contents(CORBA__::dk_Operation, false);
	len = contained_seq->length();
	for(i = 0; i < len; i++)
	{
		IR__::OperationDef_var operation = IR__::OperationDef::_narrow(((*contained_seq)[i]));
		std::string operation_name = mapName(operation);

		bool is_void = false;
		if(operation->result_def()->type()->kind() == CORBA::tk_void) { is_void = true; }

		out << map_return_type(operation->result_def()) << "\n";
		out << class_name_ << "::" << operation_name << "(";
		IR__::ParDescriptionSeq* pards = operation->params();
		CORBA::ULong len = pards->length();
		for(ii = len; ii > 0; ii--)
		{
			if(ii < len) { out << ", "; }
			IR__::ParameterDescription pardescr = (*pards)[ii - 1];
			out << map_in_parameter_type (pardescr.type_def) << " " << mapName(string(pardescr.name));
		}
		out << ")\n";
		out << "throw(CORBA::SystemException";
		handleException(operation);
		out << ")\n{\n";
		out.indent();
		out << "#ifdef TAO_ORB\n";
		out << home_name << "_ptr home_executor = dynamic_cast < ";
		out << home_name << "_ptr > (home_executor_.in());\n";
		out << "#else\n";
		out << home_name << "_var home_executor = ";
		out << home_name << "::_narrow (home_executor_.in());\n";
		out << "#endif\n";
		out << "if (CORBA::is_nil (home_executor))\n{\n";
		out.indent();
		out << "NORMAL_ERR (\"Home_servant: Cannot cast my executor\");\n";
		out << "throw Components::CreateFailure();\n";
		out.unindent();
		out << "}\n\n";
		if(!is_void) { out << "return "; }
		out << "home_executor->" << operation_name << "(";
		for(ii = len; ii > 0; ii--)
		{
			if(ii < len) { out << ", "; }
			IR__::ParameterDescription pardescr = (*pards)[ii - 1];
			out << mapName(string(pardescr.name));
		}
		out << ");\n";
		out.unindent();
		out << "}\n\n\n";
	}

	//
	// supported interfaces
	//
	IR__::InterfaceDefSeq_var supp_intfs = home -> supported_interfaces();
	for(i = 0; i < supp_intfs->length(); i++) {
		gen_supported_home_interface((*supp_intfs)[i]);
		/*
		// Attribute of supported interface
		contained_seq = (*supp_intfs)[i]->contents(CORBA__::dk_Attribute, false);
		CORBA::ULong len = contained_seq->length();
		for(i = 0; i < len; i++)
		{
			IR__::AttributeDef_var attribute = IR__::AttributeDef::_narrow(((*contained_seq)[i]));
			std::string attribute_name = mapName(attribute);
			
			// not read only
			if(attribute->mode() == IR__::ATTR_NORMAL)
			{
				out << "void" << "\n";
				out << class_name_ << "::" << attribute_name << "(";
				out << map_in_parameter_type(attribute->type_def()) << " param)\n";
				out << "throw(CORBA::SystemException";
				handleException(attribute);
				out << ")\n{\n";
				out.indent();
				out << "#ifdef TAO_ORB\n";
				out << home_name << "_ptr home_executor = dynamic_cast < ";
				out << home_name << "_ptr> (home_executor_.in());\n";
				out << "#else\n";
				out << home_name << "_var home_executor = ";
				out << home_name << "::_narrow (home_executor_.in());\n";
				out << "#endif\n";
				out << "if (CORBA::is_nil (home_executor))\n{\n";
				out.indent();
				out << "NORMAL_ERR (\"Home_servant: Cannot cast my executor\");\n";
				out << "throw Components::CreateFailure();\n";
				out.unindent();
				out << "}\n\n";
				out << "home_executor->" << attribute_name << "(param);\n";
				out.unindent();
				out << "}\n\n\n";
			}

			out << map_return_type(attribute->type_def()) << "\n";
			out << class_name_ << "::" << attribute_name << "()\n";
			out << "throw(CORBA::SystemException";
			handleException(attribute);
			out << ")\n{\n";
			out.indent();
			out << "#ifdef TAO_ORB\n";
			out << home_name << "_ptr home_executor = dynamic_cast < ";
			out << home_name << "_ptr > (home_executor_.in());\n";
			out << "#else\n";
			out << home_name << "_var home_executor = ";
			out << home_name << "::_narrow (home_executor_.in());\n";
			out << "#endif\n";
			out << "if (CORBA::is_nil (home_executor))\n{\n";
			out.indent();
			out << "NORMAL_ERR (\"Home_servant: Cannot cast my executor\");\n";
			out << "throw Components::CreateFailure();\n";
			out.unindent();
			out << "}\n\n";
			out << "return home_executor->" << attribute_name << "();\n";
			out.unindent();
			out << "}\n\n\n";
		}

		// Operation of supported interface
		contained_seq = (*supp_intfs)[i]->contents(CORBA__::dk_Operation, false);
		len = contained_seq->length();
		for(i = 0; i < len; i++)
		{
			IR__::OperationDef_var operation = IR__::OperationDef::_narrow(((*contained_seq)[i]));
			std::string operation_name = mapName(operation);

			bool is_void = false;
			if(operation->result_def()->type()->kind() == CORBA::tk_void) { is_void = true; }

			out << map_return_type(operation->result_def()) << "\n";
			out << class_name_ << "::" << operation_name << "(";
			IR__::ParDescriptionSeq* pards = operation->params();
			CORBA::ULong len = pards->length();
			for(ii = len; ii > 0; ii--)
			{
				if(ii < len) { out << ", "; }
				IR__::ParameterDescription pardescr = (*pards)[ii - 1];
				out << map_in_parameter_type (pardescr.type_def) << " " << mapName(string(pardescr.name));
			}
			out << ")\n";
			out << "throw(CORBA::SystemException";
			handleException(operation);
			out << ")\n{\n";
			out.indent();
			out << "#ifdef TAO_ORB\n";
			out << home_name << "_ptr home_executor = dynamic_cast < ";
			out << home_name << "_ptr > (home_executor_.in());\n";
			out << "#else\n";
			out << home_name << "_var home_executor = ";
			out << home_name << "::_narrow (home_executor_.in());\n";
			out << "#endif\n";
			out << "if (CORBA::is_nil (home_executor))\n{\n";
			out.indent();
			out << "NORMAL_ERR (\"Home_servant: Cannot cast my executor\");\n";
			out << "throw Components::CreateFailure();\n";
			out.unindent();
			out << "}\n\n";
			if(!is_void) { out << "return "; }
			out << "home_executor->" << operation_name << "(";
			for(ii = len; ii > 0; ii--)
			{
				if(ii < len) { out << ", "; }
				IR__::ParameterDescription pardescr = (*pards)[ii - 1];
				out << mapName(string(pardescr.name));
			}
			out << ");\n";
			out.unindent();
			out << "}\n\n\n";

		}
				*/
	
	};

	handleFactory(home);
	handleFinder(home);
}

void
GeneratorServantC::gen_supported_home_interface(IR__::InterfaceDef_ptr interf) 
{
		std::string home_name = mapFullNameLocal(home_);

		IR__::InterfaceDefSeq_var base_seq = interf->base_interfaces();
		CORBA::ULong i;
		for( i= 0; i < base_seq->length(); i++)
		{
			gen_supported_home_interface((*base_seq)[i]);
		}

		// Attribute of supported interface
		IR__::ContainedSeq_var contained_seq = interf->contents(CORBA__::dk_Attribute, false);
		CORBA::ULong len = contained_seq->length();
		for(i = 0; i < len; i++)
		{
			IR__::AttributeDef_var attribute = IR__::AttributeDef::_narrow(((*contained_seq)[i]));
			std::string attribute_name = mapName(attribute);
			
			// not read only
			if(attribute->mode() == IR__::ATTR_NORMAL)
			{
				out << "void" << "\n";
				out << class_name_ << "::" << attribute_name << "(";
				out << map_in_parameter_type(attribute->type_def()) << " param)\n";
				out << "throw(CORBA::SystemException";
				handleException(attribute);
				out << ")\n{\n";
				out.indent();
				out << "#ifdef TAO_ORB\n";
				out << home_name << "_ptr home_executor = dynamic_cast < ";
				out << home_name << "_ptr> (home_executor_.in());\n";
				out << "#else\n";
				out << home_name << "_var home_executor = ";
				out << home_name << "::_narrow (home_executor_.in());\n";
				out << "#endif\n";
				out << "if (CORBA::is_nil (home_executor))\n{\n";
				out.indent();
				out << "NORMAL_ERR (\"Home_servant: Cannot cast my executor\");\n";
				out << "throw Components::CreateFailure();\n";
				out.unindent();
				out << "}\n\n";
				out << "home_executor->" << attribute_name << "(param);\n";
				out.unindent();
				out << "}\n\n\n";
			}

			out << map_return_type(attribute->type_def()) << "\n";
			out << class_name_ << "::" << attribute_name << "()\n";
			out << "throw(CORBA::SystemException";
			handleException(attribute);
			out << ")\n{\n";
			out.indent();
			out << "#ifdef TAO_ORB\n";
			out << home_name << "_ptr home_executor = dynamic_cast < ";
			out << home_name << "_ptr > (home_executor_.in());\n";
			out << "#else\n";
			out << home_name << "_var home_executor = ";
			out << home_name << "::_narrow (home_executor_.in());\n";
			out << "#endif\n";
			out << "if (CORBA::is_nil (home_executor))\n{\n";
			out.indent();
			out << "NORMAL_ERR (\"Home_servant: Cannot cast my executor\");\n";
			out << "throw Components::CreateFailure();\n";
			out.unindent();
			out << "}\n\n";
			out << "return home_executor->" << attribute_name << "();\n";
			out.unindent();
			out << "}\n\n\n";
		}

		// Operation of supported interface
		contained_seq = interf->contents(CORBA__::dk_Operation, false);
		len = contained_seq->length();
		for(i = 0; i < len; i++)
		{
			IR__::OperationDef_var operation = IR__::OperationDef::_narrow(((*contained_seq)[i]));
			std::string operation_name = mapName(operation);

			bool is_void = false;
			if(operation->result_def()->type()->kind() == CORBA::tk_void) { is_void = true; }

			out << map_return_type(operation->result_def()) << "\n";
			out << class_name_ << "::" << operation_name << "(";
			IR__::ParDescriptionSeq* pards = operation->params();
			CORBA::ULong len = pards->length();
			CORBA::ULong ii;
			for(ii = len; ii > 0; ii--)
			{
				if(ii < len) { out << ", "; }
				IR__::ParameterDescription pardescr = (*pards)[ii - 1];
				out << map_in_parameter_type (pardescr.type_def) << " " << mapName(string(pardescr.name));
			}
			out << ")\n";
			out << "throw(CORBA::SystemException";
			handleException(operation);
			out << ")\n{\n";
			out.indent();
			out << "#ifdef TAO_ORB\n";
			out << home_name << "_ptr home_executor = dynamic_cast < ";
			out << home_name << "_ptr > (home_executor_.in());\n";
			out << "#else\n";
			out << home_name << "_var home_executor = ";
			out << home_name << "::_narrow (home_executor_.in());\n";
			out << "#endif\n";
			out << "if (CORBA::is_nil (home_executor))\n{\n";
			out.indent();
			out << "NORMAL_ERR (\"Home_servant: Cannot cast my executor\");\n";
			out << "throw Components::CreateFailure();\n";
			out.unindent();
			out << "}\n\n";
			if(!is_void) { out << "return "; }
			out << "home_executor->" << operation_name << "(";
			for(ii = len; ii > 0; ii--)
			{
				if(ii < len) { out << ", "; }
				IR__::ParameterDescription pardescr = (*pards)[ii - 1];
				out << mapName(string(pardescr.name));
			}
			out << ");\n";
			out.unindent();
			out << "}\n\n\n";
		}
}
} //
