(* *********************************************************************** *)
(*									   *)
(* Project: CATS 							   *)
(* Author:  Bartek Klin                                          	   *)
(* Date: 19.03.2000				 			   *)
(* Purpose of this file: Static semantics of architectural specifications  *)
(*			 						   *)	
(*									   *)
(* *********************************************************************** *)

(* The static analysis of architectural specifications follows the
   static semantic rules in the CASL Semantics v.0.96, July 1999
   (CoFI study note S-9). The page numbers refer to this document. *)

structure  ArchitecturalAnalysis =
struct
local open Utils AS LocalEnv GlobalEnv Morphisms Symbols
           ArchTypes SharingAnalysis in
infix mem;

exception INTERNAL_ERROR;
exception UNIMPLEMENTED;

exception ARCH_EXCEPTION of (string list);

(*exception NO_MORPHISM_EXTENSION of string;*)

(* Technical support functions *)

fun intersect(l1,l2) = 
  filter (fn x => (x mem l1) andalso (x mem l2)) (remove_dups(l1@l2))
;

fun remove_list(l1,[]) = l1
  | remove_list(l1,h::t) = remove h (remove_list(l1,t))
;

fun is_subset([],_) = true
  | is_subset(h::t,l) = h mem l andalso is_subset(t,l)
;

(* External functions *)

(* LocalEnv.local_env * GlobalEnv.global_env * L_SPEC -> 
   LocalEnv.local_env * GlobalEnv.morphism * LocalEnv.local_env * L_SPEC *)
fun stat_spec((lenv:LocalEnv.local_env),(Gamma:GlobalEnv.global_env),(sp:L_SPEC),
              (errs:string list)) =  
    let val (SPEC_ENV(lenv,_,_),sp',errs1) = 
        StructuredAnalysis.structured_analysis true (lenv,Gamma,sp)
    in
       ((lenv,sp',errs1@errs):LocalEnv.local_env * L_SPEC * string list) 
          (* morphism and second local_env removed, since they are not used; T.M. *)
    end;

(* Semantic functions *)

(* ARCH_SPEC_DEFN *)
(* p. 144 *)
(* GlobalEnv.global_env * (ARCH_SPEC_NAME * ARCH_SPEC) -> GlobalEnv.global_env *)
fun stat_arch_spec_defn(Gs as (genv,ans):GlobalEnv.global_env,(asn,ars),(errs:string list)) = 
  (case Symtab_sid.lookup(genv,asn) of
    None => let val (asig,ars',errs1) = stat_arch_spec(Gs,ars,errs) 
            in
             ((Symtab_sid.update_new((asn,arch_spec_defn_env(asig)),genv),ans),
		(asn,ars'),errs1)
            end
  | _ => (Gs,(asn,ars),"Duplicate architectural specification name"::errs))
  handle ARCH_EXCEPTION(xerrs) => (Gs,(asn,ars),xerrs)
         | Symtab_sid.DUP(id) => 
            (writeln ("Dup sid: "^BasicPrint.print_SIMPLE_ID id); raise ERROR)
         | Symtab_id.DUP(id) => 
            (writeln ("Dup id: "^BasicPrint.print_ID id); raise ERROR)
         | Symtab_str.DUP(id) => 
            (writeln ("Dup str: "^ id); raise ERROR)
and

(* ARCH_SPEC *)
(* p. 144 *)
(* GlobalEnv.global_env * ARCH_SPEC -> arch_sig *)
stat_arch_spec(Gs as (genv,_),(ars,ans),errs) = 
  case ars of 
    basic_arch_spec(uddl,ru,ans1) => 
      let val (asig,(uddl',ru'),errs') = 
         stat_basic_arch_spec(Gs,(uddl,ru),errs)
      in (asig,(basic_arch_spec(uddl',ru',ans1),ans),errs')
      end
  | named_arch_spec(asn) => (case Symtab_sid.lookup(genv,asn) of
        Some(arch_spec_defn_env(asig)) => (asig,(named_arch_spec(asn),ans),errs)
      | _ => raise ARCH_EXCEPTION(
              "Undeclared architectural specification name"::errs))
  | pos_ARCH_SPEC(r,ars1) =>
      let val (asig,(ars',ans'),errs') = stat_arch_spec (Gs,(ars1,ans),errs)
      in (asig,(pos_ARCH_SPEC(r,ars'),ans'),errs')
      end

and 

(* BASIC_ARCH_SPEC *)
(* p. 145 *)
(* GlobalEnv.global_env * (UNIT_DECL_DEFN list * RESULT_UNIT)) -> arch_sig *)
stat_basic_arch_spec(Gs:GlobalEnv.global_env,(uddl,ru),errs) = 
  let val (Cs,uddl',errs1) = stat_unit_decl_defns(Gs,uddl,errs) 
      val (usig,ru',errs2) = stat_result_unit(Gs,Cs,ru,errs1) 
  in ((Cs,usig),(uddl',ru'),errs2)
  end

and

(* UNIT_DECL_DEFN+ *)
(* p. 145 *)
(* GlobalEnv.global_env * UNIT_DECL_DEFN list -> st_unit_ctx *)
stat_unit_decl_defns(Gs:GlobalEnv.global_env,uddl,errs) =
  let val (sut',uddl',errs') =  
  foldl (fn ((suti,uddli',errsi),udd) => 
     let val (suti1,udd',errsi1) = stat_unit_decl_defn(Gs,suti,udd,errsi)
     in (suti1,udd'::uddli',errsi1)
     end)
     ((Symtab_sid.empty,[],errs),uddl)
  in
    (sut',rev uddl',errs')
  end 

and

(* UNIT_DECL_DEFN *)
(* p. 146 *)
(* GlobalEnv.global_env * st_unit_ctx * UNIT_DECL_DEFN -> st_unit_ctx *)
stat_unit_decl_defn(Gs:GlobalEnv.global_env,Cs,(udd,ans),errs) = 
  case udd of 
    unit_decl_case(ud) => 
      let val (nusig,ud',errs1) = stat_unit_decl(Gs,Cs,ud,errs) in 
      (merge_nusigs(Cs,nusig),(unit_decl_case(ud'),ans),errs1)
    end
  | unit_defn_case(ud) => 
      let val (nusig,ud',errs1) = stat_unit_defn(Gs,Cs,ud,errs) in
      (merge_nusigs(Cs,nusig),(unit_defn_case(ud'),ans),errs1)
    end
  | pos_UNIT_DECL_DEFN(r,udd1) =>
    let val (asig,(udd',ans'),errs') =
            stat_unit_decl_defn(Gs,Cs,(udd1,ans),errs)
    in (asig,(pos_UNIT_DECL_DEFN(r,udd'),ans'),errs')
    end

and

(* RESULT_UNIT *)
(* p. 146 *)
(* GlobalEnv.global_env * st_unit_ctx * RESULT_UNIT -> unit_sig *)
stat_result_unit(Gs,Cs,result_unit(ue,ans),errs) =
  let val ((ctgs,tsig),ue',errs1) = stat_unit_expression(Gs,Cs,ue,errs) in
    if forall' (fn (ss,tm) => 
      forall' (fn xi => Symtab_esym.lookup(tm,xi)=None orelse
                        Symtab_esym.lookup(tm,xi)=Some([])) 
      (e_sig_symbols(ss)))
      ctgs
    then (untag_unit_sig(ctgs,tsig),result_unit(ue',ans),errs1)
    else raise ARCH_EXCEPTION(
           "Parameters of result unit share with local environment"::errs1)
  end
  | stat_result_unit(Gs,Cs,pos_RESULT_UNIT(r,ru),errs) =
    let val (tsig,ru',errs') = stat_result_unit(Gs,Cs,ru,errs)
    in (tsig,pos_RESULT_UNIT(r,ru'),errs')
    end
and
 
(* UNIT_DECL *)
(* p. 147-148 *)
(* GlobalEnv.global_env * st_unit_ctx * UNIT_DECL -> named_unit_sig *)
stat_unit_decl(Gs:GlobalEnv.global_env,Cs,unit_decl(un,us,ui),errs) = 
  ((let val ((sigI,mI),ui',errs1) = stat_unit_imports(Gs,Cs,ui,un,errs)
      val ((sigl,sig0),us',errs2) = stat_unit_spec(sigI,Gs,us,errs1) 
  in
  if compatible_sigs([sigI,sig0]) then
    let val (_,m0)::mil = map (fn sigi => (sigi,
      let val comsyml = intersect(e_sig_symbols(sigi),e_sig_symbols(sigI)) in
      foldl (fn (mii,xi) => 
        case Symtab_esym.lookup(mI,xi) of
          Some(mIval) => Symtab_esym.update_new((xi,mIval),mii)
        | None => mii) 
        (Symtab_esym.empty,comsyml)
      end)) (sig0::sigl) 
    in
      case Symtab_sid.lookup(Cs,un) of
        Some(_) => (Symtab_sid.empty,unit_decl(un,us',ui'),
                    "Unit name already declared"::errs2)
      | None => 
        let val utsig = (mil,merge_tag_sigs([(sig0,m0),(sigI,mI)])) 
        val errs3 = 
          if ensures_model_compatibility([(sigI,mI),merge_tag_sigs(mil)]) 
          then errs2 
          else (("Warning: I am unable to prove model compatibility of "^
                "unit parameters and imports")::errs2) 
        in 
          (Symtab_sid.update_new((un,utsig),Symtab_sid.empty),
          unit_decl(un,us',ui'),errs3)
        end
    end
  else (Symtab_sid.empty,
     unit_decl(un,us',ui'),
     "Signature of unit specification is incompatible with its imports"::
      errs2)
  end)
  handle ARCH_EXCEPTION(xerrs) => (Symtab_sid.empty,unit_decl(un,us,ui),xerrs)
  )
  | stat_unit_decl (Gs,Cs,pos_UNIT_DECL(r,ud),errs) =
    let val (env,ud',errs') = stat_unit_decl(Gs,Cs,ud,errs)
    in (env,pos_UNIT_DECL(r,ud'),errs')
    end

and

(* UNIT_IMPORTS *)
(* p. 148 *)
(* GlobalEnv.global_env * st_unit_ctx * UNIT_IMPORTS -> tag_sig *)
stat_unit_imports(Gs,Cs,unit_imports(utl),un,errs) = 
  let val (tsigs,utl',errs1) = 
    foldl (fn ((tsigsi,utli',errsi),ut) => 
      let val (tsig,ut',errsi1) = stat_unit_term(Gs,Cs,ut,errsi) in
        (tsig::tsigsi,ut'::utli',errsi1)
      end)
      (([],[],errs),utl) 
  in
    if compatible_tag_sigs(tsigs) then
      let val errs2 = if ensures_model_compatibility(tsigs) then errs1 
        else (("Warning in declaration of unit "^
               BasicPrint.print_SIMPLE_ID un^
               "\nI am unable to prove model compatibility "^
              "of unit imports")::errs1)
      in
        (merge_tag_sigs(tsigs),unit_imports(rev utl'),errs2)
      end
    else raise ARCH_EXCEPTION("In declaration of unit"^
               BasicPrint.print_SIMPLE_ID un^
               ",\n              imports are not compatible"::errs1)
  end
  | stat_unit_imports(Gs,Cs,pos_UNIT_IMPORTS(r,ui),un,errs) =
    let val (tsig,ui',errs') = stat_unit_imports(Gs,Cs,ui,un,errs)
    in (tsig,pos_UNIT_IMPORTS(r,ui'),errs')
    end

and  

(* UNIT_DEFN *)
(* p. 149 *)
(* GlobalEnv.global_env * st_unit_ctx * UNIT_DEFN -> named_unit_sigs *)
stat_unit_defn(Gs as (genv,ans),Cs,unit_defn(un,ue),errs) = 
  ((let val ((utsig:unit_tag_sig),ue',errs1) = 
    stat_unit_expression(Gs,Cs,ue,errs) 
   in
    case Symtab_sid.lookup(genv,un) of 
      None => (Symtab_sid.update_new((un,utsig),Symtab_sid.empty),
               unit_defn(un,ue'),errs1)
    | _ => (Symtab_sid.empty,unit_defn(un,ue'),
            "Unit name already declared"::errs1)
  end)
  handle ARCH_EXCEPTION(xerrs) => 
    (Symtab_sid.empty,unit_defn(un,ue),xerrs))
 | stat_unit_defn(Gs,Cs,pos_UNIT_DEFN(r,ud),errs) =
   let val (env,ud',errs') = stat_unit_defn(Gs,Cs,ud,errs)
   in (env,pos_UNIT_DEFN(r,ud'),errs')
   end
and

(* UNIT_SPEC_DEFN *)
(* p. 150 *)
(* GlobalEnv.global_env * (UNIT_SPEC_NAME * UNIT_SPEC) -> GlobalEnv.global_env *)
stat_unit_spec_defn(Gs as (genv,ans),(usn,us),errs) = 
  (case Symtab_sid.lookup(genv,usn) of 
    None => let val (usig,us',errs1) = 
            stat_unit_spec(LocalEnv.empty_local_env,Gs,us,errs) in
              ((Symtab_sid.update_new((usn,unit_spec_defn_env(usig)),genv),ans),
               (usn,us'),errs1)
            end
  | _ => (Gs,(usn,us),"Duplicate unit specification name"::errs))
  handle ARCH_EXCEPTION(xerrs) => (Gs,(usn,us),xerrs)

and

(* UNIT_SPEC *)
(* p. 150-151 *)
(* CaslEnv.lenv * GlobalEnv.global_env * UNIT_SPEC -> unit_sig *)
stat_unit_spec(sigI,Gs as (genv,ans):GlobalEnv.global_env,us,errs) = case us of 
    unit_type_case(ut) => 
     let val (usig,ut',errs') = stat_unit_type(sigI,Gs,ut,errs)
     in (usig,unit_type_case(ut'),errs')
     end
  | spec_name_case(sn) => (case Symtab_sid.lookup(genv,sn) of
        Some(unit_spec_defn_env(usig)) => (usig,spec_name_case(sn),errs)
        (* A hack. Basic specification name as unit specification name. *)
      | Some(spec_defn_env(_,SPEC_ENV(sigma,_,_))) =>
           (([],sigma),spec_name_case(sn),errs)
      | _ => raise ARCH_EXCEPTION("Undeclared unit specification name"::errs))
  | arch_spec_case(ars) => 
     let val ((_,usig),ars',errs1)=stat_arch_spec(Gs,ars,errs) 
     in (usig,arch_spec_case(ars'),errs1) end
  | closed(us) => let val (usig,us',errs') = stat_closed_unit_spec(Gs,us,errs)
     in (usig,closed(us'),errs')
     end
  | pos_UNIT_SPEC(r,us1) =>
    let val (usig,us',errs') = stat_unit_spec(sigI,Gs,us1,errs)
    in (usig,pos_UNIT_SPEC(r,us'),errs')
    end

and

(* UNIT_TYPE *)
(* p. 152 *)
(* CaslEnv.lenv * GlobalEnv.global_env * UNIT_TYPE -> unit_sig *)
stat_unit_type(sigI,Gs:GlobalEnv.global_env,unit_type(spl,sp),errs) =
  let val (deltas,spl',errs1) = 
    foldl (fn ((deltasi,spli',errsi),spi) => 
      let val (deltai,spi',errsi1) =        
        stat_spec(LocalEnv.empty_local_env,Gs,spi,errsi) 
      in (deltai::deltasi,spi'::spli',errsi1)
      end)
      (([],[],errs),spl)
  val sigmas = (* map (fn (_,_,sgi) => sgi) *) deltas in  (* T.M. *)
  if compatible_sigs(sigI::sigmas) then
    let val sig0 = merge_lenvlist(sigI::sigmas) 
    val (sigma,sp',errs2) = stat_spec(sig0,Gs,sp,errs1) in (*** _,_, removed, T.M.  *)
      ((sigmas,sigma),unit_type(rev spl',sp'),errs2)
    end
  else raise ARCH_EXCEPTION("Unit type is internally incompatible"::errs1)
  end 
  | stat_unit_type(sigI,Gs,pos_UNIT_TYPE(r,ut),errs) =
    let val (sigs,ut',errs') = stat_unit_type(sigI,Gs,ut,errs)
    in (sigs,pos_UNIT_TYPE(r,ut'),errs')
    end

and

(* CLOSED_UNIT_SPEC *)
(* p. 153 *)
(* GlobalEnv.global_env * UNIT_SPEC -> unit_sig *)
stat_closed_unit_spec(Gs,us,errs) =
  stat_unit_spec(LocalEnv.empty_local_env,Gs,us,errs)

and
(* UNIT_EXPRESSION *)
(* p. 154-155 *)
(* GlobalEnv.global_env * st_unit_ctx * UNIT_EXPRESSION -> unit_tag_sig *)
stat_unit_expression(Gs,Cs,unit_expression(ubl,ut),errs) = 
  let val (uns,sigmas,ubl',errs1) = 
    foldl (fn ((unsi,sigmasi,ubsi',errsi),ub) =>
      let val (uni,sigmai,ubi',errsi1) = stat_unit_binding(Gs,ub,errsi) in
        (uni::unsi,sigmai::sigmasi,ubi'::ubsi',errsi1)
      end)
      (([],[],[],errs),ubl) 
  val bindings = zip(uns,sigmas) in
  if length(remove_dups uns)=length(uns) andalso
     (forall' (fn un => (case Symtab_sid.lookup(Cs,un) of
               None => true | _ => false)) uns) then
  if compatible_sigs(sigmas) then
    let val esymss = map (fn (un,sigma) => (un,e_sig_symbols(sigma))) bindings
    val ms = map (fn (un,esyms) => 
      foldl (fn (mi,x) => Symtab_esym.update_new((x,
        foldl (fn (refl,(unj,sigmaj)) => 
          if (not(unj=un)) andalso (x mem (e_sig_symbols(sigmaj))) 
          then (unj,x)::refl else refl)
          ([],bindings)),mi))
        (Symtab_esym.empty,esyms))
      esymss
    val nusigpairs = zip(uns,map (fn x => ([],x)) (zip(sigmas,ms)))
    val nusig = foldl (fn (nusigi,nusigpair) => 
      Symtab_sid.update_new(nusigpair,nusigi))
      (Symtab_sid.empty,nusigpairs)
    val ((sigma,m),ut',errs2) = 
      stat_unit_term(Gs,merge_nusigs(Cs,nusig),ut,errs1) in
    if is_subsig(merge_lenvlist(sigmas),sigma) andalso
       forall' (fn x => 
         exists' (fn (shun,shnm) => (shnm=x) andalso (shun mem uns)) 
           (case Symtab_esym.lookup(m,x) of
             None => [] | Some(y) => y))
         (e_sig_symbols(merge_lenvlist(sigmas)))
    then
      let val m1 = foldl (fn (m1i,x) => 
        Symtab_esym.update_new((x,
          foldl (fn (m1vali,(shun,shnm)) => if shun mem uns then m1vali
                                          else ((shun,shnm)::m1vali))
            ([],case Symtab_esym.lookup(m,x) of 
                  None => [] | Some(l) => l)),
          m1i))
        (Symtab_esym.empty,e_sig_symbols(sigma))
      val result = 
        (map (fn sigmai => (sigmai,Symtab_esym.empty)) sigmas,(sigma,m1))
      in ((result:unit_tag_sig),unit_expression(rev ubl',ut'),errs2)
      end
    else raise ARCH_EXCEPTION(
      "Unit term does not extend its bound parameters"::errs2)
    end
  else raise ARCH_EXCEPTION(
      "Unit term bindings are incompatible"::errs1)
  else raise ARCH_EXCEPTION(
      "Bounded names are duplicate or declared in environment"::errs1)
  end
 | stat_unit_expression(Gs,Cs,pos_UNIT_EXPRESSION(r,uexp),errs) =
   let val (utsigs,uexp',errs') = stat_unit_expression(Gs,Cs,uexp,errs)
   in (utsigs,pos_UNIT_EXPRESSION(r,uexp'),errs')
   end

and

(* UNIT_BINDING *)
(* p. 156 *)
(* GlobalEnv.global_env * UNIT_BINDING -> (UNIT_NAME * CaslEnv.lenv) *)
stat_unit_binding(Gs,unit_binding(un,us),errs) = 
  (case stat_unit_spec(LocalEnv.empty_local_env,Gs,us,errs) of
      (([],sigma),us',errs1) => (un,sigma,unit_binding(un,us'),errs1)
    | (_,_,errs1) => raise ARCH_EXCEPTION(
                     "Parametrized units cannot be bound"::errs1)
   )
  | stat_unit_binding(Gs,pos_UNIT_BINDING(r,ub),errs) =
    let val (un,sigma,ub',errs') = stat_unit_binding(Gs,ub,errs)
    in (un,sigma,pos_UNIT_BINDING(r,ub'),errs')
    end

and

(* UNIT_TERM *)
(* p. 157 - rules elided in the semantics *)
(* GlobalEnv.global_env * st_unit_ctx * UNIT_TERM -> tag_sig *)
stat_unit_term(Gs,Cs,ut,errs) = (case ut of
    unit_translation(ut1,rn) => 
      let val (tsig,(ut1',rn'),errs') = 
        stat_unit_translation(Gs,Cs,(ut1,rn),errs)
      in (tsig,unit_translation(ut1',rn'),errs')
      end
  | unit_reduction(ut1,rs) => 
      let val (tsig,(ut1',rs'),errs') = 
        stat_unit_reduction(Gs,Cs,(ut1,rs),errs)
      in (tsig,unit_reduction(ut1',rs'),errs')
      end
  | amalgamation(utl) =>
      let val (tsig,utl',errs') = 
        stat_amalgamation(Gs,Cs,utl,errs)
      in (tsig,amalgamation(utl'),errs')
      end
  | local_unit(udl,ut1) => 
      let val (tsig,(udl',ut1'),errs') = 
        stat_local_unit(Gs,Cs,(udl,ut1),errs)
      in (tsig,local_unit(udl',ut1'),errs')
      end
  | unit_appl(un,faul) => 
      let val (tsig,(un',faul'),errs') = 
        stat_unit_appl(Gs,Cs,(un,faul),errs)
      in (tsig,unit_appl(un',faul'),errs')
      end
  | pos_UNIT_TERM(r,b,ut1) =>
      let val (tsig,ut',errs') = stat_unit_term(Gs,Cs,ut1,errs)
      in (tsig,pos_UNIT_TERM(r,b,ut'),errs')
      end)

and

(* UNIT_TRANSLATION *)
(* p. 158 *)
(* GlobalEnv.global_env * st_unit_ctx * (UNIT_TERM * RENAMING) -> tag_sig *)
stat_unit_translation(Gs,Cs,(ut,ren),errs) =
  let val ((sigma,m),ut',errs1) = stat_unit_term(Gs,Cs,ut,errs) 
  val (mor,(sigma1,_),errs2) =   (* [] replaced by _; T.M.*)
    ((Symmaps_analysis.stat_renaming((sigma,[]),ren,errs1))   (* Empty ANNO list added; T.M. *)
      handle (STAT_EXCEPTION s) => raise 
        ARCH_EXCEPTION(("Wrong symbol map: "^s)::errs1))        
  val m1 = tagging_map_via_morphism(sigma,mor,sigma1,m)
  val errs3 = 
    if induces_translations((sigma,m),mor,sigma1) 
    then errs2 else 
      ("Warning: unable to prove model correctness of translation"::errs2)  
  in ((sigma1,embedding_closure(m1)),(ut',ren),errs3)
  end 

and

(* UNIT_REDUCTION *)
(* p. 159 *)
(* GlobalEnv.global_env * st_unit_ctx * (UNIT_TERM * RESTRICTION) -> tag_sig *)
stat_unit_reduction(Gs,Cs,(ut,res),errs) =
  let val ((sigma,m),ut',errs1) = stat_unit_term(Gs,Cs,ut,errs)
  val ((sig1,_),mor,(sig2,_),errs2) = 
    ((Symmaps_analysis.stat_restriction(
         (LocalEnv.empty_signature,[]),(sigma,[]),res,errs1)) (* Empty ANNO list added; T.M. *) 
     handle (STAT_EXCEPTION s) => raise ARCH_EXCEPTION(("Wrong symbol map"^s)::errs1))
  val m1 = foldl (fn (m1i,x1) => case Symtab_esym.lookup(m,x1) of
      None => m1i
    | Some(x1val) => Symtab_esym.update_new((x1,x1val),m1i))
    (Symtab_esym.empty,e_sig_symbols(sig1))
  val m2 = tagging_map_via_morphism(sig1,mor,sig2,m1)
  val errs3 = if induces_translations((sig1,m1),mor,sig2) 
    then errs2
    else ("Warning: unable to prove model correctness of reduction"::errs2)
  in ((sig2,embedding_closure(m2)),(ut',res),errs3)
  end
  
and

(* AMALGAMATION *)
(* p. 160 *)
(* GlobalEnv.global_env * st_unit_ctx * UNIT_TERM list -> tag_sig *)
stat_amalgamation(Gs,Cs,utl,errs) =
  let val (tsigl,utl',errs1) =
    foldl (fn ((tsigli,utli',errsi),ut) =>
      let val (tsigi,uti',errsi1) = stat_unit_term(Gs,Cs,ut,errsi) in
        (tsigi::tsigli,uti'::utli',errsi1)
      end)
      (([],[],errs),utl) 
  in
    if compatible_tag_sigs(tsigl) then
      let val errs2 = if ensures_model_compatibility(tsigl) then errs1
        else ("Warning: unable to prove model correctness of amalgamation"::
               errs1)
      in (merge_tag_sigs(tsigl),rev utl',errs2)
      end
    else raise ARCH_EXCEPTION(
      "Incompatible signatures of amalgamated units"::errs1)
  end

and

(* LOCAL_UNIT *)
(* p. 160-161 *)
(* GlobalEnv.global_env * st_unit_ctx * (UNIT_DEFN list * UNIT_TERM) -> 
    tag_sig *)
stat_local_unit(Gs,Cs,(udl,ut),errs) = 
  let val (nusigL,udl',errs1) = 
    foldl (fn ((nusigLi,udli',errsi),ud) =>
      let val (nusigLi1,udi',errsi1) = 
      stat_unit_defn(Gs,merge_nusigs(Cs,nusigLi),ud,errsi) in
        (merge_nusigs(nusigLi1,nusigLi),udi'::udli',errsi1)
      end)
      ((Symtab_sid.empty,[],errs),udl)
  val ((sigma,m),ut',errs2) = 
      stat_unit_term(Gs,merge_nusigs(Cs,nusigL),ut,errs1) 
  val m1 = foldl (fn (m1i,x) =>
    let val subl = 
      foldl (fn (subli,un) => case Symtab_sid.lookup(nusigL,un) of
        None => subli
      | Some([],unval) =>
          foldl (fn (sublii,x1) => (un,x1)::sublii)
                (subli,e_sig_symbols(untag_sig(unval)))
      | Some(_,_) => subli)
      ([],map (fn (a,b) => a) (Symtab_sid.dest(nusigL)))
    val newval = case Symtab_esym.lookup(m,x) of
        None => []
      | Some(mval) => remove_list(mval,subl)
    in Symtab_esym.update_new((x,newval),m1i)
    end)
    (Symtab_esym.empty,e_sig_symbols(sigma))
  in ((sigma,m1),(rev udl',ut'),errs2)
  end     

and

(* UNIT_APPL *)
(* p. 162-164 *)
(* GlobalEnv.global_env * st_unit_ctx * (UNIT_NAME * FIT_ARG_UNIT list)) -> 
    tag_sig *)
stat_unit_appl(Gs,Cs,(un,faul),errs) =
  let val (tagsigs1,(sigma,m)) = case Symtab_sid.lookup(Cs,un) of
    Some(x) => x 
  | None => raise ARCH_EXCEPTION("Undeclared unit name: "
                     ^BasicPrint.print_SIMPLE_ID un::errs)
  val uerr = "Error in application of "^BasicPrint.print_SIMPLE_ID un^"\n"
  val tagsigs = rev tagsigs1  (*** T.M. ***)
  val (sigF,mF) = merge_tag_sigs(tagsigs) in  
  if length(tagsigs)=length(faul) then
   if length(faul)=0 then
    let val mres = 
      foldl (fn (mresi,esym) => 
        case Symtab_esym.lookup(m,esym) of
          None => Symtab_esym.update((esym,[(un,esym)]),mresi)
        | Some(l) => Symtab_esym.update((esym,(un,esym)::l),mresi))
        (Symtab_esym.empty,e_sig_symbols(sigma))
    in
      ((sigma,mres),(un,[]),errs)
    end
   else 
    let val sigs = map untag_sig tagsigs
    val (morsA,tagsigsA,faul',errs1) = 
      foldl (fn ((morsAi,tagsigsAi,fauli',errsi),((sigi,_),fau)) =>
        let val (_,morAi,tagsigAi,faui',errsi1) = 
          stat_fit_arg_unit(Gs,sigi,Cs,fau,un,errsi)
        in
          (morAi::morsAi,tagsigAi::tagsigsAi,faui'::fauli',errsi1)
        end)
      (([],[],[],errs),zip(tagsigs,faul))    
    in
    if compatible_tag_sigs(tagsigsA) andalso
       compatible_morphisms(zip(sigs,morsA))
    then
      let val (sigA,mA) = merge_tag_sigs(tagsigsA) 
      val morA = merge_morphisms(zip(sigs,morsA)) 
      val morAE = e_morphism_symmap(sigF,morA) in
      if preserves_overloading(sigF,morA,sigA) andalso
        (* mA ensures sharing required by mF *)
        forall' (fn x => case Symtab_esym.lookup(mF,x) of
          None => true | Some([]) => true | 
          Some(mFx) => 
            let val morAEx = case Symtab_esym.lookup(morAE,x) of
              None => x | Some(y) => y
            val mAmorAEx = case Symtab_esym.lookup(mA,morAEx) of
              None => [] | Some(y) => y
            in (length(intersect(mAmorAEx,mFx))>0)
            end)
          (e_sig_symbols(sigF))
      then
        let val (morAD,sigAD) = 
          (morphism_extension_along_sig_extension(sigF,sigma,morA,sigA))
          handle NO_MORPHISM_EXTENSION(arg) => raise ARCH_EXCEPTION(
            arg::(uerr^"Actual parameters don't extend to result"::errs1))
        val morR = morAD
        val morRE = e_morphism_symmap(sigma,morR)
        (* for UNA I take "__", which is not a valid SIMPLE_ID *)
        val UNA = ("__",None)
        val m1tmp = foldl (fn (m1tmpi,x) => 
          let val morAEx = case Symtab_esym.lookup(morAE,x) of
            None => x | Some(y) => y
          val mAmorAEx = case Symtab_esym.lookup(mA,morAEx) of
            None => [] | Some(y) => y
          in (Symtab_esym.update_new((x,(UNA,morAEx)::mAmorAEx),m1tmpi))
          end)
          (Symtab_esym.empty,e_sig_symbols(sigF))
        val m1 = embedding_closure(merge_tagging_maps([m,m1tmp])) 
        val mR1 = foldl (fn (mR1i,x) =>
          let val morREx = case Symtab_esym.lookup(morAE,x) of
            None => x | Some(y) => y
          in case Symtab_esym.lookup(m,x) of
             None => mR1i | Some([]) => mR1i
           | Some(mx) => Symtab_esym.update((morREx,mx),mR1i)
          end)
          (Symtab_esym.empty,e_sig_symbols(sigma))
        val mR2 = foldl (fn (mR2i,x) =>
          let val morREx = case Symtab_esym.lookup(morRE,x) of
            None => x | Some(y) => y
          val morAEx = case Symtab_esym.lookup(morAE,x) of
            None => x | Some(y) => y
          in case Symtab_esym.lookup(mA,morAEx) of
             None => mR2i | Some([]) => mR2i
           | Some(mAmorAEx) => Symtab_esym.update((morREx,mAmorAEx),mR2i)
          end)
          (Symtab_esym.empty,e_sig_symbols(sigF))
        val mR = embedding_closure(merge_tagging_maps([mR1,mR2]))
        in
        if compatible_tag_sigs([(sigA,mA),(sigAD,mR)])
        then
          let val errs2 = if ensures_model_compatibility(tagsigsA) then errs1
            else ("Warning: unable to prove model compatibility of "^
                  "application arguments"::errs1)
          val errs3 = if induces_translations((sigma,m1),morR,sigAD) andalso
                         ensures_model_compatibility([(sigA,mA),(sigAD,mR)]) 
            then errs2 
            else ("Warning: unable to prove model correctness of application"::
                  errs2)
          in
            ((merge_lenvs(sigA,sigAD),
             embedding_closure(merge_tagging_maps([mA,mR]))),
             (un,rev faul'),
             errs3)
          end         
        else raise ARCH_EXCEPTION(
          uerr^"Application arguments are incompatible with the result"::errs1) 
        end
      else raise ARCH_EXCEPTION(
        uerr^"Actual parameters do not have enough overloading and/or sharing"
          ::errs1)     
      end
    else raise ARCH_EXCEPTION(
      uerr^"Incompatible application arguments or fitting morphisms"::errs1)
    end
  else raise ARCH_EXCEPTION(uerr^"Wrong number of arguments."^
                               " Needed: "^Int.toString (length(tagsigs))^
                               " provided: "^Int.toString (length(faul))::errs)
  end

and

(* FIT_ARG_UNIT *)
(* p. 164 *)
(* Careful! Slightly different from the semantics *)
(* GlobalEnv.global_env * CaslEnv.lenv * st_unit_ctx * FIT_ARG_UNIT -> 
                     (LocalEnv.local_env * GlobalEnv.morphism * tag_sig) *)
stat_fit_arg_unit(Gs,sigma,Cs,fit_arg_unit(ut,smil),un,errs) = 
  let val ((sigA,mA),ut',errs1) = stat_unit_term(Gs,Cs,ut,errs) 
      val (r,errs2) = Symmaps_analysis.stat_symb_map_items_star(
                        sig_symbols(sigma),smil,errs1) 
      val mor = 
        ((Stat_symmaps.induced_from_to_morphism(r,sigma,sigA))
         handle (STAT_EXCEPTION s) => 
           raise ARCH_EXCEPTION(
             (
              writeln ("Error: \n"^s^"\nFitting symbol map is wrong in application of "
                         ^BasicPrint.print_SIMPLE_ID un);
              writeln "Source signature: ";
              BasicPrint.output_local_env TextIO.stdOut sigma;
              writeln "\nTarget signature: ";
              BasicPrint.output_local_env TextIO.stdOut sigA;
              writeln "";
              "Fitting symbol map is wrong")::errs2))
  in
    (sigma,mor,(sigA,mA),fit_arg_unit(ut',smil),errs2)
  end
  | stat_fit_arg_unit(Gs,sigma,Cs,pos_FIT_ARG_UNIT(r,fau),un,errs) =
    let val (sigma,mor,sigAmA,fau',errs') =
             stat_fit_arg_unit(Gs,sigma,Cs,fau,un,errs)
    in (sigma,mor,sigAmA,pos_FIT_ARG_UNIT(r,fau'),errs')
    end
;

end
end
