(* *********************************************************************** *)
(*									   *)
(* Project: CATS 							   *)
(* Author: Till Mossakowski & Kolyang, University of Bremen		   *)
(* Date: 09.06.97				 			   *)
(* Purpose of this file: Encoding of CASL into CFOL/SOL 	           *)
(*			 						   *)	
(*									   *)
(* *********************************************************************** *)


(* This implements the encoding of the CASL logic in several other
   logics.
   The theoretical background is contained in:

     Till Mossakowski: Relating CASL with other specification languages,
       Theoretical computer science, to appear.
       Available at http://www.tzi.de/~till/publications.html

   See especially encodings (5'), (3') and (7) at the first-order level.

   The logics are:

   SubPCFOL: subsorted partial first-order logic with sort generation constraints
             (the CASL logic)
   SubCFOL:  subsorted first-order logic with sort generation constraints
   CFOL:     first-order logic with sort generation constraints
   SOL:      second-order logic

   Basically, the encoding from SubPCFOL to SubCFOL encodes partiality
   by error supersorts (called questionmark- or Q-sorts here) (encoding (5')).
   SubCFOL to CFOL encodes subsort embeddings by injective functions (encoding (3')).
   CFOL to SOL encodes sort generation constraints as induction axioms (encoding (7)).

   The encodings are also lifted to the level of CASL structured specifications.

   todo: 
    Also support HOL as a target (this is more than SOL,
    since the datastructure of terms is different).
   
   Signaturfragmente: Wenn nur neue Subsortenrelation erzeugt wird,
   geht diese im Moment verloren!

   Wieso muss in CATS/basic_env bei encoding von spec_defn_envs
   der Fall null pars speziell behndelt werden?

   Encoding gem TCS-Paper korrigieren

   Better encoding via PCFOL?

   Sortgen with partial functions: raise ERROR

   Lifting von Encode zum globalen Env korrigieren
     (insbesondere s? mit in Morphismen)

*)

structure BasicEncode :
sig
     (* first argument: should projections for "t as s" be created? *)
val encode_SubPCFOL_CFOL : bool -> GlobalEnv.global_env -> GlobalEnv.global_env
val encode_SubPCFOL_SubCFOL : bool -> GlobalEnv.global_env -> GlobalEnv.global_env
(*val encode_SubCFOL_CFOL : bool -> GlobalEnv.global_env -> GlobalEnv.global_env*)
val encode_SubPCFOL_SOL : bool -> GlobalEnv.global_env -> GlobalEnv.global_env

val encode_slenv_SubPCFOL_Sub_CFOL : 
     bool -> bool -> GlobalEnv.spec_lenv -> GlobalEnv.spec_lenv
     (* first argument = true means coding to CFOL, otherwise SubCFOL *)

val lift_encoding : 
    (LocalEnv.sign * AS.L_FORMULA list -> LocalEnv.sign * AS.L_FORMULA list)
       -> 
       (GlobalEnv.global_env -> GlobalEnv.global_env)
val encode_basic_SubPCFOL_CFOL 
  : bool -> LocalEnv.sign * AS.L_FORMULA list -> LocalEnv.sign * AS.L_FORMULA list
val encode_basic_SubPCFOL_SubCFOL 
  : bool -> LocalEnv.sign * AS.L_FORMULA list -> LocalEnv.sign * AS.L_FORMULA list
val encode_ax_SubPCFOL_SubCFOL : AS.SORT list -> AS.L_FORMULA -> AS.L_FORMULA
val encode_basic_SubCFOL_CFOL
  : bool -> LocalEnv.sign * AS.L_FORMULA list -> LocalEnv.sign * AS.L_FORMULA list
val encode_ax_SubCFOL_CFOL : AS.L_FORMULA -> AS.L_FORMULA

val get_bottom : AS.SORT list -> LocalEnv.Fun_list
end
= 
struct

open Utils AS LocalEnv GlobalEnv Subsorts BasicPrint BasicAnalysis IDOrder;

infix mem \\ inter upto;

(*********************************** Special names *********************************)
fun quest (x:ID):ID = 
 compound_id (([token "?"],"?",None),[x]);

fun is_q (compound_id (([token "?"],"?",None),[x]):ID) = true
  | is_q x = false;

fun is_2q (compound_id (([token "?"],"?",None),[x]):ID) = is_q x
  | is_2q x = false;

fun force_q x = if is_q x then x else quest x

val DEFINEDNESS_NAME = simple_id ([token "def$"], "def$", None);
val INJECTION_NAME = simple_id ([token "inj$"],"inj$", None);
val EQUIPRED_NAME  = simple_id ([token "eeq$"],"eeq$",None);
val PROJECTION_NAME = simple_id([token  "proj$"],"proj$", None);
val MEMBERSHIP_NAME = simple_id ([token "memb$"],"memb$", None);
val BOTTOM_NAME =  simple_id ([token "bottom"],"bottom$", None);
fun strict_function_name f = 
( simple_id ([token ((print_ID f))], (print_ID f), None));

fun get_membership_name (sub,s) =
simple_id ([token ("memb$"^(print_ID sub)^"$"^(print_ID s))],
 "memb$"^(print_ID sub)^"$"^(print_ID s),None);

fun get_eeq_typelist s =(EQUIPRED_NAME, [pred_type (sorts [s,s])]);
fun get_defined_typelist s =(DEFINEDNESS_NAME, [pred_type (sorts [quest s])])
fun make_q_definedness_symb s = 
      pred_symb(DEFINEDNESS_NAME,Some (pred_type (sorts [quest s])))
fun make_definedness_symb s = pred_symb(DEFINEDNESS_NAME, Some (pred_type (sorts [s])))
fun make_definedness_predsymb s = pred_symb(DEFINEDNESS_NAME, Some (pred_type (sorts [s])))


fun inject_symb (s1,s2) = 
    op_symb (INJECTION_NAME, 
             Some (total_op_type (sorts [s1], s2)))

fun project_symb (s1,s2) =
    op_symb (PROJECTION_NAME, 
             Some (total_op_type (sorts [s2], quest s1)))

fun inject (s1:SORT,s2:SORT,t:TERM):TERM =
    if s1=s2 then t
    else application (inject_symb (s1,s2),terms [t]);

fun project (s1:SORT,s2:SORT,t:TERM):TERM =
    if s1=s2 then t
    else application (project_symb (s1,s2),terms [t]);

fun bottom_symb s =
    op_symb(BOTTOM_NAME,Some (total_op_type(sorts [],quest s)))                     

fun bottom_const s =
    application(bottom_symb s,terms [])

fun get_bottom_type s = total_op_type(sorts [],quest s)

fun get_bottom nil = nil
  | get_bottom (sq: SORT list) = [(BOTTOM_NAME, map get_bottom_type sq)]

(****************************************************************************************)
exception NOT_DEFINED;



fun mk_sub (sor, []) = []
| mk_sub (sor,subs::entries) = (subs,sor)::(mk_sub (sor, entries));

fun mk_subs x = foldl (op @) ([], map mk_sub x);


(*fun map f [] = []
| map f (x::xs) = (f x) :: (map f xs);*)

fun make_sort (s:string):SORT  = simple_id ([token s],s,None);


fun make_label s = [label_anno(simple_id([token s], s,None))]
(*([(simple_id([token s], s,None),None)],None)*)

val mkatom  = map atom; 
                               
fun var_term1 (x,v,s) = qual_var ( ( v,None),s);
fun var_term (v,s) = qual_var ( ( v,None),s);
fun mk_var_decllist ([],s) =  []
  | mk_var_decllist ((v1::vl),s) =  ([( v1,None)],s)::mk_var_decllist (vl,s);



fun make_vars cnt (s::rest) = 
let val newname = concat["X",radixstring (10, "0",cnt)]
in ([token newname], newname,s)::make_vars (cnt+1) rest
end
  | make_vars cnt nil = nil;
fun make_var_decl (x,v,s) = ([(v,None)],s)

fun varsortlist []  = []
  | varsortlist fargtypes = (make_vars 0 fargtypes);
fun vardecl_list [] = []
  | vardecl_list s = map make_var_decl (varsortlist s);

(*val vardecl_list = map make_var_decl o varsortlist ;*)

fun lift_sort sq s = 	if s mem sq then quest s else s

fun varslist l = map var_term1 l;
	
fun lift_op_type sq a =
 case a of
  total_op_type (sorts x, y) => 
     total_op_type (sorts (map (lift_sort sq) x), lift_sort sq y)
| partial_op_type(sorts x, y) => 
     total_op_type (sorts (map (lift_sort sq) x), lift_sort sq y)
| _ => raise ERROR (*lift_op_type sq t*)

fun totalize_op_type  a =
 case a of
  total_op_type (sorts x, y) => total_op_type (sorts x,  y)
| partial_op_type(sorts x, y) => total_op_type (sorts x, quest y)
| _ => raise ERROR (*totalize_op_type sq t*)


fun lift_op_symb sq (op_symb(f,Some t)) =
    op_symb(f,Some (lift_op_type sq t))
  | lift_op_symb _ sy = sy

fun lift_pred_type sq (pred_type (sorts x)) = pred_type (sorts (map (lift_sort sq) x))
  | lift_pred_type _ _ = raise ERROR (* lift_pred_type sq t*)




(* *********************************************************************** *)
(*									   *)
(*	Encode SubPCFOL into SubCFOL	 				   *)	
(*									   *)
(* *********************************************************************** *)

local

(**************** SubPCFOL--> SubCFOL: Sorts ****************)


 (* how to get the sorts of the functions with question marks*)

(***** SubPCFOL--> SubCFOL: Construct the new set of sorts *****)

fun make_s_prime (sq:SORT list,env:local_list):SORT list = 
let
	val (sortlist, subsortlist,varlist,funlist,predlist) = env
in 
	sortlist @ (map quest sq)
end

(* es fehlt noch der filter, naemlich (?s not <= s) ??? *)



(***** SubPCFOL--> SubCFOL: Construct the new subsort relation *****)

fun order (sq:SORT list,env:local_list):Subsort_list =
let 
   val (sortlist, subsortlist,varlist,funlist,predlist) = env;
   fun issq s = s mem sq
   fun  mksubs [] = []
      | mksubs ((supersort, subsorts)::l) = 
          if  supersort mem sq then
                       (quest supersort, (subsorts@(map quest (filter issq subsorts))))::mksubs l
           else mksubs l
 in 
      subsortlist @ (mksubs subsortlist)
 end;


exception NOT_A_FUNCTION;

(**************** SubPCFOL--> SubCFOL: Functions ****************)


(***** SubPCFOL--> SubCFOL: Projection functions *****)

fun get_projection_funs1 (sq:SORT list) [] =[]
  | get_projection_funs1 sq ((sub,s)::sexp) =
    if s=sub then get_projection_funs1 sq sexp
    else  
     (if s mem sq 
      then [(PROJECTION_NAME,[total_op_type (sorts [quest s],(quest sub))])]
      else [])@
      (PROJECTION_NAME,[total_op_type (sorts [s],quest sub)])
           ::(get_projection_funs1 sq sexp);
   
fun get_projection_funs (sq:SORT list, []) =[]
  | get_projection_funs (sq,(s,entries)::S:Subsort_list):Fun_list = 
    let
       val sexp = mk_subs ((s,entries)::S);
    in get_projection_funs1 sq sexp   end;


(** get_fun receives s_question_mark**)

(***** SubPCFOL--> SubCFOL: Functions in the signature *****)

fun get_fun1 (sq,[]) = []
|   get_fun1 (sq,total_op_type (sorts x, y)::rest) =
       if y mem sq orelse test_question_mark (sq,x) then
           total_op_type (sorts (map (lift_sort sq) x), quest y)::get_fun1 (sq,rest)  (* strict extension *)
       else get_fun1 (sq,rest) 
|   get_fun1 (sq,partial_op_type (sorts x, y)::rest) =
       (if test_question_mark (sq,x) then
           total_op_type (sorts (map (lift_sort sq) x), quest y)::get_fun1 (sq,rest)  (* strict extension *)
       else get_fun1 (sq,rest))
|   get_fun1 _ = raise ERROR
     

fun get_fun s (f,tlist) =
 let val strict_type_list = get_fun1 (s, tlist)
     val    total_type_list = map totalize_op_type tlist
in if strict_type_list = nil then [(f,total_type_list)] 
   else [(f,total_type_list), (strict_function_name f, strict_type_list)]
end
   
 

fun get_funs (sq:SORT list,f:Fun_list):Fun_list = flat (map (get_fun sq) f);


(**************** SubPCFOL--> SubCFOL: Predicates ****************)

  
exception NOT_A_PREDICATE;

(***** SubPCFOL--> SubCFOL: Existential equality  and definedness predicate *****)

fun get_special_preds (sq, []) =[]
| get_special_preds (sq,s::sortlist) = 
  (if s mem sq 
   then [get_eeq_typelist (quest s),get_defined_typelist s]
   else [])
  @get_special_preds (sq,sortlist)

(***** SubPCFOL--> SubCFOL: Predicates in the signature *****)

fun get_pred1 (sq,[]) = []
|   get_pred1 (sq,pred_type (sorts y)::rest) =
       pred_type (sorts y) ::
       (if test_question_mark (sq,y) then
           pred_type (sorts (map (lift_sort sq) y))::get_pred1 (sq,rest)
       else get_pred1 (sq,rest)) ;
     

    
fun get_pred s (p,tlist) =   (p, get_pred1 (s,tlist));
   
 

fun get_preds (sq:SORT list,p:Pred_list):Pred_list = map (get_pred sq) p;



(**************** SubPCFOL--> SubCFOL: Axioms ****************)

(***** SubPCFOL--> SubCFOL: forall x:?s. not D_s(x) <=> x=_bottom  (bottom) *****)

fun get_special_axiom_sort_defQuest  (s:SORT):L_FORMULA =
let val (name, [typePred]) = get_defined_typelist s
in
    ((quantification (forall, mk_var_decllist(["x"], quest s),
          equivalence
                (negation(atom (predication
                    (pred_symb
                       (name, Some typePred),
                       terms [var_term ("x",quest s)]))),
                 atom(strong_equation(var_term ("x",quest s),
                                 bottom_const s)))
            )),
      make_label("ga_bottom_"^print_SORTS1[s]))
 end;

(***** SubPCFOL-->SubCFOL: forall x:s'. in_{s}(x) <=> D_s(pr_{s',?s}(x)) (s<=s') (projection_membership) *****)

fun get_special_axiom_projection_membership1 [] =[]
  | get_special_axiom_projection_membership1 ((s,s')::sexp)=
    if s=s' then get_special_axiom_projection_membership1 sexp
    else
      ((quantification (forall, mk_var_decllist (["x"],  s'), 
         (equivalence
             ((atom 
                 (predication 
                    ((pred_symb 
                     (get_membership_name (s,s'), Some (pred_type (sorts [s'])))), 
                     terms [var_term ("x",s')]))),
              (atom 
               (predication 
                     ((make_q_definedness_symb s),
                     (terms
                        [project(s,s',var_term ("x",s'))])))  ))))),
       (make_label("ga_projection_membership_"^(print_ID s)^"_"^(print_ID s'))))
         ::get_special_axiom_projection_membership1 sexp 
                                         
fun get_special_axiom_projection_membership [] =[]
|  get_special_axiom_projection_membership ((s,entries)::S:Subsort_list):L_FORMULA list=
   let fun notsq (x,y) = not (is_q y)
       val sexp = filter notsq  (mk_subs ((s,entries)::S));         
   in
         get_special_axiom_projection_membership1 sexp                                       
   end;

(***** SubPCFOL--> SubCFOL: forall x:s. pr_{s',?s}(inj_{s,s'}(x))=inj_{s,?s}(x)  (s<=s') 
                                                    (projection) *****)

fun get_special_axiom_pro1 [] =[]
   | get_special_axiom_pro1 ((s,s')::sexp) =
    if s=s' then get_special_axiom_pro1 sexp
    else
              ((quantification (forall, mk_var_decllist (["x"], s), 
                  atom (strong_equation
                          (project(s,s',inject (s,s',var_term ("x",s))),
                           inject (s,quest s,var_term ("x",s)) )) )),
                           make_label ("ga_projection_"^ print_SORTS1 [s']^"_"^ print_SORTS1 [s]))
      ::get_special_axiom_pro1 sexp;

fun get_special_axiom_pro [] =[]
   | get_special_axiom_pro ((s,entries)::S:Subsort_list) :L_FORMULA list =
   let
      val sexp = mk_subs ((s,entries)::S);
   in
         get_special_axiom_pro1 sexp
  end;

(***** SubPCFOL--> SubCFOL: forall x,y:s'. pr_{s',?s}(x)=pr_{s',?s}(y) => x=y (s<=s')
                                              (projection-injectivity) *****)

fun get_special_axiom_proj_inj1 [] =[]
  | get_special_axiom_proj_inj1 ((s,s')::sexp)=
    if (s=s') then get_special_axiom_proj_inj1 sexp
    else
               ((quantification (forall, mk_var_decllist (["x","y"],  s'), 
                 (equivalence
                    (atom(strong_equation
                          (project (s,s',var_term ("x",s')),
                           project (s,s',var_term ("y",s'))) ),
                                         
                     atom(strong_equation
                           (var_term ("x",s'),var_term ("y",s'))))))),
                           make_label ("ga_projection_injectivity_"^print_SORTS1 [s']^"_"^ print_SORTS1[s]))
          ::get_special_axiom_proj_inj1 sexp;

fun get_special_axiom_proj_inj [] =[]
  | get_special_axiom_proj_inj ((s, entries)::S:Subsort_list):L_FORMULA list=
    let val sexp = mk_subs ((s,entries)::S);
     in
       get_special_axiom_proj_inj1 sexp 
     end;


(***** SubPCFOL--> SubCFOL: forall x:?s. D_s(x) <=> x in s *****)

(* Leave this out because it is not necessary to have
   two equivalent predicates *)


(***** SubPCFOL--> SubCFOL: forall x:s. D_s(inj_{s,?s}(x)) (definedness) *****)

fun get_special_axiom_sort_def (sq: SORT list)(s:SORT) =
    if s mem sq then
    let
    val (name, [typePred]) = get_defined_typelist s 
    in
        Some ((quantification (forall, mk_var_decllist(["x"],  s),
                          (atom (predication
                              (pred_symb
                                 (name, Some typePred),
                                 terms [inject(s,quest s,var_term ("x", s))])))
                  )),
     make_label( "ga_definedness_"^print_SORTS1[s]))
     end
     else None;

(* This is taken from the SubCFOL-->CFOL encoding, since it does no harm 
   and is easier to construct here *)
(***** SubCFOL-->CFOL: forall x:?s. D_s(x) <=> exists y:s . x=inj_{s,?s}(y) (membership) *****)

fun get_special_axiom_membership (s:SORT):L_FORMULA =
let val (name, [typePred]) = get_defined_typelist s
    val qs = quest s
in      ((quantification (forall, mk_var_decllist(["x"],qs), 
         (equivalence
             ((atom 
                 (predication 
                    (pred_symb 
                     (name, Some typePred), 
                     terms [var_term ("x",qs)]))),
              (quantification (exists, mk_var_decllist (["y"], s),
                  (atom (strong_equation 
                            (var_term ("x",qs),
                             inject(s,quest s,var_term ("y",s))
                             )))
              )))
         ))), 
        (make_label(  "ga_membership_"^(print_ID s))))
end                                      


(***** SubPCFOL--> SubCFOL: generated {sort ?s; ops inj:s->?s; _bottom:?s *****)

(*
fun get_special_axiom_sort_gen_bottom  (s:SORT):L_FORMULA =
   (sort_gen_ax([quest s],[bottom_symb s,inject_symb(s,quest s)]),
      make_label("ga_generated_"^print_SORTS1[quest s]))
*)

(***** SubPCFOL--> SubCFOL: forall x,y:?s. x=.=y <=> x=y /\ D_s(x)  (existential-equality) *****)

fun get_special_axiom_sprime (s:SORT):L_FORMULA =
 let val (ename, ptype) = get_eeq_typelist (quest s);
     val (dname, pdtype) = get_defined_typelist s;
 in
   ((quantification
          (forall,mk_var_decllist(["x","y"],(quest s)),
           equivalence
             (atom (predication (pred_symb (ename,Some (hd ptype)),
                   terms [var_term ("x",quest s), var_term ("y",quest s)])),
               (conjunction [atom(strong_equation (var_term ("x",quest s), var_term ("y",quest s))), 
                   atom (predication(pred_symb (dname,Some (hd pdtype)),terms [var_term ("x",quest s)]))])))),
       make_label("ga_existential_equality_"^print_SORTS1[s]))

 end;
  


(***** SubPCFOL--> SubCFOL: 
       forall x_1,...x_n. D(f(x_1,...,x_n)) => D(x_1)/\ ... /\ D(x_n)    (f \in PF)  (strictness)
       forall x_1,...x_n. D(f(x_1,...,x_n)) <=> D(x_1)/\ ... /\ D(x_n)   (f \in TF)  (strictness-and-totality) *****)
              
fun mkpred1 (x1,x,s) = 
   atom (predication (make_definedness_symb s, terms [var_term (x,s)]));
           
fun get_special_axiom_fun (sq:SORT list,(name,typelist):OP_NAME * fun_entry):L_FORMULA list =
let fun issq (x1,x,s) = is_q s

    fun get_special_axiom_fun1 t =
     case t of
     total_op_type (sorts s, res) 
       => ( let val liste = vardecl_list (map (lift_sort sq) s);
                val varsl = varsortlist (map (lift_sort sq) s);
                val varl = varslist varsl
                val qvars = filter issq varsl
         in 
         if qvars=[] then []
         else
            [((quantification (forall,liste,           
                equivalence 
                  ((atom (predication (make_q_definedness_symb res, terms
                             [(application (op_symb (name, Some (total_op_type (sorts (map (lift_sort sq) s), 
                                                                                lift_sort sq res))),
                                   terms varl))]))),
                  conjunction (map mkpred1 qvars)))),
           make_label( "ga_strictness_and_totality_"^print_OP_SYMB(op_symb(name,Some t))))]
          end)                         
     | partial_op_type (sorts s, res) 
       => ( let val liste = vardecl_list (map (lift_sort sq) s);
                val varsl = varsortlist (map (lift_sort sq) s);
                val varl = varslist varsl
                val qvars = filter issq varsl
         in 
         if qvars=[] then []
         else
            [((quantification (forall,liste,           
                implication 
                  ((atom (predication (make_q_definedness_symb res, terms
                             [(application (op_symb (name, Some (total_op_type (sorts (map (lift_sort sq) s), 
                                                                                lift_sort sq res))),
                                   terms varl))]))),
                  conjunction (map mkpred1 qvars)))),
            make_label("ga_strictness_"^print_OP_SYMB(op_symb(name,Some t))))]
          end)
     | _ => raise ERROR
in
	flat (map get_special_axiom_fun1 typelist)
end;                        
   
fun get_special_axiom_funs (sq,[]) = []
   | get_special_axiom_funs (sq,(f::funlist):Fun_list):L_FORMULA list=
    (get_special_axiom_fun (sq,f))@(get_special_axiom_funs (sq,funlist));


(***** SubPCFOL--> SubCFOL: 
       forall x_1,...x_n. P(x_1,...,x_n) => D(x_1)/\ ... /\ D(x_n)  (predicate-strictness) *****)
       
fun get_special_axiom_pred_Axiom (sq:SORT list,
          (name, typelist):PRED_NAME * pred_entry):L_FORMULA list=
let fun get_special_axiom_pred_Axiom1 (pred_type (sorts s)) =
     let fun issq (x2,x,s) = is_q s
        val liste = vardecl_list (map (lift_sort sq) s);
        val varsl = varsortlist (map (lift_sort sq) s);
        val varl = varslist varsl
        val qvars = filter issq varsl
    in if qvars=[] then []
       else [((quantification
               (forall,liste,           
                    implication 
                      ((atom
                          (predication 
                  (pred_symb (name,Some(pred_type(sorts (map (lift_sort sq) s)))),
                            terms varl))),
                       conjunction 
                   (map mkpred1 qvars)))),
              make_label( "ga_strictness_"^print_PRED_SYMB(pred_symb(name,Some (pred_type (sorts s))))))]
    end
   | get_special_axiom_pred_Axiom1 _ = raise ERROR
in
	flat (map get_special_axiom_pred_Axiom1 typelist)
end;     
     
fun get_special_axiom_pred_Axioms (sq,[]) = []
| get_special_axiom_pred_Axioms (sq,(p::predlist):Pred_list):L_FORMULA list =
  (get_special_axiom_pred_Axiom (sq,p)) @ (get_special_axiom_pred_Axioms (sq,predlist))
     



fun get_special_axioms (sqlist:SORT list,env:local_list) =
let 	val (sortlist, subsortlist,varlist,funlist,predlist) = env
        val newSprime = map  get_special_axiom_sprime sqlist
        val newSmem = map get_special_axiom_membership sqlist
        val newSdef = mapfilter (get_special_axiom_sort_def sqlist) sortlist
(*        val newEq = map  get_special_axiom_sort_eq sqlist*)
        val newQuest = map  get_special_axiom_sort_defQuest sqlist
                     (* @map get_special_axiom_sort_gen_bottom sqlist*)
      (*  val newSorts = map get_special_axiom sqlist *)
        val newPreds =  get_special_axiom_pred_Axioms (sqlist,predlist);
        val newFuns =  get_special_axiom_funs (sqlist,funlist);
         
in  newSmem @
 newSprime @ newSdef @ newFuns @ (*newEq  @ *) newQuest @ newPreds
end;



(***** SubPCFOL--> SubCFOL: Axioms in the specification *****)
fun make_sort (s:string):SORT  = simple_id ([token s],s,None);


fun get_term (t,q,s) = t
fun get_q (t,q,s) = q
fun get_sort (t,q,s) = s
fun liftq_term sq (t,q,s) = 
	if q orelse not (s mem sq) then (t,s)
	else (inject(s, quest s,t),quest s)

fun trans_TERM (sq:SORT list) (a:TERM):(TERM * bool * SORT)= 
case a of
   qual_var (x,s)  => (qual_var (x,s),false,s)
 | var_or_const x  => raise ERROR (*(var_or_const x, false,make_sort "")*)
 | application (op_symb(f,Some ftype), tt) => 
     let
       val  (transtt,isqtt,_) = (trans_TERMS sq tt)
       val new_type = if isqtt then lift_op_type sq ftype 
                               else totalize_op_type ftype
       val res_sort = get_res new_type
 (*      val x = if true
          then writeln("Application of "^print_ID f^" to "^print_TERMS transtt^" old type "^print_OP_TYPE ftype^" new type "^print_OP_TYPE new_type)
          else (); *)
     in (application (op_symb(f,Some new_type), transtt), 
                              is_q res_sort,  res_sort)
    end
 | sorted_term (T, S) =>  
     let val (transtt,isqtt,s) = trans_TERM sq T
     in (sorted_term (transtt, s), isqtt, s)
     end
 | cast (T,S ) =>  
     let  val (transtt,isqtt,srts) = trans_TERM sq T;
     in (cast (transtt, S), true, force_q S) 
     end
 | conditional (T1,Phi,T2) =>
 let val (terms [t1,t2], q, [s1,s2]) = trans_TERMS sq (terms [T1,T2])
     val phi = trans_FORMULA sq Phi
     in (conditional (t1,phi,t2),q,s1)
     end
 | pos_TERM(r,b,t) => 
   let val (t1,q,s) = trans_TERM sq t
   in (pos_TERM(r,b,t1),q,s)
   end
 |t => raise (WRONG (1,its_a_term t))
 
and trans_TERMS (sq:SORT list) (l:TERMS):(TERMS * bool*SORT list) = 
    case l of
         terms ll => 
         let val transtt = map (trans_TERM sq) ll;
         in if (true mem (map get_q transtt)) 
            then let val lifttt = map (liftq_term sq) transtt
                 in (terms (map fst lifttt), true , map snd lifttt)
                 end 
            else (terms (map get_term transtt), false , map get_sort transtt)
           end
       | pos_TERMS(r,ts) => 
         let val (ts1,q,s) = trans_TERMS sq ts
         in (pos_TERMS(r,ts1),q,s)
         end


and trans_two_terms sq (T1,T2) =
    let val (terms [t1,t2], q, [s1,s2]) = trans_TERMS sq (terms [T1,T2])
        val q1 = is_q s1
        val q2 = is_q s2
        val (newt1,news1) = 
              if q2 andalso not q1 
              then (inject(s1, quest s1,t1),quest s1)
              else (t1,s1)
        val (newt2,news2) =
              if q1 andalso not q2 
              then (inject(s2, quest s2,t2),quest s2)
              else (t2,s2)
    in ((newt1,newt2),q,news1)
    end

and trans_ATOM (sq:SORT list) (a:ATOM):ATOM =
case a of
(predication (pred_symb(p,Some ptype),ts)) =>  
                 (case trans_TERMS sq ts of
		      (ts1,true,s) =>(predication (pred_symb(p,Some (lift_pred_type sq ptype)),ts1)) (*"with ?"*)
		      | (ts1,false,s) => (predication (pred_symb(p,Some ptype),ts1)))
 | (definedness T) =>  
    let val (t,q,s) = (trans_TERM sq T) 
    in
    if q then (* if (not (s mem sq)) then*)
              predication(make_definedness_symb s,terms [t]) 
              (*else predication(make_definedness_symb (quest s),terms [inject(s, quest s,t)])*)
    else ttrue
    end
 | (existl_equation (T1 , T2)) =>  
 let val ((t1,t2), q, s) = trans_two_terms sq (T1,T2)
     val (ename, ptype) = get_eeq_typelist s
     in if q
        then predication (pred_symb(ename, Some (hd ptype)), terms[t1,t2])
        else strong_equation (t1,t2)
     end     
 | (strong_equation (T1 , T2)) =>  
 let val ((t1,t2), q, s) = trans_two_terms sq (T1,T2)
     in strong_equation (t1,t2)
     end
 | membership (T, S)  => 
  let val (t, q, s) = trans_TERM sq T
     in membership (t, if q then quest S else S)
  end 
 | ttrue => ttrue
 | ffalse => ffalse
 | a => raise (WRONG (2,its_an_atom a))

and trans_FORMULA (sq:SORT list) (phi:FORMULA):FORMULA =
case phi of
 quantification (quant ,vlist, f) =>  
     quantification (quant ,vlist, trans_FORMULA sq f)
 | pred_quantification (quant,vlist,f) =>
     pred_quantification (quant ,vlist, trans_FORMULA sq f)
 | conjunction ff =>  conjunction (map (trans_FORMULA sq) ff)
 | disjunction ff => disjunction (map (trans_FORMULA sq) ff)
 | implication (f1, f2) => implication (trans_FORMULA sq f1, trans_FORMULA sq f2)
 | equivalence (f1, f2) => equivalence (trans_FORMULA sq f1, trans_FORMULA sq f2)
 | negation f =>  negation (trans_FORMULA sq f)
 | atom a => atom (trans_ATOM sq a)
 | sort_gen_ax x => sort_gen_ax (trans_sort_gen_ax sq x)
 | sort_cogen_ax x => raise ERROR (*sort_cogen_ax x*)
 | sort_cofree_ax x => raise ERROR (*sort_cofree_ax x*)
 | pos_FORMULA(r,b,phi) => pos_FORMULA(r,b,trans_FORMULA sq phi)
 | a => raise (WRONG (3,its_a_formula a)) 

and trans_sort_gen_ax sq (sl: SORT list,funs:OP_SYMB list) =
    if Utils.forall is_total funs then (sl,funs)
    else (map (lift_sort sq) sl, 
          map (lift_op_symb sq) funs 
            @ map bottom_symb (sl inter sq))

and is_total (op_symb (f,Some(total_op_type _))) = true
  | is_total _ = false

fun trans_L_FORMULA (sq:SORT list) ((f,l):L_FORMULA):L_FORMULA = (trans_FORMULA sq f, l)


fun get_axioms (sq:SORT list) (l:L_FORMULA list):L_FORMULA list =  
	map (trans_L_FORMULA sq) l;

in

fun encode_basic_SubPCFOL_SubCFOL 
    (use_projections:bool) 
    (env:sign,ax:L_FORMULA list):(sign * L_FORMULA list) =
let
        val list = env_to_list env
	val (sortlist, subsortlist,varlist,funlist,predlist) = list
        val sq = compute_s_question_mark use_projections list
	val newEnv = make_s_prime (sq,list);
        val newBottom = get_bottom sq
        val newProj      = if use_projections 
                           then get_special_axiom_pro subsortlist
                           else nil
        val newProjInj   = if use_projections 
                           then get_special_axiom_proj_inj subsortlist
                           else nil
        val newMemberDef = if use_projections 
                           then get_special_axiom_projection_membership subsortlist
                           else nil
	val newSubsort = order (sq,list);
	val newFuns = get_funs (sq,funlist);
	val newSpeFuns = if use_projections 
                         then get_projection_funs (sq,subsortlist)
                         else nil;
	val newPreds = get_special_preds (sq,sortlist);
	val newSpePreds = get_preds (sq, predlist);
	val newAx = newProj @ newProjInj @ newMemberDef @ (get_axioms sq ax) @ (get_special_axioms (sq,list))
in 
	(list_to_env(newEnv, 
                     newSubsort, 
                     varlist, 
                     newFuns@newBottom@ newSpeFuns, 
                     newPreds@newSpePreds), newAx)
end

and  encode_ax_SubPCFOL_SubCFOL (sq:SORT list) ((phi,n):L_FORMULA):L_FORMULA =
	(trans_FORMULA sq phi,n)

end


(* *********************************************************************** *)
(*									   *)
(*	Encode SubCFOL into CFOL		 				   *)	
(*									   *)
(* *********************************************************************** *)


local


(**************** SubCFOL-->CFOL: Functions ****************)

(***** SubCFOL-->CFOL: Injection functions *****)

fun get_injection_funs1 [] =[]
  | get_injection_funs1 ((s,s')::sexp) =
    if s=s' then get_injection_funs1 sexp
    else  (INJECTION_NAME, [total_op_type (sorts [s],s')])
           ::(get_injection_funs1 sexp)
   ;

fun get_injection_funs [] =[]
  | get_injection_funs ((s,entries)::S:Subsort_list):Fun_list = 
    let
      val sexp = mk_subs ((s,entries)::S);           
      in (get_injection_funs1 sexp)
   end;



fun get_funs (env:local_list):Fun_list =
let 	val (sortlist, subsortlist,varlist,funlist,predlist) = env
in funlist @ (get_injection_funs subsortlist)
end;
 


(**************** Predicates ****************)

(***** SubCFOL-->CFOL: get membership and definedness predicates *****)
 
fun get_membership1 [] = []
 | get_membership1 ((a,b)::sexp) = 
   if a=b orelse quest a = b then get_membership1 sexp
   else
  (get_membership_name (a,b),[pred_type (sorts[b])])::(get_membership1 sexp)
 

fun get_memberships [] = []
  | get_memberships ((s, entries)::S:Subsort_list):Pred_list =
      let
      val sexp = mk_subs ((s,entries)::S) 
       in (get_membership1 sexp)
    end;
 


 fun members [] = []
    |members ((s, entries)::S) =(members S);  

fun get_special_preds s = (members s)@(get_memberships s);


fun get_preds (s:Subsort_list, p:Pred_list):Pred_list = (get_special_preds s) @ p;



(**************** SubCFOL-->CFOL: Axioms ****************)

(******** SubCFOL-->CFOL: Construct set J of axioms ********)


(***** SubCFOL-->CFOL: forall x,y:s. inj_{s,s'}(x)=inj_{s,s'}(y) => x=y (s<=s') *****)

fun get_special_axiom_inj_embed1 [] = []
   | get_special_axiom_inj_embed1 ((s,s')::sexp) =
     if s=s' then get_special_axiom_inj_embed1 sexp
     else
          ((quantification (forall, mk_var_decllist(["x", "y"], s), 
             (equivalence
               ((atom(strong_equation
                 (inject(s,s',var_term ("x",s)), 
                  inject(s,s',var_term ("y",s)))),
                (atom(strong_equation
                 ((var_term ("x",s), var_term ("y",s)))))))))),
            make_label("ga_embedding_injectivity_"^(print_ID s)^"_"^ (print_ID s')))
                 ::(get_special_axiom_inj_embed1 sexp)
     ;


fun get_special_axiom_inj_embed [] = []
   | get_special_axiom_inj_embed ((s,entries)::S:Subsort_list):L_FORMULA list =
    let
            val sexp = mk_subs ((s,entries)::S) 
    in
          get_special_axiom_inj_embed1 sexp
     end;


(***** SubCFOL-->CFOL: forall x:s. inj_{s',s''}(inj_{s,s'}(x))=inj_{s,s''}(x) (s<=s'<=s'') *****)

fun get_special_axiom_inj_transitiv3(s'':SORT,s':SORT) (s:SORT):L_FORMULA list =
           if s=s' orelse s'=s'' then []
           else
           [((quantification (forall, mk_var_decllist(["x"],  s), 
              atom (strong_equation
                    (inject(s',s'',inject(s,s',var_term ("x",s))),
                     inject(s,s'',var_term ("x",s)))))),
          make_label( "ga_transitivity_"^(print_ID s)^"_"^(print_ID s')^"_"^ (print_ID s'')))]

fun get_special_axiom_inj_transitiv2 (subsorts:Subsort_list,s3:SORT) (s2:SORT) =
	flat(map (get_special_axiom_inj_transitiv3 (s3,s2)) (lookup_subsorts (s2,its_a_list subsorts)))
	
fun get_special_axiom_inj_transitiv1 (subsorts:Subsort_list) (s,subs) =
	flat(map (get_special_axiom_inj_transitiv2 (subsorts,s)) subs)
	
fun get_special_axiom_inj_transitiv (subsorts:Subsort_list) =
	flat(map (get_special_axiom_inj_transitiv1 subsorts) subsorts)

 


(***** SubCFOL-->CFOL: forall x:s'. in_{s}(x) <=> exists y:s . x=in_{s,s'}(y) (s<=s') (membership) *****)

fun get_special_axiom_membership1 [] =[]
  | get_special_axiom_membership1 ((s,s')::sexp)=
    if s=s' then get_special_axiom_membership1 sexp
    else
      ((quantification (forall, mk_var_decllist(["x"],  s'), 
         (equivalence
             ((atom 
                 (predication 
                    ((pred_symb 
                     (get_membership_name (s,s'), Some (pred_type (sorts [s'])))), 
                     terms [var_term ("x",s')]))),
              (quantification (exists, mk_var_decllist (["y"], s),
                  (atom (strong_equation 
                            (var_term ("x",s'),
                             inject(s,s',var_term ("y",s))
                             )))
              )))
         ))), 
        (make_label(  "ga_membership_"^(print_ID s)^"_"^(print_ID s'))))
         ::get_special_axiom_membership1 sexp 
                                         
fun get_special_axiom_membership [] =[]
|  get_special_axiom_membership ((s,entries)::S:Subsort_list):L_FORMULA list=
   let fun notsq (x,y) = not (is_q y)
       val sexp = filter notsq  (mk_subs ((s,entries)::S));         
   in
         get_special_axiom_membership1 sexp                                       
   end;



(***** SubCFOL-->CFOL: Overloading axioms for functions *****)
     
fun mk_pair [] = []
  | mk_pair (a::ll) = (map (pair a) ll)@(mk_pair ll) ;


fun order_profs S (prof1,prof2) =
    (if leq_S (its_a_list S,get_res prof1,get_res prof2)
     then if leq_S (its_a_list S,get_res prof2,get_res prof1)
          then let val args1 = get_args prof1
                   val args2 = get_args prof2
                   fun leq (s1,s2) = leq_S (its_a_list S,s1,s2)
               in if Library.exists leq (zip (args1,args2))
                  then Some(prof2,prof1)
                  else Some(prof1,prof2)
               end 
          else Some(prof2,prof1)
     else Some(prof1,prof2)
    )
    handle ZIP_ERROR => None

fun get_special_axiom_overload_F ([],f) =[]
  | get_special_axiom_overload_F (S, f):L_FORMULA list =
   let  val (funname, profiles) = f;
        val listOfprofiles = mapfilter (order_profs S) (mk_pair profiles);

        fun appl_map_inj [] _  _  = []
          | appl_map_inj _  [] _  = []
          | appl_map_inj _  _  [] = []
          | appl_map_inj ((a1,a2,a3)::variables) (s1::w) (s2::w') =
               inject (s1,s2,var_term (a2,s1)) :: (appl_map_inj variables w w');
                                            
                                            
       fun appl_w (w, w', w'',s, s', s'') = 
            let val number = length w;
               val variables = make_vars number w;
               val sortedlist = (make_vars number w);
	       val vardecl_list  = map make_var_decl sortedlist ;
	       val funsymb1 = op_symb (funname, Some (total_op_type (sorts w', s')))
	       val funsymb2 = op_symb (funname, Some (total_op_type (sorts w'', s'')))
            in 
              ((quantification 
                 (forall, vardecl_list, 
                   (atom
                     (strong_equation
                       ((
                         (inject(s',s,application 
                                   (funsymb1,
                                    (terms 
                                        (appl_map_inj variables w w')))))),
                                       
                     (inject (s'', s , application 
                                      (funsymb2,
                                       (terms (appl_map_inj variables w w''))))))) ))),
             (make_label( "ga_overload_F_"^print_OP_SYMB funsymb1^"_"^print_OP_SYMB funsymb2)))
                                     
        end         
   in
        map appl_w (flat (map (overload_F (its_a_list S)) listOfprofiles ))     
   end;

  
fun get_special_axiom_overloading_F ([],[]) =[]
  | get_special_axiom_overloading_F (s,[]) =[]
  | get_special_axiom_overloading_F ([],s) =[]
  | get_special_axiom_overloading_F ((S):Subsort_list,((f::funlist):Fun_list)):L_FORMULA list =           
 (get_special_axiom_overload_F (S,f))@get_special_axiom_overloading_F(S, funlist);



(***** SubCFOL-->CFOL: Overloading axioms for predicates *****)

fun order_args_pred S (prof1,prof2) =
    let val args1 = get_args_pred prof1
        val args2 = get_args_pred prof2
        fun leq (s1,s2) = leq_S (its_a_list S,s1,s2)
    in if Library.exists leq (zip (args1,args2))
       then Some (prof2,prof1)
       else Some (prof1,prof2)
    end
    handle ZIP_ERROR => None

fun get_special_axiom_overload_P ([],f) =[]
  | get_special_axiom_overload_P (S, p):L_FORMULA list =
   let  val (predname, profiles) = p;
        val listOfprofiles = mapfilter (order_args_pred S) (mk_pair profiles);

        fun appl_map_inj [] _  _  = []
          | appl_map_inj _  [] _  = []
          | appl_map_inj _  _  [] = []
          | appl_map_inj ((a1,a2,a3)::variables) (s1::w) (s2::w') =
                               (inject(s1, s2,var_term (a2,s1)))
                                            ::(appl_map_inj variables w w');
                                            
                                            
       fun appl_w (w, w', w'') = 
            let val number = length w;
               val variables = make_vars number w;
               val sortedlist = (make_vars number w);
	       val vardecl_list  = map make_var_decl sortedlist ;
	       val predsymb1 = pred_symb (predname,Some (pred_type (sorts w')))
	       val predsymb2 = pred_symb (predname,Some (pred_type (sorts w'')))
            in 
              ((quantification 
               (forall, vardecl_list, 
                (equivalence
                  (
                   (atom
                     (predication 
                              (predsymb1,
                               (terms(appl_map_inj variables w w')))),
                                       
                     (atom(predication 
                                (predsymb2,
                                (terms (appl_map_inj variables w w'')))))))))),
               (make_label(  "ga_overload_P_"^print_PRED_SYMB predsymb1^"_"^print_PRED_SYMB predsymb2)))

                                     
            end       
   in
        map appl_w (flat (map (overload_P (its_a_list S)) listOfprofiles ))     
   end;


  
fun get_special_axiom_overloading_P ([],[]) =[]
  | get_special_axiom_overloading_P (s,[]) =[]
  | get_special_axiom_overloading_P ([],s) =[]
  | get_special_axiom_overloading_P ((S):Subsort_list,((p::predlist):Pred_list)):L_FORMULA list =           
 (get_special_axiom_overload_P (S,p))@get_special_axiom_overloading_P(S, predlist);


fun get_special_axioms (env:local_list):L_FORMULA list =
let	val (sortlist,subsortlist,varlist,funlist,predlist) = env
       (* val newSortAxioms = map get_special_axiom_inj_id sortlist *)
        val newEmbedd    = get_special_axiom_inj_embed subsortlist
        val newTrans     = get_special_axiom_inj_transitiv subsortlist
        val newMember    = get_special_axiom_membership subsortlist
        val newfuns     = get_special_axiom_overloading_F (subsortlist, funlist)
       val newpreds    = get_special_axiom_overloading_P (subsortlist, predlist)
  in  
   newEmbedd @ newTrans  @ newMember  @ newfuns @ newpreds
   end;
 


(***** SubCFOL-->CFOL: Axioms in the specification *****)

fun get_term (t,s) = t

fun trans_TERM (a:TERM):(TERM * SORT)= 
case a of
    qual_var (x,s) => (qual_var(x,s),s)
 |  var_or_const x  => (var_or_const x, make_sort "")  
 | application (op_symb(f,Some ftype), tt) => 
     let
       val  (transtt,srts) = (trans_TERMS tt)
       val  res_sort = get_res ftype
     in (application 
           (op_symb
              (f, Some ftype), transtt), res_sort)
    end
 | sorted_term (T, S) =>  
     let val (transtt,srts) = (trans_TERM T)
     in (sorted_term (transtt, S), S)
     end
 | cast (T,S) =>  
     let  val (transtt,srt) = (trans_TERM T);
     in if S=srt then (inject(S,quest S,transtt),quest S)
        else (project(S,srt,transtt), quest S) 
     end 
 | conditional (T1,Phi,T2) =>
 let val (t1,s) = trans_TERM T1
     val (t2,_) = trans_TERM T2
     val phi = trans_FORMULA Phi
     in (conditional (t1,phi,t2),s)
     end
 | pos_TERM(r,b,t) => 
   let val (t1,s) = trans_TERM t
   in (pos_TERM(r,b,t1),s)
   end
 |_ => raise NOT_DEFINED
 
and trans_TERMS (l:TERMS):(TERMS * SORT) = 
    case l of
         terms ll => 
         let val transtt = (map trans_TERM ll);
             in (terms (map get_term transtt), make_sort "") 
           end
       | pos_TERMS(r,tt) => 
         let val (tt1,s) = trans_TERMS tt
         in (pos_TERMS(r,tt1),s)
         end

and trans_ATOM (a:ATOM):ATOM =
case a of
(predication (psymb,ts)) =>
   let val (ts,s) = trans_TERMS ts
   in predication (psymb,ts)
   end  
(* | (definedness T) =>  
    let val (t,_) = (trans_TERM T) 
    in
    definedness t
    end*)
 | (existl_equation (T1 , T2)) =>  
 let val (t1,_) = trans_TERM T1
     val (t2,_) = trans_TERM T2
     in existl_equation (t1,t2)
     end     
 | (strong_equation (T1 , T2)) =>  
 let val (t1,_) = trans_TERM T1
     val (t2,_) = trans_TERM T2
     in strong_equation (t1,t2)
     end
 | membership (T, S)  => 
  let val (t, s) = trans_TERM T  
  in 
  if s=S then ttrue
  else predication(pred_symb(get_membership_name (S,s),
                   Some(pred_type(sorts [s]))), terms [t])  
  end
  | ttrue => ttrue
  | ffalse => ffalse 
  | _ => raise ERROR
  
and trans_FORMULA (phi:FORMULA):FORMULA =
case phi of
 quantification (quant ,vlist, f) =>  
     quantification (quant ,vlist, trans_FORMULA f)
 | pred_quantification (quant ,vlist, f) =>  
     pred_quantification (quant ,vlist, trans_FORMULA f)
 | conjunction ff =>  conjunction (map trans_FORMULA ff)
 | disjunction ff => disjunction (map trans_FORMULA ff)
 | implication (f1, f2) => implication (trans_FORMULA f1, trans_FORMULA f2)
 | equivalence (f1, f2) => equivalence (trans_FORMULA f1, trans_FORMULA f2)
 | negation f =>  negation (trans_FORMULA f)
 | atom a => atom (trans_ATOM a)
 | sort_gen_ax x => sort_gen_ax x
 | sort_cogen_ax x => sort_cogen_ax x
 | sort_cofree_ax x => sort_cofree_ax x
 | pos_FORMULA(r,b,phi) => pos_FORMULA(r,b,trans_FORMULA phi)
 | _ => raise ERROR

in


fun encode_basic_SubCFOL_CFOL
    (use_projections:bool) 
    (env:sign,ax:L_FORMULA list):(sign * L_FORMULA list) =

let     val list =  env_to_list env
 	val (sortlist, subsortlist,varlist,funlist,predlist)  = list
        fun mk_no_subs s = (s,[s])
        val newSubsorts = map mk_no_subs sortlist
	val newFuns = get_funs list;
	val newPreds = get_preds (subsortlist, predlist);
        fun get_axioms (env:local_list,l:L_FORMULA list):L_FORMULA list =
                               (get_special_axioms env) @(map encode_ax_SubCFOL_CFOL l);
	val newAx = get_axioms (list,ax) 
in 
	(list_to_env(sortlist, newSubsorts, varlist, newFuns, newPreds),newAx )

end

and  encode_ax_SubCFOL_CFOL ((phi,n):L_FORMULA):L_FORMULA =
	(trans_FORMULA phi,n)

end




(* *********************************************************************** *)
(*									   *)
(*	Encode CFOL into SOL		 				   *)	
(*									   *)
(* *********************************************************************** *)

fun mk_var_decl_x (s,n) = ([("x"^Int.toString n,None)],s)
fun mk_qual_var_x (s,n) = qual_var(("x"^Int.toString n,None),s)

local
    fun make_P s = pred_symb(make_sid("P"^print_ID s),Some (pred_type (sorts [s])))
    fun full_pred s = 
        let val x = ("x"^print_ID s,None)
            val body = atom (predication (make_P s,
                                    terms([qual_var (x,s)])))
        in quantification(forall,[([x],s)],body)
        end
    fun get_prem srts (s,n) =
        if s mem srts 
        then Some (atom (predication(make_P s,terms [mk_qual_var_x(s,n)])))
        else None
    fun get_ind_hyp srts 
         (f_symb as (op_symb (f,Some(ftype as (total_op_type (sorts args,res)))))) =
          let val args_nos = zip (args,1 upto length args)
              val concl = atom(predication(make_P res,
                                 terms ([application(f_symb,terms (map mk_qual_var_x args_nos))]) ))
              val prems = mapfilter (get_prem srts) args_nos
              val phi = case prems of
                        nil =>    concl
                        | [p1] => implication (p1,concl)
                        | _ =>    implication (conjunction prems, concl)
          in
          quantification (forall,map mk_var_decl_x args_nos,phi)
          end
      | get_ind_hyp _ _ = raise ERROR

fun encode_FORMULA (sort_gen_ax (srts,funs)) =
    let val ind_hyps = map (get_ind_hyp srts) funs
        val ind_hyp = case ind_hyps of
            nil => atom(ffalse)
            | [phi] => phi
            | phis => conjunction phis
        val ind_concls = map full_pred srts
        val ind_concl = case ind_concls of
            nil => atom(ttrue)
            | [phi] => phi
            | phis => conjunction phis
    in
    implication (ind_hyp,ind_concl)
    end
  | encode_FORMULA (sort_cogen_ax (srts,funs)) = raise ERROR
  | encode_FORMULA (sort_cofree_ax (strs,funs)) = raise ERROR
  | encode_FORMULA phi = phi
    	      
fun encode_ax_CFOL_SOL (phi,l) =
    (encode_FORMULA phi,l)
in

fun encode_basic_CFOL_SOL use_projections
    (Sigma:sign,ax:L_FORMULA list) =
    (Sigma, map encode_ax_CFOL_SOL ax)
end
  
 
(* *********************************************************************** *)
(*									   *)
(*	Overall encoding		 				   *)	
(*									   *)
(* *********************************************************************** *)

fun encode_basic_SubPCFOL_CFOL (use_projections:bool)
        (Sigma:sign,ax:L_FORMULA list) =
  let 
 	val (Sigma1,ax1) = encode_basic_SubPCFOL_SubCFOL use_projections (Sigma,ax)
	val (Sigma2,ax2) = encode_basic_SubCFOL_CFOL use_projections (Sigma1,ax1)
  in (Sigma2,ax2)
  end;


fun encode_basic_SubPCFOL_SOL  (use_projections:bool)
        (Sigma:sign,ax:L_FORMULA list) =
  let 
 	val (Sigma1,ax1) = encode_basic_SubPCFOL_CFOL use_projections (Sigma,ax)
	val (Sigma2,ax2) = encode_basic_CFOL_SOL use_projections (Sigma1,ax1)
  in (Sigma2,ax2)
  end;

fun encode_basic_SubCFOL_SOL  (use_projections:bool)
        (Sigma:sign,ax:L_FORMULA list) =
  let 
 	val (Sigma1,ax1) = encode_basic_SubCFOL_CFOL use_projections (Sigma,ax)
	val (Sigma2,ax2) = encode_basic_CFOL_SOL use_projections (Sigma1,ax1)
  in (Sigma2,ax2)
  end;

(* *********************************************************************** *)
(*									   *)
(*	Lifting to the level of structured specifications		   *)	
(*									   *)
(* *********************************************************************** *)


fun encode_senv be (basic_env (Sigma,ax)) = basic_env (be(Sigma,ax))
  | encode_senv be (translate_env (senv,sigma)) =
    translate_env (encode_senv be senv,sigma)  (* Needs to be elaborated !!! (sigma)*) 
  | encode_senv be (derive_env (senv,sigma)) =
    derive_env (encode_senv be senv ,sigma)   (* Needs to be elaborated !!! (sigma)*)
  | encode_senv be (union_env senvs) =
    union_env (map (encode_senv be) senvs)
  | encode_senv be (extension_env senvs) =
    extension_env (map (encode_senv1 be) senvs)
  | encode_senv be (free_spec_env senv) =
    free_spec_env (encode_senv be senv)
  | encode_senv be (cofree_spec_env senv) =
    cofree_spec_env (encode_senv be senv)
  | encode_senv be (closed_spec_env senv) =
    closed_spec_env (encode_senv be senv)
  | encode_senv be (spec_inst_env (name,body,sigma,args)) =
    spec_inst_env (name,
                   encode_senv be body,
                   sigma,
                   map (encode_senv be) args)
  | encode_senv _ _ = raise ERROR

and encode_senv1 be (senv,an) = (encode_senv be senv,an)

fun encode_slenv be (SPEC_ENV (Sigma,HSigma,senv)) =
    (SPEC_ENV (fst (be (Sigma,[])),fst (be (HSigma,[])),encode_senv be senv))

fun encode_gen_env be (slenv,slenv_list,Sigma) =
    (encode_slenv be slenv,map (encode_slenv be) slenv_list,fst (be (Sigma,[])))

fun encode_global_entry be (spec_defn_env (gen_env,slenv)) =
    spec_defn_env (encode_gen_env be gen_env,
                     encode_slenv be slenv)
  | encode_global_entry be (view_defn_env (gen_env,slenv1,sigma,slenv2)) =
    view_defn_env (encode_gen_env be gen_env,
                     encode_slenv be slenv1,
                     sigma,
                     encode_slenv be slenv2)
  | encode_global_entry be (arch_spec_defn_env archentry) =
    arch_spec_defn_env archentry  (* Needs to be elaborated !!!*)
  | encode_global_entry be (unit_spec_defn_env unitentry) =
    unit_spec_defn_env unitentry (* Needs to be elaborated !!!*)

fun encode_global_entry_pair be (n,entry) =
    (n,encode_global_entry be entry)
    
fun lift_encoding basic_encoding (genv,an:ANNO list) =
    (Symtab_sid.make (map (encode_global_entry_pair basic_encoding) 
                          (Symtab_sid.dest genv)),
     an)


(* *********************************************************************** *)
(*									   *)
(*	Special lifting to the level of structured specifications	   *)	
(*	for encoding of subsorting - (co)freeness needs special treatment  *)
(*									   *)
(* *********************************************************************** *)

fun mk_opsym f t = op_symb(f, Some t)
fun mk_opsyms (f,types) =
    map (mk_opsym f) types

fun matches_sort (s,(compound_id (([token "?"],"?",None),[s']):ID)) =
    ID_eq(s,s')
  | matches_sort (s,s') =
    ID_eq(s,s')

fun matches_op_type (total_op_type (sorts args,res)) 
                    ((total_op_type (sorts args',res')),_,_) = 
    matches_sort(res,res') andalso
    Utils.forall matches_sort (zip (args,args'))
  | matches_op_type (partial_op_type (sorts args,res)) 
                    ((total_op_type (sorts args',res')),_,_) = 
    matches_sort(res,res') andalso
    Utils.forall matches_sort (zip (args,args'))

fun find_op_type t tf_list =
    let val (_,g,_) = the (find_first (matches_op_type t) tf_list)
    in g
    end

fun matches_pred_type (pred_type (sorts args)) 
                      ((pred_type (sorts args')),_) = 
    Utils.forall matches_sort (zip (args,args'))

fun find_pred_type t tf_list =
    let val (_,q) = the (find_first (matches_pred_type t) tf_list)
    in q
    end

fun lift_morphism Sigma sigma ESigma =
    let val (srts,_,_,funs,preds) = env_to_list_multi Sigma
        val (Esrts,_,_,Efuns,Epreds) = env_to_list_multi ESigma
        val (smap,fmap,pmap) = sigma
        val Qsrts = Esrts \\ srts
        fun map_sort (compound_id (([token "?"],"?",None),[s]):ID) =
            quest (Morphisms.sort_via_morphism sigma s)
          | map_sort s = s
        fun add_sort (m,s) = Symtab_id.update ((s,map_sort s),m)
        val smap' = foldl add_sort (smap,Qsrts)

        fun add_fun (l,(f,t)) = 
            let val tf_list = the (Symtab_id.lookup (fmap,f))
                val g = find_op_type t tf_list
            in (f,(t,g,true))::l
            end handle _ => (f,(t,f,true))::l
        val fmap' = Symtab_id.make_multi (foldl add_fun ([],Efuns))

        fun add_pred (l,(p,t)) = 
            let val tf_list = the (Symtab_id.lookup (pmap,p))
                val q = find_pred_type t tf_list
            in (p,(t,q))::l
            end handle _ => (p,(t,p))::l
        val pmap' = Symtab_id.make_multi (foldl add_pred ([],Epreds))

    in (smap',fmap',pmap')
    end

fun is_phorn_senv (basic_env (Sigma,ax)) = Utils.forall HORN1.is_phorn (map fst ax)
  | is_phorn_senv (translate_env (senv,sigma)) =
    is_phorn_senv senv 
  | is_phorn_senv (derive_env (senv,sigma)) = false
  | is_phorn_senv (union_env senvs) =
    Utils.forall is_phorn_senv senvs
  | is_phorn_senv (extension_env senvs) =
    Utils.forall is_phorn_senv (map fst senvs)
  | is_phorn_senv (free_spec_env senv) = false
  | is_phorn_senv (cofree_spec_env senv) = false
  | is_phorn_senv (closed_spec_env senv) =
    is_phorn_senv senv
  | is_phorn_senv (spec_inst_env (name,body,sigma,args)) =
    is_phorn_senv body

fun check_all_TERM pred (t:TERM) = 
case t of
    qual_var (x,s) => true
 |  var_or_const x  => true  
 | application (_, tt) => 
    check_all_TERMS pred tt
 | sorted_term (T, S) =>  
     check_all_TERM pred T
 | cast (T,S) =>  
     check_all_TERM pred T
 | conditional (T1,Phi,T2) =>
     check_all_TERM pred T1 andalso
     check_all_TERM pred T2 andalso
     check_all_FORMULA pred Phi
 | pos_TERM(r,b,t) => 
     check_all_TERM pred t
 
and check_all_TERMS pred (ts:TERMS) = 
    case ts of
         terms tt => 
           Utils.forall (check_all_TERM pred) tt
       | pos_TERMS(r,tt) => 
           check_all_TERMS pred tt

and check_all_ATOM pred (a:ATOM) =
case a of
  (predication (psymb,ts)) =>
    check_all_TERMS pred ts
 | (definedness T) => check_all_TERM pred T
 | (existl_equation (T1 , T2)) =>  
   check_all_TERM pred T1
   andalso
   check_all_TERM pred T2 
 | (strong_equation (T1 , T2)) =>  
   check_all_TERM pred T1
   andalso
   check_all_TERM pred T2
 | membership (T, S)  => 
   check_all_TERM pred T  
 | ttrue => true
 | ffalse => true

and check_all_FORMULA pred phi =
case phi of
 quantification (quant ,vlist, f) =>  
     check_all_FORMULA pred f
 | pred_quantification (quant ,vlist, f) =>  
     check_all_FORMULA pred f
 | conjunction ff =>  Utils.forall (check_all_FORMULA pred) ff
 | disjunction ff => Utils.forall (check_all_FORMULA pred) ff
 | implication (f1, f2) => check_all_FORMULA pred f1 andalso check_all_FORMULA pred f2
 | equivalence (f1, f2) => check_all_FORMULA pred f1 andalso check_all_FORMULA pred f2
 | negation f =>  check_all_FORMULA pred f
 | atom a => pred a andalso check_all_ATOM pred a
 | sort_gen_ax x => false
 | sort_cogen_ax x => false
 | sort_cofree_ax x => false
 | pos_FORMULA(r,b,phi) => check_all_FORMULA pred phi

fun check_all_L_FORMULA pred (phi,_) = check_all_FORMULA pred phi

fun is_pred_in_list predlist (predication (psymb,_)) =
    psymb mem predlist
  | is_pred_in_list _ _ = false

fun check_all_predication predlist ax = check_all_L_FORMULA (is_pred_in_list predlist) ax

(* Beware! Formulas in conditionals are not substituted, since
   these cannot occur in a Horn formula *)
fun subst_ATOM subst (pr as predication (psymb,ts)) =
    (case Utils.assoc (subst,psymb) of
       Some (p,ty) => (predication (pred_symb(p,Some ty),ts))
     | None => pr)
  | subst_ATOM _  a = a

fun subst_FORMULA subst phi =
case phi of
 quantification (quant ,vlist, f) =>  
     quantification (quant ,vlist, subst_FORMULA subst f)
 | pred_quantification (quant ,vlist, f) =>  
     pred_quantification (quant ,vlist, subst_FORMULA subst f)
 | conjunction ff =>  conjunction (map (subst_FORMULA subst) ff)
 | disjunction ff => disjunction (map (subst_FORMULA subst) ff)
 | implication (f1, f2) => implication (subst_FORMULA subst f1, subst_FORMULA subst f2)
 | equivalence (f1, f2) => equivalence (subst_FORMULA subst f1, subst_FORMULA subst f2)
 | negation f =>  negation (subst_FORMULA subst f)
 | atom a => atom (subst_ATOM subst a)
 | sort_gen_ax x => sort_gen_ax x
 | sort_cogen_ax x => sort_cogen_ax x
 | sort_cofree_ax x => sort_cofree_ax x
 | pos_FORMULA(r,b,phi) => pos_FORMULA(r,b,subst_FORMULA subst phi)
 | _ => raise ERROR

fun subst_L_FORMULA subst (phi,l) = (subst_FORMULA subst phi,l)

fun prefix_id s (simple_id (toks,id,line)) =
    simple_id (token (s^"_")::toks,s^"_"^id,line)
  | prefix_id s (compound_id ((toks,id,line),cids)) =
    compound_id ((token (s^"_")::toks,s^"_"^id,line),cids)

(*
fun mk_predvar (p,nil) = nil
  | mk_predvar (p,[ty]) = [(prefix_id "P_" p,ty)]
  | mk_predvar (p,tys) = 
    map (fn (ty,n) => (prefix_id ("P"^string_of_int n^"_") p,ty)) 
        (zip(tys,1 upto length tys)) *)

fun mk_predvar ((pred_symb(p,Some ty)),n) = (make_sid ("P"^string_of_int n),ty)
 
fun mk_predsym (p,tys) = map (fn ty => pred_symb(p,Some ty)) tys

fun least_pred_ax (psymb,(p,ty as pred_type(tlist))) =
    let val argsrts = get_sorts tlist
        val n = length argsrts
        val args_nos = zip (argsrts,1 upto length argsrts)
        val varlist = map mk_var_decl_x args_nos
        val premise = atom(predication(psymb,
                                 terms (map mk_qual_var_x args_nos) ))
        val conclusion = atom(predication(pred_symb(p,Some ty),
                                 terms (map mk_qual_var_x args_nos) ))
        val phi = implication (premise,conclusion)
    in phi (*quantification (forall, varlist, phi) *)
    end

fun get_pred_induction (nil:Pred_list) senv = nil
  | get_pred_induction (predlist:Pred_list) senv =
    let val (_,ax) =  Flatten.flatten_senv senv
        val predsyms = flat (map mk_predsym predlist)
    in if Utils.forall (check_all_predication predsyms) ax 
       then 
       let val predvars = map mk_predvar (zip (predsyms,1 upto length predsyms))
           val subst = zip (predsyms,predvars)
           val premise = conjunction(map (fst o subst_L_FORMULA subst) ax)
           val conclusion = conjunction(map least_pred_ax subst)
           val phi = implication (premise,conclusion)
           val lab = [label_anno (prefix_idn "ga_pred_induction" (map fst predlist))]
       in [(pred_quantification (forall, predvars, phi),lab)]
       end
       else []
    end

fun encode_senv_SubPCFOL_Sub_CFOL sub pr lSigma (basic_env (Sigma,ax)) = 
    let (* val _ = writeln("Union in the encoding!")*)
        val united_Sigma = signature_union(Sigma,lSigma)
        val encoder = if sub then encode_basic_SubPCFOL_CFOL
                      else encode_basic_SubPCFOL_SubCFOL
        (* Compute the difference between the encoding of
           the current basic theory and the local environment.
           The encoding of the latter could be carried around
           to optimize performance *)
        val (ElSigma,Elax) = encoder pr (lSigma,[])
        val (ESigma,Eax) = encoder pr (united_Sigma,ax)
        val ESigma_diff = signature_diff(ESigma,ElSigma)
        val Eax_diff = Eax \\ Elax
      (*  val _ = writeln("Sig: "^BasicPrint.print_sign Sigma^"\nend\n");
        val _ = writeln("lSig: "^BasicPrint.print_sign lSigma^"\nend\n");
        val _ = writeln("ESigdiff: "^BasicPrint.print_sign ESigma_diff^"\nend\n");*)
    in
    (united_Sigma,
     basic_env (ESigma_diff,Eax_diff))
    end
  | encode_senv_SubPCFOL_Sub_CFOL sub pr lSigma (translate_env (senv,sigma)) =
    let val (Sigma,senv') = encode_senv_SubPCFOL_Sub_CFOL sub pr lSigma senv
        val sigma_Sigma = Rename.rename_sign sigma Sigma
        val (ESigma,_) = Flatten.flatten_senv senv'
        val sigma' = lift_morphism Sigma sigma ESigma
    in (sigma_Sigma,translate_env (senv',sigma'))
    end  
  | encode_senv_SubPCFOL_Sub_CFOL sub pr lSigma (derive_env (senv,sigma)) =
    let val (Sigma,senv') = encode_senv_SubPCFOL_Sub_CFOL sub pr lSigma senv
    in (Sigma,derive_env (senv',sigma))
    end
    (* Needs to be elaborated !!! (sigma)*)
  | encode_senv_SubPCFOL_Sub_CFOL sub pr lSigma (union_env senvs) =
    let val senvs' = map (encode_senv_SubPCFOL_Sub_CFOL sub pr lSigma) senvs
        val Sigma = signature_union_list (map fst senvs')
        val senv = map snd senvs'
    in (Sigma,union_env (map snd senvs'))
    end
  | encode_senv_SubPCFOL_Sub_CFOL sub pr lSigma (extension_env senvs) =
    let fun encode1 ((lSigma,senvs),(senv,ans)) =
        let (*val _ = writeln("Extension:\nlSimga:"^print_sign lSigma)
            val _ = writeln("senv: "^StructuredPrint.print_spec_env senv^"\nend")*)
            val (Sigma,senv') = encode_senv_SubPCFOL_Sub_CFOL sub pr lSigma senv
        in (Sigma,(senv',ans)::senvs)
        end
        val (Sigma,senvs') = foldl encode1 ((lSigma,[]),senvs)
    in (Sigma,extension_env (rev senvs'))
    end
  | encode_senv_SubPCFOL_Sub_CFOL sub pr lSigma (free_spec_env senv) =
    let val (Sigma,senv') = encode_senv_SubPCFOL_Sub_CFOL sub pr lSigma senv
        val (sortlist,_,_,funlist,predlist) = 
            env_to_list (signature_diff(Sigma,lSigma))
        val opsymlist = flat (map mk_opsyms funlist)
        val gen_ax = if is_phorn_senv senv then
                        if null sortlist then get_pred_induction predlist senv
                        else [(sort_gen_ax(sortlist,opsymlist),
                              [label_anno (prefix_idn "ga_generated" sortlist)])]
              else (writeln "Warning: free specification not in Horn form"; [])
        val benv = basic_env(empty_signature,gen_ax)
    in (Sigma,extension_env [(senv',[]),(benv,[])])
    end
  | encode_senv_SubPCFOL_Sub_CFOL sub pr lSigma (cofree_spec_env senv) = raise ERROR
    (*cofree_spec_env (encode_senv_SubPCFOL_Sub_CFOL sub pr senv)*)
  | encode_senv_SubPCFOL_Sub_CFOL sub pr lSigma (closed_spec_env senv) =
    let val (Sigma,senv') = encode_senv_SubPCFOL_Sub_CFOL sub pr empty_signature senv
        val Sigma' = signature_union(lSigma,Sigma)
    in (Sigma',closed_spec_env senv')
    end
  | encode_senv_SubPCFOL_Sub_CFOL sub pr lSigma (spec_inst_env (name,body,sigma,args)) =
    let val (Sigma_b,senv_b) = encode_senv_SubPCFOL_Sub_CFOL sub pr empty_signature body
        (*val _ = writeln("Body sig: "^print_sign Sigma_b)*)
        val senvs = map (encode_senv_SubPCFOL_Sub_CFOL sub pr empty_signature) args
        val Sigma_renamed_body = Rename.rename_sign sigma Sigma_b
        val Sigma_args = signature_union_list (map fst senvs)
        val Sigma = signature_union(Sigma_renamed_body,Sigma_args)
        val (ESigma_b,_) = Flatten.flatten_senv senv_b
        val sigma' = lift_morphism Sigma_b sigma ESigma_b
    in (signature_union(lSigma,Sigma),
        spec_inst_env (name,senv_b,sigma',map snd senvs))
    end
  | encode_senv_SubPCFOL_Sub_CFOL _ _ _ _ = raise ERROR

and encode_senv1_SubPCFOL_Sub_CFOL sub pr (senv,an) = (encode_senv_SubPCFOL_Sub_CFOL sub pr senv,an)

fun encode_slenv_SubPCFOL_Sub_CFOL sub pr (SPEC_ENV (Sigma,HSigma,senv)) =
   (SPEC_ENV (Sigma,HSigma,
               snd (encode_senv_SubPCFOL_Sub_CFOL sub pr empty_signature senv)))
 
(*fun encode_slenv_SubPCFOL_Sub_CFOL sub pr (SPEC_ENV (Sigma,HSigma,senv)) =
    let val be = if sub then encode_basic_SubPCFOL_CFOL pr
                 else encode_basic_SubPCFOL_SubCFOL pr
    in
    (SPEC_ENV (fst (be (Sigma,[])),
               fst (be (HSigma,[])),
               snd (encode_senv_SubPCFOL_Sub_CFOL sub pr empty_signature senv)))
    end
*)

fun encode_gen_env_SubPCFOL_Sub_CFOL sub pr (slenv,slenv_list,Sigma) =
    let val be = if sub then encode_basic_SubPCFOL_CFOL pr
                 else encode_basic_SubPCFOL_SubCFOL pr
    in
    (encode_slenv_SubPCFOL_Sub_CFOL sub pr slenv,
     map (encode_slenv_SubPCFOL_Sub_CFOL sub pr) slenv_list,
     fst (be (Sigma,[])))
    end

fun get_env (SPEC_ENV(_,_,env)) = env

fun encode_global_entry_SubPCFOL_Sub_CFOL sub pr (spec_defn_env (gen_env,body)) =
    let val (imp,pars,_) = gen_env
        val imp' = (get_env imp,[])
        val par' = (union_env (map get_env pars),[])
        val SPEC_ENV(bsig,bhsig,body_env) = body
        val body_env' = if null pars then body 
                        else SPEC_ENV(bsig,bhsig,extension_env[imp',par',(body_env,[])])
        val gen_env' = encode_gen_env_SubPCFOL_Sub_CFOL sub pr gen_env
        val body_env'' = encode_slenv_SubPCFOL_Sub_CFOL sub pr body_env'
    in spec_defn_env (gen_env',body_env'')
    end
  | encode_global_entry_SubPCFOL_Sub_CFOL sub pr (view_defn_env (gen_env,slenv1,sigma,slenv2)) =
    view_defn_env (encode_gen_env_SubPCFOL_Sub_CFOL sub pr gen_env,
                     encode_slenv_SubPCFOL_Sub_CFOL sub pr slenv1,
                     sigma,
                     encode_slenv_SubPCFOL_Sub_CFOL sub pr slenv2)
  | encode_global_entry_SubPCFOL_Sub_CFOL sub pr (arch_spec_defn_env archentry) =
    arch_spec_defn_env archentry  (* Needs to be elaborated !!!*)
  | encode_global_entry_SubPCFOL_Sub_CFOL sub pr (unit_spec_defn_env unitentry) =
    unit_spec_defn_env unitentry (* Needs to be elaborated !!!*)

fun encode_global_entry_pair_SubPCFOL_Sub_CFOL sub pr (n,entry) =
    (n,encode_global_entry_SubPCFOL_Sub_CFOL sub pr entry)
    
fun encode_SubPCFOL_Sub_CFOL sub use_projections (genv,an:ANNO list) =
    (Symtab_sid.make (map (encode_global_entry_pair_SubPCFOL_Sub_CFOL sub use_projections)
                          (Symtab_sid.dest genv)),
     an)


(* ???
fun lift_encoding basic_encoding Gamma =
    Symtab_sid.make (map (encode_global_entry_pair basic_encoding) 
                         (Symtab_sid.dest Gamma))
*)

(*fun encode_SubCFOL_CFOL use_projections = 
    lift_encoding (encode_basic_SubCFOL_CFOL use_projections) *)
fun encode_SubPCFOL_SubCFOL use_projections =
    encode_SubPCFOL_Sub_CFOL false use_projections
fun encode_SubPCFOL_CFOL use_projections =
    encode_SubPCFOL_Sub_CFOL true use_projections
fun encode_SubPCFOL_SOL use_projections = 
    lift_encoding (encode_basic_CFOL_SOL use_projections)  o encode_SubPCFOL_CFOL use_projections 



end
