(***************************************************************** *)
(*								   *)
(* Project: CATS						   *)
(* Author: Till Mossakowski, University of Bremen		   *)
(* Date: 2000						           *)
(* Purpose of this file: Operations on IDs	 		   *)
(*								   *)
(*								   *)
(***************************************************************** *)

(* This module implements ordering and equality test functions
   on IDs. Note that the position information contained in IDs
   has to be ignored.
*)

structure IDOrder = 

struct
local open AS Utils in



fun ID_ord (id1,id2):order =
  case id1 of
  AS.simple_id (_,s1,_) => (case id2 of
                      AS.simple_id (_,s2,_) => string_ord (s1,s2)
                      | AS.compound_id (_) => LESS)
  | AS.compound_id ((_,s1,_),idlist1) => 
     (case id2 of
      AS.simple_id (_) => GREATER
      | AS.compound_id ((_,s2,_),idlist2) =>
      case string_ord (s1,s2) of
        LESS => LESS
        | GREATER => GREATER
        | EQUAL => IDlist_ord (idlist1,idlist2)
      )

and IDlist_ord (nil,nil) = EQUAL
  | IDlist_ord (nil,id1::rest) = LESS
  | IDlist_ord (id1::rest,nil) = GREATER   
  | IDlist_ord (id1::rest1,id2::rest2) =
       case ID_ord (id1,id2) of
        LESS => LESS
        | GREATER => GREATER
        | EQUAL => IDlist_ord (rest1,rest2);
                               
        
fun LABEL_ord ((idlst1,_),(idlst2,_)) =
    IDlist_ord (idlst1,idlst2)

fun SIMPLE_ID_ord ((s1,_),(s2,_)) = string_ord (s1,s2)


fun SIMPLE_ID_eq (x,y:SIMPLE_ID) = SIMPLE_ID_ord(x,y)=EQUAL

fun SIMPLE_ID_list_eq (x,y) = 
    length x=length y andalso Library.forall SIMPLE_ID_eq (zip (x,y))

fun SIMPLE_ID_member (s,l) = 
    generalized_member SIMPLE_ID_list_eq (s,l)

fun ID_eq (x,y) = ID_ord(x,y)=EQUAL

fun ID_list_eq (x,y) = 
    length x=length y andalso Library.forall ID_eq (zip (x,y))

fun ID_member (s,l) = 
    generalized_member ID_eq (s,l)

val SORT_eq = ID_eq
val SORT_member = ID_member
val SORT_list_eq = ID_list_eq

val OP_eq = ID_eq
val OP_member = ID_member

val PRED_eq = ID_eq
val PRED_member = ID_member

val VAR_eq = SIMPLE_ID_eq
val VAR_member = SIMPLE_ID_member

fun OP_TYPE_eq (AS.total_op_type (AS.sorts sl1,s1), AS.total_op_type (AS.sorts sl2,s2)) =
    SORT_eq (s1,s2) andalso SORT_list_eq (sl1,sl2)
  | OP_TYPE_eq (AS.partial_op_type (AS.sorts sl1,s1), AS.partial_op_type (AS.sorts sl2,s2)) =
    SORT_eq (s1,s2) andalso SORT_list_eq (sl1,sl2)
  | OP_TYPE_eq (_,_) = false

fun OP_TYPE_member (x,l) =
    generalized_member OP_TYPE_eq (x,l)

fun PRED_TYPE_eq (AS.pred_type sl1, AS.pred_type sl2) =
    SORT_list_eq (get_sorts sl1,get_sorts sl2)
  | PRED_TYPE_eq (AS.pos_PRED_TYPE (_,pt1),pt2) = PRED_TYPE_eq (pt1,pt2)
  | PRED_TYPE_eq (pt1,AS.pos_PRED_TYPE (_,pt2)) = PRED_TYPE_eq (pt1,pt2)

fun PRED_TYPE_member (x,l) =
    generalized_member PRED_TYPE_eq (x,l)
    
fun SORT_map_eq ((s1,s2),(t1,t2)) =
    SORT_eq (s1,t1) andalso SORT_eq (s2,t2)

fun SORT_map_member (x,l) =
    generalized_member SORT_map_eq (x,l)
    
fun OP_map_eq ((s1,s2),(t1,t2)) =
    OP_eq (s1,t1) (*andalso OP_TYPE_eq (s2,t2)*)

fun OP_map_member (x,l) =
    generalized_member OP_map_eq (x,l)

fun PRED_map_eq ((s1,s2),(t1,t2)) =
    PRED_eq (s1,t1) (*andalso PRED_TYPE_eq (s2,t2)*)

fun PRED_map_member (x,l) =
    generalized_member PRED_map_eq (x,l)
           
fun ANNO_eq (number_anno (id1),number_anno (id2)) = 
    ID_eq(id1,id2)
  | ANNO_eq (floating_anno (ida1,idb1),floating_anno (ida2,idb2)) =
   ID_eq(ida1,ida2) andalso ID_eq(idb1,idb2)
  | ANNO_eq (string_anno (ida1,idb1),string_anno (ida2,idb2)) =
    ID_eq(ida1,ida2) andalso ID_eq(idb1,idb2)
  | ANNO_eq (list_anno (ida1,idb1,idc1),list_anno (ida2,idb2,idc2)) = 
    ID_eq(ida1,ida2) andalso ID_eq(idb1,idb2) andalso ID_eq(idc1,idc2)
  | ANNO_eq (display_anno (id1 , s1),display_anno (id2 , s2)) = 
    ID_eq(id1,id2) andalso s1=s2
  | ANNO_eq (prec_anno (b1,idsa1,idsb1),prec_anno (b2,idsa2,idsb2)) = 
    b1=b2 andalso ID_list_eq(idsa1,idsa2) andalso ID_list_eq(idsb1,idsb2)
  | ANNO_eq (lassoc_anno (id1),lassoc_anno (id2)) = 
    ID_list_eq(id1,id2)
  | ANNO_eq (rassoc_anno (id1),rassoc_anno (id2)) =
    ID_list_eq(id1,id2)
  | ANNO_eq (pos_ANNO(_,an1),an2) = ANNO_eq (an1,an2)
  | ANNO_eq (an1,pos_ANNO(_,an2)) = ANNO_eq (an1,an2) 
  | ANNO_eq (_,_) = false

end

end


