(***************************************************************** *)
(*								   *)
(* Project: CATS						   *)
(* Author: Pascal Schmidt, University of Bremen		           *)
(* Date: 19.05.00						   *)
(* Purpose of this file: string -> HTML formatting                 *)
(*								   *)
(*								   *)
(***************************************************************** *)


(* Example use: format_text HTML_plain "Foo\nBar\n" *)

signature TextForm
=
sig
  val print_heading : bool -> string -> string
  val format_text   : bool * string * string -> string -> string

  (* pre-defined constants for format_text *)
  val HTML_plain    : bool * string * string   (* normal HTML text *)
  val HTML_pre      : bool * string * string   (* preformatted text *)
  val HTML_courier  : bool * string * string   (* text in Courier *)
  val Text_plain    : bool * string * string   (* plain text, no HTML *)

  val print_text    : bool -> string -> unit
end

(***********************************************************************)

structure TextForm :> TextForm
=
struct

fun print_heading html s =
    if html then
      "<h2>" ^ s ^ "</h2>\n\n"
    else
      "" ^ s ^ "\n"

fun conv []         = []
  | conv (#"\n"::t) = (String.explode "<br>\n") @ (conv t)
  | conv (h::t)     = h :: (conv t)

fun chk []     = []
  | chk (h::t) = if h=(#"<") then ( (String.explode "&lt;") @ (chk t) )
                 else if h=(#">") then ( (String.explode "&gt;") @ (chk t) )
		 else if h=(#"&") then ( (String.explode "&amp;") @ (chk t) )
                 else (h :: (chk t))

fun format_text (html,heading,ending) text =
  if html then
    heading ^ (String.implode o conv o chk o String.explode) text ^ ending
  else
    text

val HTML_plain   = (true,"\n","\n")

val HTML_pre     = (true,"<pre>\n","\n</pre>\n")

val HTML_courier = (true,"<tt>\n","\n</tt>\n")

val Text_plain   = (false,"","")


fun print_text html s =
   if html then print (format_text HTML_plain s)
   else print s

end

(***********************************************************************)
