(* *********************************************************************** *)
(*									   *)
(* Project : CATS 							   *)
(* Author: Till Mossakowski, University of Bremen			   *)
(* Date: 09.05.2000				 			   *)
(* Purpose of this file: Flattening of global environment		   *)
(*			 						   *)	
(*									   *)
(* *********************************************************************** *)


(* A structured specification, represented by a global environment,
   is flattened to a signature plus axioms (f_global_env).
   Translations are eliminated by renaming formulas along the
   signature morphism.

   todo:
   Reductions are eliminated by deleting the hidden symbols and
   collecting them into a separate signature. This needs further
   consideration!

   Flachklopfen sollte nichts duplizieren
  (gesharte Teile)
  insebesondere fuer encoding (doppelte Axiomnamen)
*)

structure Flatten :
sig
val flatten : GlobalEnv.global_env -> GlobalEnv.f_global_env
val flatten_senv : GlobalEnv.spec_env -> LocalEnv.sign * AS.L_FORMULA list
val flatten_slenv : GlobalEnv.spec_lenv -> GlobalEnv.ext_signature
(* It is assumed that the pushing hiding outside already has been done ! *)
end

= struct
open Utils AS LocalEnv GlobalEnv Rename


(*********** Flattening of a global environment *****************)
(* It is assumed that the psuhing hiding outside already has been done ! *)

fun get_ax (basic_env (_,ax)) = ax
  | get_ax (translate_env (senv,sigma)) =
    rename_axs sigma (get_ax senv)
  | get_ax (derive_env (senv,sigma)) =
    get_ax senv  (* Here, renaming according to pushing hiding outside
                    may become necessary *)
  | get_ax (union_env senvs) =
    flat (map get_ax senvs)
  | get_ax (extension_env senvs) =
    flat (map (get_ax o fst) senvs)
  | get_ax (free_spec_env senv) =
    get_ax senv
  | get_ax (cofree_spec_env senv) =
    get_ax senv
  | get_ax (closed_spec_env senv) =
    get_ax senv
  | get_ax (spec_inst_env (name,body,sigma,args)) =
    rename_axs sigma (get_ax body)
    @ flat (map get_ax args)

fun flatten_slenv (SPEC_ENV (Sigma,HSigma,senv)) =
    (Sigma,HSigma,rev (remove_dups (get_ax senv)))

fun flatten_gen_env (slenv,slenv_list,Sigma) =
    (flatten_slenv slenv,map flatten_slenv slenv_list)

fun flatten_global_entry (spec_defn_env (gen_env,slenv)) =
    f_spec_defn_env (flatten_gen_env gen_env,
                     flatten_slenv slenv)
  | flatten_global_entry (view_defn_env (gen_env,slenv1,sigma,slenv2)) =
    f_view_defn_env (flatten_gen_env gen_env,
                     flatten_slenv slenv1,
                     sigma,
                     flatten_slenv slenv2)
  | flatten_global_entry (arch_spec_defn_env archentry) =
    f_arch_spec_defn_env
  | flatten_global_entry (unit_spec_defn_env unitentry) =
    f_unit_spec_defn_env

fun flatten_global_entry_pair (n,entry) =
    (n,flatten_global_entry entry)
    
fun flatten (genv,an) =
    (Symtab_sid.make (map flatten_global_entry_pair (Symtab_sid.dest genv)),
     an)


fun flatten_senv (basic_env (Sigma,ax)) = (Sigma,ax)
  | flatten_senv (translate_env (senv,sigma)) =
    let val (Sigma,ax) = flatten_senv senv
        val Sigma' = Rename.rename_sign sigma Sigma
        val ax' = Rename.rename_axs sigma ax
    in (Sigma',ax')
    end
  | flatten_senv (derive_env (senv,sigma)) =
    flatten_senv senv  (* Hiding should be eliminated already *)
  | flatten_senv (union_env senvs) =
    let val fl = map flatten_senv senvs
    in (signature_union_list (map fst fl),
        flat (map snd fl))
    end
  | flatten_senv (extension_env senvs) =
    let val fl = map (flatten_senv o fst) senvs
    in (signature_union_list (map fst fl),
        flat (map snd fl))
    end
  | flatten_senv (free_spec_env senv) =
    flatten_senv senv
  | flatten_senv (cofree_spec_env senv) =
    flatten_senv senv
  | flatten_senv (closed_spec_env senv) =
    flatten_senv senv
  | flatten_senv (spec_inst_env (name,body,sigma,args)) =
    let val fl = map flatten_senv args
        val (body_sig,body_ax) = flatten_senv body
        (*val _ = writeln("Flattened body: "^BasicPrint.print_sign body_sig)*)
        val body_sig' = Rename.rename_sign sigma body_sig
        val body_ax' = Rename.rename_axs sigma body_ax
    in (signature_union_list (body_sig'::map fst fl),
        flat (body_ax'::map snd fl))
    end



end
