(* *********************************************************************** *)
(*									   *)
(* Project: HOL-CASL 							   *)
(* Author: Till Mossakowski, University of Bremen          		   *)
(* Date: 24.08.2001				 			   *)
(* Purpose of this file: Proof script for Basic/Numbers                    *)
(*			 						   *)	
(*									   *)
(* *********************************************************************** *)

print_mode := ["Inj"];  (* Show injections, but not types *)
(* print_mode := ["InjTypes",""]; This would show injections and types *)


(************************ Specification Nat *******************************)

CASL_context Nat.casl;

(* Abbreviations for getting axioms *)
val getNat = get_caslaxiom Nat.casl;

(* Feed all axioms into simplifier - a bit dangerous ... *)
AddsimpAll ();



(********** Inductive theorems about addition **********)

Goal "forall x,y,z:Nat . (x+y)+z=x+(y+z)";
by (induct_tac "x" 1);
by (ALLGOALS Asm_simp_tac);
qed "Nat_add_assoc";

Goal "forall x:Nat . x+0=x";
by (induct_tac "x" 1);
by (ALLGOALS Simp_tac);
qed "Nat_add_0_right";

Goal "forall x,y:Nat . x+(op suc:Nat->Nat)(y) = (op suc:Nat->Nat)(x+y)";
by (induct_tac "x" 1);
by (ALLGOALS Simp_tac);
qed "Nat_add_suc_right";

Addsimps [Nat_add_0_right,Nat_add_suc_right];

Goal "forall x,y:Nat . x+y=y+x";
by (induct_tac "x" 1);
by (ALLGOALS Simp_tac);
qed "Nat_add_comm";

Goal "forall x,y,z:Nat . x+(y+z)=y+(x+z)";
by (induct_tac "x" 1);
by (ALLGOALS (simp_tac (simpset() addsimps [Nat_add_comm])));
qed "Nat_add_left_commute";

(*Addition is an AC-operator*)
bind_thms ("Nat_add_ac", [Nat_add_assoc, Nat_add_comm, Nat_add_left_commute]);

(* Simple facts about multiplication *)

Goal "(1:Pos)*(1:Pos)=1:Pos";
by (simp_tac (overload_up()) 1);
qed "one_times_one_pos";
Addsimps [one_times_one_pos, one_times_one_pos RS (ga_embedding_injectivity_Pos_Nat RS iffD1)];


(********** Inductive theorems about multiplication **********)

Goal "forall x:Nat . x*0=0";
by (induct_tac "x" 1);
by (ALLGOALS Simp_tac);
qed "Nat_mult_0_right";


(*right successor law for multiplication*)
Goal "forall m,n:Nat . m * (op suc:Nat->Nat)(n) = m + (m * n)";
by (induct_tac "m" 1);
by (ALLGOALS(asm_simp_tac (simpset() addsimps Nat_add_ac)));
qed "Nat_mult_suc_right";

Addsimps [Nat_mult_0_right, Nat_mult_suc_right];


(*addition distributes over multiplication*)
Goal "forall x,y,z:Nat . (x+y)*z=(x*z)+(y*z)";
by (induct_tac "x" 1);
by (ALLGOALS (asm_simp_tac (simpset() addsimps Nat_add_ac)));
qed "Nat_add_mult_distrib";

(*Associative law for multiplication*)
Goal "forall x,y,z:Nat . (x*y)*z=x*(y*z)";
by (induct_tac "x" 1);
by (ALLGOALS (asm_simp_tac (simpset() addsimps [Nat_add_mult_distrib])));
qed "Nat_mult_assoc";

Goal "forall n:Nat . (op 1:Nat) * n = n";
by (Asm_simp_tac 1);
qed "Nat_mult_1";

Goal "forall n:Nat . n * (op 1:Nat) = n";
by (Asm_simp_tac 1);
qed "Nat_mult_1_right";

(*Commutative law for multiplication*)
Goal "forall m,n:Nat . m * n = n * m";
by (induct_tac "m" 1);
by (ALLGOALS (asm_simp_tac (simpset() addsimps [Nat_add_comm])));
qed "Nat_mult_comm";

Goal "forall x,y,z:Nat . x*(y*z) = y*(x*z)";
by (rtac trans 1);
by (rtac Nat_mult_comm 1);
by (rtac trans 1);
by (rtac Nat_mult_assoc 1);
by (rtac (Nat_mult_comm RS arg_cong) 1);
qed "Nat_mult_left_comm";

bind_thms ("Nat_mult_ac", [Nat_mult_assoc,Nat_mult_comm,Nat_mult_left_comm]);



(********** Theorems about ordering **********)


Goal "forall n:Nat . not n = 0 => exists m:Nat.  n = (op suc:Nat->Nat)(m)";
by (induct_tac "n" 1);
by Auto_tac;
qed "Nat_not0_implies_suc";

Goal "forall n:Nat . not n = 0 <=> exists m:Nat.  n = (op suc:Nat->Nat)(m)";
by Auto_tac;
by (etac (Nat_not0_implies_suc RS mp) 1);         
qed "Nat_not0_equiv_suc";

Goal "forall m:Nat . m <= 0 => m = 0";
by (induct_tac "m" 1);
by (ALLGOALS Asm_simp_tac);
qed "Nat_leq_0_imp_eq_0";

Addsimps [Nat_leq_0_imp_eq_0];


Goal "forall m,n:Nat. m<n => not n = 0";
by (induct_tac "n" 1);
by (ALLGOALS Simp_tac);
qed "Nat_gr_implies_not0";


Goal "forall n:Nat. not (n = 0) <=> (0 < n)";
by (induct_tac "n" 1);
by (ALLGOALS Simp_tac);
qed "Nat_neq0_conv";
AddIffs [Nat_neq0_conv];

(*This theorem is useful with blast_tac: (n=0 ==> False) ==> 0<n *)
bind_thm ("Nat_gr0I", [Nat_neq0_conv, notI] MRS iffD1);

Goal "forall n:Nat . (0<n) <=> (exists m:Nat . n = (op suc:Nat->Nat)(m))";
by (Step_tac 1);
by (ALLGOALS (Asm_full_simp_tac));
by (rtac ((neq_commute RS iffD1) RS (Nat_not0_implies_suc RS mp)) 1);
by (Simp_tac 1);
qed "Nat_gr0_conv_Suc";

Goal "forall n:Nat. not(0 < n) <=> n=0";
by (rtac iffI 1);
 by (rtac ccontr 1);
 by (ALLGOALS Asm_full_simp_tac);
qed "Nat_not_gr0";
AddIffs [Nat_not_gr0];

Goal "forall n,m':Nat . (op suc:Nat->Nat)(n) <= m' => (exists m:Nat. m' = (op suc:Nat->Nat)(m))";
by (induct_tac "m'" 1);
by  Auto_tac;
qed "Nat_suc_le_D";

(*Useful in certain inductive arguments*)
Goal "forall m,n:Nat . m < (op suc:Nat->Nat)(n) <=> (m=0 \\/ (exists j:Nat. m = (op suc:Nat->Nat)(j) /\\ j < n))";
by (induct_tac "m" 1);
by Auto_tac;
qed "Nat_less_suc_eq_0_disj";


Goal "forall n:Nat . min(0,n) = 0";
by (Simp_tac 1);
qed "Nat_min_0L";

Goal "forall n:Nat . min(n,0) = 0";
by (Simp_tac 1);
qed "Nat_min_0R";

Goal "forall m,n:Nat . min((op suc:Nat->Nat)(m),(op suc:Nat->Nat)(n)) = (op suc:Nat->Nat)(min(m,n))";
by (Simp_tac 1);
qed "Nat_min_suc_suc";

Addsimps [Nat_min_0L,Nat_min_0R,Nat_min_suc_suc];

Goal "forall n:Nat . max(0,n) = n";
by (Simp_tac 1);
qed "Nat_max_0L";

Goal "forall n:Nat . max(n,0) = n";
by (Simp_tac 1);
br impI 1;  
br (Nat_leq_0_imp_eq_0 RS mp RS sym) 1; 
ba 1;
qed "Nat_max_0R";


Goal "forall n:Nat . n<=n";
by (induct_tac "n" 1);
by (ALLGOALS Simp_tac);
qed "Nat_le_refl";

Addsimps [Nat_le_refl];

Goal "forall m,n:Nat . m<=n => m <=suc(n)";
(* We need induction over m for the goal quantified over n;
   therefore, we introduce a universal quantifier *)
br spec 1;  
back(); back(); back();    
by (induct_tac "m" 1);
by (ALLGOALS Simp_tac);
(* Now we have to get rid off the quantifier *)
br impI 1;
br allI 1;
(* The following three lines should be replaced by
   a datatype driven case_tac *)
by (case_tac "x=0" 1);
by (Asm_full_simp_tac 1);
by (asm_full_simp_tac (HOL_ss addsimps [Nat_not0_equiv_suc]) 1);
by (Step_tac 1);    
by (Asm_full_simp_tac 1);
qed "Nat_le_SucI";

Addsimps [Nat_le_SucI RS mp];


Goal "forall m,n:Nat . suc(m)<=n => m <=n";
by (induct_tac "n" 1);
by Auto_tac;
qed "Nat_le_SucE";


Goal "forall m,n:Nat . n <= m + n";
by (induct_tac "m" 1);
by (ALLGOALS (simp_tac (simpset() addsimps [Nat_le_SucI])));    
qed "Nat_le_add2";

Goal "forall m,n:Nat . n <= n + m";
by (simp_tac (simpset() addsimps Nat_add_ac) 1);
by (rtac Nat_le_add2 1);
qed "Nat_le_add1";

Goal "forall x,y,v:Nat . x <= y => x <= v + y";
by (induct_tac "v" 1);
by (ALLGOALS (simp_tac (simpset() addsimps [Nat_le_SucI])));
qed "Nat_le_mon1";


Goal "forall m,n:Nat . m<=suc(n) => (m<=n \\/ m=suc(n))";
br spec 1;
back(); back(); back(); back(); back();
by (induct_tac "m" 1);
by (ALLGOALS Simp_tac); 
br impI 1;
br allI 1;
by (case_tac "x=0" 1);
by (Asm_full_simp_tac 1);
by (asm_full_simp_tac (HOL_ss addsimps [Nat_not0_equiv_suc]) 1);
by (Auto_tac);    
qed "Nat_le_suc";

val Nat_le_sucE = Nat_le_suc RS mp RS disjE;

Goal "forall m,n:Nat . m<=n /\\ not m=n => suc(m)<=n";
br spec 1;  
back(); back();  
by (induct_tac "m" 1);
by (ALLGOALS Simp_tac); 
br allI 1;
by (induct_tac "x" 1);
by (ALLGOALS Simp_tac); 
br impI 1;
br allI 1;
by (case_tac "x=0" 1);
by (Asm_full_simp_tac 1);
by (asm_full_simp_tac (HOL_ss addsimps [Nat_not0_equiv_suc]) 1);
by (Auto_tac);    
qed "Nat_less_suc1";


Goal "forall m,n:Nat . m<suc(n) => (m<n \\/ m=n)";
br spec 1;  
back(); back(); back();   
by (induct_tac "m" 1);
by (ALLGOALS Simp_tac);
by (step_tac (claset() addSIs [Nat_le_suc]) 1);
by (asm_full_simp_tac (simpset() addsimps [Nat_less_suc1]) 1); 
qed "Nat_less_suc";

val Nat_less_sucE = Nat_le_suc RS mp RS disjE;

Goal "forall m,n:Nat . m<n => exists k:Nat . n=suc(m+k)";
by (induct_tac "n" 1);
by (ALLGOALS Simp_tac);
br impI 1;
by (etac (Nat_leq_0_imp_eq_0 RS mp RS sym)1);    
by (blast_tac (claset() addSEs [Nat_less_sucE]
                        addSIs [Nat_add_0_right RS sym, Nat_add_suc_right RS sym]) 1);
qed "Nat_less_imp_suc_add";

Goal "forall m,n:Nat . m <= n => m<suc(n)";
br spec 1;  
back(); back();  back();
by (induct_tac "m" 1);
by (ALLGOALS Simp_tac); 
br impI 1;
br allI 1;
by (case_tac "x=0" 1);
by (Asm_full_simp_tac 1);
by (asm_full_simp_tac (HOL_ss addsimps [Nat_not0_equiv_suc]) 1);
by (Auto_tac);    
qed "Nat_le_less_suc";



Goal "forall m,n:Nat . m <= n => exists k:Nat. n = (m+k)";
br impI 1;
br ex_forward 1;
br (( Nat_le_less_suc RS mp) RS (Nat_less_imp_suc_add RS mp)) 1;       
by (ALLGOALS Asm_full_simp_tac);     
qed "Nat_le_imp_add";


Goal "forall x,y:Nat . suc(x)<=y => x<=y";
by (induct_tac "y" 1);
by Auto_tac;
qed "Nat_suc_le";


Goal "forall x,y,z:Nat . x<=y /\\ y<=z => x<=z";
br spec 1;
back(); back();
br spec 1;
back();
by (induct_tac "y" 1);
by (Step_tac 2);
bd (Nat_le_suc RS mp) 2;
by (Step_tac 2);
bd (Nat_suc_le RS mp) 2;
by (Asm_simp_tac 2);
bd (Nat_leq_0_imp_eq_0 RS mp) 1;
by Auto_tac;
qed "Nat_le_trans";


Goal "forall u,x,y:Nat . x<=y => x+u<=y+u";
by (induct_tac "u" 1);
by Auto_tac;
qed "Nat_leq_add_right";


Goal "forall u,v,x,y:Nat . u<=v /\\ x<=y => u+x <= v+y";
by (induct_tac "y" 1);
by (induct_tac "u" 2);
by Auto_tac;
bd (Nat_leq_0_imp_eq_0 RS mp) 1;
by (simp_tac (HOL_ss addsimps [Nat_add_suc_right RS sym]) 2);
br (Nat_le_mon1 RS mp) 2;
bd (Nat_suc_le RS mp) 3; back();
br (Nat_le_SucE RS mp) 4;
bd (Nat_le_suc RS mp) 5;
bd (Nat_leq_add_right RS mp) 5;
br (Nat_le_SucE RS mp) 6;
bd (Nat_suc_le RS mp) 7; back();
by Auto_tac;
qed "Nat_add_monotone";

Goal "forall m,n: Nat . def(m-?n) <=> m >= n";
by (auto_tac (clasimpset() addsimps2 [Nat_le_add2,ga_membership_Nat]));
br ex_forward 1;
br (Nat_le_imp_add RS mp) 1;
by (REPEAT (asm_full_simp_tac (simpset() addsimps [Nat_add_comm]) 1));   
qed "Nat_sub_dom";

Addsimps [Nat_sub_dom];


(************************ Specification Int *******************************)

CASL_context Int.casl;

(* Abbreviations for getting axioms *)
val getInt = get_caslaxiom Int.casl;

AddsimpAll (); (* Feed all axioms into simplifier - a bit dangerous ... *)
AddSimpParents (); (* also add the simplifiers of the parent theories *)

(* Addition *)

Goal "forall x:Int . x+0=x";
by (induct_tac "x" 1);
by (simp_tac (overload_down()) 1);
qed "Int_add_0_left";
Addsimps [Int_add_0_left];


Goal "forall x:Int . 0+x=x";
by (induct_tac "x" 1);
by (simp_tac (overload_down()) 1);
qed "Int_add_0_right";
Addsimps [Int_add_0_right];

Goal "forall x,y:Nat . (x+y)-y = x";
by Auto_tac ; 
qed "Nat_add_minus";

Addsimps [Nat_add_minus];

Goal "forall m,n: Nat . (op __-__:Nat*Nat->Int)(suc(m),suc(n))=(op __-__:Nat*Nat->Int)(m,n)";
by Auto_tac;
qed "minus_suc_Int";
Addsimps[minus_suc_Int];

Goal "forall n:Nat . 0:Nat <= n:Int <=> 0<=n";
by Auto_tac;
by (simp_tac (simpset() delsimps [Int_Nat_embedding] addsimps [Int_Nat_embedding RS sym])1);
qed "leq_zero_Nat_Int";
Addsimps[leq_zero_Nat_Int];

(*Goal "forall m,n:Nat . m:Int <= n:Int <=> m<=n";
by (induct_tac "m" 1);
by (Simp_tac 1);
by (simp_tac (simpset() delsimps [Int_Nat_embedding] addsimps [Int_Nat_embedding RS sym])1);
by Auto_tac;
*)

Goal "forall m,n:Nat . (op __-__:Nat*Nat->Int)(suc(m),suc(n)) = (op __-__:Nat*Nat->Int)(m,n) ";
by Auto_tac;
qed "Int_minus_suc";
Addsimps [Int_minus_suc];

Goal "forall m,n:Nat . (op __-__:Nat*Nat->Int)(0,suc(m)) = n:Int => false";
by Auto_tac;
qed "Int_neg_not_in_Nat1";
Addsimps [Int_neg_not_in_Nat1];

Goal "forall m:Nat . not (op __-__:Nat*Nat->Int)(0,suc(m)) in Nat";
by (simp_tac (HOL_ss addsimps [ga_membership_Nat_Int]) 1);
by Auto_tac;
qed "Int_neg_not_in_Nat2";
Addsimps [Int_neg_not_in_Nat2];

Goal "forall m,n:Nat . (op __-__:Nat*Nat->Int)(m,n) in Nat <=> n<=m";
br spec 1; back(); back();
by (induct_tac "n" 1);
br impI 2;
br allI 2;
by (induct_tac "x" 2);
by (simp_tac (HOL_ss addsimps [Int_neg_not_in_Nat2]) 2);
by Auto_tac;
by (simp_tac (simpset() delsimps [Int_Nat_embedding] addsimps [Int_Nat_embedding RS sym]) 1);
qed "Int_dom_minus1";
Addsimps [Int_dom_minus1];

Goal "forall m,n:Nat . def (op __-__:Nat*Nat->Int)(m,n) as Nat <=> n<=m";
by (simp_tac (simpset() delsimps [ga_projection_membership_Nat_Int] addsimps [ga_projection_membership_Nat_Int RS sym]) 1);
qed "Int_dom_minus2";
Addsimps [Int_dom_minus2];


Goal "forall a,b: Nat . def(a -? b) => a -? b = a - b";
(* leave injections there, for transitivity *)
by (auto_tac (clasimpset() addsimps2 [ga_membership_Nat] delsimps2 [Int_Nat_embedding]));
(* We now have to unfold one injection into two injections, using transitivity,
   and then can delete the same injection on both sides, using injectivity.
   We need to use HOL_ss for this purpose, since the simpset already
   contains transitivity in the other direction!
*)
by (simp_tac (HOL_ss addsimps [
                               ga_transitivity_Nat_Q_Nat_Q_Int_ RS sym,
                               ga_embedding_injectivity_Q_Nat_Q_Int_]) 1);
by (Simp_tac 1);
qed "Int_Nat_sub_compat";


Goal "forall x,y,z:Int . (x+y)+z = x+(y+z)";
(* Express x,y, and z as differences of natural numbers *)
by (induct_tac "x" 1);
by (induct_tac "y" 1);
by (induct_tac "z" 1);
by (simp_tac (simpset() addsimps [Nat_add_assoc]) 1);
qed "Int_add_assoc";


Goal "forall x:Int . x-x=0";
by (induct_tac "x" 1);
(* We need the overloading equation 0:Int=0:Nat *)
by (simp_tac (overload_down() addsimps [Nat_add_comm]) 1); 
qed "Int_minus_0";

Goal "-0:Int = 0:Int";
by (simp_tac (overload_down()) 1);
qed "Int_minus_zero";
Addsimps [Int_minus_zero];

(* Multiplication *)

Goal "forall x:Int . 0*x=0";
by (induct_tac "x" 1);
by (simp_tac (overload_down()) 1);
qed "Int_mult_0_left";
Addsimps [Int_mult_0_left];

Goal "forall x:Int . x*0=0";
by (induct_tac "x" 1);
by (simp_tac (overload_down()) 1);
qed "Int_mult_0_right";
Addsimps [Int_mult_0_right];


Goal "forall x:Int . 1*x=x";
by (induct_tac "x" 1);
by (simp_tac (overload_down()) 1);
 by (simp_tac (overload_up()) 1);
qed "Int_mult_1_left";
Addsimps [Int_mult_1_left];
 

Goal "forall x:Int . x*1=x";
by (induct_tac "x" 1);
by (simp_tac (overload_down()) 1);
 by (simp_tac (overload_up()) 1);
qed "Int_mult_1_right";
Addsimps [Int_mult_1_right];


Goal "forall x:Int . (1:Pos:Int)*x=x";
by (induct_tac "x" 1);
by (simp_tac (overload_down()) 1);
 by (simp_tac (overload_up()) 1);
qed "Int_mult_1pos_left";
Addsimps [Int_mult_1pos_left];


Goal "forall x:Int . x*(1:Pos:Int)=x";
by (induct_tac "x" 1);
by (simp_tac (overload_down()) 1);
 by (simp_tac (overload_up()) 1);
qed "Int_mult_1pos_right";
Addsimps [Int_mult_1pos_right];


Goal "forall x,y:Int . (-x)*y = -(x*y)";
by (induct_tac "x" 1);
by (induct_tac "y" 1);
by Auto_tac;
qed "Int_mult_minus";
Addsimps [Int_mult_minus];




(************************ Specification Rat *******************************)

CASL_context Rat.casl;
AddSimpParents (); (* also add the simplifiers of the parent theories *)

(* Abbreviations for getting axioms *)
val getRat = get_caslaxiom Rat.casl;

AddsimpAll (); (* Feed all axioms into simplifier - a bit dangerous ... *)
AddSimpParents (); (* also add the simplifiers of the parent theories *)

(* The numerals *)

Goal "0:Rat = (op __/__:Int*Pos->Rat)(0:Int,1:Pos)";
by (simp_tac (HOL_ss addsimps [embeddingIntToRat]) 1);
by (simp_tac (overload_down()) 1);
qed "zero_Rat";

Goal "0:Rat = (op __/__:Int*Pos->Rat)(0:Nat,1:Pos)";
by (simp_tac (HOL_ss addsimps [embeddingIntToRat]) 1);
by (simp_tac (overload_down()) 1);
qed "zero_Rat_Nat";

Goal "1:Rat = (op __/__:Int*Pos->Rat)(1:Nat,1)";
by (simp_tac (HOL_ss addsimps [embeddingIntToRat]) 1);
by (simp_tac (overload_down()) 1);
by (simp_tac (HOL_ss addsimps [ga_embedding_injectivity_Pos_Nat RS sym]) 1);
br ((ga_overload_F_op_1_Nat_op_1_Pos_ RS (Nat_1_def RS sym RS trans))
 RS (ga_overload_F_op_suc_NatToNat_op_suc_NatToPos_ RS sym RS trans) RS sym) 1;
qed "one_Rat";

Goal "2:Rat = (op __/__:Int*Pos->Rat)(2:Nat,1)";
by (simp_tac (HOL_ss addsimps [embeddingIntToRat]) 1);
by (simp_tac (overload_down()) 1);
qed "two_Rat";

(* Simple facts about addition *)


Goal "1:Rat + 1:Rat = 2:Rat";
by (simp_tac (simpset() delsimps [embeddingIntToRat] 
               addsimps [one_Rat,two_Rat,ga_transitivity_Nat_Int_Rat RS sym,ga_transitivity_Pos_Nat_Int RS sym]) 1);
by (Simp_tac 1);
by (simp_tac (overload_up()) 1);
qed "one_plus_one_Rat";
Addsimps[one_plus_one_Rat];

(* Inductive theorems *)

Goal "forall i,j: Int; p,q:Pos . (i / p) - (j / q) =  ((i * q) - (j * p)) / (p * q)";
by (induct_tac "i" 1);
by (induct_tac "j" 1);
by (simp_tac (simpset() delsimps [ga_transitivity_Pos_Nat_Int]
                        addsimps [ga_transitivity_Pos_Nat_Int RS sym]) 1);
qed "Rat_sub_rule";

Goal "forall x,y,z:Rat . (x+y)+z=x+(y+z)";
by (induct_tac "x" 1);
by (induct_tac "y" 1);
by (induct_tac "z" 1);
by (simp_tac (simpset() delsimps [ga_transitivity_Pos_Nat_Int]
                        addsimps [ga_transitivity_Pos_Nat_Int RS sym,Nat_add_assoc]) 1);

(* by (simp_tac (overload_up() delsimps [Int_Nat_embedding]) 1); *)

by (simp_tac (simpset() addsimps [getNat "CASL_Nat.ga_overload_F_(op __*__:Nat*Nat->Nat)_(op __*__:Pos*Pos->Pos)" RS sym,
                                  Nat_add_assoc]) 1);
by (induct_tac "x1" 1);
by (induct_tac "x1a" 1);
by (induct_tac "x1b" 1);
by (simp_tac (simpset() addsimps [Nat_add_mult_distrib])1);
by (simp_tac (simpset() addsimps (Nat_add_ac@Nat_mult_ac))1);
qed "Rat_add_assoc";
Addsimps [Rat_add_assoc];

(* Simple facts about ordering *)
Goal "1:Rat <= 1:Rat";
by (simp_tac (HOL_ss addsimps [one_Rat]) 1);
by Auto_tac;
by (simp_tac (HOL_ss addsimps [ga_overload_P_pred_LtEq_RatxRat_pred_LtEq_IntxInt_]) 1);
by Auto_tac;
qed "one_leq_one_Rat";
Addsimps [one_leq_one_Rat];

Goal "forall x,y:Rat . x<=y <=> y-x>=0";
by (induct_tac "x" 1);
by (induct_tac "y" 1);
by (simp_tac (HOL_ss addsimps [zero_Rat_Nat]) 1);
by (simp_tac (HOL_ss addsimps [ga_overload_F_op_0_Int_op_0_Nat_ RS sym]) 1);
by (simp_tac (simpset() delsimps [embeddingIntToRat]) 1);
qed "Rat_leq_minus";


(*
Goal "forall x,y:Rat . x>=0 /\\ y>=0 => x+y >=0";
by (induct_tac "x" 1);
by (induct_tac "y" 1);
by (simp_tac (HOL_ss addsimps [zero_Rat]) 1);
by (simp_tac (simpset() delsimps [embeddingIntToRat] addsimps [ga_overload_F_op_1_Int_op_1_Pos_ RS sym]) 1);
by (simp_tac (simpset() delsimps [ga_transitivity_Pos_Nat_Int]
                        addsimps [ga_transitivity_Pos_Nat_Int RS sym]) 1);
by Auto_tac;
br (Nat_add_monotone RS mp) 1;

Goal "forall u,v,x,y:Rat . u<=v /\\ x<=y => u+x <= v+y";
by (induct_tac "u" 1);
by (induct_tac "v" 1);
by (induct_tac "x" 1);
by (induct_tac "y" 1);
by (induct_tac "x1" 1);
by (induct_tac "x1a" 1);
by (induct_tac "x1b" 1);
*)

Goal"(op __/__:Int*Pos->Rat)(2,1)+(op __/__:Int*Pos->Rat)(4,1)=(op __/__:Int*Pos->Rat)(6,1)";
by (simp_tac (simpset() delsimps [embeddingIntToRat]) 1);
qed "Rat_Two_plus_four_is_six";

(*
Goal "forall i:Int; p:Pos . (i:Rat)/(p:Rat) =  (op __/__:Int*Pos->Rat)(i,p)";
by (simp_tac (simpset() delsimps [embeddingIntToRat] addsimps[embeddingIntToRat RS sym,ga_transitivity_Pos_Int_Rat RS sym] ) 1);
br (Rat_divide_rule RS trans) 1;
*)

Goal"(2/1)+(4/1)=6/1";
by (simp_tac (simpset() delsimps [embeddingIntToRat] addsimps[ga_transitivity_Nat_Int_Rat RS sym] ) 1);
by Auto_tac;
by (simp_tac (simpset() delsimps [embeddingIntToRat,Int_Nat_embedding,Nat_2_def,Nat_1_def,Nat_4_def,Nat_6_def,ga_transitivity_Nat_Int_Rat] 
               addsimps [one_Rat,two_Rat,ga_transitivity_Nat_Int_Rat RS sym,ga_transitivity_Pos_Nat_Int RS sym]) 1);

