(*  Title:      Pure/context.ML
    ID:         $Id: context.sml,v 1.1 2004/02/13 11:28:53 7till Exp $
    Author:     Markus Wenzel, TU Muenchen

Global theory context.

Adapted for HOL-CASL by Till Mossakowski
*)

signature CASL_BASIC_CONTEXT =
sig
  val CASL_context: casltheory -> unit
  val CASL_the_context: unit -> casltheory
end;

signature CASL_CONTEXT =
sig
  include CASL_BASIC_CONTEXT
  val get_context: unit -> casltheory option
  val set_context: casltheory option -> unit
  val reset_context: unit -> unit
  val setmp: casltheory option -> ('a -> 'b) -> 'a -> 'b
  val pass: casltheory option -> ('a -> 'b) -> 'a -> 'b * casltheory option
  val pass_casltheory: casltheory -> ('a -> 'b) -> 'a -> 'b * casltheory
  val save: ('a -> 'b) -> 'a -> 'b
  val >> : (casltheory -> casltheory) -> unit
  val ml_output: (string -> unit) * (string -> unit)
  val use_mltext: string -> bool -> casltheory option -> unit
  val use_mltext_casltheory: string -> bool -> casltheory -> casltheory
  val use_let: string -> string -> string -> casltheory -> casltheory
  val use_setup: string -> casltheory -> casltheory
end;

structure CASL_Context: CASL_CONTEXT =
struct


(* casltheory context *)

local
  val current_theory = ref (None: casltheory option);
in
  fun get_context () = ! current_theory;
  fun set_context opt_thy = 
      (current_theory := opt_thy;
       case opt_thy of 
       Some th => Context.context (#thy (CASLTheory.rep_casltheory th)) 
       | None => ()
      )
  fun setmp opt_thy f x = Library.setmp current_theory opt_thy f x;
end;

fun CASL_the_context () =
  (case get_context () of
    Some thy => thy
  | _ => error "Unknown theory context");

fun CASL_context thy = set_context (Some thy);
fun reset_context () = set_context None;

fun pass opt_thy f x =
  setmp opt_thy (fn x => let val y = f x in (y, get_context ()) end) x;

fun pass_casltheory thy f x =
  (case pass (Some thy) f x of
    (y, Some thy') => (y, thy')
  | (_, None) => error "Lost theory context in ML");

fun save f x = setmp (get_context ()) f x;


(* map context *)

nonfix >>;
fun >> f = set_context (Some (f (CASL_the_context ())));


(* use ML text *)

val ml_output = (writeln, error_msg: string -> unit);
fun use_output verb txt = use_text ml_output verb (Symbol.plain_output txt);

fun use_mltext txt verb opt_thy = setmp opt_thy (fn () => use_output verb txt) ();
fun use_mltext_casltheory txt verb thy = #2 (pass_casltheory thy (use_output verb) txt);

fun use_context txt = use_mltext_casltheory ("Context.>> (" ^ txt ^ ");") false;

fun use_let bind body txt =
  use_context ("let " ^ bind ^ " = " ^ txt ^ " in\n" ^ body ^ " end");

val use_setup =
  use_let "val setup: (casltheory -> casltheory) list" "Library.apply setup";


end;

structure CASL_BasicContext: CASL_BASIC_CONTEXT = CASL_Context;
open CASL_BasicContext;
