(* ************************************************************************* *)
(*									     *)
(* Project: CATS 							     *)
(* Author:  Bartek Klin                                          	     *)
(* Date: 2000				 			             *)
(* Purpose of this file: Computation of signature morphisms from symbol maps *)
(*			 						     *)	
(*									     *)
(* ************************************************************************* *)

(* This module implements the functions computing signature morphisms
   from symbol maps that are described in the CASL semantics,
   see CoFI study note S-9, chapter 5.
   It uses algorithms described in Bartek Klin's masters thesis.
*)
(* todo
  Adapt symbol map analysis to new semantics 
    (instead of minimality: preservation of names)
     Example from WADT01 paper takes too long
     (with preservation of names instead of minimality,
      consider rsymmap first in try_to_induce?
      at least check first if source of raw symbol
      map actually is matchable to source signature,
      and same for target )
*)

structure Stat_symmaps : sig
  val sig_symbols: LocalEnv.local_env -> (Symbols.symbol list);
  val inclusion_morphism: LocalEnv.local_env -> GlobalEnv.morphism;
  val matches: (Symbols.symbol * Symbols.raw_symbol) -> bool;
  val generated_signat: 
    Symbols.symbol list * LocalEnv.local_env * AS.ANNO list ->   
    (LocalEnv.local_env * AS.ANNO list * GlobalEnv.morphism);
  val cogenerated_signat: 
    Symbols.symbol list * LocalEnv.local_env * AS.ANNO list ->
    (LocalEnv.local_env * AS.ANNO list * GlobalEnv.morphism);
  val induced_from_morphism: 
    Symbols.raw_symbol_map * LocalEnv.local_env * AS.ANNO list ->
    (LocalEnv.local_env * AS.ANNO list * GlobalEnv.morphism);
  val induced_from_to_morphism:
    Symbols.raw_symbol_map * LocalEnv.local_env * LocalEnv.local_env ->
    GlobalEnv.morphism;

end = struct 

open AS;
open Utils;
open LocalEnv;
(*open Bk_stat_symmaps;
open Bk_bridge;*)
open Symbols;
open ArchTypes;
open BasicPrint;
open StructuredPrint;

infix mem;

exception INTERNAL_ERROR;

(* Inclusion morphism *)
(* LocalEnv.local_env -> GlobalEnv.morphism *)
(* It seems that 3 empty tables make an inclusion morphism.
   In case they don't, uncomment the following function ... *)
(* fun inclusion_morphism(lenv:LocalEnv.local_env) = 
  let val (sl,_,_,fl,pl) = LocalEnv.env_to_list(lenv) in
    let val sm = foldl (fn (smi,s) => Symtab_id.update_new((s,s),smi)) 
                       (Symtab_id.empty,sl)
    val fm = foldl (fn (fmi,(on,f_e)) => 
      Symtab_id.update_new((on,map (fn ot => (ot,on)) f_e),fmi))
                   (Symtab_id.empty,fl)
    val pm = foldl (fn (pmi,(pn,p_e)) =>
      Symtab_id.update_new((pn,map (fn pt => (pt,pn)) p_e),pmi))
                   (Symtab_id.empty,pl)
    in
      ((sm,fm,pm):GlobalEnv.morphism)
    end
  end
;
*)

(*... and comment this one. *)
fun inclusion_morphism(_) = (Symtab_id.empty,Symtab_id.empty,Symtab_id.empty);

(******* Some additional stuff for renaming/hiding annotations *******)

(*** hiding ***)

(* LocalEnv.local_env -> AS.ID -> bool *)
fun is_sort_in_lenv (subsortenv,_,_,_) id =
  not(Symtab_id.lookup(subsortenv,id)=None)
;

fun is_fun_in_lenv (_,_,funenv,_) id =
  not(Symtab_id.lookup(funenv,id)=None)
;

fun is_pred_in_lenv (_,_,_,predenv) id =
  not(Symtab_id.lookup(predenv,id)=None)
;

fun is_id_in_lenv lenv id =
  (is_sort_in_lenv lenv id) orelse (is_fun_in_lenv lenv id) orelse
  (is_pred_in_lenv lenv id)
;

(* LocalEnv.local_env * AS.ANNO -> bool *)
fun is_anno_in_lenv(lenv,annot) = let val checkfun = is_fun_in_lenv lenv in
  case annot of
    number_anno(id) => checkfun id
  | floating_anno(id1,id2) => checkfun id1 andalso checkfun id2
  | string_anno(id1,id2) => checkfun id1 andalso checkfun id2
  | list_anno(id1,id2,id3) => checkfun id1 andalso checkfun id2 andalso
                              checkfun id3
  | display_anno(id,s) => is_id_in_lenv lenv id
  | prec_anno(_,idl1,idl2) => forall' checkfun idl1 andalso
                              forall' checkfun idl2
  | lassoc_anno(idl) => forall' checkfun idl
  | rassoc_anno(idl) => forall' checkfun idl
  | _ => raise (ERR "is_anno")
 end 
;

(* LocalEnv.local_env * AS.ANNO list -> AS.ANNO list *)
fun cut_annos(lenv,annos) = 
  filter (fn annot => is_anno_in_lenv(lenv,annot)) annos
;

(*** translating ***)

(* GlobalEnv.morphism -> AS.ID -> AS.ID *)
fun translate_sort (smap,_,_) id = 
  case Symtab_id.lookup(smap,id) of Some(id2) => id2 | None => id
;

fun translate_fun (_,fmap,_) id = 
  case Symtab_id.lookup(fmap,id) of 
    Some((_,on,_)::_) => on |
    Some([]) => id | 
    None => id
;

fun translate_pred (_,_,pmap) id = 
  case Symtab_id.lookup(pmap,id) of 
   Some((_,pn)::_) => pn | 
   Some([]) => id | 
   None => id
;

(* GlobalEnv.morphism -> AS.ANNO -> AS.ANNO *)
(* Doubt: How should we really translate display annotations? *)
fun translate_anno mor annot = let val transfun = translate_fun mor in
  case annot of
    number_anno(id) => number_anno(transfun id)
  | floating_anno(id1,id2) => floating_anno(transfun id1,transfun id2)
  | string_anno(id1,id2) => string_anno(transfun id1,transfun id2)
  | list_anno(id1,id2,id3) => list_anno(transfun id1,transfun id2,transfun id3)
  | display_anno(id,s) => display_anno(transfun id,s)  (* Wrong?? *)
  | prec_anno(bl,id1,id2) => prec_anno(bl,map transfun id1,map transfun id2)
  | lassoc_anno(idl) => lassoc_anno(map transfun idl)
  | rassoc_anno(idl) => rassoc_anno(map transfun idl)
 end
;

(* GlobalEnv.morphism * AS.ANNO list -> AS.ANNO *)
fun translate_annos(mor,annos) =
  map (translate_anno mor) annos
;

(* Auxiliary function: adding one element to a list in Symtab_id value *)
fun add_to_symtabid((key,value),stab) = 
  let val oldval = (case Symtab_id.lookup(stab,key) of 
      None => [] | Some(x) => x) in
    Symtab_id.update((key,value::oldval),stab)
  end
;

(*************** Some auxiliary functions, taken from morphisms.sml *****)
(* (Where should they belong, anyway? *)

fun sort_via_morphism (sm,_,_) s = 
  case Symtab_id.lookup(sm,s) of
    None => s
  | Some(s1) => s1
;

fun fun_via_morphism (_,fm,_) (on,(oss,os),total) = 
  case Symtab_id.lookup(fm,on) of
    None => on
  | Some(vall) => foldl 
      (fn (onri,(ot1,on1)) => 
        if (if total then total_op_type(sorts(oss),os)
            else partial_op_type(sorts(oss),os))=ot1 then on1 else onri)
      (on,vall)
;

fun pred_via_morphism (_,_,pm) (pn,pt) =
  case Symtab_id.lookup(pm,pn) of
    None => pn
  | Some(vall) => foldl (fn (pnri,(pt1,pn1)) => 
      if pred_type(sorts(pt))=pt1 then pn1 else pnri)
                        (pn,vall)
;
fun sorts_via_morphism mor sl = map (sort_via_morphism mor) sl;

fun fprof_via_morphism mor (sl,s) = 
  (sorts_via_morphism mor sl,sort_via_morphism mor s);

fun symbol_via_morphism(mor,sym) =
  case sym of
    SORT_SYMBOL(s) => SORT_SYMBOL(sort_via_morphism mor s)
  | TOTAL_FUN_SYMBOL(on,fprof) =>
      TOTAL_FUN_SYMBOL(fun_via_morphism mor (on,fprof,true),
                       fprof_via_morphism mor fprof)
  | PARTIAL_FUN_SYMBOL(on,fprof) =>
      PARTIAL_FUN_SYMBOL(fun_via_morphism mor (on,fprof,false),
                         fprof_via_morphism mor fprof)
  | PRED_SYMBOL(pn,pprof) =>
      PRED_SYMBOL(pred_via_morphism mor (pn,pprof),
                  sorts_via_morphism mor pprof)
;

(*************** Four main functions of symbol maps **************)

(* Signature generated by a symbol set.*)
(* symbol list * LocalEnv.local_env * AS.ANNO list ->
     (LocalEnv.local_env * AS.ANNO list * GlobalEnv.morphism)  *)
(* p. 86 *)
fun generated_signat(syml,(sv,vv,fv,pv),an) =
  if not (Finset.is_subset(syml,sig_symbols(sv,vv,fv,pv))) then
    raise (STAT_EXCEPTION
	  ("The following revealed symbols are not in the signature:\n"
             ^print_symbol_set(Finset.remove_set(sig_symbols(sv,vv,fv,pv),syml)))) 
  else 
  let val (sv1,vv1,fv1,pv1) = (* first without subsorting relations *)
    foldl (fn ((svi,vvi,fvi,pvi),sym) =>
      case sym of
        SORT_SYMBOL(sn) =>
		(Symtab_id.update((sn,[]),svi),vvi,fvi,pvi)
      | TOTAL_FUN_SYMBOL(fnm,(sseq,sn)) =>
		(foldl (fn (svii,sni) => Symtab_id.update((sni,[]),svii))
		       (Symtab_id.update((sn,[]),svi),sseq),
                 vvi,
		 add_to_symtabid((fnm,total_op_type(sorts(sseq),sn)),fvi),
		 pvi)
      | PARTIAL_FUN_SYMBOL(fnm,(sseq,sn)) =>
		(foldl (fn (svii,sni) => Symtab_id.update((sni,[]),svii))
		       (Symtab_id.update((sn,[]),svi),sseq),
                 vvi,
		 add_to_symtabid((fnm,partial_op_type(sorts(sseq),sn)),fvi),
		 pvi)
      | PRED_SYMBOL(fnm,sseq) =>
		(foldl (fn (svii,sni) => Symtab_id.update((sni,[]),svii))
		       (svi,sseq),
                 vvi,fvi,
		 add_to_symtabid((fnm,pred_type(sorts(sseq))),pvi)))
      (empty_local_env,syml)
  (* now adding subsorting relations *)
  val ss = map fst (Symtab_id.dest sv1) 
  val sigma = (foldl (fn (sv1i,sn) => 
        let val sls = case Symtab_id.lookup(sv,sn) of 
 			None => [] | Some(x) => x 
        val newsls = filter (fn sni => sni mem ss) sls
        in
   	  Symtab_id.update_new((sn,newsls),sv1i)
        end) (Symtab_id.empty,ss),
       vv1,fv1,pv1)
  val mor = inclusion_morphism(sigma)
  val new_an = cut_annos(sigma,an)
  in 
    (sigma,new_an,mor)
  end
;

(* Signature cogenerated by a symbol set *)
(* symbol set * LocalEnv.local_env * AS.ANNO list ->
     (LocalEnv.local_env * AS.ANNO list * GlobalEnv.morphism)  *)
(* p. 86 *)
fun cogenerated_signat(syml,cur_env,an) = 
  let val sigsyms = sig_symbols(cur_env)
  val wosyml = Finset.remove_set(syml,sigsyms)
  val woss = map (fn SORT_SYMBOL(sn) => sn
                    | _ => raise ERR ("cogenerated 1"))
                 (filter (fn SORT_SYMBOL(_) => true | _ => false) wosyml)
  val cogsyms = filter (
        fn SORT_SYMBOL(x) => true
         | TOTAL_FUN_SYMBOL(fnm,(sseq,sn)) =>
             (sn mem woss) andalso (Finset.is_subset(sseq,woss))
         | PARTIAL_FUN_SYMBOL(fnm,(sseq,sn)) =>
             (sn mem woss) andalso (Finset.is_subset(sseq,woss))
         | PRED_SYMBOL(fnm,sseq) => (Finset.is_subset(sseq,woss))) wosyml
  in
    if Finset.is_subset(syml,sigsyms) then generated_signat(cogsyms,cur_env,an)
    else raise (STAT_EXCEPTION 
            ("The following hidden symbols are not in the signature:\n"
              ^print_symbol_set(Finset.remove_set(sigsyms,syml))))
  end
;

(* Inducing a morphism from a raw symbol map and a source signature *)
(* raw_symbol_map * LocalEnv.local_env * AS.ANNO list ->
      LocalEnv.local_env * AS.ANNO list * GlobalEnv.morphism *)
(* p. 88 *)
fun induced_from_morphism(rsmap,cur_env,an) =
  (* Preparation - construction of mapping relevant symbols to raw symbols *) 
  let val sgsym = sig_symbols(cur_env)
  val ssm1 = remove_dups (flat (map (fn (rsym1,rsym2) => 
                     map (fn sym => (sym,rsym2))
                         (filter (fn sym => matches(sym,rsym1)) sgsym))rsmap)) 
  (* Overloaded symbols should be mapped similarly *)
  val ovrrel = overloading_relation(cur_env)
  val ssmovr = flat (map (fn (sym,rsym) => 
    let val irsym = IMPLICIT_RAW_SYMBOL(raw_symbol_name rsym)
    val ovrsyms = snd (if_none (find_first (fn (sym1,syms) => sym1=sym) ovrrel) 
                               (sym,[]))
    in map (fn sym1 => (sym1,irsym)) ovrsyms
    end) ssm1)
  val ssm = remove_dups (ssmovr@ssm1)  (* hard-coded trick: maps generated
                                         from overloading should be
                                         considered last, so they are at the
                                         beginning of the map *)
  (* "sort" part of morphism *)
  val mrph1 = foldl (fn ((smap,fmap,pmap),(sym,rsym)) =>
	case sym of
	  SORT_SYMBOL(sn) => (let val sn1 = case rsym of
	      IMPLICIT_RAW_SYMBOL(snn) => snn
            | SORT_RAW_SYMBOL(snn) => snn
	    | SYMBOL_RAW_SYMBOL(SORT_SYMBOL(snn)) => snn
	    | _ => raise (STAT_EXCEPTION 
			 (print_symbol sym^"mapped to non-sort")) in
            ((Symtab_id.update_new((sn,sn1),smap),fmap,pmap) 
             handle DUP => 
               raise (STAT_EXCEPTION (print_symbol sym^"mapped illegaly")))
            end) 
        | _ => (smap,fmap,pmap))
     ((Symtab_id.empty,Symtab_id.empty,Symtab_id.empty),ssm)
  (* identities on sorts - may be ommitted now *)
  val mrph2 = mrph1
  (* Function mapping sort names according to the morphism *)
  val smapfun = (let val (smap,_,_) = mrph2 in
         (fn sn => if_none (Symtab_id.lookup(smap,sn)) sn)
         end)
  (* subsorting relations - now we start to construct the target signature *)
  val mrph2_5 = 
    let val (svold,_,_,_) = cur_env
    val ssold = Symtab_id.dest(svold)
    val ssnew = map (fn (sn,sl) => (smapfun sn,map smapfun sl)) ssold
    val svnew = foldl (fn (svi,(sn,sl)) => 
		Symtab_id.update((sn,
                  remove_dups(sl@(if_none (Symtab_id.lookup(svi,sn)) []))),
                  svi))
                (Symtab_id.empty,ssnew)
    in
      (mrph2,(svnew,Symtab_sid.empty,Symtab_id.empty,Symtab_id.empty))
    end
  (* "total operation" part of morphism *)
  val mrph3 = 
    foldl (fn (((smap,fmap,pmap),(sv,vv,fv,pv)),(sym,rsym)) =>
      case sym of
        TOTAL_FUN_SYMBOL(fnm,(sseq,sn)) =>
         (let val (sseq1,sn1) = (map smapfun sseq,smapfun sn)
          val (fn2,(sseq2,sn2)) = case rsym of
              SYMBOL_RAW_SYMBOL(TOTAL_FUN_SYMBOL(fnn,fprofn)) => (fnn,fprofn)
            | FUN_RAW_SYMBOL(fnn) => (fnn,(sseq1,sn1))
            | IMPLICIT_RAW_SYMBOL(fnn) => (fnn,(sseq1,sn1))
            | _ => raise (STAT_EXCEPTION(
                            print_symbol sym^" mapped illegally (1)"))
          in
          if not((sseq1,sn1)=(sseq2,sn2)) then
             raise (STAT_EXCEPTION(print_symbol sym^" mapped to symbol of wrong type")) 
          else
             let val fntype = total_op_type(sorts(sseq),sn)
             val newtype = total_op_type(sorts(sseq2),sn2)
             val fnmap = if_none (Symtab_id.lookup(fmap,fnm)) []
             val fnmnew = (case   
                (find_first (fn (fntype1,_,_) => fntype1=fntype) fnmap) 
              of None => (fntype,fn2,true)::fnmap
               | Some(_,fn3,_) => if (fn2=fn3) then fnmap
                                else raise(STAT_EXCEPTION(
                                       print_symbol sym^" mapped illegally (2)")))
             val fvset = if_none (Symtab_id.lookup(fv,fn2)) []
             val fvnew = if newtype mem fvset then fvset else (newtype::fvset)
             in
                ((smap,Symtab_id.update((fnm,fnmnew),fmap),pmap),
                (sv,vv,Symtab_id.update((fn2,fvnew),fv),pv))
             end
           end)
      | _ => ((smap,fmap,pmap),(sv,vv,fv,pv)))
     (mrph2_5,ssm)
  (* "partial operation" part of morphism *)
  val mrph4 = 
    foldl (fn (((smap,fmap,pmap),(sv,vv,fv,pv)),(sym,rsym)) =>
      case sym of
        PARTIAL_FUN_SYMBOL(fnm,(sseq,sn)) => 
          let val (sseq1,sn1) = (map smapfun sseq,smapfun sn)
          val (fn2,(sseq2,sn2),reqtot,reqpart) = case rsym of 
             SYMBOL_RAW_SYMBOL(TOTAL_FUN_SYMBOL(fnn,fprofn)) => 
                   (fnn,fprofn,true,false)
           | SYMBOL_RAW_SYMBOL(PARTIAL_FUN_SYMBOL(fnn,fprofn)) => 
                   (fnn,fprofn,false,true)
           | FUN_RAW_SYMBOL(fnn) => (fnn,(sseq1,sn1),false,false)
           | IMPLICIT_RAW_SYMBOL(fnn) => (fnn,(sseq1,sn1),false,false)
           | _ => raise (STAT_EXCEPTION(
                            print_symbol sym^" mapped illegally (3)"))
          in if not((sseq1,sn1)=(sseq2,sn2)) then
            raise (STAT_EXCEPTION(print_symbol sym^" mapped to symbol of wrong type")) 
          else
            let val fntype = partial_op_type(sorts(sseq),sn)
            val fnmap = if_none (Symtab_id.lookup(fmap,fnm)) []
            val fvset = if_none (Symtab_id.lookup(fv,fn2)) [] 
            val totality = (total_op_type(sorts(sseq2),sn2)) mem fvset
            val fnmnew = (case   
                (find_first (fn (fntype1,_,_) => fntype1=fntype) fnmap) 
              of None => (fntype,fn2,totality orelse reqtot)::fnmap
               | Some(_,fn3,_) => if (fn2=fn3) then fnmap
                                else raise(STAT_EXCEPTION(
                                       print_symbol sym^" mapped illegally (4)")))
            val fvnew = 
              if (total_op_type(sorts(sseq2),sn2)) mem fvset then
                if reqpart then 
                  raise (STAT_EXCEPTION(print_symbol sym^" mapped illegally (5)"))       
                else fvset
              else if (partial_op_type(sorts(sseq2),sn2)) mem fvset then
                if reqtot then
                  raise (STAT_EXCEPTION(print_symbol sym^" mapped illegally (6)"))       
                else fvset
              else 
                if reqtot then
                     (total_op_type(sorts(sseq2),sn2))::fvset
                else
                     (partial_op_type(sorts(sseq2),sn2))::fvset
            in
               ((smap,Symtab_id.update((fnm,fnmnew),fmap),pmap),
                (sv,vv,Symtab_id.update((fn2,fvnew),fv),pv))
             end
           end
      | _ => ((smap,fmap,pmap),(sv,vv,fv,pv)))
     (mrph3,ssm)
  (* "predicate" part of the morphism *)
  val ((smap5,fmap5,pmap5),sigma5) = 
    foldl (fn (((smap,fmap,pmap),(sv,vv,fv,pv)),(sym,rsym)) =>
      case sym of
        PRED_SYMBOL(pnm,sseq) =>
         (let val sseq1 = map smapfun sseq
          val (pn2,sseq2) = case rsym of
              SYMBOL_RAW_SYMBOL(PRED_SYMBOL(pnn,pprofn)) => (pnn,pprofn)
            | PRED_RAW_SYMBOL(pnn) => (pnn,sseq1)
            | IMPLICIT_RAW_SYMBOL(pnn) => (pnn,sseq1)
            | _ => raise (STAT_EXCEPTION(
                            print_symbol sym^" mapped illegally (7)"))
          in
          if not (sseq1=sseq2) then
             raise (STAT_EXCEPTION(print_symbol sym^" mapped to symbol of wrong type")) 
          else
             let val ptype = pred_type(sorts(sseq))
             val newtype = pred_type(sorts(sseq2))
             val prmap = if_none (Symtab_id.lookup(pmap,pnm)) []
             val prmnew = (case   
                (find_first (fn (ptype1,_) => ptype1=ptype) prmap) 
              of None => (ptype,pn2)::prmap
               | Some(_,pn3) => if (pn2=pn3) then prmap
                                  else raise(STAT_EXCEPTION(
                                     print_symbol sym^" mapped illegally (8)")))
             val pvset = if_none (Symtab_id.lookup(pv,pn2)) []
             val pvnew = if newtype mem pvset then pvset else (newtype::pvset)
             in
               ((smap,fmap,Symtab_id.update((pnm,prmnew),pmap)),
                (sv,vv,fv,Symtab_id.update((pn2,pvnew),pv)))
             end
           end)
      | _ => ((smap,fmap,pmap),(sv,vv,fv,pv)))
     (mrph4,ssm)
  (* Identities on operations and predicates *)
  val unmsym = filter (fn sym => case sym of
      SORT_SYMBOL(_) => false
    | TOTAL_FUN_SYMBOL(fnm,(sseq,sn)) => 
        not (exists' (fn (fntype,_,_) => fntype = total_op_type(sorts(sseq),sn))
                     (if_none (Symtab_id.lookup(fmap5,fnm)) []))
    | PARTIAL_FUN_SYMBOL(fnm,(sseq,sn)) =>
        not (exists' (fn (fntype,_,_) => fntype = partial_op_type(sorts(sseq),sn))
                     (if_none (Symtab_id.lookup(fmap5,fnm)) []))
    | PRED_SYMBOL(pnm,sseq) =>
        not (exists' (fn (ptype,_) => ptype = pred_type(sorts(sseq)))
                     (if_none (Symtab_id.lookup(pmap5,pnm)) []))) sgsym
  val (mrph,sigma) = foldl (fn ((morph,(sv,vv,fv,pv)),sym) =>
    case sym of
      SORT_SYMBOL(_) => (morph,(sv,vv,fv,pv))
    | TOTAL_FUN_SYMBOL(fnm,(sseq,sn)) =>
        let val (sseq1,sn1) = (map smapfun sseq,smapfun sn)
        val fvset = if_none (Symtab_id.lookup(fv,fnm)) []
        val fvnew = if (total_op_type(sorts(sseq1),sn1)) mem fvset then fvset
                    else ((total_op_type(sorts(sseq1),sn1))::fvset)
        in
          (morph,(sv,vv,Symtab_id.update((fnm,fvnew),fv),pv))
        end
    | PARTIAL_FUN_SYMBOL(fnm,(sseq,sn)) =>
        let val (sseq1,sn1) = (map smapfun sseq,smapfun sn)
        val fvset = if_none (Symtab_id.lookup(fv,fnm)) []
        val fvnew = if ((total_op_type(sorts(sseq1),sn1)) mem fvset) orelse
                       ((partial_op_type(sorts(sseq1),sn1)) mem fvset) 
                    then fvset
                    else ((partial_op_type(sorts(sseq1),sn1))::fvset)
        in
          (morph,(sv,vv,Symtab_id.update((fnm,fvnew),fv),pv))
        end
    | PRED_SYMBOL(pnm,sseq) =>
        let val sseq1 = map smapfun sseq
        val pvset = if_none (Symtab_id.lookup(pv,pnm)) []
        val pvnew = if (pred_type(sorts(sseq1))) mem pvset then pvset
                    else ((pred_type(sorts(sseq1)))::pvset)
        in
          (morph,(sv,vv,fv,Symtab_id.update((pnm,pvnew),pv)))
        end)
    (((smap5,fmap5,pmap5),sigma5),unmsym)
  val new_an = translate_annos(mrph,an)
  (* Overloading preservation checking. This is redundant now. *)
(*  val ovr_ok = forall' (fn (sym1,syml) =>
     let val sym1n = symbol_via_morphism(mrph,sym1) in
       forall' (fn sym2 => symbols_overload(sigma,sym1n,
                                            symbol_via_morphism(mrph,sym2)))
               syml
     end) (overloading_relation(cur_env)) *)
  in
   (* if ovr_ok then *) (sigma,new_an,mrph) (* else 
      raise(STAT_EXCEPTION("Symbol map does not preserve overloading")) *)  
  end
;

(* The following 3 auxiliary functions I took from arch/arch_analysis. BK
*)

(* sub_sig -> e_symbol list *)
(* p. 131 *)
local
  open Symbols;
in
fun e_sig_symbols((ssenv,venv,fenv,penv):sub_sig):e_symbol list =
  let val sigsyms = sig_symbols(ssenv,venv,fenv,penv)
  val esyms1 = map (fn x => SYM_ESYMBOL(x)) sigsyms
  val senvl = Symtab_id.dest(ssenv)
  val esyms =
    foldl (fn (esymsi,(s,ss)) =>
      foldl (fn (esymsii,s1) => EMB_ESYMBOL(s,s1)::esymsii)
            (esymsi,ss))
      (esyms1,senvl) in
  esyms
  end
end;

fun is_subset([],_) = true
  | is_subset(h::t,l) = h mem l andalso is_subset(t,l)
;

(* sub_sig * sub_sig -> bool *)
fun is_subsig(sig1,sig2) =
  is_subset(e_sig_symbols(sig1),e_sig_symbols(sig2))
;

(*************************************************************)
(* Inducing a morphism with known source and target. *)
(* Expotential pessimistic time *)
(*************************************************************)

(* Computing core of a symbol map *)
(* symbol_map -> symbol_map *)
(* Note S-10, p. 15, with an error corrected: only names at second position *)
fun symbol_map_core sm = 
  map (fn (sym1,sym2) => (sym1,symbol_name sym2))
      (filter (fn (sym1,sym2) => not(symbol_name sym1 = symbol_name sym2)) sm)
;

(* Can a symbol be mapped to another symbol, taking their kind and arity
   into account? *)
(* symbol * symbol -> bool *)
fun can_be_mapped(sym1,sym2) = case sym1 of
    SORT_SYMBOL(_) => (case sym2 of
        SORT_SYMBOL(_) => true
      | _ => false)
  | TOTAL_FUN_SYMBOL(_,(sseq1,_)) => (case sym2 of
        TOTAL_FUN_SYMBOL(_,(sseq2,_)) => length(sseq1)=length(sseq2) 
      | _ => false)
  | PARTIAL_FUN_SYMBOL(_,(sseq1,_)) => (case sym2 of
        PARTIAL_FUN_SYMBOL(_,(sseq2,_)) => length(sseq1)=length(sseq2)
      | TOTAL_FUN_SYMBOL(_,(sseq2,_)) => length(sseq1)=length(sseq2)
      | _ => false)
  | PRED_SYMBOL(_,sseq1) => (case sym2 of
        PRED_SYMBOL(_,sseq2) => length(sseq1)=length(sseq2)
      | _ => false)
;

(* Does addition of one symbol pair to a map not loose overloading? *)
(* local_env * local_env * symbol_map * (symbol*symbol) -> bool *)
fun map_addition_preserves_overloading(sg1,sg2,akmap,(sym1,sym2)) =
  forall' (fn (sym3,sym4) =>
      if symbols_overload(sg1,sym1,sym3) then 
         symbols_overload(sg2,sym2,sym4) else true) akmap
;

(* What a symbol is mapped to by a map? *)
(* symbol_map * symbol -> symbol *)
fun symbol_map_value(akmap,sym,defvalue) = 
  snd (if_none (find_first (fn (sym1,sym2) => sym1=sym) akmap)
               (sym,defvalue))
;

(* Can we add one symbol pair to a map, hence getting a correct map? *)
(* local_env * local_env * symbol_map * (symbol * symbol) -> bool *)
fun map_addition_is_possible(sg1,sg2,akmap,(sym1,sym2)) =
  if sym1 mem (sig_symbols sg1) andalso
     sym2 mem (sig_symbols sg2) andalso
     map_addition_preserves_overloading(sg1,sg2,akmap,(sym1,sym2)) andalso
     symbol_map_value(akmap,sym1,sym2)=sym2
  then
    case sym1 of
      SORT_SYMBOL(s1) => (case sym2 of
          SORT_SYMBOL(s2) =>    (* Preserving sort order *)
            forall' (fn (SORT_SYMBOL(s3),SORT_SYMBOL(s4)) => 
                      (if sort_is_leq(sg1,s1,s3) then sort_is_leq(sg2,s2,s4)
                       else true) andalso
                      (if sort_is_leq(sg1,s3,s1) then sort_is_leq(sg2,s4,s2)
                       else true)
                      | _ => true) akmap
        | _ => false)
    | TOTAL_FUN_SYMBOL(f1,(sseq1,s1)) => (case sym2 of
          TOTAL_FUN_SYMBOL(f2,(sseq2,s2)) => 
            map_addition_is_possible(sg1,sg2,akmap,
                                     (SORT_SYMBOL(s1),SORT_SYMBOL(s2))) 
            andalso
            sortseq_map_addition_is_possible(sg1,sg2,
              ((SORT_SYMBOL(s1),SORT_SYMBOL(s2))::akmap),sseq1,sseq2)
        | _ => false)
    | PARTIAL_FUN_SYMBOL(f1,(sseq1,s1)) => 
      (let
	   fun get_value_tup symb2 = 
	       (case symb2 of
		    TOTAL_FUN_SYMBOL(fnm,(sseqn,sn)) => (fnm,(sseqn,sn))
		  | PARTIAL_FUN_SYMBOL(fnm,(sseqn,sn)) => (fnm,(sseqn,sn))
		  | _ => raise (STAT_EXCEPTION 
				    (print_symbol sym1^
				     " mapped illegally (9)")))
	   fun map_addition_possiblities (f2,(sseq2,s2)) =
	       map_addition_is_possible(sg1,sg2,akmap,
					(SORT_SYMBOL(s1),SORT_SYMBOL(s2))) 
               andalso
               sortseq_map_addition_is_possible(sg1,sg2,
						((SORT_SYMBOL(s1),
						  SORT_SYMBOL(s2))::akmap),
						sseq1,sseq2)    
       in
	   (case sym2 of
		TOTAL_FUN_SYMBOL(_) =>  
		map_addition_possiblities (get_value_tup sym2)
              | PARTIAL_FUN_SYMBOL(_) =>
		map_addition_possiblities (get_value_tup sym2)
(*		(let val (f2,(sseq2,s2)) = get_value_tup sym2 in
		     map_addition_possiblities (f2,(sseq2,s2))
		 end)*)
              | _ => false)
       end)
    | PRED_SYMBOL(f1,sseq1) => (case sym2 of
          PRED_SYMBOL(f2,sseq2) =>
            sortseq_map_addition_is_possible(sg1,sg2,akmap,sseq1,sseq2)
        | _ => false)
  else false
  
and 

(* local_env * local_env * symbol_map * sort list * sort list *)
sortseq_map_addition_is_possible(sg1,sg2,akmap,sseq1,sseq2) =
  let val symseq1 = map (fn s => SORT_SYMBOL(s)) sseq1
  val symseq2 = map (fn s => SORT_SYMBOL(s)) sseq2
  val (akmres,ok) =
    foldl (fn ((akmapi,oki),(sym1,sym2)) => if oki then
       (((sym1,sym2)::akmapi),
        map_addition_is_possible(sg1,sg2,akmapi,(sym1,sym2)))
       else (akmapi,false)) 
    ((akmap,true),zip(symseq1,symseq2)) in ok
  end 
;

(* Add a single map to a symbol map. Add appropriate sort maps, too. *)
fun add_map(akmap,(sym1,sym2)) =
let val akmap1 = (sym1,sym2)::akmap in 
  case sym1 of
      TOTAL_FUN_SYMBOL(_,(sseq1,s1)) => (case sym2 of
          TOTAL_FUN_SYMBOL(_,(sseq2,s2)) =>
            add_sortseq_map(akmap1,s1::sseq1,s2::sseq2)
        | _ => akmap1)
    | PARTIAL_FUN_SYMBOL(_,(sseq1,s1)) => (case sym2 of
          TOTAL_FUN_SYMBOL(_,(sseq2,s2)) =>
            add_sortseq_map(akmap1,s1::sseq1,s2::sseq2)
        | PARTIAL_FUN_SYMBOL(_,(sseq2,s2)) =>
            add_sortseq_map(akmap1,s1::sseq1,s2::sseq2)
        | _ => akmap1)
    | PRED_SYMBOL(_,sseq1) => (case sym2 of
          PRED_SYMBOL(_,sseq2) => add_sortseq_map(akmap1,sseq1,sseq2)
        | _ => akmap1)
    | _ => akmap1
  end

and

add_sortseq_map(akmap,sseq1,sseq2) =
  let val symseq1 = map (fn s => SORT_SYMBOL(s)) sseq1 
      val symseq2 = map (fn s => SORT_SYMBOL(s)) sseq2 in
    remove_dups((zip(symseq1,symseq2))@akmap)
  end
;

type map_core = (symbol * AS.ID) list;

(* Some simple datatypes to flow information through the depth search *)
(*datatype map_or_not = 
    NO_MAP 
  | SOME_MAP of symbol_map
;*)

datatype map_and_core = 
    NO_MAP (* no map at all *)
  | NO_MAPS of map_core 
  | MANY_MAPS of map_core
  | ONE_MAP of (map_core * symbol_map)
;

fun get_core NO_MAP = None
  | get_core (NO_MAPS(x)) = Some(x)
  | get_core (MANY_MAPS(x)) = Some(x)
  | get_core (ONE_MAP(x,_)) = Some(x)
;

(* We have found some maps with (possibly) different cores. What does it
   mean? *)
(* Important observation: always m<>m1. *)
fun combine_maps_and_cores(m,m1) = case m of
    NO_MAP => m1
  | NO_MAPS(mc) => (case m1 of
      NO_MAPS(mc1) => NO_MAPS(Finset.intersect(mc,mc1))
    | MANY_MAPS(mc1) => if Finset.is_subset(mc1,mc) then
        MANY_MAPS(mc1) else NO_MAPS(Finset.intersect(mc,mc1))
    | ONE_MAP(mc1,mm1) => if Finset.is_subset(mc1,mc) then
        ONE_MAP(mc1,mm1) else NO_MAPS(Finset.intersect(mc,mc1))
    | _ => combine_maps_and_cores(m1,m))
  | MANY_MAPS(mc) => (case m1 of
      MANY_MAPS(mc1) => if Finset.is_subset(mc,mc1) then
        MANY_MAPS(mc) else if Finset.is_subset(mc1,mc) then
        MANY_MAPS(mc1) else NO_MAPS(Finset.intersect(mc,mc1))
    | ONE_MAP(mc1,mm1) => if Finset.is_subset(mc,mc1) then
        MANY_MAPS(mc) else if Finset.is_subset(mc1,mc) then
        ONE_MAP(mc1,mm1) else NO_MAPS(Finset.intersect(mc,mc1))
    | _ => combine_maps_and_cores(m1,m))
  | ONE_MAP(mc,mm) => (case m1 of
      ONE_MAP(mc1,mm1) => if Finset.is_equal(mc,mc1) then
        MANY_MAPS(mc) else if Finset.is_subset(mc,mc1) then
        ONE_MAP(mc,mm) else if Finset.is_subset(mc1,mc) then
        ONE_MAP(mc1,mm1) else NO_MAPS(Finset.intersect(mc,mc1))
   | _ => combine_maps_and_cores(m1,m))
;

(* The depth search for unique least map. *)
(* local_env * local_env * symbol_map * symbol_map * 
   (symbol * symbol list) list -> map_and_core *)
(* akmap - actually constructed (part of a) map *)
(* akcore - the least actually known core of a possible map *)
(* posmap - still possible mappings of yet unmapped symbols *)
fun unique_mapping(sg1,sg2,akmap,akcore,posmap) = 
  let val akmapcore = (symbol_map_core akmap)
  val akcore1 = if_none akcore akmapcore
  in
  if Finset.is_subset(akcore1,akmapcore) andalso
     not(Finset.is_subset(akmapcore,akcore1)) then NO_MAP
  else 
    case posmap of
      [] => ONE_MAP(symbol_map_core akmap,akmap)
                             (* claim: akmap is always defined on all
				symbols. *)
    | ((sym1,sym2set)::t) => foldl 
	 (fn ((mapsofar:map_and_core),sym2) => 
	    if map_addition_is_possible(sg1,sg2,akmap,(sym1,sym2)) then
              combine_maps_and_cores(mapsofar,
                 unique_mapping(sg1,sg2,add_map(akmap,(sym1,sym2)),
                                get_core mapsofar,t))
             else mapsofar)
	 ((case akcore of None => NO_MAP
	 	        | Some(x) => NO_MAPS(x)),sym2set)
  end
;

(* Some heuristics: the search goes faster if we search sorts first and if
   we start from trying as many identities as possible. *) 
fun identities_first(sym,syms) = 
  let val sname = symbol_name sym in 
    (filter (fn sym1 => sname = symbol_name sym1) syms)@
    (filter (fn sym1 => not(sname = symbol_name sym1)) syms)
  end
;

fun sorts_first(posmap) =
  (filter (fn (SORT_SYMBOL(_),_) => true | _ => false) posmap)@
  (filter (fn (SORT_SYMBOL(_),_) => false | _ => true) posmap)
;

(* Constructing a morphism from a symmap. *)
(* Copied from morphism.sml; the structure of modules should be improved. *)
fun morphism_from_symmap(smap) = 
  foldl (fn ((smapi,fmapi,pmapi),(sym1,sym2)) => case sym1 of
      SORT_SYMBOL(s1) => (case sym2 of
          SORT_SYMBOL(s2) => 
            (Symtab_id.update_new((s1,s2),smapi),fmapi,pmapi)
        | _ => raise INTERNAL_ERROR)
    | TOTAL_FUN_SYMBOL(on1,(ss,s)) => (case sym2 of
          TOTAL_FUN_SYMBOL(on2,_) => (smapi,Symtab_id.update(
               (on1,(total_op_type(sorts(ss),s),on2,true)::
                  (case Symtab_id.lookup(fmapi,on1) of
                     None => [] | Some(x) => x)),fmapi),pmapi)
        | _ => raise INTERNAL_ERROR)
    | PARTIAL_FUN_SYMBOL(on1,(ss,s)) => (case sym2 of
          PARTIAL_FUN_SYMBOL(on2,_) => (smapi,Symtab_id.update(
               (on1,(partial_op_type(sorts(ss),s),on2,false)::
                  (case Symtab_id.lookup(fmapi,on1) of
                     None => [] | Some(x) => x)),fmapi),pmapi)
        | TOTAL_FUN_SYMBOL(on2,_) => (smapi,Symtab_id.update(
               (on1,(partial_op_type(sorts(ss),s),on2,true)::
                  (case Symtab_id.lookup(fmapi,on1) of
                     None => [] | Some(x) => x)),fmapi),pmapi)
        | _ => raise INTERNAL_ERROR)
    | PRED_SYMBOL(pn1,ss) => (case sym2 of
          PRED_SYMBOL(pn2,_) => (smapi,fmapi,Symtab_id.update(
               (pn1,(pred_type(sorts(ss)),pn2)::
                  (case Symtab_id.lookup(pmapi,pn1) of
                     None => [] | Some(x) => x)),pmapi))
        | _ => raise INTERNAL_ERROR))
   ((Symtab_id.empty,Symtab_id.empty,Symtab_id.empty),smap)
;


(* Now the main function: try to induce a morphism from a raw symbol map
   knowing its source and target. *)
(* Essentially a wrapper to the depth search above. *)
(* Does not employ Theorem 6 from BK Master's *)
(* raw_symbol_map * local_env * local_env -> GlobalEnv.morphism *)
fun try_to_induce(rsm,sig1,sig2) = 
  let val sg1syms = sig_symbols(sig1)
  val sg2syms = sig_symbols(sig2) in
  (* Does the domain of rsm match sg1syms? *)
  if exists' (fn (rsym,_) =>
     forall' (fn sym => not(matches(sym,rsym))) sg1syms) rsm
  then raise (STAT_EXCEPTION ("Wrong raw symbol map domain"))
  else
  (* All possible mappings, not caring about rsm yet *)
  let val posmap2 = 
    (map (fn sym1 => (sym1,filter (fn sym2 => can_be_mapped(sym1,sym2))  
			           sg2syms))
         sg1syms)
  (* Identities and sorts first! *)
  val posmap1 = map (fn (sym1,sym2s) => (sym1,identities_first(sym1,sym2s)))
	            (sorts_first(posmap2))
  (* Now exclude mappings clashing with rsm *)
  val posmap = map (fn (sym1,sym2s) =>
    (sym1,filter (fn sym2 =>
            forall' (fn (rsym1,rsym2) => if matches(sym1,rsym1) then
                                            matches(sym2,rsym2) else true)
                    rsm)
            sym2s))
    posmap1
  in
  case unique_mapping(sig1,sig2,[],None,posmap) of
    NO_MAP => raise (STAT_EXCEPTION 
       ("Error in implicit fitting: no possible signature morphism"))
  | NO_MAPS(_) => raise (STAT_EXCEPTION 
       ("Error in implicit fitting: no least map"))      
  | MANY_MAPS(_) => raise (STAT_EXCEPTION 
            ("Error in implicit fitting: many least maps"))
  | ONE_MAP(_,symmap) => morphism_from_symmap(symmap)
  end end
;

(* And now we also use Theorem 6 from BK's Master's *)
fun induced_from_to_morphism (rsm,sig1,sig2) = 
  let val (trysig,_,trymor) = induced_from_morphism(rsm,sig1,[]) in
    if (is_subsig(trysig,sig2)) then trymor
    else try_to_induce(rsm,sig1,sig2)
  end
;

end;
