(* ----------------------------------------------------------------------- *)
(* xml_convert.sml                        Pascal Schmidt <pharao90@tzi.de> *)
(* ----------------------------------------------------------------------- *)
(* conversion from AS tree to XML datatype                                 *)
(* ----------------------------------------------------------------------- *)

(* $Id: xml_convert.sml,v 1.1 2004/02/13 11:29:20 7till Exp $ *)

(* todo

Erweiterung von extension_env korrekt umsetzen
   
Restliche Datentypen auch automatisch erzeugen
  erstmal fxp das Sharing auffalten lassen
  spter Sharing erst bei bersetzung in AS auffalten, dazu Programm automatisch aus DTD erzeugen

*)

structure xml_convert
=
struct

  open AS GlobalEnv LocalEnv Symbols ArchTypes xml_types

  (* --------------------------------------------------------------------- *)
  (* helper functions                                                      *)
  (* --------------------------------------------------------------------- *)

  fun escape ("<"::t)  = (Utils.explode "&lt;")   @ (escape t)
    | escape (">"::t)  = (Utils.explode "&gt;")   @ (escape t)
    | escape ("&"::t)  = (Utils.explode "&amp;")  @ (escape t)
    | escape ("\""::t) = (Utils.explode "&quot;") @ (escape t)
    | escape ("'"::t)  = (Utils.explode "&apos;") @ (escape t)
    | escape ("%"::t)  = (Utils.explode "&#x25;") @ (escape t)
    | escape (x::t)    = x :: (escape t)
    | escape []        = []

  fun esc x = (Utils.implode o escape o Utils.explode) x

  (* --------------------------------------------------------------------- *)
  (* library definitions                                                   *)
  (* --------------------------------------------------------------------- *)
  
  (* ********************************************************************* *)
  (* NOTE: cannot put region before initial casl tag, so x_lib_defn and    *)
  (*       x_lib_defn2 discard the region element                          *)
  (* ********************************************************************* *)
  
  fun x_lib_defn (lib_defn (n,i,a)) = tag ("casl",[],[
                                      tag ("lib-defn",[],[(x_lib_name n)] @
                                      ((map x_lib_item) i) @ ((map x_anno) a))
                                      ])
     | x_lib_defn (pos_LIB_DEFN (r,x)) = x_lib_defn x

  and x_lib_defn2 (lib_defn (n,i,a),e) = tag ("casl",[],[
                                         tag ("lib-defn",[],[(x_lib_name n)] @
                                         ((map x_lib_item) i) @ ((map x_anno)
                                         a) @ [x_global_env2 e])])
    | x_lib_defn2 (pos_LIB_DEFN (r,x),e) = x_lib_defn2 (x,e)

  and x_lib_name (x) = tag ("lib-name",[],[x_lib_name1 x])
                                      
  and x_lib_name1 (lib (i)) = tag ("lib-id",[],[x_lib_id i])
    | x_lib_name1 (versioned_lib (i,v)) = tag ("lib-version",[],[
                                          tag ("lib-id",[],[x_lib_id i]),
                                          x_version v])
    | x_lib_name1 (pos_LIB_NAME (r,x)) = tag ("pos-lib-name",[],[x_region r,
                                              x_lib_name x])

  and x_lib_id (url (u)) = tag ("direct-link",[],[x_url u])
    | x_lib_id (path_name (p)) = tag ("indirect-link",[],[x_path p])
    | x_lib_id (pos_LIB_ID (r,x)) = tag ("pos-lib-id",[],[x_region r,
                                         x_lib_id x])

  and x_url u = tag ("url",[],[pcdata (esc u)])

  and x_srcpos (i,j) = tag ("srcpos",[],[x_int i,x_int j])

  and x_region (a,b) = tag ("region",[],[x_srcpos a,x_srcpos b])

  and x_int i = tag ("int",[],[pcdata (esc (Int.toString i))])
  
  and x_path p = tag ("path",[],[pcdata (esc p)])

  and x_version (version (v)) = tag ("version",[],(map x_number) v)
    | x_version (pos_VERSION (r,x)) = tag ("pos-version",[],[x_region r,
                                           x_version x])
  
  and x_number n = tag ("number",[],[pcdata (esc n)])

  and x_lib_item (x) = tag ("lib-item",[],[x_lib_item1 x])

  and x_lib_item1 (spec_defn (n,g,s,a)) = tag ("spec-defn",[],[x_spec_name n,
                                          x_genericity g,x_l_spec s] @
                                          ((map x_anno) a))
    | x_lib_item1 (view_defn (n,g,t,l,a)) = tag ("view-defn",[],[x_view_name
                                            n,x_genericity g,x_view_type t] @
                                            ((map x_symb_map_items) l) @
                                            ((map x_anno) a))
    | x_lib_item1 (arch_spec_defn (n,s,a)) = tag ("arch-spec-defn",[],
                                             [x_arch_spec_name n,
                                             x_l_arch_spec s] @
                                             ((map x_anno) a))
    | x_lib_item1 (unit_spec_defn (n,s,a)) = tag ("unit-spec-defn",[],
                                             [x_spec_name n,x_unit_spec s] @
                                             ((map x_anno) a))
    | x_lib_item1 (download_items (n,l,a)) = tag ("download-items",[],
                                             [x_lib_name n] @
                                             ((map x_item_name_or_map) l) @
                                             ((map x_anno) a))
    | x_lib_item1 (pos_LIB_ITEM (r,x)) = tag ("pos-lib-item",[],[x_region r,
                                              x_lib_item x])
  
  and x_item_name_or_map (x) = tag ("item-name-or-map",[],[x_item_nom x])
  
  and x_item_nom (item_name (n)) = (x_item_name n)
    | x_item_nom (item_name_map (a,b)) = (x_item_name_map (a,b))
    | x_item_nom (pos_ITEM_NAME_OR_MAP (r,x)) = tag ("pos-item-name-or-map",
                              [],[x_region r,x_item_name_or_map x])
    
  and x_item_name n = tag ("item-name",[],[x_simple_id n])
  
  and x_item_name_map (a,b) = tag ("item-name-map",[],[x_item_name a,
                              x_item_name b])

  (* --------------------------------------------------------------------- *)
  (* structured specifications                                             *)
  (* --------------------------------------------------------------------- *)
  
  and x_spec_name n = tag ("spec-name",[],[x_simple_id n])
  
  and x_simple_id (n,_) = tag ("simple-id",[],[x_words n])
  
  and x_words n = tag ("words",[],[pcdata (esc n)])
  
  and x_genericity (genericity (p,i)) = tag ("genericity",[],[x_params p,
                                        x_imports i])
    | x_genericity (pos_GENERICITY (r,x)) = tag ("pos-genericity",[],
                                            [x_region r,x_genericity x])

  and x_params (params (l)) = tag ("params",[],(map x_l_spec) l)
    | x_params (pos_PARAMS (r,x)) = tag ("pos-params",[],[x_region r,
                                         x_params x])
  
  and x_imports (imports (l)) = tag ("imports",[],(map x_l_spec) l)
    | x_imports (pos_IMPORTS (r,x)) = tag ("pos-imports",[],[x_region r,
                                           x_imports x])
  
  and x_l_spec (s,a) = tag ("anno-spec",[],(x_spec s)::((map x_anno) a))
  
  and x_spec (x) = tag ("spec",[],[x_spec1 x])
  
  and x_spec1 (basic (b)) = tag ("basic",[],[x_basic_spec b])
    | x_spec1 (translation (s,r,a)) = tag ("translation",[],[x_l_spec s,
                                      x_renaming r] @ ((map x_anno) a))
    | x_spec1 (reduction (s,r,a)) = tag ("reduction",[],[x_l_spec s,
                                    x_restriction r] @ ((map x_anno) a))
    | x_spec1 (union_spec (l)) = tag ("union",[],x_union l)
    | x_spec1 (extension (l)) = tag ("extension",[],x_union l)
    | x_spec1 (free_spec (s,a)) = tag ("free-spec",[],(x_l_spec s)::
                                 ((map x_anno) a))
    | x_spec1 (cofree_spec (s,a)) = tag ("cofree-spec",[],(x_l_spec s)::
                                 ((map x_anno) a))
    | x_spec1 (local_spec (s,a,t,b)) = tag ("local-spec",[],[x_l_spec s] @
                                       ((map x_anno) a) @ [x_l_spec t] @
                                       ((map x_anno) b))
    | x_spec1 (closed_spec (s,a)) = tag ("closed-spec",[],(x_l_spec s)::
                                    ((map x_anno) a))
    | x_spec1 (spec_inst (n,f)) = tag ("spec-inst",[],(x_spec_name n)::
                                  ((map x_fit_arg) f))
    | x_spec1 (pos_SPEC (r,b,x)) = tag ("pos-spec",[],[x_region r,
                                   x_bool_data b,x_spec x])

  and x_union l = List.concat ((map x_union_one) l)
  
  and x_union_one (s,a) = [tag ("anno-anno-spec",[],(x_l_spec s) ::
                           ((map x_anno) a))]

  and x_renaming (renaming (s)) = tag ("renaming",[],(map x_symb_map_items) s)
    | x_renaming (pos_RENAMING (r,x)) = tag ("pos-renaming",[],[x_region r,
                                             x_renaming x])
  
  and x_symb_map_items (symb_map_items (k,s)) = tag ("symb-map-items",[],
                                                (x_symb_kind k)::
                                                ((map x_symb_or_map) s))
    | x_symb_map_items (pos_SYMB_MAP_ITEMS (r,x)) = tag ("pos-symb-map-items",
                                         [],[x_region r,x_symb_map_items x])
  
  and x_symb_kind (x) = tag ("symb-kind",[],[x_symb_kind1 x])
  
  and x_symb_kind1 (implicitk) = empty_tag ("implicit",[])
    | x_symb_kind1 (sortsk)    = empty_tag ("sort-kind",[])
    | x_symb_kind1 (opsk)      = empty_tag ("op-kind",[])
    | x_symb_kind1 (predsk)    = empty_tag ("pred-kind",[])
    | x_symb_kind1 (pos_SYMB_KIND (r,x)) = tag ("pos-symb-kind",[],[x_region r,
                                                x_symb_kind x])
    
  and x_symb_or_map (x) = tag ("symb-or-map",[],[x_symb_o_map x])

  and x_symb_o_map (symb (s)) = tag ("symb",[],[x_symb s])
    | x_symb_o_map (symb_or_map (a)) = x_symb_map a
    | x_symb_o_map (pos_SYMB_OR_MAP (r,x)) = tag ("pos-symb-or-map",[],
                                                [x_region r,x_symb_or_map x])

  and x_symb_map (symb_map (a,b)) = tag ("symb-map",[],[tag ("symb",[],
                                         [x_symb a]),tag ("symb",[],
                                         [x_symb b])])
    | x_symb_map (pos_SYMB_MAP (r,x)) = tag ("pos-symb-map",[],[x_region r,
                                             x_symb_map x])
  
  and x_symb (symb_id (i)) = (x_id i)
    | x_symb (qual_id (i,t)) = tag ("qual-id",[],[x_id i,x_type t])
    | x_symb (pos_SYMB (r,x)) = tag ("pos-symb",[],[x_region r,x_symb x])
    
  and x_type (op_symb_type (t)) = tag ("type",[],[x_op_type t])
    | x_type (pred_symb_type (t)) = tag ("type",[],[x_pred_type t])
    | x_type (pos_TYPE (r,x)) = tag ("pos-type",[],[x_region r,x_type x])
    
  and x_restriction (x) = tag ("restriction",[],[x_restriction1 x])
  
  and x_restriction1 (hide (s)) = tag ("hide",[],(map x_symb_items) s)
    | x_restriction1 (reveal (s)) = tag ("reveal",[],(map x_symb_map_items) s)
    | x_restriction1 (pos_RESTRICTION (r,x)) = tag ("pos-restriction",[],
                                                 [x_region r,x_restriction x])
    
  and x_symb_list x = tag ("symb",[],[x_symb x])

  and x_symb_items (symb_items (k,s)) = tag ("symb-items",[],(x_symb_kind k)::
                                        ((map x_symb_list) s))
    | x_symb_items (pos_SYMB_ITEMS (r,x)) = tag ("pos-symb-items",[],
                                                  [x_region r,x_symb_items x])
  
  and x_fit_arg (x) = tag ("fit-arg",[],[x_fit_arg1 x])
  
  and x_fit_arg1 (fit_spec (s,l)) = tag ("fit-spec",[],(x_l_spec s)::
                                    ((map x_symb_map_items) l))
    | x_fit_arg1 (fit_view (n,l)) = tag ("fit-view",[],(x_view_name n)::
                                    ((map x_fit_arg) l))
    | x_fit_arg1 (pos_FIT_ARG (r,x)) = tag ("pos-fit-arg",[],[x_region r,
                                                              x_fit_arg x])
  
  and x_view_name n = tag ("view-name",[],[x_simple_id n])
  
  and x_view_type (view_type (a,b)) = tag ("view-type",[],[x_l_spec a,
                                      x_l_spec b])
    | x_view_type (pos_VIEW_TYPE (r,x)) = tag ("pos-view-type",[],[x_region r,
                                                               x_view_type x])

  (* --------------------------------------------------------------------- *)
  (* annotations                                                           *)
  (* --------------------------------------------------------------------- *)
  
  and x_anno (x) = tag ("anno",[],[x_anno1 x])
  
  and x_anno1 (comment (s)) = tag ("comment-anno",[],[pcdata (esc s)])
    | x_anno1 (comment_line (s)) = tag ("comment-line-anno",[],
                                        [pcdata (esc s)])
    | x_anno1 (label_anno (i)) = tag ("label-anno",[],[x_id i])
    | x_anno1 (unparsed_anno (s)) = tag ("unparsed-anno",[],[pcdata (esc s)])
    | x_anno1 (number_anno (i)) = tag ("number-anno",[],[x_id i])
    | x_anno1 (floating_anno (i,j)) = tag ("floating-anno",[],[x_id i,x_id j])
    | x_anno1 (string_anno (i,j)) = tag ("string-anno",[],[x_id i,x_id j])
    | x_anno1 (list_anno (i,j,k)) = tag ("list-anno",[],[x_id i,x_id j,x_id k])
    | x_anno1 (display_anno (i,s)) = tag ("display-anno",[],[x_id i,x_disp_anno s])
    | x_anno1 (prec_anno (b,s,t)) = tag ("prec-anno",[],[x_bool_data b,x_id_list s,
                                    x_id_list t])
    | x_anno1 (lassoc_anno (i)) = tag ("lassoc-anno",[],(map x_id) i)
    | x_anno1 (rassoc_anno (i)) = tag ("rassoc-anno",[],(map x_id) i)
    | x_anno1 (conservative) = empty_tag ("conservative-anno",[])
    | x_anno1 (definitional) = empty_tag ("definitional-anno",[])
    | x_anno1 (mono) = empty_tag ("mono-anno",[])
    | x_anno1 (implies) = empty_tag ("implies-anno",[])
    | x_anno1 (pos_ANNO (r,x)) = tag ("pos-anno",[],[x_region r,x_anno x])
    
  and x_bool_data (x) = tag ("bool-data",[],[x_bool_data1 x])
  
  and x_bool_data1 (true) = empty_tag ("true",[])
    | x_bool_data1 (false) = empty_tag ("false",[])
    
  and x_disp_anno (s) = tag ("display-anno-content",[],[pcdata (esc s)])
  
  and x_id_list (l) = tag ("id-list",[],(map x_id) l)

  (* --------------------------------------------------------------------- *)
  (* basic specifications                                                  *)
  (* --------------------------------------------------------------------- *)

  and x_basic_spec (basic_spec (l)) = tag ("basic-spec",[],
                                      (map x_basic_items) l)
    | x_basic_spec (pos_BASIC_SPEC (r,x)) = tag ("pos-basic-spec",[],
                                            [x_region r,x_basic_spec x])
  
  and x_basic_items (x) = tag ("basic-items",[],[x_basic_items1 x])

  and x_basic_items1 (sig_items (s)) = x_sig_items s
    | x_basic_items1 (free_datatype (l,a)) = tag ("free-datatype",[],
                                             ((map x_datatype_decl) l) @
                                             ((map x_anno) a))
    | x_basic_items1 (cofree_datatype (l,a)) = tag ("cofree-datatype",[],
                                             ((map x_datatype_decl) l) @
                                             ((map x_anno) a))
    | x_basic_items1 (sort_gen (l,a)) = tag ("sort-gen",[],((map x_sig_items)
                                        l) @ ((map x_anno) a))
    | x_basic_items1 (sort_cogen (l,a)) = tag ("sort-cogen",[],((map
                                          x_sig_items) l) @ ((map x_anno) a))
    | x_basic_items1 (var_items (l,a)) = tag ("var-items",[],((map x_var_decl)
                                         l) @ ((map x_anno) a))
    | x_basic_items1 (local_var_axioms (a,b,c)) = tag ("local-var-axioms",[],
                                                  ((map x_var_decl) a) @
                                                  ((map x_l_formula) b) @
                                                  ((map x_anno) c))
    | x_basic_items1 (axiom_items (a,b)) = tag ("axiom-items",[], ((map
                                           x_l_formula) a) @ ((map x_anno) b))
    | x_basic_items1 (pos_BASIC_ITEMS (r,x)) = tag ("pos-basic-items",[],
                                            [x_region r,x_basic_items x])
 
  and x_sig_items (x) = tag ("sig-items",[],[x_sig_items1 x])

  and x_sig_items1 (sort_items (l,a)) = tag ("sort-items",[],((map
                                        x_l_sort_item) l) @ ((map x_anno) a))
    | x_sig_items1 (op_items (l,a)) = tag ("op-items",[],((map x_l_op_item) l)
                                      @ ((map x_anno) a))
    | x_sig_items1 (pred_items (l,a)) = tag ("pred-items",[],((map
                                        x_l_pred_item) l) @ ((map x_anno) a))
    | x_sig_items1 (datatype_items (l,a)) = tag ("datatype-items",[],
                                            ((map x_datatype_decl) l) @
                                            ((map x_anno) a))
    | x_sig_items1 (pos_SIG_ITEMS (r,x)) = tag ("pos-sig-items",[],
                                           [x_region r,x_sig_items x])
  
  and x_l_sort_item (i,a) = tag ("anno-sort-item",[],(x_sort_item i)::
                            ((map x_anno) a))

  and x_l_op_item (i,a) = tag ("anno-op-item",[],(x_op_item i)::
                          ((map x_anno) a))

  and x_l_pred_item (i,a) = tag ("anno-pred-item",[],(x_pred_item i)::
                            ((map x_anno) a))

  and x_sort_item (x) = tag ("sort-item",[],[x_sort_item1 x])
  
  and x_sort_item1 (sort_decl (l)) = tag ("sort-decl",[],(map x_sort) l)
    | x_sort_item1 (subsort_decl (l,s)) = tag ("subsort-decl",[],
                                          [x_sorts (sorts l),x_sort s])
    | x_sort_item1 (subsort_defn (s,v,t,f,a)) = tag ("subsort-defn",[],
                                                [x_sort s,x_var v,x_sort t,
                                                x_formula f] @
                                                ((map x_anno) a))
    | x_sort_item1 (iso_decl (l)) = tag ("iso-decl",[],(map x_sort) l)
    | x_sort_item1 (pos_SORT_ITEM (r,x)) = tag ("pos-sort-item",[],[x_region r,
                                                        x_sort_item x])
  
  and x_sort (i) = tag ("sort",[],[x_id i])
                              
  and x_sorts (sorts (l)) = tag ("sorts",[],(map x_sort) l)
    | x_sorts (pos_SORTS (r,x)) = tag ("pos-sorts",[],[x_region r,x_sorts x])

  and x_op_item (x) = tag ("op-item",[],[x_op_item1 x])
  
  and x_op_item1 (op_decl (l,t,a)) = tag ("op-decl",[],((map x_op_name) l) @
                                     [x_op_type t] @ ((map x_op_attr) a))
    | x_op_item1 (op_defn (n,h,t,a)) = tag ("op-defn",[],[x_op_name n,
                                       x_op_head h,x_term t] @ ((map x_anno) a))
    | x_op_item1 (pos_OP_ITEM (r,x)) = tag ("pos-op-item",[],[x_region r,
                                            x_op_item x])

  and x_op_name (i) = tag ("op-name",[],[x_id i])
  
  and x_op_type (x) = tag ("op-type",[],[x_op_type1 x])
  
  and x_op_type1 (total_op_type (s,t)) = tag ("total-op-type",[],[x_sorts s,
                                         x_sort t])
    | x_op_type1 (partial_op_type (s,t)) = tag ("partial-op-type",[],
                                           [x_sorts s,x_sort t])
    | x_op_type1 (pos_OP_TYPE (r,x)) = tag ("pos-op-type",[],[x_region r,
                                            x_op_type x])

  and x_op_head (x) = tag ("op-head",[],[x_op_head1 x])
  
  and x_op_head1 (total_op_head (l,s)) = tag ("total-op-head",[],
                                         ((map x_arg_decl) l) @ [x_sort s])
    | x_op_head1 (partial_op_head (l,s)) = tag ("partial-op-head",[],
                                           ((map x_arg_decl) l) @ [x_sort s])
    | x_op_head1 (pos_OP_HEAD (r,x)) = tag ("pos-op-head",[],[x_region r,
                                            x_op_head x])
  
  and x_arg_decl (arg_decl (v)) = tag ("arg-decl",[],[x_var_decl v])
    | x_arg_decl (pos_ARG_DECL (r,x)) = tag ("pos-arg-decl",[],[x_region r,
                                              x_arg_decl x])

  and x_op_attr (x) = tag ("op-attr",[],[x_op_attr1 x])
  
  and x_op_attr1 (associative) = empty_tag ("associative",[])
    | x_op_attr1 (commutative) = empty_tag ("commutative",[])
    | x_op_attr1 (idempotent)  = empty_tag ("idempotent",[])
    | x_op_attr1 (unit_op_attr (t)) = tag ("unit-op-attr",[],[x_term t])
    | x_op_attr1 (pos_OP_ATTR (r,x)) = tag ("pos-op-attr",[],[x_region r,
                                            x_op_attr x])
    
  and x_op_symb (x) = tag ("op-symb",[],[x_op_symb1 x])
  
  and x_op_symb1 (op_symb (n,Utils.None)) = (x_op_name n)
    | x_op_symb1 (op_symb (n,Utils.Some t)) = tag ("qual-op-name",[],
                                              [x_op_name n,x_op_type t])
    | x_op_symb1 (pos_OP_SYMB (r,x)) = tag ("pos-op-symb",[],[x_region r,
                                            x_op_symb x])
  
  and x_pred_item (x) = tag ("pred-item",[],[x_pred_item1 x])
  
  and x_pred_item1 (pred_decl (l,t)) = tag ("pred-decl",[],((map x_pred_name)
                                       l) @ [x_pred_type t])
    | x_pred_item1 (pred_defn (n,h,f,a)) = tag ("pred-defn",[],[x_pred_name n,
                                           x_pred_head h,x_l_formula f] @
                                           ((map x_anno) a))
    | x_pred_item1 (pos_PRED_ITEM (r,x)) = tag ("pos-pred-item",[],[x_region r,
                                                x_pred_item x])

  and x_pred_type (pred_type (s)) = tag ("pred-type",[],[x_sorts s])
    | x_pred_type (pos_PRED_TYPE (r,x)) = tag ("pos-pred-type",[],[x_region r,
                                               x_pred_type x])
  
  and x_pred_head (pred_head (l)) = tag ("pred-head",[],(map x_arg_decl) l)
    | x_pred_head (pos_PRED_HEAD (r,x)) = tag ("pos-pred-head",[],[x_region r,
                                               x_pred_head x])
  
  and x_pred_symb (x) = tag ("pred-symb",[],[x_pred_symb1 x])
  
  and x_pred_symb1 (pred_symb (n,Utils.None)) = (x_pred_name n)
    | x_pred_symb1 (pred_symb (n,Utils.Some t)) = tag ("qual-pred-name",[],
                                                [x_pred_name n,x_pred_type t])
    | x_pred_symb1 (pos_PRED_SYMB (r,x)) = tag ("pos-pred-symb",[],[x_region r,
                                                x_pred_symb x])
  
  and x_pred_name (i) = tag ("pred-name",[],[x_id i])
  
  and x_datatype_decl (datatype_decl (s,l,a)) = tag ("datatype-decl",[],
                                                [x_sort s] @
                                                ((map x_l_alternative) l) @
                                                ((map x_anno) a))
    | x_datatype_decl (pos_DATATYPE_DECL (r,x)) = tag ("pos-datatype-decl",[],
                                           [x_region r,x_datatype_decl x])
                                                
  and x_l_alternative (a,b) = tag ("anno-alternative",[],(x_alternative a)::
                              ((map x_anno) b))
                              
  and x_alternative (x) = tag ("alternative",[],[x_alternative1 x])
  
  and x_alternative1 (total_construct (n,l)) = tag ("total-construct",[],
                                               (x_op_name n)::
                                               ((map x_components) l))
    | x_alternative1 (partial_construct (n,l)) = tag ("partial-construct",[],
                                                 (x_op_name n)::
                                                 ((map x_components) l))
    | x_alternative1 (subsort (l)) = tag ("subsorts",[],(map x_sort) l)
    | x_alternative1 (pos_ALTERNATIVE (r,x)) = tag ("pos-alternative",[],
                                         [x_region r,x_alternative x])
    
  and x_components (x) = tag ("components",[],[x_components1 x])
  
  and x_components1 (total_select (l,s)) = tag ("total-select",[],
                                           ((map x_op_name) l) @ [x_sort s])
    | x_components1 (partial_select (l,s)) = tag ("partial-select",[],
                                             ((map x_op_name) l) @ [x_sort s])
    | x_components1 (sort_component (s)) = (x_sort s)
    | x_components1 (pos_COMPONENTS (r,x)) = tag ("pos-components",[],
                                        [x_region r,x_components x])

  and x_var_decl (l,s) = tag ("var-decl",[],((map x_var) l) @ [x_sort s])
  
  and x_var (i) = tag ("var",[],[x_simple_id i])
  
  and x_l_formula (f,a) = tag ("anno-formula",[],(x_formula f)::
                          ((map x_anno) a))
                          
  and x_formula (x) = tag ("formula",[],[x_formula1 x])
  
  and x_formula1 (quantification (q,l,f)) = tag ("quantification",[],
                                            [x_quantifier q] @
                                            ((map x_var_decl) l) @
                                            [x_formula f])
    | x_formula1 (pred_quantification (q,l,f)) = tag ("pred-quantification",
                                   [],[x_quantifier q] @
                                   ((map x_var_pred_type) l) @
                                   [x_formula f])
    | x_formula1 (conjunction (l)) = tag ("conjunction",[],(map x_formula) l)
    | x_formula1 (disjunction (l)) = tag ("disjunction",[],(map x_formula) l)
    | x_formula1 (equivalence (a,b)) = tag ("equivalence",[],[x_formula a,
                                       x_formula b])
    | x_formula1 (implication (a,b)) = tag ("implication",[],[x_formula a,
                                       x_formula b])
    | x_formula1 (negation (f)) = tag ("negation",[],[x_formula f])
    | x_formula1 (atom (a)) = (x_atom (a))
    | x_formula1 (sort_gen_ax (s,l)) = tag ("sort-gen-ax",[],((map x_sort) s) @
                                       ((map x_op_symb) l))
    | x_formula1 (sort_cogen_ax (s,l)) = tag ("sort-cogen-ax",[],((map x_sort)
                                       s) @ ((map x_op_symb) l))
    | x_formula1 (sort_cofree_ax (s,l)) = tag ("sort-cofree-ax",[],((map
                                        x_sort) s) @ ((map x_op_symb) l))
    | x_formula1 (unparsed_formula (s)) = tag ("unparsed",[],[pcdata (esc s)])
    | x_formula1 (pos_FORMULA (r,b,x)) = tag ("pos-formula",[],[x_region r,
                                               x_bool_data b,x_formula x])

  and x_var_pred_type (v,t) = tag ("var-pred-type",[],[x_id v,x_pred_type t])
 
  and x_atom (ttrue) = empty_tag ("true",[])
    | x_atom (ffalse) = empty_tag ("false",[])
    | x_atom (predication (s,t)) = tag ("predication",[],[x_pred_symb s,
                                   x_terms t])
    | x_atom (definedness (t)) = tag ("definedness",[],[x_term t])
    | x_atom (existl_equation (a,b)) = tag ("existl-equation",[],[x_term a,
                                       x_term b])
    | x_atom (strong_equation (a,b)) = tag ("strong-equation",[],[x_term a,
                                       x_term b])
    | x_atom (membership (t,s)) = tag ("membership",[],[x_term t,x_sort s])
    
  and x_quantifier (x) = tag ("quantifier",[],[x_quantifier1 x])
  
  and x_quantifier1 (forall) = empty_tag ("forall",[])
    | x_quantifier1 (exists) = empty_tag ("exists",[])
    | x_quantifier1 (exists_uniquely) = empty_tag ("exists-uniquely",[])
    | x_quantifier1 (pos_QUANTIFIER (r,x)) = tag ("pos-quantifier",[],
                                         [x_region r,x_quantifier x])
    
  and x_terms (terms l) = tag ("terms",[],(map x_term) l)
    | x_terms (pos_TERMS (r,x)) = tag ("pos-terms",[],[x_region r,x_terms x])
  
  and x_term (x) = tag ("term",[],[x_term1 x])
  
  and x_term1 (var_or_const (i)) = (x_simple_id i)
    | x_term1 (qual_var (v,s)) = tag ("qual-var",[],[x_var v,x_sort s])
    | x_term1 (application (s,t)) = tag ("application",[],[x_op_symb s,
                                    x_terms t])
    | x_term1 (sorted_term (t,s)) = tag ("sorted-term",[],[x_term t,x_sort s])
    | x_term1 (cast (t,s)) = tag ("cast",[],[x_term t,x_sort s])
    | x_term1 (conditional (a,f,b)) = tag ("conditional",[],[x_term a,
                                      x_formula f,x_term b])
    | x_term1 (unparsed_term (s)) = tag ("unparsed",[],[pcdata (esc s)])
    | x_term1 (pos_TERM (r,b,x)) = tag ("pos-term",[],[x_region r,
                                         x_bool_data b,x_term x])
 
  (* --------------------------------------------------------------------- *)
  (* identifiers                                                           *)
  (* --------------------------------------------------------------------- *)

  and x_id (simple_id (t)) = tag ("id",[],[x_sid t])
    | x_id (compound_id (t,l)) = tag ("id",[],[x_cid (t,l)])
 
  and is_token [token _] = true
    | is_token _         = false
     
  and x_sid (l,s,n) = if (is_token l) then
                        tag ("token-id",[],[x_tokop (hd l)])
                      else
                        tag ("mixfix-id",[],[x_token_places l])
                        
  and x_tokop (token t) = tag ("token",[],[x_tok t])
    | x_tokop (place)   = empty_tag ("place",[])
 
  and is_words x = (Char.isAlpha o hd o String.explode) x
  
  and is_dotwords x = (((hd o String.explode) x)=(#".")) andalso
                      ((String.size x) > 1)
    
  and x_tok t = if (is_words t) then
                  tag ("words",[],[pcdata (esc t)])
                else
                  if (is_dotwords t) then
                    tag ("dot-words",[],[pcdata (esc t)])
                  else
                    tag ("signs",[],[pcdata (esc t)])
  
  and x_token_places (l) = tag ("token-places",[],(map x_token_or_place) l)
  
  and x_token_or_place (x) = tag ("token-or-place",[],[x_tokop x])

  and x_cid ((l,s,n),x) = if (is_token l) then
                            tag ("token-id",[],[x_tcid ((hd l),x)])
                          else
                            tag ("mixfix-id",[],[x_mcid (l,x)])
                            
  and x_tcid (t,l) = tag ("comp-token-id",[],(x_tokop t)::
                     ((map x_id) l))
  
  and x_mcid (m,l) = tag ("comp-mixfix-id",[],(x_token_places m)::
                     ((map x_id) l))
  
  (* --------------------------------------------------------------------- *)
  (* architectural specifications                                          *)
  (* --------------------------------------------------------------------- *)

  and x_arch_spec_name n = tag ("arch-spec-name",[],[x_simple_id n])
  
  and x_l_arch_spec (s,a) = tag ("anno-arch-spec",[],(x_arch_spec s)::
                            ((map x_anno) a))
                            
  and x_arch_spec (x) = tag ("arch-spec",[],[x_arch_spec1 x])
  
  and x_arch_spec1 (basic_arch_spec (l,r,a)) = tag ("basic-arch-spec",[],
                                               (x_comp l) @ [x_result_unit r]
                                               @ ((map x_anno) a))
    | x_arch_spec1 (named_arch_spec (n)) = (x_arch_spec_name n)
    | x_arch_spec1 (pos_ARCH_SPEC (r,x)) = tag ("pos-arch-spec",[],
                                           [x_region r,x_arch_spec x])
    
  and x_comp l = List.concat ((map x_comp_one) l)
  
  and x_comp_one (u,a) = [tag ("anno-unit-decl-defn",[],(x_unit_decl_defn u)
                          :: ((map x_anno) a))]

  and x_unit_decl_defn (x) = tag ("unit-decl-defn",[],[x_unit_decl_defn1 x])
  
  and x_unit_decl_defn1 (unit_decl_case (u)) = (x_unit_decl u)
    | x_unit_decl_defn1 (unit_defn_case (u)) = (x_unit_defn u)
    | x_unit_decl_defn1 (pos_UNIT_DECL_DEFN (r,x)) = tag ("pos-unit-decl-defn",
                                  [],[x_region r,x_unit_decl_defn x])
    
  and x_unit_decl (unit_decl (n,s,i)) = tag ("unit-decl",[],[x_unit_name n,
                                        x_unit_spec s,x_unit_imports i])
    | x_unit_decl (pos_UNIT_DECL (r,x)) = tag ("pos-unit-decl",[],[x_region r,
                                               x_unit_decl x])
                                        
  and x_unit_defn (unit_defn (n,e)) = tag ("unit-defn",[],[x_unit_name n,
                                      x_unit_expression e])
    | x_unit_defn (pos_UNIT_DEFN (r,x)) = tag ("pos-unit-defn",[],[x_region r,
                                               x_unit_defn x])
                                      
  and x_unit_imports (unit_imports (l)) = tag ("unit-imports",[],
                                          (map x_unit_term) l)
    | x_unit_imports (pos_UNIT_IMPORTS (r,x)) = tag ("pos-unit-imports",[],
                                          [x_region r,x_unit_imports x])

  and x_l_unit_spec (s,a) = tag ("anno-unit-spec",[],(x_unit_spec s)::
                            ((map x_anno) a))

  and x_unit_spec (x) = tag ("unit-spec",[],[x_unit_spec1 x])
  
  and x_unit_spec1 (unit_type_case (t)) = (x_unit_type t)
    | x_unit_spec1 (spec_name_case (n)) = (x_spec_name n)
    | x_unit_spec1 (arch_spec_case (s)) = (x_l_arch_spec s)
    | x_unit_spec1 (closed (s)) = tag ("closed",[],[x_unit_spec s])
    | x_unit_spec1 (pos_UNIT_SPEC (r,x)) = tag ("pos-unit-spec",[],[x_region r,
                                                x_unit_spec x])
    
  and x_unit_type (unit_type (l,s)) = tag ("unit-type",[],[x_anno_specs l,
                                      x_l_spec s])
    | x_unit_type (pos_UNIT_TYPE (r,x)) = tag ("pos-unit-type",[],[x_region r,
                                               x_unit_type x])
                                      
  and x_anno_specs (l) = tag ("anno-specs",[],(map x_l_spec) l)
  
  and x_result_unit (result_unit (e,a)) = tag ("result-unit",[],
                                          (x_unit_expression e)::
                                          ((map x_anno) a))
    | x_result_unit (pos_RESULT_UNIT (r,x)) = tag ("pos-result-unit",[],
                                       [x_region r,x_result_unit x])
                                          
  and x_unit_expression (unit_expression (l,t)) = tag ("unit-expression",[],
                                                  ((map x_unit_binding) l) @
                                                  [x_unit_term t])
    | x_unit_expression (pos_UNIT_EXPRESSION (r,x)) = tag
              ("pos-unit-expression",[],[x_region r,x_unit_expression x])
  
  and x_unit_binding (unit_binding (n,s)) = tag ("unit-binding",[],
                                            [x_unit_name n,x_unit_spec s])
    | x_unit_binding (pos_UNIT_BINDING (r,x)) = tag ("pos-unit-binding",[],
                                    [x_region r,x_unit_binding x])
  
  and x_unit_term (x) = tag ("unit-term",[],[x_unit_term1 x])
  
  and x_unit_term1 (unit_translation (t,r)) = tag ("unit-translation",[],
                                              [x_unit_term t,x_renaming r])
    | x_unit_term1 (unit_reduction (t,r)) = tag ("unit-reduction",[],
                                            [x_unit_term t,x_restriction r])
    | x_unit_term1 (amalgamation (l)) = tag ("amalgamation",[],
                                        (map x_unit_term) l)
    | x_unit_term1 (local_unit (l,t)) = tag ("local-unit",[],
                                        ((map x_unit_defn) l) @
                                        [x_unit_term t])
    | x_unit_term1 (unit_appl (n,l)) = tag ("unit-appl",[],(x_unit_name n)::
                                       ((map x_fit_arg_unit) l))
    | x_unit_term1 (pos_UNIT_TERM (r,b,x)) = tag ("pos-unit-term",[],
						  [x_region r,
						   x_bool_data b,
                                                   x_unit_term x])
  
  and x_fit_arg_unit (fit_arg_unit (t,l)) = tag ("fit-arg-unit",[],
                                            (x_unit_term t)::
                                            ((map x_symb_map_items) l))
    | x_fit_arg_unit (pos_FIT_ARG_UNIT (r,x)) = tag ("pos-fit-arg-unit",[],
                                       [x_region r,x_fit_arg_unit x])

  and x_unit_name n = tag ("unit-name",[],[x_simple_id n])

  (* --------------------------------------------------------------------- *)
  (* casenv                                                                *)
  (* --------------------------------------------------------------------- *)

  and x_global_env (t,a) = tag ("casl",[],[x_global_env2 (t,a)])
                           
  and x_global_env2 (t,a) = tag ("casenv",[],(x_genv (Symtab_sid.dest t)) @
                            ((map x_anno) a))
  
  and x_genv l = List.concat ((map x_genv1) l)

  and x_genv1 (i,e) = [tag ("sid-global-entry",[],[x_simple_id i,
                       x_global_entry e])]
  
  and x_global_entry (x) = tag ("global-entry",[],[x_global_entry1 x])
  
  and x_global_entry1 (spec_defn_env (g,s)) = tag ("spec-defn-env",[],
                                              [x_genericity_env g,
                                              x_spec_lenv s])
    | x_global_entry1 (view_defn_env (g,s,m,t)) = tag ("view-defn-env",[],
                                                  [x_genericity_env g,
                                                  x_spec_lenv s,x_morphism m,
                                                  x_spec_lenv t])
    | x_global_entry1 (arch_spec_defn_env (e)) = tag ("arch-spec-defn-env",[],
                                                 [x_arch_sig e])
    | x_global_entry1 (unit_spec_defn_env (e)) = tag ("unit-spec-defn-env",[],
                                                 [x_unit_sig e])
  
  and x_genericity_env (s,l,t) = tag ("genericity-env",[],[x_spec_lenv s] @
                                 ((map x_spec_lenv) l) @ [x_sign t])
  
  and x_spec_lenv (SPEC_ENV (s,t,e)) = tag ("spec-lenv",[],[x_sign s,x_sign t,
                                       x_spec_env e])
  
  and x_spec_env (x) = tag ("spec-env",[],[x_spec_env1 x])
  
  and x_spec_env1 (basic_env (s,l)) = tag ("basic-env",[],(x_sign s)::
                                      ((map x_l_formula) l))
    | x_spec_env1 (translate_env (e,m)) = tag ("translate-env",[],
                                          [x_spec_env e,x_morphism m])
    | x_spec_env1 (derive_env (e,m)) = tag ("derive-env",[],[x_spec_env e,
                                       x_morphism m])
    | x_spec_env1 (union_env (l)) = tag ("union-env",[],(map x_spec_env) l)
    | x_spec_env1 (extension_env (l)) = tag ("extension-env",[],
                                        (map (x_spec_env o Utils.fst)) l)
    | x_spec_env1 (free_spec_env (e)) = tag ("free-spec-env",[],[x_spec_env e])
    | x_spec_env1 (closed_spec_env (e)) = tag ("closed-spec-env",[],
                                          [x_spec_env e])
    | x_spec_env1 (spec_inst_env (n,e,m,l)) = tag ("spec-inst-env",[],
                                              [x_spec_name n,x_spec_env e,
                                              x_morphism m] @
                                              ((map x_spec_env) l))
    | x_spec_env1 (dummy_spec_env) = empty_tag ("dummy-spec-env",[])
    | x_spec_env1 (cofree_spec_env (e)) = tag ("cofree-spec-env",[],
                                          [x_spec_env e])
  
  and x_morphism (s,f,p) = tag ("morphism",[],[x_sort_map s,x_fun_map f,
                           x_pred_map p])
  
  and x_sort_map (t) = tag ("sort-map",[],x_smap (Symtab_id.dest t))

  and x_smap l = List.concat ((map x_smap1) l)

  and x_smap1 (i,e) = [tag ("id-sort",[],[x_id i,x_sort e])]

  and x_fun_map (t) = tag ("fun-map",[],x_fmap (Symtab_id.dest t))

  and x_fmap l = List.concat ((map x_fmap1) l)

  and x_fmap1 (i,l) = [tag ("id-op-type-name",[],(x_id i) ::
                       (List.concat ((map x_fmap2) l)))]
  
  and x_fmap2 (t,n,b) = [tag ("op-type-name",[],[x_op_type t,x_op_name n,x_bool_data b])]

  and x_pred_map (t) = tag ("pred-map",[],x_pmap (Symtab_id.dest t))

  and x_pmap l = List.concat ((map x_pmap1) l)

  and x_pmap1 (i,l) = [tag ("id-pred-type-name",[],(x_id i) ::
                       (List.concat ((map x_pmap2) l)))]
  
  and x_pmap2 (t,n) = [tag ("pred-type-name",[],[x_pred_type t,
                       x_pred_name n])]

  and x_ext_signature (s,t,l) = tag ("ext-signature",[],[x_sign s,x_sign t] @
                                ((map x_l_formula) l))

  and x_sign (x) = tag ("sign",[],[x_local_env x])
  
  and x_local_env (s,v,f,p) = tag ("local-env",[],[x_subsort_env s,
                              x_var_env v,x_fun_env f,x_pred_env p])

  and x_subsort_env (t) = tag ("subsort-env",[],x_senv (Symtab_id.dest t))

  and x_senv l = List.concat ((map x_senv1) l)

  and x_senv1 (i,e) = [tag ("id-sorts",[],(x_id i) :: ((map x_sort) e))]
  
  and x_var_env (t) = tag ("var-env",[],x_venv (Symtab_sid.dest t))

  and x_venv l = List.concat ((map x_venv1) l)

  and x_venv1 (i,e) = [tag ("sid-sort",[],[x_simple_id i,x_sort e])]

  and x_fun_env (t) = tag ("fun-env",[],x_fenv (Symtab_id.dest t))

  and x_fenv l = List.concat ((map x_fenv1) l)

  and x_fenv1 (i,e) = [tag ("id-op-types",[],(x_id i) :: ((map x_op_type) e))]

  and x_pred_env (t) = tag ("pred-env",[],x_penv (Symtab_id.dest t))

  and x_penv l = List.concat ((map x_penv1) l)

  and x_penv1 (i,e) = [tag ("id-pred-types",[],(x_id i) ::
                       ((map x_pred_type) e))]
  
  (* --------------------------------------------------------------------- *)
  (* types from symmaps/symmaps_types.ml                                   *)
  (* --------------------------------------------------------------------- *)

  and x_symbol (x) = tag ("symbol",[],[x_symbol1 x])
  
  and x_symbol1 (SORT_SYMBOL (s)) = tag ("sort-symbol",[],[x_sort s])
    | x_symbol1 (TOTAL_FUN_SYMBOL (n,(l,s))) = tag ("total-fun-symbol",[],
                                             [x_op_name n,x_sorts (sorts l),
                                             x_sort s])
    | x_symbol1 (PARTIAL_FUN_SYMBOL (n,(l,s))) = tag ("partial-fun-symbol",[],
                                               [x_op_name n,x_sorts (sorts l),
                                               x_sort s])
    | x_symbol1 (PRED_SYMBOL (n,l)) = tag ("pred-symbol",[],(x_pred_name n)::
                                      ((map x_sort) l))

  (* --------------------------------------------------------------------- *)
  (* types from arch/arch_types.ml                                         *)
  (* --------------------------------------------------------------------- *)

  and x_arch_sig (n,u) = tag ("arch-sig",[],[x_named_unit_sigs n,x_unit_sig u])

  and x_unit_sig (c,s) = tag ("unit-sig",[],[x_comp_sub_sigs c,x_sub_sig s])
  
  and x_comp_sub_sigs (l) = tag ("comp-sub-sigs",[],(map x_sub_sig) l)
  
  and x_named_unit_sigs (t) = tag ("named-unit-sigs",[],x_nsub 
                              (Symtab_sid.dest t))

  and x_nsub l = List.concat ((map x_nsub1) l)

  and x_nsub1 (i,e) = [tag ("sig-unit-tag-sig",[],[x_simple_id i,
                       x_unit_tag_sig e])]

  and x_unit_tag_sig (c,t) = tag ("unit-tag-sig",[],[x_comp_tag_sigs c,
                             x_tag_sig t])
  
  and x_comp_tag_sigs (l) = tag ("comp-tag-sigs",[],(map x_tag_sig) l)
  
  and x_tag_sig (s,t) = tag ("tag-sig",[],[x_sub_sig s,x_tagging_map t])

  and x_tagging_map (t) = tag ("tagging-map",[],x_tmap 
                          (Symtab_esym.dest t))

  and x_tmap l = List.concat ((map x_tmap1) l)

  and x_tmap1 (i,e) = [tag ("e-ref-sym",[],(x_e_symbol i) :: 
                       ((map x_ref_sym) e))]
  
  and x_ref_sym (n,e) = tag ("ref-sym",[],[x_unit_name n,x_e_symbol e])
  
  and x_e_symbol (x) = tag ("e-symbol",[],[x_e_symbol1 x])
  
  and x_e_symbol1 (SYM_ESYMBOL (s)) = tag ("sym-esymbol",[],[x_symbol s])
    | x_e_symbol1 (EMB_ESYMBOL (s,t)) = tag ("emb-esymbol",[],[x_sort s,
                                        x_sort t])
  
  and x_sub_sig (e) = tag ("sub-sig",[],[x_local_env e])

  (* --------------------------------------------------------------------- *)
  (* fcasenv                                                               *)
  (* --------------------------------------------------------------------- *)

  and x_f_global_env (t,a) = tag ("casl",[],[
                             tag ("fcasenv",[],(x_f_genv (Symtab_sid.dest t))
                             @ ((map x_anno) a))])
  
  and x_f_genv l = List.concat ((map x_f_genv1) l)

  and x_f_genv1 (i,e) = [tag ("sid-f-global-entry",[],[x_simple_id i,
                         x_f_global_entry e])]
  
  and x_f_global_entry (x) = tag ("f-global-entry",[],[x_f_global_entry1 x])
  
  and x_f_global_entry1 (f_spec_defn_env (g,s)) = tag ("f-spec-defn-env",[],
                                                  [x_f_genericity_env g,
                                                  x_ext_signature s])
    | x_f_global_entry1 (f_view_defn_env (g,s,m,t)) = tag ("f-view-defn-env",
                                                      [],[x_f_genericity_env g,
                                                      x_ext_signature s,
                                                      x_morphism m,
                                                      x_ext_signature t])
    | x_f_global_entry1 (f_arch_spec_defn_env) = empty_tag
                                                 ("f-arch-spec-defn-env",[])
    | x_f_global_entry1 (f_unit_spec_defn_env) = empty_tag
                                                 ("f-unit-spec-defn-env",[])

  and x_f_genericity_env (s,l) = tag ("f-genericity-env",[],
                                 [x_ext_signature s,x_ext_signatures l])
  
  and x_ext_signatures l = tag ("ext-signatures",[],(map x_ext_signature) l)

  (* --------------------------------------------------------------------- *)
  (* end of code                                                           *)
  (* --------------------------------------------------------------------- *)

end
