(* *********************************************************************** *)
(*									   *)
(* Project: CATS 							   *)
(* Author:  Bartek Klin                                          	   *)
(* Date: 19.03.2000				 			   *)
(* Purpose of this file: Data structures for architectural specifications  *)
(*			 						   *)	
(*									   *)
(* *********************************************************************** *)

(* The data structures for architectural specifications follow the
   semantic concepts in the CASL Semantics v.0.96, July 1999
   (CoFI study note S-9). The page numbers refer to this document. *)

structure ArchTypes =
struct

(* General assumption *)
type sub_sig = LocalEnv.local_env;

(* p. 130 *)
(* This declaration should be added to as.sml *)
type UNIT_SPEC_NAME = AS.SIMPLE_ID;

(* p. 131 *)
datatype e_symbol = 
    SYM_ESYMBOL of Symbols.symbol
  | EMB_ESYMBOL of (AS.SORT * AS.SORT)
;

(*** Tables indexed by extended symbols. ***)
(*** Maybe should be moved to symtab.ML (but probably not) ***)
local
  open Utils Symbols IDOrder;
in
fun sym_ord(SORT_SYMBOL(s1),SORT_SYMBOL(s2)) = ID_ord(s1,s2)
  | sym_ord(SORT_SYMBOL(_),_) = LESS
  | sym_ord(TOTAL_FUN_SYMBOL(_),SORT_SYMBOL(_)) = GREATER
  | sym_ord(TOTAL_FUN_SYMBOL(f1,(ss1,s1)),TOTAL_FUN_SYMBOL(f2,(ss2,s2))) = 
      let val ord1 = ID_ord(f1,f2) in
        if ord1=EQUAL then
          let val ord2 = IDlist_ord(ss1,ss2) in
            if ord2=EQUAL then ID_ord(s1,s2) else ord2
          end
        else ord1
      end
  | sym_ord(TOTAL_FUN_SYMBOL(_),_) = LESS
  | sym_ord(PARTIAL_FUN_SYMBOL(_),PRED_SYMBOL(_)) = LESS
  | sym_ord(PARTIAL_FUN_SYMBOL(f1,(ss1,s1)),PARTIAL_FUN_SYMBOL(f2,(ss2,s2))) =
      let val ord1 = ID_ord(f1,f2) in
        if ord1=EQUAL then
          let val ord2 = IDlist_ord(ss1,ss2) in
            if ord2=EQUAL then ID_ord(s1,s2) else ord2
          end
        else ord1
      end
  | sym_ord(PARTIAL_FUN_SYMBOL(_),_) = GREATER
  | sym_ord(PRED_SYMBOL(p1,ss1),PRED_SYMBOL(p2,ss2)) = 
      let val ord1 = ID_ord(p1,p2) in
        if ord1=EQUAL then IDlist_ord(ss1,ss2) else ord1
      end
  | sym_ord(PRED_SYMBOL(_),_) = GREATER

fun esym_ord(SYM_ESYMBOL(s1),SYM_ESYMBOL(s2)) = sym_ord(s1,s2)
  | esym_ord(SYM_ESYMBOL(_),_) = LESS
  | esym_ord(EMB_ESYMBOL(s1,s2),EMB_ESYMBOL(s3,s4)) = 
      let val ord1 = ID_ord(s1,s3) in
        if ord1=EQUAL then ID_ord(s2,s4) else ord1
      end
  | esym_ord(EMB_ESYMBOL(_,_),_) = GREATER
end;

structure Symtab_esym = TableFun(type key = e_symbol val ord = esym_ord);

(******** End of tables indexed by extended symbols. ******)

(* p. 131 again *)
type ref_sym = AS.UNIT_NAME * e_symbol;
type tagging_map = ref_sym list Symtab_esym.table;
type tag_sig = sub_sig * tagging_map;
type comp_tag_sigs = tag_sig list;
type unit_tag_sig = comp_tag_sigs * tag_sig;
type named_unit_sigs = unit_tag_sig Symtab_sid.table;

(* p. 130 again *)
type comp_sub_sigs = sub_sig list;
type unit_sig = comp_sub_sigs * sub_sig;
type arch_sig = named_unit_sigs * unit_sig;

(* p. 141 *)
type st_unit_ctx = named_unit_sigs;

end;
