(* *********************************************************************** *)
(*									   *)
(* Project: CATS 							   *)
(* Author: Frederic Voisin and Till Mossakowski         		   *)
(* Date: 1998/2000				 			   *)
(* Purpose of this file: ML lex grammar for CASL			   *)
(*			 						   *)	
(*									   *)
(* *********************************************************************** *)

(* The lexical grammar follows the lexical grammar in the summary, but has been
   modified at a few places.
   Paths and URLs are recognized as a whole lexeme together with the
   preceding keyword, because it would be difficult to recognize
   them standalone (because of the overlap with normals WORDS).

   Annotations are recognized as whole lexems.
*)

type pos = int*int
type svalue = Tokens.svalue
type ('a,'b) token = ('a,'b) Tokens.token
type lexresult = (svalue,pos) token


val html = ref false

(* Compute a (row,col) position from a byte position *)
fun gp index = 
    let fun gp1 row (l::rowlengths) col =
            if col>l then gp1 (row+1) rowlengths (col-l-1)
            else (row,col)
          | gp1 _ _ _ = (~1,~1) (*raise Utils.ERROR*)
    in gp1 1 (!Global.Lexer_line_lengths) (index-1)
    end 

fun error pos s = Utils.writeln (Utils.perr (gp pos,(0,0))^s);

fun eof () = Tokens.EOF (((0,0),(0,0)))

fun take2 x =
    (Utils.implode o tl o tl o Utils.explode) x

fun take2both x =
    (Utils.implode o rev o tl o tl o rev o tl o tl o Utils.explode) x

%%
%header (functor CASLLexFun(structure Tokens : CASL_TOKENS));

%full

alpha = [A-Za-z];
digit = [0-9];
ws = [\ \t];
wsn = [\ \t\n\015];
sign = (["+" "*" "/" "\\" "&" "=" "<" ">" "|" "^" "~" "!" "?" ":" "." "$" "@" 
   "#" "" "" "" "" "" "" "" "" "" "" "" "" 
   "" "" "" ""] | "-" | "");  
WORD = ({alpha}|{digit}|"'")+;
LETTER_WORD = {alpha}({alpha}|{digit}|"'")*;
WORDS = {LETTER_WORD} (_{WORD})*;

FRACTION = {digit}+ "." {digit}+;
EXP = [eE][+-]?{digit}+;

odigit = [0-7];
hdigit = [0-9A-Fa-f];
printable = " " | [!-];
escape =   ("\\"[tnvrbfa "?" "'" "\"" "\\"])
         | ("\\"{digit}{digit}{digit})
         | ("\\"o{odigit}{odigit}{odigit})
         | ("\\"x{hdigit}{hdigit});
rawChar = {escape} | {printable};

version = {digit}+ ("." {digit}+)*;

anyexceptrbra = [!-Z_-~ -]
                | {wsn} | "[" | "\\" | "^"; 

%%

\n	=> (lex ());
{ws}+	=> (lex ());

"%%".*"\n" => (Tokens.COMMENT_LINE (take2 yytext,gp(yypos),gp(yypos+String.size yytext-1)));
"%{"([^\}]|\}[^\%])*"}%" => (Tokens.COMMENT (take2both yytext,gp(yypos),gp(yypos+String.size yytext-1)));
"%["({anyexceptrbra}|"]"[^\%])*"]%" => (lex ());
"%"{WORDS}.*"\n" => (Tokens.ANNO (yytext,gp(yypos),gp(yypos+String.size yytext-1)));
"%"{WORDS}"("([^\)]|\)[^\%])*")%" => (Tokens.ANNO (yytext,gp(yypos),gp(yypos+String.size yytext-1)));


"%!%display" => (Tokens.DISPLAY_ANNO ((0,0),(0,0)));
"%!%prec" => (Tokens.PREC_ANNO ((0,0),(0,0)));
"%!%left_assoc" => (Tokens.LEFT_ASSOC ((0,0),(0,0)));
"%!%right_assoc" => (Tokens.RIGHT_ASSOC ((0,0),(0,0)));
"%!%cons" => (Tokens.CONS_ANNO ((0,0),(0,0)));
"%!%def" => (Tokens.DEF_ANNO ((0,0),(0,0)));
"%!%mono" => (Tokens.MONO_ANNO ((0,0),(0,0)));
"%!%implies" => (Tokens.IMPLIES_ANNO ((0,0),(0,0)));
"%!%number" => (Tokens.NUMBER_ANNO ((0,0),(0,0)));
"%!%floating" => (Tokens.FLOATING_ANNO ((0,0),(0,0)));
"%!%string" => (Tokens.STRING_ANNO ((0,0),(0,0)));
"%!%list" => (Tokens.LIST_ANNO ((0,0),(0,0)));
"%!%author".* => (Tokens.OTHER_ANNO (yytext,(0,0),(0,0)));
"%!%authors".* => (Tokens.OTHER_ANNO (yytext,(0,0),(0,0)));
"%!%date".* => (Tokens.OTHER_ANNO (yytext,(0,0),(0,0)));
"%!%dates".* => (Tokens.OTHER_ANNO (yytext,(0,0),(0,0)));



"%("([^\)]|\)[^\%])*")%" => 	
  (Tokens.LABEL 
      (take2both yytext,
       gp(yypos),gp(yypos+String.size yytext-1)));
")%"	=>  (Tokens.CLOSE_ANNO (gp(yypos),gp(yypos+String.size yytext-1)));

"("	=>  (Tokens.LPAREN (gp(yypos),gp(yypos+String.size yytext-1))); 
")"	=>  (Tokens.RPAREN (gp(yypos),gp(yypos+String.size yytext-1)));
"["	=>  (Tokens.LSQBRA (gp(yypos),gp(yypos+String.size yytext-1)));
"]"	=>  (Tokens.RSQBRA (gp(yypos),gp(yypos+String.size yytext-1)));
"{"	=>  (Tokens.LBRACE (gp(yypos),gp(yypos+String.size yytext-1)));
"}"	=>  (Tokens.RBRACE (gp(yypos),gp(yypos+String.size yytext-1)));
","	=>  (Tokens.COMMA (gp(yypos),gp(yypos+String.size yytext-1)));
"|->"        =>  (Tokens.MAPSTO (gp(yypos),gp(yypos+String.size yytext-1)));
"/\\"	=>  (Tokens.AND (gp(yypos),gp(yypos+String.size yytext-1)));
"=e="	=>  (Tokens.EXEQUALTO (gp(yypos),gp(yypos+String.size yytext-1)));
"=>"	=>  (Tokens.IMPLIES (gp(yypos),gp(yypos+String.size yytext-1)));
"<=>"	=>  (Tokens.EQUIVALENTTO (gp(yypos),gp(yypos+String.size yytext-1)));
"."	=>  (Tokens.DOT (gp(yypos),gp(yypos+String.size yytext-1)));
""	=>  (Tokens.DOT (gp(yypos),gp(yypos+String.size yytext-1)));
"|"	=>  (Tokens.BAR (gp(yypos),gp(yypos+String.size yytext-1)));
"\\/"	=>  (Tokens.OR (gp(yypos),gp(yypos+String.size yytext-1)));
""	=>  (Tokens.NOT (gp(yypos),gp(yypos+String.size yytext-1)));
"<"	=>  (Tokens.LESSTHAN (gp(yypos),gp(yypos+String.size yytext-1)));
"?"	=>  (Tokens.QUESTMARK (gp(yypos),gp(yypos+String.size yytext-1)));
"*"	=>  (Tokens.TIMES (gp(yypos),gp(yypos+String.size yytext-1)));
"/"	=>  (Tokens.SLASH (gp(yypos),gp(yypos+String.size yytext-1)));
""	=>  (Tokens.TIMES (gp(yypos),gp(yypos+String.size yytext-1)));
"="	=>  (Tokens.EQUALTO (gp(yypos),gp(yypos+String.size yytext-1)));
":"	=>  (Tokens.COLON (gp(yypos),gp(yypos+String.size yytext-1)));
":?"	=>  (Tokens.COLON_QUEST (gp(yypos),gp(yypos+String.size yytext-1)));
"::="	=>  (Tokens.ISDEFBY (gp(yypos),gp(yypos+String.size yytext-1)));
"->"	=>  (Tokens.ARROW (gp(yypos),gp(yypos+String.size yytext-1)));
"->?"	=>  (Tokens.PARROW (gp(yypos),gp(yypos+String.size yytext-1)));
")?"	=>  (Tokens.RPAREN_QUEST (gp(yypos),gp(yypos+String.size yytext-1)));
";"	=>  (Tokens.SEMICOLON (gp(yypos),gp(yypos+String.size yytext-1)));
"__"	=>  (Tokens.PLACE (gp(yypos),gp(yypos+String.size yytext-1)));
"<>"   =>  (Tokens.UNEQUAL (gp(yypos),gp(yypos+String.size yytext-1)));
"and"	=>  (Tokens.UNION (gp(yypos),gp(yypos+String.size yytext-1)));
"arch"	=>  (Tokens.ARCH (gp(yypos),gp(yypos+String.size yytext-1)));
"as"	=>  (Tokens.AS (gp(yypos),gp(yypos+String.size yytext-1)));
"assoc"	=>  (Tokens.ASSOC (gp(yypos),gp(yypos+String.size yytext-1)));
"axiom"	=>  (Tokens.AXIOM (gp(yypos),gp(yypos+String.size yytext-1)));
"axioms"	=>  (Tokens.AXIOM (gp(yypos),gp(yypos+String.size yytext-1)));
"closed"	=>  (Tokens.CLOSED (gp(yypos),gp(yypos+String.size yytext-1)));
"comm"	=>  (Tokens.COMM (gp(yypos),gp(yypos+String.size yytext-1)));
"def"	=>  (Tokens.DEF (gp(yypos),gp(yypos+String.size yytext-1)));
"else"	=>  (Tokens.ELSE (gp(yypos),gp(yypos+String.size yytext-1)));
"end"	=>  (Tokens.END (gp(yypos),gp(yypos+String.size yytext-1)));
"exists!" =>  (Tokens.EXISTS_UNIQUELY (gp(yypos),gp(yypos+String.size yytext-1)));
"exists"	=>  (Tokens.EXISTS (gp(yypos),gp(yypos+String.size yytext-1)));
"false"	=>  (Tokens.FALSE (gp(yypos),gp(yypos+String.size yytext-1)));
"fit"	=>  (Tokens.FIT (gp(yypos),gp(yypos+String.size yytext-1)));
"forall"	=>  (Tokens.FORALL (gp(yypos),gp(yypos+String.size yytext-1)));
"free"	=>  (Tokens.FREE (gp(yypos),gp(yypos+String.size yytext-1)));
"cofree"	=>  (Tokens.COFREE (gp(yypos),gp(yypos+String.size yytext-1)));
"from"	=>  (Tokens.FROM (gp(yypos),gp(yypos+String.size yytext-1)));
"generated" =>  (Tokens.GENERATED (gp(yypos),gp(yypos+String.size yytext-1)));
"cogenerated" =>  (Tokens.COGENERATED (gp(yypos),gp(yypos+String.size yytext-1)));
"get"	=>  (Tokens.GET (gp(yypos),gp(yypos+String.size yytext-1)));
"given"	=>  (Tokens.GIVEN (gp(yypos),gp(yypos+String.size yytext-1)));
"hide"	=>  (Tokens.HIDE (gp(yypos),gp(yypos+String.size yytext-1)));
"idem"	=>  (Tokens.IDEM (gp(yypos),gp(yypos+String.size yytext-1)));
"if"	=>  (Tokens.IF (gp(yypos),gp(yypos+String.size yytext-1)));
"in"	=>  (Tokens.IN (gp(yypos),gp(yypos+String.size yytext-1)));
"lambda" =>  (Tokens.LAMBDA (gp(yypos),gp(yypos+String.size yytext-1)));
"library"	=>  (Tokens.LIBRARY (gp(yypos),gp(yypos+String.size yytext-1)));
"local"	=>  (Tokens.LOCAL (gp(yypos),gp(yypos+String.size yytext-1)));
"not"	=>  (Tokens.NOT (gp(yypos),gp(yypos+String.size yytext-1)));
"op"	=>  (Tokens.OP (gp(yypos),gp(yypos+String.size yytext-1)));
"ops"	=>  (Tokens.OP (gp(yypos),gp(yypos+String.size yytext-1)));
"pred"	=>  (Tokens.PRED (gp(yypos),gp(yypos+String.size yytext-1)));
"preds"	=>  (Tokens.PRED (gp(yypos),gp(yypos+String.size yytext-1)));
"result"	=>  (Tokens.RESULT (gp(yypos),gp(yypos+String.size yytext-1)));
"reveal"	=>  (Tokens.REVEAL (gp(yypos),gp(yypos+String.size yytext-1)));
"sort"	=>  (Tokens.SORT (gp(yypos),gp(yypos+String.size yytext-1)));
"sorts"	=>  (Tokens.SORT (gp(yypos),gp(yypos+String.size yytext-1)));
"spec"	=>  (Tokens.SPEC (gp(yypos),gp(yypos+String.size yytext-1)));
"then"	=>  (Tokens.THEN (gp(yypos),gp(yypos+String.size yytext-1)));
"to"	=>  (Tokens.TO (gp(yypos),gp(yypos+String.size yytext-1)));
"true"	=>  (Tokens.TRUE (gp(yypos),gp(yypos+String.size yytext-1)));
"type"	=>  (Tokens.TYPE (gp(yypos),gp(yypos+String.size yytext-1)));
"types"	=>  (Tokens.TYPE (gp(yypos),gp(yypos+String.size yytext-1)));
"unit"	=>  (Tokens.UNIT (gp(yypos),gp(yypos+String.size yytext-1)));
"units"	=>  (Tokens.UNIT (gp(yypos),gp(yypos+String.size yytext-1)));
"var"	=>  (Tokens.VAR (gp(yypos),gp(yypos+String.size yytext-1)));
"vars"	=>  (Tokens.VAR (gp(yypos),gp(yypos+String.size yytext-1)));
"view"	=>  (Tokens.VIEW (gp(yypos),gp(yypos+String.size yytext-1)));
"when"	=>  (Tokens.WHEN (gp(yypos),gp(yypos+String.size yytext-1)));
"with"	=>  (Tokens.WITH (gp(yypos),gp(yypos+String.size yytext-1)));
"within"	=>  (Tokens.WITHIN(gp(yypos),gp(yypos+String.size yytext-1)));

"version" ({wsn}*) {version} 
       =>  (Tokens.VERSION (yytext,gp(yypos),gp(yypos+String.size yytext-1))); 

{WORDS} => (Tokens.WORDS (yytext,gp(yypos),gp(yypos+String.size yytext-1)));
"."{WORDS} => (Tokens.DOT_WORDS (yytext,gp(yypos),gp(yypos+String.size yytext-1)));
{digit} => (Tokens.DIGIT (yytext,gp(yypos),gp(yypos+String.size yytext-1)));
{digit}({digit}+) => (Tokens.NNUMBER (yytext,gp(yypos),gp(yypos+String.size yytext-1)));
{FRACTION}{EXP}?  => (Tokens.FLOAT (yytext,gp(yypos),gp(yypos+String.size yytext-1)));
{digit}+{EXP}  => (Tokens.FLOAT (yytext,gp(yypos),gp(yypos+String.size yytext-1)));
"\"" ( ("\\" (["\"" "\\" "'" ntrvbfa"?"] | {digit}{3} | x[0-9a-fA-F]{2} | o[0-7]{3})) | [^"\\""\""] )* "\""
	=> (Tokens.STRING (Utils.implode (rev (tl (rev (tl (Utils.explode yytext))))),gp(yypos),gp(yypos+String.size yytext-1)));
{sign}+ =>  (Tokens.NO_BRACKET_SIGNS (yytext,gp(yypos),gp(yypos+String.size yytext-1)));
"'"{rawChar}"'" => (Tokens.CHAR (yytext,gp(yypos),gp(yypos+String.size yytext-1)));
"\""{rawChar}*"\"" => (Tokens.STRING (yytext,gp(yypos),gp(yypos+String.size yytext-1)));

.       => (error yypos ("ignoring bad character "^yytext); lex ());

